/*                  AFDISK  -  Amiga RDB Fdisk for Linux
 *                amigastuff.c part  -  amiga specific stuff
 *                   written in 1996 by Stefan Reinauer
 *
 *    Copyright (C) 1996,1997 Stefan Reinauer
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 1, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA
 */

#include <fcntl.h>
#include <unistd.h>
#include <linux/hdreg.h>
#include <sys/ioctl.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>

#include <asm/system.h>
#include <asm/byteorder.h>

#include "amiga/types.h"
#include "amiga/filehandler.h"
#include "amiga/hardblocks.h"

#include "fdisk.h"

#define SECTOR_SIZE 512
#define MAXPARTS     16

#define BBB(pos)  ((struct BadBlockBlock *)(pos))
#define PART(pos) ((struct PartitionBlock *)(pos))
#define FSHB(pos) ((struct FileSysHeaderBlock *)(pos))
#define LSEG(pos) ((struct LoadSegBlock *)(pos))
#define RDSK(pos) ((struct RigidDiskBlock *)(pos)) 

struct AmigaBlock {
    ULONG   amiga_ID;             /* 4 character identifier */
    ULONG   amiga_SummedLongs;    /* size of this checksummed structure */
    LONG    amiga_ChkSum;         /* block checksum (longword sum to zero) */
};

extern char disk_device[256];
char *initsectors;
char *sectortable;
struct RigidDiskBlock *rdb;
struct PartitionBlock *pa[MAXPARTS];
int parts,firstblock,lastblock,minblock,maxblock;

int rigiddisk_new(int first);
int pt(void);


int checksum(struct AmigaBlock *ptr)
{
  int i, end;
  LONG sum;
  ULONG *pt=(ULONG *)ptr;
  sum=htonl(pt[0]); end=htonl(pt[1]);
  if (end>SECTOR_SIZE) end=SECTOR_SIZE;
  for(i=1;i<end;i++) sum+=htonl(pt[i]);
  return (int)sum;
}

int sector_correct(struct AmigaBlock *ablock)
{
 ULONG btype;
 btype=htonl(ablock->amiga_ID);
 if (btype==IDNAME_RIGIDDISK
  || btype==IDNAME_BADBLOCK || btype==IDNAME_FILESYSHEADER
  || btype==IDNAME_LOADSEG  || btype==IDNAME_PARTITION)
  ablock->amiga_ChkSum = ntohl( htonl(ablock->amiga_ChkSum) -
                                checksum((struct AmigaBlock  *)ablock));
  else return -1;
 return 0;
}

int check_lseg(int blk, char *data)
{
  char *act;
  while (blk!=-1) {
    if (blk>lastblock) lastblock=blk;
    sectortable[blk]='x';
    act=data+(blk*SECTOR_SIZE);
    if (htonl(LSEG(act)->lsb_ID)!=IDNAME_LOADSEG) {
      fprintf (stderr, "LoadSeg Block messed up at block %d\n",blk);
      return -1;
    }
    if (checksum((struct AmigaBlock *)act)!=0) {
      fprintf (stderr, "LoadSeg Block %d has wrong checksum. Corrected\n",blk);
      sector_correct((struct AmigaBlock *)act);
    }
    blk=htonl(LSEG(act)->lsb_Next);
  }
  return 0;
}

int check_bbb(int blk, char *data)
{
  char *act;
  while (blk!=-1) {
    if (blk>lastblock) lastblock=blk;
    sectortable[blk]='x';
    act=data+(blk*SECTOR_SIZE);
    if (htonl(BBB(act)->bbb_ID)!=IDNAME_BADBLOCK) {
      fprintf (stderr, "BadBlockList messed up at block %d\n",blk);
      return -1;
    }
    if (checksum((struct AmigaBlock *)act)!=0) {
      fprintf (stderr, "BadBlockList block %d has wrong checksum. Corrected\n",blk);
      sector_correct((struct AmigaBlock *)act);
    }
    blk=htonl(BBB(act)->bbb_Next);
  }
  return 0;
}

int connectivity(void)
{
 char *actual;
 int  block;
 char *data=initsectors;

/* - Check if all checksums in init sectors are ok
 * - and repair them (ought to work)
 * - fill partition info into struct *pa
 */

 lastblock=firstblock;
 sectortable[firstblock]='x';

 /* Check BadBlockBlocks, LoadSegBlocks & FileSystemHeaderBlocks */
  if (check_bbb(htonl(rdb->rdb_BadBlockList), data)==-1) return -1;
  if (check_lseg(htonl(rdb->rdb_DriveInit), data)==-1) return -1;
  block=htonl(rdb->rdb_FileSysHeaderList);

   while (block!=-1) {
    if (block>lastblock) lastblock=block;
    sectortable[block]='x';
    actual=data+(block*SECTOR_SIZE);
    if (htonl(FSHB(actual)->fhb_ID)!=IDNAME_FILESYSHEADER) {
      fprintf (stderr, "Filesystem Header Block messed up at block %d\n",block);
      return -1;
    }
    if (checksum((struct AmigaBlock *)actual)!=0) {
      fprintf (stderr, "Filesystem Header Block %d has wrong checksum. Corrected\n",block);
      sector_correct((struct AmigaBlock *)actual);
    }
    if (check_lseg(htonl(FSHB(actual)->fhb_SegListBlocks), data)==-1) return -1;
    block=htonl(FSHB(actual)->fhb_Next);
  }

 /* Partition Blocks */

  parts=0;
  block=htonl(rdb->rdb_PartitionList);
  
  while (block!=-1) {
    if (block>lastblock) lastblock=block;
    sectortable[block]='x';
    actual=data+(block*SECTOR_SIZE);
    if (htonl(PART(actual)->pb_ID)!=IDNAME_PARTITION) {
      fprintf (stderr, "Partition Block messed up at block %d\n",block);
      return -1;
    }
    if (checksum((struct AmigaBlock *)actual)!=0) {
      fprintf (stderr, "Partition Block %d has wrong checksum. Corrected\n",block);
      sector_correct((struct AmigaBlock *)actual);
    }

    pa[parts]=PART(actual);
    parts++;
    if (parts>MAXPARTS) {
      fprintf (stderr, "Too many partitions.  MAXPARTS=%d\n",MAXPARTS);
      return -1;
    }

    block=htonl(PART(actual)->pb_Next);
  }

  /* Everything seems to be alright */
  return 0;
}

int get_rdb(void)
{
 /* Initial work.
  * - Get RDB
  * - Check size of all init blocks
  * - get init sectors
  */

 int f,i;

 if ((rdb=(struct RigidDiskBlock *)malloc(SECTOR_SIZE))==NULL) {
   fprintf (stderr,"Not enough main memory for one sector (poor guy).\n");
   return -1;
 }

 if ((f=open(disk_device,O_RDONLY))<0) {
   if (!list_only || get_dev)
     fprintf (stderr,"Cannot open device %s\n",disk_device);
   free((char *)rdb);
   return -1;
 }

 i=0;
 firstblock=-1;
 while ((i<RDB_LOCATION_LIMIT)&&(firstblock==-1)) {

   if (lseek(f,(i*SECTOR_SIZE),SEEK_SET)<0) {
     close(f); free((char *)rdb);
     fprintf (stderr,"RDB seek error.\n");
     return -1;
   }

   if (read(f,(char *)rdb,SECTOR_SIZE)!=SECTOR_SIZE) {
     close(f); free((char *)rdb);
     fprintf (stderr,"RDB read error.\n");
     return -1;
   }

   if (htonl(rdb->rdb_ID)==IDNAME_RIGIDDISK) {
     firstblock=i;
     if (checksum((struct AmigaBlock *)rdb)) {
       fprintf(stderr,"RDB on block %d has bad checksum. Corrected.\n",i);
       sector_correct((struct AmigaBlock *)rdb);
     }
   }
   i++;
 }
 if (firstblock==-1) {
   fprintf(stderr,"No valid RDB found.\n");
   close(f); free((char *)rdb);
   rigiddisk_new(0);
   return 0;
 }

 i=htonl(rdb->rdb_RDBBlocksHi);
 free ((char *)rdb);

 if ((initsectors=malloc(i*SECTOR_SIZE))==NULL) {
   fprintf (stderr,"Not enough memory for all init sectors.\n");
   return -1;
 }

 if ((sectortable=malloc(i))==NULL) {
   fprintf (stderr,"Not enough memory for sector table.\n");
   return -1;
 }

 memset(sectortable,-1,i);

 if (lseek(f,0,SEEK_SET)<0) {
   close(f); free(initsectors); free(sectortable);
   fprintf (stderr,"Disk seek error.\n");
   return -1;
 }

 if (read(f,initsectors,i*SECTOR_SIZE)!=i*SECTOR_SIZE) {
   close(f); free(initsectors); free (sectortable);
   fprintf (stderr,"Disk read error.\n");
   return -1;
 }

 close (f);

 rdb=(struct RigidDiskBlock *)(initsectors+firstblock*SECTOR_SIZE);
 maxblock=htonl(rdb->rdb_RDBBlocksHi);
 minblock=htonl(rdb->rdb_RDBBlocksLo);

 if (connectivity()==-1) {
   free(initsectors); free(sectortable); fprintf (stderr,"Could not repair checksums.\n");
   return -1;
 }

 /* Everything alright ? */
 return 0;
}

int countblk(void)
{
 int i,j=0;
 for (i=0;i<(maxblock+1);i++) if (sectortable[i]=='x') j++;
 return j;
}
 

int findblk(void)
{
 int i;
 for (i=firstblock;i<(maxblock+1);i++) if (sectortable[i]!='x') return i;
 return -1;
}

int room_for_partition(void)
{
 int i;
 if (parts==MAXPARTS) {
  fprintf (stderr,"Too many partitions.   MAXPARTS=%d\n",MAXPARTS);
  return -1;
 }

 if ((i=findblk())<minblock) {
  fprintf (stderr,"No room in RDB area for new partition. Sorry.\n");
  return -1;
 }
 return i; 
}

int partition_new(int nr)
{
  int i,j;
  struct DosEnvec *de;

  if ((nr<1)||(nr>parts+1)) {
      fprintf (stderr,"Invalid partition number.   existing: %d, tried: %d\n",parts,nr);
      return -1;
    }

  if ((i=room_for_partition())==-1) {
      fprintf (stderr,"Could not create new partition.\n");
      return -1;
    }

  fprintf (stdout, "Creating new partition entry at block %d.\n",i);

  nr--;
  for (j=parts;j>nr;j--) pa[j]=pa[j-1];
  parts++;

  sectortable[i]='x';
  if (i>lastblock) lastblock=i;  

  rdb->rdb_HighRDSKBlock=ntohl(lastblock);
  pa[nr]=PART(initsectors+i*SECTOR_SIZE);
  memset(pa[nr],-1,SECTOR_SIZE);

  pa[nr]->pb_ID          =  ntohl(IDNAME_PARTITION);
  pa[nr]->pb_SummedLongs =  ntohl(64);
  pa[nr]->pb_HostID      =  rdb->rdb_HostID;
  pa[nr]->pb_Flags       =  ntohl(0);
  pa[nr]->pb_DevFlags    =  ntohl(0);

  memcpy(pa[nr]->pb_DriveName,"\003dhx",4);

  if (nr==0) {
    pa[nr]->pb_Next=rdb->rdb_PartitionList;
    rdb->rdb_PartitionList=ntohl(i);
  } else {
    pa[nr]->pb_Next=pa[nr-1]->pb_Next;
    pa[nr-1]->pb_Next=ntohl(i);
  } 

  de=(struct DosEnvec *)pa[nr]->pb_Environment;
  de->de_TableSize       = ntohl(19);                   /* 20 ? */
  de->de_SizeBlock       = ntohl(128);
  de->de_SecOrg          = ntohl(0);
  de->de_Surfaces	= ntohl(1);
	/* On my old Seagate this was 1. If it does not work, try
         * de->de_Surfaces=rdb->rdb_Heads   */

  de->de_SectorPerBlock = ntohl(1);
  de->de_BlocksPerTrack  = rdb->rdb_CylBlocks;
  /* BlocksPerTrack should be rdb->rdb_sectors?! But CylBlocks
   * seems to be better if a hd has more than 1 _LOGICAL_ surface
   * (Does this ever happen? Correct me if I am wrong.. 
   */
  de->de_Reserved        = ntohl(2);
  de->de_PreAlloc        = ntohl(0);
  de->de_Interleave      = ntohl(0);
  de->de_NumBuffers      = ntohl(30);
  de->de_BufMemType      = ntohl(0);
  de->de_MaxTransfer     = ntohl(0x7fffffff);
  de->de_Mask            = ntohl(0xffffffff);
  de->de_BootPri         = ntohl(0);
  de->de_DosType         = ntohl(0x444f5301);
  de->de_Baud            = ntohl(0);
  de->de_Control         = ntohl(0);
  de->de_BootBlocks      = ntohl(0);

  return 0;
}

int partition_delete (int nr)
{
  if ((parts<nr)||(nr<1)) {
    fprintf (stderr,"Could not delete partition %d.\n",nr);
    return -1;
  }

  nr--;

  if (nr==0) {
    sectortable[htonl(rdb->rdb_PartitionList)]=(char) -1;
    if (htonl(rdb->rdb_PartitionList)==(ULONG)lastblock) lastblock--;
    rdb->rdb_PartitionList=pa[0]->pb_Next;
  } else {
    sectortable[htonl(pa[nr-1]->pb_Next)]=(char) -1;
    if (htonl(pa[nr-1]->pb_Next)==(ULONG)lastblock) lastblock--;
    pa[nr-1]->pb_Next=pa[nr]->pb_Next;
  }

  memset(pa[nr],0,SECTOR_SIZE);
  while (nr<parts) {
    pa[nr]=pa[nr+1];
    nr++;
  }
  parts--;
  return 0;
}

int partition_togglebootable(int nr)
{
 if ((nr<1)||(nr>parts)) return -1;
 nr--;
 pa[nr]->pb_Flags=ntohl(htonl(pa[nr]->pb_Flags)^PBFF_BOOTABLE);
 return 0;
}

int partition_togglenomount(int nr)
{
 if ((nr<1)||(nr>parts)) return -1;
 nr--;
 pa[nr]->pb_Flags=ntohl(htonl(pa[nr]->pb_Flags)^PBFF_NOMOUNT);
 return 0;
}

int partition_locyl(int nr,int cyl)
{
 if ((nr<1)||(nr>parts)) return -1;
 nr--;
 ((struct DosEnvec *)(pa[nr]->pb_Environment))->de_LowCyl=ntohl(cyl);
 return 0;
}

int partition_hicyl(int nr,int cyl)
{
 if ((nr<1)||(nr>parts)) return -1;
 nr--;
 ((struct DosEnvec *)pa[nr]->pb_Environment)->de_HighCyl=ntohl(cyl);
 return 0;
}

int partition_bootpri(int nr,int pri)
{
 if ((nr<1)||(nr>parts)) return -1;
 nr--;
 ((struct DosEnvec *)pa[nr]->pb_Environment)->de_BootPri=ntohl(pri);
 return 0;
}

int partition_bootblk(int nr,int blks)
{
 if ((nr<1)||(nr>parts)) return -1;
 nr--;
 ((struct DosEnvec *)pa[nr]->pb_Environment)->de_BootBlocks=ntohl(blks);
 return 0;
}

int partition_dostype(int nr, ULONG dostype)
{
 if ((nr<1)||(nr>parts)) return -1;
 nr--;
 ((struct DosEnvec *)pa[nr]->pb_Environment)->de_DosType=ntohl(dostype);
 return 0;
}
int rigiddisk_correct(void)
{
 /* Correct checksums of all Blocks */
 int i,ts;
 struct DosEnvec *de;

 rdb->rdb_HighRDSKBlock=ntohl(lastblock);

 /* Historic overhead from the very old days. The DosEnvec is a 
    dynamically growing table, and some fields are only present in newer
    versions of AmigaDos. So if we decide to create acient looking RDB's,
    we are ready for it. (If a partion is mounted under AmigaDos, unkown 
    fields should be ignored, and missing fields given a senseful default.)
    NB, AmigaLilo depends on the presense of fields 19 and 20, which are
    only supportted by V37 or newer Roms.
 */

 for (i=firstblock;i<maxblock+1;i++)
   sector_correct((struct AmigaBlock *)(initsectors+i*SECTOR_SIZE));
 connectivity();

 for (i=0;i<parts;i++) {
  de=(struct DosEnvec *)pa[i]->pb_Environment;
  ts=htonl(de->de_TableSize);
  if (ts<12) de->de_BufMemType=ntohl(0);
  if (ts<13) de->de_MaxTransfer=ntohl(0x7fffffff);
  if (ts<14) de->de_Mask=ntohl(0xffffffff);
  if (ts<15) de->de_BootPri=ntohl(0);
  if (ts<16) de->de_DosType=ntohl(0x444f5300);
  if (ts<17) de->de_Baud=ntohl(0);
  if (ts<18) de->de_Control=ntohl(0);
  if (ts<19) {
    de->de_BootBlocks=ntohl(0);
    de->de_TableSize=ntohl(19);
  }
  sector_correct((struct AmigaBlock *)(pa[i]));
 }
 return 0;
}

int rigiddisk_new(int first)
{
  struct hd_geometry geo={-1,};
  long hdsize;
  int i,f;

  if ((f=open(disk_device,O_RDONLY))<0) {
   fprintf (stderr,"Cannot open device %s\n",disk_device);
   return -1;
  }

  printf ("Creating new Rigid Disk Block\n");

  if (ioctl(f,HDIO_GETGEO,&geo)!=0) {
      fprintf(stdout,"Can't get geometry data. Let's try lseek/trivial mapping now.\n");

      hdsize=lseek(f,0,SEEK_END);
      if (hdsize != -1) {
       if (hdsize==0) {
        fprintf(stdout, "LSEEK: Could not lseek (Floppy?). Exit.\n");
        exit(0);
       }

       if ((hdsize%SECTOR_SIZE)!=0) {
        fprintf(stdout, "LSEEK: File length error (corrupt disk image?). Exit.\n");
        exit(0);
       }

       geo.heads=1;    /* Maybe we should use 2 here to assume a floppy disk */
       geo.sectors=1;
       geo.cylinders=hdsize/SECTOR_SIZE;
       geo.start=0;
      } else fprintf (stderr, "LSEEK ERROR!\n");
  }

  printf("geometry: %d heads, %d secs, %d cyl, %ld start\n",
          geo.heads,geo.sectors,geo.cylinders,geo.start);


  i=geo.sectors*geo.heads*2;
  if ((initsectors=malloc(i*SECTOR_SIZE))==NULL) {
    fprintf (stderr,"Not enough main memory for all init sectors.\n");
    return -1;
  }

  if ((sectortable=malloc(i))==NULL) {
    fprintf (stderr,"Not enough main memory for sector table.\n");
    return -1;
  }
 memset(sectortable,-1,i);

 rdb=(struct RigidDiskBlock *)(initsectors+first*SECTOR_SIZE);

 memset(rdb,-1,SECTOR_SIZE);

 rdb->rdb_ID              = ntohl(IDNAME_RIGIDDISK);
 rdb->rdb_SummedLongs     = ntohl(64);
 rdb->rdb_BlockBytes      = ntohl(SECTOR_SIZE);
 rdb->rdb_Flags           = ntohl(0);
 rdb->rdb_Cylinders       = ntohl(geo.cylinders);
 rdb->rdb_Sectors         = ntohl(geo.sectors);
 rdb->rdb_Heads           = ntohl(geo.heads);
 rdb->rdb_Interleave      = ntohl(0);
 rdb->rdb_Park            = ntohl(geo.cylinders);
 rdb->rdb_WritePreComp    = ntohl(geo.cylinders);
 rdb->rdb_ReducedWrite    = ntohl(geo.cylinders);
 rdb->rdb_StepRate        = ntohl(0);

 rdb->rdb_RDBBlocksLo     = ntohl(first);
 rdb->rdb_RDBBlocksHi     = ntohl(geo.sectors*geo.heads*2-1);
 rdb->rdb_LoCylinder      = ntohl(2);
 rdb->rdb_HiCylinder      = ntohl(geo.cylinders-1);
 rdb->rdb_CylBlocks       = ntohl(geo.sectors*geo.heads);
 rdb->rdb_AutoParkSeconds = ntohl(0);
 rdb->rdb_HighRDSKBlock   = ntohl(first);

 firstblock=first;
 lastblock=first;
 minblock=first;
 maxblock=geo.sectors*geo.heads*2-1;
 rigiddisk_correct();
 return 0;
}

int rigiddisk_reorg(int startblk)
{
 /* Put all Blocks together and/or move RDB to another block */
 char *actual, *act2, *newblks;
 int  i,j,block,blk2;
 ULONG *crk, btype;

 i=countblk()+startblk-1; /* Is this correct? */
 if (i>maxblock) {
   fprintf (stderr,"Not enough RDB space to place RDB at Block %d. Have %d, need %d.\n",
            startblk,maxblock,i);
   return -1;
 }

 if ((newblks=malloc(maxblock*SECTOR_SIZE))==NULL) {
   fprintf (stderr,"Not enough main memory for reorganisation.\n");
   return -1;
 }
 
 memset(newblks,-1,maxblock*SECTOR_SIZE); 

 j = (firstblock<startblk)?firstblock:startblk;
 j = (j<2)?j:2;
 if (j>0) fprintf (stdout,"Trying to save %d sector(s) \n",j);

 for (i=0; i<j; i++) {
  btype=htonl(((struct AmigaBlock *)(initsectors+i*SECTOR_SIZE))->amiga_ID);
  if (btype!=IDNAME_RIGIDDISK
   && btype!=IDNAME_BADBLOCK && btype!=IDNAME_FILESYSHEADER
   && btype!=IDNAME_LOADSEG  && btype!=IDNAME_PARTITION) {
   fprintf (stdout,"Raw Copying Block Nr. %d (MBR?)\n",i);
   memcpy  (newblks+i*SECTOR_SIZE,initsectors+i*SECTOR_SIZE,SECTOR_SIZE);
  }
 }

 i=startblk; actual=newblks+i*SECTOR_SIZE;

 memcpy (actual,rdb,SECTOR_SIZE);

 crk=&(RDSK(newblks+startblk*SECTOR_SIZE)->rdb_BadBlockList);
 i++; actual=newblks+i*SECTOR_SIZE;

 block=htonl(rdb->rdb_BadBlockList);

 while (block!=-1) {
   memcpy (actual,initsectors+block*SECTOR_SIZE,SECTOR_SIZE);
   (*crk)=ntohl(i);
   crk=&(BBB(actual)->bbb_Next);
   block=htonl(BBB(initsectors+block*SECTOR_SIZE)->bbb_Next);
   i++; actual=newblks+i*SECTOR_SIZE;
}

 (*crk)=ntohl(-1);
 crk=&(RDSK(newblks+startblk*SECTOR_SIZE)->rdb_DriveInit);
 block=htonl(rdb->rdb_DriveInit);

 while (block!=-1) {
   memcpy (actual,initsectors+block*SECTOR_SIZE,SECTOR_SIZE);
   (*crk)=ntohl(i);
   crk=&(LSEG(actual)->lsb_Next);
   block=htonl(LSEG(initsectors+block*SECTOR_SIZE)->lsb_Next);
   i++; actual=newblks+i*SECTOR_SIZE;
 }

 (*crk)=ntohl(-1);
 crk=&(RDSK(newblks+startblk*SECTOR_SIZE)->rdb_FileSysHeaderList);
 block=htonl(rdb->rdb_FileSysHeaderList);

 while (block!=-1) {
   memcpy (actual,initsectors+block*SECTOR_SIZE,SECTOR_SIZE);
   (*crk)=ntohl(i);
   crk=&(FSHB(actual)->fhb_SegListBlocks);

   act2=actual; 
   i++; actual=newblks+i*SECTOR_SIZE;
 
   blk2=htonl(FSHB(initsectors+block*SECTOR_SIZE)->fhb_SegListBlocks);

   while (blk2!=-1) {
     memcpy (actual,initsectors+blk2*SECTOR_SIZE,SECTOR_SIZE);
     (*crk)=ntohl(i);
     crk=&(LSEG(actual)->lsb_Next);
     blk2=htonl(LSEG(initsectors+blk2*SECTOR_SIZE)->lsb_Next);
     i++; actual=newblks+i*SECTOR_SIZE;
   }

   crk=&(FSHB(act2)->fhb_Next); 
   block=htonl(FSHB(initsectors+block*SECTOR_SIZE)->fhb_Next);
 }
 
 crk=&(RDSK(newblks+startblk*SECTOR_SIZE)->rdb_PartitionList);
 block=htonl(rdb->rdb_PartitionList);

 while (block!=-1) {
   memcpy (actual,initsectors+block*SECTOR_SIZE,SECTOR_SIZE);
   (*crk)=ntohl(i);
   crk=&(PART(actual)->pb_Next);
   block=htonl(PART(initsectors+block*SECTOR_SIZE)->pb_Next);
   i++; actual=newblks+i*SECTOR_SIZE;
 }

 lastblock=i;
 firstblock=startblk;
 minblock=startblk;

 free (initsectors);
 initsectors=newblks;
 rdb=(struct RigidDiskBlock *)(initsectors+firstblock*SECTOR_SIZE);

 memset(sectortable,-1,maxblock);
 rigiddisk_correct();
 return 0;
}

int rigiddisk_save(void)
{
 /* save RDB to disk */
 int f;
 rigiddisk_correct();
 if ((f=open(disk_device,O_WRONLY))<0) {
   fprintf (stderr,"Cannot open device %s\n",disk_device);
   return -1;
 }

 if (lseek(f,0,SEEK_SET)<0) {
   close(f); fprintf (stderr,"RDB seek error.\n");
   return -1;
 }

 if (write(f,initsectors,maxblock*SECTOR_SIZE)!=maxblock*SECTOR_SIZE) {
   close(f);
   fprintf (stderr,"RDB write error.\n");
   return -1;
 }
 return 0;
}

char *DosType(ULONG dostype)
{
 static char type[32];
 union {
 ULONG dlong;
 char dchar[4];
 } dt;
 int i,j=0;

 dt.dlong = htonl(dostype);
 if (!list_only)
 {
   j = sprintf(type, "0x%08lX = ",dostype);
   for (i = 0; i < 4; i++)
     if (isprint(dt.dchar[i]))
       type[j++] = dt.dchar[i];
     else
       j += sprintf(type + j, "\\%o", dt.dchar[i]);
   type[j]=0;
 }

 switch (dt.dlong) { 
   case 0x444f5300:      j += sprintf(type + j, "Amiga OFS");break;
   case 0x444f5301:      j += sprintf(type + j, "Amiga FFS");break;
   case 0x444f5302:      j += sprintf(type + j, "Amiga OFS Int.");break;
   case 0x444f5303:      j += sprintf(type + j, "Amiga FFS Int.");break;
   case 0x444f5304:      j += sprintf(type + j, "Amiga OFS DirCache");break;
   case 0x444f5305:      j += sprintf(type + j, "Amiga FFS DirCache");break;
   case 0x4C4E5800:      j += sprintf(type + j, "Linux native");break;
   case 0x53575000:      j += sprintf(type + j, "Linux swap");break;
   default:              j += sprintf(type + j, "[unknown]");break;
 }

 return type;
}

int rigiddisk_show(void)
{
 fprintf (stdout,"Disk %s: %ld heads, %ld sectors, %ld cylinders, RDB: %d\n", disk_device,
          htonl(rdb->rdb_Heads),htonl(rdb->rdb_Sectors),htonl(rdb->rdb_Cylinders),firstblock);
 fprintf (stdout,"Logical Cylinders from %ld to %ld, %d  bytes/Cylinder \n\n", 
          htonl(rdb->rdb_LoCylinder), htonl(rdb->rdb_HiCylinder), SECTOR_SIZE);

 return 0;
}

int partition_show(void)
{
 int i;
 struct DosEnvec *de;

 fprintf (stdout,"   Device  Boot Mount   Begin      End     Size   Pri  BBlks    System\n");
 for (i=0;i<parts;i++) {
   de=(struct DosEnvec *)pa[i]->pb_Environment;
   fprintf (stdout, "%s%-2d   ",disk_device,(i+1));
   fprintf (stdout, "%s    ", htonl(pa[i]->pb_Flags)&PBFF_BOOTABLE?"*":" ");
   fprintf (stdout, "%s    ", htonl(pa[i]->pb_Flags)&PBFF_NOMOUNT?" ":"*");
   fprintf (stdout, "%6ld   %6ld   ", htonl(de->de_LowCyl),htonl(de->de_HighCyl));
   fprintf (stdout, "%6ld   ", (htonl(de->de_HighCyl)-htonl(de->de_LowCyl)+1)*
                            htonl(de->de_BlocksPerTrack)*htonl(de->de_Surfaces)/2);
   fprintf (stdout, "%3ld   ",htonl(de->de_BootPri));
   fprintf (stdout, "%3ld  ",htonl(de->de_BootBlocks));
   fprintf (stdout, "%s\n",DosType(htonl(de->de_DosType)));
 }
 return 0;
}

int quitall(void)
{
 free (initsectors); free (sectortable);
 return 0;
}

int pt(void)
{
 unsigned int i;
 for (i=0;i<htonl(rdb->rdb_RDBBlocksHi)+1;i++)
  fprintf (stdout,"%d",i%10);printf("\n");
 for (i=0;i<htonl(rdb->rdb_RDBBlocksHi)+1;i++)
  fprintf (stdout,"%s",sectortable[i]=='x'?"x":" ");
 fprintf (stdout,"!\n");
 return 0;
}
