/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include "loader.h"
#include "loaderlib.h"

/*--------------------------------------------------------------------------*/

extern const FILEMAP * find_file_map(const char *path);

/*--------------------------------------------------------------------------*/

typedef struct memfrag_str
{
	unsigned long		size; 		/* size of free block including header  */
	struct memfrag_str	*next;		/* pointer to next memory fragment		*/

} memfrag_t;

/*--------------------------------------------------------------------------*/

static memfrag_t		*free_list = NULL;	/* ptr to first free mem frag	*/

/*--------------------------------------------------------------------------*/
/* initialise memory allocator
 */

void
mem_alloc_init
(	void			*heap_base,
	unsigned long	heap_size
)
{
	if (free_list == NULL)
	{
		free_list       = heap_base;
		free_list->size = heap_size;
		free_list->next = NULL;
	}
}

/*--------------------------------------------------------------------------*/

void *
mem_alloc
(	unsigned long	size
)
{	memfrag_t		*frag;
	memfrag_t		*last;

	/* long word align and add space to store allocation size */
	size = (size + sizeof(frag->size) + 3) & ~3;

	frag = free_list;
	last = NULL;

	while (frag)
	{
		if (frag->size >= size)
		{
			goto frag_found;
		}

		/* onwards ... */
		last = frag;
		frag = frag->next;
	}

	/* nothing found! */
	return NULL;

	/* A suitable fragment has been found
	 * now we must unlink it from the free list.
	 * If the fragment is large enough just take
	 * the allocation from the end.
	 */
frag_found:
	/* is it big enough to split into two fragments */
	if (frag->size >= (size + sizeof(memfrag_t)))
	{
		/* adjust for amount removed */
		frag->size -= size;

		/* point at allocated chunk at end */
		frag = (memfrag_t *) ((char *)frag + frag->size);

		/* store allocated size */
		frag->size = size;
	}
	else
	{
		/* unlink found fragment */
		if (last)
		{
			last->next = frag->next;
		}
		else
		{
			free_list  = frag->next;
		}
	}

	/* zero intialise allocated memory */
	mem_clear(&(frag->next), frag->size - sizeof(frag->size));

	/* return pointer to allocated memory */
	return &(frag->next);
}

/*--------------------------------------------------------------------------*/
/* return size of largest free memory block
 */

#ifdef NOT_USED

unsigned long
mem_maxalloc
(	void
)
{	memfrag_t		*frag;
	unsigned long	size = 0;

	/* find size of largest fragment */
	for (frag = free_list; frag; frag = frag->next)
	{
		if (frag->size > size)
		{
			size = frag->size;
		}
	}

	/* anything found? */
	if (size)
	{
		/* adjust for allocator overhead */
		size -= sizeof(frag->size);
	}

	/* return size available */
	return size;
}

#endif

/*--------------------------------------------------------------------------*/
/* resize block of allocated memory
 * (can only handle size reduction)
 */

#ifdef NOT_USED

int
mem_resize
(	void			*memaddr,
	unsigned long	newsize
)
{	memfrag_t		*frag;
	memfrag_t		*newfrag;

	/* recover pointer to base of allocation */
	frag = (memfrag_t *) ((char *)memaddr - sizeof(frag->size));

	/* long word align and add space to store allocation size */
	newsize = (newsize + sizeof(frag->size) + 3) & ~3;

	/* do they want more than we have */
	if (newsize > frag->size)
	{
		return -1;
	}

	/* is there enough to make a new fragment from remainder */
	if ((frag->size - newsize) >= sizeof(memfrag_t))
	{
		/* make new fragment from end of old */
		newfrag       = (memfrag_t *)((char *)frag + newsize);
		newfrag->size = frag->size - newsize; 

		/* give fragment back to heap */
		mem_free(&(newfrag->next));

		/* remember new fragment size */
		frag->size = newsize;
	}

	return 0;
}

#endif

/*--------------------------------------------------------------------------*/

int
mem_free
(	void		*memaddr
)
{	memfrag_t	*last;
	memfrag_t	*frag;
	memfrag_t	*dead;

	if (memaddr == NULL)
	{
		return -1;
	}

	/* recover pointer to base of allocation */
	dead = (memfrag_t *) ((char *)memaddr - sizeof(frag->size));

	frag = free_list;
	last = NULL;

	/* find fragment that will follow us */
	while (frag && (frag < dead))
	{
		last = frag;
		frag = frag->next;
	}

	/* make sure it doesn't overlap previous fragment */
	if (last)
	{
		if ((char *)dead < ((char *)last + last->size))
		{
			return -1;
		}
	}

	/* make sure it doesn't overlap next fragment */
	if (frag)
	{
		if (((char *)dead + dead->size) > (char *)frag)
		{
			return -1;
		}
	}

	/* can we merge with next fragment */
	if (((char *)dead + dead->size) == (char *)frag)
	{
		/* merge with next fragment */
		dead->next  = frag->next;
		dead->size += frag->size;
	}
	else
	{
		/* link to next */
		dead->next = frag;
	}

	/* if there is a previous fragment */
	if (last)
	{
		/* can we merge with previous fragment */
		if (((char *)last + last->size) == (char *)dead)
		{
			/* merge with previous fragment */
			last->next  = dead->next;
			last->size += dead->size;
		}
		else
		{
			/* just link to previous fragment */
			last->next  = dead;
		}
	}

	/* no previous fragment */
	else
	{
		/* make new head of list */
		free_list = dead;
	}

	return 0;
}

/*--------------------------------------------------------------------------*/

unsigned long
disable_icache
(	void
)
{	unsigned long	old_cacr;

	/* disable and invalidate instruction cache */
	__asm__ __volatile__ (
	"	movec	%%cacr,%%d1
		move.l	%%d1,%0
		bclr	#15,%%d1
		movec	%%d1,%%cacr
		.word	0xf498			| cinva	ic
	"
	: "=d" (old_cacr)
	: /* no inputs */
	: "d1"
	);

	return old_cacr;
}

/*--------------------------------------------------------------------------*/

unsigned long
enable_icache
(	void
)
{	unsigned long	old_cacr;

	/* enable instruction cache */
	__asm__ __volatile__ (
	"	movec	%%cacr,%%d1
		move.l	%%d1,%0
		bset	#15,%%d1
		movec	%%d1,%%cacr
	"
	: "=d" (old_cacr)
	: /* no inputs */
	: "d1"
	);

	return old_cacr;
}

/*--------------------------------------------------------------------------*/

void
invalidate_icache
(	void
)
{
	__asm__ __volatile__ (
	"	.word	0xf498		| cinva ic - invalidate instruction cache
	"
	);
}

/*--------------------------------------------------------------------------*/

int
str_len
(	const char	*s
)
{	const char	*e = s;

	while (*e++)
		;

	return (e - s) - 1;
}

/*--------------------------------------------------------------------------*/

void
str_cpy
(	char		*d,
	const char	*s
)
{
	while ((*d++ = *s++) != '\0')
		;
}

/*--------------------------------------------------------------------------*/

void
str_cat
(	char		*d,
	const char	*s
)
{
	while (*d++ != '\0')
		;

	d--;

	while ((*d++ = *s++) != '\0')
		;
}

/*--------------------------------------------------------------------------*/

void
str_ncat
(	char			*d,
	const char		*s,
	unsigned long	n
)
{
	while (n && (*d++ != '\0'))
	{
		n--;
	}

	d--;

	while (n-- && *s)
	{
		*d++ = *s++;
	}

	*d = '\0';
}

/*--------------------------------------------------------------------------*/

void
str_ncpy
(	char			*d,
	const char		*s,
	unsigned long	n
)
{	const char		*save;
	unsigned long	max;

	save = s;
	max  = 1;

	while (*s++) max++;

	s = save;

	if (n < max) max = n;

	n -= max;

	while (max--)
	{
		*d++ = *s++;
	}

	while (n--)
	{
		*d++ = '\0';
	}
}

/*--------------------------------------------------------------------------*/
/* Check whether two strings exist and second is a prefix of the first
 * while ignoring case.
 */

int
prefix_string
(	const char	*s,						/* string							*/
	const char	*p						/* prefix							*/
)
{	char		sc;
	char		pc;

	if ((s == NULL) || (p == NULL))
	{
		return FALSE;
	}

	/* until we reach end of prefix */
	while (*p)
	{
		sc = *s++;
		pc = *p++;

		/* convert to lower case */
		if ((sc >= 'A') && (sc <= 'Z')) sc += 'a' - 'A';
		if ((pc >= 'A') && (pc <= 'Z')) pc += 'a' - 'A';

		if (sc != pc)
		{
			return FALSE;
		}
	}

	/* make sure we are either at end of source or in white space */
	if (*s >= '!')
	{
		return FALSE;
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Check whether two strings exist and are equal ignoring case.
 */

int
equal_strings
(	const char	*s1,
	const char	*s2
)
{	char		c1;
	char		c2;

	if ((s1 == NULL) || (s2 == NULL))
	{
		return FALSE;
	}

	while (*s1 || *s2)
	{
		c1 = *s1++;
		c2 = *s2++;

		/* convert to lower case */
		if ((c1 >= 'A') && (c1 <= 'Z')) c1 += 'a' - 'A';
		if ((c2 >= 'A') && (c2 <= 'Z')) c2 += 'a' - 'A';

		if (c1 != c2)
		{
			return FALSE;
		}
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Check whether two strings exist and are equal including case
 */

int
case_equal_strings
(	const char	*s1,
	const char	*s2
)
{
	if ((s1 == NULL) || (s2 == NULL))
	{
		return FALSE;
	}

	while (*s1 || *s2)
	{
		if (*s1++ != *s2++)
		{
			return FALSE;
		}
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/

int
file_size
(	const char		*path
)
{	const FILEMAP	*map;

	if ((map = find_file_map(path)) == NULL)
	{
		return -1;
	}

	return MAP_FILESIZE(map);
}

/*--------------------------------------------------------------------------*/

static unsigned	pcnt_current;
static unsigned	pcnt_total;
static int		pcnt_backspaces;

void
percent_init
(	unsigned long	total
)
{
	pcnt_total      = total;
	pcnt_current    = 999;
	pcnt_backspaces = 0;
}

/*--------------------------------------------------------------------------*/

void
percent_term
(	const char	*msg
)
{
	/* remove old percentage */
	while (pcnt_backspaces--)
		put_str("\b \b");

	put_str(msg);
}

/*--------------------------------------------------------------------------*/

void
percent_show
(	unsigned long	current
)
{	unsigned 		percentage;

	/* calculate new percentage */
	percentage = current * 100 / pcnt_total;

	/* if percentage changed */
	if (percentage != pcnt_current)
	{
		/* remove old percentage */
		while (pcnt_backspaces--)
			put_char('\b');

		/* output new percentage */
		Printf("%ld%% %n", percentage, &pcnt_backspaces);

		/* remember it for next time */
		pcnt_current = percentage;
	}
}

/*-----------------------------< end of file >------------------------------*/
