/*
  copyright : GPL
    (author: joost witteveen, joostje@debian.org)

 */


#include <dlfcn.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/sem.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "fakeroot.h"

#define LIBC_HACK

#define S_ISTXT S_ISVTX

/*libfakeroot debuging output is only generated if *both* debug=1 here, and
  debug=1 in the fakeroot.c "main" programme (commandline argument -d)*/
const static int debug=0;

static int msg_get=-1, msg_snd=-1, sem_id=-1;

#define DEBUG(A)  {char b[MAXPATH];sprintf(b,A); wdebug(b);}
#define DEBUG2(A,B)  {char b[MAXPATH];sprintf(b,A,B); wdebug(b);}
#define DEBUG3(A,B,C)  {char b[MAXPATH];sprintf(b,A,B,C); wdebug(b);}
#define DEBUG4(A,B,C,D)  {char b[MAXPATH];sprintf(b,A,B,C,D); wdebug(b);}
void wdebug(const char *);
/*
  Note that this code contains a few rather bad hacks to
  get atleast libc5 code executed. I hope to remove those
  when  most of the main distribution is libc6.
 */


/*********************************/
/*                               */
/* storage for the old functions */
/*                               */
/*********************************/

void *libc_handle(){
  /*static void *handle=0;
  
  if(handle==NULL)
    handle = dlopen("/lib/libc.so.6", RTLD_LAZY);
    */
  return RTLD_NEXT;//handle;
}

int old_chown(const char *path, uid_t owner, gid_t group){
  static int (*fptr)(const char *, uid_t, gid_t)=0;
  
  if(fptr==NULL)
    fptr = (int (*)(const char *, uid_t, gid_t))dlsym(libc_handle(), "chown");
  
  return (*fptr)(path,owner,group);
}
int old_fchown(int fd, uid_t owner, gid_t group){
  static int (*fptr)(int, uid_t, gid_t)=0;
  
  if(fptr==NULL)
    fptr = (int (*)(int, uid_t, gid_t))dlsym(libc_handle(), "fchown");
  
  return (*fptr)(fd,owner,group);
}
int old_chmod(const char *path, mode_t mode){
  static int  (*fptr)(const char *, mode_t mode)=0;

  if(fptr==NULL)
    fptr = (int (*)(const char *,mode_t))dlsym(libc_handle(), "chmod");
  return (*fptr)(path,mode);
}
int old_fchmod(int fd, mode_t mode){
  static int  (*fptr)(int, mode_t mode)=0;

  if(fptr==NULL)
    fptr = (int (*)(int,mode_t))dlsym(libc_handle(), "fchmod");
  return (*fptr)(fd,mode);
}
int old_mkdir(const char *path, mode_t mode){
  static int  (*fptr)(const char *, mode_t mode)=0;

  if(fptr==NULL)
    fptr = (int (*)(const char *,mode_t))dlsym(libc_handle(), "mkdir");
  return (*fptr)(path,mode);
}
int old_xstat(int ver, const char *file_name, struct stat *buf){
  static  (*fptr)(int ver, const char *file_name, struct stat *buf)=0;
  
  if(fptr==NULL)
    fptr = (int (*)(int ver, const char *file_name, struct stat *buf))
      dlsym(libc_handle(), "__xstat");

  return (*fptr)(ver, file_name,buf);
}
int old_lxstat(int ver, const char *file_name, struct stat *buf){
  static  (*fptr)(int ver, const char *file_name, struct stat *buf)=0;
  
  if(fptr==NULL)
    fptr = (int (*)(int ver, const char *file_name, struct stat *buf))
      dlsym(libc_handle(), "__lxstat");

  return (*fptr)(ver, file_name,buf);
}
int old_fxstat(int ver, int fd, struct stat *buf){
  static  (*fptr)(int ver, int fd, struct stat *buf)=0;
  
  if(fptr==NULL)
    fptr = (int (*)(int ver, int tf, struct stat *buf))
      dlsym(libc_handle(), "__fxstat");

  return (*fptr)(ver, fd, buf);
}
int old_unlink(const char *pathname){
  static int  (*fptr)(const char *)=0;

  if(fptr==NULL)
    fptr = (int (*)(const char *))dlsym(libc_handle(), "unlink");
  return (*fptr)(pathname);
}
int old_rmdir(const char *pathname){
  static int  (*fptr)(const char *)=0;

  if(fptr==NULL)
    fptr = (int (*)(const char *))dlsym(libc_handle(), "rmdir");
  return (*fptr)(pathname);
}
int old_remove(const char *pathname){
  static int  (*fptr)(const char *)=0;

  if(fptr==NULL)
    fptr = (int (*)(const char *))dlsym(libc_handle(), "remove");
  return (*fptr)(pathname);
}
int old_rename(const char *newpath, const char *oldpath){
  static int  (*fptr)(const char *, const char *)=0;

  if(fptr==NULL)
    fptr = (int (*)(const char *, const char *))dlsym(libc_handle(), "rename");
  return (*fptr)(newpath, oldpath);
}
int old_execve(const char *filename,  char *const argv[], char * const envp[]){
  static int  (*fptr)(const char *filename, char *const argv[],
		      char *const envp[])=0;
  
  if(fptr==NULL)
    fptr = (int (*)(const char *filename, 
		    char *const argv [],
		    char *const envp[]))dlsym(libc_handle(), "execve");
  return (*fptr)(filename,argv,envp);
}
int old_execvp(const char *file, char *const argv[]){
  static int (*fptr)(const char *file, char *const argv[])=0;

  if(fptr==NULL)
    fptr = (int (*)(const char *file, char *const argv[]))
      dlsym(libc_handle(), "execvp");
  return (*fptr)(file,argv);
}
int old_exect(const char *path, char *const argv[]){
  static int (*fptr)(const char *path, char *const argv[])=0;
  if(fptr==NULL)
    fptr = (int (*)(const char *path, char *const argv[]))
      dlsym(libc_handle(), "execvt");
  return (*fptr)(path,argv);
}
int old_execv(const char *path, char *const argv[]){
  static int (*fptr)(const char *path, char *const argv[])=0;

  if(fptr==NULL)
    fptr = (int (*)(const char *path, char *const argv[]))
      dlsym(libc_handle(), "execv");
  return (*fptr)(path,argv);
}
FILE *old_popen(const char *command, const char *type){
  static FILE* (*fptr)(const char *command, const char *type)=0;
  if(fptr==NULL)
    fptr = (FILE * (*)(const char *command, const char *type))
      dlsym(libc_handle(), "popen");
  return (*fptr)(command,type);
}


/*********************************/
/*                               */
/* communication with daemon     */
/*                               */
/*********************************/


void set_func(struct my_msgbuf *buf,func_id id){
  ((struct q_and_a_func_id *)&(buf->mtext))->id=id;
};

key_t get_ipc_key(){
  char *s;
  s=getenv(FAKEROOTKEY_ENV);
  if(s==NULL){
    fprintf(stderr,"variable "FAKEROOTKEY_ENV" undefined -- probably not running from within fakeroot?\n");
    exit(1);
  }
  return atoi(s);
}
void init_get_msg(){
  /* a msgget call generates a fstat() call. As fstat() is wrapped,
     that call will in turn call semaphone_up(). So, before 
     the semaphones are setup, we should make sure we already have
     the msg_get and msg_set id.
     This is why semaphone_up() calls this function.*/

  if(msg_snd==-1){
    msg_snd=msgget(get_ipc_key(),IPC_CREAT|0600);
    DEBUG2("sendmsg msg=%i", msg_snd);
  }

  if(msg_get==-1){
    msg_get=msgget(get_ipc_key()+1,IPC_CREAT|0600);
    DEBUG2("get_stat msg=%i", msg_get);
  }
}

void semaphone_up(){
  struct sembuf op;
  if(sem_id==-1)
    sem_id=semget(get_ipc_key()+2,1,IPC_CREAT|0600);
  op.sem_num=0;
  op.sem_op=-1;
  op.sem_flg=SEM_UNDO;
  init_get_msg();
  semop(sem_id,&op,1);
  DEBUG2("SEM UP (%i)",sem_id);
}
void semaphone_down(){
  struct sembuf op;
  if(sem_id==-1)
    sem_id=semget(get_ipc_key()+2,1,IPC_CREAT|0600);
  DEBUG2("SEM DOWN (%i)",sem_id);
  op.sem_num=0;
  op.sem_op=1;
  op.sem_flg=SEM_UNDO;
  semop(sem_id,&op,1);
}


void  send_msg_size(struct my_msgbuf *buf, int l){
  int r;
  
  init_get_msg();
  buf->mtype=1;
  r=msgsnd(msg_snd,buf, sizeof(struct q_and_a_func_id)+l, 0);
  if(r==-1)
    perror("libfakeroot, when sending message");	 
}

void send_msg(struct my_msgbuf *buf){
  send_msg_size(buf,strlen(buf->mtext)+1);
}

void send_msg_par(struct my_msgbuf *buf, int l){
  char *s;
  int sizepar;
  sizepar=l-MAXPATH;
  s=buf->mtext+sizepar+sizeof(struct q_and_a_func_id);
  send_msg_size(buf,sizepar+l+strlen(s)+1);
}

int get_msg(struct my_msgbuf *buf, int length){
  int r;

  /*  DEBUG("get_msg, begin");*/
  init_get_msg();
  /*  DEBUG("get_msg, mid");*/
  r=msgrcv(msg_get,buf,length,0,0);
  if(r==-1){
    fprintf(stderr, "errno=%i ",errno);
    perror("libfakeroot, when recieving message");	 
  }

  /*  DEBUG("get_msg, end");*/
  return r;
}

void wdebug(const char *msg){
  struct my_msgbuf buf;
  struct debug_param *par;

  if(debug){
    par=(struct debug_param *)(buf.mtext+sizeof(struct q_and_a_func_id));
    
    ((struct q_and_a_func_id*)buf.mtext)->id=debug_func;
    strcpy(par->msg,msg);
    par->pid=getpid();
    send_msg_par(&buf,sizeof(struct debug_param));
  }
}

void q_and_a_size(struct my_msgbuf *buf, int buf_length,
		 void * ret, int ret_length){
  struct q_and_a_func_id *ser;
  static int serial=0;
  int l;

  ret_length+=sizeof(struct q_and_a_func_id);
  
  
  serial++;
  ser=(struct q_and_a_func_id *)&(buf->mtext);
  ser->pid=getpid();
  ser->serial=serial;


  semaphone_up();  
  send_msg_size(buf,buf_length);
  do{
    l=get_msg(buf, ret_length);
    /* DEBUG3("check_ser =(%i,%i)",ser->serial, ser->pid);
       DEBUG3("check_ser =(%i,%i)",serial, getpid());*/
  }while((ser->serial!=serial)||ser->pid!=getpid());
  semaphone_down();

  
  if(l!=ret_length)
    DEBUG3("******************* q_and_a_size: **** ret_length=%i != l=%i",
	   ret_length,l);
  memcpy(ret,buf->mtext+sizeof(struct q_and_a_func_id),
	 ret_length-sizeof(struct q_and_a_func_id));
}


void q_and_a_stat(struct my_msgbuf *buf, int buf_length,
		 struct stat *st){
  q_and_a_size(buf,buf_length,(void *)st,sizeof(*st));
}
 

#ifdef LIBC_HACK

void remove_fake(char **environ, const char *file){
  char **e;
  static char *buf="";

  DEBUG2("fakeroot: Warning: about to execute %s, a libc5 binary"
	    "fakeroot: I will do this without using libfakeroot (that's libc6)",
	    file);
  for(e=environ; *e; e++){
    if(!strncmp(*e,"LD_PRELOAD=",strlen("LD_PRELOAD="))){
      *e=buf;
    }
  }
}
int check_libc(const char *path){
  /* check what libc the binary in path uses. return 0 for "OK", 1 for
     different libc, and -1 for error.
  
     I could do the checking itself here, but then I'd have to
     get rid of the wrapper functions and all that. It's much easier
     to do in the "main" fakeroot, so I'll do it via the messages*/
  struct my_msgbuf buf;
  struct checklibc_param *par;

  par=(struct checklibc_param *)get_par(&buf);

  set_func(&buf,checklibc_func);

  strcpy(par->path,path);
  DEBUG2("check_libc %s",path);

  
  q_and_a_size(&buf,
	       sizeof(struct checklibc_param)-
	         MAXPATH+strlen(path)+1,
	       (void *)(buf.mtext), 
	       sizeof(par->check));

  DEBUG2("check_libc =%i",par->check);
  return(par->check);
}

int debianpedantic(){
  static int debian=-1;
  struct my_msgbuf buf;
  struct debianpedantic_param *par;
  
  if(debian==-1){
    
    par=(struct debianpedantic_param *)get_par(&buf);
    set_func(&buf, debian_pedantic_func);
    
    q_and_a_size(&buf,sizeof(struct debianpedantic_param),
		 (void *)(buf.mtext),
		 sizeof(struct debianpedantic_param));
    debian=par->debian;
  }
  DEBUG2("debianpedantic = %i ", debian);
  return debian;
}
#endif

/*********************************/
/*                               */
/* the 'fake' wrappers           */
/*                               */
/*********************************/

void send_chown(uid_t owner, gid_t group, struct stat *st){
  struct chown_param *par;
  struct my_msgbuf buf;

  set_func(&buf,chown_func);
  par=(struct chown_param*)get_par(&buf);

  par->owner=owner;
  par->group=group;
  memcpy(&(par->st), st, sizeof(struct stat));

  send_msg_size(&buf,sizeof(struct chown_param));
}

int chown(const char *path, uid_t owner, gid_t group){
  struct stat st;
  int r;
  
  DEBUG4("chown called path=%s, uid=%i, gid=%i",path,owner,group);

  /*chown(sym-link) works on the symlink itself, use lstat: */
  r=old_lxstat(_STAT_VER,path, &st);

  if(r)
    return r;

  send_chown(owner, group, &st);
  r=old_chown(path,owner,group);
  
  if(r&&(errno==EPERM))
    r=0;
  
  return r;
}

int fchown(int fd, uid_t owner, gid_t group){
  struct stat st;
  int r;
  
  DEBUG4("fchown called fd=%i, uid=%i, gid=%i",fd,owner,group);

  r=old_fxstat(_STAT_VER,fd,&st);
  
  if(r)
    return r;

  send_chown(owner, group, &st);
  
  r=old_fchown(fd,owner,group);
  
  if(r&&(errno==EPERM))
    r=0;
  
  return r;
}

void send_chmod(mode_t mode, struct stat *st){
  struct chmod_param *par;
  struct my_msgbuf buf;
  /*dont clear any of the 0700 mode bits, because of the following:
      touch file; chmod 000 file; echo "foo" > file
    this works for root, but doesn't work for ordinary users (or fakeroot).
    by not resetting the 0700 mode bits, (executable for dirs), we work
    around that (we do ofcource reset them in the "fake" data)
    */

  set_func(&buf,chmod_func);
  par=(struct chmod_param*)get_par(&buf);
  par->mode=mode; 

  memcpy(&(par->st), st, sizeof(struct stat));

  send_msg_size(&buf,sizeof(struct chmod_param));
  
}
int debian_modedeny(struct stat *st, mode_t mode){
  /* see debian Policy manual 3.3.8 */
  
  if(S_ISDIR(st->st_mode)){
    if((mode==0755)||(mode==02775))
      return 0;          
  } else if((st->st_mode&0107777)!=st->st_mode)
    return 0;
  else if ((mode&0644)==0644)
    return 0;

  return 1;
}
int chmod(const char *path, mode_t mode){
  struct stat st;
  int r;

  DEBUG3("chown called path=%s, mode%i",
	   path,mode);

  /*chmod(sym-link) works on the file pointed to, use stat: */
  r=old_xstat(_STAT_VER,path,&st);

  if(r)
    return r;
  if(debianpedantic())
    if(debian_modedeny(&st,mode)){
      errno=EPERM;
      return -1;
    }
  send_chmod((mode&ALLPERMS)|(st.st_mode&~ALLPERMS), &st);

  mode |= 0600;
  if(S_ISDIR(st.st_mode))
    mode |= 0100;

  r=old_chmod(path, mode);
  if(r&&(errno==EPERM))
    r=0;
  return r;
}

int fchmod(int fd, mode_t mode){
  int r;
  struct stat st;

  DEBUG3("fchown called fd=%i, mode%i",
	   fd,mode);
  r=old_fxstat(_STAT_VER,fd,&st);

  if(r)
    return(r);
  if(debianpedantic())
    if(debian_modedeny(&st,mode)){
      errno=EPERM;
      return -1;
    }
  
  send_chmod((mode&ALLPERMS)|(st.st_mode&~ALLPERMS), &st);  

  mode |= 0600;
  if(S_ISDIR(st.st_mode))
    mode |= 0100;

  r=old_fchmod(fd, mode);
  if(r&&(errno==EPERM))
    r=0;
  return r;

}

int __xmknod(int ver, const char *pathname, 
	     mode_t mode, dev_t *dev){
  struct my_msgbuf buf;
  int mask=umask(022);
  int fd,r;
  struct mknod_param *par;

  umask(mask);
  
  DEBUG4("mknod called path=%s, mode=%i, dev=%Li",
	   pathname, mode, *dev);

  /*Don't bother to mknod the file, that probably doesn't work.
    just create it as normal file, and leave the premissions
    to the fakemode.*/
  fd=open(pathname, O_WRONLY|O_CREAT|O_TRUNC, 0644);
  if(fd>=0){
    r=0;
    close(fd);
  }else
    r=-1;

  set_func(&buf,mknod_func);
  par=(struct mknod_param*)get_par(&buf);

  par->mode=mode & ~mask;
  par->dev=*dev;
  r=old_lxstat(_STAT_VER,pathname,&par->st);
  if(r)
    return r;

  send_msg_size(&buf,sizeof(struct mknod_param));

  return r;
}

int mkdir(const char *path, mode_t mode){
  struct stat st;
  int r;

  DEBUG3("mkdir called path=%s, mode%i",
	   path,mode);

  r=old_mkdir(path, mode|0700);

  if(r)
    return r;

  r=old_xstat(_STAT_VER,path,&st);

  if(r)
    return r;

  if(debianpedantic())
    if(debian_modedeny(&st,mode)){
      errno=EPERM;
      return -1;
    }
  send_chmod((mode&ALLPERMS)|(st.st_mode&~ALLPERMS)|S_IFDIR,&st);

  if(r&&(errno==EPERM))
    r=0;
  return r;
}

/***************************/
/*           stat          */
/***************************/


void send_stat(int ver, struct stat *st){
  struct stat_param *par;
  struct my_msgbuf buf;

  init_get_msg(); /*see init_get_msg for comment*/

  set_func(&buf,stat_func);
  par=(struct stat_param*)get_par(&buf);

  memcpy(&(par->st), st, sizeof(struct stat));

  q_and_a_stat(&buf,sizeof(struct stat_param), st);
  DEBUG3("After send_stat: uid=%i, ino=%li\n",st->st_uid, st->st_ino);
}

int __lxstat(int ver, const char *file_name, struct stat *statbuf){
  int r;

  DEBUG2("lstat called with filename=%s",file_name);

  r=old_lxstat(ver, file_name,statbuf);

  if(r)
    return r;
  
  send_stat(ver,statbuf);
  return 0;
}
int __xstat(int ver, const char *file_name, struct stat *statbuf){
  int r;

  DEBUG2("lstat called with filename=%s",file_name);

  r=old_xstat(ver, file_name,statbuf);

  if(r)
    return r;
  
  send_stat(ver,statbuf);
  return 0;
}

int __fxstat(int ver, int fd, struct stat *statbuf){
  int r;
  
  DEBUG2("fstat called with fd=%i",fd);

  r=old_fxstat(ver, fd, statbuf);

  if(r)
    return r;
  
  send_stat(ver,statbuf);
  return 0;
}

/***************************/
/*        remove           */
/***************************/


void send_remove(const struct stat *st){
  struct my_msgbuf buf;
  struct unlink_param *par;
  
  set_func(&buf,unlink_func);
  par=(struct unlink_param*)get_par(&buf);
  memcpy(&par->st, st, sizeof(*st));
  
  send_msg_size(&buf,sizeof(struct unlink_param));
}

int unlink(const char *pathname){
  int r,s;
  struct stat st;

  DEBUG2("unlink called path=%s",pathname);

  s=old_lxstat(_STAT_VER,pathname,&st);

  r=old_unlink(pathname);  

  if((r==0)&&(s==0))
    send_remove(&st);

  return r;
}

int rmdir(const char *pathname){
  int r,s;
  struct stat st;

  DEBUG2("rmdir called path=%s",pathname);

  s=old_lxstat(_STAT_VER,pathname,&st);

  r=old_rmdir(pathname);  
  if((r==0)&&(s==0))
    send_remove(&st);  

  return r;
}

int remove(const char *pathname){
  int r, s;
  struct stat st;

  DEBUG2("remove called path=%s",pathname);

  s=old_lxstat(_STAT_VER,pathname,&st);
  r=old_remove(pathname);  
  if((r==0)&&(s==0))
    send_remove(&st);

  return r;
}

int rename(const char *oldpath, const char *newpath){
  int r,s;
  struct stat st;

  s=old_lxstat(_STAT_VER,newpath,&st);

  r=old_rename(oldpath, newpath);
  if((r==0)&&(s==0))
    send_remove(&st);

  return r;
}

/***************************/
/*           exec          */
/***************************/

int execve(const char *filename, char *const argv[],
	   char *const envp[]){
  int r;

#ifdef LIBC_HACK
  if(check_libc(filename)==1)
    remove_fake((char **)envp,filename);
#endif

  DEBUG2("execve %s",filename);
  r=old_execve(filename,argv,envp);
  return r;
}
/*
int execl( const char *path, const char *arg, ...){
  if(debug)
  fprintf(stderr,"!!libfakeroot: execl %s", path);
}
int execlp( const char *file, const char *arg, ...){
  if(debug)
  fprintf(stderr,"!!libfakeroot: execlp %s", file);
}
int execle(const char *path, const char *arg, ...){
  int r;

  if(debug)
  fprintf(stderr,"!!libfakeroot: execle %s", path);
}
*/
int exect(const char *path, char *const argv[]){
  int r;
#ifdef LIBC_HACK
  if(check_libc(path)==1)
    remove_fake(__environ,path);
#endif

  DEBUG2("exect %s",path);
  r=old_exect(path,argv);
  return r;
}
int execv( const char *path, char *const argv[]){
  int r;
  
#ifdef LIBC_HACK
  if(check_libc(path)==1)
    remove_fake(__environ,path);
#endif

  DEBUG2("execv %s",path);
  r=old_execv(path,argv);
  return r;
}
int execvp( const char *file, char *const argv[]){
  int r;

#ifdef LIBC_HACK
  if(check_libc(file)==1)
    remove_fake(__environ,file);
#endif

  DEBUG2("execvp %s",file);
  r=old_execvp(file,argv);
  return r;
}

/* the next ones are the most fun: */
uid_t geteuid(void){
  return 0;
}
uid_t getuid(void){
  return 0;
}
uid_t getegid(void){
  return 0;
}
uid_t getgid(void){
  return 0;
}


/* for communication with fakedebug programme */

void fakeroot_debug(){
  struct my_msgbuf buf;
  
  set_func(&buf,debugdata_func);
  send_msg_size(&buf,0);
};
