
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * The Accelerator Keys dialog for selecting an accelerator key.
 */

#include <string.h>

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "keys.h"
#include "glade_keys_dialog.h"

static void glade_keys_dialog_class_init   (GladeKeysDialogClass  *klass);
static void glade_keys_dialog_init         (GladeKeysDialog       *dialog);

static GtkDialogClass *parent_class = NULL;


guint
glade_keys_dialog_get_type (void)
{
  static guint glade_keys_dialog_type = 0;

  if (!glade_keys_dialog_type)
    {
      GtkTypeInfo glade_keys_dialog_info =
      {
	"GladeKeysDialog",
	sizeof (GladeKeysDialog),
	sizeof (GladeKeysDialogClass),
	(GtkClassInitFunc) glade_keys_dialog_class_init,
	(GtkObjectInitFunc) glade_keys_dialog_init,
	(GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL,
      };

      glade_keys_dialog_type = gtk_type_unique (gtk_dialog_get_type (),
						&glade_keys_dialog_info);
    }

  return glade_keys_dialog_type;
}

static void
glade_keys_dialog_class_init (GladeKeysDialogClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;

  parent_class = gtk_type_class (gtk_dialog_get_type ());
}


static void
glade_keys_dialog_init (GladeKeysDialog       *dialog)
{
  int i, row;
  gchar *titles[1];

  gtk_window_set_title (GTK_WINDOW (dialog), _("Select Accelerator Key"));

  titles[0] = _("Keys");
  dialog->clist = gtk_clist_new_with_titles (1, titles);
  gtk_clist_column_titles_passive (GTK_CLIST (dialog->clist));
  gtk_clist_set_policy (GTK_CLIST (dialog->clist), GTK_POLICY_AUTOMATIC,
			GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (dialog->clist, 200, 300);
  gtk_widget_show (dialog->clist);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), dialog->clist,
		      TRUE, TRUE, 0);

  /* Insert events & descriptions */
  gtk_clist_freeze (GTK_CLIST (dialog->clist));

  i = 0;
  while (GbKeys[i].name)
    {
      /* For GTK 1.0 we can only use keys 0-255. */
#ifdef GLD_HAVE_GTK_1_1
      row = gtk_clist_append (GTK_CLIST (dialog->clist), &GbKeys[i].name);
      gtk_clist_set_row_data (GTK_CLIST (dialog->clist), row,
			      GINT_TO_POINTER (i));
#else
      if (GbKeys[i].key < 256)
	{
	  row = gtk_clist_append (GTK_CLIST (dialog->clist), &GbKeys[i].name);
	  gtk_clist_set_row_data (GTK_CLIST (dialog->clist), row,
				  GINT_TO_POINTER (i));
	}
#endif
      i++;
    }

  gtk_clist_thaw (GTK_CLIST (dialog->clist));

  dialog->ok_button = gtk_button_new_with_label (_("OK"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		      dialog->ok_button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (dialog->ok_button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (dialog->ok_button);
  gtk_widget_show (dialog->ok_button);

  dialog->cancel_button = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		      dialog->cancel_button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (dialog->cancel_button, GTK_CAN_DEFAULT);
  gtk_widget_show (dialog->cancel_button);
}


GtkWidget*
glade_keys_dialog_new ()
{
  GladeKeysDialog *dialog;

  dialog = gtk_type_new (glade_keys_dialog_get_type ());

  return GTK_WIDGET (dialog);
}


/* This returns TRUE if a key is currently selected, and the given pointer
   is set to the key value. */
gboolean
glade_keys_dialog_get_key (GladeKeysDialog *dialog,
			   guint	   *key)
{
  GList *selection = GTK_CLIST (dialog->clist)->selection;
  gint row, index;

  if (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      index = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (dialog->clist), row));
      *key = GbKeys[index].key;
      return TRUE;
    }
  return FALSE;
}


/* This returns the name of the key currently selected, or NULL if none is
   selected. */
gchar*
glade_keys_dialog_get_key_symbol (GladeKeysDialog *dialog)
{
  GList *selection = GTK_CLIST (dialog->clist)->selection;
  gint row, index;

  if (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      index = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (dialog->clist), row));
      return GbKeys[index].name;
    }
  return NULL;
}


/* This sets the current key, using the given key value. */
void
glade_keys_dialog_set_key (GladeKeysDialog *dialog,
			   guint	    key)
{
  gint row, index;

  for (row = 0; row < GTK_CLIST (dialog->clist)->rows; row++)
    {
      index = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (dialog->clist), row));
      if (GbKeys[index].key == key)
	{
	  gtk_clist_select_row (GTK_CLIST (dialog->clist), row, 0);
	  break;
	}
    }
}


/* This sets the current key, using the given key name. */
void
glade_keys_dialog_set_key_symbol (GladeKeysDialog *dialog,
				  gchar           *key)
{
  gint row, index;

  for (row = 0; row < GTK_CLIST (dialog->clist)->rows; row++)
    {
      index = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (dialog->clist), row));
      if (!strcmp (GbKeys[index].name, key))
	{
	  gtk_clist_select_row (GTK_CLIST (dialog->clist), row, 0);
	  break;
	}
    }
}


/* This finds the key value corresponding to the given string, which may be
   a symbol in gdkkeysyms.h, e.g. 'GDK_Tab', or a simple character, e.g. 'S'.
   Note that for GTK 1.0 this is a gchar, but for GTK 1.1 it is a guint. */
guint
glade_keys_dialog_find_key (gchar *symbol)
{
  gint i;

  /* Search the GbKeys array. */
  i = 0;
  while (GbKeys[i].name)
    {
      if (!strcmp (GbKeys[i].name, symbol))
	return GbKeys[i].key;
      i++;
    }
  /* If it's not found, return the first char. */
  return (guint) symbol[0];
}


