/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/JRV/NJN
 *
 *	$Id: lamreduce.c,v 6.1 96/11/23 22:53:11 nevin Rel $
 *
 *	Function:	- all predefined reduce operations
 *			- min, max, sum, product
 *			- and, or, xor (bitwise and logical)
 *			- minloc, maxloc
 */

#include <mpi.h>
#include <mpisys.h>

/*
 * local structures
 */
struct dblcplex { double r; double i; };
struct complex { float r; float i; };
struct fltflt { float v; float k; };
struct fltint { float v; int k; };
struct dbldbl { double v; double k; };
struct dblint { double v; int k; };
struct intint { int v; int k; };
struct longint { long v; int k; };
struct shortint { short v; int k; };
#if __STDC__
struct ldblint { long double v; int k; };
#else
struct ldblint { double v; int k; };
#endif

void
lam_max(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;
	float		*pfin, *pfout;
	double		*pdin, *pdout;
#if __STDC__
	long double	*pldin, *pldout;
#else
	double		*pldin, *pldout;
#endif
	
	if (*datatype == MPI_INT || *datatype == MPI_F_INTEGER) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			if (*piin > *piout) {
				*piout = *piin;
			}
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			if (*plin > *plout) {
				*plout = *plin;
			}
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			if (*psin > *psout) {
				*psout = *psin;
			}
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			if (*pusin > *pusout) {
				*pusout = *pusin;
			}
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			if (*puin > *puout) {
				*puout = *puin;
			}
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			if (*pulin > *pulout) {
				*pulout = *pulin;
			}
		}
		return;
	}

	if (*datatype == MPI_FLOAT || *datatype == MPI_F_REAL) {
		pfin = (float *) invec;
		pfout = (float *) inoutvec;
		for (i = 0; i < *len; ++i, ++pfin, ++pfout) {
			if (*pfin > *pfout) {
				*pfout = *pfin;
			}
		}
		return;
	}

	if (*datatype == MPI_DOUBLE || *datatype == MPI_F_DOUBLE_PRECISION) {
		pdin = (double *) invec;
		pdout = (double *) inoutvec;
		for (i = 0; i < *len; ++i, ++pdin, ++pdout) {
			if (*pdin > *pdout) {
				*pdout = *pdin;
			}
		}
		return;
	}

	if (*datatype == MPI_LONG_DOUBLE) {
#if __STDC__
		pldin = (long double *) invec;
		pldout = (long double *) inoutvec;
#else
		pldin = (double *) invec;
		pldout = (double *) inoutvec;
#endif
		for (i = 0; i < *len; ++i, ++pldin, ++pldout) {
			if (*pldin > *pldout) {
				*pldout = *pldin;
			}
		}
		return;
	}
}

void
lam_min(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;
	float		*pfin, *pfout;
	double		*pdin, *pdout;
#if __STDC__
	long double	*pldin, *pldout;
#else
	double		*pldin, *pldout;
#endif

	if (*datatype == MPI_INT || *datatype == MPI_F_INTEGER) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			if (*piin < *piout) {
				*piout = *piin;
			}
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			if (*plin < *plout) {
				*plout = *plin;
			}
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			if (*psin < *psout) {
				*psout = *psin;
			}
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			if (*pusin < *pusout) {
				*pusout = *pusin;
			}
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			if (*puin < *puout) {
				*puout = *puin;
			}
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			if (*pulin < *pulout) {
				*pulout = *pulin;
			}
		}
		return;
	}

	if (*datatype == MPI_FLOAT || *datatype == MPI_F_REAL) {
		pfin = (float *) invec;
		pfout = (float *) inoutvec;
		for (i = 0; i < *len; ++i, ++pfin, ++pfout) {
			if (*pfin < *pfout) {
				*pfout = *pfin;
			}
		}
		return;
	}

	if (*datatype == MPI_DOUBLE || *datatype == MPI_F_DOUBLE_PRECISION) {
		pdin = (double *) invec;
		pdout = (double *) inoutvec;
		for (i = 0; i < *len; ++i, ++pdin, ++pdout) {
			if (*pdin < *pdout) {
				*pdout = *pdin;
			}
		}
		return;
	}

	if (*datatype == MPI_LONG_DOUBLE) {
#if __STDC__
		pldin = (long double *) invec;
		pldout = (long double *) inoutvec;
#else
		pldin = (double *) invec;
		pldout = (double *) inoutvec;
#endif
		for (i = 0; i < *len; ++i, ++pldin, ++pldout) {
			if (*pldin < *pldout) {
				*pldout = *pldin;
			}
		}
		return;
	}
}

void
lam_sum(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;
	float		*pfin, *pfout;
	double		*pdin, *pdout;
	struct complex	*pxin, *pxout;
	struct dblcplex	*pdxin, *pdxout;
#if __STDC__
	long double	*pldin, *pldout;
#else
	double		*pldin, *pldout;
#endif

	if (*datatype == MPI_INT || *datatype == MPI_F_INTEGER) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			*piout += *piin;
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			*plout += *plin;
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			*psout += *psin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			*pusout += *pusin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			*puout += *puin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			*pulout += *pulin;
		}
		return;
	}

	if (*datatype == MPI_FLOAT  || *datatype == MPI_F_REAL) {
		pfin = (float *) invec;
		pfout = (float *) inoutvec;
		for (i = 0; i < *len; ++i, ++pfin, ++pfout) {
			*pfout += *pfin;
		}
		return;
	}

	if (*datatype == MPI_F_COMPLEX) {
		pxin = (struct complex *) invec;
		pxout = (struct complex *) inoutvec;
		for (i = 0; i < *len; ++i, ++pxin, ++pxout) {
			pxout->r += pxin->r;
			pxout->i += pxin->i;
		}
		return;
	}

	if (*datatype == MPI_F_DOUBLE_COMPLEX) {
		pdxin = (struct dblcplex *) invec;
		pdxout = (struct dblcplex *) inoutvec;
		for (i = 0; i < *len; ++i, ++pdxin, ++pdxout) {
			pdxout->r += pdxin->r;
			pdxout->i += pdxin->i;
		}
		return;
	}

	if (*datatype == MPI_DOUBLE || *datatype == MPI_F_DOUBLE_PRECISION) {
		pdin = (double *) invec;
		pdout = (double *) inoutvec;
		for (i = 0; i < *len; ++i, ++pdin, ++pdout) {
			*pdout += *pdin;
		}
		return;
	}

	if (*datatype == MPI_LONG_DOUBLE) {
#if __STDC__
		pldin = (long double *) invec;
		pldout = (long double *) inoutvec;
#else
		pldin = (double *) invec;
		pldout = (double *) inoutvec;
#endif
		for (i = 0; i < *len; ++i, ++pldin, ++pldout) {
			*pldout += *pldin;
		}
		return;
	}
}

void
lam_prod(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;
	float		*pfin, *pfout;
	float		xr, xi;
	double		dxr, dxi;
	double		*pdin, *pdout;
	struct complex	*pxin, *pxout;
	struct dblcplex	*pdxin, *pdxout;
#if __STDC__
	long double	*pldin, *pldout;
#else
	double		*pldin, *pldout;
#endif

	if (*datatype == MPI_INT || *datatype == MPI_F_INTEGER) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			*piout *= *piin;
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			*plout *= *plin;
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			*psout *= *psin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			*pusout *= *pusin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			*puout *= *puin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			*pulout *= *pulin;
		}
		return;
	}

	if (*datatype == MPI_FLOAT || *datatype == MPI_F_REAL) {
		pfin = (float *) invec;
		pfout = (float *) inoutvec;
		for (i = 0; i < *len; ++i, ++pfin, ++pfout) {
			*pfout *= *pfin;
		}
		return;
	}

	if (*datatype == MPI_F_COMPLEX) {
		pxin = (struct complex *) invec;
		pxout = (struct complex *) inoutvec;
		for (i = 0; i < *len; ++i, ++pxin, ++pxout) {
			xr = pxout->r;
			xi = pxout->i;
			pxout->r = (xr * pxin->r) - (xi * pxin->i);
			pxout->i = (xr * pxin->i) + (xi * pxin->r);
		}
		return;
	}

	if (*datatype == MPI_F_DOUBLE_COMPLEX) {
		pdxin = (struct dblcplex *) invec;
		pdxout = (struct dblcplex *) inoutvec;
		for (i = 0; i < *len; ++i, ++pdxin, ++pdxout) {
			dxr = pdxout->r;
			dxi = pdxout->i;
			pdxout->r = (dxr * pdxin->r) - (dxi * pdxin->i);
			pdxout->i = (dxr * pdxin->i) + (dxi * pdxin->r);
		}
		return;
	}

	if (*datatype == MPI_DOUBLE || *datatype == MPI_F_DOUBLE_PRECISION) {
		pdin = (double *) invec;
		pdout = (double *) inoutvec;
		for (i = 0; i < *len; ++i, ++pdin, ++pdout) {
			*pdout *= *pdin;
		}
		return;
	}

	if (*datatype == MPI_LONG_DOUBLE) {
#if __STDC__
		pldin = (long double *) invec;
		pldout = (long double *) inoutvec;
#else
		pldin = (double *) invec;
		pldout = (double *) inoutvec;
#endif
		for (i = 0; i < *len; ++i, ++pldin, ++pldout) {
			*pldout *= *pldin;
		}
		return;
	}
}

void
lam_land(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;

	if ((*datatype == MPI_INT) || (*datatype == MPI_F_LOGICAL)
			|| (*datatype == MPI_F_INTEGER)) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			*piout = *piout && *piin;
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			*plout = *plout && *plin;
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			*psout = *psout && *psin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			*pusout = *pusout && *pusin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			*puout = *puout && *puin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			*pulout = *pulout && *pulin;
		}
		return;
	}
}

void
lam_lor(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;

	if ((*datatype == MPI_INT) || (*datatype == MPI_F_LOGICAL)
			|| (*datatype == MPI_F_INTEGER)) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			*piout = *piout || *piin;
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			*plout = *plout || *plin;
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			*psout = *psout || *psin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			*pusout = *pusout || *pusin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			*puout = *puout || *puin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			*pulout = *pulout || *pulin;
		}
		return;
	}
}

void
lam_lxor(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;

	if ((*datatype == MPI_INT) || (*datatype == MPI_F_LOGICAL)
			|| (*datatype == MPI_F_INTEGER)) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			*piout = (*piout || *piin) && !(*piout && *piin);
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			*plout = (*plout || *plin) && !(*plout && *plin);
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			*psout = (*psout || *psin) && !(*psout && *psin);
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			*pusout = (*pusout || *pusin) && !(*pusout && *pusin);
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			*puout = (*puout || *puin) && !(*puout && *puin);
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			*pulout = (*pulout || *pulin) && !(*pulout && *pulin);
		}
		return;
	}
}

void
lam_band(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;
	unsigned char	*pucin, *pucout;

	if ((*datatype == MPI_INT) || (*datatype == MPI_F_LOGICAL)
			|| (*datatype == MPI_F_INTEGER)) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			*piout &= *piin;
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			*plout &= *plin;
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			*psout &= *psin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			*pusout &= *pusin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			*puout &= *puin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			*pulout &= *pulin;
		}
		return;
	}

	if (*datatype == MPI_BYTE) {
		pucin = (unsigned char *) invec;
		pucout = (unsigned char *) inoutvec;
		for (i = 0; i < *len; ++i, ++pucin, ++pucout) {
			*pucout &= *pucin;
		}
		return;
	}
}

void
lam_bor(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;
	unsigned char	*pucin, *pucout;

	if ((*datatype == MPI_INT) || (*datatype == MPI_F_LOGICAL)
			|| (*datatype == MPI_F_INTEGER)) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			*piout |= *piin;
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			*plout |= *plin;
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			*psout |= *psin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			*pusout |= *pusin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			*puout |= *puin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			*pulout |= *pulin;
		}
		return;
	}

	if (*datatype == MPI_BYTE) {
		pucin = (unsigned char *) invec;
		pucout = (unsigned char *) inoutvec;
		for (i = 0; i < *len; ++i, ++pucin, ++pucout) {
			*pucout |= *pucin;
		}
		return;
	}
}

void
lam_bxor(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	int		*piin, *piout;
	long		*plin, *plout;
	short		*psin, *psout;
	unsigned short	*pusin, *pusout;
	unsigned	*puin, *puout;
	unsigned long	*pulin, *pulout;
	unsigned char	*pucin, *pucout;

	if ((*datatype == MPI_INT) || (*datatype == MPI_F_LOGICAL)
			|| (*datatype == MPI_F_INTEGER)) {
		piin = (int *) invec;
		piout = (int *) inoutvec;
		for (i = 0; i < *len; ++i, ++piin, ++piout) {
			*piout ^= *piin;
		}
		return;
	}

	if (*datatype == MPI_LONG) {
		plin = (long *) invec;
		plout = (long *) inoutvec;
		for (i = 0; i < *len; ++i, ++plin, ++plout) {
			*plout ^= *plin;
		}
		return;
	}

	if (*datatype == MPI_SHORT) {
		psin = (short *) invec;
		psout = (short *) inoutvec;
		for (i = 0; i < *len; ++i, ++psin, ++psout) {
			*psout ^= *psin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_SHORT) {
		pusin = (unsigned short *) invec;
		pusout = (unsigned short *) inoutvec;
		for (i = 0; i < *len; ++i, ++pusin, ++pusout) {
			*pusout ^= *pusin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED) {
		puin = (unsigned *) invec;
		puout = (unsigned *) inoutvec;
		for (i = 0; i < *len; ++i, ++puin, ++puout) {
			*puout ^= *puin;
		}
		return;
	}

	if (*datatype == MPI_UNSIGNED_LONG) {
		pulin = (unsigned long *) invec;
		pulout = (unsigned long *) inoutvec;
		for (i = 0; i < *len; ++i, ++pulin, ++pulout) {
			*pulout ^= *pulin;
		}
		return;
	}

	if (*datatype == MPI_BYTE) {
		pucin = (unsigned char *) invec;
		pucout = (unsigned char *) inoutvec;
		for (i = 0; i < *len; ++i, ++pucin, ++pucout) {
			*pucout ^= *pucin;
		}
		return;
	}
}

void
lam_maxloc(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	struct fltflt	*pffin, *pffout;
	struct fltint	*pfiin, *pfiout;
	struct dbldbl	*pddin, *pddout;
	struct dblint	*pdiin, *pdiout;
	struct ldblint	*pldiin, *pldiout;
	struct intint	*piiin, *piiout;
	struct longint	*pliin, *pliout;
	struct shortint	*psiin, *psiout;

	if (*datatype == MPI_2FLOAT || *datatype == MPI_F_2REAL) {
		pffin = (struct fltflt *) invec;
		pffout = (struct fltflt *) inoutvec;
		for (i = 0; i < *len; ++i, ++pffin, ++pffout) {
			if (pffin->v > pffout->v) {
				pffout->v = pffin->v;
				pffout->k = pffin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_FLOAT_INT) {
		pfiin = (struct fltint *) invec;
		pfiout = (struct fltint *) inoutvec;
		for (i = 0; i < *len; ++i, ++pfiin, ++pfiout) {
			if (pfiin->v > pfiout->v) {
				pfiout->v = pfiin->v;
				pfiout->k = pfiin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_2DOUBLE || *datatype == MPI_F_2DOUBLE_PRECISION) {
		pddin = (struct dbldbl *) invec;
		pddout = (struct dbldbl *) inoutvec;
		for (i = 0; i < *len; ++i, ++pddin, ++pddout) {
			if (pddin->v > pddout->v) {
				pddout->v = pddin->v;
				pddout->k = pddin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_DOUBLE_INT) {
		pdiin = (struct dblint *) invec;
		pdiout = (struct dblint *) inoutvec;
		for (i = 0; i < *len; ++i, ++pdiin, ++pdiout) {
			if (pdiin->v > pdiout->v) {
				pdiout->v = pdiin->v;
				pdiout->k = pdiin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_LONG_DOUBLE_INT) {
		pldiin = (struct ldblint *) invec;
		pldiout = (struct ldblint *) inoutvec;
		for (i = 0; i < *len; ++i, ++pldiin, ++pldiout) {
			if (pldiin->v > pldiout->v) {
				pldiout->v = pldiin->v;
				pldiout->k = pldiin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_2INT || *datatype == MPI_F_2INTEGER) {
		piiin = (struct intint *) invec;
		piiout = (struct intint *) inoutvec;
		for (i = 0; i < *len; ++i, ++piiin, ++piiout) {
			if (piiin->v > piiout->v) {
				piiout->v = piiin->v;
				piiout->k = piiin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_LONG_INT) {
		pliin = (struct longint *) invec;
		pliout = (struct longint *) inoutvec;
		for (i = 0; i < *len; ++i, ++pliin, ++pliout) {
			if (pliin->v > pliout->v) {
				pliout->v = pliin->v;
				pliout->k = pliin->k;
			}
		}
		return;
	}
	
	if (*datatype == MPI_SHORT_INT) {
		psiin = (struct shortint *) invec;
		psiout = (struct shortint *) inoutvec;
		for (i = 0; i < *len; ++i, ++psiin, ++psiout) {
			if (psiin->v > psiout->v) {
				psiout->v = psiin->v;
				psiout->k = psiin->k;
			}
		}
		return;
	}
}


void
lam_minloc(invec, inoutvec, len, datatype)

void			*invec;
void			*inoutvec;
int			*len;
MPI_Datatype		*datatype;

{
	int		i;
	struct fltflt	*pffin, *pffout;
	struct fltint	*pfiin, *pfiout;
	struct dbldbl	*pddin, *pddout;
	struct dblint	*pdiin, *pdiout;
	struct ldblint	*pldiin, *pldiout;
	struct intint	*piiin, *piiout;
	struct longint	*pliin, *pliout;
	struct shortint	*psiin, *psiout;

	if (*datatype == MPI_2FLOAT || *datatype == MPI_F_2REAL) {
		pffin = (struct fltflt *) invec;
		pffout = (struct fltflt *) inoutvec;
		for (i = 0; i < *len; ++i, ++pffin, ++pffout) {
			if (pffin->v < pffout->v) {
				pffout->v = pffin->v;
				pffout->k = pffin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_FLOAT_INT) {
		pfiin = (struct fltint *) invec;
		pfiout = (struct fltint *) inoutvec;
		for (i = 0; i < *len; ++i, ++pfiin, ++pfiout) {
			if (pfiin->v < pfiout->v) {
				pfiout->v = pfiin->v;
				pfiout->k = pfiin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_2DOUBLE || *datatype == MPI_F_2DOUBLE_PRECISION) {
		pddin = (struct dbldbl *) invec;
		pddout = (struct dbldbl *) inoutvec;
		for (i = 0; i < *len; ++i, ++pddin, ++pddout) {
			if (pddin->v < pddout->v) {
				pddout->v = pddin->v;
				pddout->k = pddin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_DOUBLE_INT) {
		pdiin = (struct dblint *) invec;
		pdiout = (struct dblint *) inoutvec;
		for (i = 0; i < *len; ++i, ++pdiin, ++pdiout) {
			if (pdiin->v < pdiout->v) {
				pdiout->v = pdiin->v;
				pdiout->k = pdiin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_LONG_DOUBLE_INT) {
		pldiin = (struct ldblint *) invec;
		pldiout = (struct ldblint *) inoutvec;
		for (i = 0; i < *len; ++i, ++pldiin, ++pldiout) {
			if (pldiin->v < pldiout->v) {
				pldiout->v = pldiin->v;
				pldiout->k = pldiin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_2INT || *datatype == MPI_F_2INTEGER) {
		piiin = (struct intint *) invec;
		piiout = (struct intint *) inoutvec;
		for (i = 0; i < *len; ++i, ++piiin, ++piiout) {
			if (piiin->v < piiout->v) {
				piiout->v = piiin->v;
				piiout->k = piiin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_LONG_INT) {
		pliin = (struct longint *) invec;
		pliout = (struct longint *) inoutvec;
		for (i = 0; i < *len; ++i, ++pliin, ++pliout) {
			if (pliin->v < pliout->v) {
				pliout->v = pliin->v;
				pliout->k = pliin->k;
			}
		}
		return;
	}

	if (*datatype == MPI_SHORT_INT) {
		psiin = (struct shortint *) invec;
		psiout = (struct shortint *) inoutvec;
		for (i = 0; i < *len; ++i, ++psiin, ++psiout) {
			if (psiin->v < psiout->v) {
				psiout->v = psiin->v;
				psiout->k = psiin->k;
			}
		}
		return;
	}
}
