/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: lamtest.c,v 6.1 96/11/23 22:53:16 nevin Rel $
 *
 *	Function:	- test if request is done
 *			- destructive and non-destructive modes
 *	Accepts:	- request
 *			- destructive mode flag
 *			- flag (returned)
 *			- status (returned)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>
#include <terror.h>

/*
 * external functions
 */
extern void		lam_emptystat();
extern void		lam_tr_msg();
extern void		_mpi_req_blkclr();
extern void		_mpi_req_blkset();
extern void		_mpi_req_rem();
extern int		lam_mkerr();
extern int		lam_tr_incff();
extern int		_mpi_req_advance();
extern int		_mpi_req_destroy();
extern int		_mpi_req_end();


int
lam_test(preq, destroy, flag, stat)

MPI_Request		*preq;
int			destroy;
int			*flag;
MPI_Status		*stat;

{
	MPI_Request	req;			/* request */
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	double		startt;			/* start time */
	double		finisht;		/* finish time */

	if (preq == 0) return(lam_mkerr(MPI_ERR_REQUEST, 0));

	if ((flag == 0) || (stat == 0)) return(lam_mkerr(MPI_ERR_ARG, 0));

	if ((fl_trace = LAM_TRACE_TOP())) {
	    startt = MPI_Wtime();
	    _kio.ki_blktime = 0.0;
	}
/*
 * Advance the system.
 */
	_mpi_req_blkclr();

	err = _mpi_req_advance();
	if (err != MPI_SUCCESS) return(err);
/*
 * Inactive requests get an empty status.
 */
	req = *preq;

	if ((req == MPI_REQUEST_NULL) || (req->rq_state == LAM_RQSINIT)) {
	    *flag = 1;
	    lam_emptystat(stat);
	}
/*
 * If not done reset flag.
 */
	else if (req->rq_state != LAM_RQSDONE) {
	    *flag = 0;
	}
/*
 * Finish the request if in destructive-mode.
 */
	else {
	    *flag = 1;

	    if (destroy) {
		_mpi_req_rem(req);

		err = _mpi_req_end(req);
		if (err != MPI_SUCCESS) return(err);

		*stat = req->rq_status;
/*
 * Generate a run time trace except in the case of cancelled request or
 * a request where the peer is MPI_PROC_NULL.
 */
		if (fl_trace && (req->rq_rank != MPI_PROC_NULL)
				&& !(req->rq_flags & LAM_RQFCANCEL)) {
		    finisht = MPI_Wtime();

		    lam_tr_msg( (req->rq_type == LAM_RQIRECV)
			    ? TRTINPUT : TRTNOIO, startt,
			    LAM_S2US(finisht - startt - _kio.ki_blktime),
			    LAM_S2US(_kio.ki_blktime), req->rq_rank,
			    req->rq_tag, req->rq_comm, req->rq_dtype,
			    req->rq_count, stat->MPI_SOURCE,
			    stat->MPI_TAG, req->rq_seq, req->rq_type);
		}
/*
 * Reset persistent requests.
 * Destroy requests that are not persistent.
 */
		if (req->rq_marks & LAM_RQFPERSIST) {
		    req->rq_state = LAM_RQSINIT;
		} else {
		    err = _mpi_req_destroy(preq);
		    if (err != MPI_SUCCESS) return(err);
		}

		if (stat->MPI_ERROR != MPI_SUCCESS) {
		    return(stat->MPI_ERROR);
		}
	    }
	}
	
	return(MPI_SUCCESS);
}
