/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: wait.c,v 6.1 96/11/23 22:54:52 nevin Rel $
 *
 *	Function:	- wait for completion of immediate request
 *	Accepts:	- request
 *			- status
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>

/*
 * external functions
 */
extern void		lam_emptystat();
extern void		lam_initerr();
extern void		lam_resetfunc();
extern void		lam_setfunc();
extern void		lam_tr_msg();
extern void		_mpi_req_blkclr();
extern void		_mpi_req_blkset();
extern void		_mpi_req_rem();
extern int		lam_errfunc();
extern int		lam_mkerr();
extern int		lam_tr_incff();
extern int		_mpi_req_advance();
extern int		_mpi_req_destroy();
extern int		_mpi_req_end();


int
MPI_Wait(preq, stat)

MPI_Request		*preq;
MPI_Status		*stat;

{
	MPI_Request	req;			/* request */
	MPI_Comm	comm;			/* communicator */
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	double		startt;			/* start time */
	double		finisht;		/* finish time */
	
	lam_initerr();
	lam_setfunc(BLKMPIWAIT);

	if (preq == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAIT,
				lam_mkerr(MPI_ERR_REQUEST, 0)));
	}

	if (stat == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAIT,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Inactive requests get an empty status.
 */
	req = *preq;

	if ((req == MPI_REQUEST_NULL) || (req->rq_state == LAM_RQSINIT)) {
		lam_emptystat(stat);
		lam_resetfunc(BLKMPIWAIT);
		return(MPI_SUCCESS);
	}

	if ((fl_trace = LAM_TRACE_TOP())) {
		startt = MPI_Wtime();
		_kio.ki_blktime = 0.0;
	}
/*
 * Block till request is done.
 */
	comm = req->rq_comm;

	if (req->rq_state != LAM_RQSDONE) {
		_mpi_req_blkclr();
		_mpi_req_blkset(req);

		err = _mpi_req_advance();
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIWAIT, err));
		}
	}

	_mpi_req_rem(req);

	err = _mpi_req_end(req);
	if (err != MPI_SUCCESS) return(lam_errfunc(comm, BLKMPIWAIT, err));

	*stat = req->rq_status;
/*
 * Generate a run time trace except in the case of a cancelled request or
 * a request where the peer is MPI_PROC_NULL.
 */
	if (fl_trace && (req->rq_rank != MPI_PROC_NULL)
			&& !(req->rq_flags & LAM_RQFCANCEL)) {
		finisht = MPI_Wtime();

		lam_tr_msg(
			(req->rq_type == LAM_RQIRECV) ? TRTINPUT : TRTNOIO,
			startt, LAM_S2US(finisht - startt - _kio.ki_blktime),
			LAM_S2US(_kio.ki_blktime), req->rq_rank, req->rq_tag,
			req->rq_comm, req->rq_dtype, req->rq_count,
			stat->MPI_SOURCE, stat->MPI_TAG, req->rq_seq,
			req->rq_type);
	}
/*
 * Reset persistent requests.
 * Destroy requests that are not persistent.
 */
	if (req->rq_marks & LAM_RQFPERSIST) {
		req->rq_state = LAM_RQSINIT;
	} else {
		err = _mpi_req_destroy(preq);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIWAIT, err));
		}
	}

	lam_resetfunc(BLKMPIWAIT);

	if (stat->MPI_ERROR != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWAIT, stat->MPI_ERROR));
	}

	return(MPI_SUCCESS);
}
