/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	GDB/RBD
 *
 *	$Id: hboot.c,v 6.1 96/11/23 19:36:35 nevin Rel $
 *
 *	Function:	- boots OTB operating system
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <all_list.h>
#include <args.h>
#include <boot.h>
#include <debug.h>
#include <kio.h>
#include <portable.h>
#include <proc_schema.h>
#include <sfh.h>
#include <terror.h>

#define MAXCLOSEFD	32			/* max close-on-exec fdecs. */

/*
 * external functions
 */
extern int		_lam_few();
extern int		psc_parse();
extern struct psc	*psc_find();

/*
 * local functions
 */
static void		help();
static void		setdefaults();
static void		setvars();

/*
 * local variables
 */
static char		*f_psc;			/* process schema file */
static char		**av_psc_path;		/* f_psc pathname prefixes */
static char		*t_tkill;		/* tkill tool */
static char		rtfbuf[32];		/* RTF env. var. */

static int		ac_psc_path;		/* # of p_psc prefixes */
static int		fl_debug;		/* debugging option */
static int		fl_verbose;		/* verbose option */

static char *usage = "usage: hboot [-dhstvN] [-c <schema>] [-I <inet_topo>] [-R <rtr_topo>]\n";


int
main(argc, argv)

int			argc;
char			*argv[];

{
	FILE		*fp_psc;	/* process schema file ptr */
	LIST		*list_psc;	/* parsed process schema list */
	struct psc	*p;
	int		i, n;
	int		fd;		/* file descriptor for /dev/null */
	int		pid;		/* child PID */
	int		ac_cmd;		/* # command arguments */
	int		ac_topo;	/* # topology variable overrides */
	char		**av_cmd;	/* command arguments */
	char		**av_topo;	/* topology variable overrides */
	char		buf[32];	/* formatting buffer */
	char		*full;		/* full pathname */
	char		*tail;		/* tail of full pathname */
/*
 * Initialize option parser.
 */
	validopts("dhstvNV");
	followed("cIR");

	if ((do_args(&argc, argv) < 0) || (errno = (argc == 1) ? 0 : EUSAGE)) {
		fprintf(stderr, usage);
		exit(errno);
	}

	if (opt_taken('h')) {
		help();
		exit(0);
	}

	setdefaults();
/*
 * Locate the conf file.
 */
	full = sfh_path_find(f_psc, av_psc_path, R_OK);

	if (! full) {
		fprintf(stderr, "hboot: cannot find process schema %s: ",
				f_psc);
		perror("");
		exit(errno);
	} else {
		f_psc = full;
	}
/*
 * Dump debugging information.
 */
	DBUG("hboot: process schema = \"%s\"\n", f_psc);
	fflush(stdout);
/*
 * Open the conf file.
 */
	fp_psc = fopen(f_psc, "r");

	if (! fp_psc) {
		fprintf(stderr, "hboot: cannot open %s: ", f_psc);
		perror("");
		exit(errno);
	}
/*
 * Handle command line override of variables.
 */
	ac_topo = 0;
	av_topo = 0;

	if (opt_taken('I')) {
		setvars(&ac_topo, &av_topo, "$inet_topo", getparam('I'));
	}

	if (opt_taken('R')) {
		setvars(&ac_topo, &av_topo, "$rtr_topo", getparam('R'));
	}
/*
 * Parse the conf file.
 */
	if (psc_parse(fp_psc, &list_psc, av_topo)) {
		fprintf(stderr, "hboot: cannot parse %s: ", f_psc);
		perror("");
		exit(errno);
	}

	argvfree(av_topo);
	fclose(fp_psc);

	if (al_count(list_psc) == 0) {
		fprintf(stderr, "hboot: nothing to do in %s\n", f_psc);
		exit(0);
	}
/*
 * Find programs with sufficient permissions.
 */
	p = psc_find(list_psc);

	if (p) {
		fprintf(stderr, "hboot: cannot find executable %s: ",
				p->psc_argv[0]);
		perror("");
		exit(errno);
	}

	if (fl_debug) {

		for (p = al_top(list_psc); p; p = al_next(list_psc, p)) {
			printf("hboot: found %s\n", p->psc_argv[0]);
		}
	}
/*
 * Bail out here, if pretending.
 */
	if (opt_taken('N')) {
		exit(0);
	}
/*
 * Tkill if needed.
 */
	if (opt_taken('t')) {
		VERBOSE("%s ...\n", t_tkill);

		ac_cmd = 0;
		av_cmd = 0;
		argvadd(&ac_cmd, &av_cmd, t_tkill);

		if (_lam_few(av_cmd)) {
			fprintf(stderr, "hboot: cannot tkill: ");
			perror("");
			exit(errno);
		}
	}
/*
 * Boot.
 */
	VERBOSE("boot ...\n");

	sprintf(rtfbuf, "TROLLIUSRTF=%d", RTF_SYSGEN);

	if (putenv(rtfbuf) < 0) {
		perror("hboot (putenv)");
		exit(errno);
	}

	setsid();

	if (opt_taken('s')) {
/*
 * Make any extraneous file descriptors close-on-exec.
 */
		for (i = 3; i < MAXCLOSEFD; ++i) {

			if ((fcntl(i, F_SETFD, 1) != 0) && (errno != EBADF)) {
				fprintf(stderr,
					"hboot: cannot close-on-exec, fd=%d: ",
					i);
				perror("");
				exit(errno);
			}
		}
	}

	n = 0;
/*
 * Loop through all the programs in the parsed config file.
 */
	for (p = al_top(list_psc); p; p = al_next(list_psc, p)) {
		DBUG("hboot: fork %s\n", p->psc_argv[0]);

		if ((pid = fork()) < 0) {
			perror("hboot (fork)");
			exit(errno);
		}

		else if (pid == 0) {		/* child */

			if (opt_taken('s')) {
/*
 * Safely get rid of the stdio descriptors.
 */
				if ((fd = open("/dev/null", O_RDWR)) < 0) {
					perror("hboot (open)");
					exit(errno);
				}

				if ((dup2(fd, 0) < 0) || (dup2(fd, 1) < 0) ||
						(dup2(fd, 2) < 0)) {
					perror("hboot");
					exit(errno);
				}

				close(fd);
			}

			execvp(p->psc_argv[0], p->psc_argv);
			exit(errno);
		}

		else {				/* parent */
			n++;

			if (fl_verbose) {
				tail = strrchr(p->psc_argv[0], STRDIR);
				tail = (tail) ? tail + 1 : p->psc_argv[0];
				sprintf(buf, "[%d]", n);
				printf("%-4.4s %5d %s", buf, pid, tail);

				for (i = 1; i < p->psc_argc; i++) {
					printf(" %s", p->psc_argv[i]);
				}

				printf("\n");
			}
		}

		if (p->psc_delay > 0) {
			sleep((unsigned int) p->psc_delay);
		}
	}

	return(0);
}

/*
 *	setdefaults
 *
 *	Function:	- sets default files and paths
 */
static void
setdefaults()

{
/*
 * prefix paths
 */
	ac_psc_path = 0;
	av_psc_path = 0;
	argvadd(&ac_psc_path, &av_psc_path, "");
	argvadd(&ac_psc_path, &av_psc_path, "$LAMHOME/boot");
	argvadd(&ac_psc_path, &av_psc_path, "$TROLLIUSHOME/boot");
	argvadd(&ac_psc_path, &av_psc_path, DEFPSCHEMA);
/*
 * tools
 */
	t_tkill = DEFTRESETH;
/*
 * flags
 */
	fl_debug = opt_taken('d');;
	fl_verbose = opt_taken('v');
/*
 * files
 */
	if (opt_taken('c')) {
		f_psc = getparam('c');
	} else {
		f_psc = DEFFCONFIGH;
	}
}

/*
 *	setvars
 *
 *	Function:	- handles command line override of variables
 *			- vars: $inet_topo, $rtr_topo
 */
static void
setvars(ac, av, var, opt)

int			*ac;
char			***av;
char			*var;
char			*opt;

{
	char		*override;

	override = malloc((unsigned) (strlen(var) + strlen(opt) + 1));
	if (override == 0) {
		perror("hboot (malloc)");
		exit(errno);
	}

	strcpy(override, var);
	strcat(override, " ");
	strcat(override, opt);

	if (argvadd(ac, av, override)) {
		perror("hboot (argvadd)");
		exit(errno);
	}

	free(override);
}

/*
 *	help
 *
 *	Function:	- prints helpful information on the hboot command
 */
static void
help()

{
	printf("\nSynopsis:	hboot [options]\n");
	printf("\nDescription:	Start LAM on one node.\n");
	printf("\nOptions:	-h		Print this message.\n");
	printf("\t\t-d		Print debug information.\n");
	printf("\t\t-s		Close stdio of processes.\n");
	printf("\t\t-t		Kill existing session first.\n");
	printf("\t\t-v		Be verbose.\n");
	printf("\t\t-N		Pretend to hboot.\n");
	printf("\t\t-c <conf>	Use <conf> as the process schema.\n");
	printf("\t\t-I <inet_topo>	Set $inet_topo variable.\n");
	printf("\t\t-R <rtr_topo>	Set $rtr_topo variable.\n");
}
