/*
 *	Ohio Trollius
 *	Copyright 1995 The Ohio State University
 *	GDB/RBD
 *
 *	$Log:	wipe.c,v $
 * Revision 6.1  96/11/23  19:37:59  nevin
 * Ohio Release
 * 
 * Revision 6.0  96/02/29  14:22:28  gdburns
 * Ohio Release
 * 
 * Revision 5.2.1.2  96/02/28  18:44:38  gdburns
 * major overhaul
 * 
 * Revision 5.2.1.1  94/10/18  02:18:09  raja
 * Upgrade to new network description.
 * 
 * Revision 5.2  94/08/22  14:19:19  gdburns
 * Ohio Release
 * 
 * Revision 5.1.1.1  94/08/19  16:58:57  gdburns
 * Add -n option to limit # of wiped nodes.
 * Remove -V option.
 * 
 * Revision 5.1  94/05/18  13:02:22  gdburns
 * Ohio Release
 * 
 * Revision 2.3  94/04/22  12:50:50  gdburns
 * Ohio Release
 * 
 *	Function:	- topology shutdown tool
 *			- kills Trollius on a network of OTB nodes
 *			- topology given using host file syntax
 *			- uses tkill
 *      Options:        - see man page and help menu
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <args.h>
#include <debug.h>
#include <lamnet.h>
#include <net.h>
#include <terror.h>

/*
 * static variables
 */
static char		f_input[80];	/* input filename */
static int		fl_verbose;	/* verbose mode */
static int		fl_debug;	/* debug mode */

static char		*usage = "wipe [-dhv] [-n <#>] [<bhost>]";

/*
 * external functions
 */
extern char		*findpath();
extern int		_lam_few();
extern int		inetexec();
extern int		bhostparse();

/*
 * static functions
 */
static void		help();
static char		*findbhost();

int
main(argc, argv)

int			argc;
char			*argv[];

{
	FILE		*fp;		/* boot schema file pointer */
	char		*bhost;		/* boot schema filename */
	char		**cmdv;		/* cmd arguments */
	int		cmdn;		/* # cmd arguments */
	int		global_ret;	/* global error status */
	int		i;
	int		iorigin;	/* index of origin node */
	int		limit;		/* limit # nodes to wipe */
	int		nlamnet;	/* lamnet size */
	int		r;
	struct lamnode	*lamnet;	/* network description array */
/*
 * Parse the command line.
 */
	validopts("dhv");
	followed("n");

	if (do_args(&argc, argv)) {
		fprintf(stderr, "usage: %s\n", usage);
		exit(EUSAGE);
	}

	if ((errno = (argc <= 2) ? 0 : EUSAGE)) {
		fprintf(stderr, "usage: %s\n", usage);
		exit(errno);
	}

	if (opt_taken('h')) {
		help();
		exit(0);
	}

	fl_debug = opt_taken('d');
	fl_verbose = opt_taken('v') || fl_debug;
/* 
 * Locate the system file.
 */
	bhost = findbhost(argc, argv);
/*
 * Open the system file.
 */
	fp = fopen(bhost, "r");
	if (fp == 0) lampanic("wipe (fopen)");
/*
 * Parse the system file.
 */
	if (bhostparse(fp, &lamnet, &nlamnet)) {
		fprintf(stderr, "wipe: cannot parse host file: ");
		lampanic("");
	}
/*
 * Close the system file.
 */
	if (fclose(fp)) lampanic("wipe (fclose)");
/*
 * Locate the host nodes.
 */
	if (lamnet_findhosts(lamnet, nlamnet)) {

		if (errno == EBADHOST) {
			lampanic("wipe");
		} else {
			lampanic("wipe (lamnet_findhosts)");
		}
	}
/*
 * Find the origin node.
 */
	iorigin = lamnet_findorig(lamnet, nlamnet);

	if (iorigin >= 0) {
		lamnet[iorigin].lnd_type |= NT_ORIGIN | NT_ME;
	}

	DBUG("wipe: found %d host node(s)\n", nlamnet);
	DBUG("wipe: origin node is %d\n", iorigin);
/*
 * Build the tkill command.
 */
	cmdn = 0;
	cmdv = 0;
	argvadd(&cmdn, &cmdv, DEFTTKILL);

	if (fl_debug) {
		argvadd(&cmdn, &cmdv, "-d");
	}

	if (opt_taken('n')) {
		intparam('n', &limit);
	} else {
		limit = -1;
	}
/*
 * Loop over all host nodes.
 */
	global_ret = 0;

	for (i = 0; (i < nlamnet) && limit; ++i) {

		if (limit > 0) --limit;

		VERBOSE("%s n%d (%s)...\n", DEFTTKILL,
				lamnet[i].lnd_nodeid, lamnet[i].lnd_hname);

		if (lamnet[i].lnd_type & NT_ORIGIN) {
			r = _lam_few(cmdv);

			if (r) {
				errno = r;
			}
		} else {
			r = inetexec(lamnet[i].lnd_hname,
					lamnet[i].lnd_uname, cmdv);
		}

		if (r) {
			fprintf(stderr, "wipe: %s failed on n%d (%s)\n",
					DEFTTKILL, lamnet[i].lnd_nodeid,
					lamnet[i].lnd_hname);

			if (errno != EUNKNOWN) {
				terror("wipe");
			}

			global_ret = errno;
		}
	}

	argvfree(cmdv);
	return(global_ret);
}

/*
 *	findbhost
 *
 *	Function:	- locates boot schema file
 *	Accepts:	- argc of remaining command line
 *			- argv of remaining command line
 *	Returns:	- full pathname of boot schema file
 */
static char *
findbhost(cmdc, cmdv)

int			cmdc;
char			**cmdv;

{
	char		*bhost;
	char		*full;		/* temporary for full pathname */
	char		**pathv;
	int		pathc;
/*
 * Set the directories for the boot schema file search.
 */
	pathc = 0;
	pathv = 0;
	argvadd(&pathc, &pathv, "");
	argvadd(&pathc, &pathv, "$TROLLIUSHOME/boot");
	argvadd(&pathc, &pathv, "$LAMHOME/boot");
	argvadd(&pathc, &pathv, DEFPBHOST);
/*
 * Set the boot schema file names.
 */
	if (cmdc == 2) {
		bhost = cmdv[1];
	} else if ((bhost = getenv("LAMBHOST"))) {
	} else if ((bhost = getenv("TROLLIUSBHOST"))) {
	} else {
		bhost = f_input;
		printf("Enter boot schema filename (host file): ");
		fflush(stdout);
		scanf("%79s", f_input);
	}
/*
 * Search the directories.
 */
	full = findpath(bhost, pathv, R_OK);

	if (full == 0) {
		fprintf(stderr, "wipe: cannot locate \"%s\"\n", bhost);
		exit(errno);
	}

	DBUG("wipe: boot schema file: %s\n", full);
	return(full);
}

/*
 *	help
 *
 *	Function:	- prints the help menu
 */
static void
help()

{
	printf("\nSynopsis:\twipe [options] [<bhost>]\n");
	printf("\nDescription:\tShutdown a LAM multicomputer.\n");
	printf("\nOptions:\t-h\t\tPrint this message.\n");
	printf("\t\t-d\t\tTurn on debugging.\n");
	printf("\t\t-v\t\tBe verbose.\n");
	printf("\t\t-n <#>\t\tWipe the first <#> nodes.\n");
	printf("\t\t<bhost>\t\tUse <bhost> as boot schema.\n\n");
}
