/*	$Id: Files.c,v 1.11 1998/06/29 15:46:54 acken Exp $	*/
/*  Access to files and file attributes.
    Copyright (C) 1997, 1998  Michael van Acken

    This file is part of OOC.

    OOC is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.  

    OOC is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details. 

    You should have received a copy of the GNU General Public License
    along with OOC. If not, write to the Free Software Foundation, 59
    Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#include <utime.h>
#include <limits.h>

#include "__oo2c.h"
#include "__mini_gc.h"
#include "__StdTypes.h"
#include "__config.h"

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

/* the minimum number of temporary files supported by any system; GNU libc info
   says this is 25 */
#define TMP_MIN 25
/* prepare for the worst and assume that really just TMP_MIN temporary names
   are available; maintain a buffer of discarded names */
static char *tmp_name[TMP_MIN];

/* this is the currently active umask of the process; it is used by procedure
   Register */
static mode_t active_umask;

/* if the system imposes no limit on the length of a file name, choose an 
   arbitrary large value instead */
#ifndef PATH_MAX
#define PATH_MAX 2048
#endif


/* --- begin #include "Files.d" */
#include "Files.h"
#include "Termination.h"

/* local definitions */

/* function prototypes */

/* module and type descriptors */
static const struct {
  int length;
  void* pad;
  const char name[6];
} _n0 = {6, NULL, {"Files"}};
static struct _MD Files__md = {
  NULL, 
  &Kernel_ModuleDesc__td.td, 
  {
    NULL, 
    (const unsigned char*)_n0.name, 
    -1, 
    NULL
  }
};

static const struct {
  int length;
  void* pad;
  const char name[9];
} _n1 = {9, NULL, {"FileDesc"}};
static const struct {
  int length;
  void* pad;
  _Type btypes[3];
} Files_FileDesc__tdb = {
  3, 
  NULL, 
  {
    &Channel_ChannelDesc__td.td, 
    &PosixFileDescr_ChannelDesc__td.td, 
    &Files_FileDesc__td.td
  }
};
static const struct {
  int length;
  void* pad;
  const void* tbprocs[9];
} _tb0 = {9, NULL, {
  (void*)PosixFileDescr_ChannelDesc_Length, 
  (void*)PosixFileDescr_ChannelDesc_GetModTime, 
  (void*)Files_FileDesc_NewReader, 
  (void*)Files_FileDesc_NewWriter, 
  (void*)PosixFileDescr_ChannelDesc_Flush, 
  (void*)Files_FileDesc_Close, 
  (void*)Files_FileDesc_ErrorDescr, 
  (void*)Channel_ChannelDesc_ClearError, 
  (void*)Files_FileDesc_Register
}};
struct _TD Files_FileDesc__td = {
  NULL, 
  &Types_TypeDesc__td.td, 
  {
    Files_FileDesc__tdb.btypes, 
    _tb0.tbprocs, 
    (const unsigned char*)_n1.name, 
    &Files__md.md, 
    2, 
    '0', '1', 
    sizeof(Files_FileDesc), 
    NULL
  }
};

static const struct {
  int length;
  void* pad;
  const char name[11];
} _n2 = {11, NULL, {"ReaderDesc"}};
static const struct {
  int length;
  void* pad;
  _Type btypes[3];
} Files_ReaderDesc__tdb = {
  3, 
  NULL, 
  {
    &Channel_ReaderDesc__td.td, 
    &PosixFileDescr_ReaderDesc__td.td, 
    &Files_ReaderDesc__td.td
  }
};
static const struct {
  int length;
  void* pad;
  const void* tbprocs[7];
} _tb1 = {7, NULL, {
  (void*)PosixFileDescr_ReaderDesc_Pos, 
  (void*)PosixFileDescr_ReaderDesc_Available, 
  (void*)PosixFileDescr_ReaderDesc_SetPos, 
  (void*)PosixFileDescr_ReaderDesc_ReadByte, 
  (void*)PosixFileDescr_ReaderDesc_ReadBytes, 
  (void*)Files_ReaderDesc_ErrorDescr, 
  (void*)Channel_ReaderDesc_ClearError
}};
struct _TD Files_ReaderDesc__td = {
  NULL, 
  &Types_TypeDesc__td.td, 
  {
    Files_ReaderDesc__tdb.btypes, 
    _tb1.tbprocs, 
    (const unsigned char*)_n2.name, 
    &Files__md.md, 
    2, 
    '0', '1', 
    sizeof(Files_ReaderDesc), 
    &Files_FileDesc__td.td
  }
};

static const struct {
  int length;
  void* pad;
  const char name[11];
} _n3 = {11, NULL, {"WriterDesc"}};
static const struct {
  int length;
  void* pad;
  _Type btypes[3];
} Files_WriterDesc__tdb = {
  3, 
  NULL, 
  {
    &Channel_WriterDesc__td.td, 
    &PosixFileDescr_WriterDesc__td.td, 
    &Files_WriterDesc__td.td
  }
};
static const struct {
  int length;
  void* pad;
  const void* tbprocs[7];
} _tb2 = {7, NULL, {
  (void*)PosixFileDescr_WriterDesc_Pos, 
  (void*)PosixFileDescr_WriterDesc_SetPos, 
  (void*)PosixFileDescr_WriterDesc_WriteByte, 
  (void*)PosixFileDescr_WriterDesc_WriteBytes, 
  (void*)Files_WriterDesc_ErrorDescr, 
  (void*)Channel_WriterDesc_ClearError, 
  (void*)Files_WriterDesc_Truncate
}};
struct _TD Files_WriterDesc__td = {
  NULL, 
  &Types_TypeDesc__td.td, 
  {
    Files_WriterDesc__tdb.btypes, 
    _tb2.tbprocs, 
    (const unsigned char*)_n3.name, 
    &Files__md.md, 
    2, 
    '0', '1', 
    sizeof(Files_WriterDesc), 
    &Files_ReaderDesc__td.td
  }
};

/* local strings */

/* --- end #include "Files.d" */


#define Channel_Flush 4


/* this variable holds a list of open files; it is used to remove any temporary
   files on program exit */
static Files_File open_files = NULL;

static _ModId _mid;


/* function definitions */

static INTEGER file_error(void) {
  switch (errno) {
  case EACCES:
    return Files_dirWriteDenied;
  case EISDIR:
    return Files_isDirectory;
  case EMFILE:
  case ENFILE: 
    return Files_tooManyFiles;
  case ENOENT:
    return Files_noSuchFile;
  case ENOSPC:
    return Files_directoryFull;
  case EROFS:
    return Files_readOnlyFileSystem;
  case ENAMETOOLONG:
    return Files_nameTooLong;
  case ENOTDIR:
    return Files_notDirectory;
#ifdef ELOOP
  case ELOOP:
    return Files_linkLoop;
#endif
  default:
    return Files_fileError;
  }
}

void Files_ErrorDescr(short int res, unsigned char* descr, int descr_0d) {
  char *str;

  if (res == Files_channelClosed) {
    str = "File has been closed";
  } else if (res == Files_noReadAccess) {
    str = "No read permission for file";
  } else if (res == Files_noWriteAccess) {
    str = "No write permission for file";
  } else if (res == Files_closeError) {
    str = "Error while closing the file";
  } else if (res == Files_accessDenied) {
    str = "Failed to open file with requested access rights";
  } else if (res == Files_isDirectory) {
    str = "Can't get write access to directory file";
  } else if (res == Files_tooManyFiles) {
    str = "Too many open files at the moment";
  } else if (res == Files_noSuchFile) {
    str = "The named file does not exist";
  } else if (res == Files_directoryFull) {
    str = "Can't add new files to directory";
  } else if (res == Files_readOnlyFileSystem) {
    str = "File system is read-only";
  } else if (res == Files_invalidTime) {
    str = "Invalid modification time";
  } else if (res == Files_notOwner) {
    str = "Must be owner of file to change its modification time";
  } else if (res == Files_anonymousFile) {
    str = "Can't register anonymous file";
  } else if (res == Files_dirWriteDenied) {
    str = "Don't have write permission for directory";
  } else if (res == Files_fileError) {
    str = "Failed to open the file";
  } else if (res == Files_nameTooLong) {
    str = "The file name or one of its components is too long";
  } else if (res == Files_notDirectory) {
    str = "A directory component of the file name exists, but isn't a directory";
  } else if (res == Files_linkLoop) {
    str = "Resolved too many symbolic links while looking up the file";
  } else {
    str = NULL;
  } 

  if (str) {
    _string_copy(descr, str, descr_0d);
  } else {
    PosixFileDescr_ErrorDescr(res, descr, descr_0d);
  }
}

void Files_FileDesc_ErrorDescr(Files_File f, unsigned char* descr, int descr_0d) {
  Files_ErrorDescr(f->res, descr, descr_0d);
}

static void free_tmp_name (Files_File f) {
/* pre: f->tmpName != NULL */
  int i;
  
  /* try to find an unused slot in tmp_name */
  i = 0;
  while ((i < TMP_MIN) && tmp_name[i]) {
    i++;
  }
  if (i < TMP_MIN) {		/* store name for later use */
    tmp_name[i] = (char*)f->tmpName;
  } else {			/* discard file name */
    GC_free(f->tmpName);
  }
  f->tmpName = NULL;
  if(f->name) {
    GC_free(f->name);
    f->name = NULL;
  }
}

void Files_FileDesc_Register(Files_File f) {
  if (!f->open) {
    f->res = Files_channelClosed;
  } else if (f->anonymous) {
    f->res= Files_anonymousFile; 
  } else {
    int res;
    
    /* flush file */
    TB_CALL(f, Channel_Flush, (void(*)(Files_File)), (f));
    
    /* the temporary file has only write permissions for the user; now 
       add group and others permissions to the file as far as the umask 
       allows it */
    res = chmod((const char*)f->tmpName, 
		(S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH) & 
		~active_umask);
    
    if (res != -1) {	/* rename file atomically */
      res = rename((const char*)f->tmpName, (const char*)f->name);
    }
    
    if (res == -1) {
      f->res = file_error();
    } else {
      free_tmp_name(f);
    }
  }
}

Files_Reader Files_FileDesc_NewReader(Files_File f) {
  Files_Reader r = NULL;

  if (!f->open) {
    f->res = Channel_channelClosed;
  } else if (f->readable) {
    NEW_REC(r, Files_ReaderDesc);
    PosixFileDescr_InitReader ((PosixFileDescr_Reader)r, (PosixFileDescr_Channel)f);
  } else {
    f->res = Channel_noReadAccess;
  }

  return r;
}

Files_Writer Files_FileDesc_NewWriter(Files_File f) {
  Files_Writer w = NULL;

  if (!f->open) {
    f->res = Channel_channelClosed;
  } else if (f->writable) {
    NEW_REC(w, Files_WriterDesc);
    PosixFileDescr_InitWriter ((PosixFileDescr_Writer)w, (PosixFileDescr_Channel)f);
  } else {
    f->res = Channel_noWriteAccess;
  }

  return w;
}

void Files_FileDesc_Close(Files_File f) {
  PosixFileDescr_ChannelDesc_Close ((PosixFileDescr_Channel)f);

  /* remove `f' from the list of open files */
  if (open_files == f) {
    open_files = f->next;
  } else {
    Files_File ptr = open_files;

    while (ptr->next != f) {
      ptr = ptr->next;
    }
    ptr->next = f->next;
  }

  /* delete temporary file and free its name */
  if (f->tmpName) {
    (void)unlink((const char*)f->tmpName);
    free_tmp_name(f);
  }
}

void Files_ReaderDesc_ErrorDescr(Files_Reader r, unsigned char* descr, int descr_0d) {
  Files_ErrorDescr(r->res, descr, descr_0d);
}

void Files_WriterDesc_ErrorDescr(Files_Writer w, unsigned char* descr, int descr_0d) {
  Files_ErrorDescr(w->res, descr, descr_0d);
}

void Files_WriterDesc_Truncate(Files_Writer w, int newLength) {
  PosixFileDescr_Truncate ((PosixFileDescr_Writer)w->base, newLength);
}



static int open_file(const unsigned char* name, int open_flags, int pflags) {
  int fd;

#ifdef O_BINARY  /* be kind to MS-DOG based systems */
  open_flags |= O_BINARY;
#endif

  do {
    fd = open((const char*)name, open_flags, pflags);
  } while ((fd == -1) && (errno == EINTR));
  return fd;
}
  

#define NO_ERROR ((fd == -2) || ((fd == -1) && (errno == EACCES)))
#define RD_FLAGS ((1<<Files_read) | (1<<Files_tryRead))
#define WR_FLAGS ((1<<Files_write) | (1<<Files_tryWrite))
#define MODE_OLD 0
#define MODE_NEW 1
#define MODE_TMP 2
#define MODE_TMP_GEN_NAME 3

static int call_open (const CHAR* name, SET flags, int mode, int *access_mode) {
  /* create a new file or open one; try to open the file first with read and 
     write, permissions, then just read, then just write; if everything fails
     report `access denied'; for temporary files the permissions on the file
     system are set to zero, i.e. even the user has no read/write permissions,
     except through the current file descriptor */
  int fd, open_flags, permissions;

  if (mode == MODE_TMP_GEN_NAME) {
    open_flags = O_CREAT|O_EXCL;
  } else if (mode == MODE_NEW) {
    /* get rid of any previous file, since open will use the permissions of
       the previous file otherwise; this would break New() if we don't have 
       the requested permissions for the existing file */
    if (flags & (RD_FLAGS|WR_FLAGS)) {
      (void)unlink((const char*)name);
    }
    open_flags = O_CREAT|O_TRUNC;
  } else if (mode == MODE_TMP) {
    open_flags = O_CREAT|O_TRUNC;
  } else {  /* MODE_OLD */
    open_flags = 0;
  }
  if (mode == MODE_NEW) {
    permissions = S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH;
  } else {
    permissions = S_IWUSR;
  }

  fd = -2;
  if ((flags & RD_FLAGS) && (flags & WR_FLAGS)) {
    fd = open_file(name, open_flags|O_RDWR, permissions);
    if ((fd == -1) && (errno == EACCES) &&
	(flags & (1<<Files_read)) &&
	(flags & (1<<Files_write))) {
      return -3;
    }
    *access_mode = PosixFileDescr_readWrite;
  } 
  if (NO_ERROR && (flags & RD_FLAGS)) {
    fd = open_file(name, open_flags|O_RDONLY, permissions);
    if ((fd == -1) && (errno == EACCES) && (flags & (1<<Files_read))) {
      return -3;
    }
    *access_mode = PosixFileDescr_readOnly;
  }
  if (NO_ERROR && (flags & WR_FLAGS)) {
    fd = open_file(name, open_flags|O_WRONLY, permissions);
    if ((fd == -1) && (errno == EACCES) && (flags & (1<<Files_write))) {
      return -3;
    }
    *access_mode = PosixFileDescr_writeOnly;
  }
  if (NO_ERROR) {
    return -3;
  }

  return fd;
}

static CHAR* local_strdup (const CHAR* str) {
  /* strdup is a SVID function; it can't be used in this file */
  CHAR *new;
  new = (CHAR*)GC_malloc_atomic(strlen((const char*)str)+1);
  if (new) (void)strcpy((char*)new, (const char*)str);
  return new;
}

static Files_File create_file(const CHAR* name, SET flags, int mode,
			      INTEGER *res) {
/* Open the file `name' using the flags in `flags' that were initially passed 
   to New() or Tmp().  `mode' characterizes one of the four ways a file can
   be treated when opening it: old file, new file, temporary file, and 
   anonymous temporary file.  An error code is returned in `res'.  */
  int fd, access_mode;
  Files_File ch = NULL;
  char tname[PATH_MAX+16];
  
  if (strlen((const char*)name) > PATH_MAX) {
    *res = Files_nameTooLong;
    return NULL;
  }

  if (mode == MODE_TMP_GEN_NAME) {
    /* lets try to find an unused name for our temporary file */
    int count = 0;
    
    do {
      /* repeat trying to open the file until an unused file is found, or
	 an error occured */
      if (count) {
	(void)sprintf(tname, "%s^%d", (const char*)name, count);
      } else {
	(void)sprintf(tname, "%s^", (const char*)name);
      }
      fd = call_open((const unsigned char*)tname, flags, mode, &access_mode);
      count++;
    } while ((fd == -1) && (errno == EEXIST));
  } else {
    fd = call_open(name, flags, mode, &access_mode);
  }
  
  if (fd == -3) {
    /* couldn't get the requested access rights */
    *res = Files_accessDenied;
  } else if (fd == -1) {
    /* some other error */
    *res = file_error();
  } else {
    /* file was opened with the requested access rights */
    NEW_REC(ch, Files_FileDesc);
    *res = Channel_done;
    PosixFileDescr_Init((PosixFileDescr_Channel)ch, fd, access_mode);
    ch->next = open_files;
    if (mode == MODE_TMP_GEN_NAME) {
      ch->tmpName = local_strdup((const CHAR*)tname);
    } else {
      ch->tmpName = NULL;
    }
    ch->name = NULL;
    ch->anonymous = 0;
    open_files = ch;
  }

  return ch;
}

Files_File Files_New(const unsigned char* file__ref, int file_0d, unsigned int flags, short int *res) {
  return create_file(file__ref, flags, MODE_NEW, res);
}

Files_File Files_Old(const unsigned char* file__ref, int file_0d, unsigned int flags, short int *res) {
  return create_file(file__ref, flags, MODE_OLD, res);
}

Files_File Files_Tmp(const unsigned char* file__ref, int file_0d, unsigned int flags, short int *res) {
  Files_File ch;
  char new_name[L_tmpnam];
  char *tname;
  int i, anonymous;
  
  anonymous = (!file__ref[0]);

  if (anonymous) {
    /* first check if we have an unused name in stock */
    i = 0;
    while ((i < TMP_MIN) && !tmp_name[i]) {
      i++;
    }
    if (i < TMP_MIN) {
      /* this is our lucky day, we found an unused name */
      tname = tmp_name[i];
      tmp_name[i] = NULL;
    } else {
      /* there aren't any discarded names available right now; try to 
	 get a new one */
      tname = tmpnam(new_name);
      if (tname) tname = (char*)local_strdup((CHAR*)tname);
    }
  } else {
    tname = (char*)file__ref;
  }

  if (tname) {
    /* create file with minimal permissions; the permissions are extended
       upon registration if the umask allows it */
    ch = create_file((const unsigned char*)tname, flags, 
		     file__ref[0]?MODE_TMP_GEN_NAME:MODE_TMP, res);
    if (ch) {
      ch->anonymous = anonymous;
      if (anonymous) {
	ch->tmpName = (CHAR*)tname;
      } else {
	ch->name = local_strdup(file__ref);
      }
    }
    return ch;
  } else {
    *res = Channel_noTmpName;
    return NULL;
  }
}

/* define the day count of the Unix epoch (Jan 1 1970 00:00:00 GMT) for the
   Time.TimeStamp format */
#define days_to_epoch 40587
#define end_of_epoch 65442
#define secs_per_day 86400

void Files_SetModTime(const unsigned char* file__ref, int file_0d, const Time_TimeStamp *mtime__ref, short int *fres) {
  if ((mtime__ref->days < days_to_epoch) || 
      (mtime__ref->days >= end_of_epoch) ||
      (mtime__ref->msecs < 0) ||
      (mtime__ref->msecs > secs_per_day*1000)) {
    *fres = Files_invalidTime;
  } else {
    int res;
    int seconds = (mtime__ref->days - days_to_epoch) * secs_per_day +
                  (mtime__ref->msecs / 1000);
#if HAVE_UTIMES			/* we have microsecond resolution */
    struct timeval tpv[2];
    
    tpv[0].tv_sec = tpv[1].tv_sec = seconds;
    tpv[0].tv_usec = tpv[1].tv_usec = (mtime__ref->msecs % 1000) * 1000;
    res = utimes((char*)file__ref, tpv);
#else                           /* we only have second resolution */
    struct utimbuf times;
    
    times.actime = seconds;
    times.modtime = seconds;
    res = utime((const char*)file__ref, &times);
#endif

    if (res == ENOENT) {
      *fres = Files_noSuchFile;
    } else if (res == EPERM) {
      *fres = Files_notOwner;
    } else if (res == EROFS) {
      *fres = Files_readOnlyFileSystem;
    } else {
      *fres = Files_done;
    }
  }
}

/* define the day count of the Unix epoch (Jan 1 1970 00:00:00 GMT) for the
   Time.TimeStamp format */
#define days_to_epoch 40587
#define secs_per_day 86400

void Files_GetModTime(const CHAR* file__ref, int file_0d, Time_TimeStamp *mtime, _Type mtime__tag, short int *fres) {
  int res;
  struct stat stat_buf;

  res = stat((const char*)file__ref, &stat_buf);
  if (res == -1) {
    *fres = file_error();
  } else {
    mtime->days = days_to_epoch + stat_buf.st_mtime / secs_per_day;
    mtime->msecs = (stat_buf.st_mtime % secs_per_day) * 1000;
#if HAVE_ST_MTIME_USEC
    mtime->msecs += (stat_buf.st_mtime_usec / 1000);
#endif
    *fres = Channel_done;
  }
}

extern BOOLEAN Files_Exists(const unsigned char* file__ref, int file_0d) {
  return (access((const char*)file__ref, F_OK) == 0);
}

static void close_all_files (void) {
  while (open_files) {
    Files_FileDesc_Close (open_files);
  }
}

void Files__init(void) {
  int i;

  _mid = _register_module(&Files__md.md, &Files_WriterDesc__td.td);
  active_umask = umask (0);
  umask (active_umask);
  for(i=0; i<TMP_MIN; i++) {
    tmp_name[i] = NULL;
  }
  /* make sure that all files are closed upon program termination */
  Termination_RegisterProc (&close_all_files);
}
