(*	$Id: Log.Mod,v 1.2 1998/09/22 14:36:26 acken Exp $	*)
MODULE Log;
(*  Simple logging facility to stderr.
    Copyright (C) 1998 Michael van Acken
 
    This module is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
 
    This module is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  *)

   
IMPORT
  SYSTEM, StdChannels, TextRider, Types;

VAR
  writer-: TextRider.Writer;
  (* This is the rider used by the output procedures.  Initialized to hold a
     text writer to the `StdChannels.stderr' channel.  Can be changed by 
     calling SetWriter.  *)
     
PROCEDURE Done* () : BOOLEAN;
(* Returns TRUE if the last operation was successful.  *)
  BEGIN
    RETURN writer. Res() = TextRider.done
  END Done;

PROCEDURE ClearError*;
  BEGIN
    writer. ClearError
  END ClearError;


PROCEDURE SetWriter* (w: TextRider.Writer);
(* Changes the rider `writer'.  All following calls to output procedures will 
   write to `w'.  The preferred method of connecting to the standard error
   channel is with a call like SetWriter(NIL).
 *)
  BEGIN
    IF w=NIL THEN writer:=TextRider.ConnectWriter (StdChannels.stderr)
    ELSE writer:=w
    END  
  END SetWriter;

PROCEDURE Flush*;
(* Flushes all buffers associated with `writer'.  *)
  BEGIN
    writer. base. Flush
  END Flush;


PROCEDURE Char* (msg: ARRAY OF CHAR; ch: CHAR);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteChar (ch);
    writer. WriteLn
  END Char;
  
PROCEDURE String* (msg: ARRAY OF CHAR; s: ARRAY OF CHAR);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteString (s);
    writer. WriteLn
  END String;
  
PROCEDURE Bool* (msg: ARRAY OF CHAR; bool: BOOLEAN);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteBool (bool);
    writer. WriteLn
  END Bool;
  
PROCEDURE LongInt* (msg: ARRAY OF CHAR; lint: LONGINT);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteLInt (lint, 0);
    writer. WriteLn
  END LongInt;

PROCEDURE ShortInt* (msg: ARRAY OF CHAR; sint: SHORTINT);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteSInt (sint, 0);
    writer. WriteLn
  END ShortInt;

PROCEDURE Int* (msg: ARRAY OF CHAR; int: INTEGER);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteInt (int, 0);
    writer. WriteLn
  END Int;

PROCEDURE Hex* (msg: ARRAY OF CHAR; lint: LONGINT);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteHex (lint, 8);
    writer. WriteLn
  END Hex;
 
PROCEDURE LongReal* (msg: ARRAY OF CHAR; lreal: LONGREAL);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteLReal (lreal, 24, 17);
    writer. WriteLn
  END LongReal;

PROCEDURE Real* (msg: ARRAY OF CHAR; real: REAL);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteReal (real, 16, 9);
    writer. WriteLn
  END Real;

PROCEDURE LongRealEng* (lreal: LONGREAL);
  BEGIN
    writer. WriteLRealEng (lreal, 26, 17)
  END LongRealEng;

PROCEDURE RealEng* (real: REAL);
  BEGIN
    writer. WriteRealEng (real, 18, 9)
  END RealEng;
  
PROCEDURE Set* (msg: ARRAY OF CHAR; s: SET);
  BEGIN
    writer. WriteString (msg);
    writer. WriteString (": ");
    writer. WriteSet (s);
    writer. WriteLn
  END Set;

PROCEDURE Msg* (msg: ARRAY OF CHAR);
  BEGIN
    writer. WriteString (msg);
    writer. WriteLn
  END Msg;

PROCEDURE Ln*;
  BEGIN
    writer. WriteLn
  END Ln;

PROCEDURE Ptr* (msg: ARRAY OF CHAR; ptr: SYSTEM.PTR);
  BEGIN
    IF (ptr = NIL) THEN
      String (msg, "NIL")
    ELSE
      Hex (msg, SYSTEM.VAL (SYSTEM.ADDRESS, ptr))
    END
  END Ptr;

PROCEDURE Adr* (msg: ARRAY OF CHAR; VAR x: ARRAY OF SYSTEM.BYTE);
  BEGIN
    Hex (msg, SYSTEM.ADR (x))
  END Adr;

PROCEDURE Type* (msg: ARRAY OF CHAR; ptr: SYSTEM.PTR);
(* pre: `ptr' is a POINTER TO R, where R is a record type whose type is 
     declared in a normal module.  'o' must have a legal value, i.e., it is
     either NIL or was initialized through NEW.  *)
  VAR
    type: Types.Type;
  BEGIN
    IF (ptr = NIL) THEN
      String (msg, "NIL")
    ELSE
      type := Types.TypeOf (ptr);
      writer. WriteString (msg);
      writer. WriteString (": ");
      writer. WriteString (type. module. name^);
      writer. WriteString (".");
      writer. WriteString (type. name^);
      writer. WriteLn
    END
  END Type;

BEGIN
  SetWriter(NIL) 
END Log.
