/*
 * PDBNET.C - routines to run over networks (special stuff)
 *
 * Routines and Usage:
 *
 *   PN_target   - returns a structure chart for the target machine
 *               - given the apropriate data_standard and data_alignment
 *               - used either to target remote machine as source or sink
 *               - (see PN_conv_in and PN_conv_out)
 *
 *   PN_open     - open a PDBfile which encapsulates a memory buffer
 *               - into and out of which all IO goes
 *
 *   PN_close    - close a PDBfile which was opened by PN_open
 *
 *   PN_defstr   - defines data structures to the two relevant structure
 *               - charts (host chart and target chart)
 *
 *   PN_conv_in  - convert the data in the input buffer which is in the
 *               - format of the remote machine (from which it has been read)
 *               - to the format of the host machine and place it in the
 *               - output buffer
 *
 *   PN_conv_out - convert the data in the output buffer which is in the
 *               - format of the host machine (from which it will be written)
 *               - to the format of the remote machine and place it in the
 *               - output buffer
 *
 *   PN_sizeof   - return the size of an item of the specified type
 *               - on the target machine (this is a macro)
 *
 *  These routines do NO network communications, they are provided so that
 *  PDBLib's data conversion and structured data handling facilities can
 *  be integrated with any network communication software.
 *
 * Source Version: 9.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pdb.h"

typedef struct BF_FILE_s BF_FILE;

struct BF_FILE_s
   {long length;
    SC_address addr;
    SC_address bf;};

#define MEM(x)   ((x)->addr.memaddr)
#define DISK(x)  ((x)->addr.diskaddr)

static PFfopen io_open_save;
static PFftell io_tell_save;
static PFfread io_read_save;
static PFfwrite io_write_save;
static PFsetvbuf io_setvbuf_save;
static PFfclose io_close_save;
static PFfseek io_seek_save;
static PFfprintf io_printf_save;
static PFfputs io_puts_save;
static PFfgetc io_getc_save;
static PFungetc io_ungetc_save;
static PFfflush io_flush_save;
static PFfgets io_gets_save;

static HASHTAB
 *_PN_host_chart;

static data_standard
 *_PN_host_std;

static data_alignment
 *_PN_host_align;

static char
 _PN_bf[LRG_TXT_BUFFER];

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BOPEN - prepare the pseudo file */

static FILE *_PN_bopen(name, mode)
   char *name, *mode;
   {BF_FILE *fp;

    fp = FMAKE(BF_FILE, "_PN_BOPEN:fp");

    fp->length       = SC_arrlen(name);
    fp->bf.memaddr   = name;
    fp->addr.memaddr = name;

    return((FILE *) fp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BSETVBUF - set the pseudo file buffer
 *              - binds to io_setvbuf_hook
 */

static int _PN_bsetvbuf(stream, bf, type, size)
   FILE *stream;
   char *bf;
   int type;
   size_t size;
   {BF_FILE *fp;
    int ret;

    ret = TRUE;
    fp  = (BF_FILE *) stream;
    if (fp == NULL)
       ret = FALSE;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       ret = setvbuf(stream, bf, type, size);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BCLOSE - close the pseudo file
 *            - binds to io_close_hook
 */

static int _PN_bclose(stream)
   FILE *stream;
   {BF_FILE *fp;
    int ret;

    ret = 0;
    fp  = (BF_FILE *) stream;
    if (fp == NULL)
       ret = EOF;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       ret = fclose(stream);

    SFREE(fp);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BFLUSH - do an fflush on the pseudo file
 *            - binds to io_flush_hook
 */

static int _PN_bflush(stream)
   FILE *stream;
   {int ret;
    BF_FILE *fp;

    ret = FALSE;
    fp  = (BF_FILE *) stream;
    if (fp == NULL)
       ret = TRUE;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       ret = fflush(stream);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BTELL - do an ftell on the pseudo file
 *           - binds to io_tell_hook
 */

static long _PN_btell(stream)
   FILE *stream;
   {BF_FILE *fp;
    long addr;

    addr = -1L;
    fp   = (BF_FILE *) stream;
    if (fp == NULL)
       addr = -1L;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       addr = ftell(stream);

    else
       addr = DISK(fp);

    return(addr);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BSEEK - do an fseek on the pseudo file
 *           - binds to io_seek_hook
 */

static int _PN_bseek(stream, addr, offset)
   FILE *stream;
   long addr;
   int offset;
   {int ret;
    BF_FILE *fp;

    ret = 0;
    fp  = (BF_FILE *) stream;
    if (fp == NULL)
       ret = -1;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       ret = fseek(stream, addr, offset);

    else
       {switch (offset)
           {case SEEK_SET :
		 DISK(fp) = addr;
                 break;

            case SEEK_CUR :
		 DISK(fp) += addr;
                 break;

            case SEEK_END :
		 DISK(fp) = fp->bf.diskaddr + fp->length + addr;
                 break;

            default :
	         ret = -1;
	         break;};};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BREAD - do an fread on the pseudo file
 *           - binds to io_read_hook
 */

static size_t _PN_bread(s, nbi, ni, stream)
   char *s;
   size_t nbi, ni;
   FILE *stream;
   {BF_FILE *fp;
    size_t nbw;

    nbw = nbi*ni;
    fp  = (BF_FILE *) stream;
    if (fp == NULL)
       nbw = 0;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       nbw = fread(s, nbi, ni, stream);

    else
       {memcpy(s, MEM(fp), nbw);

/* adjust the current address */
        DISK(fp) += nbw;};

    return(ni);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BWRITE - do an fwrite on the pseudo file
 *            - binds to io_write_hook
 */

static size_t _PN_bwrite(s, nbi, ni, stream)
   char *s;
   size_t nbi, ni;
   FILE *stream;
   {long nbw;
    BF_FILE *fp;

    nbw = nbi*ni;
    fp  = (BF_FILE *) stream;
    if ((fp == NULL) || (nbi*ni == 0))
       nbw = 0;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       nbw = fwrite(s, nbi, ni, stream);

    else
       {memcpy(MEM(fp), s, nbw);

/* adjust the current address */
        DISK(fp) += nbw;};

    return(ni);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BPRINTF - do an fprintf on the pseudo file
 *             - binds to io_puts_hook
 */

#ifdef PCC

int _PN_bprintf(fp, fmt, va_alist)
   FILE *fp;
   char *fmt;
   va_dcl

#endif

#ifdef ANSI

int _PN_bprintf(FILE *fp, char *fmt, ...)

#endif

   {size_t ni, nw;

    nw = 0;

    if (fp != NULL)
       {SC_VA_START(fmt);
	SC_VSPRINTF(_PN_bf, fmt);
	SC_VA_END;

	ni = strlen(_PN_bf);
	nw = io_write(_PN_bf, (size_t) 1, ni, fp);};

    return((int) nw);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BPUTS - do an fputs on the pseudo file
 *           - binds to io_puts_hook
 */

static int _PN_bputs(s, stream)
   char *s;
   FILE *stream;
   {int nc;
    long nbw;
    BF_FILE *fp;

    fp = (BF_FILE *) stream;
    if (fp == NULL)
       nbw = 0;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       nbw = fputs(s, stream);

    else
       {nc  = strlen(s);
	nbw = io_write(s, 1, nc, stream);};

    return(nbw);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BGETS - do an fgets on the pseudo file
 *           - binds to io_gets_hook
 */

static char *_PN_bgets(s, nc, stream)
   char *s;
   int nc;
   FILE *stream;
   {int ns;
    char *pb, *ps;
    BF_FILE *fp;

    ps = NULL;
    fp = (BF_FILE *) stream;
    if (fp == NULL)
       ps = NULL;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       ps = fgets(s, nc, stream);

    else
       {pb = strchr(MEM(fp), '\n');
	if (pb != NULL)
	   {ns = pb - MEM(fp) + 1;
	    ns = min(nc, ns);
	    memcpy(s, MEM(fp), ns);
	    s[ns] = '\0';

/* adjust the local idea of the current disk address */
	    DISK(fp) += ns;};};

    return(ps);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BGETC - do an fgetc on the pseudo file
 *           - binds to io_getc_hook
 */

static int _PN_bgetc(stream)
   FILE *stream;
   {int c;
    BF_FILE *fp;

    fp = (BF_FILE *) stream;
    if (fp == NULL)
       c = EOF;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       c = fgetc(stream);

    else
       c = *MEM(fp)++;

    return(c);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_BUNGETC - do an fungetc on the pseudo file
 *             - binds to io_ungetc_hook
 */

static int _PN_bungetc(c, stream)
   int c;
   FILE *stream;
   {BF_FILE *fp;

    fp = (BF_FILE *) stream;
    if (fp == NULL)
       c = 0;

    else if ((stream == stdout) || (stream == stdin) || (stream == stderr))
       c = ungetc(c, stream);

    else
       MEM(fp)--;

    return(c);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_SAVE_HOOKS - save the current io_hooks */

static void _PN_save_hooks()
   {static int first = TRUE;

    if (first)
       {first = FALSE;

	io_open_save    = io_open_hook;
	io_tell_save    = io_tell_hook;
	io_read_save    = io_read_hook;
	io_write_save   = io_write_hook;
	io_printf_save  = io_printf_hook;
	io_setvbuf_save = io_setvbuf_hook;
	io_close_save   = io_close_hook;
	io_seek_save    = io_seek_hook;
	io_puts_save    = io_puts_hook;
	io_getc_save    = io_getc_hook;
	io_ungetc_save  = io_ungetc_hook;
	io_flush_save   = io_flush_hook;
	io_gets_save    = io_gets_hook;};

    io_open_hook    = (PFfopen) _PN_bopen;
    io_tell_hook    = (PFftell) _PN_btell;
    io_read_hook    = (PFfread) _PN_bread;
    io_write_hook   = (PFfwrite) _PN_bwrite;
    io_printf_hook  = (PFfprintf) _PN_bprintf;
    io_setvbuf_hook = (PFsetvbuf) _PN_bsetvbuf;
    io_close_hook   = (PFfclose) _PN_bclose;
    io_seek_hook    = (PFfseek) _PN_bseek;
    io_puts_hook    = (PFfputs) _PN_bputs;
    io_getc_hook    = (PFfgetc) _PN_bgetc;
    io_ungetc_hook  = (PFungetc) _PN_bungetc;
    io_flush_hook   = (PFfflush) _PN_bflush;
    io_gets_hook    = (PFfgets) _PN_bgets;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_RESTORE_HOOKS - restore the io_hooks */

static void _PN_restore_hooks()
   {

    io_open_hook    = io_open_save;
    io_tell_hook    = io_tell_save;
    io_read_hook    = io_read_save;
    io_write_hook   = io_write_save;
    io_setvbuf_hook = io_setvbuf_save;
    io_close_hook   = io_close_save;
    io_seek_hook    = io_seek_save;
    io_puts_hook    = io_puts_save;
    io_getc_hook    = io_getc_save;
    io_ungetc_hook  = io_ungetc_save;
    io_flush_hook   = io_flush_save;
    io_gets_hook    = io_gets_save;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PN_SIZEOF - sizeof operator for PDBNet */

static int _PN_sizeof(s)
   char *s;
   {

    if (_PN_host_chart != NULL)
       return(_PD_lookup_size(s, _PN_host_chart));

    else
       return(SC_sizeof(s));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PN_CONV_IN - convert from one machine format to another after input
 *            - from a remote host with different architecture
 *            - NITEMS of type, TYPE, from IN and put them in OUT
 *            - all additional information comes from OUT_CHART
 */

void PN_conv_in(out, in, type, nitems, in_chart)
   byte *out, *in;
   char *type;
   long nitems;
   HASHTAB *in_chart;
   {long ino, outo;
    data_standard *istd;

    switch (setjmp(_PD_trace_err))
       {case ABORT    : return;
        case ERR_FREE : return;
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    ino  = outo = 0L;
    istd = (data_standard *) SC_def_lookup("standard", in_chart);

    PD_convert((char **) &out, (char **) &in, type, type, nitems,
               istd, _PN_host_std, _PN_host_std, &ino, &outo,
               in_chart, _PN_host_chart, 0, PD_TRACE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PN_CONV_OUT - convert from one machine format to another before output
 *             - to a remote host with different architecture
 *             - NITEMS of type, TYPE, from IN and put them in OUT
 *             - all additional information comes from OUT_CHART
 */

void PN_conv_out(out, in, type, nitems, out_chart)
   byte *out, *in;
   char *type;
   long nitems;
   HASHTAB *out_chart;
   {long ino, outo;
    data_standard *ostd;

    switch (setjmp(_PD_trace_err))
       {case ABORT    : return;
        case ERR_FREE : return;
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    ino  = outo = 0L;
    ostd = (data_standard *) SC_def_lookup("standard", out_chart);

    PD_convert((char **) &out, (char **) &in, type, type, nitems,
               _PN_host_std, ostd, _PN_host_std, &ino, &outo,
               _PN_host_chart, out_chart, 0, PD_TRACE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PN_TARGET - allocate, initialize, and return a structure chart
 *           - for the associated standard and alignemnt
 *           - for network PDB
 */

HASHTAB *PN_target(std, align)
   data_standard *std;
   data_alignment *align;
   {HASHTAB *chart;
    static int first = TRUE;

/* if the first time initialize some stuff */
    if (PD_DEFSTR_S == NULL)
       {LAST           = FMAKE(int, "PN_TARGET:LAST");
        *LAST          = 0;
        PD_DEFSTR_S    = SC_strsavef("defstr *", "char*:PN_TARGET:defstr");
        PD_SYMENT_S    = SC_strsavef("syment *", "char*:PN_TARGET:syment");
        PD_STANDARD_S  = SC_strsavef("data_standard", "char*:PN_TARGET:ds");
        PD_ALIGNMENT_S = SC_strsavef("data_alignment",
                                        "char*:PN_TARGET:da");};

/* initialize the host chart the first time only */
    if (first)
       {first          = FALSE;
	_PN_host_std   = _PD_copy_standard(INT_STANDARD);
	_PN_host_align = _PD_copy_alignment(INT_ALIGNMENT);
        _PN_host_chart = PN_target(_PN_host_std, _PN_host_align);};

/* initialize the chart */
    chart  = SC_make_hash_table(1, NODOC);

    _PD_setup_chart(chart, std,
		    _PN_host_std, align, _PN_host_align, TRUE);

/* special hack for jollies and to save typing */
    SC_install("standard", std, PD_STANDARD_S, chart);
    SC_install("alignment", align, PD_ALIGNMENT_S, chart);

    return(chart);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PN_DEFSTR - a structure definition mechanism for PDBLib
 *           -
 *           - sample syntax:
 *           -
 *           -   PN_defstr(<PDB file>, "<struct name>",
 *           -                         "<member1>", "<member2>",
 *           -                 ...     "<membern>", LAST);
 *           -
 *           - where
 *           - 
 *           -   <member> := <primitive type> <member name>[(<dimensions>)] |
 *           -               <derived type> <member name>[(<dimensions>)]
 *           -
 *           -   <dimensions> := <non-negative int> |
 *           -                   <non-negative int>,<dimensions> |
 *           -                   <non-negative int>, <dimensions> |
 *           -                   <non-negative int> <dimensions>
 *           -
 *           -   <primitive type> := short | integer | long | float |
 *           -                       double | char | short * | integer *
 *           -                       long * | float * | double * | char *
 *           - 
 *           -   <derived type> := any defstr'd type | any defstr'd type *
 *           -
 *           - LAST is a pointer to a integer zero and is specifically
 *           - allocated by PDBLib to be used to terminate argument lists
 *           - which consist of pointers
 *           -
 *           - Returns NULL if member types are unknown
 */

#ifdef PCC

defstr *PN_defstr(chart, name, align, defoff, va_alist)
   HASHTAB *chart;
   char *name;
   data_alignment *align;
   int defoff;
   va_dcl

#endif

#ifdef ANSI

defstr *PN_defstr(HASHTAB *chart, char *name, data_alignment *align,
                  int defoff, ...)

#endif

   {char *nxt, *ptype, *type;
    defstr *dp;
    memdes *desc, *lst, *prev;

    SC_sizeof_hook = _PN_sizeof;

    SC_VA_START(defoff);

    prev   = NULL;
    lst    = NULL;
    for (nxt = SC_VA_ARG(char *); (int) *nxt != 0;
         nxt = SC_VA_ARG(char *))
        {desc  = _PD_mk_descriptor(nxt, defoff);
         type  = SC_strsavef(nxt, "char*:PN_DEFSTR:type");
         ptype = SC_firsttok(type, " \n");
         if (SC_lookup(ptype, chart) == NULL)
            if ((strcmp(ptype, name) != 0) || !_PD_indirection(nxt))
               {sprintf(PD_err, "ERROR: %s BAD MEMBER TYPE - PN_DEFSTR\n",
                                nxt);
                return(NULL);};
         SFREE(type);
         if (lst == NULL)
            lst = desc;
         else
            prev->next = desc;
         prev = desc;};

    SC_VA_END;

/* install the type in both charts */
    dp = _PD_defstr_inst(name, lst, -1, NULL, NULL,
                         chart, _PN_host_chart,
                         align, _PN_host_align, FALSE);
    if (dp == NULL)
       sprintf(PD_err, "ERROR: CAN'T HANDLE PRIMITIVE TYPE - PN_DEFSTR\n");

    return(dp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PN_OPEN - create a special pseudo PDBfile and set all the
 *         - io hooks appropriately
 */

PDBfile *PN_open(fm, bf)
   PDBfile *fm;
   char *bf;
   {PDBfile *file;
    FILE *fp;

    _PN_save_hooks();

    switch (setjmp(_PD_create_err))
       {case ABORT    : io_close(fp);
	                _PN_restore_hooks();
                        return(NULL);
        case ERR_FREE : return(file);
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    fp = io_open(bf, BINARY_MODE_WPLUS);

    if (fp == NULL)
       PD_error("CAN'T CREATE FILE - PN_OPEN", PD_CREATE);

/* make the PDBfile */
    file = FMAKE(PDBfile, "PN_OPEN:file");
    if (file == NULL)
       PD_error("CAN'T ALLOCATE PDBFILE - PN_OPEN", PD_CREATE);

/* copy over stuff from the template file */
    *file = *fm;

    file->name             = SC_strsavef("pseudo", "char*:PN_OPEN:name");
    file->stream           = fp;
    file->mode             = PD_APPEND;
    file->chrtaddr         = io_tell(fp);
    file->system_version   = PDB_SYSTEM_VERSION;
    file->date             = SC_date();
    file->virtual_internal = FALSE;
    file->current_prefix   = NULL;
    file->symtab           = SC_make_hash_table(HSZSMALL, NODOC);

    return(file);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PN_CLOSE - close the special pseudo PDBfile and reset all the
 *          - io hooks
 */

int PN_close(file)
   PDBfile *file;
   {int ret;
    FILE *fp;

    fp  = file->stream;
    ret = TRUE;

    if (io_close(fp) != 0)
       PD_error("CAN'T CLOSE FILE - PD_CLOSE", PD_CLOSE);

    _PN_restore_hooks();

/* free the space */
    PD_reset_ptr_list(file);

    _PD_clr_table(file->symtab, _PD_rl_syment_d);

    SFREE(file->current_prefix);
    SFREE(file->date);
    SFREE(file->name);
    SFREE(file);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PN_WRITE - write the given data to the special pseudo PDBfile */

int PN_write(file, type, nitems, vr)
   PDBfile *file;
   char *type;
   long nitems;
   byte *vr;
   {int ret;
    char bf[MAXLINE];

    PD_reset_ptr_list(file);

    sprintf(bf, "s[%ld]", nitems);

    ret = PD_write(file, bf, type, vr);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PN_READ - read the given data from the special pseudo PDBfile */

int PN_read(file, type, nitems, vr)
   PDBfile *file;
   char *type;
   long nitems;
   byte *vr;
   {int ret;
    char bf[MAXLINE];
    syment *ep;
    FILE *fp;

    PD_reset_ptr_list(file);

    fp = file->stream;

    sprintf(bf, "s");

    ep = _PD_mk_syment(type, nitems, io_tell(fp), NULL, NULL);
    _PD_e_install(bf, ep, file->symtab);

    ret = PD_read(file, bf, vr);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

