/*
 * GSAXIS.C - axis routines for Portable Graphics System
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

#define EPSILON     1.0e-4
#define MIN_DECADE  0.1

#define DELTA(del, c, lower, upper, flag, logs)                              \
    {if (flag)                                                               \
        {del = 1.0 + logs*ABS(c*log10(upper/lower));                         \
         if (c < 0.0)                                                        \
	    del = 1.0/del;                                                   \
         if (upper < lower)                                                  \
	    del = 1.0/del;}                                                  \
     else                                                                    \
        del = c*(upper - lower);}

#define HIGH   1.000001
#define LOW    0.999999
#define LOW10  9.9999999

/* axis drawing attributes - these define the axis */

int
 _PG_axis_char_size       = 1,
 _PG_axis_grid_on         = FALSE,
 _PG_axis_grid_style      = 3,
 _PG_axis_line_style      = SOLID,
 _PG_axis_max_major_ticks = 10,
 _PG_axis_n_minor_ticks   = -1,
 _PG_axis_n_minor_x_ticks = -1,
 _PG_axis_n_minor_y_ticks = -1,
 _PG_axis_on              = TRUE,
 _PG_axis_tick_type       = RIGHT_OF_AXIS,
 _PG_axis_type            = CARTESIAN;

REAL
 _PG_axis_char_angle       = 0.0,
 _PG_axis_label_x_standoff = 2.0,
 _PG_axis_label_y_standoff = 2.0,
 _PG_axis_line_width       = 0.1,
 _PG_axis_major_tick_size  = 0.018,
 _PG_axis_n_decades        = 8.0;

char
 *_PG_axis_label_x_format,
 *_PG_axis_label_y_format,
 *_PG_axis_type_face = NULL;

/* these are not static so the test driver GSLAXT can access them */

PG_axis_def
 SC_DECLARE(*PG_mk_axis_def,
         (int atype, int ttype, int ltype,
          double xl, double yl, double xr, double yr,
          double t1, double t2, double v1, double v2,
          double as, int xlog, int ylog));

void
 SC_DECLARE(_PG_tick, (PG_axis_def *ad, int tick));


static void
 SC_DECLARE(_PG_aux_axis, (PG_device *dev, int axis_type)),
 SC_DECLARE(_PG_draw_polar_grid_lines,
         (PG_device *dev, PG_axis_def *ad,
          double xmn, double ymn, double xmx, double ymx)),
 SC_DECLARE(_PG_draw_grid_lines,
         (PG_device *dev, PG_axis_def *ad,
          double x1n, double y1n, double x1x, double y1x,
          double x2n, double y2n, double x2x, double y2x,
          int logsp)),
 SC_DECLARE(_PG_comp_tick_spacings, (PG_axis_def *ad)),
 SC_DECLARE(_PG_log_tick,
         (double v1, double v2, int *pn,
          double *pva, double *pvb, REAL **pd, int tick)),
 SC_DECLARE(_PG_write_label,
         (PG_device *dev, char *format, double xa, double ya,
          double lss, double tol, double fx, double fy, int flg)),
 SC_DECLARE(_PG_sprintf,
         (char *string, char *format, double value, double tol));

static int
 SC_DECLARE(_PG_draw_tick,
         (PG_device *dev, PG_axis_def *ad, double sz, int tick)),
 SC_DECLARE(_PG_draw_label, 
            (PG_device *dev, PG_axis_def *ad, char *fmt));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_AXIS - draw the axes
 *         - the information for this is set up by a setlimits call
 */
 
void PG_axis(dev, axis_type)
   PG_device *dev;
   int axis_type;
   {PG_dev_attributes *attr;

    if (_PG_axis_on == FALSE)
       return;

/* save user's values for various attributes */
    attr = PG_get_attributes(dev);

    switch (axis_type)
       {case INSEL     : PG_set_axis_attributes(dev,
                            0);
                         _PG_aux_axis(dev, INSEL);
                         break;

        case POLAR     : if (dev->grid == ON)          /* dotted grid lines */
                            {PG_set_axis_attributes(dev,
                                AXIS_GRID_ON, TRUE,
                                0);
                             _PG_aux_axis(dev, POLAR);}
                         else
                            {PG_set_axis_attributes(dev,
                                0);
                             _PG_aux_axis(dev, POLAR);};
                         break;

        case CARTESIAN : if (dev->grid == ON)          /* dotted grid lines */
                            {PG_set_axis_attributes(dev,
                                                    AXIS_GRID_ON, TRUE,
                                                    0);
                             _PG_aux_axis(dev, CARTESIAN);}
                         else
                            {PG_set_axis_attributes(dev,
                                0);
                             _PG_aux_axis(dev, CARTESIAN);}
        default        : break;};

/* reset user's values for various attributes */
    PG_set_attributes(dev, attr);
    SFREE(attr);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_AXIS_ATTRIBUTES - set the parameters which control the look
 *                        - of the axes being drawn
 *                        -
 *                        - the control parameters are:
 *                        -
 *                        - AXIS_LINESTYLE    - the style of the lines
 *                        -                     (see PG_set_line_style)
 *                        - AXIS_LINETHICK    - the thickness of the lines
 *                        -                     (see PG_set_line_width)
 *                        - AXIS_LINECOLOR    - the color of the lines
 *                        -                     (see PG_set_color_line)
 *                        - AXIS_LABELCOLOR   - the color of the labels
 *                        -                     (see PG_set_color_text)
 *                        - AXIS_LABELFONT    - the label font type_face
 *                        -                     (see PG_set_font)
 *                        - AXIS_LABELPREC    - the character precision
 *                        -                     (see PG_set_char_precision)
 *                        - AXIS_X_FORMAT     - the format of the x labels
 *                        -                     (e.g. %10.2f)
 *                        - AXIS_Y_FORMAT     - the format of the y labels
 *                        -                     (e.g. %10.2f)
 *                        - AXIS_TICKSIZE     - set the tick size in
 *                        -                     fraction of axis length
 *                        - AXIS_GRID_ON      - turn on grid iff TRUE
 *                        - AXIS_CHAR_ANGLE   - orientation angle for label
 *                        -                     characters
 */

#ifdef PCC

int PG_set_axis_attributes(dev, va_alist)
   PG_device *dev;
   va_dcl

#endif

#ifdef ANSI

int PG_set_axis_attributes(PG_device *dev, ...)

#endif

   {int type, linecolor, txtcolor, prec;
    REAL charspace, chupx, chupy, chpthx, chpthy;

/* load default values for external variables */
    linecolor     = dev->WHITE;
    txtcolor      = dev->WHITE;
    prec          = TEXT_CHARACTER_PRECISION;

    if (_PG_axis_label_x_format == NULL)
       _PG_axis_label_x_format = SC_strsavef("%10.2g",
                               "char*:PG_SET_AXIS_ATTRIBUTES:xlab_format");

    if (_PG_axis_label_y_format == NULL)
       _PG_axis_label_y_format = SC_strsavef("%10.2g",
                               "char*:PG_SET_AXIS_ATTRIBUTES:ylab_format");

    if (_PG_axis_type_face == NULL)
       _PG_axis_type_face = SC_strsavef("helvetica",
                             "char*:PG_SET_AXIS_ATTRIBUTES:type_face");

    charspace    = 0.0;
    chpthx       = 1.0;
    chpthy       = 0.0;
    chupx        = 0.0;
    chupy        = 1.0;

    _PG_axis_grid_on = FALSE;

    SC_VA_START(dev);

/* the type of the second part of the pair is dependent on the
 * value of type 
 */
    while ((type = SC_VA_ARG(int)) != 0)
       {switch (type)
           {case AXIS_LINESTYLE :
	         _PG_axis_grid_style = SC_VA_ARG(int);
		 break;

            case AXIS_LINETHICK :
	         _PG_axis_line_width = SC_VA_ARG(double);
		 break;

            case AXIS_LINECOLOR :
	         linecolor = SC_VA_ARG(int);
		 break;

            case AXIS_LABELCOLOR :
	         txtcolor = SC_VA_ARG(int);
		 break;

            case AXIS_LABELSIZE :
	         _PG_axis_char_size = SC_VA_ARG(int);
		 break;

            case AXIS_CHAR_ANGLE :
	         _PG_axis_char_angle = (REAL) SC_VA_ARG(double);
		 break;

            case AXIS_LABELPREC :
	         prec = SC_VA_ARG(int);
		 break;

            case AXIS_TICKSIZE :
	         _PG_axis_major_tick_size = (REAL) SC_VA_ARG(double);
		 break;

            case AXIS_GRID_ON :
	         _PG_axis_grid_on = SC_VA_ARG(int);
		 break;

            case AXIS_LABELFONT :
	         SFREE(_PG_axis_type_face);
		 _PG_axis_type_face = SC_strsavef(SC_VA_ARG(char *),
                                "char*:PG_SET_AXIS_ATTRIBUTES:labelfont");
		 break;

            case AXIS_X_FORMAT :
	         strcpy(_PG_axis_label_x_format, SC_VA_ARG(char *));
		 break;

            case AXIS_Y_FORMAT :
	         strcpy(_PG_axis_label_y_format, SC_VA_ARG(char *));
		 break;

            default :
	         return(FALSE);};};

    SC_VA_END;

    chpthx = cos(_PG_axis_char_angle);
    chpthy = sin(_PG_axis_char_angle);
    chupx  = -chpthy;
    chupy  =  chpthx;

/* set attribute values */
    PG_set_clipping(dev, FALSE);
    PG_set_color_text(dev, txtcolor, TRUE);
    PG_set_font(dev, _PG_axis_type_face, dev->type_style, dev->type_size);
    PG_set_char_precision(dev, prec);
    PG_set_char_path(dev, chpthx, chpthy);
    PG_set_char_up(dev, chupx, chupy);
    PG_set_char_space(dev, charspace);
    PG_set_color_line(dev, linecolor, TRUE);
    PG_set_line_style(dev, _PG_axis_line_style);
    PG_set_line_width(dev, _PG_axis_line_width);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_AUX_AXIS - Draw a set of axes for a rectangular plot complete with
 *              - labels, ticks, and/or grid lines.  The 
 *              - starting values and interval for the labels and ticks are 
 *              - determined by this routine.
 */

static void _PG_aux_axis(dev, axis_type)
   PG_device *dev;
   int axis_type;
   {int ndim, nt, troax, lroax, troay, lroay;
    double xsep, xp1, xp2, scale, majoren, majorst;
    REAL wxextent, wyextent;
    REAL wxmin, wxmax, wymin, wymax;
    PG_axis_def *ad;

    _PG_axis_type = axis_type;

/* get window and viewport limits */
    PG_get_bound(dev, &wxmin, &wxmax, &wymin, &wymax);
    wxextent = wxmax - wxmin;
    wyextent = wymax - wymin;

/* if the coordinate system has x increasing to the right */
    if (wxmax >= wxmin)
       {lroax = RIGHT_OF_AXIS;
	troax = (_PG_axis_tick_type == RIGHT_OF_AXIS) ?
	        RIGHT_OF_AXIS : LEFT_OF_AXIS;}

/* otherwise use the left hand rule for ticks */
    else
       {lroax = LEFT_OF_AXIS;
	troax = (_PG_axis_tick_type == RIGHT_OF_AXIS) ?
	        LEFT_OF_AXIS : RIGHT_OF_AXIS;};

/* if the coordinate system has y increasing up */
    if (wymax >= wymin)
       {lroay = RIGHT_OF_AXIS;
	troay = (_PG_axis_tick_type == RIGHT_OF_AXIS) ?
	        RIGHT_OF_AXIS : LEFT_OF_AXIS;}

/* otherwise use the left hand rule for ticks */
    else
       {lroay = LEFT_OF_AXIS;
	troay = (_PG_axis_tick_type == RIGHT_OF_AXIS) ?
	        LEFT_OF_AXIS : RIGHT_OF_AXIS;};

    if (_PG_axis_tick_type == STRADDLE_AXIS)
       {troax = STRADDLE_AXIS;
        troay = STRADDLE_AXIS;};

    nt = _PG_axis_n_minor_ticks;
            
/* draw a set of rectangular axes */
    if (axis_type == CARTESIAN)

/* if grid desired draw axes with minor ticks, save major tick data, then
 * draw the grid
 */
       {if (_PG_axis_grid_on == TRUE)
           {_PG_axis_n_minor_ticks = (_PG_axis_n_minor_x_ticks >= 0) ?
	                             _PG_axis_n_minor_x_ticks :
	                             nt;

	    ad = PG_draw_axis(dev, wxmin, wymin, wxmax, wymin,
                              0.0, 1.0, wxmin, wxmax, 1.0,
                              _PG_axis_label_x_format,
                              troax, lroax, TRUE,
                              MAJOR, MINOR, LABEL, 0);
	    if (wxmin < wxmax)
	       {majorst = ad->major_start + wxmin;
		majoren = ad->major_end + wxmin;}
	    else
	       {majoren = -ad->major_start + wxmin;
		majorst = -ad->major_end + wxmin;};
            _PG_draw_grid_lines(dev, ad, majorst, wymin,
                                majorst, wymax, majoren,
                                wymin, majoren, wymax,
                                dev->ifxlog);

            PG_draw_axis(dev, wxmax, wymax, wxmin, wymax,
                         0.0, 1.0, wxmax, wxmin, 1.0,
                         _PG_axis_label_x_format,
                         troax, NOTHING_ON_AXIS, FALSE,
                         MAJOR, MINOR, 0);

            _PG_axis_n_minor_ticks = (_PG_axis_n_minor_y_ticks >= 0) ?
	                             _PG_axis_n_minor_y_ticks :
	                             nt;

            PG_draw_axis(dev, wxmin, wymax, wxmin, wymin,
                         0.0, 1.0, wymax, wymin, 1.0,
                         _PG_axis_label_y_format,
                         troay, lroay, FALSE,
                         MAJOR, MINOR, LABEL, 0);

            ad = PG_draw_axis(dev, wxmax, wymin, wxmax, wymax,
                              0.0, 1.0, wymin, wymax, 1.0,
                              _PG_axis_label_y_format,
                              troay, NOTHING_ON_AXIS, TRUE,
                              MAJOR, MINOR, 0);
	    if (wymin < wymax)
	       {majorst = ad->major_start + wymin;
		majoren = ad->major_end + wymin;}
	    else
	       {majoren = -ad->major_start + wymin;
		majorst = -ad->major_end + wymin;};
            _PG_draw_grid_lines(dev, ad, wxmin, majorst,
                                wxmax, majorst, wxmin,
                                majoren, wxmax, majoren,
                                dev->ifylog);}

/* without grid simply draw the axes */
        else
           {_PG_axis_n_minor_ticks = (_PG_axis_n_minor_x_ticks >= 0) ?
	                             _PG_axis_n_minor_x_ticks :
	                             nt;

	    PG_draw_axis(dev, wxmin, wymin, wxmax, wymin,
                         0.0, 1.0, wxmin, wxmax, 1.0,
                         _PG_axis_label_x_format,
                         troax, lroax, FALSE,
                         MAJOR, MINOR, LABEL, 0);

            PG_draw_axis(dev, wxmax, wymax, wxmin, wymax,
                         0.0, 1.0, wxmax, wxmin, 1.0,
                         _PG_axis_label_x_format,
                         troax, NOTHING_ON_AXIS, FALSE,
                         MAJOR, MINOR, 0);

            _PG_axis_n_minor_ticks = (_PG_axis_n_minor_y_ticks >= 0) ?
	                             _PG_axis_n_minor_y_ticks :
	                             nt;

	    PG_draw_axis(dev, wxmin, wymax, wxmin, wymin,
                         0.0, 1.0, wymax, wymin, 1.0,
                         _PG_axis_label_y_format,
                         troay, lroay, FALSE,
                         MAJOR, MINOR, LABEL, 0);

            PG_draw_axis(dev, wxmax, wymin, wxmax, wymax,
                         0.0, 1.0, wymin, wymax, 1.0,
                         _PG_axis_label_y_format,
                         troay, NOTHING_ON_AXIS, FALSE,
                         MAJOR, MINOR, 0);};}

/* draw a set of Inselberg axes */
    else if (axis_type == INSEL)
       {ndim  = 2;
        xsep  = wxextent/((REAL) (++ndim));
        xp1   = wxmin + xsep;
        xp2   = wxmin + 2.0*xsep;
        scale = wyextent/wxextent;

/* draw the axis lines */
        PG_draw_axis(dev, xp1, wymin, xp1, wymax,
                     0.0, 1.0, wxmin, wxmax, scale,
                     _PG_axis_label_x_format,
                     STRADDLE_AXIS, ENDS, FALSE,
                     MAJOR, MINOR, LABEL, 0);

        PG_draw_axis(dev, xp2, wymin, xp2, wymax,
                     0.0, 1.0, wymin, wymax, 1.0,
                     _PG_axis_label_y_format,
                     STRADDLE_AXIS, ENDS, FALSE,
                     MAJOR, MINOR, LABEL, 0);}

/* draw a set of polar axes */
    else if (axis_type == POLAR)

/* with grid */
       {if (_PG_axis_grid_on == TRUE)

/* if the y = 0 line is in the range draw the axis at y = 0 */
           {if ((wymin <= 0.0) && (0.0 <= wymax))
               PG_draw_axis(dev, wxmin, 0.0, wxmax, 0.0,
                            0.0, 1.0, wxmin, wxmax, 1.0,
                            _PG_axis_label_x_format,
                            STRADDLE_AXIS, ENDS, FALSE,
                            MAJOR, MINOR, LABEL, 0);

/* otherwise draw it at the bottom edge */
            else
               PG_draw_axis(dev, wxmin, wymin, wxmax, wymin,
                            0.0, 1.0, wxmin, wxmax, 1.0,
                            _PG_axis_label_x_format,
                            STRADDLE_AXIS, ENDS, FALSE,
                            MAJOR, MINOR, LABEL, 0);

/* if the x = 0 line is in the domain draw the axis at x = 0 */
            if ((wxmin <= 0.0) && (0.0 <= wxmax))
               ad = PG_draw_axis(dev, 0.0, wymin, 0.0, wymax,
                                 0.0, 1.0, wymin, wymax, 1.0,
                                 _PG_axis_label_y_format,
                                 STRADDLE_AXIS, ENDS, TRUE,
                                 MAJOR, MINOR, LABEL, 0);

/* otherwise draw it at the left edge */
            else
               ad = PG_draw_axis(dev, wxmin, wymin, wxmin, wymax,
                                 0.0, 1.0, wymin, wymax, 1.0,
                                 _PG_axis_label_y_format,
                                 STRADDLE_AXIS, ENDS, TRUE,
                                 MAJOR, MINOR, LABEL, 0);

/* draw the grid */
            _PG_draw_polar_grid_lines(dev, ad, wxmin, wymin, wxmax, wymax);}

/* no grid */
        else

/* if the y = 0 line is in the range draw the axis at y = 0 */
           {if ((wymin <= 0.0) && (0.0 <= wymax))
               PG_draw_axis(dev, wxmin, 0.0, wxmax, 0.0,
                                 0.0, 1.0, wxmin, wxmax, 1.0,
                                 _PG_axis_label_x_format,
                                 STRADDLE_AXIS, ENDS, FALSE,
                                 MAJOR, MINOR, LABEL, 0);

/* otherwise draw it at the bottom edge */
            else
               PG_draw_axis(dev, wxmin, wymin, wxmax, wymin,
                                 0.0, 1.0, wxmin, wxmax, 1.0,
                                 _PG_axis_label_x_format,
                                 STRADDLE_AXIS, ENDS, FALSE,
                                 MAJOR, MINOR, LABEL, 0);

/* if the x = 0 line is in the domain draw the axis at x = 0 */
            if ((wxmin <= 0.0) && (0.0 <= wxmax))
               PG_draw_axis(dev, 0.0, wymin, 0.0, wymax,
                                 0.0, 1.0, wymin, wymax, 1.0,
                                 _PG_axis_label_y_format,
                                 STRADDLE_AXIS, ENDS, FALSE,
                                 MAJOR, MINOR, LABEL, 0);

/* otherwise draw it at the left edge */
            else
               PG_draw_axis(dev, wxmin, wymin, wxmin, wymax,
                                 0.0, 1.0, wymin, wymax, 1.0,
                                 _PG_axis_label_y_format,
                                 STRADDLE_AXIS, ENDS, FALSE,
                                 MAJOR, MINOR, LABEL, 0);};};

    _PG_axis_n_minor_ticks = nt;
            
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_GRID_LINES - draw in grid lines */

static void _PG_draw_grid_lines(dev, ad, x1n, y1n, x1x, y1x,
                                x2n, y2n, x2x, y2x, logsp)
   PG_device *dev;
   PG_axis_def *ad;
   double x1n, y1n, x1x, y1x, x2n, y2n, x2x, y2x;
   int logsp;
   {int n;
    REAL *dx;

    n  = ad->n_major;
    dx = ad->major_dx;

    PG_set_line_style(dev, _PG_axis_grid_style);

    PG_draw_multiple_line(dev, n, x1n, y1n, x1x, y1x,
			  x2n, y2n, x2x, y2x, dx);

    PG_set_line_style(dev, _PG_axis_line_style);

    SFREE(ad->major_dx);
    SFREE(ad->minor_dx);
    SFREE(ad);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_POLAR_GRID_LINES - draw in a set of polar grid lines */

static void _PG_draw_polar_grid_lines(dev, ad, xmn, ymn, xmx, ymx)
   PG_device *dev;
   PG_axis_def *ad;
   double xmn, ymn, xmx, ymx;
   {REAL fr, st, en, tpi;
    int i, n, na, nr;
    REAL dr, r, r1, r2, r3, r4, rmn, rmx, a1, a2, amx, aincr;

    tpi = 2.0*PI;

    st = ad->major_start;
    en = ad->major_end;
    n  = ad->n_major;

    fr    = max(ABS(ymn), ABS(ymx))/(ymx - ymn);
    nr    = fr*n;
    fr    = max(ABS(xmn), ABS(xmx))/(xmx - xmn);
    nr    = max(nr, fr*n);
    na    = 32;
    aincr = tpi/na;
    amx   = na*aincr;
    a1    = 0.0;
    a2    = na*aincr;
    nr    = max(nr, fr*n);
    r1    = HYPOT(ymn, xmn);
    r2    = HYPOT(ymn, xmx);
    r3    = HYPOT(ymx, xmx);
    r4    = HYPOT(ymx, xmn);
    rmn   = min(r1, r2);
    rmn   = min(rmn, r3);
    rmn   = min(rmn, r4);
    rmx   = max(r1, r2);
    rmx   = max(rmx, r3);
    rmx   = max(rmx, r4);

    PG_set_line_style(dev, _PG_axis_grid_style);
    PG_set_clipping(dev, TRUE);

    dr = (en - st)/((double) n - 1.0);
    for (i = 1; i <= nr; i++)
        {r = i*dr;

         if (r < rmn)
            {a1 = 0.0;
             a2 = tpi;}
         else
            {a1 = atan2(ymn,  ABS(xmn));
             a2 = tpi + atan2(ymn, -ABS(xmn));};

         PG_draw_arc(dev, r, a1, a2, 0.0, 0.0, RADIAN);};

    for (i = 0, a1 = 0.0; a1 <= (amx*(1.0 - TOLERANCE)); a1 += aincr, i++)
        {if ((i % 8) != 0)
            PG_draw_rad(dev, 0.0, rmx, a1, 0.0, 0.0, RADIAN);};

    PG_set_clipping(dev, FALSE);
    PG_set_line_style(dev, _PG_axis_line_style);

    SFREE(ad->major_dx);
    SFREE(ad->minor_dx);
    SFREE(ad);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_AXIS - Labels the axes of a rectangular plot
 *              - The y-axis will normally be labeled to the 
 *              - left of the grid and the x-axis will be labeled
 *              - below it.  The routine will produce labels and
 *              - ticks or just ticks, depending on the arguments.
 *              -
 *              -    PG_draw_axis(dev, xl, yl, xr, yr,
 *              -                 t1, t2, v1, v2, sc,
 *              -                 format, tick_type, label_type, ...);
 *              -
 *              -    (xl, yl)   - coordinate of beginning end
 *              -    (xr, yr)   - coordinate of terminating end
 *              -    (t1, t2)   - fractional position of v1 and v2
 *              -    (v1, v2)   - first and last tick or label value
 *              -    sc         - an additional scale factor which is
 *              -                 used, for example, when doing an
 *              -                 Inselberg axis in which the range may
 *              -                 correspond to one apropriate for
 *              -                 the perpendicular dimension
 *              -    format     - specifies the format in the standard C way
 *              -    tick_type  - types of ticks
 *              -                 RIGHT_OF_AXIS - ticks on right
 *              -                 LEFT_OF_AXIS  - ticks on left
 *              -                 STRADDLE_AXIS - ticks straddle (both)
 *              -    label_type - types of labels
 *              -                 RIGHT_OF_AXIS   - labels on right
 *              -                 LEFT_OF_AXIS    - labels on left
 *              -                 NOTHING_ON_AXIS - no labels
 *              -                 ENDS            - labels at ends of axis
 *              -    tickdef    - specifies the labels and ticks
 *              -                 MAJOR - major ticks
 *              -                 MINOR - minor ticks
 *              -                 LABEL - labels
 *              -
 *              - An axis is a directed line segment (from Xl to Xr) with ticks
 *              - The label values as defined by v1, v2, t1, t2, and ticks
 *              - associate with the line segment as follows:
 *              -
 *              -  (xl, yl)                                 (xr, yr)
 *              -     .------------------------------------------>
 *              -       |                                      |
 *              -   st <-> v1 = v(t1)                      en <-> v2 = v(t2)
 *              -   
 *              - where st and en are magnitudes only (rotations done by theta)
 *              - st and en cannot be magnitudes because the interval (-1, 1)
 *              - would have st = en
 *              - This is independent of tick type!
 */

#ifdef PCC

PG_axis_def *PG_draw_axis(dev, xl, yl, xr, yr, t1, t2, v1, v2, sc,
                          format, tick_type, label_type, flag, va_alist)
   PG_device *dev;
   double xl, yl, xr, yr, t1, t2, v1, v2, sc;
   char *format;
   int tick_type, label_type, flag;
   va_dcl

#endif

#ifdef ANSI

PG_axis_def *PG_draw_axis(PG_device *dev, double xl, double yl,
                          double xr, double yr, double t1, double t2,
                          double v1, double v2, double sc,
                          char *format, int tick_type, int label_type,
                          int flag, ...)

#endif

   {int tickdef;
    double minorsz;
    REAL wxmin, wxmax, wymin, wymax;
    PG_axis_def *ad;

    if (v1 == v2)
       {v2 += SMALL;
	v1 -= SMALL;};

    PG_set_color_line(dev, dev->line_color, TRUE);
    PG_draw_line(dev, xl, yl, xr, yr);

    if (_PG_axis_on == FALSE)
       return(NULL);

    PG_get_bound(dev, &wxmin, &wxmax, &wymin, &wymax);

    ad = PG_mk_axis_def(_PG_axis_type, tick_type, label_type,
                        xl, yl, xr, yr, t1, t2, v1, v2, sc,
                        dev->ifxlog, dev->ifylog);

/* set the minor tick size */
    minorsz = _PG_axis_major_tick_size/2.0;

    _PG_comp_tick_spacings(ad);

    SC_VA_START(flag);

    while ((tickdef = SC_VA_ARG(int)) != 0)
       {switch (tickdef)
           {case MAJOR :
                 _PG_tick(ad, MAJOR);
                 _PG_draw_tick(dev, ad, _PG_axis_major_tick_size, MAJOR);
                 break;

            case MINOR :
                 _PG_tick(ad, MINOR);
                 _PG_draw_tick(dev, ad, minorsz, MINOR);
                 break;

            case LABEL :
                 PG_set_color_text(dev, dev->text_color, TRUE);
                 _PG_tick(ad, LABEL);
                 _PG_draw_label(dev, ad, format);
                 break;

            case NOTHING_ON_AXIS :
            default              :
                 break;};};

    SC_VA_END;

    if (flag)
       return(ad);

    else
       {SFREE(ad->major_dx);
        SFREE(ad->minor_dx);
        SFREE(ad);
        return(NULL);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MK_AXIS_DEF - initialize and return a PG_axis_def which contains
 *                - the information necessary to draw an axis
 */

PG_axis_def *PG_mk_axis_def(atype, ttype, ltype,
                            xl, yl, xr, yr, t1, t2, v1, v2, as,
                            xlog, ylog)
   int atype, ttype, ltype;
   double xl, yl, xr, yr, t1, t2, v1, v2, as;
   int xlog, ylog;
   {PG_axis_def *ad;
    double dx, dy, dr, cosa, sina, toler;

    ad = FMAKE(PG_axis_def, "PG_MK_AXIS_DEF:ad");

/* compute various quantities needed by the axis routines */
    dx = xr - xl;
    dy = yr - yl;
    dr = HYPOT(dx, dy);

/* compute the orientation of the axis */
    cosa = dx/dr;
    sina = dy/dr;

    ad->axis_type   = atype;
    ad->tick_type   = ttype;
    ad->label_type  = ltype;
    ad->x_log_scale = xlog;
    ad->y_log_scale = ylog;
    ad->x0          = xl;
    ad->y0          = yl;
    ad->dr          = dr;

    ad->n_major     = 0;
    ad->major_end   = 0.0;
    ad->major_start = 0.0;
    ad->major_space = 0.0;
    ad->major_dx    = NULL;
    ad->n_minor     = 0;
    ad->minor_end   = 0.0;
    ad->minor_start = 0.0;
    ad->minor_space = 0.0;
    ad->minor_dx    = NULL;
    ad->n_label     = 0;
    ad->label_end   = 0.0;
    ad->label_start = 0.0;
    ad->label_space = 0.0;

/* v1 and v2 must be further apart than toler apart */
    toler = 1.0e-8;
    if (ABS(v1 - v2) < toler*(ABS(v1) + ABS(v2)))
       {if (v1 == 0.0)
	   {v1 -= 0.1;
	    v2 += 0.1;}
	else
	   {v1 -= 0.5*toler*v1;
	    v2 += 0.5*toler*v2;};};

    ad->scale       = as;
/*    ad->x_scale     = ABS(as*cosa); */
/*    ad->y_scale     = ABS(as*sina); */
    ad->x_scale     = as;
    ad->y_scale     = as;
    ad->cosa        = cosa;
    ad->sina        = sina;
    ad->t1          = t1;
    ad->t2          = t2;
    ad->v1          = v1;
    ad->v2          = v2;
    ad->eps_rel     = 1.0e-4*(v2 - v1);

    return(ad);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_TICK - draw the ticks
 *               -    ad   - the axis definition info
 *               -    sz   - the size of the ticks
 *               -    tick - major or minor ticks
 */

static int _PG_draw_tick(dev, ad, sz, tick)
   PG_device *dev;
   PG_axis_def *ad;
   double sz;
   int tick;
   {int n, tick_type;
    double st, en, xs, ys, sx, sy, csa, sna;
    REAL v1, v2, x1, y1, x2, y2, x3, y3, x4, y4;
    REAL xa, ya, xb, yb;
    REAL wxmin, wymin, wxmax, wymax;
    REAL dxp, dyp, tdx, tdy, *dx;
    REAL tick_scale, log_scale;

    switch (tick)
       {case MAJOR :
	     st = ad->major_start;
	     en = ad->major_end;
	     n  = ad->n_major;
	     dx = ad->major_dx;
	     break;

        case MINOR :
	     st = ad->minor_start;
	     en = ad->minor_end;
	     n  = ad->n_minor;
	     dx = ad->minor_dx;
	     break;

        default :
	    return(FALSE);};

    xs        = ad->x0;
    ys        = ad->y0;
    sx        = ad->x_scale;
    sy        = ad->y_scale;
    csa       = ad->cosa;
    sna       = ad->sina;
    tick_type = ad->tick_type;

/* compute the actual point on the axis of the first and last tick */
    xa = xs + ABS(st)*csa*sx;
    ya = ys + ABS(st)*sna*sy;
    xb = xs + ABS(en)*csa*sx;
    yb = ys + ABS(en)*sna*sy;

/* compute the actual length of the tick */
    if (tick_type == STRADDLE_AXIS)
       tick_scale = 0.5;
    else
       tick_scale = 1.0;

    log_scale = 2.0;

    PG_get_bound(dev, &wxmin, &wxmax, &wymin, &wymax);

    tdx = tick_scale*sz*sna;
    tdy = tick_scale*sz*csa;

    DELTA(dxp, tdx, wxmin, wxmax, dev->ifxlog, log_scale);
    DELTA(dyp, tdy, wymin, wymax, dev->ifylog, log_scale);

/* depending on the tick type compute the actual starting and ending points
 * for the first and last ticks
 */
    switch (tick_type)
       {case LEFT_OF_AXIS :
	     x2 = xa;
	     y2 = ya;
	     x4 = xb;
	     y4 = yb;
	     if (dev->ifxlog)
	        {if (((ABS(xs) - ABS(en)) == 0.0) && (xb == 0.0))
		    x4 = x2*POW(10.0, -_PG_axis_n_decades);
		 x1 = x2/dxp;
		 x3 = x4/dxp;}

	     else
	        {x1 = x2 - dxp;
		 x3 = x4 - dxp;};

	     if (dev->ifylog)
	        {if (((ABS(ys) - ABS(en)) == 0.0) && (yb == 0.0))
		    y4 = y2*POW(10.0, -_PG_axis_n_decades);
		 y1 = y2*dyp;
		 y3 = y4*dyp;}

	     else
	        {y1 = y2 + dyp;
		 y3 = y4 + dyp;};

	     break;

        case RIGHT_OF_AXIS :
	     x1 = xa;
	     y1 = ya;
	     x3 = xb;
	     y3 = yb;
	     if (dev->ifxlog)
	        {if (((ABS(xs) - ABS(en)) == 0.0) && (xb == 0.0))
		    x3 = x1*POW(10.0, -_PG_axis_n_decades);
		 x2 = x1*dxp;
		 x4 = x3*dxp;}

	     else
	        {x2 = x1 + dxp;
		 x4 = x3 + dxp;};

	     if (dev->ifylog)
	        {if (((ABS(ys) - ABS(en)) == 0.0) && (yb == 0.0))
		    y3 = y1*POW(10.0, -_PG_axis_n_decades);
		 y2 = y1/dyp;
		 y4 = y3/dyp;}

	     else
	        {y2 = y1 - dyp;
		 y4 = y3 - dyp;};

	     break;

        case STRADDLE_AXIS :
	     if (dev->ifxlog)
	        {if (((ABS(xs) - ABS(en)) == 0.0) && (xb == 0.0))
		    xb = xa*POW(10.0, -_PG_axis_n_decades);
		 x1 = xa/dxp;
		 x2 = xa*dxp;
		 x3 = xb/dxp;
		 x4 = xb*dxp;}

	     else
	        {x1 = xa - dxp;
		 x2 = xa + dxp;
		 x3 = xb - dxp;
		 x4 = xb + dxp;}

	     if (dev->ifylog)
	        {if (((ABS(ys) - ABS(en)) == 0.0) && (yb == 0.0))
		    yb = ya*POW(10.0, -_PG_axis_n_decades);
		 y1 = ya*dyp;
		 y2 = ya/dyp;
		 y3 = yb*dyp;
		 y4 = yb/dyp;}

	     else
	        {y1 = ya + dyp;
		 y2 = ya - dyp;
		 y3 = yb + dyp;
		 y4 = yb - dyp;};

	     break;

        default :
	     return(FALSE);};

/* draw the ticks */
    v1 = ad->v1;
    v2 = ad->v2;

    if (v1 < v2)
       PG_draw_multiple_line(dev, n, x1, y1, x2, y2, x3, y3, x4, y4, dx);

    else
       PG_draw_multiple_line(dev, n, x3, y3, x4, y4, x1, y1, x2, y2, dx);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_COMP_TICK_SPACINGS - compute and return the
 *                        - major and minor tick spacings
 *                        - and the label spacings
 */

static void _PG_comp_tick_spacings(ad)
   PG_axis_def *ad;
   {double drl, divs, sp, isp, dv, d5, d2, dt;
    int nt, n5, n2;

    dv = ABS(ad->v2 - ad->v1);

/* find the integral points (10**sp) */
    sp = log10(dv);

    isp = 0.0;
    if (sp < 0.0)
       isp = PM_fix(sp - (1.0 + EPSILON));

    else if (sp > 0.0)
       isp = PM_fix(sp + EPSILON);

/* find the number of integral points in the interval */
    drl = (1.0 - TOLERANCE)*POW(10.0, isp);
    nt  = PM_fix(dv/drl);

    if (nt <= 2)
       {drl *= 0.1;
        nt  *= 10;};

    while (nt >= _PG_axis_max_major_ticks)
       {n5 = nt/5;
        d5 = 5.0*drl;
        n2 = nt/2;
        d2 = 2.0*drl;
        dt = ((double) n5)*d5 - ((double) n2)*d2;
        if (dt >= -TOLERANCE)
           {nt  = n5;
            drl = d5;}
	else
           {nt  = n2;
            drl = d2;};};

    if (nt <= 2)
       divs = 20.0;
    else
       divs = 10.0;

    if (ad->v2 < ad->v1)
       drl *= -1.0;

    ad->major_space = drl;
    if (_PG_axis_n_minor_ticks == -1)
       ad->minor_space = drl/divs;
    else
       ad->minor_space = drl/((double) _PG_axis_n_minor_ticks);

    ad->label_space = drl;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_TICK - given a spacing compute the following additional tick
 *          - information:
 *          -
 *          -    st - first tick position
 *          -    en - last tick position
 *          -    n  - number of ticks
 */

void _PG_tick(ad, tick)
   PG_axis_def *ad;
   int tick;
   {int i, n;
    double va, vb, v2, v1, t1, t2, a, b, dvi;
    double lv1, lv2, dlv, sp, dx0;
    REAL *dx;

    switch (tick)
       {case MAJOR :
	     sp = ad->major_space;
	     break;

        case MINOR :
	     sp = ad->minor_space;
	     break;

        case LABEL :
	     sp = ad->label_space;
	     break;

        default :
	     return;};

    dx = NULL;
    v2 = ad->v2;
    v1 = ad->v1;
    t2 = ad->t2;
    t1 = ad->t1;

    dvi = ad->dr/((v2 - v1)*ad->scale);
    a   = (v1*t2 - v2*t1)*dvi;
    b   = (t2 - t1)*dvi;

    if ((ad->x_log_scale && (ABS(ad->cosa) >= 0.9999)) ||
        (ad->y_log_scale && (ABS(ad->sina) >= 0.9999)))
       {lv1 = log10(v1);
        lv2 = log10(v2);
        dlv = ABS(lv1 - lv2);

        if (dlv > 1)
           {lv1 = floor(lv1);
            lv2 = floor(lv2 + 0.01);};

        sp = PM_fix(log10(dlv));
        sp = max(MIN_DECADE, sp);

        va = sp*PM_fix(lv1/sp);
        vb = sp*PM_fix(lv2/sp);

        if (v1 < v2)
           {if (va < lv1 - TOLERANCE)
               va += sp;
            if (vb > lv2 + TOLERANCE)
               vb -= sp;}

        else
           {if (va > lv1 + TOLERANCE)
               va -= sp;
            if (vb < lv2 - TOLERANCE)
               vb += sp;};

        dlv = ABS(va - vb);
        n = 1.0 + dlv/sp + EPSILON;

        va = POW(10.0, va);
        vb = POW(10.0, vb);

        if ((tick == LABEL) && (ad->label_type == ENDS))
           n = 2;
        else
           _PG_log_tick(v1, v2, &n, &va, &vb, &dx, tick);}

    else
       {sp = ABS(sp);

        va = sp*PM_fix(v1/sp);
        vb = sp*PM_fix(v2/sp);

        if (v1 < v2)
           {if (va < v1)
               va += sp;
            if (vb > v2)
               vb -= sp;}

        else
           {if (va > v1)
               va -= sp;
            if (vb < v2)
               vb += sp;};

        n = ABS((vb - va)/sp) + 1.0 + EPSILON;};

    n = max(n, 2.0 + EPSILON);

    if (dx == NULL)
       {dx = FMAKE_N(REAL, n, "_PG_TICK:dx");
        dx0 = 1.0/((double) n - 1.0);
        for (i = 0; i < n; i++)
            dx[i] = i*dx0;};

    switch (tick)
       {case MAJOR :
	     ad->major_start = va*b - a;
	     ad->major_end   = vb*b - a;
	     ad->n_major     = n;
	     ad->va_major    = va;
	     ad->vb_major    = vb;
	     SFREE(ad->major_dx);
	     ad->major_dx    = dx;
	     break;

        case MINOR :
	     ad->minor_start = va*b - a;
	     ad->minor_end   = vb*b - a;
	     ad->n_minor     = n;
	     ad->va_minor    = va;
	     ad->vb_minor    = vb;
	     ad->minor_dx    = dx;
	     break;

        case LABEL :
	     ad->label_start = va*b - a;
	     ad->label_end   = vb*b - a;
	     ad->n_label     = n;
	     ad->va_major    = va;
	     ad->vb_major    = vb;
	     SFREE(ad->major_dx);
	     ad->major_dx    = dx;
	     break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_LOG_TICK - compute a nice set of log spacings
 *              - the minor spacings are chosen as follows:
 *              -     step = 1, spaces = (2)
 *              -     step = 2, spaces = (2, 5)
 *              -     step = 3, spaces = (2, 3, 5)
 *              -     step = 4, spaces = (2, 3, 4, 5)
 *              -     step = 5, spaces = (2, 3, 4, 5, 6)
 *              -     step = 6, spaces = (2, 3, 4, 5, 6, 7)
 *              -     step = 7, spaces = (2, 3, 4, 5, 6, 7, 8)
 *              -     step = 8, spaces = (2, 3, 4, 5, 6, 7, 8, 9)
 */

static void _PG_log_tick(v1, v2, pn, pva, pvb, pd, tick)
   double v1, v2;
   int *pn;
   double *pva, *pvb;
   REAL **pd;
   int tick;
   {int n, lv1, lv2, dexp, na, step, rmnd, decade, j, jin, jout, k;
    REAL *pdx, *dx;
    double sub[10], v1d, v2d, s, t, dxm, txm;
    static double log_value[] = {0.00000000000000,
                                 0.30102999566398, 
                                 0.47712125471966, 
                                 0.60205999132796, 
                                 0.69897000433601, 
                                 0.77815125038364, 
                                 0.84509804001424, 
                                 0.90308998699193, 
                                 0.95424250943931,
                                 1.00000000000000};

    n  = *pn;
    dx = FMAKE_N(REAL, 2*n, "_PG_LOG_TICK:dx");

/* switch min and max? */
    if (v1 > v2)
       {s  = v1;
        v1 = v2;
        v2 = s;};
 
    if (v1 == 0.0)
       {if (v2 > 100.0)
           v1 = 1.0;
        else
           v1 = POW(10.0, -_PG_axis_n_decades)*v2;};

    lv1 = (int) log10(v1);
    lv2 = (int) log10(v2);
    lv1 = max(lv1, lv2 - _PG_axis_n_decades - 1);

    na = lv2 - lv1 + 1;
    na = max(2, na);
    if (na > _PG_axis_max_major_ticks)
       dexp = na/_PG_axis_max_major_ticks;
    else
       dexp = 1;
    rmnd = n - na;
    step = rmnd/na;
    step = min(step, 8);

    if (step > 0)
       {for (j = 0; j <= step; j++)
            sub[j] = log_value[j];
        if (step < 4)
           sub[step] = log_value[4];};

    na = (na/dexp) + step*(na - 1);
    step++;
    for (j = 0, decade = lv1; j < na; decade += dexp)
        {pdx    = dx + j;
         pdx[0] = (double) decade;
         j++;
         if (step > 1)
            {for (k = 1; (k < step) && (j < na); k++, j++)
                 pdx[k] = (double) decade + sub[k];};};

/* exponentiate the spacings and remove anybody outside the range
 * also MAJOR and LABEL ticks go on only on the decades
 */
    if (na > 2)
       {dxm = -HUGE;
        v1d = 0.9999*v1;
        v2d = 1.0001*v2;
        for (jin = 0, jout = 0; jin < na; jin++)
            {t = POW(10.0, dx[jin]);

             if ((v1d <= t) && (t <= v2d))
                {if ((tick == MAJOR) || (tick == LABEL))
                    {s = floor(log10(1.0000000001*t));
                     if (((s != 0.0) || (dx[jin] != 0.0)) &&
                         !PM_CLOSETO_REL(s, dx[jin]))
                        continue;};
                 txm = POW(10.0, (double) floor(dx[jin]));
                 dxm = max(dxm, txm);
                 dx[jout++] = t;};};

        na = jout;}

    else
       {dxm = -HUGE;
        for (j = 0; j < na; j++)
            {t     = POW(10.0, dx[j]);
             dxm   = max(dxm, t);
             dx[j] = t;};};

    if (*pva > *pvb)
       *pvb = dx[0];
    else
       *pva = dx[0];

    if (na > 1)
       {t = dx[0];

/* shift by the mimimum value */
        dxm -= t;
        for (j = 0; j < na; j++)
            dx[j] -= t;

/* normalize the spacings */
        if (dxm == 0.0)
           dxm = 1.0;
        else
           dxm = 1.0/dxm;
        for (j = 0; j < na; j++)
            dx[j] *= dxm;}

    else
       dx[0] = 1;

    *pn = na;
    *pd = dx;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_LABEL - draw the axis numerics */

static int _PG_draw_label(dev, ad, fmt)
   PG_device *dev;
   PG_axis_def *ad;
   char *fmt;
   {int i, tick_type, lt, ln, ti, inc, flg;
    int xovlp, yovlp;
    double xs, ys, sx, sy, csa, sna, Dr, ll;
    REAL ldx, ldy, fdx, fdy, tdx, tdy, lst, let, lsp;
    REAL xa, ya, xb, yb, fx, fy, yo, idm;
    REAL sp, ls, le, lss, lse, v1, v2, va, vb, dv, sv, db, tol;
    REAL log_scale;
    REAL wxmin, wxmax, wymin, wymax;
    REAL *dx, dx0, dy0;
    char s[20], t[20], format[10], td;

    ls = ad->label_start;
    le = ad->label_end;
    ln = ad->n_label;
    sp = ad->label_space;

    xs        = ad->x0;
    ys        = ad->y0;
    sx        = ad->x_scale;
    sy        = ad->y_scale;
    csa       = ad->cosa;
    sna       = ad->sina;
    tick_type = ad->tick_type;
    lt        = ad->label_type;
    dx        = ad->major_dx;

    strcpy(format, fmt);

    lss = ad->va_major;
    lse = ad->vb_major;
    if (dev->ifxlog || dev->ifylog)
       tol = 0.0;
    else
       tol = 1.0e-6*ABS(lse - lss);

    v1 = ad->v1;
    v2 = ad->v2;
    if (v1 > v2)
       {v2 = ad->v1;
        v1 = ad->v2;};

/* determine whether to use 'e' or 'f' format since 'g' format is a dog */
    yo = max(ABS(ls), ABS(le));
    ti = strlen(format) - 1;
    td = format[ti];
    if (td == 'g')
       {if ((yo > 1000.0) || (yo < 0.011))
           format[ti] = 'e';
        else
           format[ti] = 'f';};

    ldx = 0.0;
    ldy = 0.0;
    lst = min(lss, lse);
    let = max(lss, lse);
    lsp = ABS(sp);
    for (ll = lst; ll <= let; ll += lsp)
        {_PG_sprintf(s, format, ll, tol);
         PG_get_text_ext(dev, s, &tdx, &tdy);
         ldx = max(ldx, tdx);
         ldy = max(ldy, tdy);};

/* compute the offset from the major tick for the aligned label text
 * alignment is horizontal and vertical centering
 */
    log_scale  = 2.0;

    PG_get_bound(dev, &wxmin, &wxmax, &wymin, &wymax);

    tdx = _PG_axis_label_x_standoff*_PG_axis_major_tick_size*sna;
    tdy = _PG_axis_label_y_standoff*_PG_axis_major_tick_size*csa;

    DELTA(fdx, tdx, wxmin, wxmax, dev->ifxlog, log_scale);
    DELTA(fdy, tdy, wymin, wymax, dev->ifylog, log_scale);

/* set up the label increments */
    xa = xs + ls*csa*sx;
    ya = ys + ls*sna*sy;
    xb = xs + le*csa*sx;
    yb = ys + le*sna*sy;
    inc = 1;

    switch (lt)
       {case RIGHT_OF_AXIS :
             dx0 = (le - ls)*csa*sx;
             dy0 = (le - ls)*sna*sy;

             if (dev->ifxlog)
                {xa *= fdx;
                 xb *= fdx;}
             else
                {xa += fdx;
                 xb += fdx;};

             if (dev->ifylog)
                {ya /= fdy;
                 yb /= fdy;}
             else
                {ya -= fdy;
                 yb -= fdy;};

             fx = -0.47*(1.0 - sna);
/*             fy  = -csa - 0.5*ABS(sna); */
             fy = -0.5*csa - 0.25*ABS(csa) - 0.33*ABS(sna);
             break;

        case LEFT_OF_AXIS :
             dx0 = (le - ls)*csa*sx;
             dy0 = (le - ls)*sna*sy;

             if (dev->ifxlog)
                {xa /= fdx;
                 xb /= fdx;}
             else
                {xa -= fdx;
                 xb -= fdx;};

             if (dev->ifylog)
                {ya *= fdy;
                 yb *= fdy;}
             else
                {ya += fdy;
                 yb += fdy;};

             fx  = -0.47*(1.0 + sna);
/*             fy  = csa - 0.5*ABS(sna); */
             fy = 0.5*csa - 0.25*ABS(csa) - 0.33*ABS(sna);
             break;

        case ENDS :
             sp = ABS(ad->v2 - ad->v1);

             xa = 0.015*sp*csa*sx;
             ya = 0.015*sp*sna*sy;

             fx = -0.47*(1.0 + csa);
             fy = -0.33*(1.0 + sna);

             Dr = ad->dr;

             dx0 = Dr*csa;
             if (dev->ifxlog)
                {idm = log_scale/
                       (ABS(0.5*(wxmin + wxmax)) + SMALL);
                 dx0 *= (1.0 + (2.0*xa - fx*ldx*csa)*idm);
                 xb   = (xs + 1.015*sp*csa*sx)*(1.0 + xa*idm);
                 xa   = xs/(1.0 + xa*idm);}
             else
                {dx0 += 2.0*xa - fx*ldx*csa;
                 xb   = xs + 1.015*sp*csa*sx;
                 xa   = xs - xa;};

             dy0 = Dr*sna;
             if (dev->ifylog)
                {idm = log_scale/
                       (ABS(0.5*(wymin + wymax)) + SMALL);
                 dy0 *= (1.0 + (2.0*ya - fy*ldy*sna)*idm);
                 yb   = (ys + 1.015*sp*sna*sy)*(1.0 + ya*idm);
                 ya   = ys/(1.0 + ya*idm);}
             else
                {dy0 += 2.0*ya - fy*ldy*sna;
                 yb   = ys + 1.015*sp*sna*sy;
                 ya   = ys - ya;};

             inc = ln - 1;
             break;

        default :
             return(FALSE);};

/* draw the labels (always low to high - for numerics to come out right) */
    va = ad->va_major;
    vb = ad->vb_major;
    if (ad->v2 < ad->v1)
       {dx0 = -dx0;
	dy0 = -dy0;
	xa = xb;
        ya = yb;
        va = ad->vb_major;
        vb = ad->va_major;};

/* if the labels cannot be distinguished because of the precision
 * try a different tack
 */
    dv = ABS(vb - va);
    sv = ABS(va) + ABS(vb);
    flg = (dv < 0.001*sv);
    xb = HUGE;
    yb = HUGE;
    if (flg)
       {db = dx[0];
	xs = xa + db*dx0;
        ys = ya + db*dy0;
        sprintf(s, format, va);
        sprintf(t, format, va + dx[1]*dv);
	xb = xs;
	yb = ys;
        if (strcmp(s, t) == 0)
           _PG_write_label(dev, format, xs, ys, va, tol, fx, fy, '~');
        else
           _PG_write_label(dev, format, xs, ys, va, tol, fx, fy, '>');
        va = 0.0;};

    for (i = flg; i < ln; i += inc)
        {xs = xa + dx[i]*dx0;
         ys = ya + dx[i]*dy0;
         ls = va + dx[i]*dv;

	 if (dev->ifxlog)
	    xovlp = ldx > max(xs/xb, xb/xs);
	 else
	    xovlp = ldx > ABS(xs - xb);

	 if (dev->ifylog)
	    yovlp = ldy > max(ys/yb, yb/ys);
	 else
	    yovlp = ldy > ABS(ys - yb);

	 if (xovlp && yovlp)
	    continue;

	 xb = xs;
	 yb = ys;

         _PG_write_label(dev, format, xs, ys, ls, tol, fx, fy, FALSE);};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_WRITE_LABEL - write a single label */

static void _PG_write_label(dev, format, xa, ya, lss, tol, fx, fy, flg)
   PG_device *dev;
   char *format;
   double xa, ya, lss, tol, fx, fy;
   int flg;
   {REAL tdx, tdy;
    double xc, yc;
    char s[80];

    if (flg)
       {s[0] = flg;
        _PG_sprintf(s+1, format, lss, tol);}
    else
       _PG_sprintf(s, format, lss, tol);
    
    PG_get_text_ext(dev, s, &tdx, &tdy);

    if (dev->ifxlog)
       xc = xa*POW(tdx, fx);
    else
       xc = xa + fx*tdx;

    if (dev->ifylog)
       yc = ya*POW(tdy, fy);
    else
       yc = ya + fy*tdy;

    PG_write_WC(dev, xc, yc, "%s", s);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SPRINTF - converts a number to a character string
 *             - and removes additional spaces
 *             - arguments are the same as for sprintf
 */

static void _PG_sprintf(string, format, value, tol)
   char *string, *format;
   double value, tol;
   {char temp[40], *token, *s;

    if (ABS(value) < tol)
       value = 0.0;

    if (strchr(format, 'd') != NULL)
       sprintf(temp, format, (int) value);
    else
       sprintf(temp, format, value);

    token = SC_strtok(temp, " ", s);
    strcpy(string, token);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
