/*
 * SXPAN.C - PANACEA extensions in SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

object
 *_SX_var_tab;

PFInt
 SX_pan_data_hook = NULL;

#if 0

static void
 SC_DECLARE(_SX_wr_gsource_variable, 
            (object *obj, object *strm)),
 SC_DECLARE(_SX_wr_giv_specification, 
            (object *obj, object *strm));

static object
 SC_DECLARE(*SX_mk_source_variable, (PA_src_variable *sv)),
 SC_DECLARE(*SX_mk_iv_specification, 
            (PA_iv_specification *iv));

#endif

static int
 SC_DECLARE(*_SX_index_ptr, (object **pargl, char *msg));

static void
 SC_DECLARE(_SX_wr_gpackage, (object *obj, object *strm)),
 SC_DECLARE(_SX_wr_gvariable, (object *obj, object *strm));

static object
 SC_DECLARE(*SX_advance_name, (object *obj)),
 SC_DECLARE(*SX_advance_time, (object *argl)),
 SC_DECLARE(*SX_db_numeric_data, (object *obj)),
 SC_DECLARE(*SX_def_var, (object *argl)),
 SC_DECLARE(*SX_dump_pp, (object *argl)),
 SC_DECLARE(*SX_fin_system, (object *argl)),
 SC_DECLARE(*SX_init_problem, (object *argl)),
 SC_DECLARE(*SX_inst_com, (byte)),
 SC_DECLARE(*SX_intern_packages, (byte)),
 SC_DECLARE(*SX_iv_specp, (object *obj)),
 SC_DECLARE(*SX_packagep, (object *obj)),
 SC_DECLARE(*SX_package_name, (object *obj)),
 SC_DECLARE(*SX_pan_cmmnd, (object *argl)),
 SC_DECLARE(*SX_pan_simulate, (object *argl)),
 SC_DECLARE(*SX_panvarp, (object *obj)),
 SC_DECLARE(*SX_readh, (object *argl)),
 SC_DECLARE(*SX_rd_restart, (object *argl)),
 SC_DECLARE(*SX_run_package, (object *argl)),
 SC_DECLARE(*SX_srcvarp, (object *obj)),
 SC_DECLARE(*SX_wr_restart, (object *argl));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_INSTALL_PANACEA_FUNCS - install the PANACEA extensions to Scheme */
 
void SX_install_panacea_funcs()
   {
    SS_install("pa-advance-name",
               "Advance the given file name",
               SS_sargs,
               SX_advance_name, SS_PR_PROC);

    SS_install("pa-advance-time",
               "Sets the problem time and time step",
               SS_nargs,
               SX_advance_time, SS_PR_PROC);

    SS_install("pa-command",
               "Process a command line string",
               SS_nargs,
               SX_pan_cmmnd, SS_PR_PROC);

    SS_install("pa-define-variable",
               "Define a new variable to the database",
               SS_nargs,
               SX_def_var, SS_PR_PROC);

    SS_install("pa-describe-entity",
               "Displays a description of a panacea package or variable",
               SS_sargs,
               SX_desc_pan, SS_PR_PROC);

    SS_install("pa-display",
               "Displays a panacea object in nice form",
               SS_sargs,
               SX_display_pan_object, SS_PR_PROC);

    SS_install("pa-finish-simulation",
               "Gracefully conclude a numerical simulation",
               SS_nargs,
               SX_fin_system, SS_PR_PROC);

    SS_install("pa-init-simulation",
               "Initialize a numerical simulation",
               SS_nargs,
               SX_init_problem, SS_PR_PROC);

    SS_install("pa-install-commands",
               "Install the commands from all packages",
               SS_zargs,
               SX_inst_com, SS_PR_PROC);

    SS_install("pa-intern-packages",
               "Returns a list of variables which are bound to the PANACEA packages",
               SS_zargs,
               SX_intern_packages, SS_PR_PROC);

    SS_install("pa-iv-specification?",
               "Returns #t if the object is a PANACEA initial value specification, and #f otherwise",
               SS_sargs,
               SX_iv_specp, SS_PR_PROC);

    SS_install("pa-package?",
               "Returns #t if the object is a PANACEA package, and #f otherwise",
               SS_sargs,
               SX_packagep, SS_PR_PROC);

    SS_install("pa-package-name",
               "Returns a the name of the PANACEA package",
               SS_sargs,
               SX_package_name, SS_PR_PROC);

    SS_install("pa-read-commands",
               "Read the commands in the specified source file",
               SS_nargs,
               SX_readh, SS_PR_PROC);

    SS_install("pa-read-state-file",
               "Reads the named state file and does the specified conversions",
               SS_nargs,
               SX_rd_restart, SS_PR_PROC);

    SS_install("pa-run-package",
               "Executes the given package and returns its time step and controlling zone",
               SS_nargs,
               SX_run_package, SS_PR_PROC);

    SS_install("pa-save-to-pp",
               "Save the data for the output requests from this cycle",
               SS_nargs,
               SX_dump_pp, SS_PR_PROC);

    SS_install("pa-simulate",
               "Runs a simulation from Ti to Tf",
               SS_nargs,
               SX_pan_simulate, SS_PR_PROC);

    SS_install("pa-source-variable?",
               "Returns #t if the object is a PANACEA source variable, and #f otherwise",
               SS_sargs,
               SX_srcvarp, SS_PR_PROC);

    SS_install("pa-variable?",
               "Returns #t if the object is a PANACEA variable, and #f otherwise",
               SS_sargs,
               SX_panvarp, SS_PR_PROC);

    SS_install("pa-variable->pm-array",
               "Save the data for the output requests from this cycle",
               SS_sargs,
               SX_db_numeric_data, SS_PR_PROC);

    SS_install("pa-write-state-file",
               "Write a state file",
               SS_sargs,
               SX_wr_restart, SS_PR_PROC);

    SS_define_constant(0,
                       "dimension",     SC_STRING_I, "dimension",
                       "upper-lower",   SC_STRING_I, "upper-lower",
                       "offset-number", SC_STRING_I, "offset-number",
                       "units",         SC_INTEGER_I, -1,
                       "per",           SC_INTEGER_I, -2,
                       "attribute",     SC_INTEGER_I, 102,
                       NULL);

    SS_define_constant(0,
                       "scope",         SC_INTEGER_I, 97,
                       "defn",          SC_INTEGER_I, -1,
                       "restart",       SC_INTEGER_I, -2,
                       "demand",        SC_INTEGER_I, -3,
                       "runtime",       SC_INTEGER_I, -4,
                       "edit",          SC_INTEGER_I, -5,
                       "scratch",       SC_INTEGER_I, -6,
                       NULL);

    SS_define_constant(0,
                       "class",         SC_INTEGER_I, 98,
                       "required",      SC_INTEGER_I, 1,
                       "optional",      SC_INTEGER_I, 2,
                       "pseudo",        SC_INTEGER_I, 3,
                       NULL);

    SS_define_constant(0,
                       "persist",       SC_INTEGER_I, 99,
                       "release",       SC_INTEGER_I, -10,
                       "keep",          SC_INTEGER_I, -11,
                       "cache",         SC_INTEGER_I, -12,
                       NULL);

    SS_define_constant(0,
                       "center",        SC_INTEGER_I, 100,
                       "zone-centered", SC_INTEGER_I, -1,
                       "node-centered", SC_INTEGER_I, -2,
                       "face-centered", SC_INTEGER_I, -3,
                       "edge-centered", SC_INTEGER_I, -4,
                       "uncentered",    SC_INTEGER_I, -5,
                       NULL);

    SS_define_constant(0,
                       "allocation",    SC_INTEGER_I, 101,
                       "static",        SC_INTEGER_I, -100,
                       "dynamic",       SC_INTEGER_I, -101,
                       NULL);

    SS_install_cv("radian",    &PA_radian,          SC_INTEGER_I);
    SS_install_cv("steradian", &PA_steradian,       SC_INTEGER_I);
    SS_install_cv("mole",      &PA_mole,            SC_INTEGER_I);
    SS_install_cv("Q",         &PA_electric_charge, SC_INTEGER_I);
    SS_install_cv("cm",        &PA_cm,              SC_INTEGER_I);
    SS_install_cv("sec",       &PA_sec,             SC_INTEGER_I);
    SS_install_cv("g",         &PA_gram,            SC_INTEGER_I);
    SS_install_cv("eV",        &PA_eV,              SC_INTEGER_I);
    SS_install_cv("K",         &PA_kelvin,          SC_INTEGER_I);
    SS_install_cv("erg",       &PA_erg,             SC_INTEGER_I);
    SS_install_cv("cc",        &PA_cc,              SC_INTEGER_I);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PACKAGEP - function version of SX_PACKAGEP macro */

static object *SX_packagep(obj)
   object *obj;
   {return(SX_PACKAGEP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PANVARP - function version of SX_PANVARP macro */

static object *SX_panvarp(obj)
   object *obj;
   {return(SX_PANVARP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SRCVARP - function version of SX_SOURCE_VARIABLEP macro */

static object *SX_srcvarp(obj)
   object *obj;
   {return(SX_SOURCE_VARIABLEP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_IV_SPECP - function version of SX_IV_SPECIFICATIONP macro */

static object *SX_iv_specp(obj)
   object *obj;
   {return(SX_IV_SPECIFICATIONP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_INTERN_PACKAGES - make a Scheme variable for each PANACEA
 *                    - package and return them in a list
 */

static object *SX_intern_packages()
   {object *obj, *lst, *lst_nxt;
    PA_package *pck;
    char *name;
    hashel *hp;

    lst = SS_null;
    for (pck = Packages; pck != NULL; pck = pck->next)
        {name = pck->name;
         if ((hp = SC_lookup(name, SS_symtab)) != NULL)
            obj = (object *) hp->def;
         else
            {obj = SS_mk_variable(name, SS_null);
             SS_UNCOLLECT(obj);
             if (SC_install(name, obj, SS_POBJECT_S, SS_symtab) == NULL)
                SS_error("CAN'T INSTALL PACKAGE - SX_INTERN_PACKAGE", obj);};

/* put these in the top level environment frame
 * not the best place but SS_Env won't last through most expression
 * evaluations
 */
         SS_def_var(obj,
                    SX_mk_package(pck),
                    SS_Global_Env);
         SS_end_cons(lst, lst_nxt, obj);};

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_PACKAGE - make and return a g_package */

object *SX_mk_package(pck)
   PA_package *pck;
   {object *op;

    op = SS_mk_object(pck, G_PACKAGE, SELF_EV, pck->name);
    op->print   = _SX_wr_gpackage;
    op->release = SS_rl_object;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GPACKAGE - print a g_package */

static void _SX_wr_gpackage(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<PACKAGE|%s>", PACKAGE_NAME(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_VARIABLE - make and return a g_variable */

object *SX_mk_variable(pp)
   PA_variable *pp;
   {object *op;

    op = SS_mk_object(pp, G_PANVAR, SELF_EV, pp->name);
    op->print   = _SX_wr_gvariable;
    op->release = SS_rl_object;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GVARIABLE - print a g_variable */

static void _SX_wr_gvariable(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<PANVAR|%s>",
                              PANVAR_NAME(obj));

    return;}

/*--------------------------------------------------------------------------*/

#if 0

/*--------------------------------------------------------------------------*/

/* SX_MK_SOURCE_VARIABLE - make and return a g_source_variable */

static object *SX_mk_source_variable(sv)
   PA_src_variable *sv;
   {object *op;

    op = SS_mk_object(sv, G_SOURCE_VARIABLE, SELF_EV, sv->name);
    op->print   = _SX_wr_gsource_variable;
    op->release = SS_rl_object;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GSOURCE_VARIABLE - print a g_source_variable */

static void _SX_wr_gsource_variable(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<SOURCE_VARIABLE|%s>",
                              SOURCE_VARIABLE_NAME(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_IV_SPECIFICATION - make and return a g_iv_specification */

static object *SX_mk_iv_specification(iv)
   PA_iv_specification *iv;
   {object *op;

    op = SS_mk_object(iv, G_IV_SPECIFICATION, SELF_EV, iv->name);
    op->print   = _SX_wr_giv_specification;
    op->release = SS_rl_object;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GIV_SPECIFICATION - print a g_iv_specification */

static void _SX_wr_giv_specification(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<IV_SPECIFICATION|%s>",
                              IV_SPECIFICATION_NAME(obj));

    return;}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* SX_PACKAGE_NAME - return the package name */

static object *SX_package_name(obj)
   object *obj;
   {if (SX_PACKAGEP(obj))
       return(SS_mk_string(PACKAGE_NAME(obj)));
    else
       SS_error("BAD PACKAGE - SX_PACKAGE_NAME", obj);

    return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PAN_SIMULATE - Scheme level call to PA_simulate
 *                 - arguments are :
 *                 -
 *                 -    ti - initial time
 *                 -    tf - final time
 */

static object *SX_pan_simulate(argl)
   object *argl;
   {double t, ti, tf;
    double *pt, *pdt;
    int *pcy, *pnz;
    static char *rsname = NULL,
                *edname = NULL,
                *ppname = NULL,
                *gfname = NULL;
    static double dtf_min  = 1.0e-15,
                  dtf_max  = 1.0e-3,
                  dtf_inc  = 2.0;

    ti = tf = 0.0;
    SS_args(argl,
            SC_DOUBLE_I, &ti,
            SC_DOUBLE_I, &tf,
            SC_DOUBLE_I, &dtf_min,
            SC_DOUBLE_I, &dtf_max,
            SC_DOUBLE_I, &dtf_inc,
            SC_STRING_I, &rsname,
            SC_STRING_I, &edname,
            SC_STRING_I, &ppname,
            SC_STRING_I, &gfname,
            0);

    ti *= unit[SEC]/convrsn[SEC];
    tf *= unit[SEC]/convrsn[SEC];

    pt  = (double *) SS_var_reference("current-time");
    pdt = (double *) SS_var_reference("current-timestep");
    pcy = (int *) SS_var_reference("current-cycle");
    pnz = (int *) SS_var_reference("number-of-zones");

    t = *pt;

    if ((t < ti) || (tf < t) || (tf < ti))
       {PRINT(stdout, "PROBLEM OUTSIDE OF TIME WINDOW: ");
        PRINT(stdout, "ti = %11.3e, t = %11.3e, tf = %11.3e", ti, t, tf);}

/* run the simulation */
    else
       PA_simulate(t, *pcy, *pnz, ti, tf, *pdt,
                   dtf_min, dtf_max, dtf_inc,
                   rsname, edname, ppname, gfname);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RUN_PACKAGE - execute the given package from Scheme level
 *                - return the timestep and zone controlling it
 *                - in a cons
 */

static object *SX_run_package(argl)
   object *argl;
   {object *numdt, *zondt;
    PA_package *pck;
    PFInt pck_entry;
    char *pck_name;
    double t, dt;
    int cycle;

    SS_args(argl,
            G_PACKAGE, &pck,
            SC_DOUBLE_I, &t,
            SC_DOUBLE_I, &dt,
            SC_INTEGER_I, &cycle,
            0);

    pck_entry = pck->main;
    pck_name  = pck->name;

/* execute the package */
    if (pck_entry != NULL)
       {pck->p_t     = t;
        pck->p_dt    = dt;
        pck->p_cycle = cycle;

        PA_control_set(pck_name);
        _PA_allocate_mapping_space(pck);

        (*pck_entry)(pck);

        _PA_dump_package_mappings(pck, t, dt, cycle);};

/* reconnect the global controls */
    PA_control_set("global");

    numdt = SS_mk_float(pck->dt);
    zondt = SS_mk_integer(pck->dt_zone);

    return(SS_mk_cons(numdt, zondt));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_ADVANCE_NAME - advance the given file family name */

static object *SX_advance_name(obj)
   object *obj;
   {char *name;

    name = NULL;
    if (SS_stringp(obj))
       name = SS_STRING_TEXT(obj);

    if (name != NULL)
       PA_advance_name(name);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_ADVANCE_TIME - set new dt from given value at Scheme level
 *                 - add old dt to t
 *                 - advance the cycle count
 */

static object *SX_advance_time(argl)
   object *argl;
   {double dt, pck_dt, t1, t2, t3, tconv;
    double *pt, *pdt;
    int *pcy;
    PA_package *pck;
    int nargs;

    pt  = (double *) SS_var_reference("current-time");
    pdt = (double *) SS_var_reference("current-timestep");
    pcy = (int *) SS_var_reference("current-cycle");

    dt = *pdt;

    *pt += dt;
    (*pcy)++;

    nargs = SS_args(argl,
                    SC_DOUBLE_I, &t1,
                    SC_DOUBLE_I, &t2,
                    SC_DOUBLE_I, &t3,
                    0);

    tconv = unit[SEC]/convrsn[SEC];
    t1   *= tconv;
    t2   *= tconv;
    t3   *= tconv;
    switch (nargs)
       {case 0  : dt = HUGE;
                  for (pck = Packages; pck != NULL; pck = pck->next)
                      {pck_dt = pck->dt;
                       dt = min(pck_dt, dt);};
                  break;
        case 1  : dt = t1;
                  break;
        case 2  : dt = HUGE;
                  for (pck = Packages; pck != NULL; pck = pck->next)
                      {pck_dt = pck->dt;
                       dt = min(pck_dt, dt);};
                  dt = min(dt, t1);
                  dt = max(dt, t2);
                  break;
        default : dt = t1;
                  for (pck = Packages; pck != NULL; pck = pck->next)
                      {pck_dt = pck->dt;
                       dt = min(pck_dt, dt);};
                  dt = min(dt, t2);
                  dt = max(dt, t3);
                  break;};

    *pdt = dt;

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEF_VAR - the most general variable installer for PANACEA
 *            - usage:
 *            -
 *            - (define-variable <name> <type> <init_val> <init_func>
 *            -            <attrribute_spec> ... ATTRIBUTE
 *            -            <dimension_spec> ... DIMENSION
 *            -            <unit_spec> ... UNITS
 *            -            [<ptr to data>])
 *            -
 *            - <attribute_spec> := <attribute_id> <attribute_val>
 *            - <dimension_spec> := <ptr to # values> |
 *            -                     PA_DUL <ptr to lower> <ptr to upper> |
 *            -                     PA_DON <ptr to offset> <ptr to # elem>
 *            - <unit_spec>      := <unit_index> | PER
 *            -
 *            - The valid attribute_id's are:
 *            -      SCOPE, CLASS, PERSIST, CENTER, ALLOCATION
 *            -
 *            - The valid SCOPE's are:
 *            -      DEFN, RESTART, DMND, RUNTIME, EDIT, SCRATCH
 *            -
 *            - The valid CLASS's are:
 *            -      REQU, OPTL, PSEUDO
 *            -
 *            - The valid PERSIST's are:
 *            -      REL, KEEP, CACHE_F, CACHE_R
 *            -
 *            - The valid CENTER's are:
 *            -      Z_CENT, N_CENT, F_CENT, E_CENT, U_CENT
 *            -
 *            - The valid ALLOCATION's are:
 *            -      STATIC, DYNAMIC
 *            -
 *            - The pre-defined unit_index's are:
 *            -      RAD, STER, MOLE, Q, CM, SEC, G, EV, K, ERG, CC
 *            -
 *            - No attributes are required to be set (ATTRIBUTE must appear
 *            - in the arg list) and the default values are:
 *            -      RUNTIME, OPTL, REL, U_CENT, DYNAMIC
 */
 
static object *SX_def_var(argl)
   object *argl;
   {char *vname, *vtype, *s;
    byte *viv;
    DECLFPTR(byte, vif, (byte *p, long sz, char *s));
    hashel *hp;
    SC_address addr;
    object *oviv, *ovif, *obj;
    PA_variable *pp;
    int i, dm, *pv, *mini, *maxi, meth;
    int enough, at, vattr[N_ATTRIBUTES];
    PA_dimens *vdims, *next, *prev;
    pcons *nu, *du, *nxt, *prv;

    SS_args(argl,
            SC_STRING_I, &vname,
            SC_STRING_I, &vtype,
            SS_OBJECT_I, &oviv,
            SS_OBJECT_I, &ovif,
            0);

    for (i = 0; i < 4; i++, argl = SS_cdr(argl));

    if (!SS_nullobjp(oviv))
       {if (!SS_numberp(oviv))
           {s = SS_get_string(oviv);
            hp = SC_lookup(s, PA_symbol_tab);
            if (strcmp(hp->type, "variable") == 0)
               viv = hp->def;};};

    if (!SS_nullobjp(ovif))
       {s = SS_get_string(ovif);
        hp = SC_lookup(s, PA_symbol_tab);
        if (strcmp(hp->type, "procedure") == 0)
           {addr.memaddr = hp->def;
            vif          = (PFByte) addr.funcaddr;};};

/* make the variable hash table if it doesn't exist yet */
    if (PA_variable_tab == NULL)
       PA_variable_tab = SC_make_hash_table(HSZLARGE, DOC);

    vattr[0] = RUNTIME;
    vattr[1] = OPTL;
    vattr[2] = REL;
    vattr[3] = U_CENT;
    vattr[4] = DYNAMIC;
    enough   = FALSE;
    while (!enough)
       {SX_GET_INTEGER_FROM_LIST(at, argl,
                                 "CAN'T GET ATTRIBUTE - SX_DEF_VAR");
        switch (at)
           {case SCOPE      :
                 SX_GET_INTEGER_FROM_LIST(vattr[0], argl,
                                          "CAN'T GET ATTRIBUTE VALUE - SX_DEF_VAR");
                 break;
            case CLASS      :
                 SX_GET_INTEGER_FROM_LIST(vattr[1], argl,
                                          "CAN'T GET ATTRIBUTE VALUE - SX_DEF_VAR");
                 break;
            case PERSIST    :
                 SX_GET_INTEGER_FROM_LIST(vattr[2], argl,
                                          "CAN'T GET ATTRIBUTE VALUE - SX_DEF_VAR");
                 break;
            case CENTER     :
                 SX_GET_INTEGER_FROM_LIST(vattr[3], argl,
                                          "CAN'T GET ATTRIBUTE VALUE - SX_DEF_VAR");
                 break;
            case ALLOCATION :
                 SX_GET_INTEGER_FROM_LIST(vattr[4], argl,
                                          "CAN'T GET ATTRIBUTE VALUE - SX_DEF_VAR");
                 break;
            case ATTRIBUTE  :
                 enough = TRUE;
                 break;
            default         :
                 PA_ERR(TRUE,
                        "BAD ATTRIBUTE %d - SX_DEF_VAR",
                        at);};};

/* get the dimensions */
    vdims = NULL;
    while (TRUE)
       {maxi = _SX_index_ptr(&argl, "BAD DIMENSION NAME - SX_DEF_VAR");
        if (maxi == NULL)
           break;

        if (maxi == PA_DUL)
           {mini = _SX_index_ptr(&argl, "BAD LOWER INDEX - SX_DEF_VAR");
            maxi = _SX_index_ptr(&argl, "BAD UPPER INDEX - SX_DEF_VAR");
            meth = *PA_DUL;}

        else if (maxi == PA_DON)
           {mini = _SX_index_ptr(&argl, "BAD OFFSET - SX_DEF_VAR");
            maxi = _SX_index_ptr(&argl, "BAD NUMBER - SX_DEF_VAR");
            meth = *PA_DON;}

        else
           {mini = &Zero_I;
            meth = *PA_DON;};

        next = _PA_mk_dimens(mini, maxi, meth);
        if (vdims == NULL)
           vdims = next;
        else
           prev->next = next;
        prev = next;};

/* get the units */
    nu = NULL;
    while (TRUE)
       {SX_GET_INTEGER_FROM_LIST(dm, argl,
                                 "BAD UNIT - SX_DEF_VAR");
        if ((dm == PER) || (dm == UNITS))
           break;
        pv  = FMAKE(int, "SX_DEF_VAR:pv");
        *pv = dm;
        nxt = SC_mk_pcons("integer *", pv, SC_PCONS_P_S, NULL);
        if (nu == NULL)
           nu = nxt;
        else
           prv->cdr = (byte *) nxt;
        prv = nxt;};

    du = NULL;
    if (dm != UNITS)
       while (TRUE)
          {SX_GET_INTEGER_FROM_LIST(dm, argl,
                                    "BAD DENOMINATOR UNIT - SX_DEF_VAR");
           if (dm == UNITS)
              break;

           pv  = FMAKE(int, "SX_DEF_VAR:pv");
           *pv = dm;
           nxt = SC_mk_pcons("integer *", pv, SC_PCONS_P_S, NULL);
           if (du == NULL)
              du = nxt;
           else
              prv->cdr = (byte *) nxt;
           prv = nxt;};

    pp = _PA_mk_variable(vname, vdims, viv, vif,
                         vattr, vtype,
                         1.0, 1.0, nu, du, NULL);

    PA_VARIABLE_DATA(pp) = NULL;

    SC_install(vname, pp, PAN_VARIABLE, PA_variable_tab);

/* install scalars as implicit commands */
    if ((pp->n_dimensions == 0) && (PA_commands != NULL))
       {byte *vaddr;
        int itype;

        if (strcmp(vtype, SC_INTEGER_S) == 0)
           itype = SC_INTEGER_I;
        else if (strcmp(vtype, SC_DOUBLE_S) == 0)
           itype = SC_DOUBLE_I;
        else if (strcmp(vtype, SC_CHAR_S) == 0)
           itype = SC_CHAR_I;
        else
           PA_WARN(TRUE,
                   "TYPE %s UNSUPPORTED AS IMPLICIT COMMAND - SX_DEF_VAR",
                   vtype);

        PA_CONNECT(vaddr, vname, FALSE);

        PA_inst_c(vname, vaddr, itype, 0, PA_pshand, PA_sargs);};

    return(SX_mk_variable(pp));}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_INDEX_PTR - get the next dimension index pointer from the list
 *               - return special values if the dimension name matches
 *               -   "dimension"     - NULL
 *               -   "upper-lower"   - PA_DUL
 *               -   "offset-number" - PA_DON
 *               - also return NULL on error
 */

static int *_SX_index_ptr(pargl, msg)
   object **pargl;
   char *msg;
   {char *ds;
    object *obj;
    hashel *hp;

    SX_GET_STRING_FROM_LIST(ds, *pargl, msg);
    if (strcmp(ds, "dimension") == 0)
       return(NULL);

    else if (strcmp(ds, "upper-lower") == 0)
       return(PA_DUL);

    else if (strcmp(ds, "offset-number") == 0)
       return(PA_DON);

    else if ((hp = SC_lookup(ds, PA_symbol_tab)) != NULL)
       {if (strcmp(hp->type, "variable") == 0)
           return((int *) hp->def);};

    return((int *) SS_var_reference(ds));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RD_RESTART - read a restart dump from Scheme level */

static object *SX_rd_restart(argl)
   object *argl;
   {int convs;
    char *name;

    name  = NULL;
    convs = NONE;
    SS_args(argl,
            SC_STRING_I, &name,
            SC_INTEGER_I, &convs,
            0);

    if (name == NULL)
       SS_error("BAD FILE NAME - SX_RD_RESTART", argl);

    PA_rd_restart(name, convs);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_INIT_PROBLEM - execute PA_init_system at Scheme level
 *                 - this should happen after the dump has been read
 */

static object *SX_init_problem(argl)
   object *argl;
   {int nc;
    double t, dt;
    char *edname, *ppname, *gfname;

    t  = 0.0;
    dt = 0.0;
    nc = 0;
    edname = NULL;
    ppname = NULL;
    gfname = NULL;
    SS_args(argl,
            SC_DOUBLE_I, &t,
            SC_DOUBLE_I, &dt,
            SC_INTEGER_I, &nc,
            SC_STRING_I, &edname,
            SC_STRING_I, &ppname,
            SC_STRING_I, &gfname,
            0);

    t  *= unit[SEC]/convrsn[SEC];
    dt *= unit[SEC]/convrsn[SEC];

/* read the source files, initialize the packages */
    PA_init_system(t, dt, nc, edname, ppname, gfname);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_INST_COM - execute PA_inst_com at Scheme level */

static object *SX_inst_com()
   {PA_inst_com();

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_READH - execute PA_readh at Scheme level */

static object *SX_readh(argl)
   object *argl;
   {char *name;

    name = NULL;
    SS_args(argl,
            SC_STRING_I, &name,
            0);

    PA_readh(name);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PAN_CMMND - execute the command specified in the given string */

static object *SX_pan_cmmnd(argl)
   object *argl;
   {char *s, *token, *t;
    hashel *hp;
    PA_command *cp;

    s = NULL;
    SS_args(argl,
            SC_STRING_I, &s,
            0);

    token = SC_strtok(s, " \n\r\t/(", t);
    if (token != NULL)
       {hp = SC_lookup(token, PA_commands);
        if (hp != NULL)
           {cp = (PA_command *) hp->def;
            (*(cp->handler))(cp);};

        return(SS_t);}

   else
      return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_FIN_SYSTEM - execute fin_system at Scheme level */

static object *SX_fin_system(argl)
   object *argl;
   {int nz, cy;

    nz = 0;
    cy = 0;
    SS_args(argl,
            SC_INTEGER_I, &nz,
            SC_INTEGER_I, &cy,
            0);

    PA_fin_system(nz, cy, FALSE);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DUMP_PP - execute PA_dump_pp and PA_file_mon at Scheme level */

static object *SX_dump_pp(argl)
   object *argl;
   {double t, dt;
    int cy;
    char *ed, *pp, *gf;

    t  = 0.0;
    dt = 0.0;
    cy = 0;
    ed = NULL;
    pp = NULL;
    gf = NULL;
    SS_args(argl,
            SC_DOUBLE_I, &t,
            SC_DOUBLE_I, &dt,
            SC_INTEGER_I, &cy,
            SC_STRING_I, &ed,
            SC_STRING_I, &pp,
            SC_STRING_I, &gf,
            0);

    PA_dump_pp(t, dt, cy);

    PA_file_mon(ed, pp, gf);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DB_NUMERIC_DATA - connect to numeric data in PANACEA data base
 *                    -
 *                    - (pa-variable->pm-array name)
 */

static object *SX_db_numeric_data(obj)
   object *obj;
   {char *name, *type;
    double *dp;
    float *fp;
    int *ip, i, n;
    long *lp;
    short *sp;
    char *cp;
    C_array *arr;
    PA_variable *pp;
    byte *pd;

    name = NULL;
    SS_args(obj,
            SC_STRING_I, &name,
            0);

/* if no other arguments, read the variable */
    pp = PA_INQUIRE_VARIABLE(name);
    if (pp == NULL)
       SS_error("VARIABLE NOT FOUND - SX_DB_NUMERIC_DATA", obj);

    PA_CONNECT(pd, name, FALSE);
    n    = PD_entry_number(pp->desc);
    type = PD_entry_type(pp->desc);

/* all arrays will be REALS */
    arr         = FMAKE(C_array, "SX_DB_NUMERIC_DATA:arr");
    arr->type   = SC_DOUBLE_S;
    arr->length = n;
    if (strcmp(type, "double") == 0)
       arr->data = pd;
    else
       {dp        = FMAKE_N(double, n, "SX_DB_NUMERIC_DATA:dp");
        arr->data = (byte *) dp;
        if (strcmp(type, "float") == 0)
           {fp = (float *) pd;
            for (i = 0; i < n; i++)
                *dp++ = *fp++;}
        else if (strcmp(type, "integer") == 0)
           {ip = (int *) pd;
            for (i = 0; i < n; i++)
                *dp++ = *ip++;}
        else if (strcmp(type, "short") == 0)
           {sp = (short *) pd;
            for (i = 0; i < n; i++)
                *dp++ = *sp++;}
        else if (strcmp(type, "long") == 0)
           {lp = (long *) pd;
            for (i = 0; i < n; i++)
                *dp++ = *lp++;}
        else if (strcmp(type, "char") == 0)
           {cp = (char *) pd;
            for (i = 0; i < n; i++)
                *dp++ = *cp++;}
        else
           SS_error("BAD DATA TYPE - SX_DB_NUMERIC_DATA", obj);};

    return(SX_mk_C_array(arr));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WR_RESTART - write a restart dump from Scheme level */

static object *SX_wr_restart(argl)
   object *argl;
   {char *name;

    name = NULL;
    SS_args(argl,
            SC_STRING_I, &name,
            0);

    if (name == NULL)
       PA_wr_restart("state.tmp");
    else
       PA_wr_restart(name);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
