/* Quinn Diff, Compares two Packages files looking for differences between Archs */
/* Copyright (C) 1997-1998, James Troup <james@nocrew.org> */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#include <db.h>
#include "common.h"
#include "xmalloc.h"
#include "error.h"
#include "utils.h"
#include "arch_specific.h"

/* Public functions */

HASHINFO *arch_specific_hash_info;
DB *arch_specific_table = NULL;
DBT *arch_specific_dbt, *arch_specific_key;
char package_key[PACKAGE_NAME_LENGTH];

void
arch_specific_hash_create (void)
{

  arch_specific_hash_info = xmalloc (sizeof(*arch_specific_hash_info));

  arch_specific_hash_info->bsize = 256;
  arch_specific_hash_info->cachesize = 1048576;
  arch_specific_hash_info->ffactor = 8;
  arch_specific_hash_info->hash = NULL; 
  arch_specific_hash_info->lorder = 0;
  arch_specific_hash_info->nelem = 1;

  arch_specific_table = dbopen (NULL, O_RDWR, 0, DB_HASH, arch_specific_hash_info);

  if (arch_specific_table == NULL)
    fubar (SYSERR, "Unable to create hash table for arch_specific buffer");

  arch_specific_dbt = xmalloc (sizeof(*arch_specific_dbt));
  arch_specific_dbt->size = sizeof (struct arch_specific_info);

  arch_specific_key = xmalloc (sizeof(*arch_specific_key));
  arch_specific_key->data = package_key;

}

void arch_specific_read(void)
{

  FILE *fp = NULL; 
  char *buffer_p;
  long buffer_length;

  open_arch_specific_file (fp, &buffer_p, &buffer_length);
  debug (debug_as, "read_arch_specific_file: read %ld bytes of \"%s\"", 
	buffer_length, packages_arch_specific_filename);

  read_arch_specific_packages (buffer_p, buffer_length);

}

int is_arch_specific(const char *package_name)
{

  strcpy (package_key, package_name); 
  arch_specific_key->size = strlen (package_name); 
  
  /* FIXME: WTF do I do with flags?  Presently set to 0. */

  if (arch_specific_table->get (arch_specific_table, arch_specific_key, arch_specific_dbt, 0) != 0)
    {
      debug (debug_as, "is_arch_specific: %s is not.", package_name);
      return 0;
    }
  else
    {
      debug (debug_as, "is_arch_specific: %s is.", package_name);
      return 1;
    }
}


/* Private functions */

void open_arch_specific_file(FILE *fp, char **buffer_p, long *buffer_length)
{
  long n_read;                   

  fp = fopen (packages_arch_specific_filename, "r");
  if (fp == NULL)
    fubar (SYSERR, "%s: couldn't open file \"%s\"",
	   program_name, packages_arch_specific_filename);
  if ((fseek (fp, 0, SEEK_END)) != 0)
    fubar (SYSERR, "%s: couldn't seek to the end of file \"%s\"",
	   program_name, packages_arch_specific_filename);
  *buffer_length = ftell (fp);
  rewind (fp);

  /* FIXME: this probably shouldn't bomb out */

  if (*buffer_length == 0)
    fubar (NONSYS, "%s: \"%s\" is zero-length.", program_name, packages_arch_specific_filename);

  /* We don't do any memory management, we just ask for a chunk of
   * memory the size of the file */

  *buffer_p = xmalloc (*buffer_length);

  n_read = fread (*buffer_p, sizeof (char), *buffer_length, fp);
  if (n_read == 0)
    fubar (SYSERR, "%s: couldn't read file \"%s\" into buffer",
	   program_name, packages_arch_specific_filename);
  else if (n_read < *buffer_length)
    fubar (SYSERR, "%s: short read (found %ld, expected %ld) in file \"%s\"",
	   program_name, n_read, *buffer_length, packages_arch_specific_filename);

  if (fclose (fp) == EOF)
    fubar (SYSERR, "%s: couldn't close file \"%s\"", program_name, 
	   packages_arch_specific_filename);
}


void read_arch_specific_packages (const char *buffer_p, const long buffer_length)
{

  char line_buffer[LINE_BUFFER_LENGTH];
  char architectures[LINE_BUFFER_LENGTH]; 
  char package_name[LINE_BUFFER_LENGTH];
  int i, j, k, l, found_end;

  check_parseable (buffer_p, buffer_length);

  i = 0;
  
  while (i < buffer_length)
    {
      
      j = 0;
      line_buffer[j] = '\0';
      found_end = 0;
      
      /* Parse a line from the buffer, until we: */
      
      while (!found_end)
	{
	  if (buffer_p[i] == '#')                /* find a comment marker */
	    {
	      if (j == 0)
		line_buffer[j++]='\0';
	      else
		line_buffer[j++]='\n';
	      skip_line (buffer_p, &i, buffer_length);
	      found_end = 1;
	    }
	  else if (buffer_p[i] == '\n')          /* hit the end of line */
	    {
	      if (j == 0)
		line_buffer[j++]='\0';
	      else
		line_buffer[j++]='\n';
	      i++;
	      found_end = 1;
	    }
	  else if (j == LINE_BUFFER_LENGTH - 2)   /* run out of space in the line buffer */
	    {
	      line_buffer[j++]='\n';
	      skip_line (buffer_p, &i, buffer_length);
	      found_end = 1;
	    }
	  else
	    {
	      line_buffer[j++] = buffer_p[i++];
	    }
	}
      line_buffer[j]='\0';

      if (line_buffer[0] != '\0')
	{
	  /* Split up the line into package and supported architectures */

	  j = 0;
	  k = 0;
	  while (line_buffer[j] != ':' && j < LINE_BUFFER_LENGTH)
	    package_name[k++] = line_buffer[j++];
	  package_name[k] ='\0';
	  k = 0;
	  l = LINE_BUFFER_LENGTH - j;
	  architectures[k] = '\0';
	  while (k < l)
	    architectures[k++] = line_buffer[j++];

	  /* Check for "!arch" or "arch" */

	  if (strchr (architectures, '!') != NULL)
	    {
	      /* "!arch" is exclusionary; if there's a !something, but
                 not !arch there is support for arch */
	      if (strstr (architectures, secondary_architecture) != NULL)
		{
		  debug (debug_as, "read_arch_specficic_packages: %s doesn't need compiled on %s", 
			 package_name, secondary_architecture);
		  arch_specific_hash_add(package_name);
		}
	    }
	  else
	    {
	      /* "arch" is inclusionary; if arch is not listed, it's not
                 supported by the package */
	      if (strstr (architectures, secondary_architecture) != NULL)
		debug (debug_as, "read_arch_specficic_packages: %s has %s support", 
		       package_name, secondary_architecture);
	      else
		{
		  debug (debug_as, "read_arch_specficic_packages: %s doesn't need compiled on %s", 
			 package_name, secondary_architecture);
		  arch_specific_hash_add(package_name);
		}
	    }
	}
    }

}

void arch_specific_hash_add (const char *package_name)
{

  int i;

  debug (debug_as, "arch_specific_hash_add: Adding \"%s\" to arch specific hash table", package_name);

  strcpy (package_key, package_name);
  arch_specific_key->size = strlen (package_name);
  
  debug (debug_as, "arch_specific_hash_add: key is %s and key size is %d", 
	(char *) arch_specific_key->data, arch_specific_key->size);

  i = arch_specific_table->put (arch_specific_table, arch_specific_key, arch_specific_dbt, R_NOOVERWRITE);
  if (i == 1)
    debug (debug_warn, "\"%s\" is duplicated in Packages-arch-specific", package_name);
  else if (i == -1)
    fubar (SYSERR, "couldn't add %s to the arch-specific hash table", package_name);

}

  /* First line should be a special comment field giving the version
     number of the Packages-arch-specific file; use this to ensure
     we're not trying to parse something we don't understand */

void check_parseable (const char *buffer_p, const long buffer_length)
{

  char line_buffer[LINE_BUFFER_LENGTH];
  char *arch_specific_version_string, *errors;
  int i;
  long arch_specific_version;

  i = 0;

  if (buffer_p[i] == '#')
    {
      while (i < buffer_length && i < LINE_BUFFER_LENGTH && buffer_p[i] != '\n')
	{
	  line_buffer[i] = buffer_p[i];
	  i++;
	}
      line_buffer[i] = '\0';
      arch_specific_version_string = strstr (line_buffer, "Packages-arch-specific-version: ");
      if (arch_specific_version_string != NULL)
	{
	  arch_specific_version_string += 32;
	  arch_specific_version = strtol(arch_specific_version_string, &errors, 10);
	  if (errors != arch_specific_version_string)
	    {
	      if (arch_specific_version > NEWEST_PARSEABLE_VERSION)
		fubar (NONSYS, "I can't understand this type of Packages-arch-specific file; please upgrade.");
	    }
	  else
	    fubar (NONSYS, "I can't understand this type of Packages-arch-specific file; please upgrade.");
	}
    }

}
