/*-----------------------------------------------------------------*-C-*---
 * File:    handc/cfg/gc/irc/writebar.c
 *
 *          Copyright (C)1997 Donovan Kolbly <d.kolbly@rscheme.org>
 *          as part of the RScheme project, licensed for free use.
 *          See <http://www.rscheme.org/> for the latest information.
 *
 * File version:     1.9
 * File mod date:    1997.12.18 22:11:02
 * System build:     v0.7.2, 97.12.21
 *
 * Purpose:          IRC write barrier implementation
 *------------------------------------------------------------------------*/

#include "irc.h"
#include "alloc.h"

#ifdef INLINES
#include <rscheme/gcserver1.ci>
#include <rscheme/gcclient1.ci>
#endif

/*
   the write barrier tripped
   do the appropriate action
*/

static void store_into_pstore( IRC_Heap *heap, 
			      void *lvalue, UINT_32 offset, 
			      void *rvalue );

int IRC_tripWriteBarrier( IRC_Heap *heap, 
		          void *lvalue, UINT_32 offset, 
		          void *rvalue )
{
    if (IRCH_WRITEPROT(lvalue))
    {
	IRC_clientWriteViolation( heap, lvalue, offset );
	return 1;
    }
    else
    {
      char wbcode;
      
      wbcode = IRC_writeBarrierCode( heap, lvalue, rvalue );
      if (atracef) {
	fprintf( atracef, 
		"write barrier trip (%d): lvalue = %p (offset +%u), "
		"rvalue = %p\n", 
		wbcode,
		lvalue, offset, rvalue );
	fflush( atracef );
      }
	switch (wbcode)
	{
	    case WB_NONE:
	    		/* why were we tripped? */
			break;
	    case WB_COLOR:
			{
			  struct IRC_Header *l = IRCH(lvalue);

			  /* (note that we can't set the color
			     bit in the lvalue back to WHITE, because
			     then during traversal, we would crawl
			     over it again (snapping it out of the
			     black list an into the gray list)
			     (is that the only reason?)
			  */
	    		/* write of a pointer to a white object
			   into a black (or gray) object
			*/
			/* regray the lvalue */
			/* but only if it's not already being regrayed */
			if (!(l->flagBits & IRC_MASK_REGRAYQ))
			{
			  if (atracef) {
			    fprintf(atracef,
				    "  -- %p added to regray list\n", l);
			    fflush(atracef);
			  }
			    IRC_ptrListAdd( &l->sizeClass->gen->regrayObjects,
					    l );
			    l->flagBits |= IRC_MASK_REGRAYQ;
			}
			  else
			    {
			      if (atracef) {
				fprintf(atracef,
					"  -- %p already in regrayed list\n", 
					IRCH(lvalue));
				fflush(atracef);
			      }
			    }
			}
			return 1;
	    case WB_GENERATION:
	    		/* write of a pointer to a younger object
			   into an older object
			*/
			/* put rvalue into the IGP list */
			return 1;
	    case WB_GLOBAL:
	    		/* write into a global shared object */
			return 1;
	    case WB_PERSISTENT:
			/* a store into a persistent object --
			   record lptr as an "external possptr",
			   but only if we're not overwriting a 
			   pointer into the same generation */
			store_into_pstore( heap, lvalue, offset, rvalue );
			return 1;
	}
    }
    return 0;
}

#ifndef GC_MACROS
int IRC_writeBarrier( IRC_Heap *heap, 
		      void *lvalue, UINT_32 offset, 
		      void *rvalue )
{
    if (IRC_writeBarrierCode( heap, lvalue, rvalue ))
	return IRC_tripWriteBarrier( heap, lvalue, offset, rvalue );
    return 0;
}
#endif

static void store_into_pstore( IRC_Heap *heap, 
			       void *lvalue, UINT_32 offset, 
			       void *rvalue )
{
  struct IRC_Gen *rgen = IRCH(rvalue)->sizeClass->gen;
  pos_ptr_addr pp;
  gc_obj_addr overwriting;

  pp = (pos_ptr_addr) (offset + (char *)lvalue);
  overwriting = cast_and_deref_ptr( pp );
  if (!overwriting || IRCH(overwriting)->sizeClass->gen != rgen)
    IRC_ptrListAdd( &rgen->extraHeapPointers, (IRC_Header *)pp );
}
