/* vi:set ts=8 sts=0 sw=8:
 * $Id: win.c,v 1.15 1998/10/16 20:21:01 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdarg.h>
#include <string.h>
#include "win.h"
#include "doc.h"
#include "file.h"
#include "menu.h"
#include "toolbar.h"
#include "msgbar.h"
#include "msgbox.h"
#include "prefs.h"
#include "recent.h"
#include "prjbar.h"


/*** external declarations ***/
extern void	win_set_title(doc_t *d);


/*** local declarations ***/
#define AUTOSAVE_MSG	"Autosaving changed files..."
#define DOC_TABS_OFF	"Document tabs OFF"
#define DOC_TABS_ON	"Document tabs ON"
#define DOC_TABS_TOP	"Document tabs on the TOP"
#define DOC_TABS_BOTTOM	"Document tabs on the BOTTOM"
#define DOC_TABS_LEFT	"Document tabs on the LEFT"
#define DOC_TABS_RIGHT	"Document tabs on the RIGHT"


/*** local variables ***/
static GList *winlist = NULL;		/* list of win_t's */
static unsigned nextid = 10000;		/* internal window id for each win_t */

static GtkWidget *wlw = NULL;		/* these three are used for the */
static GtkWidget *wlw_data = NULL;	/* winlist popup window */
static unsigned curwinrow = 0;		/* current window rownum */


/*** local function prototypes ***/
static void notebook_init(win_t *w);
static void notebook_switch_page(GtkWidget *wgt, GtkNotebookPage *page,
	int num, gpointer cbdata);
static bool_t win_close_common(win_t *w);
static void win_list_destroy(GtkWidget *wgt, gpointer cbdata);
static void win_list_remove(win_t *w);
static void win_list_add(win_t *w, char *title);
static void win_list_select(GtkWidget *wgt, unsigned row, int column,
	GdkEventButton *event, gpointer cbdata);
static void win_list_close_selected(GtkWidget *wgt, gpointer cbdata);


/*** global function definitions ***/
/*
 * PUBLIC: win_autosave
 *
 * create a new window with a document.  currently, only creates an "Untitled"
 * document.
 */
int
win_autosave(gpointer cbdata)
{
	GList *wlp, *dlp;
	win_t *w;
	doc_t *d;
	int num = 0;

	msgbox_printf("Autosave in progress...");
	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		msgbar_printf(w, AUTOSAVE_MSG);
		dlp = (GList *)(w->doclist);
		while (dlp) {
			d = (doc_t *)(dlp->data);
			if (d->changed) {
				(void)file_save_execute(d);
				num++;
			}
			dlp = dlp->next;
		}
		wlp = wlp->next;
	}
	msgbox_printf("Autosave completed (%d files saved)", num);

	return 1;	/* for GTK, must return 1 */
} /* win_new_with_new_doc */


/*
 * PUBLIC: win_redraw_doc_tab
 *
 * for all windows, redraw the document tabs.  called from prefs_save, when
 * the preferences may have changed.
 */
void
win_redraw_doc_tab(gpointer data)
{
	GList *wlp;
	win_t *wp;

	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		if (IS_SHOW_TABS())
			gtk_notebook_set_show_tabs(
					GTK_NOTEBOOK(wp->notebook), TRUE);
		else
			gtk_notebook_set_show_tabs(
					GTK_NOTEBOOK(wp->notebook), FALSE);
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(wp->notebook), tabpos);
		wlp = wlp->next;
	}
} /* win_redraw_doc_tab */


/*
 * PUBLIC: win_redraw_msgbar
 *
 * for all windows, redraw the toolbar.  called from prefs_save, when
 * the preferences may have changed.
 */
void
win_redraw_msgbar(gpointer data)
{
	GList *wlp;
	win_t *wp;

	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		msgbar_redraw(wp);
		wlp = wlp->next;
	}
} /* win_redraw_msgbar */


#ifdef WANT_PROJECT
/*
 * PUBLIC: win_redraw_prjbar
 *
 * for all windows, redraw the toolbar.  called from prefs_save, when
 * the preferences may have changed.
 */
void
win_redraw_prjbar(gpointer data)
{
	GList *wlp;
	win_t *wp;

	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		prjbar_redraw(wp);
		wlp = wlp->next;
	}
} /* win_redraw_prjbar */
#endif	/* WANT_PROJECT */


/*
 * PUBLIC: win_redraw_toolbar
 *
 * for all windows, redraw the toolbar.  called from prefs_save, when
 * the preferences may have changed.
 */
void
win_redraw_toolbar(gpointer data)
{
	GList *wlp;
	win_t *wp;

	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		tb_redraw(wp);
		wlp = wlp->next;
	}
} /* win_redraw_toolbar */


/*
 * PUBLIC: win_new_with_doc
 *
 * create a new window with a document.  currently, only creates an "Untitled"
 * document.
 */
void
win_new_with_doc(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w;

	w = win_new();
	doc_new(w, UNTITLED);
	gtk_widget_show(w->toplev);
} /* win_new_with_new_doc */


/*
 * PUBLIC: win_new
 *
 * creates a new window and adds it to the winlist.
 */
win_t *
win_new(void)
{
	win_t *w;

	w	= (win_t *)g_malloc0(sizeof(win_t));
	w->id	= nextid++;

	/* toplevel */
	w->toplev = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_name(w->toplev, "eedit");
	gtk_window_set_wmclass(GTK_WINDOW(w->toplev), "eedit", "eedit");
	gtk_window_set_title(GTK_WINDOW(w->toplev), "Title");
	gtk_window_set_policy(GTK_WINDOW(w->toplev), TRUE, TRUE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(w->toplev), 0);
	gtk_signal_connect(GTK_OBJECT(w->toplev), "destroy",
		GTK_SIGNAL_FUNC(win_close_cb), w);

	gtk_widget_set_usize(GTK_WIDGET(w->toplev), win_width, win_height);
	if (IS_SAVE_WIN_POS())
		gtk_widget_set_uposition(GTK_WIDGET(w->toplev),
					win_xpos, win_ypos);

	/* mainbox */
	w->mainbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(w->toplev), w->mainbox);
	gtk_widget_show(w->mainbox);

	/* msgbox can be initialized anywhere before a file is opened */
	msgbox_init();

	/* these need to be in order */
	menu_main_init(w);
	recent_list_init(w);
	toolbar_main_init(w);
	notebook_init(w);
#ifdef WANT_PROJECT
	prjbar_init(w);
#endif	/* WANT_PROJECT */

	/* msgbar and docinfo button go together */
	w->hbox_bot = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(w->mainbox), w->hbox_bot, FALSE, FALSE, 0);
	gtk_widget_show(w->hbox_bot);
	msgbar_init(w, w->hbox_bot);
	doc_info_init(w, w->hbox_bot);

	quickmenu_init(w);

	/* add to winlist and winlist popup */
	winlist = g_list_append(winlist, (gpointer)w);
	win_list_add(w, GTK_WINDOW(w->toplev)->title);

	return w;
} /* win_new */


/*
 * PUBLIC: win_close_cb
 *
 * window close callback
 */
void
win_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_close_common((win_t *)cbdata);
} /* win_close_cb */


/*
 * PUBLIC: win_close_all_cb
 *
 * close all windows (e.g., quit) callback
 */
void
win_close_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	msgbox_close();
	while (winlist) {
		if (win_close_common((win_t *)(winlist->data)) == FALSE)
			break;
	}

	g_assert(winlist != NULL);
} /* win_close_all_cb */


/*
 * PUBLIC: win_dtab_toggle
 *
 * callback: toggles word wrap
 */
void
win_dtab_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (IS_SHOW_TABS()) {
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->notebook), FALSE);
		CLEAR_SHOW_TABS();
		msgbar_printf(w, DOC_TABS_OFF);
	} else {
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->notebook), TRUE);
		SET_SHOW_TABS();
		msgbar_printf(w, DOC_TABS_ON);
	}
} /* win_dtab_toggle */


/*
 * PUBLIC: win_dtab_top
 *
 * callback: toggles word wrap
 */
void
win_dtab_top(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->notebook), GTK_POS_TOP);
	tabpos = GTK_POS_TOP;
	msgbar_printf(w, DOC_TABS_TOP);
} /* win_dtab_top */


/*
 * PUBLIC: win_dtab_bot
 *
 * callback: toggles word wrap
 */
void
win_dtab_bot(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->notebook), GTK_POS_BOTTOM);
	tabpos = GTK_POS_BOTTOM;
	msgbar_printf(w, DOC_TABS_BOTTOM);
} /* win_dtab_bot */


/*
 * PUBLIC: win_dtab_left
 *
 * callback: toggles word wrap
 */
void
win_dtab_left(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->notebook), GTK_POS_LEFT);
	tabpos = GTK_POS_LEFT;
	msgbar_printf(w, DOC_TABS_LEFT);
} /* win_dtab_left */


/*
 * PUBLIC: win_dtab_right
 *
 * callback: toggles word wrap
 */
void
win_dtab_right(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->notebook), GTK_POS_RIGHT);
	tabpos = GTK_POS_RIGHT;
	msgbar_printf(w, DOC_TABS_RIGHT);
} /* win_dtab_right */


/*
 * PUBLIC: win_toggle_wwrap
 *
 * callback: toggles word wrap
 */
void
win_wwrap_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);
	if (IS_USE_WORDWRAP()) {
		gtk_text_set_word_wrap(GTK_TEXT(d->txt), FALSE);
		CLEAR_USE_WORDWRAP();
	} else {
		gtk_text_set_word_wrap(GTK_TEXT(d->txt), TRUE);
		SET_USE_WORDWRAP();
	}
} /* win_wwrap_toggle */


/*** local function definitions ***/
/*
 * PRIVATE: win_close_common
 *
 * common routine for actually closing a window.  called from win_close_cb()
 * and win_close_all_cb().
 */
static bool_t
win_close_common(win_t *w)
{
	g_assert(w != NULL);

	doc_list_destroy(NULL, w);
	win_list_destroy(NULL, NULL);

	if (doc_check_if_any_changed(w) == FALSE)
		gtk_widget_hide(w->toplev);

	if (doc_close_all_common(w) == TRUE) {
		doc_new_name_t *dnnp;
		GList *wlp, *dlp, *rlp;
		doc_t *d;

		/* update preferences */
		if (IS_SAVE_WIN_POS())
			gdk_window_get_position(w->toplev->window,
						&win_xpos, &win_ypos);

		/* remove from the win list popup window */
		win_list_remove(w);

		/* remove from the linked list of win_t's */
		wlp = g_list_find(winlist, w);
		winlist = g_list_remove_link(winlist, wlp);

		/* cleanup */
		while (w->doclist) {
			dlp = w->doclist;
			w->doclist = g_list_remove_link(w->doclist, dlp);
			d = (doc_t *)(dlp->data);
			g_free(d->fname);
			g_free(d->sb);
			gtk_widget_destroy(d->txt);
			gtk_widget_destroy(d->tablabel);
			if (d->printwin)
				gtk_widget_destroy(d->printwin);
			if (d->printentry)
				gtk_widget_destroy(d->printentry);
			g_free(d);
#if GLIST_MAGIC
			g_free(dlp);
#endif
		}
		while (w->recentlist) {
			rlp = w->recentlist;
			w->recentlist = g_list_remove_link(w->recentlist, rlp);
			dnnp = (doc_new_name_t *)(rlp->data);
			g_free(dnnp->fname);
			g_free(dnnp);
#if GLIST_MAGIC
			g_free(rlp);
#endif
		}
		g_free(w->lastmsg);
		gtk_timeout_remove(w->timeout_id);
		gtk_widget_destroy(w->toplev);
		g_free(w);
#if GLIST_MAGIC
		g_free(wlp);
#endif

		/* save preferences */
		if (IS_SAVE_WIN_WIDTH() || IS_SAVE_WIN_HEIGHT() ||
				IS_SAVE_WIN_POS())
			prefs_save();

		if (winlist == NULL)
			gtk_exit(0);

		return TRUE;
	}

	return FALSE;
} /* win_close_common */


/*
 * PRIVATE: notebook_init
 *
 * creates the notebook to contain all the documents.
 */
static void
notebook_init(win_t *w)
{
	w->notebook = gtk_notebook_new();
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w->notebook), TRUE);
	gtk_signal_connect_after(GTK_OBJECT(w->notebook), "switch_page",
		GTK_SIGNAL_FUNC(notebook_switch_page), w);
	gtk_box_pack_start(GTK_BOX(w->mainbox), w->notebook, TRUE, TRUE, 0);

	/* set according to preferences */
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->notebook), tabpos);
	if (IS_SHOW_TABS())
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->notebook), TRUE);
	else
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->notebook), FALSE);

	gtk_widget_show(w->notebook);
} /* notebook_init */


/*
 * PRIVATE: notebook_switch_page
 *
 * callback invoked when a switch page has occured on the notebook.
 */
static void
notebook_switch_page(GtkWidget *wgt, GtkNotebookPage *page,
	int num, gpointer cbdata)
{
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (w->doclist == NULL)
		return;

	if (!GTK_WIDGET_REALIZED(w->toplev))
		return;

	w->curdoc = g_list_nth_data(w->doclist, num);
	if (w->dlw_data)
		gtk_clist_select_row(GTK_CLIST(w->dlw_data), num, FnumCol);
	d = (doc_t *)(w->curdoc);
	gtk_widget_grab_focus(d->txt);
	win_set_title(d);
	win_list_set_curdoc(w);
} /* notebook_switch_page */


/*
 * PUBLIC: win_set_title
 *
 * sets the window title.
 */
void
win_set_title(doc_t *d)
{
	char *title;
	int len;

	len = strlen(doc_basefname(d)) + strlen(APP_NAME) + 5;
	title = (char *)g_malloc(len);
	g_snprintf(title, len, "%s - %s", APP_NAME, doc_basefname(d));
	gtk_window_set_title(GTK_WINDOW(d->w->toplev), title);
} /* win_set_title */


/*
 * PUBLIC: win_list_set_curdoc
 *
 * set the current document column/field for the window specified.
 */
void
win_list_set_curdoc(win_t *w)
{
	GtkCList *clist;
	doc_t *d;

	if (wlw_data == NULL)
		return;

	d = doc_current(w);
	clist = GTK_CLIST(wlw_data);
	gtk_clist_freeze(clist);
	gtk_clist_set_text(clist, w->rownum, FnameCol, doc_basefname(d));
	gtk_clist_thaw(clist);
} /* win_list_set_curdoc */


/*
 * PUBLIC: win_list_set_numdoc
 *
 * set the # of docs column in the window list popup.
 */
void
win_list_set_numdoc(win_t *w)
{
	GtkCList *clist;
	char buf[8];

	if (wlw_data == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);
	gtk_clist_freeze(clist);
	g_snprintf(buf, 16, "%d", w->numdoc);
	gtk_clist_set_text(clist, w->rownum, FsizeCol, buf);
	gtk_clist_thaw(clist);
} /* win_list_set_numdoc */


/*
 * PUBLIC: win_list_show
 *
 * creates a new popup window containing a list of open files/documents
 * for the window from which it was invoked.
 */
#define WIN_LIST_NUM_COLUMNS	3
#define INSERT			0
#define APPEND			1
void
win_list_show(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget *tmp, *vbox, *hbox;
	GList *wlp;
	win_t *wp;

	char *titles[] = { " # ", " # Docs ", " Current Document " };
	win_t *w = (win_t *)cbdata;

	if (wlw) {
		gdk_window_raise(wlw->window);
		return;
	}
	g_assert(w != NULL);

	/* create top level and vbox to hold everything */
	wlw = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(wlw), "Window List");
	gtk_signal_connect(GTK_OBJECT(wlw), "destroy",
		GTK_SIGNAL_FUNC(win_list_destroy), NULL);
	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(wlw), vbox);
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);

	/* create clist */
	wlw_data = gtk_clist_new_with_titles(WIN_LIST_NUM_COLUMNS, titles);
	gtk_clist_set_column_width(GTK_CLIST(wlw_data), FnumCol, 25);
	gtk_clist_set_column_justification(GTK_CLIST(wlw_data),
		FnumCol, GTK_JUSTIFY_LEFT);
	gtk_clist_set_column_width(GTK_CLIST(wlw_data), FsizeCol, 55);
	gtk_clist_set_column_justification(GTK_CLIST(wlw_data),
		FsizeCol, GTK_JUSTIFY_CENTER);
	gtk_clist_column_titles_passive(GTK_CLIST(wlw_data));
	gtk_clist_set_policy(GTK_CLIST(wlw_data),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), wlw_data, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(wlw_data),
		"select_row", GTK_SIGNAL_FUNC(win_list_select), NULL);

	/* for each window, add info into the clist */
	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		win_list_add(wp, doc_basefname(doc_current(wp)));
		wlp = wlp->next;
	}

	/* create a separator and then some handy buttons */
	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, TRUE, 0);
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	tmp = gtk_button_new_with_label("Close Selected");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(win_list_close_selected), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 2);

	tmp = gtk_button_new_with_label("Close All");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(win_close_all_cb), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 2);

	tmp = gtk_button_new_with_label("Exit Window List");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
			GTK_SIGNAL_FUNC(win_list_destroy), NULL);
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 2);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(tmp);

	/*
	 * there should be a way to auto-detect how wide to make these widgets
	 * instead of having to use arbitrary values
	 */
	gtk_widget_set_usize(wlw, 300, 200);

	gtk_widget_show_all(wlw);
} /* win_list_show */


/*
 * PUBLIC: win_list_destroy
 *
 * zap!
 */
static void
win_list_destroy(GtkWidget *wgt, gpointer cbdata)
{
	if (wlw) {
		gtk_widget_destroy(wlw);
		wlw = NULL;
		wlw_data = NULL;
	}
} /* win_list_destroy */


/*
 * PRIVATE: win_list_close_selected
 *
 * close the window that is highlighted in the win list popup.
 */
static void
win_list_close_selected(GtkWidget *wgt, gpointer cbdata)
{
	GList *wlp;
	win_t *w;

	if (wlw == NULL)
		return;

	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		if (w->rownum == curwinrow) {
			if (win_close_common(w) == TRUE)
				break;
		}
		wlp = wlp->next;
	}
} /* win_list_close_selected */


/*
 * PUBLIC: win_list_remove
 *
 * removes an entry from the files list window.
 */
static void
win_list_remove(win_t *w)
{
	char buf[4];
	int i;
	GtkCList *clist;
	GList *wlp;
	win_t *wp;
	
	if (wlw == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);
	g_assert(clist != NULL);
	gtk_clist_freeze(clist);
	gtk_clist_remove(clist, w->rownum);

	/* renumber any entries starting with the row just removed */
	if (clist->rows > 1) {
		for (i = w->rownum; i < clist->rows; i++) {
			g_snprintf(buf, 4, "%d", i + 1);
			gtk_clist_set_text(clist, i, FnumCol, buf);
		}
	}

	/* renumber rownum field for remaining windows */
	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		if (wp->rownum > w->rownum)
			wp->rownum--;
		wlp = wlp->next;
	}

	/* pick "next most recent" row to highlight */
	if (w->rownum == (unsigned)(clist->rows))
		gtk_clist_select_row(clist, w->rownum - 1, FnumCol);
	else
		gtk_clist_select_row(clist, w->rownum, FnumCol);
	gtk_clist_thaw(clist);
} /* win_list_remove */


/*
 * PRIVATE: win_list_add
 *
 * appends an entry from the files list window.
 */
static void
win_list_add(win_t *w, char *title)
{
	char numstr[4], numdocstr[8];
	char *rownfo[WIN_LIST_NUM_COLUMNS];
	GtkCList *clist;

	
	if (wlw == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);

	g_snprintf(numstr, 4, "%d", clist->rows + 1);
	rownfo[0] = numstr;
	g_snprintf(numdocstr, 8, "%4d", w->numdoc);
	rownfo[1] = numdocstr;
	rownfo[2] = title;
	gtk_clist_freeze(clist);
	gtk_clist_append(clist, rownfo);
	gtk_clist_select_row(clist, clist->rows - 1, FnumCol);
	gtk_clist_thaw(clist);
	w->rownum = clist->rows - 1;
} /* win_list_add */


/*
 * PRIVATE: win_list_select
 *
 * callback routine when selecting a window in the winlist popup.  this does
 * two things:
 *
 * 1. sets the curwinrow variable to the row that was selected.  this is used
 * for the "close" button on the winlist popup, so we know which window to
 * close.  we need this because the winlist itself is our own creation, and
 * there is no association at all with the close button's context.
 *
 * 2. raises the window which was selected.
 */
static void
win_list_select(GtkWidget *wgt, unsigned row, int column,
	GdkEventButton *event, gpointer cbdata)
{
	GList *wlp;
	win_t *w;

	curwinrow = row;
	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		if (w->rownum == row) {
			gdk_window_raise(w->toplev->window);
			break;
		}
		wlp = wlp->next;
	}
} /* win_list_select */


/* the end */
