//  UFrame.cpp version 1.1
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998  Gaspar Sinai
// 
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "UFrame.h"
#include "UShell.h"
#include <iostream.h>

UFrame::UFrame  (UFrame* parent_, int frameSize_) 
	: UComponent(parent_->top, ((UTop*)parent_->top)->display), 
	dark (parent_->getBackground()),
	light (parent_->getBackground())
{
	XGCValues	gcv;
	parent = parent_;
	top = parent->top;
	rectangle.width = parent->rectangle.width;
	rectangle.height = parent->rectangle.height;
	
	rectangle.x = 0;
	rectangle.y = 0;
	frameSize = frameSize_;
	placement.margin = 0;
	frameStyle = IN;
	window = XCreateSimpleWindow (UTOP->display,
		parent->window, rectangle.x, rectangle.y,
		rectangle.width, rectangle.height,
		0, parent->getBackground().getPixel(), 
		parent->getBackground().getPixel());

	if (rectangle.width<=frameSize*2) rectangle.width =  frameSize*2;
	if (rectangle.height<=frameSize*2) rectangle.height =  frameSize*2;

	UTOP->addComponent (this);

	gcv.background = parent->getBackground().getPixel();
	gcv.foreground = parent->getBackground().getPixel();
	lightGC = XCreateGC (UTOP->display, window, 
		GCForeground | GCBackground, &gcv);
	darkGC = XCreateGC (UTOP->display, window, 
		GCForeground | GCBackground, &gcv);
	
	XMapRaised (UTOP->display, window);
	select (ExposureMask | SubstructureNotifyMask);
	
	setBackground (parent->getBackground());
	setForeground (parent->getForeground());
	font = parent->getFont();

	parent->addChild (this);
}

void
UFrame::addChild (UComponent* child)
{
	children.addItem (child->window, child);
}

void
UFrame::deleteChild (UComponent* child)
{
	children.deleteItem (child->window, child);
}

void
UFrame::setFont (UFont* font_)
{
	UComponent*	comp;
	unsigned int	i;

	for (i=0; i< children.getSize(); i++)
	{
		comp = (UComponent*) children.at (i);
		if (comp==0) continue;
		comp->setFont (font_);
	}
	font = font_;
}

//
// To create a toplevel....
//
UFrame::UFrame  (UTop* top_, int frameSize_)
	: UComponent(top_, top_->display), 
	dark (top_->display),
	light (top_->display)
{
	rectangle.x = 0;
	rectangle.y = 0;
	frameSize = frameSize_;
	placement.margin = 0;
	frameStyle = IN;
	parent = 0;
	font = 0;

	top=top_;
}

UFrame::~UFrame()
{
	UComponent* 	comp;

	UTOP->deleteComponent (this);
	while (children.getSize())
	{
		comp=(UComponent*) children.at (0);
		// should not happen
		if (comp==0) break;
		delete comp;
	}
	if (parent) parent->deleteChild (this);
	XFreeGC (top->display, lightGC);
	XFreeGC (top->display, darkGC);
	XDestroyWindow (((UTop*)top)->display, window);
}

void
UFrame::setFrameStyle (UFrameStyle fstyle)
{
	frameStyle=fstyle;
	redraw (0, 0, rectangle.width, rectangle.height);
}

void
UFrame::eventDown (UEvent* event)
{
	XRectangle rect;

	switch (event->type)
	{
	case UEvent::TIMER:
			break;
	case UEvent::X:
		switch (event->xevent.type)
		{
		case Expose:
			rect.x = event->xevent.xexpose.x;
			rect.y = event->xevent.xexpose.y;
			rect.width = event->xevent.xexpose.width;
			rect.height = event->xevent.xexpose.height;
			redraw (rect.x, rect.y, rect.width, rect.height);
			break;

		case GraphicsExpose:
			rect.x = event->xevent.xgraphicsexpose.x;
			rect.y = event->xevent.xgraphicsexpose.y;
			rect.width = event->xevent.xgraphicsexpose.width;
			rect.height = event->xevent.xgraphicsexpose.height;
			redraw (rect.x, rect.y, rect.width, rect.height);
			break;
		default:
			break;
		}
	default:
		break;
	}
}

int
UFrame::isA (UComponent::UType type_)
{
	if (type_ == UComponent::FRAME) return 1;
	return UComponent::isA (type_);
}

void
UFrame::pack ()
{
	resize (rectangle.width, rectangle.height);
}

void
UFrame::resize (int width, int height)
{
	unsigned int		i;
	UComponent*		comp;
	int			x0, x1;
	int			y0, y1;

	x0 = (width>frameSize * 2)? width : frameSize * 2;
	y0 = (height>frameSize * 2)? height : frameSize * 2;

	if (frameSize>0)
	{
		XClearArea (UTOP->display, window, 
			rectangle.width-frameSize, 
			0, frameSize, frameSize, False);
		XClearArea (UTOP->display, window, 
			0, rectangle.height-frameSize, 
			frameSize, frameSize, False);
	}

	rectangle.width = x0;
	rectangle.height = y0;

	// We ARE resized. This is just a notification
	if (parent != 0) 
	{
		XResizeWindow (UTOP->display, window, 
			rectangle.width, rectangle.height);
	}
	// resize children
	for (i=0; i< children.getSize(); i++)
	{
		comp=(UComponent*) children.at (i);
		if (comp==0) continue;
		// Calculate new positionX for child.
		// Negative value in position indicates opposite side.
		// Position:  256 = full

		x0 = (comp->getPlacement().flags[UPlacement::WEST]==UPlacement::PLACE) ?
			comp->getPlacement().edges[UPlacement::WEST] 
			: (int)((int)rectangle.width * 
			  comp->getPlacement().edges[UPlacement::WEST]) >> 8;
		// Measured from opposite edge - 
		x0 = (comp->getPlacement().edges[UPlacement::WEST]<0) ? rectangle.width + x0 : x0;

		x1 = (comp->getPlacement().flags[UPlacement::EAST]==UPlacement::PLACE) ?
			comp->getPlacement().edges[UPlacement::EAST] 
			: (int)((int)rectangle.width * 
			   comp->getPlacement().edges[UPlacement::EAST]) >> 8;
		// Opposite
		x1 = (comp->getPlacement().edges[UPlacement::EAST]>=0) ? rectangle.width - x1 : -x1;

		y0 = (comp->getPlacement().flags[UPlacement::NORTH]==UPlacement::PLACE) ?
			comp->getPlacement().edges[UPlacement::NORTH] 
			: (int)((int)rectangle.height * 
			   comp->getPlacement().edges[UPlacement::NORTH]) >> 8;
		// Opposite
		y0 = (comp->getPlacement().edges[UPlacement::NORTH]<0) ? rectangle.height + y0 : y0;

		y1 = (comp->getPlacement().flags[UPlacement::SOUTH]==UPlacement::PLACE) ?
			comp->getPlacement().edges[UPlacement::SOUTH] 
			: (int)((int)rectangle.height * 
			    comp->getPlacement().edges[UPlacement::SOUTH])>>8;
		// Opposite
		y1 = (comp->getPlacement().edges[UPlacement::SOUTH]>=0) ? rectangle.height - y1 : -y1;
		
		comp->move (x0+comp->getPlacement().margin, y0+comp->getPlacement().margin);
		comp->resize (x1-x0-2*(comp->getPlacement().margin), y1-y0-2*(comp->getPlacement().margin));
	}
	redraw (0, 0, rectangle.width, rectangle.height);
}

void
UFrame::move (int x, int y)
{
	if (rectangle.x == x && rectangle.y ==y) return;
	rectangle.x = x;
	rectangle.y = y;
	XMoveWindow (UTOP->display, window, rectangle.x, rectangle.y);
}

void
UFrame::redraw (int x, int y, int width, int height)
{
	int	ver, hor;
	GC	lightLeft=None;
	GC	lightRight=None;
	
	if (!isShown(this)) return;
	if (frameSize == 0) return;
	switch (frameStyle)
	{
	case OUT:
		lightLeft = lightGC;
		lightRight = darkGC;
		break;
	case IN:
		lightLeft = darkGC;
		lightRight = lightGC;
		break;
	case ETCHED_IN:
	case ETCHED_OUT:
		return;
	}
	// Draw simple -
	if (frameStyle==OUT || frameStyle == IN)
	{
		// top lines
		if (y< frameSize) for (hor=0, ver=0; ver < frameSize 
			&& ver < height + y; hor++, ver++)
		{
			XDrawLine (UTOP->display, 
				window, lightLeft, 
				0, ver, rectangle.width - hor -1, ver);
		}
		// bottom lines
		if (rectangle.height > frameSize
		   && y + height > rectangle.height-frameSize)
		   for (hor=frameSize, ver=rectangle.height-frameSize; 
			ver < rectangle.height ;hor--, ver++)
		{
			if (rectangle.height > frameSize)
			XDrawLine (UTOP->display, 
				window, lightRight, 
				hor, ver, rectangle.width, ver);
		}

		// left
		if (x < frameSize) for (hor=0, ver=1; hor < frameSize
			&& hor < height + y; hor++, ver++)
		{
			XDrawLine (UTOP->display, 
				window, lightLeft, 
				hor, frameSize, hor, rectangle.height - ver);
		}
		if (rectangle.width > frameSize
		   && x + width > rectangle.width-frameSize)
		   for (ver=frameSize, 
			hor=rectangle.width-frameSize; 
			hor < rectangle.width ;ver--, hor++)
		{
			if (rectangle.height > frameSize)
			XDrawLine (UTOP->display, 
				window, lightRight, 
				hor, ver, hor, rectangle.height-frameSize);
		}
	}
}

void
UFrame::eventUp (UEvent* event)
{
	switch (event->type)
	{
	case UEvent::CARDINAL:
		break;
	default:
		break;
	}
	if (parent) parent->eventUp (event);
}

void 
UFrame::setBackground (const UColor& color)
{
	unsigned int	i;
	UComponent*	comp;
	background = color;
	XSetWindowBackground (background.getDisplay(), window,
		background.getPixel());
	light = background;
	light.lighter ();
	XSetForeground (UTOP->display, lightGC, light.getPixel());

	dark = background;
	dark.darker ();
	XSetForeground (UTOP->display, darkGC, dark.getPixel());

	XClearArea (UTOP->display, window, 0, 0, rectangle.width,
		rectangle.height, True);

	for (i=0; i< children.getSize(); i++)
	{
		comp=(UComponent*) children.at (i);
		if (comp==0) continue;
		comp->setBackground(color);
	}
}

void 
UFrame::setBackground (const char* color)
{
	background = color;
	setBackground (background);
}

void 
UFrame::setForeground (const UColor& color)
{
	unsigned int	i;
	UComponent*	comp;

	foreground = color;
	for (i=0; i< children.getSize(); i++)
	{
		comp=(UComponent*) children.at (i);
		if (comp==0) continue;
		comp->setForeground(color);
	}
}

void 
UFrame::setForeground (const char* color)
{
	foreground = color;
	setForeground (foreground);
}

UComponent*
UFrame::getParent ()
{
	return parent;
}

int
UFrame::isChild (Window w)
{
	UComponent*	comp;
	UFrame* 	frm;
	unsigned int	i;

	if (window == w) return 1;
	for (i=0; i<children.getSize(); i++)
	{
		comp = (UComponent*) children.at (i);
		if (comp==0) continue;
		if (comp->isA(UComponent::FRAME))
		{
			frm = (UFrame*) comp;
			if (frm->isChild (w)) return 1;
		}
		else
		{
			if (comp->window ==w) return 1;
		}
	}
	return 0;
}
