/* GGlyph - a graphical utility for managing one's collection of Type
   1 fonts (and eventually, other sorts of fonts) under X11.

   Copyright (c) 1998 David Huggins-Daines
   <bn711@freenet.carleton.ca>.

   You are permitted to copy, distribute, and modify this file under the
   terms of the GNU General Public License, version 2, or any later
   version.  See the file COPYING for more details. */

/* mover_view.c - the 'Font/DA Mover'-style panel */

#include <gtk/gtk.h>
#include <unistd.h>
#include <sys/stat.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "gglyph.h"
#include "mover_view.h"
#include "viewer_window.h"
#include "external.h"

/* */
/* local enums */
/* */

typedef enum {
  INST_COPY,
  INST_MOVE,
  INST_LINK
} InstallType;

/* */
/* forward declarations (can't get enough of these) */
/* */

static FontDirRec *find_dir (GSList *dir_list, gchar *dirname);
static GtkWidget *get_tree_item_fdir (GtkTree *tree, FontDirRec
				      *fdir);
static GtkWidget *get_tree_item_ffile (GtkTree *tree, FontFileRec
				       *ffile);
static GSList *get_selected_fonts (void);
static GSList *get_selected_dirs (void);

static void cb_ffile_item_click (GtkTreeItem *item, GdkEventButton
				 *evt, GtkTree *tree);
static void cb_install_fonts (ChildRec *child, FontDirRec *fdir,
			      GSList *fonts);
static void cb_dir_drop (GtkWidget *item, GdkEvent *event);
static void cb_rtree_drop (GtkWidget *w, GdkEvent *event);
static void cb_ltree_drop (GtkWidget *w, GdkEvent *event);
static void cb_ffile_drag_request (GtkWidget *item, GdkEvent *event);
static void cb_dir_drag_request (GtkWidget *item, GdkEvent *event);
static void adjust_buttons (void);
static gint check_fonts (GList *items);
static void cb_select_child (GtkWidget *tree, GtkWidget *child);

static void install_fonts (FontDirRec *fdir, GSList *fonts,
			   InstallType itype);
static void install_dir (FontDirRec *fdir);
static void uninstall_dir (FontDirRec *fdir);

static void update_tree (GtkTree *tree);
static void add_font_to_tree (gchar *key, FontFileRec *value,
			      GtkTree *dir_tree);
static void update_font_in_tree (gchar *key, FontFileRec *value,
				 GtkWidget *item);
static void fill_inst_trees (void);
static void add_fontdir_to_tree (GtkWidget *tree, FontDirRec *fdir);
static void fill_avail_tree (void);

static void add_right_box (GtkWidget *hbox);
static void add_left_box (GtkWidget *hbox);

/* */
/* static data */
/* FIXME: too much static data */
/* */
static GtkWidget *sys_tree, *local_tree, *avail_tree;
static GtkWidget *install_button, *copy_button, *move_button,
  *preview_button;

/* */
/* global data */
/* FIXME: too much global data */
/* */
extern gchar *viewer_title;
extern gint debug_level, rescan_interval;
extern uid_t uid;
extern gchar *t1dir;
extern GSList *inst_fontdirs, *avail_fontdirs;
extern GSList *dirty_list;
extern GHashTable *avail_fontfiles;

#ifdef GTK_ITEM_FACTORY
extern GtkItemFactory *item_factory;
#else /* GTK_ITEM_FACTORY nyet */
extern GtkMenuFactory *menu_factory;
#endif /* GTK_ITEM_FACTORY */

extern gchar *wdir_drop_types[], *tree_drop_types[];
extern gchar *dir_drag_types[], *file_drag_types[];

/* */
/* utility functions */
/* */

static FontDirRec *find_dir (GSList *dir_list, gchar *dirname)
{
  GSList *i;
  FontDirRec *dir = NULL;
  
  /* find the directory */
  for (i = dir_list; i; i = i->next){
    dir = (FontDirRec *) i->data;

    if (dir->path)
      if (strcmp (dir->path, dirname) == 0)
	break;
  }
  
  return dir;
}

/* The selection is set up such that is is homogeneous within one
   tree. */
static GSList *get_selected_dirs (void)
{
  GList *selection;
  GSList *dirs = NULL;
  FontDirRec *fdir;

  selection = GTK_TREE_SELECTION(avail_tree);

  /* if there is no selection, or it's not directories */
  if (!(selection) || !(GTK_TREE_ITEM_SUBTREE (selection->data))){
    /* guaranteed to return the selection of the right box, whether
       it's in sys_tree or not */
    selection = GTK_TREE_SELECTION(sys_tree);
    if (!(selection) || !(GTK_TREE_ITEM_SUBTREE (selection->data)))
      return NULL;
  }

  while (selection){
    fdir = gtk_object_get_user_data (GTK_OBJECT (selection->data));
    
    if (fdir)
      dirs = g_slist_append (dirs, fdir);
    selection = selection->next;
  }

  return dirs;
}

static GSList *get_selected_fonts (void)
{
  GList *selection;
  GSList *fonts = NULL;
  FontFileRec *ffile;

  selection = GTK_TREE_SELECTION(avail_tree);

  /* if there is no selection, or it's not directories */
  /* I was a happy boy the day I learned about "short-circuit" logical 
     operators... */
  if (!(selection) || GTK_TREE_ITEM_SUBTREE (selection->data)){
    /* guaranteed to return the selection of the right box, whether
       it's in sys_tree or not */
    selection = GTK_TREE_SELECTION(sys_tree);
    if (!(selection) || GTK_TREE_ITEM_SUBTREE (selection->data))
      return NULL;
  }

  while (selection){
    ffile = gtk_object_get_user_data (GTK_OBJECT (selection->data));

    if (ffile)
      fonts = g_slist_append (fonts, ffile);
    
    selection = selection->next;
  }
  
  return fonts;
}

static GtkWidget *get_tree_item_fdir (GtkTree *tree, FontDirRec *fdir)
{
  GList *i;

  for (i = tree->children; i; i = i->next){
    GtkWidget *w = GTK_WIDGET (i->data);
    FontDirRec *other_fdir = gtk_object_get_user_data (GTK_OBJECT(w));

    if (fdir == other_fdir)
      return w;

    if (other_fdir)
      if (debug_level)
	g_print ("get_tree_item_fdir: %s is not %s\n",
		 other_fdir->path, fdir->path);
  }
  return NULL;
}

static GtkWidget *get_tree_item_ffile (GtkTree *tree, FontFileRec *ffile)
{
  GList *i;

  for (i = tree->children; i; i = i->next){
    GtkWidget *w = GTK_WIDGET (i->data);

    if (ffile == gtk_object_get_user_data (GTK_OBJECT(w)))
      return w;
  }
  return NULL;
}

/* */
/* callbacks (endless, endless callbacks...) */
/* */

static void cb_ffile_item_click (GtkTreeItem *item, GdkEventButton
				 *evt, GtkTree *tree)
{
  switch (evt->button){
  case 1:
    if (evt->type == GDK_2BUTTON_PRESS){
      FontFileRec *ffile = gtk_object_get_user_data (GTK_OBJECT(item));
      
      if (ffile == NULL)
	break;
      
      /* bring up the window asynchronously, otherwise it "feels" a little 
	 weird */
      gtk_idle_add ((GtkFunction) cb_viewer_window, ffile);
    }
    break;
  case 2:
    break;
  case 3:
    /* bring up a menu here, eventually */
    break;
  }
}

static void cb_install_fonts (ChildRec *child, FontDirRec *fdir,
			      GSList *fonts)
{
  if (debug_level)
    g_print
      ("cb_install_font: type1inst exited with status %d\n",
       child->exit_status);
  
  if (child->exit_status){
    /* FIXME: put up some kind of "this didn't work" messagebox */
  }
  else {
    while (fonts){
      copy_ffile_rec (fdir, (FontFileRec *) fonts->data);
      fonts = g_slist_remove_link (fonts, fonts);
    }

    rescan_fontdir (fdir);
    dirty_list = g_slist_append (dirty_list, fdir);
    if (rescan_interval == ALWAYS_RESCAN){
      update_trees();
      flush_XFontPath();
    }
  }
}

static void cb_dir_drop (GtkWidget *item, GdkEvent *event)
{
  FontDirRec *fdir = gtk_object_get_user_data (GTK_OBJECT(item));
  FontFileRec *ffile;
  GSList *fonts = NULL;
  gchar *pathname = (gchar *)event->dropdataavailable.data;

  ffile = find_font (pathname);
  
  if (ffile == NULL){
    g_print ("cb_dir_drop: Invalid drop data %s of type %s\n", pathname,
	     (gchar *)event->dropdataavailable.data_type);
    return;
  }

  if (debug_level){
    g_print ("cb_dir_drop: Dropped font is %s, record %p, id %d\n",
	     pathname, ffile, ffile->t1_fontid);
  }
  
  gtk_widget_set_state (item, GTK_WIDGET_SAVED_STATE(item));

  fonts = g_slist_append (fonts, ffile);
  /* FIXME: should use a pop-up menu? */
  install_fonts (fdir, fonts, INST_COPY);
}

static void cb_rtree_drop (GtkWidget *w, GdkEvent *event)
{
  gchar *dirname = (gchar *)event->dropdataavailable.data;
  FontDirRec *dir = find_dir (avail_fontdirs, dirname);
  
  if (dir == NULL){
    fprintf (stderr, "cb_rtree_drop: Invalid drop data %s of type %s\n",
	     dirname, (gchar *)event->dropdataavailable.data_type);
    return;
  }

  if (GTK_IS_TREE_ITEM(w)){
    gtk_widget_set_state (w, GTK_WIDGET_SAVED_STATE(w));
  }

  install_dir (dir);
}

static void cb_ltree_drop (GtkWidget *w, GdkEvent *event)
{
  gchar *dirname = (gchar *)event->dropdataavailable.data;
  FontDirRec *dir = find_dir (inst_fontdirs, dirname);
  
  if (dir == NULL){
    return;
  }

  if (GTK_IS_TREE_ITEM(w)){
    gtk_widget_set_state (w, GTK_WIDGET_SAVED_STATE(w));
  }

  uninstall_dir (dir);
}

static void cb_ffile_drag_request (GtkWidget *item, GdkEvent *event)
{
  FontFileRec *ffile;
  
  ffile = gtk_object_get_user_data (GTK_OBJECT(item));
  if (ffile == NULL)
    return;
  
  gtk_widget_dnd_data_set (item, event, ffile->pathname,
			   strlen(ffile->pathname) + 1);
}

static void cb_dir_drag_request (GtkWidget *item, GdkEvent *event)
{
  FontDirRec *fdir;
  
  fdir = gtk_object_get_user_data (GTK_OBJECT(item));
  if (fdir == NULL)
    return;
  
  gtk_widget_dnd_data_set (item, event, fdir->path,
			   strlen(fdir->path) + 1);
}

static gint check_fonts (GList *items)
{
  /* innocent until proven guilty */
  gint writable = TRUE;

  while (items && writable){
    FontFileRec *ffile = gtk_object_get_user_data
      (GTK_OBJECT(items->data));
    gchar dirname[strlen (ffile->pathname) + 1];
    gchar *slash;
    
    if (ffile == NULL)
      writable = FALSE;
    else {
      strcpy (dirname, ffile->pathname);
      slash = strrchr (dirname, '/');
      *slash = '\0';
      
      writable = !(access (dirname, R_OK | W_OK | X_OK));
    }
    items = items->next;
  }
  
  return writable;
}

/* All right, let's fix this for real...

   "Install" button is active iff:
   a) directories are selected
   b) no fonts are selected.

   "Copy" button is active iff:
   a) one directory is selected in the installed tree
   b) that directory is writable
   c) fonts are selected in the available tree

   "Move" button is active iff:
   a) "Copy" button is active
   b) fonts that are selected in the available tree are in writable
   directories.

   "Preview" button is active iff:
   a) fonts are selected.

   cb_select_child takes care of the labels on the buttons...
*/
/* hmm, this function now also takes care of menu entries, but I
   didn't bother changing the name. */
   
static void adjust_buttons (void)
{
  gint copyp, movep, installp, previewp;
#ifdef GTK_ITEM_FACTORY
  GtkWidget *item;
#else /* no hay GTK_ITEM_FACTORY */
  GtkMenuPath *path;
#endif /* GTK_ITEM_FACTORY */
  /* default: nothing is selected */
  copyp = movep = installp = previewp = FALSE;

  if (GTK_TREE_SELECTION (sys_tree) && GTK_TREE_SELECTION (avail_tree)){
    FontDirRec *fdir;
    /* selections in both trees -> fonts in avail_tree, one dir in
       sys_tree, but let's make sure... */
    g_assert (!GTK_TREE_ITEM_SUBTREE
	      (GTK_TREE_SELECTION (avail_tree)->data)
	      && !GTK_TREE_SELECTION (sys_tree)->next);

    previewp = TRUE;
    
    /* guilty until proven innocent */
    movep = copyp = FALSE;
      
    fdir = (FontDirRec *)
      gtk_object_get_user_data (GTK_OBJECT
				(GTK_TREE_SELECTION
				 (sys_tree)->data));
    if (!access (fdir->path, R_OK | W_OK | X_OK)) {
      /* it's writable - are there fonts?*/
      copyp = GTK_TREE_SELECTION (avail_tree) &&
	!GTK_TREE_ITEM_SUBTREE (GTK_TREE_SELECTION
				(avail_tree)->data);
      movep = copyp && check_fonts (GTK_TREE_SELECTION (avail_tree));
    }
  }
  else if (GTK_TREE_SELECTION (sys_tree)){
    if (GTK_TREE_ITEM_SUBTREE (GTK_TREE_SELECTION (sys_tree)->data))
      installp = TRUE;
    else
      previewp = TRUE;
  }
  else if (GTK_TREE_SELECTION (avail_tree)){
    /* er, redundant, but clear... */
    if (GTK_TREE_ITEM_SUBTREE (GTK_TREE_SELECTION (avail_tree)->data))
      installp = TRUE;
    else
      previewp = TRUE;
  }

  /* default, of course, being nothing is selected */
  
  gtk_widget_set_sensitive (install_button, installp);
  /* preview button and "Font" menu have the same sensitivity */
  gtk_widget_set_sensitive (preview_button, previewp);

#ifdef GTK_ITEM_FACTORY
  /* disabled until GtkItemFactory is fixed */
/*   item = gtk_item_factory_get_widget (item_factory, "<Menubar>/Font"); */
/*   gtk_widget_set_sensitive (item, previewp);   */
#else /* GTK_ITEM_FACTORYkh nyet */
  path = gtk_menu_factory_find (menu_factory, "Font");
  gtk_widget_set_sensitive (path->widget, previewp);
#endif /* GTK_ITEM_FACTORY */
  
  gtk_widget_set_sensitive (move_button, movep);
  gtk_widget_set_sensitive (copy_button, copyp);
}

/* Selection semantics:

   - available directories may be selected only if there are no
   installed fonts or directories selected
   
   - installed directories may be selected only if there are no
   available directories selected.
   
   - available fonts may be selected only if a single, writable
   installed directory is selected.  If more than one installed
   directory is selected, all are deselected.

   - installed fonts may be selected only if there are no available
   fonts or directories selected.

   - adjust_buttons() now takes care of the buttons (as it should have 
   from the start...)

   (You may ask, why don't I connect adjust_buttons() to
   "selection_changed"?  The answer is, that every time I call
   gtk_tree_unselect_child() in this function, that signal will be
   triggered, and I don't particularly want that.)
*/

static void cb_select_child (GtkWidget *tree, GtkWidget *child)
{
  GtkWidget *other_tree;
  GList *item, *next;

  other_tree = (tree == avail_tree) ?
    GTK_WIDGET(GTK_TREE(sys_tree)->root_tree) : avail_tree;

  if (debug_level)
    g_print ("tree=%p, avail_tree=%p, tree->selection=%p\n", tree,
	     avail_tree, GTK_TREE_SELECTION(tree));

  /* I think this is still needed... it certainly makes things more
     efficient */
  if (GTK_TREE_SELECTION (tree) == NULL){
    adjust_buttons();
    return;
  }

  /* test for directory: all and only directory items have subtrees */
  if (GTK_TREE_ITEM_SUBTREE(child)){
    if (tree == avail_tree) {
      /* selecting a directory in the available directory */
      /* set the label, even if we don't activate it */
      gtk_label_set (GTK_LABEL (GTK_BUTTON(install_button)->child),
		     ">> Install >>");
      
      /* now unselect all other selections if they're not directories
       */
      for (item = GTK_TREE_SELECTION (tree); item; item = next){
	next = item->next;
	if (!GTK_TREE_ITEM_SUBTREE(item->data)){
	  gtk_tree_unselect_child (GTK_TREE(tree),
				   GTK_WIDGET(item->data));
	}
      }
      
      /* and all selections in the other tree as well. */
      while (GTK_TREE_SELECTION(other_tree)){
	gtk_tree_unselect_child (GTK_TREE(other_tree),
				 GTK_WIDGET
				 (GTK_TREE_SELECTION(other_tree)->data));
      }
    }
    else {
      /* selecting a directory in the installed tree */
      gint have_fonts = FALSE;

      /* unselect all directories in the available tree, and check to
	 see if there are any selected fonts */
      for (item = GTK_TREE_SELECTION(avail_tree); item; item = next){
	next = item->next;
	if (GTK_TREE_ITEM_SUBTREE (item->data)) {
	  gtk_tree_unselect_child (GTK_TREE(avail_tree),
				   GTK_WIDGET(item->data));
	}
	else {
	  /* it's a font */
	  have_fonts = TRUE;
	}
      }

      for (item = GTK_TREE_SELECTION (tree); item; item = next){
	next = item->next;
	/* if there are fonts selected in the available tree, unselect
	   all other directories - unselect all fonts regardless. */
	if (!GTK_TREE_ITEM_SUBTREE(item->data) ||
	    (have_fonts && GTK_TREE_ITEM_SUBTREE(item->data)
	     && item->data != child)){
	  gtk_tree_unselect_child (GTK_TREE(tree),
				   GTK_WIDGET(item->data));
	}
      }
      
      if (!have_fonts) {
	/* no fonts - set the "Install" button appropriately */
	gtk_label_set (GTK_LABEL (GTK_BUTTON(install_button)->child),
		       "<< UnInstall <<");
	gtk_widget_set_sensitive (install_button, TRUE);
      }
    } 
  }
  else {
    /* A font has been selected */
    /* These are always disabled for installed fonts, but this might
       not always be the case */
    gtk_label_set (GTK_LABEL (GTK_BUTTON(copy_button)->child),
		   (tree == avail_tree) ? ">> Copy >>" : "<< Copy <<");
    gtk_label_set (GTK_LABEL (GTK_BUTTON(move_button)->child),
		   (tree == avail_tree) ? ">> Move >>" : "<< Move <<");
    
    /* Unselect all other selections in this tree if they're not fonts */
    for (item = GTK_TREE_SELECTION(tree); item; item = next){
      next = item->next;
      if (GTK_TREE_ITEM_SUBTREE (item->data)){
	gtk_tree_unselect_child (GTK_TREE(tree),
				 GTK_WIDGET(item->data));
      }
    }

    /* Available tree */
    if (tree == avail_tree) {
      gint dircount = 0;
      
      /* Unselect all fonts in the opposing tree's selection */
      for (item = GTK_TREE_SELECTION(other_tree); item; item = next){
	next = item->next;
	if (!GTK_TREE_ITEM_SUBTREE (item->data)){
	  gtk_tree_unselect_child (GTK_TREE(other_tree),
				   GTK_WIDGET(item->data));
	}
	else {
	  /* and additional directories */
	  if (dircount)
	    gtk_tree_unselect_child (GTK_TREE(other_tree),
				     GTK_WIDGET(item->data));
	  dircount++;
	}
      }
      
      /* and the remaining directory if there was more than one - does 
	 this make sense to the user, though? */
      if (dircount > 1){
	gtk_tree_unselect_child (GTK_TREE(other_tree),
				 GTK_WIDGET (GTK_TREE_SELECTION
					     (other_tree)->data));
      }
    }
    else {
      /* installed tree, or no directories selected */
      /* Unselect all selections in the other tree no matter what. */
      while (GTK_TREE_SELECTION(other_tree)){
	gtk_tree_unselect_child (GTK_TREE(other_tree),
				 GTK_WIDGET
				 (GTK_TREE_SELECTION(other_tree)->data));
      }
    }
  }

  /* now pray that this works correctly */
  adjust_buttons();
}

static void cb_copy_move_button (GtkWidget *w, gpointer data)
{
  GSList *fonts, *dirs;

  fonts = get_selected_fonts();
  if (fonts == NULL){
    /* selection is not a font.  Actually, we shouldn't be here. */
    return;
  }

  dirs = get_selected_dirs();
  if (dirs == NULL || dirs->next){
    /* either there is no selected directory, or there's more than
       one. either way, it's pointless */
    return;
  }
  
  install_fonts ((FontDirRec *)dirs->data, fonts, (InstallType) data);
  g_slist_free_1 (dirs);
}

static void cb_install_button (GtkWidget *w, gpointer data)
{
  GSList *fdirs;
  void (*func) (FontDirRec *);

  fdirs = get_selected_dirs();
  if (fdirs == NULL){
    /* selection is not a directory.  Actually, we shouldn't be here. */
    return;
  }
  
  if (GTK_TREE_SELECTION (avail_tree))
    func = install_dir;
  else
    func = uninstall_dir;

  while (fdirs){
    func ((FontDirRec *) fdirs->data);
    fdirs = g_slist_remove_link (fdirs, fdirs);
  }
}

static void cb_preview_button (GtkWidget *w)
{
  preview_selection (FALSE);
}

void preview_selection (gint asynchronous)
{
  GSList *fonts;

  fonts = get_selected_fonts();
  if (fonts == NULL)
    return;

  while (fonts){
    FontFileRec *ffile;

    ffile = (FontFileRec *)fonts->data;
    
    if (asynchronous)
      gtk_idle_add ((GtkFunction) cb_viewer_window, ffile);
    else {
      /* synchronous, so it's okay to do this on the stack */
      GtkType1FontSpec spec;
      
      spec.type = GTK_TYPE1_FONTID;
      spec.font.t1_fontid = ffile->t1_fontid;
      
      new_viewer_window (ffile);
    }
    
    fonts = g_slist_remove_link (fonts, fonts);
  }
}

void sample_selection (gint action)
{
  GSList *fonts;
  
  fonts = get_selected_fonts();
  if (fonts == NULL)
    return;

  while (fonts){
    FontFileRec *ffile;

    ffile = (FontFileRec *)fonts->data;
    
    do_sample_sheet (ffile, action);
    
    fonts = g_slist_remove_link (fonts, fonts);
  }
}

static void cb_addfile_ok (GtkWidget *button, GtkFileSelection
			   *filesel)
{
  gchar *name;
  struct stat buf;

  name = gtk_file_selection_get_filename (filesel);

  if (stat (name, &buf) == -1){
    return;
  }

  if (S_ISDIR (buf.st_mode)){
    FontDirRec *fdir;
    gint result;
    
    if ((result = T1_AddToFileSearchPath
	 (T1_PFAB_PATH | T1_AFM_PATH, T1_APPEND_PATH,
	  name)) < 0){
      fprintf(stderr, "Could not add pathname %s (%d)\n",
	      name, result);
      return;
    }

    fdir = add_fontdir (&avail_fontdirs, name);
    /* (lambda ()), sub {}, sigh.... */
    g_hash_table_foreach (fdir->fonts, (GHFunc) add_font_to_db,
			  GINT_TO_POINTER(TRUE));
    add_fontdir_to_tree (avail_tree, fdir);
  }
  else if (S_ISLNK (buf.st_mode) | S_ISREG (buf.st_mode) && is_psfont
	   (name)){
    /* FIXME: this is copied straight from main() */
    FontFileRec *ffile = g_new (FontFileRec, 1);
    gchar *pathname, *basename;

    if (avail_fontfiles == NULL)
      avail_fontfiles = g_hash_table_new (g_str_hash,
					  g_str_equal);

    /* except that I assume that gtk_file_selection_get_filename()
       returns a full path */
    pathname = g_strdup (name);
    basename = strrchr (pathname, '/') + sizeof(gchar);
	
    ffile->pathname = strdup (pathname);
    ffile->basename = ffile->pathname + (basename - pathname);
    ffile->xlfd = NULL;
    ffile->t1_fontid = NOT_LOADED;
    g_hash_table_insert (avail_fontfiles, ffile->basename,
			 ffile);
    if (debug_level)
      g_print ("Added %s to avaliable fonts list\n", pathname);

    /* now make t1lib aware of this fact (this will be fixed once
       I make gdk_t1lib do something...) */
    /* note that add_font_to_db totally ignores its first argument */
    add_font_to_db (NULL, ffile, FALSE);

    /* as does add_font_to_tree */
    add_font_to_tree (NULL, ffile, GTK_TREE(avail_tree));
    
    g_free (pathname);
  }
  else {
    fprintf (stderr, "%s isn't a dir or a Type 1 font file\n", name);
    return;
  }
  gtk_widget_destroy (GTK_WIDGET (filesel));
}

static void cb_add_button (GtkWidget *button)
{
  GtkWidget *filesel;

  chdir (getenv ("HOME"));
  
  filesel = gtk_file_selection_new ("Add available font file/directory");
  gtk_widget_show (filesel);
    
  gtk_signal_connect (GTK_OBJECT (filesel), "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL);
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		      "clicked", GTK_SIGNAL_FUNC (cb_addfile_ok), filesel );
  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION
			      (filesel)->cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (filesel));
}

/* so that selections don't persist */
static void cb_collapse (GtkWidget *tree_item, GtkWidget *tree)
{
  GList *selection, *next;

  for (selection = GTK_TREE_SELECTION (tree);
       selection; selection = next){
    next = selection->next;
    if (GTK_WIDGET (selection->data)->parent == tree)
      gtk_tree_unselect_child (GTK_TREE (tree),
			       GTK_WIDGET (selection->data));
  }
  adjust_buttons();
}

static void cb_drop_enter (GtkWidget *w, GdkEvent *event)
{
  gtk_widget_set_state (w, GTK_STATE_PRELIGHT);
}

static void cb_drop_leave (GtkWidget *w, GdkEvent *event)
{
  gtk_widget_set_state (w, GTK_WIDGET_SAVED_STATE(w));
}

/* */
/* functions that actually do stuff */
/* */

static void install_fonts (FontDirRec *fdir, GSList *fonts,
			  InstallType itype)
{
  ChildRec *type1instchild;
  GSList *node;
  
  for (node = fonts; node; node = node->next){
    gchar command[strlen (fdir->path) +
		 strlen (((FontFileRec *) node->data)->pathname) + 8];
    
    /* FIXME: should accept a list of fonts */
  
    switch (itype){
    case INST_COPY:
      strcpy (command, "cp -f ");
      break;
    case INST_MOVE:
      strcpy (command, "mv -f ");
      break;
    case INST_LINK:
    default:
    }
  
    strcat (command, ((FontFileRec *)node->data)->pathname);
    strcat (command, " ");
    strcat (command, fdir->path);
  
    if (debug_level)
      g_print ("install_font: Command is %s\n", command);
    
    if (system (command) == -1){
      perror ("Can't install font file");
      return;
    }
  }

  type1instchild = type1inst (fdir->path);
  on_child_exit (type1instchild, (ExitJobProc) cb_install_fonts,
		 fdir, fonts);
}

static void install_dir (FontDirRec *fdir)
{
  GtkWidget *subtree, *item;

  item = get_tree_item_fdir (GTK_TREE(avail_tree), fdir);

  if (item == NULL)
    return;
  
  avail_fontdirs = g_slist_remove (avail_fontdirs, fdir);
  inst_fontdirs = g_slist_append (inst_fontdirs, fdir);

  gtk_container_remove (GTK_CONTAINER(avail_tree), item);
  /* note: though theoretically it is possible to wrap
     gtk_container_remove with reference count increases and
     decreases, it's actually less of a pain in the ass to let it
     destroy the item, the subtree, and everything therein and then
     build the tree anew. */
  /* FIXME: if someone can suggest a better way to do this, please
     do... */
  item = gtk_tree_item_new_with_label (fdir->path);
  gtk_object_set_user_data (GTK_OBJECT(item), fdir);
  subtree = gtk_tree_new ();
  gtk_signal_connect_object (GTK_OBJECT(subtree), "select_child",
			     GTK_SIGNAL_FUNC(cb_select_child),
			     GTK_OBJECT(GTK_TREE_ROOT_TREE(sys_tree)));
  gtk_signal_connect (GTK_OBJECT (item),
		      "drag_request_event",
		      GTK_SIGNAL_FUNC(cb_dir_drag_request),
		      NULL);
  gtk_widget_show (item);

  if (uid){
    gint writable = !(access (fdir->path, R_OK | W_OK | X_OK));
    GtkWidget *tree = writable ? local_tree : sys_tree;

    gtk_tree_append (GTK_TREE(tree), item);
  }
  else {
    gtk_tree_append (GTK_TREE(sys_tree), item);
  }
  
  gtk_tree_item_set_subtree (GTK_TREE_ITEM(item), subtree);
  g_hash_table_foreach (fdir->fonts, (GHFunc) add_font_to_tree,
			subtree);
  gtk_widget_dnd_drag_set (item, TRUE,
			   dir_drag_types, 3);

  if (debug_level)
    g_print ("install_dir: Dir is %s\n", fdir->path);
  
  add_to_XFontPath (fdir->path);
}

static void uninstall_dir (FontDirRec *fdir)
{
  GtkWidget *tree, *owner, *subtree, *item;

  tree = sys_tree;
  item = get_tree_item_fdir (GTK_TREE(tree), fdir);

  if (item == NULL){
    tree = local_tree;
    item = get_tree_item_fdir (GTK_TREE(tree), fdir);
  }
  
  if (item == NULL)
    return;
  
  avail_fontdirs = g_slist_append (avail_fontdirs, fdir);
  inst_fontdirs = g_slist_remove (inst_fontdirs, fdir);

  /* see above for severe gtk_container_remove annoyance */
  gtk_widget_ref (tree);
  owner = GTK_TREE(tree)->tree_owner;
  /* since this would otherwise kill the tree if it removed the last
     item */
  gtk_container_remove (GTK_CONTAINER(tree), item);
  if (tree->parent == NULL){
    gtk_tree_item_set_subtree (GTK_TREE_ITEM(owner), tree);
    gtk_tree_item_expand (GTK_TREE_ITEM(owner));
  }
  else
    gtk_widget_unref (tree);
  
  item = gtk_tree_item_new_with_label (fdir->path);
  gtk_object_set_user_data (GTK_OBJECT(item), fdir);
  subtree = gtk_tree_new ();
  gtk_signal_connect_object (GTK_OBJECT(subtree), "select_child",
			     GTK_SIGNAL_FUNC(cb_select_child),
			     GTK_OBJECT(avail_tree));
  gtk_signal_connect (GTK_OBJECT (item),
		      "drag_request_event",
		      GTK_SIGNAL_FUNC(cb_dir_drag_request),
		      NULL);
  gtk_widget_show (item);
  gtk_tree_append (GTK_TREE(avail_tree), item);
  gtk_tree_item_set_subtree (GTK_TREE_ITEM(item), subtree);
  g_hash_table_foreach (fdir->fonts, (GHFunc) add_font_to_tree,
			subtree);
  gtk_widget_dnd_drag_set (item, TRUE,
			   dir_drag_types, 3);

  if (debug_level)
    g_print ("uninstall_dir: Dir is %s\n", fdir->path);

  remove_from_XFontPath (fdir->path);
}

/* */
/* GUI creation and maintenance code */
/* */

static void add_font_to_tree (gchar *key, FontFileRec *value,
			      GtkTree *dir_tree)
{
  GtkWidget *file_item = gtk_tree_item_new_with_label
    (value->basename);

  if (debug_level)
    g_print ("add_font_to_tree: Adding font %s to tree\n",
	     value->basename);
  
  gtk_object_set_user_data (GTK_OBJECT(file_item), value);
  gtk_signal_connect (GTK_OBJECT(file_item),
		      "button_press_event",
		      GTK_SIGNAL_FUNC(cb_ffile_item_click),
		      dir_tree);
  gtk_signal_connect (GTK_OBJECT (file_item),
		      "drag_request_event",
		      GTK_SIGNAL_FUNC(cb_ffile_drag_request),
		      NULL);
  gtk_tree_append (dir_tree, file_item);
  gtk_widget_show (file_item);
  gtk_widget_dnd_drag_set (file_item, TRUE,
			   file_drag_types, 3);
}

static void update_font_in_tree (gchar *key, FontFileRec *value,
				 GtkWidget *item)
{
  FontDirRec *fdir = gtk_object_get_user_data (GTK_OBJECT(item));

  if (debug_level)
    g_print ("update_font_in_tree: Updating %s in tree of dir %s\n",
	     value->basename,fdir->path);
  
  if (fdir == NULL)
    return;
  
  if (get_tree_item_ffile (GTK_TREE(GTK_TREE_ITEM_SUBTREE(item)),
			   value) == NULL) {
    add_font_to_tree (key, value,
		      GTK_TREE(GTK_TREE_ITEM_SUBTREE(item)));
  }
}

static void update_tree (GtkTree *tree)
{
  GList *i;

  /* "fail" silently, to fix really stupid mistake that causes
     segfault :-) */
  if (tree == NULL)
    return;
  
  g_return_if_fail (GTK_IS_TREE(tree));
  for (i = tree->children; i; i = i->next){
    GtkWidget *item = GTK_WIDGET(i->data);
    GtkWidget *subtree = GTK_TREE_ITEM_SUBTREE(item);
    
    if (subtree) {
      /* it's a directory (I hope) */
      FontDirRec *fdir = gtk_object_get_user_data (GTK_OBJECT (item));
      
      if (fdir == NULL)
	continue;

      if (fdir->dirty) {
	GList *j;

	/* add new entries */

	g_hash_table_foreach (fdir->fonts, (GHFunc)
			      update_font_in_tree, item);

	/* eliminate removed entries */
	
	for (j = GTK_TREE(subtree)->children; j; j = j->next){
	  GtkWidget *w = GTK_WIDGET (j->data);
	  FontFileRec *ffile = gtk_object_get_user_data
	    (GTK_OBJECT(w));

	  if (g_hash_table_lookup (fdir->fonts, ffile->basename) ==
	      NULL){
	    /* not there */
	    gtk_container_remove (GTK_CONTAINER(tree), item);
	  }
	}
	
	fdir->dirty = FALSE;
      }
    }
    else if (tree == GTK_TREE(avail_tree)) {
      /* it's a file - check to see if it's still supposed to be here
       */
      FontFileRec *ffile = gtk_object_get_user_data (GTK_OBJECT
						     (item));

      if (ffile == NULL)
	continue;

      if (g_hash_table_lookup (avail_fontfiles, ffile->basename) ==
	  NULL){
	/* not there */
	gtk_container_remove (GTK_CONTAINER(tree), item);
      }
    }
  }   
}

void update_trees (void)
{
  /* check avail_tree, sys_tree, local_tree */
  update_tree (GTK_TREE(avail_tree));
  update_tree (GTK_TREE(sys_tree));
  update_tree (GTK_TREE(local_tree));  
}

static void fill_inst_trees (void)
{ 
  GSList *i;

  /* static GtkWidget *sys_tree, *local_tree */
  
  for (i = inst_fontdirs; i; i = i->next){
    FontDirRec *fdir = (FontDirRec *) i->data;
    GtkWidget *dir_item = gtk_tree_item_new_with_label (fdir->path);
    GtkWidget *dir_tree = gtk_tree_new();
    gint writable = !(access (fdir->path, R_OK | W_OK | X_OK));

    gtk_signal_connect_object (GTK_OBJECT(dir_tree), "select_child",
			       GTK_SIGNAL_FUNC(cb_select_child),
			       GTK_OBJECT(GTK_TREE_ROOT_TREE(sys_tree)));
    if (debug_level)
      g_print ("fill_inst_trees: Adding font dir %s to tree\n",
	       fdir->path);
      
    gtk_object_set_user_data (GTK_OBJECT(dir_item), fdir);

    /* dropping a font on these will cause it to be installed */
    gtk_signal_connect (GTK_OBJECT (dir_item),
			"drop_data_available_event",
			GTK_SIGNAL_FUNC(cb_dir_drop),
			NULL);
    gtk_signal_connect (GTK_OBJECT (dir_item),
			"drop_enter_event",
			GTK_SIGNAL_FUNC(cb_drop_enter),
			NULL);
    gtk_signal_connect (GTK_OBJECT (dir_item),
			"drop_leave_event",
			GTK_SIGNAL_FUNC(cb_drop_leave),
			NULL);
    gtk_signal_connect (GTK_OBJECT (dir_item),
			"collapse", GTK_SIGNAL_FUNC(cb_collapse),
			dir_tree);
    /* dragging these to the "Available" side will cause them to be
       removed from the X font path */
    gtk_signal_connect (GTK_OBJECT (dir_item),
			"drag_request_event",
			GTK_SIGNAL_FUNC(cb_dir_drag_request),
			NULL);
    
      
    gtk_widget_show (dir_item);

    if (uid){
      if (writable){
	gtk_tree_append (GTK_TREE(local_tree), dir_item);
	}
      else {
	gtk_tree_append (GTK_TREE(sys_tree), dir_item);
      }
    }
    else {
      gtk_tree_append (GTK_TREE(sys_tree), dir_item);
    }

    gtk_tree_item_set_subtree (GTK_TREE_ITEM(dir_item), dir_tree);
    gtk_widget_dnd_drag_set (dir_item, TRUE,
			     dir_drag_types, 3);
    gtk_widget_dnd_drop_set (dir_item, writable,
			     wdir_drop_types, 1, FALSE);

    g_hash_table_foreach (fdir->fonts, (GHFunc) add_font_to_tree,
			  dir_tree);
  }
}

/* Right box - installed fonts */
static void add_right_box (GtkWidget *hbox)
{
  GtkWidget *scrolled_win, *tree, *vbox, *label;

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  gtk_widget_show (vbox);

  label = gtk_label_new ("Installed Fonts");
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 2);
  gtk_widget_show (label);

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (scrolled_win, 250, 350);
  gtk_box_pack_start (GTK_BOX(vbox), scrolled_win, TRUE, TRUE, 0);
  gtk_widget_show (scrolled_win);    

  tree = gtk_tree_new();
  gtk_tree_set_selection_mode (GTK_TREE(tree),
			       GTK_SELECTION_MULTIPLE);
  gtk_signal_connect (GTK_OBJECT(tree), "select_child",
		      GTK_SIGNAL_FUNC(gtk_tree_unselect_child), NULL);
  gtk_signal_connect (GTK_OBJECT (tree),
		      "drop_data_available_event",
		      GTK_SIGNAL_FUNC(cb_rtree_drop),
		      NULL);
  gtk_container_add (GTK_CONTAINER(scrolled_win), tree);
  gtk_widget_show (tree);
  
  /* This has to go after everything else, or all hell breaks loose */
  gtk_widget_dnd_drop_set (tree, TRUE,
			   tree_drop_types, 1, FALSE);
  
  sys_tree = gtk_tree_new();
  gtk_signal_connect_object (GTK_OBJECT(sys_tree), "select_child",
			     GTK_SIGNAL_FUNC(cb_select_child),
			     GTK_OBJECT(tree));

  {
    GtkWidget *tree_item;

    tree_item = gtk_tree_item_new_with_label ("System Font Directories");
    if (!uid){
      gtk_signal_connect (GTK_OBJECT (tree_item),
			  "drop_data_available_event",
			  GTK_SIGNAL_FUNC(cb_rtree_drop),
			  NULL);
      gtk_signal_connect (GTK_OBJECT (tree_item),
			  "drop_enter_event",
			  GTK_SIGNAL_FUNC(cb_drop_enter),
			  NULL);
      gtk_signal_connect (GTK_OBJECT (tree_item),
			  "drop_leave_event",
			  GTK_SIGNAL_FUNC(cb_drop_leave),
			  NULL);
    }
      
    gtk_widget_show (tree_item);
    gtk_tree_append (GTK_TREE(tree), tree_item);      
    gtk_widget_show (sys_tree);
    gtk_tree_item_set_subtree (GTK_TREE_ITEM(tree_item), sys_tree);
    gtk_tree_item_expand (GTK_TREE_ITEM(tree_item));
    gtk_widget_dnd_drop_set (tree_item, !uid,
			     tree_drop_types, 1, FALSE);

    if (uid){
      tree_item = gtk_tree_item_new_with_label ("User Font Directories");
      
      gtk_signal_connect (GTK_OBJECT (tree_item),
			  "drop_data_available_event",
			  GTK_SIGNAL_FUNC(cb_rtree_drop),
			  NULL);
      gtk_signal_connect (GTK_OBJECT (tree_item),
			  "drop_enter_event",
			  GTK_SIGNAL_FUNC(cb_drop_enter),
			  NULL);
      gtk_signal_connect (GTK_OBJECT (tree_item),
			  "drop_leave_event",
			  GTK_SIGNAL_FUNC(cb_drop_leave),
			  NULL);
      
      gtk_widget_show (tree_item);
      gtk_tree_append (GTK_TREE(tree), tree_item);
      local_tree = gtk_tree_new();
      gtk_signal_connect_object (GTK_OBJECT(local_tree), "select_child",
				 GTK_SIGNAL_FUNC(cb_select_child),
				 GTK_OBJECT(tree));

      gtk_widget_show (local_tree);
      gtk_tree_item_set_subtree (GTK_TREE_ITEM(tree_item), local_tree);
      gtk_tree_item_expand (GTK_TREE_ITEM(tree_item));
      gtk_widget_dnd_drop_set (tree_item, TRUE,
			       tree_drop_types, 1, FALSE);
    }
  }

  fill_inst_trees();
}

/* FIXME: this function should be fully general, but it's not */
static void add_fontdir_to_tree (GtkWidget *tree, FontDirRec *fdir)
{
  GtkWidget *dir_item = gtk_tree_item_new_with_label (fdir->path);
  GtkWidget *dir_tree = gtk_tree_new();

  gtk_signal_connect_object (GTK_OBJECT(dir_tree), "select_child",
			     GTK_SIGNAL_FUNC(cb_select_child),
			     GTK_OBJECT(tree));
  gtk_object_set_user_data (GTK_OBJECT(dir_item), fdir);      
  gtk_signal_connect (GTK_OBJECT (dir_item),
		      "drag_request_event",
		      GTK_SIGNAL_FUNC(cb_dir_drag_request),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (dir_item),
		      "collapse",
		      GTK_SIGNAL_FUNC(cb_collapse), dir_tree);

  gtk_tree_append (GTK_TREE(tree), dir_item);
  gtk_tree_item_set_subtree (GTK_TREE_ITEM(dir_item), dir_tree);
  gtk_widget_show (dir_item);
    
  gtk_widget_dnd_drag_set (dir_item, TRUE,
			   dir_drag_types, 3);

  g_hash_table_foreach (fdir->fonts, (GHFunc) add_font_to_tree,
			dir_tree);
}

static void fill_avail_tree (void)
{
  GSList *i;
  /* static GtkWidget *avail_tree */

  /* FIXME: should use g_slist_foreach() for consistency */
  for (i = avail_fontdirs; i; i = i->next){
    add_fontdir_to_tree (avail_tree, (FontDirRec *)i->data);
  }
    
  g_hash_table_foreach (avail_fontfiles, (GHFunc) add_font_to_tree,
			avail_tree);
}

/* Left box - avaliable fonts/directories */
static void add_left_box (GtkWidget *hbox)
{
  GtkWidget *scrolled_win, *vbox, *button, *label;
  
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  gtk_widget_show (vbox);
  
  label = gtk_label_new ("Avaliable Fonts");
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 2);
  gtk_widget_show (label);

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (scrolled_win, 250, 350);
  gtk_box_pack_start (GTK_BOX(vbox), scrolled_win, TRUE, TRUE,
		      0);
  gtk_widget_show (scrolled_win);

  button = gtk_button_new_with_label ("Add");
  gtk_signal_connect (GTK_OBJECT(button), "clicked",
		      GTK_SIGNAL_FUNC(cb_add_button), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, 2);
  gtk_widget_show (button);
  
  avail_tree = gtk_tree_new();
  gtk_tree_set_selection_mode (GTK_TREE(avail_tree),
			       GTK_SELECTION_MULTIPLE);
  gtk_signal_connect (GTK_OBJECT(avail_tree), "select_child",
		      GTK_SIGNAL_FUNC(cb_select_child), NULL);
  /* Dropping a directory here will cause it to be removed from the X
     font path; dropping a file here is undefined (but dropping a
     filename from gmc should add it to the tree) */
  gtk_signal_connect (GTK_OBJECT (avail_tree),
		      "drop_data_available_event",
		      GTK_SIGNAL_FUNC(cb_ltree_drop),
		      NULL);
  gtk_container_add (GTK_CONTAINER(scrolled_win), avail_tree);
  gtk_widget_show (avail_tree);

  /* This has to go after everything else, or all hell breaks loose */
  gtk_widget_dnd_drop_set (avail_tree, TRUE,
			   tree_drop_types, 1, FALSE);  

  fill_avail_tree();
}

/* Apple Font/DA Mover-style page (I always hated the Font/DA Mover,
   but there's no such thing as a magical "Fonts" folder in UNIX) */
void make_mover_page (GtkWidget *box)
{
  GtkWidget *hbox = gtk_hbox_new (FALSE, 0);

  gtk_box_pack_start (GTK_BOX(box), hbox, TRUE, TRUE, 5);
  gtk_widget_show (hbox);
  
  add_left_box (hbox);

  /* buttons */
  {
    GtkWidget *vbox;

    vbox = gtk_vbox_new (TRUE, 0);
    gtk_widget_set_usize (vbox, 100, 0);
    gtk_widget_show (vbox);
    gtk_box_pack_start (GTK_BOX(hbox), vbox, FALSE, FALSE, 5);

    /* these next three change their text dynamically */
    copy_button = gtk_button_new_with_label ("Copy");
    gtk_signal_connect (GTK_OBJECT(copy_button), "clicked",
			GTK_SIGNAL_FUNC(cb_copy_move_button),
			GINT_TO_POINTER (INST_COPY));
    gtk_widget_set_sensitive (copy_button, FALSE);
    gtk_widget_show (copy_button);
    gtk_box_pack_start (GTK_BOX(vbox), copy_button, FALSE, FALSE, 0);
    
    move_button = gtk_button_new_with_label ("Move");
    gtk_signal_connect (GTK_OBJECT(move_button), "clicked",
			GTK_SIGNAL_FUNC(cb_copy_move_button),
			GINT_TO_POINTER (INST_MOVE));
    gtk_widget_set_sensitive (move_button, FALSE);
    gtk_widget_show (move_button);
    gtk_box_pack_start (GTK_BOX(vbox), move_button, FALSE, FALSE, 0);
    
    install_button = gtk_button_new_with_label ("Install");
    gtk_signal_connect (GTK_OBJECT(install_button), "clicked",
			GTK_SIGNAL_FUNC(cb_install_button), NULL);
    gtk_widget_set_sensitive (install_button, FALSE);
    gtk_widget_show (install_button);
    gtk_box_pack_start (GTK_BOX(vbox), install_button, FALSE, FALSE,
			0); 
    
    preview_button = gtk_button_new_with_label ("Preview");
    gtk_signal_connect (GTK_OBJECT(preview_button), "clicked",
			GTK_SIGNAL_FUNC(cb_preview_button), NULL);
    gtk_widget_set_sensitive (preview_button, FALSE);
    gtk_widget_show (preview_button);
    gtk_box_pack_start (GTK_BOX(vbox), preview_button, FALSE, FALSE,
			0);
  }

  add_right_box (hbox);
}
