/* Types, extern data, and functions needed by more than one file
 *
 * gsumi version 0.5
 *
 * Copyright 1997 Owen Taylor <owt1@cornell.edu>
*/

#include <malloc.h>
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Xutil.h>

#include <gtk/gtk.h>
#include "xform.h"
#include "rect.h"
#include "blob.h"

typedef enum {
  PEN_TOOL,
  ERASER_TOOL
} ToolType;

typedef char int8;
typedef short int16;
typedef int int32;
typedef unsigned char uint8;
typedef unsigned short uint16;
typedef unsigned int uint32;

typedef struct _BitmapInfo BitmapInfo;

/* Hi-res bitmap images */
typedef struct image {
  int width, height;
  rect rec;
  int32 *lines[1];
} image;

#define WHITE_X(x) ((x)&0xffff)
#define BLACK_X(x) (((x)>>16)&0xffff)
#define WHITE_BLACK(x, y) ((x)|((y)<<16))
extern image *bitmap;

extern int image_width;
extern int image_height;

/* These may be 32, 16, or 8 bpp pixels, depending on bpp. */
extern gulong gray_pixels[256];
extern gulong red_pixel;
extern gulong maroon_pixel; 

typedef struct _AreaInfo AreaInfo;

/* Information about the state of one input device.
   We ignore op but keep it for compatibility with xink */ 
typedef struct input_state {
  double x, y;
  double pressure;
  double xtilt, ytilt;
  int buttons;
  int op;
} input_state;

typedef struct _ToolInfo ToolInfo;

struct _ToolInfo {
  guint32 id;
  char *name;

  input_state state;
  ToolType tool;
  double size;
  double aspect;
  double angle;
  double sens;
  
  double threshhold;
  gint use_threshhold;
  float profile[32];
  gint has_cursor;
};

typedef enum {
  FILE_BY_EXTENSION,
  FILE_PBM_GZ,
  FILE_PBM,
} FileType;

/* We encapsulate most of the global data into this one portmanteau
   object */
struct _AreaInfo {
  GtkWidget *area;
  GtkWidget *mainwin;
  GtkAdjustment *hscroll_data, *vscroll_data;
  int is_changed;
  ToolInfo *tool;
  char *name;
  FileType file_type;

  int is_updating;
};


/* user interface information exported */
extern AreaInfo info;

/* color context for application */
extern GdkColorContext *color_context;


#define MAX_TOOLS 10
extern ToolInfo *tools[MAX_TOOLS];
extern int num_tools;

/* bitmap functions */
void save_bitmap (const char *fn, int compress);
int load_bitmap (const char *fn, int compress);
void bitmap_init (int x, int y);
void undo_begin (void);
void undo_update (rect *rec);
void undo_end (void);
void undo_reset (void);
void bitmap_draw_disk (int x, int y, int r, int color);
void bitmap_draw_connector (int x0, int y0, int r0, int x1, int y1, int r1,
			    int color);
void bitmap_draw_blob (Blob *b, int color);
void bitmap_clear ();
void undo(void);
void redo(void);

/* rendering functions */
void expose_rect (rect *rec);
void add_to_render_region (rect *rec);
void draw_begin (input_state *ins, ToolInfo *tool_info);
void draw_update (input_state *last, input_state *ins, ToolInfo *tool_info);
void draw_end (input_state *ins, ToolInfo *tool_info);
void update_cursor (double x, double y, int prox);
void increment_scale(void);
void decrement_scale(void);
void scroll (int delta_x, int delta_y);
void screen_scroll (int delta_x, int delta_y);
void scroll_to (double new_x, double new_y);
int render_chunk (GtkWidget *area);
int draw (GtkWidget *area);
void init_scale(void);
void create_ximage (GtkWidget *area, int width, int height);
void bitmap_update (rect *rec);
void render_init (void);

/* file operation functions (fileops.c) */
void set_filename (void);
void do_new  (GtkWidget *w, gpointer data);
void do_open (GtkWidget *w, gpointer data);
void do_save (GtkWidget *w, gpointer data);
void do_save_as (GtkWidget *w, gpointer data);
void do_exit (GtkWidget *w, gpointer data);

/* user interface functions */
void update_scrollbars(xform *xf);
void start_idle_update(void);

void do_tool_box (GtkWidget *w, gpointer data);
void do_input_dialog (GtkWidget *w, gpointer data);

void area_enable_device (GtkWidget *w, guint32 devid);
void area_disable_device (GtkWidget *w, guint32 devid);

/* toolbox functions (toolbox.c) */
void tool_box_create(void);
void tool_box_show(void);
void tool_box_add_device(ToolInfo *tool_info);
void tool_box_remove_device(guint32 devid);
void tool_box_destroy(void);
void tool_box_set_current_device(guint32 devid);
void tool_box_toggle_current_tool(void);
void tool_box_set_values(guint32 devid);

/* options dialog functions (options.c) */
void options_dialog_create (void);
void options_dialog_set_current_device (guint32 devid);

/* for creating tool icons (icons.c) */
GdkPixmap* create_icon_pixmap (GdkWindow *window,
			       ToolType tool, gulong *gray_pixels);

/* Menu handling functions (menus.c) */
void menus_get_main_menubar   (GtkWidget           **menu,
			       GtkAcceleratorTable **table);
void menus_create             (GtkMenuEntry         *entries,
			       int                   nmenu_entries);
void menus_set_sensitive      (char                 *path,
			       int                   sensitive);
void menus_add_path           (char                 *path,
			       char                 *accelerator);
void menus_store              (void);

gchar *menus_stringify_accelerator  (guint                 keyval, 
				     GdkModifierType       modifiers);


/* from gsumirc.c (loading, saving rc file) */
void load_gsumirc(void);
void save_gsumirc(void);
void gsumirc_update_devices(void);
void gsumirc_update_menus();
void gsumirc_store_menu_path(char *path, char *accel);

