/* $Id: evqueue.inc,v 1.12 1998/09/20 18:19:32 steve Exp $
***************************************************************************

   Common event queue management.

   Copyright (C) 1997 Jason McMullan        [jmcc@ggi-project.org]
   Copyright (C) 1998 Jim Ursetto    [jim.ursetto@ggi-project.org]
   Copyright (C) 1998 Andrew Apted  [andrew.apted@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

/* 
 * This file contains the implementation of ev_queues and is included
 * by the event management files of the display targets:
 * display/X/events.c, display/svgalib/events.c, etc. 
 */

#include <ggi/events.h>

#define QUEUE_SIZE	8192
#define Q_THRESHOLD	(QUEUE_SIZE - sizeof(ggi_event) - 1)

/* The threshold the point where events wrap back to the start of the
 * buffer.  Events are stored in the buffer in whole; they do not get
 * split in half at the end of the buffer.  Instead, the event that
 * crosses the threshold will physically be the last, and the next event
 * has offset 0 in the buffer.
 *
 * Corollary: head is always <= Q_THRESHOLD.
 */

typedef struct ev_queue {
	int count;	/* number of events in the buffer */
	int head;	/* offset into buf[] where next event is put */
	int tail;	/* offset into buf[] of last event */
	uint8 buf[QUEUE_SIZE];
} ev_queue;

typedef struct ev_queue_set {
	ggi_event_mask seen;
	ev_queue * queues[evLast];
} ev_queue_set;

#ifdef WANT__NewEvent
static void __NewEvent(ggi_event *ev)
{
	memset(ev, 0, sizeof(ggi_event));
	EV_TIMESTAMP(ev);
}
#endif /* WANT__NewEvent */

/* Set up an inital, blank queue if none is there
 */
static ev_queue_set *_ggiEvQSetSetup(ggi_visual *vis)
{
	DPRINT_EVENTS("_ggiEvQSetSetup(%p) called\n", vis);
	
	if (LIBGGI_EVQUEUE(vis) == NULL) {
		LIBGGI_EVQUEUE(vis) = (ev_queue_set *) _ggi_malloc(sizeof(ev_queue_set));
		memset(LIBGGI_EVQUEUE(vis), 0, sizeof(ev_queue_set));
	}

	return (ev_queue_set *) LIBGGI_EVQUEUE(vis);
}

static ev_queue *_ggiEvQueueSetup(ev_queue_set *evq, int type)
{
	if (evq->queues[type] == NULL) {
		ev_queue *qp = (ev_queue *) _ggi_malloc(sizeof(ev_queue));
		memset(qp, 0, sizeof(ev_queue));
		evq->queues[type] = qp;
	}
	
	return evq->queues[type];
}

static ggi_event_mask _ggiEvQueueSeen(ggi_visual *vis, ggi_event_mask mask)
{
	ev_queue_set *evq;
	
	DPRINT_EVENTS("_ggiEvQueueSeen(%p, 0x%x) called\n", vis, mask);

	evq = _ggiEvQSetSetup(vis);

	return (evq->seen & mask);
}

/* Add an event. Return 0 on either success or unrecognized event type,
 * and return -1 is the buffer is full
 */
static int _ggiEvQueueAdd(ggi_visual *vis, ggi_event *ev)
{
	ev_queue_set *evq;
	ev_queue *qp;
	ggi_event_mask evm = 1 << ev->any.type;
	int freespace;

	DPRINT_EVENTS("_ggiEvQueueAdd(%p, %p) called\n", vis, ev);

	evq = _ggiEvQSetSetup(vis);

	/* Check if type is in range */

	if (ev->any.type >= evLast) {
		DPRINT_EVENTS("_ggiEvQueueAdd: bad type!\n");
		return 0;
	}
	
	/* Prevent the queueing of globally-masked-out events */
	
	if (! (evm & LIBGGI_EVMASK(vis))) {
		return 0;
	}
	
	DPRINT_EVENTS("_ggiEvQueueAdd: passed mask test.\n");

	qp = _ggiEvQueueSetup(evq, ev->any.type);

	/* Enough free space ? */

	freespace=1;

	if (qp->head < qp->tail) {
		if ((qp->tail - qp->head - 1) < ev->size) {
			freespace=0;
		}	
	} else if (qp->head > qp->tail) {
		if ((qp->head + ev->size) > Q_THRESHOLD) {
		
			/* Event crosses threshold, thus we need space
			 * at start of buffer to put head (tail may be
			 * at 0, but head == tail means an empty buffer).
			 */
			
			if (qp->tail == 0) {
				freespace=0;
			}
		}
	}

	if (! freespace) {
		DPRINT_EVENTS("_ggiEvQueueAdd: Queue overflow\n");
		return -1;
	}

	DPRINT_EVENTS("_ggiEvQueueAdd: Adding event type %d (0x%08x), size %d at pos %d.\n", 
		     ev->any.type, evm, ev->size, qp->count);

	/* Add the event, and mark that we've seen it.. */
	memcpy(qp->buf + qp->head, ev, ev->size);
	
	qp->count++;
	qp->head += ev->size;

	if (qp->head > Q_THRESHOLD) {
		qp->head = 0;
	}
	
	evq->seen |= evm;

	return 0;
}

static int _ggiEvQueueRelease(ggi_visual *vis, ggi_event *ev,
			      ggi_event_mask mask)
{
	ev_queue_set *evq;
	ev_queue *qp = NULL;
	ggi_event_mask evm;
	struct timeval t_min;
	int queue;

	DPRINT_EVENTS("_ggiEvQueueRelease(%p, %p, 0x%x) called\n",
		      vis, ev, mask);

	evq = _ggiEvQSetSetup(vis);
	evm = mask & evq->seen;

	/* Got nothing.. */
	if (evm == 0) {
		return 0;
	}

	/* Max timestamp.. */
	t_min.tv_sec= 0x7FFFFFFF;
	t_min.tv_usec=0x7FFFFFFF;

	/* Get the specified event out of the queue.  If the user asks
	 * for more than one event type, return the one that has been
	 * waiting the longest.
	 */
	for (queue=0; queue < evLast; queue++) {
		struct ev_queue *qp_tmp = evq->queues[queue];

		if (qp_tmp && qp_tmp->count && (evm & (1 << queue))) {
			ggi_event *e_tmp = (ggi_event *)
				(qp_tmp->buf + qp_tmp->tail);
			struct timeval t_tmp = e_tmp->any.time;

			if (t_tmp.tv_sec < t_min.tv_sec || 
			    (t_tmp.tv_sec == t_min.tv_sec &&
                             t_tmp.tv_usec < t_min.tv_usec)) {

				DPRINT_EVENTS("_ggiEvQueueRelease: Plausible found.\n");
				qp = qp_tmp;
				t_min = t_tmp;
			}
		}
	}

	/* Shouldn't happen.. */
	LIBGGI_ASSERT(qp != NULL, "_ggiEvQueueRelease: Arrgghh!! Nothing plausible");
	if (qp == NULL) {
		return 0;
	}
	
	/* Pull event out of queue.. */
	memcpy(ev, qp->buf + qp->tail, qp->buf[qp->tail]);

	qp->count--;
	qp->tail += ev->size;

	if (qp->tail > Q_THRESHOLD) {
		qp->tail = 0;
	}
	if (qp->count == 0) {
		evq->seen &= ~(1 << ev->any.type);
	}
	
	DPRINT_EVENTS("_ggiEvQueueRelease: Retrieved event type %d (0x%08x), size %d.\n", 
	       ev->any.type, 1 << ev->any.type, ev->size);

	return ev->size;
}

int GGIseteventmask(ggi_visual *vis, ggi_event_mask mask)
{
	ev_queue_set *evq;
	int i;
	
	DPRINT_EVENTS("GGIseteventmask(%p, 0x%x) called\n", vis, mask);
	evq = _ggiEvQSetSetup(vis);

	LIBGGI_EVMASK(vis) = mask;

	/* Flush any events whose bit is 0 in the new mask.
	 */
	for (i=0; i < evLast; i++) {
		if (((mask & (1 << i)) == 0) && evq->queues[i]) {
			evq->queues[i]->head  = 0;
			evq->queues[i]->tail  = 0;
			evq->queues[i]->count = 0;
		}
	}

	return 0;
}
