/* $Id: events.c,v 1.10 1998/10/26 04:11:10 ajapted Exp $
***************************************************************************

   TELE target.

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <ggi/internal/ggi-dl.h>
#include "../common/evqueue.inc"

#include "libtele.h"
#include "telehook.h"


#define MINSLEEPTIME  (20*1000)  /* microseconds */


static void handle_telecmd_event(TeleHook *th, TeleEvent *ev)
{
	/* handle TELE_CMD_CLOSE ??? */

	if ((th->wait_event == NULL) ||
	    (th->wait_type != ev->type) ||
	    (th->wait_sequence != ev->sequence)) {
		
		fprintf(stderr, "display-tele: UNEXPECTED CMD EVENT "
			"(type=0x%08x seq=0x%08x)\n",
			ev->type, ev->sequence);
		return;
	}

	DPRINT("display-tele: GOT REPLY (type=0x%08lx seq=0x%08lx)\n",
		ev->type, ev->sequence);

	memcpy(th->wait_event, ev, ev->size * sizeof(long));
}

static int translate_to_ggi(ggi_visual *vis, ggi_event *ev, TeleEvent *th)
{
	if ((th->type & TELE_EVENT_TYPE_MASK) != TELE_INP_BASE) {
		DPRINT("display-tele: unrecognised event from server "
			"(0x%08x).\n", th->type);
		return -1;
	}
	
	ev->any.time.tv_sec  = th->time.sec;
	ev->any.time.tv_usec = th->time.nsec / 1000;

	ev->any.origin = th->device;
	ev->any.target = 0;

	switch (th->type) {
	
		case TELE_INP_KEY:
		case TELE_INP_KEYUP:
		{
			TeleInpKeyData *d = (void *) th->data;

			ev->any.size = sizeof(ggi_key_event);
			ev->any.type = (th->type == TELE_INP_KEY) ?
				evKeyPress : evKeyRelease;
			/* !!! doesn't handle evKeyRepeat */

			ev->key.effect = d->modifiers;

			ev->key.sym    = d->key;
			ev->key.label  = d->label;
			ev->key.button = d->code;

			return 0;
		}

		case TELE_INP_BUTTON:
		case TELE_INP_BUTTONUP:
		{
			TeleInpButtonData *d = (void *) th->data;

			ev->any.size = sizeof(ggi_pbutton_event);
			ev->any.type = (th->type == TELE_INP_BUTTON) ?
				evPtrButtonPress : evPtrButtonRelease;
			
			ev->pbutton.button = d->button;
			ev->pbutton.state = 0;  /* !!! */

			return 0;
		}

		case TELE_INP_MOUSE:
		case TELE_INP_TABLET:
		{
			TeleInpAxisData *d = (void *) th->data;

			if ((d->count < 2) || (d->count > 3)) {
				return -1;
			}
			
			ev->any.size = sizeof(ggi_pmove_event);
			ev->any.type = (th->type == TELE_INP_MOUSE) ?
				evPtrRelative : evPtrAbsolute;
			
			ev->pmove.x = d->axes[0];
			ev->pmove.y = d->axes[1];

			if (d->count >= 3) {
				ev->pmove.z = d->axes[2];
			} else {
				ev->pmove.z = 0;
			}

			if (d->count >= 4) {
				ev->pmove.wheel = d->axes[3];
			} else {
				ev->pmove.wheel = 0;
			}

			return 0;
		}

		case TELE_INP_JOYSTICK:
		{
			/* !!! The following is crap, because joystick
			 * support in LibGGI is non-existent so far.
			 */
			TeleInpAxisData *d = (void *) th->data;
			int i;

			if ((d->count < 2) || (d->count > 4)) {
				return -1;
			}
			
			ev->any.size = sizeof(ggi_val_event);
			ev->any.type = evValAbsolute;
			
			ev->val.first = 0;
			ev->val.count = d->count;

			for (i=0; i < d->count; i++) {
				ev->val.value[i] = d->axes[i];
			}

			return 0;
		}

		case TELE_INP_EXPOSE:
		{
			ev->any.size = sizeof(ggi_expose_event);
			ev->any.type = evExpose;
			
			/* the whole screen */

			ev->expose.x = ev->expose.y = 0;
			ev->expose.w = LIBGGI_VIRTX(vis);
			ev->expose.h = LIBGGI_VIRTY(vis);

			return 0;
		}
	}
	
	DPRINT("display-tele: unknown input event (0x%08x).\n",
		th->type);
	return -1;
}


/* ---------------------------------------------------------------------- */


ggi_event_mask GGI_tele_eventpoll(ggi_visual_t vis, ggi_event_mask mask,
			    struct timeval *t)
{
	TeleHook *th = TELE_PRIV(vis);

	ggi_event_mask evmask;

	int  sec = t ? t->tv_sec  : 0;
	int usec = t ? t->tv_usec : 0;


  	if (! th->connected) {
  		return 0;
  	}

	evmask = _ggiEvQueueSeen(vis, mask);

	if (evmask != 0)  
		return evmask;

	do {
		ggi_event ev;

		TeleEvent th_ev;

		int err;

		if (tclient_poll(th->client) == 0) {
		
			if (t != NULL) {
				if ((sec < 0) ||
				    ((sec == 0) && (usec <= 0))) {

					/* timeout */
					return 0;
				}

				if ((sec == 0) &&
				    (usec < MINSLEEPTIME)) {
					_ggi_usleep(usec);
				} else {
					_ggi_usleep(MINSLEEPTIME);
				}

				usec -= MINSLEEPTIME;
				
				if (usec < 0) {
					sec--;
					usec += 1000000;
				}
			} else {
				_ggi_usleep(MINSLEEPTIME);
			}

			continue;
		}
		
		err = tclient_read(th->client, &th_ev);

		if (err == TELE_ERROR_SHUTDOWN) {
			TELE_HANDLE_SHUTDOWN;
		} else if (err < 0) {
			DPRINT("tclient_read: ZERO\n");
			return 0;
		}
		
		DPRINT("display-tele: got event (type=0x%08x "
			     "seq=0x%08x)\n", (int) th_ev.type,
			     (int) th_ev.sequence);

		if ((th_ev.type & TELE_EVENT_TYPE_MASK) == TELE_CMD_BASE) {
			handle_telecmd_event(th, &th_ev);
			continue;
		}

		if (translate_to_ggi(vis, &ev, &th_ev) == 0) {
			_ggiEvQueueAdd(vis, &ev);
		}
		
		evmask = _ggiEvQueueSeen(vis, mask);
		
	} while (evmask == 0);

	return evmask;
}

int GGI_tele_eventread(ggi_visual_t vis, ggi_event *ev, ggi_event_mask mask)
{
	TeleHook *th = TELE_PRIV(vis);

	if (! th->connected) {
		return 0;
	}

	/* Block if we don't have anything queued... */

	GGI_tele_eventpoll(vis, mask, NULL);

	return _ggiEvQueueRelease(vis, ev, mask);
}


/* ---------------------------------------------------------------------- */


int tele_receive_reply(ggi_visual *vis, TeleEvent *ev, 
			   long type, long seq)
{
	TeleHook *th = TELE_PRIV(vis);

	ev->size = 0;

	th->wait_event = ev;
	th->wait_type  = type;
	th->wait_sequence = seq;

	DPRINT("display-tele: WAITING FOR (type=0x%08lx seq=0x%08lx)\n",
		type, seq);

	for (;;) {
		struct timeval tv;

		tv.tv_sec = tv.tv_usec = 0;

		GGI_tele_eventpoll(vis, 0, &tv);

		if (ev->size != 0) {
			break;
		}

		_ggi_usleep(MINSLEEPTIME);
	}

	DPRINT("display-tele: WAIT OVER (type=0x%08lx seq=0x%08lx)\n",
		type, seq);

	th->wait_event = NULL;

	return 0;
}
