/* $Id: visual.c,v 1.24 1998/10/24 04:03:50 marcus Exp $
***************************************************************************

   Display-trueemu : visual management

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "trueemu.h"

static gg_option trueemu_options[3] =
{
	{ "parent", ""    },
	{ "dither", "4"   },
	{ "model",  "rgb" }
};

int GGIdlinit(ggi_visual *vis, const char *args,void *argptr)
{
	TrueemuHook *th;

	char target[1024];


	/* initialize */

	DPRINT("display-trueemu: GGIdlinit start.\n");

	LIBGGI_FD(vis) = -1;

	th = LIBGGI_PRIVATE(vis) = _ggi_malloc(sizeof(TrueemuHook));

	vis->gc = _ggi_malloc(sizeof(ggi_gc));
	th->opmansync = _ggi_malloc(sizeof(_ggi_opmansync));

	_ggiLockInit(th->flush_lock);

	th->flags  = 0;
	th->fb_ptr = NULL;

	
	/* handle arguments */

	if (getenv("GGI_TRUEEMU_OPTIONS") != NULL) {
		if (ggParseOptions(getenv("GGI_TRUEEMU_OPTIONS"),
			trueemu_options, 3) == NULL) {
			
			fprintf(stderr, "display-trueemu: error in "
				"$GGI_TRUEEMU_OPTIONS.\n");
			free(th);
			return GGI_DL_ERROR;
		}
	}
	
	if (args) {
		args = ggParseOptions((char *) args, trueemu_options, 3);

		if (args == NULL) {
			fprintf(stderr, "display-trueemu: error in "
				"arguments.\n");
			free(th);
			return GGI_DL_ERROR;
		}
	}


	/* parse the results */

	DPRINT("trueemu: parent is '%s'.\n", trueemu_options[0].result);
	DPRINT("trueemu: dither is '%s'.\n", trueemu_options[1].result);
	DPRINT("trueemu: model  is '%s'.\n", trueemu_options[2].result);
	
	ggiParseMode(trueemu_options[0].result, &th->mode);

	switch(trueemu_options[1].result[0]) {

		case '0': th->flags |= TRUEEMU_F_DITHER_0; break;
		case '2': th->flags |= TRUEEMU_F_DITHER_2; break;
		case '4': th->flags |= TRUEEMU_F_DITHER_4; break;

		default:
			fprintf(stderr, "display-trueemu: Unknown dither "
				"'%s'.\n", trueemu_options[1].result);
	}

	switch (trueemu_options[2].result[0]) {

		case 'r': th->flags |= TRUEEMU_F_RGB;    break;
		case 'c': th->flags |= TRUEEMU_F_CUBE;   break;
		case 'p': th->flags |= TRUEEMU_F_PASTEL; break;

		default:
			fprintf(stderr, "display-trueemu: Unknown model "
				"'%s'.\n", trueemu_options[2].result);
	}


	/* setup mansync */

	if (! _ggiAddDL(vis, "helper-mansync", (char *) th->opmansync, NULL, 0)) {
		fprintf(stderr, "display-trueemu: Cannot load "
			"helper-mansync!\n");
		return GGI_DL_ERROR;
	}

	MANSYNC_init(vis);


	/* open the parent visual */

	while (args && *args && isspace((unsigned char) *args)) {
		args++;
	}

	DPRINT("display-trueemu: opening target: %s\n", args);
        
	strcpy(target, "");
	
	if (args) {
		if (ggParseTarget((char *) args, target, 1024) == NULL) {
			/* error occurred */
			free(th);
			return GGI_DL_ERROR;
		}
	}

	if (*target == 0) {
		th->parent = ggiOpen(NULL);
	} else {
		th->parent = ggiOpen(target, NULL);
	}

	if (th->parent == NULL) {
		fprintf(stderr, "display-trueemu: Failed to open "
			"target: '%s'\n", target);
		free(th);
		return GGI_DL_ERROR;
	}
	
	ggiSetFlags(th->parent, GGIFLAG_ASYNC);

	/* Add giiInputs, if we have them. */
	if (th->parent->input)
	{
		vis->input=giiJoinInputs(vis->input,th->parent->input);
		th->parent->input=NULL; /* Destroy old reference */
	}

	/* Mode management */
	vis->opdisplay->getmode = GGI_trueemu_getmode;
	vis->opdisplay->setmode = GGI_trueemu_setmode;
	vis->opdisplay->checkmode = GGI_trueemu_checkmode;
	vis->opdisplay->getapi = GGI_trueemu_getapi;
	vis->opdisplay->flush = GGI_trueemu_flush;
	vis->opdisplay->setflags = GGI_trueemu_setflags;
	
	/* Event management */
	vis->opdisplay->eventpoll = GGI_trueemu_eventpoll;
	vis->opdisplay->eventread = GGI_trueemu_eventread;
	vis->opdisplay->seteventmask = GGIseteventmask;

	DPRINT("display-trueemu: GGIdlinit succeeded.\n");

	return GGI_DL_OPDISPLAY;
}


int GGIdlcleanup(ggi_visual *vis)
{
	TrueemuHook *th = TRUEEMU_PRIV(vis);
	
	DPRINT("display-trueemu: GGIdlcleanup start.\n");

	MANSYNC_deinit(vis);

	if (th->fb_ptr != NULL) {
		GGI_trueemu_resetmode(vis);
	}

	if (th->parent != NULL) {
		ggiClose(th->parent);
		th->parent = NULL;
	}

	LIBGGI_PRIVATE(vis) = NULL;

	_ggiLockDestroy(th->flush_lock);

	free(th->opmansync);
	free(vis->gc);
	free(th);

	DPRINT("display-trueemu: GGIdlcleanup done.\n");

	return 0;
}

#include <ggi/internal/ggidlinit.h>
