/* $Id: dl.c,v 1.21 1998/10/26 12:46:22 marcus Exp $
***************************************************************************

   Graphics library for GGI. Library extensions dynamic loading.

   Copyright (C) 1997 Jason McMullan	[jmcc@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef _WIN32
#include <windows.h> 
#define GGIOPENLIB(filename)	LoadLibrary((filename))
#define GGIOPENLIBGLOBAL(filename) GGIOPENLIB(filename)
#define GGIGETSYM(handle,sym)	GetProcAddress((handle),(sym))
#define GGICLOSELIB(handle)	FreeLibrary((handle))
#else /* _WIN32 */

#include <dlfcn.h>
#ifdef RTLD_LAZY
#define GGIOPENLIB(filename)		dlopen((filename), RTLD_LAZY)
#ifdef RTLD_GLOBAL
#define GGIOPENLIBGLOBAL(filename)	dlopen((filename), RTLD_LAZY | RTLD_GLOBAL)
#else
#define GGIOPENLIBGLOBAL(filename)	dlopen((filename), RTLD_LAZY)
#endif /* RTLD_GLOBAL */
#elif defined(DL_LAZY)
#define GGIOPENLIB(filename)		dlopen((filename), DL_LAZY)
#define GGIOPENLIBGLOBAL(filename)	dlopen((filename), DL_LAZY)
#else
#error This system has an unknown dlopen() call !!!
#endif /* RTLD_LAZY */
#define GGIGETSYM(handle,sym)	dlsym((handle),(sym))
#define GGICLOSELIB(handle)	dlclose((handle))
#endif /* _WIN32 */

#ifdef __OpenBSD__
#define GGI_SYMPREFIX		"_"
#else
#define GGI_SYMPREFIX		""
#endif

#define GGI_DLINIT_SYM		GGI_SYMPREFIX##"GGIdlinit"
#define GGI_DLCLEANUP_SYM	GGI_SYMPREFIX##"GGIdlcleanup" 

#include <ggi/internal/internal.h>
#include <ggi/gg.h>

/* Open the dynamic libary requested
 */
ggi_dlhandle *_ggiLoadDL(const char *filename, int type)
{
	ggi_dlhandle hand,*hp;

	DPRINT_LIBS("_ggiLoadDL(\"%s\", 0x%x) called \n", filename, type);

	hand.name=NULL;
	hand.use=0;

	if(type & GGI_DLTYPE_GLOBAL)
		hand.handle=GGIOPENLIBGLOBAL(filename);
	else
		hand.handle=GGIOPENLIB(filename);

	DPRINT_LIBS("hand.handle=%p\n",hand.handle);
	if (hand.handle==NULL) 
		return NULL;

	hand.init=(int (*)(struct ggi_visual *, const char *,void *))  GGIGETSYM(hand.handle,GGI_DLINIT_SYM);
	hand.cleanup=(int (*)(struct ggi_visual *)) GGIGETSYM(hand.handle,GGI_DLCLEANUP_SYM);

	DPRINT_LIBS("hand.init=%p\n",hand.init);
	DPRINT_LIBS("hand.cleanup=%p\n",hand.cleanup);
	if (hand.init==NULL || hand.cleanup==NULL) {
		GGICLOSELIB(hand.handle);
		return NULL;
	}

	hp=(ggi_dlhandle *)_ggi_malloc(sizeof(ggi_dlhandle));
	memcpy(hp,&hand,sizeof(ggi_dlhandle));

	return hp;
}

/****** Open and Close a DL *********/
ggi_dlhandle *_ggiAddDL(ggi_visual *vis,const char *driver,
		const char *args,void *argptr,int type)
{
	const char *fname;
	ggi_dlhandle *dlh;
	ggi_dlhandle_l *tmp;
	int err;

	DPRINT_LIBS("_ggiAddDL(%p, \"%s\", \"%s\", 0x%x) called\n",
		    vis, driver, args, type);
	
	fname = ggMatchConfig(_ggiConfigHandle, driver, args);

	if (fname == NULL) {
		fprintf(stderr, "LibGGI: libggi.conf doesn't have an entry for: %s\n", driver);
		return NULL;
	}

	dlh=_ggiLoadDL(fname, type);
	DPRINT_LIBS("_ggiLoadDL returned %p\n", dlh);
	if (dlh==NULL)
		return NULL;
	
	err=dlh->init(vis,args,argptr);
	DPRINT_LIBS("%d=dlh->init(%p,\"%s\",%p) - %s %s\n",err,vis,args,argptr,driver,fname);
	if (err & GGI_DL_ERROR) {
		GGICLOSELIB(dlh->handle);
		free(dlh);
		return NULL;
	}
	dlh->type=type;
	if (type==GGI_DLTYPE_INTERNAL) {
		if (err & GGI_DL_OPDISPLAY) {
			tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
			tmp->handle=dlh;
			tmp->next=vis->opdisplay->dlhandle;
			vis->opdisplay->dlhandle=tmp;
			dlh->use++;
		}
	
		if (err & GGI_DL_OPCOLOR) {
			tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
			tmp->handle=dlh;
			tmp->next=vis->opcolor->dlhandle;
			vis->opcolor->dlhandle=tmp;
			dlh->use++;
		}

		if (err & GGI_DL_OPDRAW) {
			tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
			tmp->handle=dlh;
			tmp->next=vis->opdraw->dlhandle;
			vis->opdraw->dlhandle=tmp;
			dlh->use++;
		}

		if (err & GGI_DL_OPGC) {
			tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
			tmp->handle=dlh;
			tmp->next=vis->opgc->dlhandle;
			vis->opgc->dlhandle=tmp;
			dlh->use++;
		}
	} else {
		dlh->use=1;
		tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
		tmp->handle=dlh;
		tmp->next=vis->extlib;
		vis->extlib=tmp;
	}

	if (dlh->use==0) {
		fprintf(stderr,"libggi: %s (%s) -> 0x%.8x - no operations in this library\n",driver,args,err);
		GGICLOSELIB(dlh->handle);
		free(dlh);
		return NULL;
	} else {
		tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
		tmp->handle=dlh;
		tmp->next=LIBGGI_DLHANDLE(vis);
		LIBGGI_DLHANDLE(vis)=tmp;
	}

	dlh->name=strdup(driver);

	return dlh;
}

ggi_dlhandle *_ggiOpenDL(ggi_visual *vis,const char *driver,const char *args,void *argptr)
{
	return _ggiAddDL(vis,driver,args,argptr,GGI_DLTYPE_INTERNAL);
}

void _ggiRemoveDL(ggi_visual *vis,ggi_dlhandle_l **lib)
{
	ggi_dlhandle_l *tmp,**prev;
	ggi_dlhandle_l *libtmp,**libprev,*libnext;

	for (libprev=lib,libtmp=*lib;libtmp!=NULL; libtmp=libnext) {
		libnext=libtmp->next;
		if (libtmp->handle->use<=0) {
			DPRINT_LIBS("Disposing \"%s\"\n",libtmp->handle->name);
			*libprev=libtmp->next;
			libtmp->handle->cleanup(vis);
			DPRINT_LIBS("Closing handle: 0x%x\n", libtmp->handle->handle);
			GGICLOSELIB(libtmp->handle->handle);
			
			/* Now, clean up the master visual */
			prev=&LIBGGI_DLHANDLE(vis);
			for (tmp=LIBGGI_DLHANDLE(vis);tmp;tmp=tmp->next) {
				if (tmp->handle==libtmp->handle) break;
				prev=&tmp->next;
			}
			if (!tmp) DPRINT_LIBS("Error: handle not in master list.\n");
			*prev=tmp->next;
			free(tmp);

			free(libtmp->handle->name);
			free(libtmp->handle);
			free(libtmp);
		} else {
			libprev=&libtmp->next;
		}
	}
}

static void _ggiZapDL(ggi_visual *vis,ggi_dlhandle_l **lib)
{
	ggi_dlhandle_l *tmp,*next;

	DPRINT_LIBS("ZAPDL call: %p %p\n",vis,lib);

	for (tmp=*lib;tmp;tmp=tmp->next) 
		tmp->handle->use--;

	_ggiRemoveDL(vis,lib);

	for (tmp=*lib;tmp;tmp=next) {
		next=tmp->next;
		free(tmp);
	}

	*lib=NULL;
}

void _ggiCloseDL(ggi_visual *vis,ggi_uint type)
{
	if (vis==NULL)
		return;
	
	if ((type & GGI_DL_OPDISPLAY) && vis->opdisplay) {
		_ggiZapDL(vis,&vis->opdisplay->dlhandle);
	}

	if ((type & GGI_DL_OPCOLOR) && vis->opcolor) {
		_ggiZapDL(vis,&vis->opcolor->dlhandle);
	}

	if ((type & GGI_DL_OPDRAW) && vis->opdraw) {
		_ggiZapDL(vis,&vis->opdraw->dlhandle);
	}

	if ((type & GGI_DL_OPGC) && vis->opgc) {
		_ggiZapDL(vis,&vis->opgc->dlhandle);
	}

	if ((type & GGI_DL_EXTENSION) && vis->extlib) {
		_ggiZapDL(vis,&vis->extlib);
	}

}
