/* $Id: input.c,v 1.3 1998/10/29 12:02:35 ajapted Exp $
***************************************************************************

   Mouse: input

   Copyright (C) 1998 Andrew Apted     [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdlib.h>
#include <unistd.h>
#include <termios.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <ggi/internal/gii.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef __linux__  /* !!! Autoconf me */
#include <linux/kdev_t.h>   /* only needed for MAJOR() macro */
#include <linux/major.h>    /* only needed for MISC_MAJOR */
#endif


/* ---------------------------------------------------------------------- */


#define MAX_MOUSE_NAMES  8
#define MAX_MOUSE_TYPES  32

#define MAX_PACKET_BUF	128

struct mouse_hook;

typedef struct mouse_type
{
	const char *names[MAX_MOUSE_NAMES];

	/* Serial parameters.  If the mouse is not a serial mouse
	 * (for example, a busmouse), then default_baud should be < 0.
	 */

	int default_baud;
	int cflag;  int iflag;
	int oflag;  int lflag;

	/* The parser function.
	 *
	 * This returns the number of bytes used by the packet.  It
	 * should return 0 when the packet is too short, and return 1
	 * when the packet is invalid (to resync).  len is guaranteed
	 * to be >= minimum_packet_size.
	 */

	int (* parse_func)(gii_input *inp, uint8 *buf, int len);

	int minimum_packet_len;
	
} MouseType;

typedef struct mouse_hook
{
	MouseType *type;

	int fd;

	int button_state;
	int parse_state;
	int left_hand;

	struct termios old_termios;

	/* mouse packet buffer */
	
	int packet_len;
	uint8 packet_buf[MAX_PACKET_BUF];

	gii_event_mask sent;

} MouseHook;

#define MOUSE_HOOK(inp)  ((MouseHook *) inp->priv)


#define C_NORM	(CREAD | CLOCAL | HUPCL)
#define I_NORM	(IGNBRK)


/* ---------------------------------------------------------------------- */


/**
 **  Event dispatching routines
 **/

static void mouse_send_movement(gii_input *inp, int dx, int dy)
{
	gii_event ev;

	if (dx || dy) {

		ev.pmove.size   = sizeof(gii_pmove_event);
		ev.pmove.type   = evPtrRelative;
		ev.pmove.origin = EV_ORIGIN_NONE;
		ev.pmove.target = EV_TARGET_NONE;

		EV_TIMESTAMP(&ev);

		ev.pmove.x = dx;
		ev.pmove.y = dy;

		_giiEvQueueAdd(inp, &ev);

		MOUSE_HOOK(inp)->sent |= emPtrRelative;
	}
}

static void mouse_send_buttons(gii_input *inp, int buttons, int last)
{
	gii_event ev;

	int mask;
	int changed = buttons ^ last;

	
	/* change in button state ? */

	for (mask = 1; mask > 0; mask <<= 1)
	if (changed & mask) {

		int state = buttons & mask;

		ev.pbutton.size = sizeof(gii_pbutton_event);
		ev.pbutton.type = state ?
			evPtrButtonPress : evPtrButtonRelease;
		ev.pmove.origin = EV_ORIGIN_NONE;
		ev.pmove.target = EV_TARGET_NONE;

		EV_TIMESTAMP(&ev);

		ev.pbutton.button = mask;  /* !!! */

		_giiEvQueueAdd(inp, &ev);

		MOUSE_HOOK(inp)->sent |= (1 << ev.any.type);
	}
}


/**
 **  Mouse Packet Parsers
 **/

static int B_microsoft[16] =
{	0x0, 0x2, 0x1, 0x3, 0x4, 0x6, 0x5, 0x7,
	0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7
};

static int M_microsoft(gii_input *inp, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0x40) != 0x40) || ((buf[1] & 0x40) != 0x00)) {

		DPRINT_EVENTS("Invalid microsoft packet\n");
		return 1;
	}

	buttons = B_microsoft[hand | ((buf[0] & 0x30) >> 4)];

	dx = (sint8) (((buf[0] & 0x03) << 6) | (buf[1] & 0x3f));
	dy = (sint8) (((buf[0] & 0x0c) << 4) | (buf[2] & 0x3f));

	mouse_send_movement(inp, dx, dy);
	mouse_send_buttons(inp, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT_EVENTS("Got microsoft packet\n");

	return 3;
} 

static MouseType T_microsoft =
{
	{ "Microsoft", "ms", NULL, },
	B1200, C_NORM | CS7, I_NORM, 0, 0,
	M_microsoft, 3 
};


static int B_mousesys[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_mousesys(gii_input *inp, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if ((buf[0] & 0xf8) != 0x80) {

		DPRINT_EVENTS("Invalid mousesys packet\n");
		return 1;
	}

	buttons = B_mousesys[hand | (~buf[0] & 0x07)];

	dx =  (sint8) buf[1] + (sint8) buf[3];
	dy = -(sint8) buf[2] - (sint8) buf[4];

	mouse_send_movement(inp, dx, dy);
	mouse_send_buttons(inp, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT_EVENTS("Got mousesys packet\n");

	return 5;
}

static MouseType T_mousesys =
{
	{ "MouseSystems", "mousesystem", "mousesys", "msc", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_mousesys, 5
};


static int B_logitech[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_logitech(gii_input *inp, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0xe0) != 0x80) || ((buf[1] & 0x80) != 0x00)) {

		DPRINT_EVENTS("Invalid logitech packet\n");
		return 1;
	}

	buttons = B_logitech[hand | (buf[0] & 0x07)];

	dx = (buf[0] & 0x10) ?	(sint8)buf[1] : -(sint8)buf[1];
	dy = (buf[0] & 0x08) ? -(sint8)buf[2] :  (sint8)buf[2];

	mouse_send_movement(inp, dx, dy);
	mouse_send_buttons(inp, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT_EVENTS("Got logitech packet\n");

	return 3;
}

static MouseType T_logitech = 
{
	{ "Logitech", "logi", "log", "Logitech", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_logitech, 3
};


static int B_sun[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_sun(gii_input *inp, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if ((buf[0] & 0xf8) != 0x80) {

		DPRINT_EVENTS("Invalid sun packet\n");
		return 1;
	}

	buttons = B_sun[hand | (~buf[0] & 0x07)];

	dx =  (sint8) buf[1];
	dy = -(sint8) buf[2];

	mouse_send_movement(inp, dx, dy);
	mouse_send_buttons(inp, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT_EVENTS("Got sun packet\n");

	return 3;
}

static MouseType T_sun =
{
	{ "Sun", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_sun, 3
};


static int B_mouseman[16] =
{	0x0, 0x2, 0x1, 0x3, 0x4, 0x6, 0x5, 0x7,
	0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7
};

static int M_mouseman(gii_input *inp, uint8 *buf, int len)
{
	/*  The damned MouseMan has 3/4 byte packets.  The extra byte
	 *  is only there if the middle button is active.
	 *
	 *  This is what we do: when we get the first 3 bytes, we parse
	 *  the info and send off the events, and set a flag to say we
	 *  have seen the first three bytes.
	 * 
	 *  When we get the fourth byte (maybe with the first three,
	 *  or maybe later on), we check if it is a header byte.
	 *  If so, we return 3, otherwise we parse the buttons in it,
	 *  send off the events, and return 4.
	 *
	 *  Note also that unlike the other mices, the mouseman parser
	 *  stores the RAW buttons in priv->button_state.
	 */

	MouseHook *mhook = MOUSE_HOOK(inp);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0x40) != 0x40) || ((buf[1] & 0x40) != 0x00)) {

		DPRINT_EVENTS("Invalid mouseman packet\n");
		return 1;
	}

	/* handle the common 3 bytes */

	if (mhook->parse_state == 0) {

		buttons = (mhook->button_state & 0x4) |
			((buf[0] & 0x30) >> 4);

		dx = (sint8) (((buf[0] & 0x03) << 6) | (buf[1] & 0x3f));
		dy = (sint8) (((buf[0] & 0x0c) << 4) | (buf[2] & 0x3f));

		mouse_send_movement(inp, dx, dy);
		mouse_send_buttons(inp, B_mouseman[hand | buttons],
			B_mouseman[hand | mhook->button_state]);

		mhook->button_state = buttons;
		mhook->parse_state  = 1;

		DPRINT_EVENTS("Got mouseman base packet\n");
	}


	/* now look for extension byte */

	if (len < 4) {
		return 0;
	}

	mhook->parse_state = 0;

	if ((buf[3] & 0xc0) != 0) {
	
		/* 4th byte must be a header byte */
		
		return 3;
	}

	/* handle the extension byte */

	buttons = (mhook->button_state & 0x3) | ((buf[3] & 0x20) >> 3);

	mouse_send_buttons(inp, B_mouseman[hand | buttons],
		B_mouseman[hand | mhook->button_state]);

	mhook->button_state = buttons;

	DPRINT_EVENTS("Got mouseman extension packet\n");

	return 4;
}

static MouseType T_mouseman =
{
	{  "MouseMan", "mman", NULL, },
	B1200, C_NORM | CS7, I_NORM, 0, 0,
	M_mouseman, 3
};


/* 
 * MMSeries mice use the same protocal as Logitech (and vice-versa)
 */

static MouseType T_mmseries =
{
	{ "MMSeries", "mm", NULL, },
	B1200, C_NORM | CS8 | PARENB | PARODD, I_NORM, 0, 0,
	M_logitech, 3
};
	

/*
 * BusMice use the same protocal as Sun
 */

static MouseType T_busmouse =
{
	{ "BusMouse", "bus", "bm", NULL, },
	-1, 0, 0, 0, 0,
	M_sun, 3
};

static int B_ps2[16] =
{	0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7,
	0x0, 0x2, 0x1, 0x3, 0x4, 0x5, 0x6, 0x7
};

static int M_ps2(gii_input *inp, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* Check header byte. */
	if ((buf[0] & 0xc0) != 0) {
		DPRINT_EVENTS("Invalid PS2 packet\n");
		return 1;
	}

	buttons = B_ps2[hand | (buf[0] & 0x07)];

	dx = (buf[0] & 0x10) ? buf[1] - 256 : buf[1];
	dy = (buf[0] & 0x20) ? -(buf[2] - 256) : -buf[2];

	mouse_send_movement(inp, dx, dy);
	mouse_send_buttons(inp, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT_EVENTS("Got PS2 packet\n");

	return 3;
}

static MouseType T_ps2 =
{
 	{ "PS2", "ps", NULL, },
	-1, 0, 0, 0, 0,
 	M_ps2, 3
};


/* ---------------------------------------------------------------------- */


/* The Holy Table of Mouse Parsers
*/

static MouseType *mice_types[MAX_MOUSE_TYPES] =
{
	& T_microsoft, 
	& T_mousesys,
	& T_logitech,
	& T_sun,
	& T_mmseries,
	& T_mouseman,
	& T_busmouse,
	& T_ps2,
	
	NULL,
};

#define DEFAULT_MOUSE  0


/* ---------------------------------------------------------------------- */


/* Parse one packet
 */
 
static int do_parse_packet(gii_input *inp)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	int used;

#if 0
	{	int i;

		fprintf(stderr, "Mouse: do_parse_packet [");

		for (i=0; i < (MOUSE_HOOK(inp)->packet_len - 1); i++) {
			fprintf(stderr, "%02x ", 
				MOUSE_HOOK(inp)->packet_buf[i]);
		}

		fprintf(stderr, "%02x]\n", MOUSE_HOOK(inp)->packet_buf[i]);
	}
#endif

	/* call parser function */
	
	used = mhook->type->parse_func(inp, mhook->packet_buf, 
					    mhook->packet_len);

	DPRINT_EVENTS("packet used %d bytes\n", used);

	return used;
}

static int find_mouse(char *name)
{
	int m, n;

	for (m=0; (m < MAX_MOUSE_TYPES) && (mice_types[m] != NULL); m++) {

		MouseType *mtype = mice_types[m];

		for (n=0; (n < MAX_MOUSE_NAMES) && 
		          (mtype->names[n] != NULL); n++) {

			if (strcasecmp(mtype->names[n], name) == 0) {
			
				return m;  /* found it */
			}
		}
	}

	fprintf(stderr, "Unknown mouse type '%s' -- using the default.\n", 
		name);

	return DEFAULT_MOUSE;
}

#if 0
static int find_baud(char *baudname)
{
	switch (atoi(baudname))
	{
		case 9600: return B9600;
		case 4800: return B4800;
		case 2400: return B2400;
		case 1200: return B1200;
	}

	fprintf(stderr, "Baud rate '%s' not supported\n", baudname);
	
	return B1200;  /* !!! */
}
#endif

static int do_mouse_open(gii_input *inp, char *filename,
			 int dtr, int rts, int baud)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	struct termios new;


	mhook->fd = open(filename, O_RDWR | O_NOCTTY);

	if (mhook->fd < 0) {
		DPRINT("Mouse: Failed to open '%s'.\n", filename);
		return -1;
	}

	if (mhook->type->default_baud >= 0) {

		/* set up the termios state and baud rate */
	
		tcflush(mhook->fd, TCIOFLUSH);

		if (tcgetattr(mhook->fd, &mhook->old_termios) < 0) {
			DPRINT("tcgetattr failed.\n");
/*			close(mhook->fd);	*/
/*			return -1;		*/
		}

		new = mhook->old_termios;

		if (baud < 0) {
			baud = mhook->type->default_baud;
		}
		
		new.c_cflag = mhook->type->cflag | baud;
		new.c_iflag = mhook->type->iflag;
		new.c_oflag = mhook->type->oflag;
		new.c_lflag = mhook->type->lflag;
		new.c_cc[VMIN]  = 1;
		new.c_cc[VTIME] = 0;

		if (tcsetattr(mhook->fd, TCSANOW, &new) < 0) {
			DPRINT("tcsetattr failed.\n");
/*			close(mhook->fd);	*/
/*			return -1;		*/
		}

		/* set up RTS and DTR modem lines */

		if ((dtr >= 0) || (rts >= 0)) {

			unsigned int modem_lines;

			if (ioctl(mhook->fd, TIOCMGET, &modem_lines) == 0) {

				if (dtr == 0) modem_lines &= ~TIOCM_DTR;
				if (rts == 0) modem_lines &= ~TIOCM_RTS;

				if (dtr > 0) modem_lines |= TIOCM_DTR;
				if (rts > 0) modem_lines |= TIOCM_RTS;
				
				ioctl(mhook->fd, TIOCMSET, &modem_lines);
			}
		}
	}

	return 0;
}

/* !!! All this parsing stuff is probably best done with the
 * ggParseOption() code, with things like "-file=/dev/mouse",
 * "-type=microsoft", "-baud=9600", and that sort of thing...
 */
 
static char *parse_field(char *dst, int max, char *src)
{
	int len=1;   /* includes trailing NUL */

	for (; *src && (*src != ','); src++) {

		if (len < max) {
			*dst++ = *src;
			len++;
		}
	}

	*dst = 0;

	if (*src == ',') {
		src++;
	}
	return src;
}

static void parse_mouse_specifier(char *spec, char *protname,
				  char *devname, char *options)
{
	*protname = *devname = *options = 0;

	/* LISP-haters should shut their eyes now :) */

	if (spec) 
	parse_field(options, 255,
		    parse_field(devname, 255,
				parse_field(protname, 255, spec)));

	/* supply defaults for missing bits */

	if (*devname == 0) {
		strcpy(devname, "/dev/mouse");
	}

	if (*protname == 0) {

		/* Protocol hasn't been specified. So try statting the
		 * file to see if it is a char device with Major == 10.
		 * In this case, the protocol is most likely BusMouse.
		 */

		struct stat m_stat;

		strcpy(protname, "microsoft");

#ifdef /* !!! */ __linux__
		if ((stat(devname, &m_stat) == 0) &&
		    S_ISCHR(m_stat.st_mode) &&
		    (MAJOR(m_stat.st_rdev) == MISC_MAJOR)) {
		    
			strcpy(protname, "busmouse");
		}
#endif
	}
}

static char *parse_opt_int(char *opt, int *val)
{
	*val = 0;

	for (; *opt && isdigit(*opt); opt++) {
		*val = ((*val) * 10) + ((*opt) - '0');
	}

	return opt;
}

static void parse_options(char *opt, int *baud, int *dtr, int *rts)
{
	while (*opt) switch (*opt++) {

		case 'b': case 'B':    /* baud */
			opt = parse_opt_int(opt, baud);
			break;

		case 'd': case 'D':    /* dtr */
			opt = parse_opt_int(opt, dtr);
			break;

		case 'r': case 'R':    /* rts */
			opt = parse_opt_int(opt, rts);
			break;

		default:
			fprintf(stderr, "Unknown mouse option "
				"'%c' -- rest ignored.\n", *opt);
			return;
	}
}


/* ---------------------------------------------------------------------- */


static int GII_mouse_init(gii_input *inp, char *typename)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	char protname[256];
	char devname[256];
	char options[256];

	int mindex;
	int dtr=-1, rts=-1, baud=-1;


	/* allocate mouse private structure */

	mhook = inp->priv = _gii_malloc(sizeof(MouseHook));

	/* parse the mouse specifier */
	
	parse_mouse_specifier(typename, protname, devname, options);
	parse_options(options, &baud, &dtr, &rts);

	if (strcmp(protname, "none") == 0) {
		return -1;
	}


 	/* open mouse */

	mindex = find_mouse(protname);

	mhook->type = mice_types[mindex];

	if (do_mouse_open(inp, devname, dtr, rts, baud) < 0) {

		fprintf(stderr, "Mouse: Couldn't open mouse.\n");
		return -1;
	}

	
	DPRINT("Mouse: init OK.\n");

	return 0;
}

static void GII_mouse_exit(gii_input *inp)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	if (tcsetattr(mhook->fd, TCSANOW, &mhook->old_termios) < 0) {
		DPRINT("tcsetattr failed.\n");
	}

	close(mhook->fd);
	mhook->fd = -1;

	free(mhook);
	inp->priv = NULL;

	DPRINT("Mouse: exit OK.\n");
}


static gii_event_mask GII_mouse_handle_data(gii_input *inp)
{
	MouseHook *mhook = MOUSE_HOOK(inp);

	int read_len;


	mhook->sent = 0;

	/* read the mouse data */

	read_len = MAX_PACKET_BUF - mhook->packet_len - 1;

	/* ASSERT(read_len >= 1) */

	read_len = read(mhook->fd, mhook->packet_buf + mhook->packet_len, 
			read_len);

	if (read_len < 1) {
		perror("Mouse: Error reading mouse");
		return 0;
	}

	mhook->packet_len += read_len;
	

	/* parse any packets */

	while (mhook->packet_len >= mhook->type->minimum_packet_len) {

		int used;

		used = do_parse_packet(inp);

		if (used <= 0) {
			break;	 /* not enough data yet */
		}

		mhook->packet_len -= used;

		if (mhook->packet_len > 0) {
		
			memmove(mhook->packet_buf, mhook->packet_buf + used,
				mhook->packet_len);
		} else {
			mhook->packet_len = 0;
		}
	}

	DPRINT_EVENTS("mouse_handle_data() done.\n");

	return mhook->sent;
}


/* ---------------------------------------------------------------------- */


static gii_event_mask GII_mouse_poll(gii_input *inp)
{
	MouseHook *mhook = MOUSE_HOOK(inp);
	
	gii_event_mask result = 0;


	DPRINT_MISC("mouse: poll(%p)\n", inp);
	
	for (;;) {

		fd_set readset;

		struct timeval t = {0,0};
		int rc;

		FD_ZERO(&readset);
		FD_SET(mhook->fd, &readset);

		/* FIXME !!! doesn't handle -EINTR */
		rc = select(inp->maxfd, &readset, NULL, NULL, &t);

		if (rc <= 0)
			break;

		result |= GII_mouse_handle_data(inp);
	}

	return result;
}


/* ---------------------------------------------------------------------- */


static int GII_mouse_close(gii_input *inp)
{
	DPRINT_MISC("Mouse cleanup\n");

	if (MOUSE_HOOK(inp)) {
		GII_mouse_exit(inp);
	}

	return 0;
}

int GIIdlinit(void *vis, const char *args)  /* !!! */
{
	gii_input *inp = (gii_input *) vis;   /* FIXME ! HACKHACKHACK !*/

	char *spec = "";

	
	DPRINT_MISC("mouse starting.\n");

	/* Initialize */

	if (args && *args) {
		spec = (char *) args;
	}

	if (GII_mouse_init(inp, spec) < 0) {
		return -1;
	}
	
	/* We leave these on the default handlers
	 *	inp->GIIseteventmask = _GIIstdseteventmask;
	 *	inp->GIIgeteventmask = _GIIstdgeteventmask;
	 *	inp->GIIgetselectfdset = _GIIstdgetselectfd;
	 */

	inp->GIIeventpoll = GII_mouse_poll;
	inp->GIIclose = GII_mouse_close;

	inp->targetcan = emPointer;
	inp->GIIseteventmask(inp, emPointer);

	inp->maxfd = MOUSE_HOOK(inp)->fd + 1;
	FD_SET(MOUSE_HOOK(inp)->fd, &inp->fdset);

	DPRINT_MISC("mouse fully up\n");

	return 0;
}
