/* cp_comps.c = radius, curvature, misc, routines for circle packing in
hyp, sph, eucl setting */

#include "cp_head.h"

extern void fillcurves();

float
s_star_area(p,v)
struct p_data *p;
int v;
{
	struct R_data *pR_ptr;
	struct K_data *pK_ptr;
	float area;
	int i,j,w,next,last;

	pR_ptr=p->packR_ptr;
	pK_ptr=p->packK_ptr;
	area=pR_ptr[v].curv;
	area-=pK_ptr[v].num * M_PI;
	for(i=1;i<=pK_ptr[v].num;i++)
	 {
		w=pK_ptr[v].flower[i];
		area+=pR_ptr[w].curv;
		area-=(pK_ptr[w].num -3)*M_PI;
		last=nghb(p,w,pK_ptr[v].flower[i-1]);
		for(j=1;j<=pK_ptr[w].num-3;j++)
		 {
			next=(last+1<=pK_ptr[w].num) ? 
				(last+1) : 1;
			if (j==1)
			  area+=acos(s_comp_cos(
				pR_ptr[pK_ptr[w].flower[next]].rad,
			  	pR_ptr[w].rad,
				pR_ptr[pK_ptr[w].flower[last]].rad));
			else
			 {
			   area+=acos(s_comp_cos(
				pR_ptr[pK_ptr[w].flower[last]].rad,
				pR_ptr[w].rad,
				pR_ptr[pK_ptr[w].flower[next]].rad));

			   area+=acos(s_comp_cos(
				pR_ptr[pK_ptr[w].flower[next]].rad,
				pR_ptr[w].rad,
				pR_ptr[pK_ptr[w].flower[last]].rad));
			  }									
			last=next;
		 }
	 }	
	
	return area;
} /* s_star_area */

int
fill_star_curves(p,v)
struct p_data *p;
int v;
{
	int i,j,flag;
	struct R_data *pR_ptr;
	struct K_data *pK_ptr;

	pR_ptr=p->packR_ptr;
	pK_ptr=p->packK_ptr;

	s_anglesum_overlap(p,v,pR_ptr[v].rad,&pR_ptr[v].curv,&flag);
	for(i=0;i<=pK_ptr[v].num;i++)
	 {
		j=pK_ptr[v].flower[i];
		s_anglesum_overlap(p,j,pR_ptr[j].rad,
			&pR_ptr[j].curv,&flag);
	 }
	return 1;
} /* fill_star_curves */

float
max_r_adjustment (p,v) /* rad limits in sph setting */
struct p_data *p;
int v;
{
	float dist,min;
	int i;
	
	min= M_PI- (p->packR_ptr[v].rad + p->packR_ptr
		[p->packK_ptr[v].flower[0]].rad + p->packR_ptr
		[p->packK_ptr[v].flower[1]].rad);

	for (i=1;i<=p->packK_ptr[v].num - 1;i++)
	  {
		dist= M_PI -(p->packR_ptr[v].rad + p->packR_ptr[p->packK_ptr
			[v].flower[i]].rad +p->packR_ptr[p->packK_ptr[v].
			flower[i+1]].rad);
		if (dist<min)
			min=dist;
	   }

	if (min>okerr)
		return min;
	else
		return -1.0;
} /* max_r_adjustment */

int
h_norm_pack(p,c,d) /* normalizes hyperbolic data of pack p by 
putting point c at origin and d on pos y-axis */
complex c,d;
struct p_data *p;
{
	int i;
	float abd;
	complex temp;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	if (cAbs(c)>=okerr) /* j vertex not origin */
		for (i=1;i<=p->nodecount;i++)
			pR_ptr[i].center=mob_norm(pR_ptr[i].center,c,d);
	else if (d.im>=okerr || d.im<=(-okerr) || d.re<=(-okerr))
		/* just a rotation is needed */
	 {
		abd=1.0/cAbs(d);
		d.re *= abd;
		d.im *= abd;
		for (i=1;i<=p->nodecount;i++)
			pR_ptr[i].center=cdiv(pR_ptr[i].center,d);
	 }
		/* now rotate another 90 degrees --- need to clean this up
		   later ?? */
	for (i=1;i<=p->nodecount;i++)
	 {
		temp=pR_ptr[i].center;
		pR_ptr[i].center.re=(-temp.im);
		pR_ptr[i].center.im=temp.re;
	 }
	return 1;
} /* h_norm_pack */

int 
face_okay(pK_ptr,face) /* return -1 if face not ready to draw, -2 if
vertices already drawn, else return index of circle to draw first */
struct K_data *pK_ptr;
f_data *face;
{
	if (pK_ptr[face->vert[0]].plot_flag 
		&& pK_ptr[face->vert[1]].plot_flag) 
	 {
		if (pK_ptr[face->vert[2]].plot_flag) return -2;
		else return 0;
	 }
	if (pK_ptr[face->vert[1]].plot_flag 
		&& pK_ptr[face->vert[2]].plot_flag) 
	 {
		if (pK_ptr[face->vert[0]].plot_flag) return -2;
		else return 1;
	 }
	if (pK_ptr[face->vert[2]].plot_flag 
		&& pK_ptr[face->vert[0]].plot_flag) 
	 {
		if (pK_ptr[face->vert[1]].plot_flag) return -2;
		else return 2;
	 }
	return -1;
} /* face_okay */

int
set_face_next(p,datastr) /* set's "next_face" index of faces */
struct p_data *p;
char *datastr;
{
	char next[256];
	int f1,f2;
	char *nextptr;

	nextptr=datastr;
	while (grab_next(&nextptr,next) && sscanf(next,"%d",&f1)==1
	   && f1>=0 && f1<=p->facecount 
	   && grab_next(&nextptr,next) && sscanf(next,"%d",&f2)==1
	   && f2>0 && f2<=p->facecount)
	 {
		if (f1==0) p->first_face=f2;
		else p->faces[f1].next_face=f2;
	 }
	return 1;
} /* set_face_next */

int
set_face_index(p,datastr) /* set "index_flag" of faces */
struct p_data *p;
char *datastr;
{
	char next[256];
	int fn,ind;
	char *nextptr;

	nextptr=datastr;
	while (grab_next(&nextptr,next) && sscanf(next,"%d",&fn)==1
	   && fn>0 && fn<=p->facecount 
	   && grab_next(&nextptr,next) && sscanf(next,"%d",&ind)==1
	   && ind >=0 && ind <3)
		p->faces[fn].index_flag=ind;
	return 1;
} /* set_face_index */

int
set_face_red(p,datastr) /* set "rwb_flag" of faces */
struct p_data *p;
char *datastr;
{
	char next[256];
	int fn,ind;
	char *nextptr;

	nextptr=datastr;
	while (grab_next(&nextptr,next) && sscanf(next,"%d",&fn)==1
	   && fn>0 && fn<=p->facecount 
	   && grab_next(&nextptr,next) && sscanf(next,"%d",&ind)==1)
		p->faces[fn].rwb_flag=ind;
	return 1;
} /* set_face_red */

int
set_face_rnext(p,datastr) /* set's "next_face" index of faces */
struct p_data *p;
char *datastr;
{
	char next[256];
	int f1,f2;
	char *nextptr;

	nextptr=datastr;
	while (grab_next(&nextptr,next) && sscanf(next,"%d",&f1)==1
	   && f1>=0 && f1<=p->facecount 
	   && grab_next(&nextptr,next) && sscanf(next,"%d",&f2)==1
	   && f2>0 && f2<=p->facecount)
		p->faces[f1].next_red=f2;
	return 1;
} /* set_face_rnext */

float
curv_aim_error(p) /* find error in curvatures compared to their targets;
disregard vertices which are free (aim<0). */
struct p_data *p;
{
	int i;
	float accum=0,err;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	for (i=1;i<=p->nodecount;i++) if (pR_ptr[i].aim>=0)
	 {
		err=pR_ptr[i].curv-pR_ptr[i].aim;
		accum += fabs(err);
	 }
	return (accum);
} /* curv_aim_error */

void
fillcurves(p) /* fill in curvatures, pack p */
struct p_data *p;
{
	int i,flag;
	struct R_data *pR_ptr;
	extern void h_anglesum_overlap(),e_anglesum_overlap();

	pR_ptr=p->packR_ptr;
	if (p->hes<0)
	   for (i=1;i<=p->nodecount;i++)
		h_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
	else if (p->hes>0)
	   for (i=1;i<=p->nodecount;i++)
		s_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
	else
	   for (i=1;i<=p->nodecount;i++)
		e_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
} /* fillcurves */

void
set_aim_current(p,flag) /* put current curvatures into aim. If flag,
put neg aim in for bdry, since they are considered free. */
int flag;
struct p_data *p;
{
	int i;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	for (i=1;i<=p->nodecount;i++)
	 {
		pR_ptr[i].aim=pR_ptr[i].curv;
		if (flag && pK_ptr[i].bdry_flag) pR_ptr[i].aim=-1;
	 }
	return;
} /* set_aim_current */

void
set_aim_default(p) /* put 2pi aim in for interior and -1 for bdry.*/
struct p_data *p;
{
	int i;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	for (i=1;i<=p->nodecount;i++)
	 {
		if (pK_ptr[i].bdry_flag) pR_ptr[i].aim=-1;
		else pR_ptr[i].aim=2.0*M_PI;
	 }
} /* set_aim_default */

geom_to_e_pack_convert(p) /* convert p to eucl data; flag means
don't change label. */
struct p_data *p;
{
	int i;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	if (p->hes==0) return;
	else if (p->hes<0) 
	 {
		for (i=1;i<=p->nodecount;i++)
	 		h_to_e_data(pR_ptr[i].center,pR_ptr[i].rad,
				&pR_ptr[i].center,&pR_ptr[i].rad);
	 }
	else if (p->hes>0)
	 {
		for (i=1;i<=p->nodecount;i++)
	 		s_to_e_data(pR_ptr[i].center,pR_ptr[i].rad,
				&pR_ptr[i].center,&pR_ptr[i].rad);
	 }
} /* geom_to_e_pack_convert */

geom_to_h_pack_convert(p) /* takes eucl pack (with eucl centers and radii)
and squeezes it to disc with alpha vertex at origin, then replaces
eucl centers and radii by hyp centers and s-radii */
struct p_data *p;
{
	int i;
	float m,mx;
	complex cent;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	if (p->hes<0) return;
	else if (p->hes==0) 
	 {
		cent=pR_ptr[p->alpha].center;
		mx=pR_ptr[p->alpha].rad;
		for (i=1;i<=p->nodecount;i++) /* translate and
			determine scaling factor */
		 {
			pR_ptr[i].center=csub(pR_ptr[i].center,cent);
			m=cAbs(pR_ptr[i].center);
			if ((m+pR_ptr[i].rad)>mx) mx=m+pR_ptr[i].rad;
		 }
		mx *=1.05; /* shrink slightly from unit circle */
		for (i=1;i<=p->nodecount;i++)
		 {
			pR_ptr[i].center.re /= mx;
			pR_ptr[i].center.im /= mx;
			pR_ptr[i].rad /= mx;
			e_to_h_data(pR_ptr[i].center,
			   pR_ptr[i].rad,&pR_ptr[i].center,&pR_ptr[i].rad);
		 }
	 }
	else if (p->hes>okerr) /* pass sph through eucl */
	 {
		geom_to_e_pack_convert(p);
		p->hes=0;
		geom_to_h_pack_convert(p);
	 }
} /* geom_to_h_pack_convert */

geom_to_s_pack_convert(p) /* takes eucl pack to sphere, with alpha 
vertex at south pole.  */
struct p_data *p;
{
	int i;
	float rad;
	complex ctr;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	if (p->hes>0) return;
	for (i=1;i<=p->nodecount;i++)
	 {
		if (p->hes<0) h_to_e_data(pR_ptr[i].center,
			pR_ptr[i].rad,&ctr,&rad);
		else {ctr=pR_ptr[i].center;rad=pR_ptr[i].rad;}
		e_to_s_data(ctr,rad,&pR_ptr[i].center,&pR_ptr[i].rad);
	 }
} /* geom_to_s_pack_convert */

int
spiral(pnum,datastr) /* compute radii of spiral pack, parameters a,b>0.
Unit circle at 0, tang circle, rad a, on x-axis, circle, rad b, in 
first quad. Rule: product of radii at ends of edge equals product of radii
for vertices on two sides of that edge. */
int pnum;
char *datastr;
{
	int i,j,m,k0,k1,k2,count,flag=0,nodecount,k,alpha,l;
	complex z1,z2,cent;
	float r0,r1,r2,r3,scale,Maxx,minx,Maxy,miny,x,y,r,a,b,factor;
	struct p_data *p;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	p=&packdata[pnum];
	if (!p->status || sscanf(datastr,"%lf%lf",&a,&b)!=2 ||
		a<=0 || b<=0)
	 {
		strcpy(msgbuf,"Data form: a b. Try again.");
		emsg();
		return 0;
	 }
	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	nodecount=p->nodecount;
	for (i=1;i<=nodecount;i++) 
	 {
		if (!pK_ptr[i].bdry_flag && pK_ptr[i].num!=6) flag++;
		pK_ptr[i].plot_flag=0; 
	 }
	if (flag)
	 {
		strcpy(msgbuf,"Pack not hexagonal.");
		emsg();
		return 0;
	 }
	alpha=p->alpha;
	pR_ptr[alpha].center.re=0.0;pR_ptr[alpha].center.im=0.0;
	pR_ptr[alpha].rad=1.0;pK_ptr[alpha].plot_flag=1;
	k=pK_ptr[alpha].flower[0];
	l=pK_ptr[alpha].flower[1];
	pR_ptr[k].center.re=1.0+a;pR_ptr[k].center.im=0;
	pR_ptr[k].rad=a;
	pK_ptr[k].plot_flag=1;
	z1=pR_ptr[alpha].center;z2=pR_ptr[k].center;
	e_compcenter(z1,z2,&pR_ptr[l].center,1.0,a,&b,1.0,1.0,1.0);
 	pR_ptr[l].rad=b;
	pK_ptr[l].plot_flag=1;
	
	count=3;
	j=0;
	while (count<nodecount)
	 {
		flag=0;
		do
		 {
j++;
if (j>nodecount) j=1;
if (!pK_ptr[j].plot_flag) /* if not plotted, see if
	there are appropriate neighbors */
 {
	k=0;
	do
	 {
		k1=pK_ptr[j].flower[k];
		k2=pK_ptr[j].flower[k+1];
		if (pK_ptr[k1].plot_flag && pK_ptr[k2].plot_flag)
		 {
			m=0;
			while (m<=pK_ptr[k2].num 
				&& pK_ptr[k2].flower[m]!=j) m++;
			if (m==(pK_ptr[k2].num -1)
				&& !pK_ptr[k2].bdry_flag
				&& pK_ptr[pK_ptr[k2].flower[0]].plot_flag)
			 {k0=pK_ptr[k2].flower[1];flag=1;}
			else if (m<(pK_ptr[k2].num -1)
				&& pK_ptr[pK_ptr[k2].flower[m+2]].plot_flag)
			 {k0=pK_ptr[k2].flower[m+2];flag=1;}
			if (flag)
			 {		
				z1=pR_ptr[k1].center;r1=pR_ptr[k1].rad;
				z2=pR_ptr[k2].center,r2=pR_ptr[k2].rad;
				r0=pR_ptr[k0].rad;
				r3=r1*r2/r0;
				e_compcenter(z1,z2,&pR_ptr[j].center,r1,r2,&r3,
					1.0,1.0,1.0);
				pK_ptr[j].plot_flag=1;
				pR_ptr[j].rad=r3;
			 }
			else k++;
		 } /* end of if */
		else k++;
	 } /* end of do loop */
	while (!flag && k<=(pK_ptr[j].num-1) );
 } /* end of if */
		 } /* end of do */
		while (!flag);
		count++;
	 } /* end of while */
/* should have all centers and radii now */
	if (p->hes<0)
	 {
		Maxx=1;minx=-1;Maxy=1;miny=-1;
		for (i=1;i<=nodecount;i++) 
		 {
			x=pR_ptr[i].center.re;y=pR_ptr[i].center.im;
			r=pR_ptr[i].rad;
			Maxx=(x+r>Maxx) ? x+r : Maxx;
			minx=(x-r<minx) ? x-r : minx;
			Maxy=(y+r>Maxy) ? y+r : Maxy;
			miny=(y-r<miny) ? y-r : miny;
		 }
		z1.re=(Maxx+minx)*0.5;z1.im=(Maxy+miny)*0.5;
		scale= ((Maxx-minx>Maxy-miny) ? Maxx-minx : Maxy-miny)*0.5;
		factor=(0.9)/(scale*1.4142136);
		for (i=1;i<=nodecount;i++) 
		 {
			cent.re=(pR_ptr[i].center.re-z1.re)*factor;
			cent.im=(pR_ptr[i].center.im-z1.im)*factor;
			r=pR_ptr[i].rad*factor;
	 		e_to_h_data(cent,r,&pR_ptr[i].center,&pR_ptr[i].rad);
		 }
	 }
	free_overlaps(p); /* outdated */
	return 1;
} /* spiral */

comp_center_face(p,f,flag) /* compute and store center/rad of last circle of
face f in pack p; flag is optional index, else use index_flag of f_data. */
int f,flag;
struct p_data *p;
{
	int j,k,v,indx;
	float o1,o2,o3;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	if (f<1 || f> p->facecount) return;
	if (flag<0 || flag>2) indx=p->faces[f].index_flag;
	else indx=flag;
	j=p->faces[f].vert[indx];
	k=p->faces[f].vert[(indx+1) % 3];
	v=p->faces[f].vert[(indx+2) % 3];
	if (p->overlap_status)
	 {
		o1=pK_ptr[k].overlaps[nghb(p,k,v)];
		o2=pK_ptr[v].overlaps[nghb(p,v,j)];
		o3=pK_ptr[j].overlaps[nghb(p,j,k)];
	 }
	else o1=o2=o3=1.0;
	any_compcenter(p->hes,pR_ptr[j].center,pR_ptr[k].center,
		&pR_ptr[v].center,pR_ptr[j].rad,pR_ptr[k].rad,
		&pR_ptr[v].rad,o1,o2,o3);
} /* comp_center_face */

float
comp_single_angle(p,v,u,w,flag) /* compute angle at vert v. flag
indicated incompatibility. */
struct p_data *p;
int v,u,w,*flag;
{
	float r,r1,r2;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	r=pR_ptr[v].rad;
	r1=pR_ptr[u].rad;
	r2=pR_ptr[w].rad;
	if (p->hes<-okerr) /* hyperbolic */
	 {
		if (r<=0) return 0.0;
		if (!p->overlap_status) 
			return (acos(h_comp_cos(r,r1,r2)));
		else return (acos(h_cos_overlap(r,r1,r2,
			pK_ptr[u].overlaps[nghb(p,u,w)],
			pK_ptr[w].overlaps[nghb(p,w,v)],
			pK_ptr[v].overlaps[nghb(p,v,u)],flag)));
	 }
	if (p->hes<okerr) /* euclidean */
	 {
		if (!p->overlap_status)
			return (acos(1-2*r1*r2/((r+r1)*(r+r2))));
		else return (acos(e_cos_overlap(r,r1,r2,
			pK_ptr[u].overlaps[nghb(p,u,w)],
			pK_ptr[w].overlaps[nghb(p,w,v)],
			pK_ptr[v].overlaps[nghb(p,v,u)],flag)));
	 }
	else /* spherical - no overlaps yet */
		return(s_comp_cos(r,r1,r2));
} /* comp_single_angle */

int
any_compcenter(hes,z1,z2,z3,r1,r2,r3,o1,o2,o3) /* Find center of third circle
in ordered tangent triple, curvature hes. */ 
int hes;
float r1,r2,*r3,o1,o2,o3;
complex z1,z2,*z3;
{
	if (hes<0) return(h_compcenter(z1,z2,z3,r1,r2,r3,o1,o2,o3));
	if (hes>0) return(s_compcenter(z1,z2,z3,r1,r2,r3,o1,o2,o3));
	return(e_compcenter(z1,z2,z3,r1,r2,r3,o1,o2,o3));
} /* any_compcenter */

int
center_point(p,ctr) /* If hyperbolic, apply a Mobius trans of disc 
	putting ctr at origin. If euclidean, translate. */
struct p_data *p;
complex ctr;
{
	int i;
	complex z1,z2,z3,zz;
	float radius;
	struct R_data *pR_ptr;
	
	pR_ptr=p->packR_ptr;
	if (p->hes<-okerr)
	 {
		if ( (ctr.re*ctr.re+ctr.im*ctr.im)>(1.0-okerr) ) return 0;
		for (i=1;i<=p->nodecount;i++) 
		 {
			radius=(-pR_ptr[i].rad);
			if (radius>0)
			 {
				z1=pR_ptr[i].center;
				z2.re=(1-2*radius)*z1.re;
				z2.im=(1-2*radius)*z1.im;
				z3.re=(1-radius)*z1.re-radius*z1.im;
				z3.im=(1-radius)*z1.im+radius*z1.re; 
				z1=mob_trans(z1,ctr);
				z2=mob_trans(z2,ctr);
				z3=mob_trans(z3,ctr);
				circle_3(z1,z2,z3,&zz,&radius);
				pR_ptr[i].rad=(-radius);
			 }
			pR_ptr[i].center=mob_trans(pR_ptr[i].center,ctr);
			pR_ptr[i].center.re *= (-1.0);
			pR_ptr[i].center.im *= (-1.0);
		 }
		return 1;
	 }
	else if (p->hes< okerr)
	 {
		for (i=1;i<=p->nodecount;i++) 
			pR_ptr[i].center=csub(pR_ptr[i].center,ctr);
		return 1;
	 }
	return 0;
} /* center_point */

reset_Mob() /* sets Mob to identity to avoid errors. */
{
	Mob.a.re=Mob.d.re=1.0;
	Mob.b.re=Mob.c.re=0.0;
	Mob.a.im=Mob.b.im=Mob.c.im=Mob.d.im=0.0;
	Mob.flip=0;
	return;
}

int
NSpole(p,datastr) /* only for spherical packings: centers specified cirs
at north/south poles and (eventually) allows parameter to adjust their
sizes. */
struct p_data *p;
char *datastr;
{
	int S,N;
	float erS,erN,factor=1.0;
	complex ecS,ecN;
	struct R_data *pR_ptr;
	extern Mobius zero_inf();

	pR_ptr=p->packR_ptr;
	if (p->hes<okerr) return 0; 
	if (sscanf(datastr,"%d %d %lf",&S,&N,&factor)<2 || N==S
		|| S<1 || N<1 || S>p->nodecount || N>p->nodecount ||factor<okerr) 
		return 0;
	s_to_e_data(pR_ptr[S].center,pR_ptr[S].rad,&ecS,&erS);
	s_to_e_data(pR_ptr[N].center,pR_ptr[N].rad,&ecN,&erN);
	Mob=zero_inf(ecS,ecN,erS,erN,factor);
	sprintf(buf,"Mobius -p%d",pack_num(p));
	handle_cmd(buf);
	reset_Mob();
	return 1;
} /* NSpole */

rotate(p,ang) /* rotate pack p by angle ang (radians). */
struct p_data *p;
float ang;
{
	int i;
	struct R_data *pR_ptr;
	
	pR_ptr=p->packR_ptr;
	for (i=1;i<=p->nodecount;i++)
		pR_ptr[i].center=mob_rotate(pR_ptr[i].center,ang);
} /* rotate */

int
place_face(p,face,indx) /* compute cents of face, indx at origin,
next vert in standard relation */
struct p_data *p;
int face,indx;
{
	int a,k;
	float r,r2,ovlp,erad,s,ss1,ss2,s1,s2;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	if (face>p->facecount || face<1 || indx <0 || indx >2) return 0;
	a=p->faces[face].vert[indx];
	k=p->faces[face].vert[(indx+1) % 3];
	if (p->overlap_status)
		ovlp=p->packK_ptr[a].overlaps[nghb(p,a,k)];
	else ovlp=1.0;
	if (p->hes<0) /* hyp case */
	 {
		s1=pR_ptr[a].rad;
		s2=pR_ptr[k].rad;
		if (s1<=0)
		 {
			s1=pR_ptr[a].rad=.01;
			strcpy(msgbuf,"Circle at origin had infinite radius; radius reset.");
			emsg();
		 }
		pR_ptr[a].center.re=0; pR_ptr[a].center.im=0;
		if (s2<=0) /* if next one is infinite radius */
		 {
			pR_ptr[k].center.re=1;pR_ptr[k].center.im=0;
			erad=(1-s1)/(1+s1);
			pR_ptr[k].rad=(-1)*(1-erad*erad)/(2.0+2.0*erad*ovlp);
		 }
		else
		 {
			ss1=s1*s1;ss2=s2*s2;
			s=exp( acosh((1/(4.0*s1*s2))*((1+ss1)*(1+ss2)
				+(1-ss1)*(1-ss2)*ovlp) ) );
			pR_ptr[k].center.re=(s-1)/(s+1);
			pR_ptr[k].center.im=0.0;
		 }
	 }
	else if (p->hes>0) /* sphere case */
	 {
			/* alpha at south pole */
		pR_ptr[a].center.re=0;pR_ptr[a].center.im=M_PI; 
			/* next out pos x-axis */
		pR_ptr[k].center.re=0.0;
		pR_ptr[k].center.im=M_PI-pR_ptr[a].rad-pR_ptr[k].rad;
	 }
	else /* eucl case */
	 {
			/* alpha at origin */
		r=pR_ptr[a].rad;
		pR_ptr[a].center.re=0;pR_ptr[a].center.im=0; 
			/* next on x-axis */
		r2=pR_ptr[k].rad;
		pR_ptr[k].center.re=sqrt(r*r+r2*r2+2*r*r2*ovlp);
		pR_ptr[k].center.im=0; 
	 }
	comp_center_face(p,face,indx);
	return 1;
} /* place_face */
		
int
comp_pack_centers(p) /* find centers based on current radii.
Start with alpha vert; normalize gamma vert on y>0 axis; then use
order of faces. Only compute each circle's center once. */
struct p_data *p;
{
	int nf,i,vert,count=0;
	complex c,d;

	for (i=1;i<=p->nodecount;i++) p->packK_ptr[i].plot_flag=0;
	nf=p->first_face;
	place_face(p,nf,p->faces[nf].index_flag);
	for (i=0;i<3;i++) p->packK_ptr[p->faces[nf].vert[i]].plot_flag=1;
	while ( (nf=p->faces[nf].next_face)!=p->first_face 
	   && nf>0 && nf<=p->facecount && count<2*p->facecount)
	 {
		vert=p->faces[nf].vert[(p->faces[nf].index_flag +2) % 3];
		if (!p->packK_ptr[vert].plot_flag) /* not yet computed */
			comp_center_face(p,nf,-1);
		p->packK_ptr[vert].plot_flag=1;
		count++;
	 }
/* normalize */
	if (nf<=0 || nf>p->facecount || count>=p->facecount) 
		return 0; /* error in data */
	c=p->packR_ptr[p->alpha].center;
	d=p->packR_ptr[p->gamma].center;
	return (norm_any_pack(p,c,d));
} /* comp_pack_centers */

int
norm_any_pack(p,a,g) /* depending on geom */
struct p_data *p;
complex a,g;
{
	if (p->hes<(-okerr)) return h_norm_pack(p,a,g);
	if (p->hes< okerr) return e_norm_pack(p,a,g);
	else return 0; /* haven't yet done sphere case */
} /* norm_any_pack */

int
e_norm_pack(p,a,g) /* normalizes eucl data of pack p by 
putting point a at origin and g on pos y-axis */
struct p_data *p;
complex a,g;
{
	int i;
	complex I,z,w,y;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	w=csub(g,a);
	I.re=0; I.im=cAbs(w);
	z=cdiv(I,w);
	for (i=1;i<=p->nodecount;i++)
	 {
		y=csub(pR_ptr[i].center,a);
		pR_ptr[i].center=cmult(z,y);
	 }
	return 1;
} /* e_norm_pack */

int
eucl_scale(p,factor) /* eucl scale by given factor. In hyp case, 
return 0 if some circles forced outside unit disc. In sph case, apply
z->t*z Mobius. */
struct p_data *p;
float factor;
{
	int flag=0,i;
	float e_rad;
	complex e_center;
	Mobius oldMob;
	struct R_data *pR_ptr;
	
	pR_ptr=p->packR_ptr;
	if (p->hes<-okerr)
	 {
		for (i=1;i<=p->nodecount;i++)
		 {
			h_to_e_data(pR_ptr[i].center,pR_ptr[i].rad,
				&e_center,&e_rad);
			e_center.re*=factor;
			e_center.im*=factor;
			e_rad*=factor;
			if (!e_to_h_data(e_center,e_rad,&pR_ptr[i].center,
				&pR_ptr[i].rad)) flag++;
		 }
		fillcurves(p);
	 }
	else 
	 {
		oldMob=Mob; /* save old */
		Mob.flip=0;
		Mob.a.re=factor;
		Mob.a.im=Mob.b.re=Mob.b.im=Mob.c.re=Mob.c.im=Mob.d.im=0.0;
		Mob.d.re=1.0;
		sprintf(buf,"Mobius -p%d a",pack_num(p));
		handle_cmd(buf);
		Mob=oldMob;
	 }
 	return flag;
} /* eucl_scale */

int
set_radii(p,rad,datastr) /* set radii  */
char *datastr;
float rad;
struct p_data *p;
{
	int count=0,hits;
	float orad;
	char *endptr;
	extern struct Vertlist *node_link_parse();
	struct Vertlist *vertlist,*trace;
	
	orad=rad;
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		if (p->hes<0)
		 {
			if (rad>0) rad=exp(-rad); /* convert rad */
			do {
			  count += h_changerad(p,trace->v,rad);
			  trace=trace->next;
		  	 } while (trace!=NULL);
		 }
		else if (p->hes==0)
		 {
		  	do {
			  count += e_changerad(p,trace->v,rad);
			  trace=trace->next;
		 	 } while (trace!=NULL);
		 }
		else if (p->hes>0)
		 {
		  	do {
			  p->packR_ptr[trace->v].rad=rad;
			  trace=trace->next;
			  count++;
		 	 } while (trace!=NULL);
		 }
		vert_free(&vertlist);
	 }
	if (count)
	 {
		fillcurves(p);
		sprintf(msgbuf,"Set %d radii of pack %d to %lf.",
			count,pack_num(p),orad);
		msg();
		return count;
	 }
	strcpy(msgbuf,"No radii adjusted.");
	emsg();
	return 0;
} /* set_radii */

int
crement_radii(p,factor,datastr) /* inc/dec radii in p from string. 
For s_radii, replace s by s=exp(factor*(log(s)))=s^(factor). 
For eucl radii, just multiply.  */
char *datastr;
struct p_data *p;
float factor;
{ 
	char *endptr;
	int count=0,vert,hits;
	float newrad;
	extern struct Vertlist *node_link_parse();
	struct Vertlist *vertlist,*trace;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
	   trace=vertlist;
	   if (p->hes<0)
	    {
		do {
			vert=trace->v;
		  	if (pR_ptr[vert].rad<=0 && factor < 1.0) 
			 {
				newrad=.1;
				count += h_changerad(p,vert,newrad);
			 }
			     /* make it finite if decr horocycle */
		   	else if (pR_ptr[vert].rad > 0)
			 {
				newrad=exp(factor*log(pR_ptr[vert].rad));
				count += h_changerad(p,vert,newrad);
			 }
			trace=trace->next;
		 } while (trace!=NULL);
	    }
	   else if (p->hes>=0)
	    {
		do {
			vert=trace->v;
		  	newrad=factor*pR_ptr[vert].rad;
			if (p->hes>okerr)
				count += s_changerad(p,vert,newrad);
		  	else
				count += e_changerad(p,vert,newrad);
			trace=trace->next;
		 } while (trace!=NULL);
	    }
	   vert_free(&vertlist);
	 }
	if (count)
	 {
		fillcurves(p);
		return count;
	 }
	strcpy(msgbuf,"No radii adjusted.");
	emsg();
	return 0;
 } /* crement_radii */

int
randize(p,datastr) /* set random radii (default) or overlaps 
(or random changes) */
char *datastr;
struct p_data *p;
{ 
	int mode=0,count=0,vert,hits,flag=0,vv,ww,k;
	float rn,rand_value,lowr,highr,tmprad;
	char *nextptr,*lastptr,*endptr,next[256];
	struct Vertlist *vertlist=NULL,*trace;
	struct Edgelist *edgelist=NULL,*etrace;
	struct R_data *pR_ptr;
	extern float gwrand();
	extern struct Vertlist *node_link_parse();
	extern struct Edgelist *node_pair_link();

	pR_ptr=p->packR_ptr;
	lowr=10.0*okerr;highr=10; /* default radii range */
	nextptr=lastptr=datastr;
	grab_next(&nextptr,next);
	if (!strncmp(next,"-o",2)) /* overlaps, not radii */
	 {
		flag=1;
		lowr=0.0; /* default overlap range */
		grab_next(&nextptr,next);
	 }
	do
	 {
		if (next[0]=='-') 
		 {
			if (next[1]=='r') /* range */
			 {
				if (sscanf(nextptr,"%lf %lf",&lowr,&highr)!=2
					|| highr<lowr) 
				 {
					if (count) fillcurves(p);
					return count;
				 }
				grab_next(&nextptr,next);
				grab_next(&nextptr,next); /* move past data */
				if (mode==1)
				 {
					if (lowr<okerr) lowr=okerr;
					if (highr<lowr) highr=2*lowr;
				 }
			 }
			else if (next[1]=='f'
			   && (!flag || p->overlap_status)) 
					/* increment (treat as factor) */
			 {
				mode=1;
				if (lowr<okerr) lowr=okerr;
				if (highr<lowr) highr=2*lowr;
			 }
			else if (!flag && next[1]=='j')
					/* jiggle radii */
			 {
				mode=1;
				lowr=1-10*okerr;
				highr=1+10*okerr;
			 }
			lastptr=nextptr;
		 }
		else if (!flag) /* adjust radii */
		 {
			if ((vertlist=node_link_parse(p,
				lastptr,&endptr,&hits))==NULL)
			 {
				if (count) fillcurves(p);
				return count;
			 } 
					/* default = no action */
			nextptr=lastptr=endptr;
			trace=vertlist;
do
 {
	vert=trace->v;
	rn=gwrand(); /* pseudo-random in (0,1) */
	rand_value=lowr+(highr-lowr)*rn;
	if (mode==1) /* multiply by factor */
	 {
		if (p->hes>0) /* sphere */
		 {
			pR_ptr[vert].rad *= rand_value;
			if (pR_ptr[vert].rad>=M_PI)
				pR_ptr[vert].rad=M_PI-okerr;
			count++;
		 }
		if (p->hes<0) /* hyperbolic */
		 {
			if (pR_ptr[vert].rad < 0) /* infinite radius */
			 {
				if (rand_value < 1.0) /* decrease to .5 */
				   pR_ptr[vert].rad=.5;
			 }
			else
			 {
				tmprad=exp(log(pR_ptr[vert].rad)*rand_value); 
				if (tmprad < .0001) pR_ptr[vert].rad = -.1;
					/* infinite */
				else pR_ptr[vert].rad = tmprad;			
					/* s-radii */
			 }
			count++;
		 }
		else /* eucl */
		 {
			pR_ptr[vert].rad *= rand_value;
			count++;
		 }
	 }
	else
	 {
		if (p->hes>0)
		 {
			pR_ptr[vert].rad = rand_value;
			if (pR_ptr[vert].rad>=M_PI)
				pR_ptr[vert].rad=M_PI-okerr;
			count++;
		 }
		if (p->hes<0) /* hyperbolic */
		 {
			tmprad= exp((-1.0)*rand_value);	
			if (tmprad < .0001) pR_ptr[vert].rad = -.1;
				/* infinite */
			else pR_ptr[vert].rad = tmprad;			
			count++;
		 }
		else 
		 {
			pR_ptr[vert].rad = rand_value;
			count++;
		 }
	 }
	trace=trace->next;
 } /* end of inner do */
			while (trace!=NULL);
			vert_free(&vertlist);
			if (count) fillcurves(p);
			return count;
		 }
		else /* adjust overlaps */
		 {
		   if ((edgelist=node_pair_link(p,
			lastptr,&endptr,&hits))==NULL)
		    {
			if (count) fillcurves(p);
			   return count;
		    }   /* default = no action */
		   alloc_overlaps(p);
		   nextptr=lastptr=endptr;
		   etrace=edgelist;			
		   do
		    {
			vv=etrace->v;
			ww=etrace->w;
			rn=gwrand(); /* pseudo-random in (0,1) */
			rand_value=lowr+(highr-lowr)*rn;
			if ((k=nghb(p,vv,ww))>=0)
			 {
			   if (mode==1) /* factor */
			      set_overlap(p,vv,k,
				rand_value*p->packK_ptr[vv].overlaps[k]);
			   else set_overlap(p,vv,k,rand_value);
			 }
			etrace=etrace->next;
		    } /* end of inner do */
		   while (etrace!=NULL);
		   edge_free(&edgelist);
		   if (count) fillcurves(p);
		   return count;
		 }
	 } /* end of do */
	while ((lastptr=nextptr)!=NULL && grab_next(&nextptr,next) );
	if (count) fillcurves(p);
	return count;		
} /* randize */

int
e_changerad(p,i,newr) /* records new eucl radius calculated
elsewhere and makes approp curvature changes */
int i;
float newr;
struct p_data *p;
{
	int k,j,flag;
	float newc;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern void e_anglesum_overlap();

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	if (newr<=0) return 0;
	e_anglesum_overlap(p,i,newr,&newc,&flag);
	pR_ptr[i].rad=newr;
	pR_ptr[i].curv=newc;
	for (k=0;k<=pK_ptr[i].num;k++)
	 {
		j=pK_ptr[i].flower[k];
		e_anglesum_overlap(p,j,pR_ptr[j].rad,&newc,&flag);
		pR_ptr[j].curv=newc;
	 }
	return 1;
} /* e_changerad */

int
h_changerad(p,i,newr) /* just records new rad, makes needed curv
changes, etc. */
int i;
float newr;
struct p_data *p;
{
	int k,j,flag;
	float newc;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern void h_anglesum_overlap();

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	if (newr>=0 && newr<.005) newr=-.1;
		 /* so large, assume infinite */
	h_anglesum_overlap(p,i,newr,&newc,&flag);
	pR_ptr[i].rad=newr;
	pR_ptr[i].curv=newc;
	for (k=0;k<=pK_ptr[i].num;k++)
	 {
		j=pK_ptr[i].flower[k];
		h_anglesum_overlap(p,j,pR_ptr[j].rad,&newc,&flag);
		pR_ptr[j].curv=newc;
	 }
	return 1;
} /* h_changerad */

int
s_changerad(p,i,newr) /* check/record new rad, make curv changes, etc. */
int i;
float newr;
struct p_data *p;
{
	int k,j,l,m,flag;
	float newc;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int s_anglesum_overlap();

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
/* try changing radius by more than direct computation suggests. I
don't understand why this works better? */
	newr=pR_ptr[i].rad+(1.3)*(newr-pR_ptr[i].rad);
	for (k=0;k<=pK_ptr[i].num;k++) /* newr too large? */
	 {
		j=pK_ptr[i].flower[k];
		m=pK_ptr[i].flower[(k+1)%(pK_ptr[i].num)];
		if ((pR_ptr[i].rad+pR_ptr[j].rad+pR_ptr[m].rad)>=M_PI)
			return 0;
	 }
	normalize_star(p,i,newr);  /*also changes pR_ptr[i].rad*/
	s_anglesum_overlap(p,i,newr,&pR_ptr[i].curv,&flag);
	for (k=0;k<=pK_ptr[i].num;k++)
	 {
		j=pK_ptr[i].flower[k];
		s_anglesum_overlap(p,j,pR_ptr[j].rad,&newc,&flag);
		pR_ptr[j].curv=newc;
		for (l=0;l<=pK_ptr[j].num;l++)
			s_anglesum_overlap(p,pK_ptr[j].flower[l],
				 pR_ptr[pK_ptr[j].flower[l]].rad,
				&pR_ptr[pK_ptr[j].flower[l]].curv,&flag);
	 }
	return 1;
} /* s_changerad */

int
normalize_star(p,v,radius) /*  Needs updated anglesums, and does update them before returning. Also changes radius. */
struct p_data *p;
float radius;
int v;
{
	int i,j;
	struct R_data *pR_ptr,*lowR,*highR;
	struct K_data *pK_ptr;
	float target,area,max_r,lowarea,higharea,error,factor=0.15;
	float norm_err=.0001;
	extern float s_star_area();
	extern float max_r_adjustment();
	extern int fill_star_curves();

	pR_ptr=p->packR_ptr;
	pK_ptr=p->packK_ptr;
	fill_star_curves(p,v);
	target=s_star_area(p,v);
	pR_ptr[v].rad=radius;
	
/* set up dup data */
	highR=(struct R_data*)malloc((p->nodecount+2)*sizeof(struct R_data));
	lowR=(struct R_data *)malloc((p->nodecount+2)*sizeof(struct R_data));
	for (i=1;i<=p->nodecount;i++) highR[i]=lowR[i]=pR_ptr[i];

	fill_star_curves(p,v);
	area=s_star_area(p,v);
	lowarea=area;
	higharea=area;
	error=area - target;
	if (fabs(error)> norm_err)
	  {
		  while (lowarea>(target))
		    {
		       lowR[v].rad-=lowR[v].rad*factor;
		       for (i=0;i<=pK_ptr[v].num;i++)
			 {
				j=pK_ptr[v].flower[i];
				if (lowR[j].rad>highR[j].rad)
				 {
					sprintf(msgbuf,"low>high for vert %d.",
						j);
					emsg();
				 }
			     if (pR_ptr[pK_ptr[v].flower[i]].aim>0) 
				lowR[j].rad-=lowR[j].rad * factor;
			 }
			p->packR_ptr=lowR;
			fill_star_curves(p,v);
		        lowarea=s_star_area(p,v);
		     }

		   while (higharea<(target))
		     {
			highR[v].rad+=highR[v].rad*factor;
	        	for (i=0;i<=pK_ptr[v].num;i++)
			  {
			    j=pK_ptr[v].flower[i];	
			     if (pR_ptr[j].aim>0 
				&& (max_r=max_r_adjustment(p,j))>0.0) 
			       {
				max_r= highR[j].rad+max_r;
				highR[j].rad
				  =((1+factor)*highR[j].rad<=max_r)
			 	 ? (1+factor)*highR[j].rad : max_r;
			       }
			  } 
			p->packR_ptr=highR;
			fill_star_curves(p,v);
			higharea=s_star_area(p,v);		     
	    	      }
	    }	
	p->packR_ptr=pR_ptr; /* CAUTION: note, pointer has been fooled with! */

	while (fabs(error)>norm_err)
	  {
		pR_ptr[v].rad=(higharea*lowR[v].rad + lowarea*highR[v].rad)
				/(higharea + lowarea);
	     	for (i=0;i<=pK_ptr[v].num;i++)
		 {
		  j=pK_ptr[v].flower[i];
		  if (pR_ptr[j].aim>=0)
		   pR_ptr[j].rad=(higharea*lowR[j].rad+lowarea*highR[j].rad)
			 	 /(higharea + lowarea);
		  }
		fill_star_curves(p,v);
		area=s_star_area(p,v);
		error= area-target;
		
		if (error>norm_err)
		   {
			highR[v]=pR_ptr[v];
			for (i=0;i<=pK_ptr[v].num;i++)
				highR[pK_ptr[v].flower[i]]
				=pR_ptr[pK_ptr[v].flower[i]];
			higharea=area;
		    }		
		else if (error<(-norm_err))
		    {
			lowR[v]=pR_ptr[v];
			for (i=0;i<=pK_ptr[v].num;i++) 
				lowR[pK_ptr[v].flower[i]]
				=pR_ptr[pK_ptr[v].flower[i]];
			lowarea=area;
		    }
	   }
	free(highR);
	free(lowR);
} /* normalize_star */

float
gwrand() /* Brock Williams random number generator. Gives
pseudo-random number between -1 and 1 */
{
	unsigned int tmp;

	rand_next=rand_next*1103515245 + 12345;
	tmp=(unsigned int)(rand_next/65536) % 32768;
	return (float)(((float)tmp)/32768.0);
}

int
set_aims(p,datastr) /* Set the intended target angle sums. */ 
char *datastr;
struct p_data *p;
{ 
	int count=0,mode=0,k,hits;
	float inc,ang;
	char *endptr,next[256];
	extern struct Vertlist *node_link_parse();
	struct Vertlist *vertlist,*trace;
	struct R_data *pR_ptr;
	struct K_data *pK_ptr;

	pR_ptr=p->packR_ptr;pK_ptr=p->packK_ptr;
	grab_next(&datastr,next);
	if (next[0]=='-' && next[1]=='d') mode=0; /* default mode */
	else if (next[0]=='-' && next[1]=='c') mode=1;/* current mode */
	else if (next[0]=='-' && next[1]=='%') /* modify mode */
	 {
		mode=2;
		grab_next(&datastr,next);
		if (sscanf(next,"%lf",&inc)!=1) return 0; 
			/* no increment was given */
	 }
	else mode=3;
	if (mode==3 && sscanf(next,"%lf",&ang)!=1) return 0;
			/* no angle given */
	if ((vertlist=node_link_parse(p,datastr,&endptr,&hits))==NULL)
	 {
		if (mode==3 || mode==2) return 0; /* no verts given */
		if (mode==0)
		 { 
			set_aim_default(p);
			sprintf(msgbuf,"set `aims' of pack %d to default.",
				pack_num(p)); 
			msg();
			return (p->nodecount);
		 }
		if (mode==1)
		 {
			set_aim_current(p,0);
			sprintf(msgbuf,"Set `aims' of pack %d to current values.",
				pack_num(p)); 
			msg();
			return (p->nodecount);
		 }
	 }
	if (mode==3 && ang==0.0 && p->hes >= 0) return 0;
			/* inappropriate angle */
	trace=vertlist;
	do {
		k=trace->v;
		if (mode==0) 
		 {
			if (!pK_ptr[k].bdry_flag) pR_ptr[k].aim=2.0*M_PI;
			else pR_ptr[k].aim=-1;
		 }
		else if (mode==1) pR_ptr[k].aim=pR_ptr[k].curv;
		else if (mode==2) pR_ptr[k].aim=pR_ptr[k].aim+inc*M_PI;
		else if (mode==3)
		 {
			if (ang!=0.0 || ((p->hes < 0) && pK_ptr[k].bdry_flag))
				pR_ptr[k].aim=ang*M_PI;
			else count--;
		 }
		count++;
		trace=trace->next;
	 } while (trace!=NULL);
	vert_free(&vertlist);
	if (count)
	 {
		if (mode==0) 
		   sprintf(msgbuf,"Set %d `aims' of pack %d to default.",
			count,pack_num(p));
		else if (mode==1)
		   sprintf(msgbuf,"Set %d `aims' of pack %d to their current angle sums.",
			count,pack_num(p));
		else if (mode==2)
		   sprintf(msgbuf,"Changed %d `aims' of pack %d by %lf",
			count,pack_num(p),inc);
		else if (mode==3) 
			sprintf(msgbuf,"Set %d `aims' of pack %d to %lf.",
			   count,pack_num(p),ang);
		msg();
		return 1;
	 }
	strcpy(msgbuf,"No aims adjusted.");
	emsg();
	return 0;
} /* set_aims */

float
seg_dist(pt,p1,p2) /* dist from pt to eucl segment [p1,p2] */
complex pt,p1,p2;
{
	float x,y,x1,y1,x2,y2,m;
	complex z,w;

	x=pt.re;y=pt.im;
	x1=p1.re;y1=p1.im;
	x2=p2.re;y2=p2.im;
	if (x1!=x2 && y1!=y2)
	 {
		m=(y2-y1)/(x2-x1);
		if (x2<x1)
		 {z=p2; p2=p1; p1=z;x1=p1.re;y1=p1.im;x2=p2.re;y2=p2.im;}
		z.re=(x+m*(y-y1+m*x1))/(m*m+1);
		z.im=m*(z.re-x1)+y1;
		if (z.re<x1)
		 {
			w=csub(pt,p1);
			return (cAbs(w));
		 }
		if (z.re>x2)
		 {
			w=csub(pt,p2);
			return (cAbs(w));
		 }
		w=csub(z,pt);
		return (cAbs(w));
	 }
	if (x1==x2)
	 {
		if (y1==y2)
		 {
			w=csub(pt,p1);
			return (cAbs(w));
		 }
		if (y2<y1) 
		 {z=p2; p2=p1; p1=z;x1=p1.re;y1=p1.im;x2=p2.re;y2=p2.im;}
		if (y<y1)
		 {
			w=csub(pt,p1);
			return (cAbs(w));
		 }
		if (y>y2)
		 {
			w=csub(pt,p2);
			return (cAbs(w));
		 }
		return (fabs(x-x1));
	 }
	if (x2<x1)
	 {z=p2; p2=p1; p1=z;x1=p1.re;y1=p1.im;x2=p2.re;y2=p2.im;}
	if (x<x1)
	 {
		w=csub(pt,p1);
		return (cAbs(w));
	 }
	if (x>x2)
	 {
		w=csub(pt,p2);
		return (cAbs(w));
	 }
	return (fabs(y-y1));
} /* seg_dist */

int
norm_scale(p,datastr) /* to normalize eucl p so vert v has eucl center on unit
circle. */
struct p_data *p;
char *datastr;
{
	int v;
	float factor;
	char buff[100];
	float ctr;
	extern float cAbs();

	if (sscanf(datastr,"%d",&v)!=1|| !p->status
	   || v<1 || v>p->nodecount 
	   || p->hes>okerr || p->hes<(-okerr)) return 0;
	ctr=cAbs(p->packR_ptr[v].center);
	if (ctr < okerr) return 0;
	factor=1.0/ctr;
	sprintf(buff,"scale -p%d %lf",pack_num(p),factor);
	if (handle_cmd(buff)) return 1;
	else return 0;
} /* special10 */

int
apply_Mobius(p,datastr,flag) /* apply current Mobius, Mob (flag==1), 
or inverse (flag==-1) */
int flag;
struct p_data *p;
char *datastr;
{
	int count=0,v,hits;
	float rad;
	complex z1,z2,z3,tmp1,tmp2,tmp3;
	char *endptr,deflt[1];
	extern struct Vertlist *node_link_parse();
	struct Vertlist *vertlist,*trace;
	struct R_data *pR_ptr;
	Mobius C,CC;

	pR_ptr=p->packR_ptr;
/* euclidean */
   if (p->hes<okerr && p->hes>-okerr) 
    {
	stripsp(datastr);
	if (strlen(datastr)==0) {deflt[0]='a';datastr=deflt;}	
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
			v=trace->v;
			rad=pR_ptr[v].rad;
			z1=pR_ptr[v].center;z1.re += rad;
			z2=pR_ptr[v].center;z2.im += rad;
			z3=pR_ptr[v].center;z3.re -= rad;
			tmp1=mobius(Mob,z1,flag);
			tmp2=mobius(Mob,z2,flag);
			tmp3=mobius(Mob,z3,flag);
			count += circle_3(tmp1,tmp2,tmp3,
				&(pR_ptr[v].center),&(pR_ptr[v].rad));
			trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&vertlist);
	 }
	return count;
    }
/* spherical */
   else if (p->hes>okerr) 
    {
	stripsp(datastr);
	if (strlen(datastr)==0) {deflt[0]='a';datastr=deflt;}	
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
		   v=trace->v;
		   s_to_matrix_data(pR_ptr[v].center,pR_ptr[v].rad,&C);
		   CC=apply_mobius(Mob,C,1);
		   matrix_to_s_data(CC,&(pR_ptr[v].center),&(pR_ptr[v].rad));
		   trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&vertlist);
	 }
	return count;
    }
/* hyperbolic */
   else
    {
	sprintf(msgbuf,"Mobius not yet applicable to hyp packings.");
	emsg();
	return 0;
    }
} /* apply_Mobius */					

marking(p,datastr) /* set mark on circles or faces */
struct p_data *p;
char *datastr;
{
	int count=0,i,vert,flg,hits;
	char *nextpoint,*endptr,next[64];
	extern struct Vertlist *node_link_parse(),*face_link_parse();
	struct Vertlist *vertlist,*trace,*facelist;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	nextpoint=datastr;
	if (!grab_next(&nextpoint,next) || next[0]!='-' 
		|| !p->status) return 1;
	do
	 {
if (next[0]!='-') return count;
else if (next[1]=='w') /* wipe out all marks */
 {
	for (i=1;i<=p->nodecount;i++) pK_ptr[i].mark=0;
	for (i=1;i<=p->facecount;i++) p->faces[i].mark=0;
	count++;
 }
else if (next[1]=='c') /* mark circles */
 {
	flg= (next[2]=='w'); /* 1==>wipe out */
	if ((vertlist=node_link_parse(p,nextpoint,
		&endptr,&hits))!=NULL)
	 {
		nextpoint=endptr;
		trace=vertlist;
		do
		 {
			vert=trace->v;
			if (flg) pK_ptr[vert].mark=0;
			else pK_ptr[vert].mark++;
			count++;
			trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&vertlist);
	 }
	else if (flg)
	 {
		for (i=1;i<=p->nodecount;i++) pK_ptr[i].mark=0;
		count++;
	 }
 }
else if (next[1]=='f') /* mark faces */
 {
	flg= (next[2]=='w'); /* 1==>wipe out */
	if ((facelist=face_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
	 {
		nextpoint=endptr;
		trace=facelist;
		do
		 {
			vert=trace->v;
			if (flg) p->faces[vert].mark=0;
			else p->faces[vert].mark++;
			count++;
			trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&facelist);
	 }
	else if (flg)
	 {
		for (i=1;i<=p->facecount;i++) p->faces[i].mark=0;
		count++;
	 }
 }
	 } /* end of do */
	while (nextpoint!=NULL && grab_next(&nextpoint,next) );
	return count;		
} /* marking */

int
set_default_overlaps(p) /* set to 1.0. */
struct p_data *p;
{
	int v,j;

	if (!p->overlap_status) return 0;
	for (v=1;v<=p->nodecount;v++)
	for (j=0;j<=(p->packK_ptr[v].num);j++)
		if (v<p->packK_ptr[v].flower[j]) set_overlap(p,v,j,1.0);
	return 1;
} /* set_default_overlaps */

float
comp_inv_dist(p,v,w) /* returns inv dist */
struct p_data *p;
int v,w;
{
	float erad1,erad2;
	complex ectr1,ectr2;
	extern float inv_dist();

	if (v<1 || w<1 || v>p->nodecount || w>p->nodecount
	   || !p->overlap_status || nghb(p,v,w)< 0)
		return 1.0; /* tangency, default */
	if (p->hes<-okerr) /* hyperbolic */
	 {
		h_to_e_data(p->packR_ptr[v].center,
			p->packR_ptr[v].rad,&ectr1,&erad1);
		h_to_e_data(p->packR_ptr[w].center,
			p->packR_ptr[w].rad,&ectr2,&erad2);
	 }
	else if (p->hes>=okerr) /* spherical */
	 {
		s_to_e_data(p->packR_ptr[v].center,
			p->packR_ptr[v].rad,&ectr1,&erad1);
		s_to_e_data(p->packR_ptr[w].center,
			p->packR_ptr[w].rad,&ectr2,&erad2);
	 }
	else
	 {
		ectr1=p->packR_ptr[v].center;
		erad1=p->packR_ptr[v].rad;
		ectr2=p->packR_ptr[w].center;
		erad2=p->packR_ptr[w].rad;
	 }
	return (inv_dist(ectr1,erad1,ectr2,erad2));
		/* note: inv_dist routine not very robust */
} /* comp_inv_dist */


/* ======================== 
routines for random walk 
=========================== */

/* only for tangency packings */

/* ========= euclidean functions =========== */

float
G(x,y,z) /* angle at x */
float x,y,z;
{
	float a;

	a=x*(x+y+z);
	return ( acos((a-y*z)/(a+y*z)) );
} /* G */

float
G1(x,y,z) /* angle ftn partial. First, convert arguments to s-radii. */
float x,y,z;
{
	float s;

	s=x+y+z;
	return ((-1.0)*(sqrt(x*y*z)*sqrt(s))*(1.0/s+1.0/x)/((x+y)*(x+z)) );
} /* G1 */

float
G2(x,y,z) /* angle ftn partial. First, convert arguments to s-radii. */
float x,y,z;
{
	return ( (sqrt(x*z)/(sqrt(y)*((x+y)*sqrt(x+y+z)))) );
} /* G2 */

float
G3(x,y,z) /* angle ftn partial. First, convert arguments to s-radii. */
float x,y,z;
{
	return ( (sqrt(x*y)/(sqrt(z)*((x+z)*sqrt(x+y+z)))) );
} /* G3 */

/* ========= hyperbolic functions ========== */

float
g(x,y,z) /* angle function. First, convert arguments to s-radii. */
float x,y,z;
{
	x=1.0/(x*x); y=1.0/(y*y); z=1.0/(z*z);
	return (acos( ((x*y+1.0)*(x*z+1.0)-2.0*x*(y*z+1.0)) / 	
		((x*y-1.0)*(x*z-1.0)) ));
} /* g */

float
g1(x,y,z) /* angle ftn partial. First, convert arguments to s-radii. */
float x,y,z;
{
	x=1.0/(x*x); y=1.0/(y*y); z=1.0/(z*z);
	return ( (-1.0)*(x*x*y*z-1.0)*sqrt((y-1.0)*(z-1.0)) / 
		((x*y-1.0)*(x*z-1.0)*sqrt(x*(x-1.0)*(x*y*z-1.0))) );
} /* g1 */


float
g2(x,y,z) /* angle ftn partial. First, convert arguments to s-radii. */
float x,y,z;
{
	x=1.0/(x*x); y=1.0/(y*y); z=1.0/(z*z);
	return (sqrt(x*(x-1.0)*(z-1.0)) / 
		((x*y-1.0)*sqrt((y-1.0)*(x*y*z-1.0))) );
} /* g2 */

float
g3(x,y,z) /* angle ftn partial. First, convert arguments to s-radii. */
float x,y,z;
{
	x=1.0/(x*x); y=1.0/(y*y); z=1.0/(z*z);
	return ( sqrt(x*(x-1.0)*(y-1.0)) / 
		((x*z-1.0)*sqrt((z-1.0)*(x*y*z-1.0))) );
} /* g3 */

float
area_div(p,v) /* deriv of hyp area. */
struct p_data *p;
int v;
{
	int k;
	float accum=0.0;
	struct R_data *pR_ptr;
	struct K_data *pK_ptr;
	extern float Av();

	pR_ptr=p->packR_ptr;pK_ptr=p->packK_ptr;
	if (p->hes>=0 || v<1 || v>p->nodecount) return 0.0; 
	for (k=0;k<pK_ptr[v].num;k++)
		accum += Av(pR_ptr[v].rad,pR_ptr[pK_ptr[v].flower[k]].rad,
			pR_ptr[pK_ptr[v].flower[k+1]].rad);
	return accum;
} /* area_div */

float
Av(v,u,w) /* rate of change of area of triangle.*/
float v,u,w;
{
	return ((-1.0)*(g1(v,u,w)+g2(u,v,w)+g3(w,u,v)));
} /* Av */

float
dtheta_dlog(p,v) /* compute the derivative of the angle_sum theta
at vertex v with respect to the log of the actual hyperbolic radius 
of v. */
struct p_data *p;
int v;
{
	float vv,dtdv,h;
	extern float node_conductance(),radius();

	vv=p->packR_ptr[v].rad;
	if (vv<=0) return 0.0;
	vv=1.0/(vv*vv);
	h=radius(p,v);
	dtdv=node_conductance(p,v)/((1.0-vv)*sqrt(vv));
	return (dtdv*2.0*h*exp(2.0*h));
} /* dtheta_dlog */

/* ============= computations ========== */

float
twin_conductance(p,v) /* conductance to ground in hyp case */
struct p_data *p;
int v;
{
	float coef,vv;

	if (p->hes == 0 || p->hes >0) return 0; 
	if ((vv=p->packR_ptr[v].rad)<=okerr) return 0.0;
	vv=1.0/(vv*vv);
	coef=(vv-1.0)*sqrt(vv);
	return (area_div(p,v)*coef);
} /* twin_conductance */

float
Cvw(v,u,w,a,flag,hes) /* conductance of edge from v to w. 
flag: 0=2 nbh, 1=only left nbh (a), 2=only right nbh (u). */
float v,u,w,a;
int flag,hes;
{
	float vv,coef;

	if (hes>0) return 0; /* spherical not yet done */
	if (hes==0)
	 {
		coef=v;
		if (flag==1) return (G3(w,a,v)*coef);
		if (flag==2) return (G3(w,u,v)*coef);
		return ( (G3(w,u,v)+G3(w,a,v))*coef );
	 }
	if (hes<0)
	 {
		if (v<=0 || w<=0) return 0;
		vv=1.0/(v*v); /* convert to s-radius */
		coef=(vv-1.0)*sqrt(vv);
		if (flag==1) return (g3(w,a,v)*coef);
		if (flag==2) return (g3(w,u,v)*coef);
		return ( (g3(w,u,v)+g3(w,a,v))*coef);
	 }
} /* Cvw */

float
node_conductance(p,v) /* add up edge (and in hyp, twin) conductances */
struct p_data *p;
int v;
{
	int k;
	float accum,coef,vv;
	struct R_data *pR_ptr;
	struct K_data *pK_ptr;

	pR_ptr=p->packR_ptr;pK_ptr=p->packK_ptr;
	if (p->hes>0) return 0.0;
	if (p->hes==0)
	 {
		accum=0.0;
		for (k=1;k<pK_ptr[v].num;k++) /* middle faces */
		  accum += Cvw(pR_ptr[v].rad,pR_ptr[pK_ptr[v].flower[k-1]].rad,
			pR_ptr[pK_ptr[v].flower[k]].rad,
			pR_ptr[pK_ptr[v].flower[k+1]].rad,0,p->hes);
		accum += Cvw(pR_ptr[v].rad,0.0,pR_ptr[pK_ptr[v].flower[0]].rad,
		  pR_ptr[pK_ptr[v].flower[1]].rad,1,p->hes); /* first face */
		accum += Cvw(pR_ptr[v].rad,
		  pR_ptr[pK_ptr[v].flower[pK_ptr[v].num-1]].rad,
		  pR_ptr[pK_ptr[v].flower[pK_ptr[v].num]].rad,
		  0.0,2,p->hes); /* last face */
		return accum;
	 }
	if (p->hes<0)
	 {
		vv=pR_ptr[v].rad;
		if (vv<okerr) return 0.0;
		vv=1.0/(vv*vv);
		coef=(vv-1.0)*sqrt(vv);
		accum=area_div(p,v)*coef;
		for (k=1;k<pK_ptr[v].num;k++) /* middle faces */
		  accum += Cvw(pR_ptr[v].rad,pR_ptr[pK_ptr[v].flower[k-1]].rad,
			pR_ptr[pK_ptr[v].flower[k]].rad,
			pR_ptr[pK_ptr[v].flower[k+1]].rad,0,p->hes);
		accum += Cvw(pR_ptr[v].rad,0.0,pR_ptr[pK_ptr[v].flower[0]].rad,
		  pR_ptr[pK_ptr[v].flower[1]].rad,1,p->hes); /* first face */
		accum += Cvw(pR_ptr[v].rad,
		  pR_ptr[pK_ptr[v].flower[pK_ptr[v].num-1]].rad,
		  pR_ptr[pK_ptr[v].flower[pK_ptr[v].num]].rad,
		  0.0,2,p->hes); /* last face */
		return accum;
	 }
} /* node_conductance */

float
edge_conduct(p,v,indx) /* from v to indx neighbor */
struct p_data *p;
int v,indx;
{
	int w,left_index;
	float dum;
	struct R_data *pR_ptr;
	struct K_data *pK_ptr;

	pR_ptr=p->packR_ptr;pK_ptr=p->packK_ptr;
	w=pK_ptr[v].flower[indx];
	if (pK_ptr[v].bdry_flag && indx==0) /* bdry and upstream ngb */
	 {
		dum=Cvw(pR_ptr[v].rad,0.0,
			pR_ptr[w].rad,
			pR_ptr[pK_ptr[v].flower[1]].rad,1,p->hes);
		return (dum);
	 }
	if (pK_ptr[v].bdry_flag && indx==pK_ptr[v].num) /* bdry, down ngb */
	 {
		dum=Cvw(pR_ptr[v].rad,
			pR_ptr[pK_ptr[v].flower[pK_ptr[v].num-1]].rad,
			pR_ptr[w].rad,0.0,2,p->hes);
		return (dum);
	 }
	if (indx==0) left_index=pK_ptr[v].num-1;
	else left_index=indx-1;
	dum=Cvw(pR_ptr[v].rad,
		pR_ptr[pK_ptr[v].flower[left_index]].rad,
		pR_ptr[w].rad,
		pR_ptr[pK_ptr[v].flower[indx+1]].rad,0,p->hes);
	return (dum);
} /* edge_conduct */


/* =============== end of random walk functions ======== */

/* --------------- Gaussian curvature routines ----------------- */

int
kappa_calc(p,v,kap) /* compute the square root of curvature for that
model in which label for v gives angle sum 2pi. return = 0 means no pos
curv will work. return =-1 means center had hyp radius infinity. */
struct p_data *p;
int v;
float *kap;
{
	float Theta,Phi,Top,Bottom, Middle, Difference,*R;
	int i,n;
	float kap_upper_bd,rad,bound,dum;
	extern float neg_k_angle_sum(),pos_k_angle_sum();
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	n=pK_ptr[v].num;
/* set up label */
	R=(float *)malloc((n+2)*sizeof(float));
	if (p->hes<(-1.0)*okerr) /* hyp data, stored as s-radii */
	 {
		if ((rad=pR_ptr[v].rad)<=0)
		 {
			*kap=0.0;
			free(R);
			return -1;
		 }
		R[0]=-log(rad);
		for (i=0;i<=n;i++)
		 {
			if ((rad=pR_ptr[pK_ptr[v].flower[i]].rad)<=0) 
				R[i+1]=10.0; /* arbitrary large value */
			else R[i+1]=-log(rad);
		 }
	 }
	else
	 {
		R[0]=pR_ptr[v].rad;
		for (i=0;i<=n;i++) R[i+1]=pR_ptr[pK_ptr[v].flower[i]].rad;
	 }
/* First get eucl angle sum to see if curvature should be +,-, or 0 */
	Theta= 0.0;
	for (i=1;i<=n;i++)
		Theta += acos (((R[0] + R[i])*(R[0]+R[i]) + 
			(R[0] + R[i+1])*(R[0]+R[i+1])-
			(R[i]+R[i+1])*(R[i]+R[i+1]))/
			(2*(R[0] + R[i])*(R[0] + R[i+1])));
/* close to euclidean */
	if (fabs(Theta-2.0*M_PI)<okerr) 
	 {
		free(R);
		*kap=0.0;
		return 1;
	 }	
/* Positive curvature case */
	if (Theta <= (2.0 * M_PI))
	 {
		bound=0.0;
		for (i=0;i<n;i++)
		 {
			if ((dum=R[0]+R[i+1]+R[i+2])>bound) bound=dum;
		 }
		kap_upper_bd=M_PI/bound; 
			/* kap too small means labels would be inconsistent */
		Phi=pos_k_angle_sum(kap_upper_bd-okerr,R,n);
		if (Phi< 2.0*M_PI-okerr) 
			/* no curvature will work for this label */
		 {
			free(R);
			*kap=0.0;
			return 0;
		 }
		if (1.0< kap_upper_bd/2.0) Top= 1.0;
		else Top= kap_upper_bd/2.0;
		Phi=pos_k_angle_sum(Top,R,n);
		while (Phi <= (2*M_PI) && Top<kap_upper_bd-okerr)
		  {
			Top=(2.0*Top<kap_upper_bd-okerr) ? 
				2.0*Top : kap_upper_bd-okerr;
			Phi=pos_k_angle_sum(Top,R,n);
		  }
		Bottom=0;
		Middle=0;  /* to handle Theta = 2*M_PI */
		Difference= Theta - 2.0* M_PI;
		while (fabs(Difference)>okerr)
		 {
			Middle= (Top + Bottom)/2.0;
			Difference= pos_k_angle_sum(Middle,R,n)-2.0*M_PI;
			if (Difference>=0) Top=Middle;
			else Bottom=Middle;
		 }
		free(R);
		*kap=Middle;
		return 1;
	  }
/* Negative curvature case */
	else 
 	  {
		Top=-1;
		Phi=neg_k_angle_sum(Top,R,n);
		while (Phi >= (2.0*M_PI))
		 {
			Top= 2.0*Top;
			Phi= neg_k_angle_sum(Top,R,n);
		 }
		Bottom= Middle=0.0;
		Difference= Theta - 2.0*M_PI;
		while (fabs(Difference)>okerr)
		 {
			Middle=(Top + Bottom)/2.0;
			Difference=neg_k_angle_sum(Middle,R,n)-2.0*M_PI;
			if (Difference<=0) Top=Middle;
			else Bottom=Middle;
		  }
		free(R);
		*kap=Middle;
		return 1;
	   }
} /* kappa_calc */

float 
pos_k_angle_sum(c,R,m) /* ang sum comp for curvature + (c*c) */
float c,*R;
int m;
{ 
	int i;
	float Sum;

	Sum=0.0;
	for (i=1;i<=m;i++)
		Sum += acos((cos(c*(R[i]+R[i+1]))-(cos(c*(R[0]+R[i]))*
			cos(c*(R[0]+R[i+1]))))/
			(sin(c*(R[0]+R[i]))*sin(c*(R[0]+R[i+1]))));
	return (Sum);
} /* pos_k_angle_sum */
			
float 
neg_k_angle_sum(c,R,m) /* ang sum comp for curvature - (c*c) */
float c,*R;
int m;
{
	int i;
	float Sum;
 
	Sum=0.0;
	c=c*(-1.0);  /* to simplify formula below */
	for (i=1;i<=m;i++)
		Sum += acos((cosh(c*(R[0]+R[i]))*cosh(c*(R[0]+R[i+1])) -
			cosh(c*(R[i]+R[i+1])))/(sinh(c*(R[0]+R[i]))*
			sinh(c*(R[0]+R[i+1]))));
	return (Sum);
} /* neg_k_angle_sum */

int
color_kappa(p,base) /* set circle colors; compare model curvature to base. 
Red==>kappa > base, Blue==> < base, white==> = base. */
struct p_data *p;
float base;
{
	int v;
	float kap;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	for (v=1;v<=p->nodecount;v++)
	 {
		if (pK_ptr[v].bdry_flag || kappa_calc(p,v,&kap)<=0) 
		 {
			pK_ptr[v].color=FG_COLOR;
			return;
		 }
		kap=kap-base;
		if (kap<=0)
		 {
			if (fabs(kap)<okerr) pK_ptr[v].color=0;
			if (kap>= -1.0) 
				pK_ptr[v].color=(int)(kap*40 + 100);
			else if (kap>= -2.0) 
				pK_ptr[v].color=(int)((kap+1)*25+60);
			else if (kap>= -3.0)
				pK_ptr[v].color=(int)((kap+2)*20+35);	
			else if (kap>= -4.0)
				pK_ptr[v].color=(int)((kap+3)*14+15);
			else 
				pK_ptr[v].color=1;
		 }
		else 
		 {
			if (kap<=1.0)
				pK_ptr[v].color=(int)(kap*40+100);
			else if (kap<=2.0)
				pK_ptr[v].color=(int)((kap-1)*25+140);
			else if (kap<=3.0)
				pK_ptr[v].color=(int)((kap-2)*20+165);
			else if (kap<=4.0)
				pK_ptr[v].color=(int)((kap-3)*14+185);
			else 
				pK_ptr[v].color=199;
		 }
	 }
} /* color_kappa */
	
