/*
 * Copyright (c) 1990,1993 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#include <errno.h>
#include <sys/syslog.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <netatalk/endian.h>
#include <atalk/adouble.h>
#include <atalk/afp.h>
#include <utime.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>

#include "directory.h"
#include "desktop.h"
#include "volume.h"
#include "file.h"
#include "globals.h"

/* the format for the finderinfo fields (from IM: Toolbox Essentials):
 * field         bytes        subfield    bytes
 * 
 * files:
 * ioFlFndrInfo  16      ->       type    4  type field
 *                             creator    4  creator field
 *                               flags    2  finder flags: 
 *					     alias, bundle, etc.
 *                            location    4  location in window
 *                              folder    2  window that contains file
 * 
 * ioFlXFndrInfo 16      ->     iconID    2  icon id
 *                              unused    6  reserved 
 *                              script    1  script system
 *                              xflags    1  reserved
 *                           commentID    2  comment id
 *                           putawayID    4  home directory id
 */

const u_char ufinderi[] = {
    'T', 'E', 'X', 'T', 'U', 'N', 'I', 'X',
    0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0
};

getfilparams( bitmap, path, dir, st, buf, buflen )
    u_short	bitmap;
    char	*path;
    struct dir	*dir;
    struct stat	*st;
    char	*buf;
    int		*buflen;
{
    struct stat		hst;
    struct adouble	ad;
    struct extmap	*em;
    char		*data, *nameoff = NULL;
    int			bit = 0, isad = 1, aint;
    u_int16_t		ashort;
    u_char              achar, fdType[4];

    if ( ad_open( mtoupath( path ), ADFLAGS_HF, O_RDONLY, 0, &ad ) < 0 ) {
	isad = 0;
    } else {
	if ( fstat( ad_hfileno( &ad ), &hst ) < 0 ) {
	    syslog( LOG_ERR, "getfilparams fstat: %m" );
	}
    }

    data = buf;
    while ( bitmap != 0 ) {
	while (( bitmap & 1 ) == 0 ) {
	    bitmap = bitmap>>1;
	    bit++;
	}

	switch ( bit ) {
	case FILPBIT_ATTR :
	    if ( isad ) {
		bcopy( ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_ATTR, &ashort,
			sizeof( u_short ));
	    } else {
		ashort = 0;
	    }
	    bcopy( &ashort, data, sizeof( u_short ));
	    data += sizeof( u_short );
	    break;

	case FILPBIT_PDID :
	    bcopy( &dir->d_did, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_CDATE :
	    if ( isad ) {
		bcopy( ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_CREATE, &aint,
			sizeof( int ));
	    } else {
		aint = htonl( st->st_mtime );
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_MDATE :
	    if ( isad ) {
		bcopy( ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_MODIFY, &aint,
			sizeof( int ));
		aint = ntohl( aint );
		if ( st->st_mtime > aint && hst.st_mtime < st->st_mtime ) {
		    aint = st->st_mtime;
		}
	    } else {
		aint = st->st_mtime;
	    }
	    aint = htonl( aint );
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_BDATE :
	    if ( isad ) {
		bcopy( ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_BACKUP, &aint,
			sizeof( int ));
	    } else {
		aint = 0;
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_FINFO :
	    memcpy(data, isad ? (void *) ad_entry(&ad, ADEID_FINDERI) :
		   (void *) ufinderi, 32);

	    if ( !isad ||
		    bcmp( ad_entry( &ad, ADEID_FINDERI ), ufinderi, 8 ) == 0 ) {
	         bcopy(ufinderi, data, 8);
		 if (( em = getextmap( path )) != NULL ) {
		   bcopy( em->em_type, data, sizeof( em->em_type ));
		   bcopy( em->em_creator, data + 4, sizeof( em->em_creator ));
		 } 
	    }
	    data += 32;
	    break;

	case FILPBIT_LNAME :
	    nameoff = data;
	    data += sizeof( u_short );
	    break;

	case FILPBIT_SNAME :
	    ashort = 0;
	    bcopy( &ashort, data, sizeof( u_short ));
	    data += sizeof( u_short );
	    break;

	case FILPBIT_FNUM :
	    /*
	     * What a fucking mess.  First thing:  DID and FNUMs are
	     * in the same space for purposes of enumerate (and several
	     * other wierd places).  While we consider this Apple's bug,
	     * this is the work-around:  In order to maintain constant and
	     * unique DIDs and FNUMs, we monotonically generate the DIDs
	     * during the session, and derive the FNUMs from the filesystem.
	     * Since the DIDs are small, we insure that the FNUMs are fairly
	     * large by setting thier high bits to the device number.
	     *
	     * AFS already does something very similar to this for the
	     * inode number, so we don't repeat the procedure.
	     */
#ifdef AFS
	    aint = st->st_ino;
#else AFS
	    aint = ( st->st_dev << 16 ) | ( st->st_ino & 0x0000ffff );
#endif AFS
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_DFLEN :
	    aint = htonl( st->st_size );
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_RFLEN :
	    if ( isad ) {
		aint = htonl( ad_getentrylen( &ad, ADEID_RFORK ));
	    } else {
		aint = 0;
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

 	    /* Current client needs ProDOS info block for this file.
 	       Use simple heuristic and let the Mac "type" string tell
 	       us what the PD file code should be.  Everything gets a
 	       subtype of 0x0000 unless the original value was hashed
 	       to "pXYZ" when we created it.  See IA, Ver 2.
 	       <shirsch@ibm.net> */
 	case FILPBIT_PDINFO :
 	    if ( isad ) {
 	      bcopy( ad_entry( &ad, ADEID_FINDERI ), fdType, 4 );
 	      
 	      if ( bcmp( fdType, "TEXT", 4 ) == 0 ) {
 		achar = '\x04';
 		ashort = 0x0000;
 	      }
 	      else if ( bcmp( fdType, "PSYS", 4 ) == 0 ) {
 		achar = '\xff';
 		ashort = 0x0000;
 	      }
 	      else if ( bcmp( fdType, "PS16", 4 ) == 0 ) {
 		achar = '\xb3';
 		ashort = 0x0000;
 	      }
 	      else if ( bcmp( fdType, "BINA", 4 ) == 0 ) {
 		achar = '\x00';
 		ashort = 0x0000;
 	      }
 	      else if ( fdType[0] == 'p' ) {
 		achar = fdType[1];
 		ashort = (fdType[2] * 256) + fdType[3];
 	      } 
 	      else {
 		achar = '\x00';
 		ashort = 0x0000;
 	      }
 	    } 
 	    else {
 	      achar = '\x00';
 	      ashort = 0x0000;
 	    }
 	    
 	    bcopy( &achar, data, sizeof( u_char ));
 	    data += sizeof( u_char );
 
 	    achar = 0x00;
 	    bcopy( &achar, data, sizeof( u_char ));
 	    data += sizeof( u_char );
 
 	    bcopy( &ashort, data, sizeof( u_short ));
 	    data += sizeof( u_short );
 
 	    ashort = 0x0000;
 	    bcopy( &ashort, data, sizeof( u_short ));
 	    data += sizeof( u_short );
 	    break;

	default :
	    if ( isad ) {
		ad_close( &ad, ADFLAGS_HF );
	    }
	    return( AFPERR_BITMAP );
	}
	bitmap = bitmap>>1;
	bit++;
    }
    if ( nameoff ) {
	ashort = htons( data - buf );
	bcopy( &ashort, nameoff, sizeof( u_short ));
	aint = strlen( path );
	aint = ( aint > MACFILELEN ) ? MACFILELEN : aint;
	*data++ = aint;
	bcopy( path, data, aint );
	data += aint;
    }
    if ( isad ) {
	ad_close( &ad, ADFLAGS_HF );
    }
    *buflen = data - buf;
    return( AFP_OK );
}

afp_createfile(obj, ibuf, ibuflen, rbuf, rbuflen )
    AFPObj      *obj;
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct stat         st;
    struct adouble	ad;
    struct timeval	tv;
    struct vol		*vol;
    struct dir		*dir;
    char		*path;
    int			creatf, did, openf;
    u_short		vid;

    *rbuflen = 0;
    ibuf++;
    creatf = (unsigned char) *ibuf++;

    bcopy( ibuf, &vid, sizeof( u_short ));
    ibuf += sizeof( u_short );

    if (( vol = getvolbyvid( vid )) == NULL ) {
	return( AFPERR_PARAM );
    }

    bcopy( ibuf, &did, sizeof( int ));
    ibuf += sizeof( int );

    if (( dir = dirsearch( vol, did )) == NULL ) {
	return( AFPERR_NOOBJ );
    }

    if (( path = cname( vol, dir, &ibuf )) == NULL ) {
	return( AFPERR_NOOBJ );
    }

    if ( creatf) {
        /* on a hard create, fail if file exists and is open */
        if ((stat(path, &st) == 0) && of_findname(curdir, path))
	  return AFPERR_BUSY;
	openf = O_RDWR|O_CREAT|O_TRUNC;
    } else {
	openf = O_RDWR|O_CREAT|O_EXCL;
    }

    if ( ad_open( mtoupath( path ), ADFLAGS_DF|ADFLAGS_HF, openf,
	    0666, &ad ) < 0 ) {
	switch ( errno ) {
	case EEXIST :
	    return( AFPERR_EXIST );
	case EACCES :
	    return( AFPERR_ACCESS );
	default :
	    return( AFPERR_PARAM );
	}
    }

    ad_setentrylen( &ad, ADEID_NAME, strlen( path ));
    bcopy( path, ad_entry( &ad, ADEID_NAME ),
	    ad_getentrylen( &ad, ADEID_NAME ));

    if ( gettimeofday( &tv, 0 ) < 0 ) {
	syslog( LOG_ERR, "afp_createfile: gettimeofday: %m" );
	ad_close( &ad, ADFLAGS_DF|ADFLAGS_HF );
	return AFPERR_PARAM;
    }
    tv.tv_sec = htonl( tv.tv_sec );
    bcopy( &tv.tv_sec, ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_CREATE,
	    sizeof( tv.tv_sec ));
    bcopy( &tv.tv_sec, ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_MODIFY,
	    sizeof( tv.tv_sec ));

    ad_flush( &ad, ADFLAGS_DF|ADFLAGS_HF );
    ad_close( &ad, ADFLAGS_DF|ADFLAGS_HF );
    setvoltime(obj, vol );
    return AFP_OK;
}

afp_setfilparams(obj, ibuf, ibuflen, rbuf, rbuflen )
    AFPObj      *obj;
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct vol	*vol;
    struct dir	*dir;
    char	*path;
    int		did, rc;
    u_short	vid, bitmap;

    *rbuflen = 0;
    ibuf += 2;

    bcopy( ibuf, &vid, sizeof( u_short ));
    ibuf += sizeof( u_short );
    if (( vol = getvolbyvid( vid )) == NULL ) {
	return( AFPERR_PARAM );
    }

    bcopy( ibuf, &did, sizeof( int ));
    ibuf += sizeof( int );
    if (( dir = dirsearch( vol, did )) == NULL ) {
	return( AFPERR_NOOBJ );
    }

    bcopy( ibuf, &bitmap, sizeof( u_short ));
    bitmap = ntohs( bitmap );
    ibuf += sizeof( u_short );

    if (( path = cname( vol, dir, &ibuf )) == NULL ) {
	return( AFPERR_NOOBJ );
    }

    if ((u_long)ibuf & 1 ) {
	ibuf++;
    }

    if (( rc = setfilparams( path, bitmap, ibuf )) == AFP_OK ) {
	setvoltime(obj, vol );
    }

    return( rc );
}

setfilparams( path, bitmap, buf )
    char	*path, *buf;
    u_short	bitmap;
{
    struct adouble	ad;
    struct extmap	*em;
    int			bit = 0;
    u_char              achar, *fdType, xyy[4];
    u_int16_t		ashort, bshort;
    u_int32_t		aint;
    time_t		atime;
    struct utimbuf	ut;

    if ( ad_open( mtoupath( path ), ADFLAGS_HF, O_RDWR|O_CREAT,
	    0666, &ad ) < 0 ) {
	syslog( LOG_INFO, "setfilparams: ad_open %s: %m", path );
	return( AFPERR_ACCESS );
    }

    if ( ad_getoflags( &ad, ADFLAGS_HF ) & O_CREAT ) {
	ad_setentrylen( &ad, ADEID_NAME, strlen( path ));
	bcopy( path, ad_entry( &ad, ADEID_NAME ),
		ad_getentrylen( &ad, ADEID_NAME ));
    }

    while ( bitmap != 0 ) {
	while (( bitmap & 1 ) == 0 ) {
	    bitmap = bitmap>>1;
	    bit++;
	}

	switch(  bit ) {
	case FILPBIT_ATTR :
	    bcopy( buf, &ashort, sizeof( u_short ));
	    bcopy( ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_ATTR, &bshort,
		    sizeof( u_short ));
	    if ( ntohs( ashort ) & ATTRBIT_SETCLR ) {
		bshort |= htons( ntohs( ashort ) & ~ATTRBIT_SETCLR );
	    } else {
		bshort &= ~ashort;
	    }
	    bcopy( &bshort, ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_ATTR,
		    sizeof( u_short ));
	    buf += sizeof( u_short );
	    break;

	case FILPBIT_CDATE :
	    bcopy( buf, ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_CREATE,
		    sizeof( int ));
	    buf += sizeof( int );
	    break;

	case FILPBIT_MDATE :
	    bcopy( buf, ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_MODIFY,
		    sizeof( int ));
	    bcopy( buf, &aint, sizeof( aint ));
	    atime = ntohl( aint );
	    ut.actime = atime;
	    ut.modtime = atime;
	    utime( mtoupath( path ), &ut );
	    buf += sizeof( int );
	    break;

	case FILPBIT_BDATE :
	    bcopy( buf, ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_BACKUP,
		    sizeof( int ));
	    buf += sizeof( int );
	    break;

	case FILPBIT_FINFO :
	    if (( ad_getoflags( &ad, ADFLAGS_HF ) & O_CREAT ) ||
		    bcmp( ad_entry( &ad, ADEID_FINDERI ), ufinderi, 8 ) == 0 ) {
		if (( em = getextmap( path )) != NULL ) {
		    if ( bcmp( buf, em->em_type, sizeof( em->em_type )) == 0 &&
			    bcmp( buf + 4, em->em_creator,
			    sizeof( em->em_creator )) == 0 ) {
			bcopy( ufinderi, buf, 8 );
		    }
		}
	    }
	    memcpy(ad_entry( &ad, ADEID_FINDERI ), buf, 32 );
	    buf += 32;
	    break;

 	    /* Client needs to set the ProDOS file info for this file.
 	       Use defined strings for the simple cases, and convert
 	       all else into pXYY per Inside Appletalk.  Always set
 	       the creator as "pdos". <shirsch@ibm.net> */
 	case FILPBIT_PDINFO :
 	    bcopy( buf, &achar, sizeof( u_char ));
 	    buf += 2 * sizeof( u_char );
 	  
 	    bcopy( buf, &ashort, sizeof( u_short ));
 	    ashort = ntohs( ashort );
 	    buf += 2 * sizeof( u_char );
 
 	    switch ( (unsigned int) achar )
 	      {
 	      case 0x04 :
 		fdType = ( u_char *) "TEXT";
 		break;
 		
 	      case 0xff :
 		fdType = ( u_char *) "PSYS";
 		break;
 
 	      case 0xb3 :
 		fdType = ( u_char *) "PS16";
 		break;
 
 	      case 0x00 :
 		fdType = ( u_char *) "BINA";
 		break;
 
 	      default :
 		xyy[0] = ( u_char ) 'p';
 		xyy[1] = achar;
 		xyy[2] = ( u_char ) ( ashort >> 8 ) & 0xff;
 		xyy[3] = ( u_char ) ashort & 0xff;
 		fdType = xyy;
 		break;
 	      }
 	    
 	    bcopy( fdType, ad_entry( &ad, ADEID_FINDERI ), 4 );
 	    bcopy( "pdos", ad_entry( &ad, ADEID_FINDERI ) + 4, 4 );
 	    break;
 

	default :
	    ad_close(&ad, ADFLAGS_HF);
	    return( AFPERR_BITMAP );
	}

	bitmap = bitmap>>1;
	bit++;
    }

    ad_flush( &ad, ADFLAGS_HF );
    if ( ad_close( &ad, ADFLAGS_HF ) < 0 ) {
	syslog( LOG_INFO, "setfilparams: ad_close %s: %m", path );
	return( AFPERR_PARAM );
    }
    return( AFP_OK );
}

/*
 * renamefile and copyfile take the old and new unix pathnames
 * and the new mac name.
 */
renamefile( src, dst, newname )
    char	*src, *dst, *newname;
{
    struct adouble	ad;
     char		adsrc[ MAXPATHLEN + 1];
    int			len, rc;

    /*
     * Note that this is only checking the existance of the data file,
     * not the header file.  The thinking is that if the data file doesn't
     * exist, but the header file does, the right thing to do is remove
     * the data file silently.
     */

    /* existence check moved to afp_moveandrename */

    if ( rename( src, dst ) < 0 ) {
	switch ( errno ) {
	case ENOENT :
	    return( AFPERR_NOOBJ );
	case EACCES :
	    return( AFPERR_ACCESS );
	case EXDEV :			/* Cross device move -- try copy */
	    if (( rc = copyfile( src, dst, newname )) != AFP_OK ) {
		deletefile( dst );
		return( rc );
	    }
	    rc = deletefile( src );
	    return( rc );
	default :
	    return( AFPERR_PARAM );
	}
    }

    strcpy( adsrc, ad_path( src, 0 ));
    if ( rename( adsrc, ad_path( dst, 0 )) < 0 ) {
	switch ( errno ) {
	case ENOENT :
	    return( AFP_OK );
	case EACCES :
	    return( AFPERR_ACCESS );
	default :
	    return( AFPERR_PARAM );
	}
    }

    if ( ad_open( dst, ADFLAGS_HF, O_RDWR, 0666, &ad ) < 0 ) {
	switch ( errno ) {
	case ENOENT :
	    return( AFPERR_NOOBJ );
	case EACCES :
	    return( AFPERR_ACCESS );
	default :
	    return( AFPERR_PARAM );
	}
    }
    len = strlen( newname );
    ad_setentrylen( &ad, ADEID_NAME, len );
    bcopy( newname, ad_entry( &ad, ADEID_NAME ), len );
    ad_flush( &ad, ADFLAGS_HF );
    ad_close( &ad, ADFLAGS_HF );

    return( AFP_OK );
}

afp_copyfile(obj, ibuf, ibuflen, rbuf, rbuflen )
    AFPObj      *obj;
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct vol	*vol;
    struct dir	*dir;
    char	newname[ MAXNAMLEN + 1], *path, *p;
    int		sdid, ddid;
    int		plen;
    short	svid, dvid;

    *rbuflen = 0;
    ibuf += 2;

    bcopy( ibuf, &svid, sizeof( short ));
    ibuf += sizeof( short );
    if (( vol = getvolbyvid( svid )) == NULL ) {
	return( AFPERR_PARAM );
    }

    bcopy( ibuf, &sdid, sizeof( int ));
    ibuf += sizeof( int );
    if (( dir = dirsearch( vol, sdid )) == NULL ) {
	return( AFPERR_PARAM );
    }

    bcopy( ibuf, &dvid, sizeof( short ));
    ibuf += sizeof( short );
    bcopy( ibuf, &ddid, sizeof( int ));
    ibuf += sizeof( int );

    if (( path = cname( vol, dir, &ibuf )) == NULL ) {
	return( AFPERR_NOOBJ );
    }
    if ( *path == '\0' ) {
	return( AFPERR_BADTYPE );
    }
    strcpy( newname, path );

    p = ctoupath( vol, curdir, newname );

    if (( vol = getvolbyvid( dvid )) == NULL ) {
	return( AFPERR_PARAM );
    }
    if (( dir = dirsearch( vol, ddid )) == NULL ) {
	return( AFPERR_PARAM );
    }

    if (( path = cname( vol, dir, &ibuf )) == NULL ) {
	return( AFPERR_NOOBJ );
    }
    if ( *path != '\0' ) {
	return( AFPERR_BADTYPE );
    }

    /* one of the handful of places that knows about the path type */
    if ( *ibuf++ != 2 ) {
	return( AFPERR_PARAM );
    }
    if (( plen = (unsigned char)*ibuf++ ) != 0 ) {
	strncpy( newname, ibuf, plen );
	newname[ plen ] = '\0';
    }
    if ( copyfile( p, mtoupath( newname ), newname ) < 0 ) {
	return( AFPERR_ACCESS );
    }

    setvoltime(obj, vol );
    return( AFP_OK );
}


static __inline__ int copy_all(const int dfd, const void *buf,
			       size_t buflen)
{
  ssize_t cc;

  while (buflen > 0) {
    if ((cc = write(dfd, buf, buflen)) < 0) {
      if (errno == EINTR)
	continue;

      return AFPERR_PARAM;
    }
    buflen -= cc;
  }

  return 0;
}


copyfile( src, dst, newname )
    char	*src, *dst, *newname;
{
    struct adouble	ad;
    char		filebuf[8192];
    int			sfd, dfd, len, err = AFP_OK;
    ssize_t             cc;


    if (( sfd = open( ad_path( src, 0 ), O_RDONLY, 0 )) < 0 ) {
	switch ( errno ) {
	case ENOENT :
	    break;
	case EACCES :
	    return( AFPERR_ACCESS );
	default :
	    return( AFPERR_PARAM );
	}
    } else {
	if (( dfd = open( ad_path( dst, 0 ), O_WRONLY|O_CREAT,
		ad_mode( ad_path( dst, 0 ), 0666 ))) < 0 ) {
	    close( sfd );
	    switch ( errno ) {
	    case ENOENT :
		return( AFPERR_NOOBJ );
	    case EACCES :
		return( AFPERR_ACCESS );
	    default :
		return( AFPERR_PARAM );
	    }
	}

	/* copy the file */
#ifdef HAVE_SENDFILE
	/* get the size of the file */
	err = sendfile(dfd, sfd, NULL, );
#else
	while (( cc = read( sfd, filebuf, sizeof(filebuf))) > 0 ) {
	  if ((err = copy_all(dfd, filebuf, cc)) < 0)
	    break;
	}
#endif

	close(sfd);
	close(dfd);
	if (err < 0)
	  return AFPERR_PARAM;
    }

    if (( sfd = open( src, O_RDONLY, 0 )) < 0 ) {
	switch ( errno ) {
	case ENOENT :
	    return( AFPERR_NOOBJ );
	case EACCES :
	    return( AFPERR_ACCESS );
	default :
	    return( AFPERR_PARAM );
	}
    }

    if (( dfd = open( dst, O_WRONLY|O_CREAT, ad_mode( dst, 0666 ))) < 0 ) {
	close( sfd );
	switch ( errno ) {
	case ENOENT :
	    return( AFPERR_NOOBJ );
	case EACCES :
	    return( AFPERR_ACCESS );
	default :
	    return( AFPERR_PARAM );
	}
    }

#ifdef HAVE_SENDFILE
    err = sendfile(dfd, sfd, NULL, );
#else
    while (( cc = read( sfd, filebuf, sizeof( filebuf ))) > 0 ) {
      if ((err = copy_all(dfd, filebuf, cc)) < 0) {
	break;
      }
    }
#endif
    
    close(sfd);
    close(dfd);
    if (err < 0)
      return AFPERR_PARAM;

    if ( ad_open( dst, ADFLAGS_HF, O_RDWR|O_CREAT, 0666, &ad ) < 0 ) {
	switch ( errno ) {
	case ENOENT :
	    return( AFPERR_NOOBJ );
	case EACCES :
	    return( AFPERR_ACCESS );
	default :
	    return( AFPERR_PARAM );
	}
    }
    len = strlen( newname );
    ad_setentrylen( &ad, ADEID_NAME, len );
    bcopy( newname, ad_entry( &ad, ADEID_NAME ), len );
    ad_flush( &ad, ADFLAGS_HF );
    ad_close( &ad, ADFLAGS_HF );

    return( AFP_OK );
}

deletefile( file )
    char		*file;
{
    struct adouble	ad;
    int			adflags, err = AFP_OK;

    adflags = ADFLAGS_DF|ADFLAGS_HF;
    if ( ad_open( file, adflags, O_RDWR, 0, &ad ) < 0 ) {
	if ( errno == ENOENT ) {
	    ad_close( &ad, adflags );
	    adflags = ADFLAGS_DF;
	    if ( ad_open( file, adflags, O_RDWR, 0, &ad ) < 0 ) {
		ad_close( &ad, adflags );
		return( AFPERR_NOOBJ );
	    }
	} else {
	    ad_close( &ad, adflags );
	    if ( errno == EACCES ) {
		return( AFPERR_ACCESS );
	    } else {
		return( AFPERR_PARAM );
	    }
	}
    }

    if ( adflags & ADFLAGS_HF ) {
      if ( ad_tmplock(&ad, ADEID_RFORK, ADLOCK_WR, 0, 0, 0) < 0 ) {
	    ad_close( &ad, adflags );
	    return( AFPERR_BUSY );
	}
    }

    if ( ad_tmplock( &ad, ADEID_DFORK, ADLOCK_WR, 0, 0, 0 ) < 0 ) {
	ad_tmplock(&ad, ADEID_RFORK, ADLOCK_CLR, 0, 0, 0);
	ad_close( &ad, adflags );
	return( AFPERR_BUSY );
    }

    if ( unlink( ad_path( file, 0 )) < 0 ) {
	switch ( errno ) {
	case EACCES :
	    err = AFPERR_ACCESS;
	    goto delete_unlock;
	case ENOENT :
	    break;
	default :
	    err = AFPERR_PARAM;
            goto delete_unlock;
	}
    }

    if ( unlink( file ) < 0 ) {
	switch ( errno ) {
	case EACCES :
	    err = AFPERR_ACCESS;
	    goto delete_unlock;
	case ENOENT :
	    err = AFPERR_NOOBJ;
	    goto delete_unlock;
	default :
	    err = AFPERR_PARAM;
	    goto delete_unlock;
	}
    }

delete_unlock:
    ad_tmplock(&ad, ADEID_RFORK, ADLOCK_CLR, 0, 0, 0);
    ad_tmplock(&ad, ADEID_DFORK, ADLOCK_CLR, 0, 0, 0);
    ad_close( &ad, adflags );
    return err;
}
