
/* 
 * LDAPObject - wrapper around an LDAP* context
 * $Id: LDAPObject.c,v 1.9 1997/12/01 22:46:09 leonard Exp $
 */

#include <math.h>
#include <sys/time.h>
#include "common.h"
#include "errors.h"
#include "constants.h"
#include "LDAPObject.h"
#include "message.h"

/* constructor */

LDAPObject*
newLDAPObject( LDAP* l ) 
{
    LDAPObject* self = (LDAPObject*) PyObject_NEW(LDAPObject, &LDAP_Type);
    if (self == NULL) 
    	return NULL;
    self->ldap = l;
    self->_save = NULL;
    self->valid = 1;
    return self;
}

/* destructor */

static void
dealloc( LDAPObject* self )
{
    if (self->ldap) {
	if (self->valid) {
	    ldap_unbind( self->ldap );
	    self->valid = 0;
	}
	self->ldap = NULL;
    }
    PyMem_DEL(self);
}

/*------------------------------------------------------------
 * utility functions
 */

/* 
 * check to see if the LDAPObject is valid, 
 * ie has been opened, and not closed. An exception is set if not valid.
 */

static int
not_valid( LDAPObject* l ) {
    if (l->valid) {
    	return 0;
    } else {
    	PyErr_SetString( LDAPerror_object, "LDAP connection invalid" );
	return 1;
    }
}
  
/* free the LDAPMod allocated in Tuple_to_LDAPMod() */

static void
free_LDAPMod( LDAPMod* lm ) {
    struct berval **bv;

    for(bv = lm->mod_bvalues; bv && *bv; bv++) {
      if ((*bv)->bv_val) free( (*bv)->bv_val );
      free( *bv );
    }
    if (lm->mod_bvalues) free(lm->mod_bvalues);
    lm->mod_bvalues = NULL;	/* paranoia */
    if (lm->mod_type != NULL) free(lm->mod_type);
}

/* convert a tuple of the form (int,str,[str,...]) 
 * or (str, [str,...]) if no_op, into an LDAPMod 
 */

/* XXX - doesn't yet handle BER objects - ie this is WRONG!! */

static LDAPMod*
Tuple_to_LDAPMod( PyObject* tup, int no_op ) 
{
    int op;
    char *type;
    struct berval **bervals;
    PyObject *list;
    int listlen;
    LDAPMod *lm;

    if (no_op) {
        if (!PyArg_ParseTuple( tup, "sO", &type, &list )) return NULL;
	op = 0;
    } else {
	if (!PyArg_ParseTuple( tup, "isO", &op, &type, &list )) return NULL;
    }

    lm = malloc( sizeof(LDAPMod) );
    if (lm == NULL) { PyErr_NoMemory(); return NULL; }

    if (PyNone_Check(list)) {

	/* None... used for delete */
    	bervals = NULL;

    } else if (PyString_Check( list )) {
        /* a single string on its own... treat as list of 1 */
	int length;

	bervals = (struct berval**) malloc( 2 * sizeof(struct berval*) );
	if (bervals==NULL) { free(lm); PyErr_NoMemory(); return NULL; }

	bervals[1] = NULL;
	bervals[0] = malloc( sizeof( struct berval ) );
	if (bervals[0] == NULL) { 
		free_LDAPMod(lm); 
		PyErr_NoMemory(); 
		return NULL; 
	}
	length = PyString_Size( list );
	bervals[0]->bv_val = 
		(char*)malloc( length * sizeof(char) );
	if (bervals[0]->bv_val == NULL) {
		free_LDAPMod(lm); 
		PyErr_NoMemory(); 
		return NULL; 
	}
	bervals[0]->bv_len = length;
	memcpy( bervals[0]->bv_val, PyString_AsString(list), length );
    } else if (!PySequence_Check( list )) {
    	/* not a list */
    	PyErr_SetObject( PyExc_TypeError, Py_BuildValue( "sO",
		"expected sequence of strings", list ));
	free(lm);
	return NULL;
    } else {
	/* a list, possible of strings */
	int i;

	listlen = PySequence_Length( list );
	for(i=0; i<listlen; i++) 
	   if (!PyString_Check( PySequence_GetItem( list, i ) ) ) {
		PyErr_SetObject( PyExc_TypeError, Py_BuildValue( "sOi",
		   "expected sequence of strings", list, i ));
	   	return NULL;
	   }

	/* a list, definitely of strings */

	bervals = (struct berval**) malloc( 
			(listlen+1) * sizeof(struct berval*) );
	if (bervals==NULL) { 
	    free(lm); 
	    PyErr_NoMemory(); 
	    return NULL; 
	}

	bervals[listlen] = NULL;

	for(i=0; i<listlen; i++) {
	   struct berval *bv;
	   char *val;
	   int len;
	   PyObject *str;

	   str = PySequence_GetItem( list, i );
	   len = PyString_Size( str );
	   val = (char*) malloc( len * sizeof(char) );
	   if (val != NULL)
	       bv = (struct berval*) malloc( sizeof(struct berval) );
	   else
	       bv = NULL;

	   bervals[i] = bv;
	   if (bv==NULL) {
	      if (val!=NULL) free(val);
	      free_LDAPMod( lm );
	      PyErr_NoMemory();
	      return NULL;
	   }
	   memcpy( val, PyString_AsString(str), len * sizeof(char) );
	   bv->bv_val = val;
	   bv->bv_len = len;
	}
    }


    /* bervals is now either NULL or pointer to a completely allocated 
     * list of pointers to duplicated strings  */
    
    lm->mod_bvalues = bervals;
    lm->mod_type = strdup(type);

    if (lm->mod_type == NULL) { 
	free_LDAPMod(lm); 
    	PyErr_NoMemory();
	return NULL; 
    }

    lm->mod_op = op | LDAP_MOD_BVALUES;
    lm->mod_next = NULL;		/* only used in server */

    return lm;
}

/* free the structure allocated in List_to_LDAPMods() */

static void
free_LDAPMods( LDAPMod** lms ) {
    LDAPMod** lmp;
    for ( lmp = lms; *lmp; lmp++ )
    	free_LDAPMod( *lmp );
    free( lms );
}

/* convert a list of tuples into a LDAPMod*[] array structure */

static LDAPMod**
List_to_LDAPMods( PyObject *list, int no_op ) {

    int i, len;
    LDAPMod** lms;

    if (!PySequence_Check(list)) {
	PyErr_SetObject( PyExc_TypeError, Py_BuildValue("sO",
			"expected list of tuples", list ));
    	return NULL;
    }

    len = PySequence_Length(list);
    lms = (LDAPMod**) malloc( (len+1) * sizeof( LDAPMod* ) );
    if (lms==NULL) { PyErr_NoMemory(); return NULL; }
    lms[len] = NULL;

    for( i=0; i<len; i++ ) {
       lms[i] = Tuple_to_LDAPMod( PySequence_GetItem( list, i ), no_op );
       if (lms[i] == NULL) {
	  free_LDAPMods( lms );
	  return NULL;
       }
    }
    return lms;
}

/*
 * convert a python list of strings into an attr list (char*[]).
 * returns 1 if successful, 0 if not (with exception set)
 */

int
attrs_from_List( PyObject *attrlist, char***attrsp ) {

    char **attrs;

    if (PyNone_Check( attrlist )) {
    	attrs = NULL;
    } else if (PySequence_Check( attrlist )) {
	int len = PySequence_Length( attrlist );
	int i;

        attrs = (char**) malloc( (1+len) * sizeof(char*) );
	if (attrs == NULL) { 
	    PyErr_NoMemory();
	    return 0;
	}

	attrs[len] = NULL;
	for(i=0; i<len; i++) {
	    PyObject *s = PySequence_GetItem( attrlist, i );
	    if (!PyString_Check(s)) {
		free(attrs);
		PyErr_SetObject( PyExc_TypeError, Py_BuildValue("sOi",
			  "expected list of strings or None", attrlist,i ));
		return 0;
	    }
	    attrs[i] = PyString_AsString(s);
	}

    } else {
    	PyErr_SetObject( PyExc_TypeError, Py_BuildValue("sO",
			  "expected list of strings or None", attrlist ));
	return 0;
    }

    *attrsp = attrs;
    return 1;
}

/* free memory allocated from above routine */

static void
free_attrs( char*** attrsp ) {
    if (*attrsp != NULL) {
   	free( *attrsp );
	*attrsp = NULL;
    }
}

static void
set_timeval_from_double( struct timeval *tv, double d ) {
	tv->tv_usec = (long) ( fmod(d, 1.0) * 1000000.0 );
	tv->tv_sec = (long) floor(d);
}

/*------------------------------------------------------------
 * methods
 */

/* ldap_unbind */

static PyObject*
l_ldap_unbind( LDAPObject* self, PyObject* args )
{
    if (!PyArg_ParseTuple( args, "")) return NULL;
    if (not_valid(self)) return NULL;
    if ( ldap_unbind( self->ldap ) == -1 )
    	return LDAPerror( self->ldap, "ldap_unbind" );
    self->valid = 0;
    Py_INCREF(Py_None);
    return Py_None;
}

/* ldap_abandon */

static PyObject*
l_ldap_abandon( LDAPObject* self, PyObject* args )
{
    int msgid;

    if (!PyArg_ParseTuple( args, "i", &msgid)) return NULL;
    if (not_valid(self)) return NULL;
    if ( ldap_abandon( self->ldap, msgid ) == -1 )
    	return LDAPerror( self->ldap, "ldap_abandon" );
    Py_INCREF(Py_None);
    return Py_None;
}

/* ldap_add */

static PyObject *
l_ldap_add( LDAPObject* self, PyObject *args )
{
    char *dn;
    PyObject *modlist;
    int msgid;
    LDAPMod **mods;

    if (!PyArg_ParseTuple( args, "sO", &dn, &modlist )) return NULL;
    if (not_valid(self)) return NULL;

    mods = List_to_LDAPMods( modlist, 1 );
    if (mods==NULL) return NULL;

    msgid = ldap_add( self->ldap, dn, mods );
    free_LDAPMods( mods );

    if (msgid == -1)
    	return LDAPerror( self->ldap, "ldap_add_s" );

    return PyInt_FromLong(msgid);
}

/* ldap_add_s */

static PyObject *
l_ldap_add_s( LDAPObject* self, PyObject *args )
{
    char *dn;
    PyObject *modlist;
    int ret;
    LDAPMod **mods;

    if (!PyArg_ParseTuple( args, "sO", &dn, &modlist )) return NULL;
    if (not_valid(self)) return NULL;

    mods = List_to_LDAPMods( modlist, 1 );
    if (mods==NULL) return NULL;

    LDAP_BEGIN_ALLOW_THREADS( self );
    ret = ldap_add_s( self->ldap, dn, mods );
    LDAP_END_ALLOW_THREADS( self );

    free_LDAPMods( mods );

    if (ret != LDAP_SUCCESS) 
    	return LDAPerror( self->ldap, "ldap_add_s" );

    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_bind */

static PyObject*
l_ldap_bind( LDAPObject* self, PyObject* args )
{
    char *who, *cred;
    int method;
    int msgid;

    if (!PyArg_ParseTuple( args, "ssi", &who, &cred, &method)) return NULL;
    if (not_valid(self)) return NULL;
    msgid = ldap_bind( self->ldap, who, cred, method );
    if (msgid == -1)
    	return LDAPerror( self->ldap, "ldap_bind" );
    return PyInt_FromLong( msgid );
}

/* ldap_bind_s */

static PyObject*
l_ldap_bind_s( LDAPObject* self, PyObject* args )
{
    char *who, *cred;
    int method;
    int result;

    if (!PyArg_ParseTuple( args, "ssi", &who, &cred, &method)) return NULL;
    if (not_valid(self)) return NULL;

    LDAP_BEGIN_ALLOW_THREADS( self );
    result =  ldap_bind_s( self->ldap, who, cred, method );
    LDAP_END_ALLOW_THREADS( self );

    if ( result != LDAP_SUCCESS )
    	return LDAPerror( self->ldap, "ldap_bind_s" );
    Py_INCREF(Py_None);
    return Py_None;
}

#ifdef LDAP_REFERRALS

/* ldap_set_rebind_proc */

/* XXX - this could be called when threads are allowed!!! */

  static PyObject* rebind_callback_func = NULL;
  static LDAPObject* rebind_callback_ld = NULL;

  static int rebind_callback( LDAP*ld, char**dnp, char**credp, 
  			      int*methp, int freeit )
  {
      PyObject *result;
      PyObject *args;
      int was_saved;

      char *dn, *cred;
      int  meth;

      if (freeit)  {
      	if (*dnp) free(*dnp);
	if (*credp) free(*credp);
	return LDAP_SUCCESS;
      }

      /* paranoia? */
      if (rebind_callback_ld == NULL)
      	Py_FatalError("rebind_callback: rebind_callback_ld == NULL");
      if (rebind_callback_ld->ldap != ld) 
      	Py_FatalError("rebind_callback: rebind_callback_ld->ldap != ld");
      if (not_valid(rebind_callback_ld)) 
      	Py_FatalError("rebind_callback: ldap connection closed");

      was_saved = (rebind_callback_ld->_save != NULL);

      if (was_saved)
	  LDAP_END_ALLOW_THREADS( rebind_callback_ld );

      args = Py_BuildValue("(O)", rebind_callback_ld);
      result = PyEval_CallObject( rebind_callback_func, args );
      Py_DECREF(args);

      if (result != NULL && 
          !PyArg_ParseTuple( result, "ssi", &dn, &cred, &meth ) )
      {
	  Py_DECREF( result );
          result = NULL;
      }

      if (result == NULL) {
	PyErr_Print();
	if (was_saved) 
	    LDAP_BEGIN_ALLOW_THREADS( rebind_callback_ld );
      	return !LDAP_SUCCESS;
      }

      Py_DECREF(result);
      *dnp = strdup(dn);
      if (!*dnp) return LDAP_NO_MEMORY;
      *credp = strdup(cred);
      if (!*credp) return LDAP_NO_MEMORY;
      *methp = meth;

      if (was_saved) 
      	  LDAP_BEGIN_ALLOW_THREADS( rebind_callback_ld );

      return LDAP_SUCCESS;
  }

static PyObject*
l_ldap_set_rebind_proc( LDAPObject* self, PyObject* args )
{
    PyObject *func;

    if (!PyArg_ParseTuple( args, "O", &func)) return NULL;
    if (not_valid(self)) return NULL;

    if ( PyNone_Check(func) ) {
    	ldap_set_rebind_proc( self->ldap, NULL );
	rebind_callback_func = NULL;
	rebind_callback_ld = NULL;
	Py_INCREF(Py_None);
	return Py_None;
    }

    if ( PyFunction_Check(func) ) {
	rebind_callback_func = func;
	rebind_callback_ld = self;
        ldap_set_rebind_proc( self->ldap, rebind_callback );
	Py_INCREF(Py_None);
	return Py_None;
    }

    PyErr_SetString( PyExc_TypeError, "expected function or None" );
    return NULL;
}

#endif /* LDAP_REFERRALS */

/* ldap_simple_bind */

static PyObject*
l_ldap_simple_bind( LDAPObject *self, PyObject *args ) 
{
    char *who, *cred;
    int msgid;

    if (!PyArg_ParseTuple( args, "zz", &who, &cred )) return NULL;
    if (not_valid(self)) return NULL;
    msgid = ldap_simple_bind( self->ldap, who, cred );
    if ( msgid == -1 )
    	return LDAPerror( self->ldap, "ldap_simple_bind" );
    return PyInt_FromLong( msgid );
}

/* ldap_simple_bind_s */

static PyObject*
l_ldap_simple_bind_s( LDAPObject *self, PyObject *args ) 
{
    char *who, *cred;
    int result;

    if (!PyArg_ParseTuple( args, "zz", &who, &cred )) return NULL;
    if (not_valid(self)) return NULL;
    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_simple_bind_s( self->ldap, who, cred );
    LDAP_END_ALLOW_THREADS( self );
    if ( result != LDAP_SUCCESS )
    	return LDAPerror( self->ldap, "ldap_simple_bind_s" );
    Py_INCREF(Py_None);
    return Py_None;
}

#ifdef WITH_KERBEROS

/* ldap_kerberos_bind_s */

static PyObject*
l_ldap_kerberos_bind_s( LDAPObject *self, PyObject *args ) 
{
    char *who;
    int result;

    if (!PyArg_ParseTuple( args, "s", &who )) return NULL;
    if (not_valid(self)) return NULL;
    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_kerberos_bind_s( self->ldap, who );
    LDAP_END_ALLOW_THREADS( self );
    if ( result != LDAP_SUCCESS )
    	return LDAPerror( self->ldap, "ldap_kerberos_bind_s" );
    Py_INCREF(Py_None);
    return Py_None;
}

/* ldap_kerberos_bind1 */

static PyObject*
l_ldap_kerberos_bind1( LDAPObject *self, PyObject *args ) 
{
    char *who;

    if (!PyArg_ParseTuple( args, "s", &who )) return NULL;
    if (not_valid(self)) return NULL;
    if ( ldap_kerberos_bind1( self->ldap, who ) == -1 )
    	return LDAPerror( self->ldap, "ldap_kerberos_bind1" );
    Py_INCREF(Py_None);
    return Py_None;
}

/* ldap_kerberos_bind1_s */

static PyObject*
l_ldap_kerberos_bind1_s( LDAPObject *self, PyObject *args ) 
{
    char *who;
    int result;

    if (!PyArg_ParseTuple( args, "s", &who )) return NULL;
    if (not_valid(self)) return NULL;
    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_kerberos_bind1_s( self->ldap, who );
    LDAP_END_ALLOW_THREADS( self );
    if ( result != LDAP_SUCCESS )
    	return LDAPerror( self->ldap, "ldap_kerberos_bind1_s" );
    Py_INCREF(Py_None);
    return Py_None;
}

/* ldap_kerberos_bind2 */

static PyObject*
l_ldap_kerberos_bind2( LDAPObject *self, PyObject *args ) 
{
    char *who;

    if (!PyArg_ParseTuple( args, "s", &who )) return NULL;
    if (not_valid(self)) return NULL;
    if ( ldap_kerberos_bind2( self->ldap, who ) == -1 )
    	return LDAPerror( self->ldap, "ldap_kerberos_bind2" );
    Py_INCREF(Py_None);
    return Py_None;
}

/* ldap_kerberos_bind2_s */

static PyObject*
l_ldap_kerberos_bind2_s( LDAPObject *self, PyObject *args ) 
{
    char *who;
    int result;

    if (!PyArg_ParseTuple( args, "s", &who )) return NULL;
    if (not_valid(self)) return NULL;
    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_kerberos_bind2_s( self->ldap, who );
    LDAP_END_ALLOW_THREADS( self );
    if ( result != LDAP_SUCCESS )
    	return LDAPerror( self->ldap, "ldap_kerberos_bind2_s" );
    Py_INCREF(Py_None);
    return Py_None;
}

#endif /* WITH_KERBEROS */

#ifndef NO_CACHE

/* ldap_enable_cache */

static PyObject*
l_ldap_enable_cache( LDAPObject* self, PyObject* args )
{
    long timeout = LDAP_NO_LIMIT;
    long maxmem  = LDAP_NO_LIMIT;

    if (!PyArg_ParseTuple( args, "|ll", &timeout, &maxmem )) return NULL;
    if (not_valid(self)) return NULL;
    if ( ldap_enable_cache( self->ldap, timeout, maxmem ) == -1 )
    	return LDAPerror( self->ldap, "ldap_enable_cache" );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_disable_cache */

static PyObject *
l_ldap_disable_cache( LDAPObject* self, PyObject *args )
{
    if (!PyArg_ParseTuple( args, "" )) return NULL;
    if (not_valid(self)) return NULL;
    ldap_disable_cache( self->ldap );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_set_cache_options */

static PyObject *
l_ldap_set_cache_options( LDAPObject* self, PyObject *args )
{
    long opts;

    if (!PyArg_ParseTuple( args, "l", &opts )) return NULL;
    if (not_valid(self)) return NULL;
    ldap_set_cache_options( self->ldap, opts );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_destroy_cache */

static PyObject *
l_ldap_destroy_cache( LDAPObject* self, PyObject *args )
{
    if (!PyArg_ParseTuple( args, "" )) return NULL;
    if (not_valid(self)) return NULL;
    ldap_destroy_cache( self->ldap );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_flush_cache */

static PyObject *
l_ldap_flush_cache( LDAPObject* self, PyObject *args )
{
    if (!PyArg_ParseTuple( args, "" )) return NULL;
    if (not_valid(self)) return NULL;
    ldap_flush_cache( self->ldap );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_uncache_entry */

static PyObject *
l_ldap_uncache_entry( LDAPObject* self, PyObject *args )
{
    char *dn;

    if (!PyArg_ParseTuple( args, "s", &dn )) return NULL;
    if (not_valid(self)) return NULL;
    ldap_uncache_entry( self->ldap, dn );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_uncache_request */

static PyObject *
l_ldap_uncache_request( LDAPObject* self, PyObject *args )
{
    int msgid;

    if (!PyArg_ParseTuple( args, "i", &msgid )) return NULL;
    if (not_valid(self)) return NULL;
    ldap_uncache_request( self->ldap, msgid );
    Py_INCREF( Py_None );
    return Py_None;
}

#endif /* !NO_CACHE */

/* ldap_compare */

static PyObject *
l_ldap_compare( LDAPObject* self, PyObject *args )
{
    char *dn, *attr, *value;
    int msgid;

    if (!PyArg_ParseTuple( args, "sss", &dn, &attr, &value )) return NULL;
    if (not_valid(self)) return NULL;
    msgid = ldap_compare( self->ldap, dn, attr, value );
    if (msgid == -1)
    	return LDAPerror( self->ldap, "ldap_compare" );
    return PyInt_FromLong( msgid );
}

/* ldap_compare_s */

static PyObject *
l_ldap_compare_s( LDAPObject* self, PyObject *args )
{
    char *dn, *attr, *value;
    int result;

    if (!PyArg_ParseTuple( args, "sss", &dn, &attr, &value )) return NULL;
    if (not_valid(self)) return NULL;
    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_compare_s( self->ldap, dn, attr, value );
    LDAP_END_ALLOW_THREADS( self );
    if (result != LDAP_COMPARE_TRUE && 
        result != LDAP_COMPARE_FALSE )
    	return LDAPerror( self->ldap, "ldap_compare_s" );
    return PyInt_FromLong( result == LDAP_COMPARE_TRUE );
}

/* ldap_delete */

static PyObject *
l_ldap_delete( LDAPObject* self, PyObject *args )
{
    char *dn;
    int msgid;

    if (!PyArg_ParseTuple( args, "s", &dn )) return NULL;
    if (not_valid(self)) return NULL;
    msgid = ldap_delete( self->ldap, dn );
    if (msgid == -1)
    	return LDAPerror( self->ldap, "ldap_delete" );
    return PyInt_FromLong(msgid);
}

/* ldap_delete_s */

static PyObject *
l_ldap_delete_s( LDAPObject* self, PyObject *args )
{
    char *dn;
    int result;

    if (!PyArg_ParseTuple( args, "s", &dn )) return NULL;
    if (not_valid(self)) return NULL;
    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_delete_s( self->ldap, dn );
    LDAP_END_ALLOW_THREADS( self );
    if (result != LDAP_SUCCESS)
    	return LDAPerror( self->ldap, "ldap_delete_s" );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_modify */

static PyObject *
l_ldap_modify( LDAPObject* self, PyObject *args )
{
    char *dn;
    PyObject *modlist;
    int msgid;
    LDAPMod **mods;

    if (!PyArg_ParseTuple( args, "sO", &dn, &modlist )) return NULL;
    if (not_valid(self)) return NULL;

    mods = List_to_LDAPMods( modlist, 0 );
    if (mods==NULL) return NULL;

    msgid = ldap_modify( self->ldap, dn, mods );
    free_LDAPMods( mods );

    if (msgid == -1)
    	return LDAPerror( self->ldap, "ldap_modify" );

    return PyInt_FromLong( msgid );
}

/* ldap_modify_s */

static PyObject *
l_ldap_modify_s( LDAPObject* self, PyObject *args )
{
    char *dn;
    PyObject *modlist;
    int result;
    LDAPMod **mods;

    if (!PyArg_ParseTuple( args, "sO", &dn, &modlist )) return NULL;
    if (not_valid(self)) return NULL;

    mods = List_to_LDAPMods( modlist, 0 );
    if (mods==NULL) return NULL;

    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_modify_s( self->ldap, dn, mods );
    LDAP_END_ALLOW_THREADS( self );

    free_LDAPMods( mods );

    if (result != LDAP_SUCCESS)
    	return LDAPerror( self->ldap, "ldap_modify_s" );

    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_modrdn */

static PyObject *
l_ldap_modrdn( LDAPObject* self, PyObject *args )
{
    char *dn, *newrdn;
    int delold = 1;
    int msgid;

    if (!PyArg_ParseTuple( args, "ss|i", &dn, &newrdn, &delold )) 
    	return NULL;
    if (not_valid(self)) return NULL;

    msgid = ldap_modrdn2( self->ldap, dn, newrdn, delold );
    if (msgid == -1)
    	return LDAPerror( self->ldap, "ldap_modrdn2" );
    return PyInt_FromLong(msgid);
}

/* ldap_modrdn_s */

static PyObject *
l_ldap_modrdn_s( LDAPObject* self, PyObject *args )
{
    char *dn, *newrdn;
    int delold = 1;
    int result;

    if (!PyArg_ParseTuple( args, "ss|i", &dn, &newrdn, &delold )) 
    	return NULL;
    if (not_valid(self)) return NULL;

    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_modrdn2_s( self->ldap, dn, newrdn, delold );
    LDAP_END_ALLOW_THREADS( self );

    if ( result != LDAP_SUCCESS )
    	return LDAPerror( self->ldap, "ldap_modrdn2_s" );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_result */

static PyObject *
l_ldap_result( LDAPObject* self, PyObject *args )
{
    int msgid = LDAP_RES_ANY;
    int all = 1;
    double timeout = -1.0;
    struct timeval tv;
    struct timeval* tvp;
    int result;
    LDAPMessage *msg = NULL;
    PyObject *result_str;

    if (!PyArg_ParseTuple( args, "|iid", &msgid, &all, &timeout ))
    	return NULL;
    if (not_valid(self)) return NULL;
    
    if (timeout >= 0) {
        tvp = &tv;
	set_timeval_from_double( tvp, timeout );
    } else {
    	tvp = NULL;
    }

    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_result( self->ldap, msgid, all, tvp, &msg );
    LDAP_END_ALLOW_THREADS( self );

    if (result == -1)
    	return LDAPerror( self->ldap, "ldap_result" );

    result_str = LDAPconstant( result );

    if (msg == NULL) {
    	return Py_BuildValue("(OO)", result_str, Py_None);
    } else {
	PyObject *pmsg = LDAPmessage_to_python( self->ldap, msg );
	if (pmsg == NULL)  {
	    Py_DECREF( result_str );
	    return NULL;
	}
	return Py_BuildValue("(OO)", result_str, pmsg );
    }
}

/* ldap_search */

static PyObject*
l_ldap_search( LDAPObject* self, PyObject* args )
{
    char *base;
    int scope;
    char *filter;
    PyObject *attrlist = Py_None;
    char **attrs;
    int attrsonly = 0;

    int msgid;

    if (!PyArg_ParseTuple( args, "sis|Oi", 
    	&base, &scope, &filter, &attrlist, &attrsonly)) return NULL;
    if (not_valid(self)) return NULL;

    if (!attrs_from_List( attrlist, &attrs )) 
   	 return NULL;

    msgid = ldap_search( self->ldap, base, scope, filter, 
                             attrs, attrsonly );

    free_attrs( &attrs );

    if (msgid == -1)
    	return LDAPerror( self->ldap, "ldap_search" );

    return PyInt_FromLong( msgid );
}	

/* ldap_search_st */

static PyObject*
l_ldap_search_st( LDAPObject* self, PyObject* args )
{
    char *base;
    int scope;
    char *filter;
    PyObject *attrlist = Py_None;
    char **attrs;
    int attrsonly = 0;
    double timeout = -1.0;
    struct timeval tv, *tvp;
    LDAPMessage *resmsg = NULL;
    int result;

    if (!PyArg_ParseTuple( args, "sis|Oid", 
    	&base, &scope, &filter, &attrlist, &attrsonly, &timeout )) return NULL;
    if (not_valid(self)) return NULL;

    if (timeout >= 0) {
    	tvp = &tv;
	set_timeval_from_double( tvp, timeout );
    } else {
    	tvp = NULL;
    }

    if (!attrs_from_List( attrlist, &attrs )) 
   	 return NULL;

    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_search_st( self->ldap, base, scope, filter, 
                             attrs, attrsonly, tvp, &resmsg );
    LDAP_END_ALLOW_THREADS( self );

    free_attrs( &attrs );

    if (result != LDAP_SUCCESS)
    	return LDAPerror( self->ldap, "ldap_search_st" );

    if (resmsg == NULL) {
    	Py_INCREF( Py_None );
	return Py_None;
    } else {
    	return LDAPmessage_to_python( self->ldap, resmsg );
    }
}	

/* ldap_search_s == ldap_search_st */

/* ldap_ufn_search_c */

/* ldap_ufn_search_ct */

/* ldap_ufn_search_s */

static PyObject*
l_ldap_ufn_search_s( LDAPObject* self, PyObject* args )
{
    char *ufn;
    PyObject *attrlist;
    char **attrs;
    int attrsonly = 0;
    LDAPMessage *resmsg = NULL;
    int result;

    if (!PyArg_ParseTuple( args, "sO|i", 
    	&ufn, &attrlist, &attrsonly)) return NULL;
    if (not_valid(self)) return NULL;

    if (!attrs_from_List( attrlist, &attrs )) 
   	 return NULL;

    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_ufn_search_s( self->ldap, ufn,
                             attrs, attrsonly, &resmsg );
    LDAP_END_ALLOW_THREADS( self );

    free_attrs( &attrs );

    if (result != LDAP_SUCCESS)
    	return LDAPerror( self->ldap, "ldap_ufn_search_s" );

    if (resmsg == NULL) {
    	Py_INCREF( Py_None );
	return Py_None;
    } else {
    	return LDAPmessage_to_python( self->ldap, resmsg );
    }
}	

/* ldap_ufn_setfilter */

static PyObject*
l_ldap_ufn_setfilter( LDAPObject* self, PyObject* args )
{
    char* filter;
    LDAPFiltDesc* res;

    if (!PyArg_ParseTuple( args, "s", &filter)) return NULL;
    if (not_valid(self)) return NULL;
    res = ldap_ufn_setfilter( self->ldap, filter );

    if (res==NULL) {
    	PyErr_SetFromErrno( LDAPerror_object );
	return NULL;
    }

    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_ufn_setprefix */

static PyObject*
l_ldap_ufn_setprefix( LDAPObject* self, PyObject* args )
{
    char* prefix;

    if (!PyArg_ParseTuple( args, "s", &prefix)) return NULL;
    if (not_valid(self)) return NULL;
    ldap_ufn_setprefix( self->ldap, prefix );
    Py_INCREF( Py_None );
    return Py_None;
}

/* ldap_sort_entries */

/* ldap_url_search */

/* ldap_url_search_s */

/* ldap_url_search_st */

static PyObject*
l_ldap_url_search_st( LDAPObject* self, PyObject* args )
{
    char *url;
    int attrsonly = 0;
    LDAPMessage *resmsg;
    int result;
    double timeout = -1.0;
    struct timeval tv, *tvp;

    if (!PyArg_ParseTuple( args, "s|id", &url, &attrsonly, &timeout )) 
    	return NULL;
    if (not_valid(self)) return NULL;

    if (timeout>=0) {
        tvp = &tv;
	set_timeval_from_double( tvp, timeout );
    } else {
    	tvp = NULL;
    }

    LDAP_BEGIN_ALLOW_THREADS( self );
    result = ldap_url_search_st( self->ldap, url, attrsonly, tvp, &resmsg );
    LDAP_END_ALLOW_THREADS( self );

    if (result != LDAP_SUCCESS)
    	return LDAPerror( self->ldap, "ldap_ufn_search_st" );
    
    if (resmsg == NULL) {
    	Py_INCREF( Py_None );
	return Py_None;
    } else {
    	return LDAPmessage_to_python( self->ldap, resmsg );
    }
}

/* methods */

static PyMethodDef methods[] = {
    {"unbind",		(PyCFunction)l_ldap_unbind,		METH_VARARGS},	
    {"unbind_s",	(PyCFunction)l_ldap_unbind,		METH_VARARGS},	
    {"abandon",		(PyCFunction)l_ldap_abandon,		METH_VARARGS},	
    {"add",		(PyCFunction)l_ldap_add,		METH_VARARGS},	
    {"add_s",		(PyCFunction)l_ldap_add_s,		METH_VARARGS},	
    {"bind",		(PyCFunction)l_ldap_bind,		METH_VARARGS},	
    {"bind_s",		(PyCFunction)l_ldap_bind_s,		METH_VARARGS},	
#ifdef LDAP_REFERRALS
    {"set_rebind_proc",	(PyCFunction)l_ldap_set_rebind_proc,	METH_VARARGS},	
#endif /* LDAP_REFERRALS */
    {"simple_bind",	(PyCFunction)l_ldap_simple_bind,	METH_VARARGS},	
    {"simple_bind_s",	(PyCFunction)l_ldap_simple_bind_s,	METH_VARARGS},	
#ifdef WITH_KERBEROS
    {"kerberos_bind_s",	(PyCFunction)l_ldap_kerberos_bind_s,	METH_VARARGS},	
    {"kerberos_bind1",	(PyCFunction)l_ldap_kerberos_bind1,	METH_VARARGS},	
    {"kerberos_bind1_s",(PyCFunction)l_ldap_kerberos_bind1_s,	METH_VARARGS},	
    {"kerberos_bind2",	(PyCFunction)l_ldap_kerberos_bind2,	METH_VARARGS},	
    {"kerberos_bind2_s",(PyCFunction)l_ldap_kerberos_bind2_s,	METH_VARARGS},	
#endif /* WITH_KERBEROS */
#ifndef NO_CACHE
    {"enable_cache",	(PyCFunction)l_ldap_enable_cache,	METH_VARARGS},	
    {"disable_cache",	(PyCFunction)l_ldap_disable_cache,	METH_VARARGS},	
    {"set_cache_options",(PyCFunction)l_ldap_set_cache_options,	METH_VARARGS},	
    {"destroy_cache",	(PyCFunction)l_ldap_destroy_cache,	METH_VARARGS},	
    {"flush_cache",	(PyCFunction)l_ldap_flush_cache,	METH_VARARGS},	
    {"uncache_entry",	(PyCFunction)l_ldap_uncache_entry,	METH_VARARGS},	
    {"uncache_request",	(PyCFunction)l_ldap_uncache_request,	METH_VARARGS},	
#endif /* !NO_CACHE */
    {"compare",		(PyCFunction)l_ldap_compare,		METH_VARARGS},	
    {"compare_s",	(PyCFunction)l_ldap_compare_s,		METH_VARARGS},	
    {"delete",		(PyCFunction)l_ldap_delete,		METH_VARARGS},	
    {"delete_s",	(PyCFunction)l_ldap_delete_s,		METH_VARARGS},	
    {"modify",		(PyCFunction)l_ldap_modify,		METH_VARARGS},	
    {"modify_s",	(PyCFunction)l_ldap_modify_s,		METH_VARARGS},	
    {"modrdn",		(PyCFunction)l_ldap_modrdn,		METH_VARARGS},	
    {"modrdn_s",	(PyCFunction)l_ldap_modrdn_s,		METH_VARARGS},
    {"result",		(PyCFunction)l_ldap_result,		METH_VARARGS},	
    {"search",		(PyCFunction)l_ldap_search,		METH_VARARGS},	
    {"search_s",	(PyCFunction)l_ldap_search_st,		METH_VARARGS},	
    {"search_st",	(PyCFunction)l_ldap_search_st,		METH_VARARGS},	
    {"ufn_search_s",	(PyCFunction)l_ldap_ufn_search_s,	METH_VARARGS},
    {"ufn_setfilter",	(PyCFunction)l_ldap_ufn_setfilter,	METH_VARARGS},
    {"ufn_setprefix",	(PyCFunction)l_ldap_ufn_setprefix,	METH_VARARGS},
    {"url_search_s",	(PyCFunction)l_ldap_url_search_st,	METH_VARARGS},	
    {"url_search_st",	(PyCFunction)l_ldap_url_search_st,	METH_VARARGS},	
    { NULL, NULL }
};

/* representation */

static PyObject*
repr( LDAPObject* self )
{
    static char buf[4096];

#   define STRFMT	"%s%s%s"
#   define STRFMTP(s)                                                  \
    		(s)==NULL?"":"'",                                       \
		(s)==NULL?"(null)":(s),                                 \
		(s)==NULL?"":"'"

#   define LIMITFMT	"%d%s"
#   define LIMITFMTP(v)                                                \
    		(v),                                                    \
		(v)==LDAP_NO_LIMIT?" (NO_LIMIT)":""
    		

    sprintf(buf,
    	"<LDAP {lberoptions:%d, deref:%s, "
	"timelimit:" LIMITFMT ", "
	"sizelimit:" LIMITFMT ", "
	"errno:%d, error:" STRFMT ", "
	"matched:" STRFMT ", refhoplimit:%d, options:< %s%s%s>}>",
	    self->ldap->ld_lberoptions,
	    (
		self->ldap->ld_deref==LDAP_DEREF_NEVER ? "DEREF_NEVER" :
		self->ldap->ld_deref==LDAP_DEREF_SEARCHING ? "DEREF_SEARCHING" :
		self->ldap->ld_deref==LDAP_DEREF_FINDING ? "DEREF_FINDING" :
		self->ldap->ld_deref==LDAP_DEREF_ALWAYS ? "DEREF_ALWAYS" :
							  "*illegal*" 
	    ),
	    LIMITFMTP(self->ldap->ld_timelimit),
	    LIMITFMTP(self->ldap->ld_sizelimit),
	    self->ldap->ld_errno,
	    STRFMTP(self->ldap->ld_error),
	    STRFMTP(self->ldap->ld_matched),
	    self->ldap->ld_refhoplimit,

#ifdef LDAP_DNS
	   (self->ldap->ld_options & LDAP_OPT_DNS ? "OPT_DNS ":""),
#else
	      "",
#endif /* LDAP_DNS */

#ifdef LDAP_REFERRALS
	   (self->ldap->ld_options & LDAP_OPT_REFERRALS ? "OPT_REFERRALS ":""),
#else
	      "",
#endif /* LDAP_REFERRALS */

	   (self->ldap->ld_options & LDAP_OPT_RESTART   ? "OPT_RESTART ":"")
    );
    return PyString_FromString( buf );
}

/* get attribute */

static PyObject*
getattr( LDAPObject* self, char* name ) 
{

	if (streq(name,"lberoptions")) 
		return PyInt_FromLong(self->ldap->ld_lberoptions);
	if (streq(name,"deref")) 
		return PyInt_FromLong(self->ldap->ld_deref);
	if (streq(name,"timelimit")) 
		return PyInt_FromLong(self->ldap->ld_timelimit);
	if (streq(name,"sizelimit")) 
		return PyInt_FromLong(self->ldap->ld_sizelimit);
	if (streq(name,"errno")) 
		return PyInt_FromLong(self->ldap->ld_errno);
	if (streq(name,"error")) 
		return PyString_FromString(self->ldap->ld_error);
	if (streq(name,"matched")) 
		return PyString_FromString(self->ldap->ld_matched);
	if (streq(name,"refhoplimit")) 
		return PyInt_FromLong(self->ldap->ld_refhoplimit);
	if (streq(name,"options")) 
		return PyInt_FromLong(self->ldap->ld_options);
	if (streq(name,"valid")) 
		return PyInt_FromLong(self->valid);

	return Py_FindMethod( methods, (PyObject*)self, name );
	return NULL;
}

/* set attribute */

static int
setattr( LDAPObject* self, char* name, PyObject* value ) 
{
	long intval;

	if (streq(name,"errno") ||
	    streq(name,"error") ||
	    streq(name,"valid") ||
	    streq(name,"matched"))
	{
	    PyErr_SetString( PyExc_AttributeError, "read-only attribute" );
	    return -1;
    	}

	if (!PyArg_Parse( value, "i", &intval )) {
	    PyErr_SetString( PyExc_TypeError, "expected integer" );
	    return -1;
	}

#       define set(a,max)                                          \
	if (streq(name,#a)) {                                       \
	    if (intval < 0 || intval > max )                        \
	    {                                                       \
		PyErr_SetString( PyExc_ValueError, "value out of range" );\
		return -1;                                          \
	    }                                                       \
	    self->ldap->ld_##a = intval;                            \
	    return 0;                                               \
	}

	set(lberoptions, ~(unsigned char)0)
	set(deref, ~0UL)
	set(timelimit, ~0UL)
	set(sizelimit, ~0UL)
	set(refhoplimit, ~0UL)
	set(options, LDAP_OPT_RESTART)

	/* it fell through to here */
	PyErr_SetString( PyExc_NameError, "cannot set that field" );
	return -1;
}

/* type entry */

PyTypeObject LDAP_Type = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,                      /*ob_size*/
	"LDAP",                 /*tp_name*/
	sizeof(LDAPObject),     /*tp_basicsize*/
	0,                      /*tp_itemsize*/
	/* methods */
	(destructor)dealloc,	/*tp_dealloc*/
	0,                      /*tp_print*/
	(getattrfunc)getattr,	/*tp_getattr*/
	(setattrfunc)setattr,	/*tp_setattr*/
	0,                      /*tp_compare*/
	(reprfunc)repr,         /*tp_repr*/
	0,                      /*tp_as_number*/
	0,                      /*tp_as_sequence*/
	0,                      /*tp_as_mapping*/
	0,                      /*tp_hash*/
};
