/* NewsCache.cc 
 * -- (c) 1996-1998 by Thomas GSCHWIND <tom@infosys.tuwien.ac.at>
 * -- implements the cache server in combination with the NServer library
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "config.h"

#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <signal.h>
#include <sys/socket.h>
#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <limits.h>
#ifdef HAVE_LIBWRAP
#include <tcpd.h>
#endif

#include <iostream.h>

#include "NServer.h"
#include "Newsgroup.h"

#include "setugid.h"
#include "Config.h"
#include "Logger.h"

/* CONF_MultiClient
 * Allow several simultaneous clients in standalone mode.
 * This switch is turned off for debug purposes mainly.
 */
#define CONF_MultiClient

#ifndef SOCKLEN_TYPE
#define SOCKLEN_TYPE size_t
#endif

Logger slog;
const char *cmnd;
const char *opt_configfile=NULL;
int opt_inetd=0;
int opt_debug=0;
Config Cfg;
int Xsignal;

typedef struct ClientData {
  struct sockaddr_in sock;
  SOCKLEN_TYPE socklen;
  char client_name[512];
  char client_logname[512];
  istream *ci;
  ostream *co;

  CServer *srvr;
  char groupname[MAXGROUPNAMELEN];
  Newsgroup *grp;
  int nbr;
  int stat_groups;
  int stat_artingrp;
  int stat_articles;
};

GroupInfo *selectgroup(ClientData *clt, const char *group)
{
  GroupInfo *gi;

  try {
    gi=clt->srvr->groupinfo(group);
  } catch(NSError &nse) {
    return NULL;
  } catch(...) {
    return NULL;
  }
  // Reset Article Pointer
  if(gi->first()<=gi->last()) clt->nbr=gi->first();
  else clt->nbr=-1;
  if(strcmp(clt->groupname,group)!=0) {
    if(clt->grp) clt->srvr->freegroup(clt->grp);
    clt->grp=NULL;
    clt->stat_groups++;
    if(clt->stat_artingrp) {
      slog.p(Logger::Notice) << clt->client_logname << " group " << clt->groupname << " " << clt->stat_artingrp << "\n";
      clt->stat_artingrp=0;
    }
    strcpy(clt->groupname,group);
  }
  return gi;
}

// Article 
int ns_article(ClientData *clt,int argc,char *argv[])
{
  if(clt->groupname[0]=='\0') {
    (*clt->co) << "412 no newsgroup has been selected\r\n";
    return -1;
  }
  //FIX! Retrieving an article by msg-id is currently not 
  //FIX! implemented
  //FIX! We do not check the 2nd argument correctly
  if(argc>2) {
    switch(argv[0][0]) {
    case 'a':
      (*clt->co) << "501 Syntax: article [nbr]\r\n";
      return -1;
    case 'h':
      (*clt->co) << "501 Syntax: head [nbr]\r\n";
      return -1;
    case 'b':
      (*clt->co) << "501 Syntax: body [nbr]\r\n";
      return -1;
    case 's':
      (*clt->co) << "501 Syntax: stat [nbr]\r\n";
      return -1;
    }
  }
  try {
    if(!clt->grp) clt->grp=clt->srvr->getgroup(clt->groupname);
  } catch(NoSuchGroupError &nsge) {
    (*clt->co) << "411 no such news group\r\n";
    return -1;
  } catch(Error &e) {
    (*clt->co) << "412 operation failed\r\n";
    return -1;
  }

  if(argc==1 && clt->nbr<0) {
    (*clt->co) << "420 no current article has been selected\r\n";
    return -1;
  }

  int nbr=(argc==2)?atoi(argv[1]):clt->nbr;
  Article *a;
  string aid;

  ASSERT(clt->grp->testdb());
  if(argv[0][0]!='s') {
    clt->stat_articles++;
    clt->stat_artingrp++;
  }
  if((a=clt->grp->getarticle(nbr))==NULL) {
    (*clt->co) << "423 no such article number in this group\r\n";
    return -1;
  }

  aid=a->getfield("Message-ID:");
  clt->nbr=nbr;
  
  switch(argv[0][0]) {
  case 'a':
    (*clt->co) << "220 " << nbr << " " << aid 
	       << " article retrieved - head and body follow\r\n";
    (*clt->co) << *a << ".\r\n";
    slog.p(Logger::Info) << "articlesize " << clt->groupname << ":" << nbr 
			 << " " << a->length() << "\n";
    break;
  case 'h':
    (*clt->co) << "221 " << nbr << " " << aid 
	       << " article retrieved - head follows\r\n";
    a->write(*clt->co,Article::Head);
    (*clt->co) << ".\r\n";
    break;
  case 'b':
    (*clt->co) << "222 " << nbr << " " << aid 
	       << " article retrieved - body follows\r\n";
    a->write(*clt->co,Article::Body);
    (*clt->co) << ".\r\n";
    break;
  case 's':
    (*clt->co) << "223 " << nbr << " " << aid 
	       << " article retrieved - request text separately\r\n";
    break;
  }
  clt->grp->freearticle(a);
  return 0;
}

/*
 */
int ns_date(ClientData *clt,int argc,char *argv[])
{
//   struct timeval tv;
//   struct timezone tz;
  struct tm *ltm;
  time_t conv;
  char buf[256];

  if(argc!=1) {
    (*clt->co) << "501 Syntax: group Newsgroup\r\n";
    return -1;
  }

//   if(gettimeofday(&tv,&tz)<0) {
//     (*clt->co) << "500 gettimeofday failed\r\n";
//     return -1;
//   }
  time(&conv);
  if((ltm=localtime(&conv))==NULL) {
    (*clt->co) << "500 localtime failed\r\n";
    return -1;
  }
  sprintf(buf,"111 %04d%02d%02d%02d%02d%02d\r\n",
	  1900+ltm->tm_year,ltm->tm_mon,ltm->tm_mday,
	  ltm->tm_hour,ltm->tm_min,ltm->tm_sec);
  (*clt->co) << buf;

  return 0;
}

/* Group name
 * Find the news server responsible for this newsgroup.
 * Connect to it and select the newsgroup.
 */
int ns_group(ClientData *clt,int argc,char *argv[])
{
  if(argc!=2) {
    (*clt->co) << "501 Syntax: group Newsgroup\r\n";
    return -1;
  }

  GroupInfo *gi;
  if((gi=selectgroup(clt,argv[1]))==NULL) {
    (*clt->co) << "411 no such news group\r\n";
    return -1;
  }
  (*clt->co) << "211 " 
	     << gi->n() << " " 
	     << gi->first() << " " 
	     << gi->last() << " "
	     << gi->name() << " group selected\r\n";
  
  return 0;
}

/* help
 * lists the available commands
 */
int ns_help(ClientData *clt,int argc,char *argv[])
{
  
  argc=0;
  argv=NULL;

  (*clt->co) << "100 Legal commands\r\n"
	     << "  article [Number]\r\n"
	     << "  body [Number]\r\n"
	     << "  date\r\n"
	     << "  group newsgroup\r\n"
	     << "  head [Number]\r\n"
	     << "  help\r\n"
	     << "  last\r\n"
	     << "  list [active [wildmat]|overview.fmt]\r\n"
	     << "  listgroup [newsgroup]\r\n"
	     << "  mode reader\r\n"
// 	     << "  newgroups date time [GMT] [<WILDMAT>]\r\n"
	     << "  next\r\n"
	     << "  over [range]\r\n"
	     << "  post\r\n"
	     << "  quit\r\n"
	     << "  stat [Number]\r\n"
	     << "  xhdr header [range]"
	     << "  xover [range]\r\n"
	     << "Report problems to " << Cfg.Admin << "\r\n"
	     << ".\r\n";
  return 0;
}

/* last,next */
int ns_lastnext(ClientData *clt,int argc,char *argv[])
{
  if(argc!=1) {
    switch(argv[0][0]) {
    case 'l':
      (*clt->co) << "501 Syntax: last\r\n";
      return -1;
    case 'n':
      (*clt->co) << "501 Syntax: next\r\n";
      return -1;
    }
  }
  if(clt->groupname[0]=='\0') {
    (*clt->co) << "412 No newsgroup currently selected\r\n";
    return -1;
  }
  if(clt->nbr<0) {
    (*clt->co) << "420 no current article has been selected\r\n";
    return -1;
  }
  try {
    if(!clt->grp) clt->grp=clt->srvr->getgroup(clt->groupname);
  } catch(NoSuchGroupError &nsge) {
    (*clt->co) << "411 no such news group\r\n";
    return -1;
  } catch(Error &e) {
    (*clt->co) << "412 operation failed\r\n";
    return -1;
  }

  unsigned int f, i=clt->nbr, l;
  const char *o;
  string aid;
  OverviewFmt *ofmt=clt->srvr->overviewfmt();
  clt->grp->getsize(&f,&l);
  switch(argv[0][0]) {
  case 'l':
    for(;;) {
      i--;
      if(i<f) {
	(*clt->co) << "422 No previous to retrieve\r\n";
	return -1;
      }
      o=clt->grp->getover(i);
      if(o[0]) {
	aid=ofmt->getfield(o,"Message-ID:",0);
	break;
      }
    }
    break;
  case 'n':
    for(;;) {
      ++i;
      if(i>l) {
	(*clt->co) << "421 No next to retrieve\r\n";
	return -1;
      }
      o=clt->grp->getover(i);
      if(o[0]) {
	aid=ofmt->getfield(o,"Message-ID:",0);
	break;
      }
    }
  }
  clt->nbr=i;

  (*clt->co) << "223 " << clt->nbr << " " << aid 
	     << " article retrieved - request text separately\r\n";
  return 0;
}

/* List [active|newsgroups|overview.fmt]
 * List the specified data
 */
int ns_list(ClientData *clt,int argc,char *argv[])
{
  //FIX! list active [WILDMAT] not supported
  //FIX! list newsgroups currently returns an empty list
  if(argc<=3) {
    if(argc==1 || strcasecmp(argv[1],"active")==0) {
      ActiveDB *active;
      try {
	active=clt->srvr->active();
      } catch(Error &e) {
	(*clt->co) << "410 operation failed\r\n";
	return -1;
      }	
      (*clt->co) << "215 List of newsgroups (Group High Low Flags) follows\r\n";
      if(argc==3) active->write((*clt->co),0,NVActiveDB::m_active,argv[2]);
      else active->write((*clt->co));
      (*clt->co) << ".\r\n";
      return 0;
    } else if(strcasecmp(argv[1],"active.times")==0) {
      ActiveDB *active;
      try {
	active=clt->srvr->active();
      } catch(Error &e) {
	(*clt->co) << "410 operation failed\r\n";
	return -1;
      }	
      (*clt->co) << "215 information follows\r\n";
      if(argc==3) active->write((*clt->co),0,NVActiveDB::m_times,argv[2]);
      else active->write((*clt->co),0,NVActiveDB::m_times);
      (*clt->co) << ".\r\n";
      return 0;
    } else if(strcasecmp(argv[1],"newsgroups")==0 && argc==2) {
      (*clt->co) << "215 Don't have any data.\r\n.\r\n";
      return 0;
    } else if(strcasecmp(argv[1],"overview.fmt")==0 && argc==2) {
      (*clt->co) << "215 Order of fields in overview database\r\n"
	         << *(clt->srvr->overviewfmt()) << ".\r\n";
      return 0;
    }
  }

  (*clt->co) << "501 Syntax: list [active|active.times|newsgroups|overview.fmt]\r\n";
  return -1;
}

/* listgroup
 * Listgroup only works for groups, where the overview
 * database has already been cached!
 */
int ns_listgroup(ClientData *clt,int argc,char *argv[])
{
  if(argc>2) {
    (*clt->co) << "501 Syntax: listgroup [Newsgroup]\r\n";
    return -1;
  }
  if(argc==1 && clt->groupname[0]=='\0') {
    (*clt->co) << "412 Not currently in a newsgroup\r\n";
    return -1;
  }

  if(argc==2) {
    if(selectgroup(clt,argv[1])==NULL) {
      (*clt->co) << "411 no such news group\r\n";
      return -1;
    }
  }
  try {
    if(!clt->grp) clt->grp=clt->srvr->getgroup(clt->groupname);
  } catch(NoSuchGroupError &nsge) {
    (*clt->co) << "411 no such news group\r\n";
    return -1;
  } catch(Error &e) {
    (*clt->co) << "412 operation failed\r\n";
    return -1;
  }
  
  (*clt->co) << "211 list of article numbers follow\r\n";
  clt->grp->printlistgroup(*clt->co);
  (*clt->co) << ".\r\n";
  return 0;
}

int ns_mode(ClientData *clt,int argc,char *argv[])
{
  if(argc!=2 || 
     (strcasecmp(argv[1],"reader")!=0 && 
      strcasecmp(argv[1],"query")!=0)) {
    (*clt->co) << "501 Syntax: mode (reader|query)\r\n";
    return -1;
  }
  (*clt->co) << "200 "PACKAGE" "VERSION", accepting NNRP commands\r\n";
  return 0;
}

int ns_newgroups(ClientData *clt,int argc,char *argv[]) 
{
  int err=0;

  if(argc<3 || argc>5) {
    err=1;
  } else {
    int i;
    struct tm rtm;

    for(i=0;isdigit(argv[1][i]);i++);
    if((i!=8 && i!=6) || argv[1][i]) err=1;
    for(i=0;isdigit(argv[2][i]);i++);
    if(i!=6 || argv[2][i]) err=1;
    //FIX! 3rd and 4th argument are ignored currently
  }
  if(err) {
    (*clt->co) << "501 Syntax:  NEWGROUPS date time [GMT] [<wildmat>]\r\n";
    return -1;
  }
  
  (*clt->co) << "231 list of new newsgroups follows\r\n.\r\n";
  return 0;
}

int ns_post(ClientData *clt,int argc,char *argv[])
{
  if(argc!=1) {
    (*clt->co) << "501 Syntax: post\r\n";
    return -1;
  }

  Article art;
  (*clt->co) << "340 send article to be posted.\r\n";
  art.read(*clt->ci);

  try {
    clt->srvr->post(&art);
    (*clt->co) << "240 Article posted\r\n";
  } catch (InvalidArticleError &iae) {
    (*clt->co) << "441 invalid article\r\n";
    return -1;
  } catch (NotAllowedError &nae) {
    (*clt->co) << "440 posting not allowed\r\n";
    return -1;
  } catch (Error &e) {
    (*clt->co) << "449 operation failed\r\n";
    return -1;
  }
  return 0;
}

int ns_quit(ClientData *clt,int argc,char *argv[])
{
  (*clt->co) << "205 Good bye\r\n";
  return 1;
}

/*
 * 221 Header follows
 * 412 No news group current selected
 * 420 No current article selected
 * 430 no such article
 * 502 no permission
 */
int ns_xover(ClientData *clt,int argc,char *argv[])
{
  int i=1,fst,lst;
  char buf[513];
  char *p;

  if(clt->groupname[0]=='\0') {
    (*clt->co) << "412 No newsgroup currently selected\r\n";
    return -1;
  }
  if(strcmp(argv[0],"xhdr")==0) { i=2; }
  switch(argc-i) {
  case 0:
    fst=lst=clt->nbr;
    break;
  case 1:
    p=argv[i];
    if(isdigit(*p)) fst=lst=strtol(argv[i],&p,10);
    if((*p)=='\0') break;
    if((*p)=='-') {
      lst=UINT_MAX;
      p++;
      if(isdigit(*p)) lst=strtol(p,&p,10);
      if((*p)=='\0') break;
    }
  default:
    (*clt->co) << "501 Syntax: xover [range]\r\n";
    return -1;
  } 
  try {
    if(!clt->grp) clt->grp=clt->srvr->getgroup(clt->groupname);
  } catch(NoSuchGroupError &nsge) {
    (*clt->co) << "411 no such news group\r\n";
    return -1;
  } catch(Error &e) {
    e.print();
    (*clt->co) << "412 operation failed\r\n";
    return -1;
  }

  ASSERT(clt->grp->testdb());
  switch(i) {
  case 1:
    (*clt->co) << "224 Overview information follows\r\n";
    clt->grp->printoverdb(*clt->co,fst,lst);
    break;
  case 2:
    (*clt->co) << "221 " << p << "[" << fst << "-" << lst << "]\r\n";
    clt->grp->printheaderdb(*clt->co,argv[1],fst,lst);
    break;
  }
  (*clt->co) << ".\r\n";
  return 0;
}

int ns_xttl(ClientData *clt,int argc,char *argv[])
{
  if(argc>3) {
    (*clt->co) << "501 Syntax: xttl list desc group\r\n";
    return -1;
  }

  clt->srvr->setttl(atoi(argv[1]),atoi(argv[2]));
  (*clt->co) << "280 New timeouts have been set for this session\r\n";
  return 0;
}

typedef struct _nnrp_command_t {
  const char *name;
  int (*func)(ClientData*,int,char*[]);
} nnrp_command_t;

static nnrp_command_t nnrp_commands[]=
{
  {"article",ns_article},
  {"head",ns_article},
  {"body",ns_article},
  {"stat",ns_article},

  {"date",ns_date},
  {"group",ns_group},
  {"help",ns_help},
  {"last",ns_lastnext},
  {"list",ns_list},
  {"listgroup",ns_listgroup},
  {"mode",ns_mode},
  {"newgroups",ns_newgroups},
  {"next",ns_lastnext},
  {"over",ns_xover},
  {"post",ns_post},
  // stat: 
  //   Necessary for tin to detect, whether the connection timed out
  // last,next should we implement these commands?
  {"quit",ns_quit},
  {"xhdr",ns_xover},
  {"xover",ns_xover},
  //  {"xttl",ns_xttl},
  
  {NULL,NULL}
};

#ifdef HAVE_LIBWRAP
#ifndef WITH_SYSLOG
/* This is from my syslog.h */
#define LOG_EMERG       0       /* system is unusable */
#define LOG_ALERT       1       /* action must be taken immediately */
#define LOG_CRIT        2       /* critical conditions */
#define LOG_ERR         3       /* error conditions */
#define LOG_WARNING     4       /* warning conditions */
#define LOG_NOTICE      5       /* normal but significant condition */
#define LOG_INFO        6       /* informational */
#define LOG_DEBUG       7       /* debug-level messages */
#endif

/* needed for TCP wrappers */
int allow_severity = LOG_INFO;
int deny_severity = LOG_NOTICE;
#endif

/* nnrpd(fd)
 */
void nnrpd(int fd)
{
  struct hostent *he;
  ClientData clt;
  char req[1024],oreq[1024],*rp;
  char *argv[256];
  int argc;
  const char *com;
  nnrp_command_t *p;
  int errc=0;
  ifstream ifs;
  ofstream ofs;
  fd_set fset;
  struct timeval tv;
  int idle;
  
  if(fd>=0) {
    ofs.attach(fd);
    clt.co=&ofs;

    ifs.attach(fd);
    ifs.unsetf(ios::skipws);
    ifs.tie(&ofs);
    clt.ci=&ifs;
  } else {
    cin.unsetf(ios::skipws);
    clt.ci=&cin;
    clt.co=&cout;
    fd=0;
  }

  // Get the network address and name of the client
  clt.socklen=sizeof(clt.sock);
  if(getpeername((fd>=0)?fd:0,(struct sockaddr*)&clt.sock,&clt.socklen)<0) {
    // Cannot get socket --- connection from stdin?
    strcpy(clt.client_name,"stdin");
    strcpy(clt.client_logname,"stdin [0]");
  } else {
    he=gethostbyaddr((const char *)&(clt.sock.sin_addr), 
		     sizeof(clt.sock.sin_addr), 
		     AF_INET);
    if(he) {
      strncpy(clt.client_name,he->h_name,sizeof(clt.client_name));
      clt.client_name[sizeof(clt.client_name)-1]='\0';    
    } else {
      strncpy(clt.client_name,
	      inet_ntoa(clt.sock.sin_addr),
	      sizeof(clt.client_name));
      clt.client_name[sizeof(clt.client_name)-1]='\0';
    }
    if(Cfg.LogStyle&Config::LogName) {
      strcpy(clt.client_logname,clt.client_name);
    } else {
      clt.client_logname[0]='\0';
    }
    if(Cfg.LogStyle&Config::LogAddr) {
      if(clt.client_logname[0]) {
	if(sizeof(clt.client_logname)-strlen(clt.client_logname)-1>=strlen(inet_ntoa(clt.sock.sin_addr))+3) {
	  strcat(clt.client_logname," [");
	  strcat(clt.client_logname,inet_ntoa(clt.sock.sin_addr));
	  strcat(clt.client_logname,"]");
	}
      } else {
	strncpy(clt.client_name,
		inet_ntoa(clt.sock.sin_addr),
		sizeof(clt.client_name));
	clt.client_name[sizeof(clt.client_name)-1]='\0';
      }
    }

#ifdef HAVE_LIBWRAP    
    // Check the hosts_access configuration; emulate INN error message
    if (!hosts_ctl(PACKAGE,he->h_name, inet_ntoa(clt.sock.sin_addr), STRING_UNKNOWN)){
      slog.p(Logger::Notice) << clt.client_logname << " denied\n";
      (*clt.co) << "502 You have no permission to talk.  Goodbye.\n";
      return;
    }
#endif
  }

  slog.p(Logger::Notice) << clt.client_logname << " connect\n";
  
  try {
    clt.srvr=new CServer(Cfg.SpoolDirectory,&(Cfg.srvrs));
    clt.srvr->setttl(Cfg.ttl_list,Cfg.ttl_desc);
    clt.grp=NULL;
    clt.groupname[0]='\0';
    clt.stat_groups=0;
    clt.stat_artingrp=0;
    clt.stat_articles=0;
  } catch(SystemError &se) {
    slog.p(Logger::Alert) << "CServer failed, please check the permissions on the spool directory\n";
    se.print();
    (*clt.co) << "400 "PACKAGE" "VERSION", service not available\r\n";
    exit(1);
  }

  (*clt.co) << "200 "PACKAGE" "VERSION", accepting NNRP commands\r\n";
  do {
    flush(*clt.co);
    // Read command
    idle=0;
    for(;;) {
      if(Xsignal>=0) {
	(*clt.co) << "400 "PACKAGE" "VERSION", service discontinued\r\n";
	goto client_exit;
      }
      FD_ZERO(&fset);
      FD_SET(fd,&fset);
      tv.tv_sec=3;
      tv.tv_usec=0;
      if(select(fd+1,&fset,NULL,NULL,&tv)) {
	clt.ci->getline(req,sizeof(req),'\n');
	break;
      } else {
	idle+=3;
	if(idle>Cfg.ClientTimeout) goto client_exit;
      }
    }
    if(!clt.ci->good()) break;
    rp=req+strlen(req);
    while(rp>req) {
      rp--;
      if(!isspace(*rp)) break;
      else *rp='\0';
    } 
    strcpy(oreq,req);
    slog.p(Logger::Info) << clt.client_logname << " " << oreq << "\n";

    // Split command into arguments
    for(rp=req, argc=0;*rp && argc<256;rp++) {
      if(isspace(*rp)) {
	*rp='\0';
      } else {
	if(rp==req || *(rp-1)=='\0') argv[argc++]=rp;
	if(argc==1) *rp=tolower(*rp);
      }
    }
    if(!argc) continue;
    if(argc==256) {
      (*clt.co) << "500 Line too long\r\n";
      continue;
    }

    // Call function for command
    com=argv[0];
    p=nnrp_commands;
    while(p->name) {
      if (strcasecmp(p->name,com)==0) {
	errc=p->func(&clt,argc,argv);
	if(!(Cfg.LogStyle&Config::LogINN)) {
	  if(errc<0) {
	    slog.p(Logger::Notice) << clt.client_logname 
				  << " failed " << oreq << "\n";
	  }
	}
	break; 
      }
      p++;
    }
    if(p->name==NULL) {
      slog.p(Logger::Notice) << clt.client_logname 
			    << " unrecognized " << oreq << "\n";
      (*clt.co) << "500 What?\r\n";
    }
  } while(errc<=0);
client_exit:
  if(clt.stat_artingrp) {
    slog.p(Logger::Notice) << clt.client_logname 
			  << " group " << clt.groupname 
			  << " " << clt.stat_artingrp << "\n";
    clt.stat_artingrp=0;
  }
  slog.p(Logger::Notice) << clt.client_logname 
			<< " exit articles " << clt.stat_articles 
			<< " groups " << clt.stat_groups << "\n";
//   clt.co->close();
  delete clt.srvr;
}

void nntpd()
{
  int sock;
  struct sockaddr_in nproxy;
  const char *cp;
  struct servent *cport;

  struct sockaddr_in clt_sa;
  // This has to be changed oon an alpha --- see man accept
  SOCKLEN_TYPE clt_salen;
  int clt_fd;
  int clt_pid;
    
  if ((sock=socket(AF_INET, SOCK_STREAM, 0))<0) {
    if(opt_debug) cerr << "socket failed: " << strerror(errno) << "\n";
    slog.p(Logger::Error) << "socket failed: " << strerror(errno) << "\n";
    exit(1);
  }
  
  { /* set some socket options */
    int one=1;
    if(setsockopt(sock,SOL_SOCKET,SO_KEEPALIVE,
		  (char*)&one,sizeof(int))<0) {
      slog.p(Logger::Error) << "setsockopt failed: " << strerror(errno) << "\n";
    }
    if(setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,
		  (char*)&one,sizeof(int))<0) {
      slog.p(Logger::Error) << "setsockopt failed: " << strerror(errno) << "\n";
    }
  }
  nproxy.sin_family=AF_INET;
  nproxy.sin_addr.s_addr=INADDR_ANY; 
  
  cp=Cfg.CachePort;
  if(cp[0]!='#') {
    if ((cport=getservbyname(cp,"tcp"))==NULL) {
      if(opt_debug) cerr << cmnd << ": Can't resolve service " << cp << "/tcp\n";
      slog.p(Logger::Error) << cmnd << ": Can't resolve service " << cp << "/tcp\n";
      exit(1);
    }
    nproxy.sin_port=cport->s_port;
  } else {
    nproxy.sin_port=htons(atoi(cp+1));
  }
  if(opt_debug) {
    cerr << cmnd << ": Binding to port " << ntohs(nproxy.sin_port) 
         << " (" << cp << ")\n";
  }

  if (bind(sock, (struct sockaddr *)&nproxy, sizeof(nproxy))<0) {
    if(opt_debug) cerr << "can't bind socket: " << strerror(errno) << "\n";
    slog.p(Logger::Error) << "can't bind socket: " << strerror(errno) << "\n";
    exit(1);
  }

  { /* store pid in Cfg.PidFile */
    ofstream pid(Cfg.PidFile);
    pid << getpid() << endl;
    if(!pid.good()) {
      slog.p(Logger::Warning) << "cannot open pid file\n";
    }
  }

  setugid(Cfg.Username,Cfg.Groupname);
  listen(sock,4);
  for(;;) {
    /* Accept connection */
    clt_salen=sizeof(clt_sa);
    do {
      errno=0;
      clt_fd=accept(sock,(struct sockaddr *)&clt_sa,&clt_salen);
      if(Xsignal>=0) {
	close(sock);
	exit(0);
      }
    } while(errno==EINTR);
    if(clt_fd<0) {
      if(opt_debug) cerr << "accept failed: " << strerror(errno) << "\n";
      slog.p(Logger::Error) << "accept failed: " << strerror(errno) << "\n";
      exit(1);
    }
    if(opt_debug) {
      cerr << "Connection from "
	   << inet_ntoa(clt_sa.sin_addr) << ":" << ntohs(clt_sa.sin_port)
	   << "\n";
    }
    
#ifdef CONF_MultiClient
    if((clt_pid=fork())<0) {
      if(opt_debug) cerr << "fork failed: " << strerror(errno) << "\n";
      slog.p(Logger::Error) << "fork failed: " << strerror(errno) << "\n";
      write(clt_fd,"503 cannot create process\r\n",28);
      close(clt_fd);
      continue;
    }
    //Success
    if(clt_pid==0) {
      //Child
      close(sock);
      nnrpd(clt_fd);
      close(clt_fd);
      exit(0);
    }
    //Parent
    close(clt_fd);
#else
    nnrpd(clt_fd);
    close(clt_fd);
#endif
  }
  
  close(sock);
}

void sigchld(int num)
{
  waitpid(0,NULL,WNOHANG);
  signal(SIGCHLD,sigchld);
}

void catchsignal(int num)
{
  Xsignal=num;
  signal(num,catchsignal);
}

main(int argc, char **argv)
{
#ifndef WITH_SYSLOG
  char logfile[MAXPATHLEN];
  time_t t;
  pid_t p;
#endif
  char conffile[MAXPATHLEN];
  int ai=1,aerr=0;

//   strcpy(ServerRoot,SYSCONFDIR);
//   if(chdir(ServerRoot)<0) {
//     cerr << "Cannot change to " << ServerRoot << endl;
//     exit(1);
//   }

  cmnd=argv[0];
  while(ai<argc && !aerr) {
    if(argv[ai][0]!='-' || !argv[ai][1] || argv[ai][2]) break;
    switch(argv[ai][1]) {
    case '-':
      if(strcmp(argv[ai],"--version")==0) {
	cout << PACKAGE" "VERSION"\n";
	exit(0);
      }
    case 'c':
      if(opt_configfile) aerr=1;
      opt_configfile=argv[ai+1];
      ai+=2;
      break;
    case 'd':
      opt_debug++;
      ai++;
      break;
    case 'i':
      if(opt_inetd) aerr=1;
      opt_inetd=1;
      ai++;
      break;
    default:
      aerr=1;
      break;
    }
  }

  if(aerr || ai!=argc) {
    cerr << "Usage: " << cmnd << " [options]\n";
    cerr << "Option          Description\n";
    cerr << "-c config-file  Read configuration from config-file\n";
    cerr << "-i              "PACKAGE" is started from inetd\n";
    cerr << "-d              Do not detach from controlling tty\n";
    exit(1);
  }
    
  if(opt_configfile) {
    strcpy(conffile,opt_configfile);
  } else {
    sprintf(conffile,"%s/newscache.conf",SYSCONFDIR);
  }
  try {
    Cfg.read(conffile);
  } catch(IOError &io) {
    cerr << "unexpected EOF in " << conffile << "\n";
    exit(-1);
  } catch(SyntaxError &se) {
    cerr << se._errtext << "\n";
    exit(-1);
  }

#ifdef WITH_SYSLOG
  slog.open(PACKAGE,LOG_NDELAY|LOG_PID,LOG_NEWS);
#else
  time(&t);
  p=getpid();
  sprintf(logfile,"%s/Trace-%ld-%d",Cfg.LogDirectory,t,p);
  slog.open(logfile);
#endif

  // signal
  Xsignal=-1;
  signal(SIGHUP,catchsignal);
  signal(SIGINT,catchsignal);
  signal(SIGPIPE,catchsignal);
  signal(SIGALRM,catchsignal);
  signal(SIGTERM,catchsignal);
  signal(SIGUSR1,catchsignal);
  signal(SIGUSR2,catchsignal);
  signal(SIGCHLD,sigchld);

  if(Cfg.ServerType==Config::inetd) opt_inetd=1;

  if(opt_inetd) {
    if(getuid()==CONF_UIDROOT) setugid(Cfg.Username,Cfg.Groupname);
    nnrpd(-1);
  } else {
    if(!opt_debug) {
      switch(fork()) {
      case -1:
	cerr << "cannot fork\n";
	slog.p(Logger::Error) << "cannot fork\n";
	break;
      case 0:
	if(setsid()==-1) {
	  cerr << "setsid failed\n";
	  slog.p(Logger::Error) << "setsid failed\n";
	  exit(1);
	}
	close(STDIN_FILENO);
	close(STDOUT_FILENO);
	close(STDERR_FILENO);      
	break;
      default:
	return 0;
      }
    }
    nntpd();
  }
}
