/*
 *   System Timer driver for ALSA sequencer
 *   Copyright (c) 1998 by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"

#include "seq.h"
#include "seq_timer.h"		/* debug time function */

static void poll_sys_tmr(unsigned long dummy);

static volatile int tmr_running = 0;
static struct timer_list sys_tmr =
{NULL, NULL, 0, 0, poll_sys_tmr};
snd_seq_real_time_t period;

static void poll_sys_tmr(unsigned long dummy)
{
	if (tmr_running) {
		/* reschedule */
		sys_tmr.expires = (1) + jiffies;
		add_timer(&sys_tmr);


		/* call sequencer client and let it know of the delta time since
		   previous call ie. time of one period            
		 */

		snd_seq_timer_interrupt(&period);
	}
}


static int sys_tmr_open(void)
{
	if (tmr_running)
		return -EBUSY;

	/* calculate period time */
	period.tv_sec = 0;
	period.tv_nsec = 1E9 / HZ;
	;
	snd_printk("registering system timer, period time = %d.%09d\n", (int)period.tv_sec, (int)period.tv_nsec);

	tmr_running = 1;
	add_timer(&sys_tmr);

	return 0;
}

static void sys_tmr_close(void)
{
	tmr_running = 0;
	del_timer(&sys_tmr);
}


/*
   should also provide identification of timer to sequencer:

   - name of timer      --> pass with callback ??
   - current period     --> passed with callback
   - min configurable period    -> no is property of timer, not of seq. 
   - max configurable period    -> no, is property of timer, not of seq. 
 */


int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(NULL) < 0)
		return -ENOMEM;
#endif

	sys_tmr_open();

	return 0;
}


void cleanup_module(void)
{
	sys_tmr_close();
}

#ifdef MODULE_PARM		/* hey - we have new 2.1.18+ kernel... */
MODULE_AUTHOR("Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>");
MODULE_DESCRIPTION("Advanced Linux Sound Architecture sequencer system timer.");
MODULE_SUPPORTED_DEVICE("sound");
#endif
