/*----------------------------------------------------------------
 * general options
 *----------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "util.h"
#include "midievent.h"
#include "controls.h"
#include "options.h"

void set_bool(int *val)
{
	if (optarg)
		*val = bool_val(optarg);
	else
		*val = !*val;
}

int adjust_volscale(char *val)
{
	double vol = atof(val);
	if (vol < 5.0) vol *= 100;
	return (int)vol;
}


/*----------------------------------------------------------------*/

enum {
	TY_BOOL, TY_INT, TY_MISC,
};

typedef struct _IndexCmd {
	int id;		/* option id */
	char *str;	/* option index name */
	int before;	/* before read midi file */
	int type;	/* option type */
	int offset;	/* offset in midi info record */
	char *desc;	/* description of the option */
} IndexCmd;

#define Offset(field)	offsetof(MidiInfo, field)

static IndexCmd idxcmds[] = {
	{OPT_CHORUS, "chorus", FALSE, TY_INT, Offset(chorus),
	 "set chorus mode (0-7)\n"
	 "\t0=chorus1, 1=chorus2, 2=chorus3, 3=chorus4,\n"
	 "\t4=feedback, 5=flanger, 6=short delay, 6=short delay2"},
	{OPT_REVERB, "reverb", FALSE, TY_INT, Offset(reverb),
	 "set reverb mode (0-7)\n"
	 "\t0=room1, 1=room2, 2=room3, 3=hall1,\n"
	 "\t4=hall2, 5=plate, 6=delay, 7=panning delay"},
	{OPT_CHORUSDEPTH, "chorusdepth", TRUE, TY_INT, Offset(chorusdepth),
	 "set chorus depth (0-127)"},
	{OPT_REVERBDEPTH, "reverbdepth", TRUE, TY_INT, Offset(reverbdepth),
	 "set reverb depth (0-127)"},
	{OPT_VOLUME, "volume", FALSE, TY_INT, Offset(volume_base),
	 "set master volume (0-100%)"},
	{OPT_REALTIME_PAN, "realpan", TRUE, TY_BOOL, Offset(realtime_pan),
	 "enable/disable realtime panning position change\n\t"},
	{OPT_AUTOSKIP, "autoskip", TRUE, TY_BOOL, Offset(skip_blank_head),
	 "enable/disable auto skip of blank events"},
	{OPT_TUNING, "tuning", TRUE, TY_BOOL, Offset(do_tuning),
	 "enable/disable tuning via RPN"},
	{OPT_SAMECSEC, "samecsec", TRUE, TY_BOOL, Offset(check_same_csec),
	 "enable/disable short note-on/off workaround\n\t"},
	{OPT_GSMACRO, "gsmacro", TRUE, TY_BOOL, Offset(check_gs_macro),
	 "enable/disable GS sysex messages"},
	{OPT_XGMACRO, "xgmacro", TRUE, TY_BOOL, Offset(check_xg_macro),
	 "enable/disable XG bank controls"},
	{OPT_XGMAP, "xgmap", TRUE, TY_BOOL, Offset(xg_mapping),
	 "enable/disable XG drum/sfx mapping"},
	{OPT_DRUM, "drum", FALSE, TY_MISC, 0,
	 "turn on (>0) or off (<0) drum channel[1-32]"},
	{OPT_DRUMFLAG, "drumflag", FALSE, TY_INT, Offset(drumflag),
	 "set all drum status as 32bit flags"},
	{OPT_ACCEPTALL, "acceptall", TRUE, TY_BOOL, Offset(accept_all_off),
	 "enable/disable all notes/sound off events"},
	{OPT_TRACKS, "tracks", TRUE, TY_INT, Offset(track_nums),
	 "set number of tracks in one midi part\n\t(-1=no multi part)"},
	{OPT_MULTIPART, "multipart", TRUE, TY_BOOL, Offset(multi_part),
	 "enable/disable multi midi parts"},
	{OPT_TITLE, "title", FALSE, TY_MISC, 0,
	 "set title of the midi file"},
	{OPT_VOLSCALE, "volscale", FALSE, TY_MISC, 0,
	 "set volume scale (0-500%)"},
	{OPT_MT32, "mt32", TRUE, TY_INT, Offset(midi_mode),
	 "set MT32 program mode\n\t(0:GM/GS, 1:MT32, 2:MT32 emulation)"},
	{OPT_CHN_PRIOR, "chnprior", TRUE, TY_BOOL, Offset(chn_prior),
	 "enable/disable channel priority mode"},
	{OPT_CHN_VOLUME, "chnvolume", TRUE, TY_MISC, 0,
	 "set channel volume scale (channel[1-32] scale[%])"},
	{OPT_OFFSET, "offset", TRUE, TY_INT, Offset(base_offset),
	 "set base key offset in semitone"},
	{OPT_PARSETITLE, "parsetitle", TRUE, TY_BOOL, Offset(parse_title),
	 "enable/disable parse title string in midi file\n\t"},
	{OPT_DYNAMIC, "dynamic", FALSE, TY_MISC, 0,
	 "load the specified soundfont file dynamically"},
	{OPT_SUBSF, "subsf", FALSE, TY_MISC, 0,
	 "load the specified soundfont file as optional font"},
	{OPT_XGLOAD, "xgload", FALSE, TY_MISC, 0,
	 "load the specified soundfont file in XG mode"},
	{OPT_VERBOSE, NULL, TRUE, TY_MISC, 0,
	 "set verbosity level"},
	{OPT_HELP, NULL, FALSE, TY_MISC, 0,
	 "print help message"},
	{OPT_SEQBUF, "seqbuf", TRUE, TY_BOOL, Offset(seq_buffered),
	 "enable/disable sequencer buffering mode"},
	{OPT_SEQECHO, "seqecho", TRUE, TY_BOOL, Offset(use_echoback),
	 "enable/disable sequencer echo back system"},
	{OPT_USEFX, "usefx", TRUE, TY_BOOL, Offset(use_fx),
	 "enable effect controls (experimental)"},
	{OPT_FX_CUTOFF, "fx_cutoff", TRUE, TY_INT, Offset(fx_sense[FX_CUTOFF]),
	 "set cutoff sensitivity"},
	{OPT_FX_RESONANCE, "fx_resonance", TRUE, TY_INT, Offset(fx_sense[FX_RESONANCE]),
	 "set resonance sensitivity"},
	{OPT_FX_ATTACK, "fx_attack", TRUE, TY_INT, Offset(fx_sense[FX_ATTACK]),
	 "set attack sensitivity"},
	{OPT_FX_RELEASE, "fx_release", TRUE, TY_INT, Offset(fx_sense[FX_RELEASE]),
	 "set attack sensitivity"},
	{OPT_FX_VIBRATE, "fx_vibrate", TRUE, TY_INT, Offset(fx_sense[FX_VIBRATE]),
	 "set vibrato rate sensitivity"},
	{OPT_FX_VIBDEPTH, "fx_vibdepth", TRUE, TY_INT, Offset(fx_sense[FX_VIBDEPTH]),
	 "set vibrato depth sensitivity"},
	{OPT_FX_VIBDELAY, "fx_vibdelay", TRUE, TY_INT, Offset(fx_sense[FX_VIBDELAY]),
	 "set vibrato delay sensitivity"},
	{OPT_CONVERT, NULL, TRUE, TY_MISC, 0,
	 "set file conversion rule (given as \"ext/command\")"},
};

int do_option_index(MidiInfo *mp, char *tok, char *val, int before_reading)
{
	int i;
	for (i = 0; i < numberof(idxcmds); i++) {
		if (idxcmds[i].str && strcmp(idxcmds[i].str, tok) == 0) {
			if (idxcmds[i].before != before_reading)
				return OPT_NONE;
			return do_option_arg(mp, idxcmds[i].id, val, TRUE);
		}
	}
	return OPT_NONE;
}

int do_option_arg(MidiInfo *mp, int type, char *arg, int msg)
{
	int i, *iptr, mode;
	char *ext;
	int len;

	switch (type) {
	case OPT_VERBOSE:
		if (arg)
			verbose = atoi(arg);
		else
			verbose++;
		break;
	case OPT_HELP:
		print_usage();
		exit(1);
	case OPT_DRUM:
		if (arg && *arg) {
			mode = atoi(arg);
			if (mode < 0)
				BITOFF(mp->drumflag, DRUMBIT(-mode - 1));
			else
				BITON(mp->drumflag, DRUMBIT(mode - 1));
			break;
		}
		return type;
	case OPT_VOLSCALE:
		mp->volscale = adjust_volscale(arg);
		if (msg)
			ctl->cmsg(CMSG_INFO, 0, "volume scale %d", mp->volscale);
		return type;
	case OPT_CONVERT:
		if ((ext = strchr(arg, '/')) == NULL)
			break;
		*ext = 0;
		ext++;
		if ((len = strlen(ext)) == 0)
			break;
		CmpAddList(arg, ext);
		break;
	}

	for (i = 0; i < numberof(idxcmds); i++) {
		if (idxcmds[i].id == type)
			break;
	}
	if (i >= numberof(idxcmds))
		return OPT_NONE;
	
	iptr = (int*)((char*)mp + idxcmds[i].offset);
	switch (idxcmds[i].type) {
	case TY_BOOL:
		if (arg && *arg)
			*iptr = bool_val(arg);
		else
			*iptr = !*iptr;
		break;
	case TY_INT:
		if (arg && *arg) {
			mode = (int)strtol(arg, NULL, 0);
			if (msg)
				ctl->cmsg(CMSG_INFO, 0, "set %s %d", idxcmds[i].str, mode);
			*iptr = mode;
		}
		break;
	}
	return type;
}


void general_options(char *optflags, int cp)
{
	do_option_arg(&glinfo, cp, optarg, FALSE);
}


void print_general_options(FILE *fp, char *optflags, awe_option_args *options)
{
	int i, *iptr;
	awe_option_args *p;

	for (p = options; p->str; p++) {
		for (i = 0; i < numberof(idxcmds); i++) {
			if (idxcmds[i].id == p->val)
				break;
		}
		if (i >= numberof(idxcmds))
		    continue;
		fprintf(fp, "  ");
		if (p->val < 0x1000 && strchr(optflags, p->val))
			fprintf(fp, "-%c, ", p->val);
		fprintf(fp, "--%s", p->str);
		if (p->has_arg > 0) {
			if (p->has_arg == 2)
				fprintf(fp, "[");
			switch (idxcmds[i].type) {
			case TY_BOOL:
				fprintf(fp, "=bool"); break;
			case TY_INT:
				fprintf(fp, "=val"); break;
			default:
				fprintf(fp, "=arg"); break;
			}
			if (p->has_arg == 2)
				fprintf(fp, "]");
		}
		fprintf(fp, ": %s", idxcmds[i].desc);

		iptr = (int*)((char*)&gl_default + idxcmds[i].offset);
		switch (idxcmds[i].type) {
		case TY_BOOL:
			fprintf(fp, " (default:%s)", BOOLSTR(*iptr)); break;
		case TY_INT:
			fprintf(fp, " (default:%d)", *iptr); break;
		}
		fprintf(fp, "\n");
	}
}
