
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include "widget.h"
#include "../econfig.h"

void HideListBox(ListBox * list)
{
   EDBUG(5, "HideListBox");
   if (list->visible) {
      HideSlider(list->vertical);
      HideSlider(list->horizontal);
      XUnmapWindow(disp, list->win);
      list->visible = FALSE;
   }
   EDBUG_RETURN_;
}

void UnhideListBox(ListBox * list)
{
   EDBUG(5, "UnhideListBox");
   if (!list->visible) {
      XMapWindow(disp, list->win);
      UnhideSlider(list->vertical);
      UnhideSlider(list->horizontal);
      list->visible = TRUE;
      DisplayListBox(list);
   }
   EDBUG_RETURN_;
}

void DefaultsListBox(ListBox * list)
{
   list->x = list->y = list->w = list->h = 0;
   list->back = NULL;
   list->state = 1;
   list->exposed = FALSE;
   list->fontheight = 1;
   list->font_info = NULL;
   list->fontset = 0;
   list->items = Emalloc(sizeof(XTextItem));
   list->items->chars = NULL;
   list->fontname = duplicate("fixed");
   list->vertical = list->horizontal = NULL;
   list->current_selection = list->last_selection = list->click_time = 0;
   list->list = NULL;
   list->start = 1;
   list->action = 0;
   list->visible = FALSE;
   list->use_fn = FALSE;
   list->fn = NULL;
}

void CreateListBox(ListBox * list, Window parent)
{
   EDBUG(5, "CreateListBox");
   list->parent = parent;
   if (!list->parent)
      list->parent = root;
   if (deferred) {
      CreateSlider(list->vertical, list->parent);
      CreateSlider(list->horizontal, list->parent);
   }
   if ((list->x + list->w) > list->vertical->x)
      list->w = list->vertical->x - list->x;
   if ((list->y + list->h) > list->horizontal->y)
      list->h = list->horizontal->y - list->y;
   if (list->back) {
      list->win = CreateWin(list->parent, list->x, list->y, list->w, list->h);
      XMapWindow(disp, list->win);
      Imlib_apply_image(imlib, list->back->im, list->win);
   }
   else {
      list->win = CreateClearWin(list->parent, list->x, list->y, list->w, list->h);
      XMapWindow(disp, list->win);
   }
   XClearWindow(disp, list->win);
   RaiseSlider(list->vertical);
   RaiseSlider(list->horizontal);
   list->visible = TRUE;
   if ((list->fn = Fnlib_load_font(fnlib, list->fontname))) {
      list->use_fn = TRUE;
      list->fnstyle.mode = MODE_VERBATIM;
      list->fnstyle.orientation = FONT_TO_RIGHT;
      list->fnstyle.justification = 0;
      list->fnstyle.spacing = 0;
   }
   else if (strchr(list->fontname, ',')) {
      int missing_cnt;
      char **missing_list, *def_str;

      list->fontset = XCreateFontSet(disp, list->fontname, &missing_list,
				     &missing_cnt, &def_str);
      if (missing_cnt)
	 XFreeStringList(missing_list);
      if (!list->fontset || missing_cnt) {
	 Efree(list->fontname);
	 list->fontname = duplicate("fixed");
      }
      else {
	 XRectangle r1, r2;

	 XmbTextExtents(list->fontset, "A", 1, &r1, &r2);
	 list->fontheight = r2.height;
      }
   }
   if (!list->use_fn && !list->fontset) {
      if ((list->font_info = XLoadQueryFont(disp, list->fontname)) == NULL) {
	 fprintf(stderr, "The font %s could not be loaded.  Using fixed instead\n", list->fontname);
	 if ((list->font_info = XLoadQueryFont(disp, "fixed")) == NULL) {
	    fprintf(stderr, "Couldn't load fixed either, quitting\n");
	    exit(-1);
	 }
      }
      list->items->font = list->font_info->fid;
      list->fontheight = list->font_info->max_bounds.ascent + list->font_info->max_bounds.descent;
   }
   list->items->delta = 0;
   if (!list->state)
      HideListBox(list);
   EDBUG_RETURN_;
}

ListBox *ConfigListBox(Window parent, FILE * file)
{
   char st[FILEPATH_LEN_MAX];
   char s1[FILEPATH_LEN_MAX], s2[FILEPATH_LEN_MAX];
   int end = FALSE;
   ListBox *list = Emalloc(sizeof(ListBox));

   EDBUG(5, "ConfigListBox");
   DefaultsListBox(list);
   list->parent = parent;
   while (!end) {
      if (GetNextLine(st, file)) {
	 splitstring(s1, s2, st);
	 if (!strncmp("END", s1, 3))
	    end = TRUE;
	 else if (!strncmp("BACKGROUND", s1, 10)) {
	    list->back = LoadImage(s2);
	    list->w = list->back->w;
	    list->h = list->back->h;
	 }
	 else if (!strncmp("LOCATION", s1, 8))
	    getcoords(s2, &list->x, &list->y);
	 else if (!strncmp("SIZE", s1, 4))
	    getcoords(s2, &list->w, &list->h);
	 else if (!strncmp("VISIBLE", s1, 7))
	    list->state = parsebool(s2);
	 else if (!strncmp("FONTNAME", s1, 8)) {
	    Efree(list->fontname);
	    list->fontname = duplicate(s2);
	 }
	 else if (!strncmp("FONTHEIGHT", s1, 10))
	    list->fontheight = atoi(s2);
	 else if (!strncmp("BEGIN", s1, 5)) {
	    upstr(s2);
	    if (!strncmp("VERTICAL", s2, 8))
	       list->vertical = ConfigSlider(parent, file);
	    else if (!strncmp("HORIZONTAL", s2, 10))
	       list->horizontal = ConfigSlider(parent, file);
	 }
	 else if (!strncmp("ACTION", s1, 6))
	    list->action = parse_action(s2);
	 else
	    fprintf(stderr, "Unknown line in ListBox config\n");
      }
      else
	 end = TRUE;
   }
   if (!deferred)
      CreateListBox(list, parent);
   EDBUG_RETURN(list);
}

void ResizeListBox(ListBox * list, int new_w, int new_h)
{
   int old_w = list->w + list->vertical->w;
   int old_h = list->h + list->horizontal->h;
   double scale_w = (double) new_w / old_w;
   double scale_h = (double) new_h / old_h;

   EDBUG(5, "ResizeListBox");
   XResizeWindow(disp, list->win, list->w * scale_w, list->h * scale_h);
   if (list->back) {
      Imlib_render(imlib, list->back->im, list->w * scale_w, list->h * scale_h);
      Imlib_apply_image(imlib, list->back->im, list->win);
   }
   MoveSlider(list->vertical, list->x + new_w - list->vertical->w, list->vertical->y);
   if (new_h != old_h)
      ResizeSlider(list->vertical, list->vertical->w, list->vertical->h * scale_h);
   MoveSlider(list->horizontal, list->horizontal->x, list->y + new_h - list->horizontal->h);
   if (new_w != old_w)
      ResizeSlider(list->horizontal, list->horizontal->w * scale_w, list->horizontal->h);
   list->w = list->w * scale_w;
   list->h = list->h * scale_h;
   RedrawListBoxText(list);
   EDBUG_RETURN_;
}

void EventListBox(ListBox * list, XEvent ev)
{
   struct timeval tv;

   EDBUG(5, "EventListBox");
   gettimeofday(&tv, NULL);
   if (isEventSlider(list->vertical, ev)) {
      EventSlider(list->vertical, ev);
      DisplayListBox(list);
   }
   else if (isEventSlider(list->horizontal, ev)) {
      EventSlider(list->horizontal, ev);
      DisplayListBox(list);
   }
   else if (ev.type != ButtonRelease) {
      EDBUG_RETURN_;
   }
   else {
      int temp = list->current_selection;

      list->current_selection = ceil((double) (ev.xbutton.y) / (double) list->fontheight) + list->start - 1;
      if (temp != list->current_selection)
	 RedrawListBoxText(list);
      if ((abs(list->click_time - tv.tv_sec) < 1) && (list->current_selection == list->last_selection)) {
	 list->last_selection = 0;
	 do_action(list->action);
      }
      else {
	 list->last_selection = list->current_selection;
	 list->click_time = tv.tv_sec;
      }
   }
   EDBUG_RETURN_;
}

int isEventListBox(ListBox * list, XEvent ev)
{
   EDBUG(5, "isEventListBox");
   if (isEventSlider(list->vertical, ev))
      EDBUG_RETURN(TRUE);
   if (isEventSlider(list->horizontal, ev))
      EDBUG_RETURN(TRUE);
   if (ev.xbutton.subwindow == list->win) {
      list->forceredraw = TRUE;
      EDBUG_RETURN(TRUE);
   }
   if (ev.xany.window == list->win) {
      list->forceredraw = TRUE;
      EDBUG_RETURN(TRUE);
   }
   EDBUG_RETURN(FALSE);
}

void RedrawListBoxText(ListBox * list)
{
   struct playlist_item *cur;
   int max_items, y_pos = 0, i;

   EDBUG(5, "RedrawListBoxText");
   if (!list || !list->list)
      EDBUG_RETURN_;
   XClearWindow(disp, list->win);
   max_items = list->h / list->fontheight;
   if (!list->list->first) {
      EDBUG_RETURN_;
   }
   else {
      cur = list->list->first;
      while (cur && (cur->pos != list->start)) {
	 cur = cur->next;
      }
   }
   y_pos = list->fontheight - list->fontheight / 3;
   for (i = 1; i <= max_items; i++) {
      if (!cur)
	 break;
      if (cur->name) {
	 list->items->nchars = strlen(cur->name);
	 list->items->chars = Emalloc(strlen(cur->name) + 20);
	 strcpy(list->items->chars, cur->name);
	 if (cur->length == 1) {
	    list->items->nchars++;
	    strcat(list->items->chars, "/");
	 }
	 if (cur->pos != list->current_selection) {
	    if (!list->use_fn) {
	       if (list->fontset) {
		  XmbDrawString(disp, list->win, list->fontset, fg, 0,
				y_pos, list->items->chars,
				strlen(list->items->chars));
	       }
	       else
		  XDrawText(disp, list->win, fg, 0, y_pos, list->items, 1);
	    }
	 }
	 else {
	    GC tempgc;
	    XGCValues tempgcv;

	    if (list->fontset)
	       XmbDrawString(disp, list->win, list->fontset, fg, 0,
			     y_pos, list->items->chars,
			     strlen(list->items->chars));
	    else
	       XDrawText(disp, list->win, fg, 0, y_pos, list->items, 1);
	    tempgc = XCreateGC(disp, root, 0, &tempgcv);
	    if (tempgc) {
	       XCopyGC(disp, fg, GCForeground, tempgc);
	       tempgcv.function = GXinvert;
	       XChangeGC(disp, tempgc, GCFunction, &tempgcv);
	       XFillRectangle(disp, list->win, tempgc, 0, y_pos - list->fontheight + 2, list->w - list->vertical->w, list->fontheight);
	       XFlush(disp);
	       XFreeGC(disp, tempgc);
	    }
	 }
	 Efree(list->items->chars);
      }
      if (!cur->next)
	 break;
      else {
	 cur = cur->next;
	 y_pos += list->fontheight;
      }
   }
   EDBUG_RETURN_;
}

void DisplayListBox(ListBox * list)
{
   int temp;

   EDBUG(5, "DisplayListBox");
   if (list->vertical->dragging) {
      temp = list->start;
      if (list->list->last)
	 list->start = GetSliderPosition(list->vertical) * list->list->last->pos;
      else
	 list->start = 1;
      if (list->start <= 0)
	 list->start = 1;
      if (list->list->last && (list->start >= (list->list->last->pos - 5)))
	 list->start = list->list->last->pos - 5;
      if (temp != list->start)
	 RedrawListBoxText(list);
      EDBUG_RETURN_;
   }
   else if (list->horizontal->dragging) {
      temp = list->items->delta;
      list->items->delta = -(list->horizontal->slider_position - list->horizontal->min);
      if (temp != list->items->delta)
	 RedrawListBoxText(list);
      EDBUG_RETURN_;
   }
   else if (list->forceredraw)
      RedrawListBoxText(list);
   EDBUG_RETURN_;
}

void SetPlaylistListBox(ListBox * list, PlayList * plist)
{
   EDBUG(5, "SetPlaylistListBox");
   list->list = plist;
   if (list->visible)
      DisplayListBox(list);
   EDBUG_RETURN_;
}

void FreeListBox(ListBox * list)
{
   EDBUG(5, "FreeListBox");
   if (!list)
      EDBUG_RETURN_;
   FreeImageDat(list->back);
   if (list->font_info)
      Efree(list->font_info);
   else if (list->fontset)
      XFreeFontSet(disp, list->fontset);
   if (list->items) {
      Efree(list->items);
      list->items = NULL;
   }
   if (list->fn)
      Fnlib_free_font(fnlib, list->fn);
   FreeSlider(list->vertical);
   FreeSlider(list->horizontal);
   Efree(list);
   list = NULL;
   EDBUG_RETURN_;
}
