
#include "widget.h"
#include "../econfig.h"

void HideSlider(Slider * slider)
{
   EDBUG(5, "HideSlider");
   if (slider->visible) {
      if (slider->arrows) {
	 HideButton(slider->arrow1);
	 HideButton(slider->arrow2);
      }
      HideButton(slider->button);
      XUnmapWindow(disp, slider->win);
      slider->visible = FALSE;
   }
   EDBUG_RETURN_;
}

void UnhideSlider(Slider * slider)
{
   EDBUG(5, "UnhideSlider");
   if (!slider->visible) {
      XMapWindow(disp, slider->win);
      if (slider->arrows) {
	 UnhideButton(slider->arrow1);
	 UnhideButton(slider->arrow2);
      }
      UnhideButton(slider->button);
      slider->visible = TRUE;
   }
   EDBUG_RETURN_;
}

void RaiseSlider(Slider * slider)
{
   EDBUG(5, "RaiseSlider");
   XRaiseWindow(disp, slider->win);
   if (slider->arrows) {
      XRaiseWindow(disp, slider->arrow1->win);
      XRaiseWindow(disp, slider->arrow2->win);
   }
   XRaiseWindow(disp, slider->button->win);
   EDBUG_RETURN_;
}

void DefaultsSlider(Slider * slider)
{
   slider->x = slider->y = slider->w = slider->h = 0;
   slider->button = slider->arrow1 = slider->arrow2 = NULL;
   slider->back = NULL;
   slider->arrows = FALSE;
   slider->percent = 0;
   slider->horizontal = TRUE;
   slider->slider_size = 0;
   slider->slider_position = 0;
   slider->min = 0;
   slider->max = 0;
   slider->state = 1;
   slider->callback = NULL;
   slider->dragging = FALSE;
   slider->backs = NULL;
   slider->thumbs = NULL;
   slider->progressbar = FALSE;
}

void CreateSlider(Slider * slider, Window parent)
{
   EDBUG(5, "CreateSlider");
   slider->parent = parent;
   if (!slider->parent)
      slider->parent = root;
   if (deferred) {
      if (slider->arrows) {
	 CreateButton(slider->arrow1, parent);
	 CreateButton(slider->arrow2, parent);
      }
      CreateButton(slider->button, parent);
   }
   slider->win = CreateWin(slider->parent, slider->x, slider->y, slider->w, slider->h);
   XMapWindow(disp, slider->win);
   Imlib_apply_image(imlib, slider->back->im, slider->win);
   XClearWindow(disp, slider->win);
   slider->visible = TRUE;
   if (slider->horizontal)
      slider->slider_size = slider->button->w;
   else
      slider->slider_size = slider->button->h;
   if (slider->arrows) {
      XMoveWindow(disp, slider->arrow1->win, slider->x, slider->y);
      slider->arrow1->x = slider->x;
      slider->arrow1->y = slider->y;
      if (slider->horizontal) {
	 slider->min = slider->arrow1->w + slider->x;
	 slider->max = slider->w - slider->arrow2->w + slider->x;
	 XMoveWindow(disp, slider->arrow2->win, slider->max, slider->y);
	 slider->arrow2->x = slider->max;
	 slider->arrow2->y = slider->y;
      }
      else {
	 slider->min = slider->arrow1->h + slider->y;
	 slider->max = slider->h - slider->arrow2->h + slider->y;
	 XMoveWindow(disp, slider->arrow2->win, slider->x, slider->max);
	 slider->arrow2->x = slider->x;
	 slider->arrow2->y = slider->max;
      }
   }
   else {
      if (slider->horizontal) {
	 slider->min = slider->x;
	 slider->max = slider->w + slider->x;
      }
      else {
	 slider->min = slider->y;
	 slider->max = slider->h + slider->y;
      }
   }
   slider->max -= slider->slider_size;
   if (slider->horizontal) {
      XMoveWindow(disp, slider->button->win, slider->min, slider->y);
      slider->button->x = slider->min;
      slider->button->y = slider->y;
   }
   else {
      XMoveWindow(disp, slider->button->win, slider->x, slider->min);
      slider->button->x = slider->x;
      slider->button->y = slider->min;
   }
   if (slider->progressbar)
      ScaleButton(slider->button, 1 / (double) slider->button->w, slider->button->h);
   XRaiseWindow(disp, slider->button->win);
   slider->percent = 0;
   slider->slider_position = slider->min;
   if (!slider->state)
      HideSlider(slider);
   EDBUG_RETURN_;
}

Slider *ConfigSlider(Window parent, FILE * file)
{
   char st[FILEPATH_LEN_MAX];
   char s1[FILEPATH_LEN_MAX], s2[FILEPATH_LEN_MAX];
   int end = FALSE;
   Slider *slider = Emalloc(sizeof(Slider));

   EDBUG(5, "ConfigSlider");
   DefaultsSlider(slider);
   slider->parent = parent;
   while (!end) {
      if (GetNextLine(st, file)) {
	 splitstring(s1, s2, st);
	 if (!strncmp("END", s1, 3))
	    end = TRUE;
	 else if (!strncmp("BACKGROUND", s1, 10)) {
	    slider->back = LoadImage(s2);
	    slider->w = slider->back->w;
	    slider->h = slider->back->h;
	 }
	 else if (!strncmp("ARROWS", s1, 6))
	    slider->arrows = parsebool(s2);
	 else if (!strncmp("HORIZONTAL", s1, 10))
	    slider->horizontal = parsebool(s2);
	 else if (!strncmp("LOCATION", s1, 8))
	    getcoords(s2, &slider->x, &slider->y);
	 else if (!strncmp("VISIBLE", s1, 7))
	    slider->state = parsebool(s2);
	 else if (!strncmp("PROGRESSBAR", s1, 9))
	    slider->progressbar = parsebool(s2);
	 else if (!strncmp("BEGIN", s1, 5)) {
	    upstr(s2);
	    if (!strncmp("BUTTON", s2, 6))
	       slider->button = ConfigButton(parent, file);
	    else if (!strncmp("ARROW1", s2, 6))
	       slider->arrow1 = ConfigButton(parent, file);
	    else if (!strncmp("ARROW2", s2, 6))
	       slider->arrow2 = ConfigButton(parent, file);
	    else if (!strncmp("BACK_IMAGES", s2, 11))
	       slider->backs = ConfigImageList(file);
	    else if (!strncmp("SLIDER_IMAGES", s2, 13))
	       slider->thumbs = ConfigImageList(file);
	 }
	 else
	    fprintf(stderr, "Unknown line in Slider config\n");
      }
      else
	 end = TRUE;
   }
   if (!deferred)
      CreateSlider(slider, parent);
   EDBUG_RETURN(slider);
}

void MoveSlider(Slider * slider, int x, int y)
{
   int diff_x = x - slider->x;
   int diff_y = y - slider->y;

   EDBUG(5, "MoveSlider");
   slider->x = x;
   slider->y = y;
   XMoveWindow(disp, slider->win, x, y);
   if (slider->arrows) {
      MoveButton(slider->arrow1, slider->arrow1->x + diff_x, slider->arrow1->y +
		 diff_y);
      MoveButton(slider->arrow2, slider->arrow2->x + diff_x, slider->arrow2->y +
		 diff_y);
   }
   if (slider->horizontal) {
      slider->slider_size = slider->button->w;
      if (slider->arrows) {
	 slider->min = slider->arrow1->w + slider->x;
	 slider->max = slider->w - slider->arrow2->w + slider->x;
      }
      else {
	 slider->min = slider->x;
	 slider->max = slider->w + slider->x;
      }
   }
   else {
      slider->slider_size = slider->button->h;
      if (slider->arrows) {
	 slider->min = slider->arrow1->h + slider->y;
	 slider->max = slider->h - slider->arrow2->h + slider->y;
      }
      else {
	 slider->min = slider->y;
	 slider->max = slider->h + slider->y;
      }
   }
   slider->max -= slider->slider_size;
   UpdateSlider(slider);
   EDBUG_RETURN_;
}

void ResizeSlider(Slider * slider, int new_w, int new_h)
{
   EDBUG(5, "ResizeSlider");
   slider->w = new_w;
   slider->h = new_h;
   XResizeWindow(disp, slider->win, new_w, new_h);
   Imlib_render(imlib, slider->back->im, new_w, new_h);
   Imlib_apply_image(imlib, slider->back->im, slider->win);
   if (slider->arrows) {
      /* FIXME: fill in */
   }
   if (slider->backs) {
      /* FIXME: fill in */
   }
   SliderUpdateInfo(slider);
   EDBUG_RETURN_;
}

void ScaleSlider(Slider * slider, double scale_w, double scale_h)
{
   EDBUG(5, "ScaleSlider");
   slider->w *= scale_w;
   slider->h *= scale_h;
   slider->x *= scale_w;
   slider->y *= scale_h;
   XMoveResizeWindow(disp, slider->win, slider->x, slider->y, slider->w, slider->h);
   Imlib_render(imlib, slider->back->im, slider->w, slider->h);
   Imlib_apply_image(imlib, slider->back->im, slider->win);
   ScaleButton(slider->button, scale_w, scale_h);
   if (slider->arrows) {
      ScaleButton(slider->arrow1, scale_w, scale_h);
      ScaleButton(slider->arrow2, scale_w, scale_h);
   }
   if (slider->backs)
      ScaleImageList(slider->backs, scale_w, scale_h);
   if (slider->thumbs)
      ScaleImageList(slider->thumbs, scale_w, scale_h);
   SliderUpdateInfo(slider);
   EDBUG_RETURN_;
}

void SliderUpdateInfo(Slider * slider)
{
   EDBUG(5, "SliderUpdateInfo");
   if (slider->horizontal) {
      slider->slider_size = slider->button->w;
      if (slider->arrows) {
	 slider->min = slider->arrow1->w + slider->x;
	 slider->max = slider->w - slider->arrow2->w + slider->x;
      }
      else {
	 slider->min = slider->x;
	 slider->max = slider->w + slider->x;
      }
   }
   else {
      slider->slider_size = slider->button->h;
      if (slider->arrows) {
	 slider->min = slider->arrow1->h + slider->y;
	 slider->max = slider->h - slider->arrow2->h + slider->y;
      }
      else {
	 slider->min = slider->y;
	 slider->max = slider->h + slider->y;
      }
   }
   slider->max -= slider->slider_size;
   UpdateSlider(slider);
   EDBUG_RETURN_;
}

void EventSlider(Slider * slider, XEvent ev)
{
   float per;
   static int save_pos = 0;
   static int ref_click = 0;

   EDBUG(5, "EventSlider");
   if (ev.type == ButtonPress) {
      if (ev.xbutton.window == slider->button->win) {
	 EventButton(slider->button, ev);
	 slider->dragging = -1;
	 if (slider->horizontal)
	    ref_click = ev.xbutton.x_root;
	 else
	    ref_click = ev.xbutton.y_root;
	 save_pos = slider->slider_position;
      }
      else if (slider->arrows) {
	 if (ev.xbutton.subwindow == slider->arrow1->win) {
	    EventButton(slider->arrow1, ev);
	    per = GetSliderPosition(slider);
	    if (per >= 2)
	       per -= 2.0;
	    SetSliderPosition(slider, per);
	 }
	 else if (ev.xbutton.subwindow == slider->arrow2->win) {
	    EventButton(slider->arrow2, ev);
	    per = GetSliderPosition(slider);
	    if (per <= 98)
	       per += 2.0;
	    SetSliderPosition(slider, per);
	 }
      }
   }
   else if (ev.type == MotionNotify) {
      if (slider->horizontal) {
	 if ((slider->dragging != 0) && (slider->dragging != ev.xmotion.x_root)) {
	    slider->dragging = ev.xmotion.x_root;
	    slider->slider_position = ev.xmotion.x_root - ref_click + save_pos;
	    UpdateSlider(slider);
	 }
      }
      else {
	 if ((slider->dragging != 0) && (slider->dragging != ev.xmotion.y_root)) {
	    slider->dragging = ev.xmotion.y_root;
	    slider->slider_position = ev.xmotion.y_root - ref_click + save_pos;
	    UpdateSlider(slider);
	 }
      }
      if ((slider->dragging != 0) && (slider->callback))
	 slider->callback();
   }
   else if (ev.type == ButtonRelease) {
      slider->dragging = FALSE;
      EventButton(slider->button, ev);
      if (slider->callback)
	 slider->callback();
   }
   EDBUG_RETURN_;
}

int isEventSlider(Slider * slider, XEvent ev)
{
   EDBUG(5, "isEventSlider");
   if (isEventButton(slider->button, ev))
      EDBUG_RETURN(TRUE);
   if (slider->arrows) {
      if (isEventButton(slider->arrow1, ev))
	 EDBUG_RETURN(TRUE);
      if (isEventButton(slider->arrow2, ev))
	 EDBUG_RETURN(TRUE);
   }
   if (ev.xbutton.subwindow == slider->win)
      EDBUG_RETURN(TRUE);
   if (ev.xany.window == slider->win)
      EDBUG_RETURN(TRUE);
   EDBUG_RETURN(FALSE);
}

void UpdateSlider(Slider * slider)
{
   EDBUG(5, "UpdateSlider");
   if (slider->slider_position < slider->min)
      slider->slider_position = slider->min;
   if (slider->slider_position > slider->max)
      slider->slider_position = slider->max;
   if (slider->backs) {
      float pos = (GetSliderPosition(slider) * slider->backs->number);
      int slot = (int) floor(pos);

      if (slot >= slider->backs->number)
	 slot = slider->backs->number - 1;
      Imlib_apply_image(imlib, slider->backs->images[slot]->im, slider->win);
      XClearWindow(disp, slider->win);
   }
   if (slider->thumbs) {
      float pos = (GetSliderPosition(slider) * slider->thumbs->number);
      int slot = (int) floor(pos);

      if (slot >= slider->thumbs->number)
	 slot = slider->thumbs->number - 1;
      Imlib_apply_image(imlib, slider->thumbs->images[slot]->im, slider->button->win);
      XClearWindow(disp, slider->button->win);
   }
   if (slider->horizontal) {
      if (slider->progressbar)
	 ResizeButton(slider->button, slider->slider_position * (slider->w / slider->max), slider->button->h);
      else {
	 XMoveWindow(disp, slider->button->win, slider->slider_position, slider->y);
	 slider->button->x = slider->slider_position;
      }
   }
   else {
      if (slider->progressbar)
	 ResizeButton(slider->button, slider->button->w, slider->slider_position * (slider->h / slider->max));
      else {
	 XMoveWindow(disp, slider->button->win, slider->x, slider->slider_position);
	 slider->button->y = slider->slider_position;
      }
   }
   EDBUG_RETURN_;
}

float GetSliderPosition(Slider * slider)
{
   int total;

   EDBUG(5, "GetSliderPosition");
   total = slider->max - slider->min;
   slider->percent = (float) (slider->slider_position - slider->min) / (float) total;
   EDBUG_RETURN(slider->percent);
}

void SetSliderPosition(Slider * slider, float percent)
{
   int total;

   EDBUG(5, "SetSliderPosition");
   if (percent == slider->percent)
      EDBUG_RETURN_;
   if (slider->dragging)
      EDBUG_RETURN_;
   total = slider->max - slider->min;
   slider->slider_position = total * percent + slider->min;
   slider->percent = percent;
   UpdateSlider(slider);
   EDBUG_RETURN_;
}

void SliderSetCallback(Slider * slider, void (*call) (void))
{
   slider->callback = call;
}

void FreeSlider(Slider * slider)
{
   EDBUG(5, "FreeSlider");
   if (!slider)
      EDBUG_RETURN_;
   FreeImageDat(slider->back);
   FreeButton(slider->button);
   FreeButton(slider->arrow1);
   FreeButton(slider->arrow2);
   Efree(slider);
   slider = NULL;
   EDBUG_RETURN_;
}
