// -*-C++-*-
// This file is part of the gmod package
// Copyright (C) 1997 by Andrew J. Robinson

#include <fcntl.h>

#include "CursesScr.h"
#include "Sample.h"

CursesScr::CursesScr(unsigned char background)
{
  background_ = background;

  if (!background)
    {
#ifdef USE_NCURSES
      initscr();
      cbreak();
      noecho();
      infoWin = newwin(6, 0, 0, 0);
      scrollok(infoWin, FALSE);
      memWin = newwin(1, 0, LINES - 2, 0);
      scrollok(memWin, FALSE);
      posWin = newwin(1, 0, LINES - 1, 0);
      scrollok(posWin, FALSE);
      keypad(posWin, TRUE);
      wtimeout(posWin, 0);
      mvwaddstr(infoWin, 0, 0, "File:");
      mvwaddstr(infoWin, 1, 0, "Name:");
      mvwaddstr(infoWin, 2, 0, "Channels:");
      mvwaddstr(infoWin, 2, 14, "Type:");
      waddstr(memWin, "===         bytes of         total onboard used ===");
      waddstr(posWin, "Position 000/000, Pattern 000");
#else
      struct termios newTermio;

      oldFlags_ = fcntl(0, F_GETFL);
      fcntl(0, F_SETFL, oldFlags_ | O_NONBLOCK);
      tcgetattr(0, &oldTermio_);
      newTermio = oldTermio_;
      newTermio.c_lflag = 0;
      newTermio.c_cc[VTIME] = 0;
      newTermio.c_cc[VMIN] = 0;
      tcsetattr(0, TCSANOW, &newTermio);             
#endif
    }
}

CursesScr::~CursesScr()
{
  if (!background_)
    {
#ifdef USE_NCURSES
      endwin();
#else
      fcntl(0, F_SETFL, oldFlags_);
      tcsetattr(0, TCSANOW, &oldTermio_);                               
#endif
    }
}

void
CursesScr::moduleFile(const char *fileName)
{
  if (!background_)
    {
#ifdef USE_NCURSES
      mvwaddstr(infoWin, 0, 6, fileName);
      wclrtoeol(infoWin);
#else
      fputs("\n\n\nFile: ", stdout);
      fputs(fileName, stdout);
      fputs("\n", stdout);
#endif
    }
}

void
CursesScr::moduleTitle(const char *name)
{
  if (!background_)
    {
#ifdef USE_NCURSES
      mvwaddstr(infoWin, 1, 6, name);
      wclrtoeol(infoWin);
#else
      fputs("Name: ", stdout);
      fputs(name, stdout);
      fputs("\n", stdout);
#endif
    }
}

void
CursesScr::setChannels(int numChannels)
{
  if (!background_)
    {
#ifdef USE_NCURSES
      mvwprintw(infoWin, 2, 10, "%02d", numChannels);
#else
      printf("Channels: %02d  ", numChannels);
#endif
    }
}

void
CursesScr::setType(const char *type)
{
  if (!background_)
    {
#ifdef USE_NCURSES
      mvwaddstr(infoWin, 2, 20, type);
      wclrtoeol(infoWin);
      wrefresh(infoWin);
#else
      fputs("Type: ", stdout);
      fputs(type, stdout);
      fputs("\n\n", stdout);
#endif
    }
}

void
CursesScr::setMaxPosition(int maxPosition)
{
  if (!background_)
    {
#ifdef USE_NCURSES
      mvwprintw(posWin, 0, 13, "%03d", maxPosition);
#else
      // fill in later
#endif
    }
}

void
CursesScr::setMem(int avail, int total)
{
  if (!background_)
    {
#ifdef USE_NCURSES
      mvwprintw(memWin, 0, 4, "%7d", total - avail);
      mvwprintw(memWin, 0, 21, "%7d", total);
      wrefresh(memWin);
#else
      printf("=== %d bytes of %d total onboard used ===\n", total - avail,
	     total);
#endif
    }
}

void
CursesScr::setPos(unsigned int pos, unsigned int pat)
{
  if (!background_)
    {
#ifdef USE_NCURSES
      //mvwprintw(posWin, 0, 0, "Position %03d, Pattern %03d", pos, pat);
      mvwprintw(posWin, 0, 9, "%03d", pos);
      mvwprintw(posWin, 0, 26, "%03d", pat);
      wrefresh(posWin);
#else
      printf("Position %03d, Pattern %03d\r", pos, pat);
      fflush(stdout);
#endif  
    }
}
