/*==============================================================================

  $Id: mloader.c,v 1.7 1998/09/20 21:45:23 miod Exp $

  These routines are used to access the available module loaders

==============================================================================*/

/*
	This library is free software; you can redistribute it and/or modify
	it under the terms of the GNU Library General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.
 
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Library General Public License for more details.
 
	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <mikmod.h>

#include <string.h>

FILE *modfp;
UNIMOD of;

static MLOADER *firstloader=NULL;

UWORD finetune[16]={
	8363,8413,8463,8529,8581,8651,8723,8757,
	7895,7941,7985,8046,8107,8169,8232,8280
};

CHAR* ML_InfoLoader(void)
{
	int t,len=0;
	MLOADER *l;
static	CHAR *list=NULL;

	if (list) free(list);	/* just in case someone registered a new driver */
	/* compute size of buffer */
	for(l=firstloader;l;l=l->next) len+=4+(l->next?1:0)+strlen(l->version);

	if((list=_mm_malloc(len*sizeof(CHAR)))) {
		list[0]=0;
		/* list all registered module loders */
		for(t=1,l=firstloader;l;l=l->next,t++)
			sprintf(list,(l->next)?"%s%2d %s\n":"%s%2d %s",list,t,l->version);
	}
	return list;
}

void ML_RegisterLoader(MLOADER* ldr)
{
	MLOADER *cruise=firstloader;

	/* if we try to register an invalid loader, or an already registered loader,
	   ignore this attempt */
	if ((!ldr)||(ldr->next))
		return;

	if(cruise) {
		while(cruise->next) cruise = cruise->next;
		cruise->next=ldr;
	} else
		firstloader=ldr; 
}

BOOL ReadComment(UWORD len)
{
	if(len) {
		if(!(of.comment=(CHAR*)_mm_malloc(len+1))) return 0;
		fread(of.comment,len,1,modfp);
		of.comment[len]=0;	/* just in case */
	}
	return 1;
}

BOOL ReadLinedComment(UWORD lines,UWORD linelen)
{
	CHAR *tempcomment,*line,*storage;
	UWORD total,t,i,len=lines*linelen;

	if (lines) {
		if(!(tempcomment=(CHAR*)_mm_malloc(len))) return 0;
		if(!(storage=(CHAR*)_mm_malloc(linelen+1))) {
			free(tempcomment);
			return 0;
		}
		fread(tempcomment,len,1,modfp);

		/* compute message length */
		for(line=tempcomment,total=t=0;t<lines;t++,line+=linelen) {
			for(i=0;i<linelen;i++) if (!line[i]) break;
			total+=1+i;
		}

		if(!(of.comment=(CHAR*)_mm_malloc(total+1))) {
			free(tempcomment);
			return 0;
		}

		/* convert message */
		for(line=tempcomment,t=0;t<lines;t++,line+=linelen) {
			for(i=0;i<linelen;i++) if(!(storage[i]=line[i])) break;
			storage[i]=0; /* if (i==linelen) */
			strcat(of.comment,storage);strcat(of.comment,"\r");
		}
		free(storage);
	}
	return 1;
}

BOOL AllocPositions(int total)
{
	if(!(of.positions=_mm_calloc(total,sizeof(UWORD)))) return 0;
	return 1;
}

BOOL AllocPatterns(void)
{
	int s,t,tracks = 0;

	/* Allocate track sequencing array */
	if(!(of.patterns=(UWORD*)_mm_calloc((ULONG)(of.numpat+1)*of.numchn,sizeof(UWORD)))) return 0;
	if(!(of.pattrows=(UWORD*)_mm_calloc(of.numpat+1,sizeof(UWORD)))) return 0;

	for(t=0;t<=of.numpat;t++) {
		of.pattrows[t]=64;
		for(s=0;s<of.numchn;s++)
		of.patterns[(t*of.numchn)+s]=tracks++;
	}

	return 1;
}

BOOL AllocTracks(void)
{
	if(!(of.tracks=(UBYTE **)_mm_calloc(of.numtrk,sizeof(UBYTE *)))) return 0;
	return 1;
}

BOOL AllocInstruments(void)
{
	int t,n;
	
	if(!(of.instruments=(INSTRUMENT*)_mm_calloc(of.numins,sizeof(INSTRUMENT))))
		return 0;

	for(t=0;t<of.numins;t++) {
		for(n=0;n<120;n++) { 
			/* Init note / sample lookup table */
			of.instruments[t].samplenote[n]   = n;
			of.instruments[t].samplenumber[n] = t;
		}   
		of.instruments[t].globvol = 64;
	}
	return 1;
}

BOOL AllocSamples(void)
{
	UWORD u;

	if(!(of.samples=(SAMPLE*)_mm_calloc(of.numsmp,sizeof(SAMPLE)))) return 0;

	for(u=0;u<of.numsmp;u++) {
		of.samples[u].panning = 128;
		of.samples[u].handle  = -1;
		of.samples[u].globvol = 64;
		of.samples[u].volume  = 64;
	}
	return 1;
}

BOOL ML_LoadSamples(void)
{
	SAMPLE *s;
	int u;

	for(u=of.numsmp,s=of.samples;u;u--,s++)
	if(s->length) SL_RegisterSample(s,MD_MUSIC,modfp);

	return 1;
}

/* Creates a CSTR out of a character buffer of 'len' bytes, but strips any
   terminating non-printing characters like 0, spaces etc.                    */
CHAR *DupStr(CHAR* s,UWORD len)
{
	UWORD t;
	CHAR *d=NULL;

	/* Scan for last printing char in buffer [includes high ascii up to 254] */
	while(len) {
		if(s[len-1]>0x20) break;
		len--;
	}
	/* Scan forward for possible NULL character */
/*	for(t=0;t<len;t++) if (!s[t]) break;
	if (t<len) len=t;
*/
	/* When the buffer wasn't completely empty, allocate a cstring and copy the
	   buffer into that string, except for any control-chars */
	if((d=(CHAR*)_mm_malloc(len+1))) {
		for(t=0;t<len;t++) d[t]=(s[t]<32)?' ':s[t];
		d[len]=0;
	}
	return d;
}

void ML_XFreeSample(SAMPLE *s)
{
	if(s->handle>=0)
		MD_SampleUnLoad(s->handle);
	if(s->samplename) free(s->samplename);
}

void ML_XFreeInstrument(INSTRUMENT *i)
{
	if(i->insname) free(i->insname);
}

void ML_FreeEx(UNIMOD *mf)
{
	UWORD t;

	if(mf->songname) free(mf->songname);
	if(mf->composer) free(mf->composer);
	if(mf->comment)  free(mf->comment);

	if(mf->modtype)   free(mf->modtype);
	if(mf->positions) free(mf->positions);
	if(mf->patterns)  free(mf->patterns);
	if(mf->pattrows)  free(mf->pattrows);

	if(mf->tracks) {
		for(t=0;t<mf->numtrk;t++)
			if(mf->tracks[t]) free(mf->tracks[t]);
		free(mf->tracks);
	}
	if(mf->instruments) {
		for(t=0;t<mf->numins;t++)
			ML_XFreeInstrument(&mf->instruments[t]);
		free(mf->instruments);
	}
	if(mf->samples) {
		for(t=0;t<mf->numsmp;t++)
			if(mf->samples[t].length) ML_XFreeSample(&mf->samples[t]);
		free(mf->samples);
	}
	memset(mf,0,sizeof(UNIMOD));
}

UNIMOD *ML_AllocUniMod(void)
{
	UNIMOD *mf;

	return (mf=_mm_malloc(sizeof(UNIMOD)));
}

void MikMod_FreeSong(UNIMOD *mf)
{
	if(mf) {
		Player_Exit(mf);
		ML_FreeEx(mf);
	}
}

CHAR* MikMod_LoadSongTitle(CHAR* filename)
{
	MLOADER *l;
	CHAR *retval;
	FILE *fp;

	if(!(fp=_mm_fopen(filename,"r"))) return NULL;

	_mm_errno = 0;
	_mm_critical = 0;
	_mm_iobase_setcur(modfp);

	/* Try to find a loader that recognizes the module */
	for(l=firstloader;l;l=l->next) {
		_mm_rewind(modfp);
		if(l->Test()) break;
	}

	if(!l) {
		_mm_errno = MMERR_NOT_A_MODULE;
		if(_mm_errorhandler) _mm_errorhandler();
		_mm_iobase_revert();
		return NULL;
	}

	retval=l->LoadTitle();
	fclose(fp);
	return(retval);
}

/* Loads a module given a file pointer.
   File is loaded from the current file seek position. */
UNIMOD* MikMod_LoadSongFP(FILE* fp,int maxchan)
{
	int t;
	MLOADER *l;
	BOOL ok;
	UNIMOD *mf;

	modfp = fp;
	_mm_errno = 0;
	_mm_critical = 0;
	_mm_iobase_setcur(modfp);

	/* Try to find a loader that recognizes the module */
	for(l=firstloader;l;l=l->next) {
		_mm_rewind(modfp);
		if(l->Test()) break;
	}

	if(!l) {
		_mm_errno = MMERR_NOT_A_MODULE;
		if(_mm_errorhandler) _mm_errorhandler();
		_mm_iobase_revert();
		return NULL;
	}

	/* init unitrk routines */
	if(!UniInit()) {
		if(_mm_errorhandler) _mm_errorhandler();
		return NULL;
	}

	/* load the song using the song's loader variable */
	memset(&of,0,sizeof(UNIMOD));
	of.initvolume = 128;

	/* init panning array */
	for(t=0; t<64; t++) of.panning[t] = ((t+1)&2) ? 255 : 0;
	for(t=0; t<64; t++) of.chanvol[t] = 64;

	/* init module loader and load the header / patterns */
	if(l->Init()) {
		_mm_rewind(modfp);
		ok = l->Load();
	} else
		ok = 0;

	/* free loader and unitrk allocations */
	l->Cleanup();
	UniCleanup();

	if(!ok) {
		ML_FreeEx(&of);
		if(_mm_errorhandler) _mm_errorhandler();
		_mm_iobase_revert();
		return NULL;
	}

	if(!ML_LoadSamples()) {
		ML_FreeEx(&of);
		if(_mm_errorhandler) _mm_errorhandler();
		_mm_iobase_revert();
		return NULL;
	}

	if(!(mf=ML_AllocUniMod())) {
		ML_FreeEx(&of);
		_mm_iobase_revert();
		if(_mm_errorhandler) _mm_errorhandler();
		return NULL;
	}

	/* Copy the static UNIMOD contents into the dynamic UNIMOD struct. */
	memcpy(mf,&of,sizeof(UNIMOD));
	_mm_iobase_revert();

	if(maxchan>0) {
		if(!(mf->flags&UF_NNA)&&(mf->numchn<maxchan))
			maxchan = mf->numchn;
		else
		  if((mf->numvoices)&&(mf->numvoices<maxchan))
			maxchan = mf->numvoices;

		if(maxchan<mf->numchn) mf->flags |= UF_NNA;

		if(MikMod_SetNumVoices(maxchan,-1)) {
			MikMod_FreeSong(mf);
			return NULL;
		}
	}
	return mf;
}

/* Open a module via it's filename.  The loader will initialize the specified
   song-player 'player'. */
UNIMOD* MikMod_LoadSong(CHAR* filename,int maxchan)
{
	FILE *fp;
	UNIMOD *mf;

	if(!(fp=_mm_fopen(filename,"r"))) return NULL;

	if((mf=MikMod_LoadSongFP(fp, maxchan))) 
		if(SL_LoadSamples() || Player_Init(mf)) {
			MikMod_FreeSong(mf);
			mf=NULL;
		}

	fclose(fp);
	return mf;
}
