/************************************************************/
/*    Midifile interface                                    */ 
/*            V 1.00 (okt 91)             by wini           */
/************************************************************/
/* A) systemdefs */

/* B) standard includes */

#include	<stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>

#include "mididefs.h"
#include "midifile.h"

/* D) defines */
/* #define mydebug() {} */

void mydebug(void) { }

/* E) prototypes */
int read_varlen(int fd,unsigned long *dt);

/* F) globals */
byte buf[MD_MAX_MESSAGE_LEN];  /* buffer for reading from file */
int verbose = 1;

/* ------------ FUNCTIONS ---------------------------------- */

MIDIFile *mf_read(char *filename)
{
  MIDIFile *mifi; /* midifile to read */
  MThd *mifihd;   /* short for header */
  MTrk *trk;      /* track */
  MTrk *ptrk;     /* previous track */
  MTevt *me;      /* metaevent */
  MTevt *pe;      /* previous metaevent */
  METAstr *meta;

  char *name;
  int i;
  unsigned long j;
  int count;
  unsigned long len;
  int fd;
  int bytes;
  int status;
  byte *pbuf;

  unsigned long evtcount;

  if((mifi = (MIDIFile *) malloc(sizeof(MIDIFile))) == NULL){
	 if(verbose > 0)
		fputs("Error: could not alloc memory for MIDIFile",stderr);
	 return NULL;
  };

  if((mifi->MThd = (MThd *) malloc(sizeof(MThd))) == 0l){
	 if(verbose > 0)
		fputs("Error: could not alloc memory for MIDIFile",stderr);
	 free(mifi);
	 return NULL;
  };

  mifihd = mifi->MThd;

  if(filename == NULL){
	 mifi->name = name = NULL;
	 fd = STDIN_FILENO;
	 if(verbose > 2)
		fprintf(stderr,"Reading midifile from STDIN\n");

	 }
  else{
	 name = filename;

	 if((mifi->name = (char *) malloc(strlen(name)+1)) == NULL){
		if(verbose > 0)
		  fputs("Error: could not alloc memory for name of MIDIFile",stderr);
		free(mifi->MThd);
		free(mifi);
		return NULL;
	 }

	 strcpy(mifi->name,name);

	 if( (fd = open(mifi->name,O_RDONLY)) < 0){
		if(verbose > 0)
		  fprintf(stderr,"Error: could not open MIDIFile %s\n",mifi->name);
		if(mifi->name)free(mifi->name);free(mifi->MThd);
		free(mifi);
		return NULL;
	 };

	 if(verbose > 1)
		fprintf(stderr,"Reading midifile %s ...\n",mifi->name);

  }/* else NULL filename */

  /* Header einlesen */
  if( (read(fd,buf,  14) <= 0)){
	 if(verbose > 0)
		fprintf(stderr,"Error: EOF or read error on MIDIFile %s\n",
				  (mifi->name?mifi->name:"stdin"));
	 if(mifi->name)free(mifi->name);
	 free(mifi->MThd);
	 free(mifi);
	 if(mifi->name)close(fd);
	 return NULL;
  }

  /* und testen auf MThd */

  mifihd->type = "MThd";

  if( strncmp(mifihd->type,buf,4) != 0){
	 if(verbose > 0)
		fprintf(stderr,"Error: %s is no MIDIFile \n",
				  (mifi->name?mifi->name:"stdin"));
	 if(mifi->name)free(mifi->name);
	 free(mifi->MThd);
	 free(mifi);
	 if(mifi->name)close(fd);
	 return NULL;
  }

  if(verbose > 2){
	 fprintf(stderr,"MThd read:");
	 for(i=0;i<14;i++)
		fprintf(stderr," %02x",buf[i]);
	 fprintf(stderr,"\n");
  }

  mifihd->length = 6;
  mifihd->format = buf[8] * 0x100 + buf[9];
  mifihd->ntrks = buf[10] * 0x100 + buf[11];
  mifihd->divisions = buf[12] * 0x100 + buf[13];
  mifihd->starttrack = NULL;

  if(verbose > 1)
	 fprintf(stderr,"MIDIhead read with %d tracks\n",mifihd->ntrks);

  /* read tracks */
  ptrk = NULL;

  for(i=0;i<mifihd->ntrks;i++){

	 evtcount = 0l;
	 pe = NULL;

	 if((trk = (MTrk *) malloc(sizeof(MTrk))) == NULL){
		if(verbose > 0)
		  fputs("Error: could not alloc MTrk memory for MIDIFile\n",stderr);
		if(mifi->name)free(mifi->name);free(mifi->MThd);free(mifi);
		if(mifi->name)close(fd);
		return NULL;
	 };

	 /* read trk header */
	 if( (read(fd,buf,4+4) <= 0)){
		if(verbose > 0)
		  fprintf(stderr,"Error: EOF or read error MTrk on %s\n",
					 (mifi->name?mifi->name:"stdin"));
		free(trk);free(mifi->MThd);if(mifi->name)free(mifi->name);free(mifi);	 
		if(mifi->name)close(fd);
		return NULL;
	 }

	 if(verbose > 2){
		fprintf(stderr,"MTrk read:");
		for(j=0;j<(4+4);j++)
		  fprintf(stderr," %02x",buf[j]);
		fprintf(stderr,"\n");
	 }

    if(verbose > 1)
		fprintf(stderr,"Reading miditrack %d:",i);


	 /* test on type */
	 if(strncmp(buf,"MTrk",4) != 0){
		if(verbose > 0)
		  fprintf(stderr,"Error: reading MTrk type on MIDIFile %s\n",
					 (mifi->name?mifi->name:"stdin"));
		free(trk);free(mifi->MThd);free(mifi);	 if(mifi->name)close(fd);
		return NULL;
	 }

	 trk->type = "MTrk";
	 trk->length = (long) buf[4]*0x1000000l + (long) buf[5]*0x10000l
		+ (long) buf[6]*0x100l + (long) buf[7];

	 if(mifihd->starttrack == NULL)
		mifihd->starttrack = trk;
	 else{
		ptrk->next = trk;
		ptrk = trk;
	 }

	 trk->next = NULL;
	 trk->events = NULL;
	 pe = NULL;

	 /* read MTevt */

	 if(verbose > 1)fprintf(stderr,"len=%ld bytes\n",trk->length);

	 j= 0l;

	 while(j < trk->length){

		evtcount++;

		if(verbose > 2)
		  fprintf(stderr,"%ld=",j);

		if((me = (MTevt *) malloc(sizeof(MTevt))) == 0l){
		  if(verbose > 0)
			 fputs("Error: could not alloc MTevt memory for MIDIFile\n",stderr);
		  mf_free(mifi);
		  if(mifi->name)close(fd);
		  return NULL;
		};

		me->data.status = MD_NOTE_OFF; /* no sysex or meta (for correct free)*/

		me->next = NULL;
		me->prev = NULL;
		/* first event ? make double linked list*/

		if(trk->events == NULL)
		  trk->events = me;

		if(pe != NULL){
		  pe->next = me;
		  me->prev = pe;
		}

		pe = me;

		if( (count=read_varlen(fd, &(me->dt))) <= 0){
		  if(verbose > 0)
			 fprintf(stderr,"Error: reading varlen for MTevt %ld\n",me->dt);
		  mf_free(mifi);
		  if(mifi->name)close(fd);
		  return NULL;
		};
		j+=count;

		/* read event */
		if( (count=read(fd,buf,1)) <= 0){
		  if(verbose > 0)
			 fputs("Error: reading data for MTevt\n",stderr);
		  mf_free(mifi);
		  if(mifi->name)close(fd);
		  return NULL;
		};
		j+=count;

		if(verbose > 2)fprintf(stderr,",status=%x,dt=%ld,j=%ld:",*buf,me->dt,j);

		if(*buf == MF_META_EVENT){

		  if(verbose > 2)fprintf(stderr,"M:");

		  if((meta = (METAstr *) malloc(sizeof(METAstr))) == 0l){
			 if(verbose > 0)
				fputs("Error:could not alloc METAstr memory for MIDIFile\n",stderr);
			 mf_free(mifi);
			 if(mifi->name)close(fd);
			 return NULL;
		  };

 		  me->data.status = *buf;
		  me->data.data.meta_data = meta;

		  meta->len = 0;

		  if( (count=read(fd,buf,1)) <= 0){
			 if(verbose > 0)
				fputs("Error: reading data for MTevt\n",stderr);
			 mf_free(mifi);
			 if(mifi->name)close(fd);
			 return NULL;
		  };

		  j+=count;
		  meta->type = *buf;

		  if( (count=read_varlen(fd, &len)) <= 0){
			 if(verbose > 0)
				fputs("Error: reading varlen for METAstr\n",stderr);
			 mf_free(mifi);
			 if(mifi->name)close(fd);
			 return NULL;
		  };
		  j+=count;

		  meta->len = (int) len;

		  if(verbose > 2)
			 fprintf(stderr,",META=%x,len=%d\n",meta->type,meta->len);

		  if(meta->type == 0x2F){ /* end of track */
			 if(verbose > 2)
				fprintf(stderr,"End of track %d\n",i);

			 if(j != trk->length){
				if(verbose > 0)
				  fprintf(stderr,
							 "Warning:End of MTrk%d with wrong length %ld"
							 " instead of %ld ??\n"
							 ,i,j,trk->length);
			 }
		  }

		  /* plus 1 for strings to put a 0 at the end */
		  if((meta->data = (byte *) malloc((size_t) len+1)) == 0l){
			 if(verbose > 0)
				fprintf(stderr,
						  "Error:could not alloc %ld bytes METAstr memory"
						  "for MIDIFile\n",len+1);
			 mf_free(mifi);
			 if(mifi->name)close(fd);
			 return NULL;
		  };


		  if(meta->len > 0){
			 if( (count=read(fd,meta->data,meta->len)) <= 0){
				if(verbose > 0)
				  fputs("Error: reading data for METAstr\n",stderr);
				mf_free(mifi);
				if(mifi->name)close(fd);
				return NULL;
			 };
		  j+=count;
		  };
		}
		else if (*buf == MF_SYSEX_EVENT || *buf == MF_ESC_EVENT){

		  MIDIstr *sysex;

		  if(verbose > 0)fprintf(stderr,"S:");

		  if((sysex = (MIDIstr *) malloc(sizeof(MIDIstr))) == 0l){
			 if(verbose > 0)
				fputs("Error:could not alloc MIDIstr memory for MIDIFile\n",stderr);
			 mf_free(mifi);
			 if(mifi->name)close(fd);
			 return NULL;
		  };

 		  me->data.status = *buf;
		  me->data.data.sysex_data = sysex;
		  
		  sysex->len = 0;

		  if( (count=read_varlen(fd, &len)) <= 0){
			 if(verbose > 0)
				fputs("Error: reading varlen for MIDIstr\n",stderr);
			 mf_free(mifi);
			 if(mifi->name)close(fd);
			 return NULL;
		  };
		  j+=count;
		  
		  sysex->len = (int) len;
		  if(verbose > 2)fprintf(stderr,",len=%d\n",sysex->len);

		  if((sysex->data = (byte *) malloc((size_t) len)) == 0l){
			 if(verbose > 0)
				fputs("could not alloc MIDIstr->data memory for MIDIFile\n",stderr);
			 mf_free(mifi);
			 if(mifi->name)close(fd);
			 return NULL;
		  };


		  if(sysex->len > 0){
			 if( (count=read(fd,sysex->data,sysex->len)) < sysex->len){
				if(verbose > 0)
				  fputs("Error: reading data for MIDIstr\n",stderr);
				mf_free(mifi);
				if(mifi->name)close(fd);
				return NULL;
			 };
			 j+=count;
		  };

		}
		else if(*buf >= 0xF8){ /* MIDI realtime event */
			 me->data.status = *buf;
			 if(verbose > 2)fprintf(stderr,"RT:%x\n",*buf);
		}
		else{ /* midi event */

		  if(verbose > 2)fprintf(stderr,"E:");

		  /* how many MIDI bytes ? */

		  if((*buf & 0x80) == 0x80){ /* new status */
			 me->data.status = status = *buf;
			 pbuf = &me->data.data.message[0];
		  };

		  bytes = md_msglen(status);

		  if((*buf & 0x80) == 0x00){ /* running status */
			 me->data.data.message[0] = *buf;
			 me->data.status = status;
			 bytes--;
			 pbuf = &(me->data.data.message[1]);
		  }

		  if( bytes > 0){
			 if((count=read(fd,pbuf,bytes)) <= 0){
				if(verbose > 0)
				  fprintf(stderr,
							 "\nError: reading data for MIDI Message:%d"
							 " bytes status=%x\n",
							 bytes,*buf);
				mf_free(mifi);
				if(mifi->name)close(fd);
				return NULL;
			 };
			 j+=count;
		  };

		  if(verbose > 2){
			 if((*buf & 0x80)==0)bytes++;
			 if(verbose > 2)
				fprintf(stderr,"status=%x",me->data.status);
			 if(bytes >= 1)fprintf(stderr,",%x",me->data.data.message[0]);
			 if(bytes >= 2)fprintf(stderr,",%x",me->data.data.message[1]);
			 if(bytes >= 3)fprintf(stderr,",...");
			 fprintf(stderr,"\n");
		  }
		} /* read MIDI event */
	 } 

	 if(verbose > 1)
		fprintf(stderr,"read Track with %ld Events\n",
				 evtcount);

  }; /* for each track */


  if(mifi->name)close(fd); 
  
  return mifi;
}

void mf_free(MIDIFile *mifi)
{
  MTevt *me,*nme;
  MTrk *ntrk;
  MTrk *trk = mifi->MThd->starttrack;

  while(trk != NULL){

	 me = trk->events;
	 while(me != NULL){

		if(me->data.status == MF_META_EVENT){
		  if(me->data.data.meta_data->len > 0)
			 free(me->data.data.meta_data->data);
		  free(me->data.data.meta_data);
		}
		else if(me->data.status == MF_SYSEX_EVENT || 
				  me->data.status == MF_ESC_EVENT){
		  if(me->data.data.sysex_data->len > 0)
			 free(me->data.data.sysex_data->data);
		  free(me->data.data.sysex_data);
		}

		nme = me->next;
		free(me);
		me = nme;		  
	 };

	 ntrk = trk->next;
	 free(trk);
	 trk = ntrk;
  };

  free(mifi->MThd);
  if(mifi->name)free(mifi->name);
  free(mifi);

  return;
}


MIDIFile *mf_save(MIDIFile *mifi,char *outfile)
{
  MTevt *me,*pme;
  MTrk *trk;
  unsigned long count;
  int fd;
  byte vnp[MF_MAX_VARNUM_LEN];
  int i,j;
  unsigned long evtcount;

  if(outfile == NULL){
	 fd =  STDOUT_FILENO;
  }
  else{
	 
	 if( (fd = open(outfile,O_WRONLY|O_CREAT|O_TRUNC,
						 S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP)) < 0){
		if(verbose > 0)
		  fprintf(stderr,"Error: could not open MIDIFile %s for writing\n",
					 outfile);
		return NULL;
	 };
  }

  strcpy(buf,"MThd");
  buf[4] = buf[5] = buf[6] = 0;  buf[7] = 6;
  buf[8] = (byte) (mifi->MThd->format>>8) & 0xFF;
  buf[9] = (byte) mifi->MThd->format & 0xFF;
  buf[10] = (byte) (mifi->MThd->ntrks>>8) & 0xFF;
  buf[11] = (byte) mifi->MThd->ntrks & 0xFF;
  buf[12] = (byte) (mifi->MThd->divisions>>8) & 0xFF;
  buf[13] = (byte) mifi->MThd->divisions & 0xFF;
 
  if(write(fd,buf,14) <= 0){
	 if(verbose > 0)
		fprintf(stderr,"Error: writing MThd to file %s\n",
				  (outfile?outfile:"stdout"));
	 if(outfile)close(fd);
	 return NULL;
  };

  trk = mifi->MThd->starttrack;

  while(trk != NULL){

	 evtcount = 0l; 

	 strcpy(buf,"MTrk");
	 i=4;

	 /* calculate trk len */
	 count = 0l;
	 me = trk->events;
	 pme = NULL;
	 while(me != NULL){
		count +=  mf_int_to_varnum(vnp,me->dt); /* delta time */

		if(pme == NULL ||  
			me->data.status >= 0xF7 || me->data.status == 0xF0 || 
			me->data.status != pme->data.status ||
			md_msglen(me->data.status) == 0)
		count++; /*status byte */
  
		if(me->data.status == MF_META_EVENT){
		  count++; /* type byte */
		  count += mf_int_to_varnum(vnp,me->data.data.meta_data->len);
		  count += me->data.data.meta_data->len;		  
		}
		else if(me->data.status == MF_SYSEX_EVENT 
				  || me->data.status == MF_ESC_EVENT){
		  count += mf_int_to_varnum(vnp,me->data.data.sysex_data->len);
		  count += me->data.data.sysex_data->len;		  
		}
		else{ /* MIDI Message */
		  count += md_msglen(me->data.status);
		}

		if(me->data.status < 0xF8 || me->data.status == 0xFF)
		  pme = me;

		me = me->next;
	 }; /* count trk len */


	 if(verbose >2)
		fprintf(stderr,"Track count=%ld\n",count);

	 buf[4] = (byte) (count>>24 & 0xFFl);
	 buf[5] = (byte) (count>>16 & 0xFFl);
	 buf[6] = (byte) (count>>8 & 0xFFl);
	 buf[7] = (byte) (count & 0xFFl);

	 if(write(fd,buf,4+4) <= 0){
		if(verbose > 0)
		  fprintf(stderr,"Error: writing MTrk to file %s\n",
					 (outfile?outfile:"stdout"));
		if(outfile)close(fd);
		return NULL;
	 };

	 /* write events */
	 me = trk->events;
	 pme = NULL;
	 while(me != NULL){

		evtcount++;

		/* delta time */
		mf_int_to_varnum(vnp,me->dt);
		i = mf_copy_varnum(buf,vnp);

		if(verbose>2)
		  fprintf(stderr,"writing dt=%ld(%d)",me->dt,i);

		/* if previous event  status different or REALTIMe no running status */
		if(pme == NULL ||  
			me->data.status >= 0xF7 || me->data.status == 0xF0 || 
			me->data.status != pme->data.status ||
			md_msglen(me->data.status) == 0)
		  buf[i++] = me->data.status;

		if(me->data.status == MF_META_EVENT){

		  buf[i++] = me->data.data.meta_data->type;

		  mf_int_to_varnum(vnp,(unsigned long) me->data.data.meta_data->len);
		  i += mf_copy_varnum(&(buf[i]),vnp);

		  for(j=0;j<me->data.data.meta_data->len;j++)
			 buf[i+j] = me->data.data.meta_data->data[j];
		  i+=j;

		  if(verbose > 2)
			 fprintf(stderr,"write METAstr %x len=%d(%d)\n",
					  me->data.data.meta_data->type,
					  me->data.data.meta_data->len,i);

		}
		else if(me->data.status == MF_SYSEX_EVENT || 
				  me->data.status == MF_ESC_EVENT){

		  mf_int_to_varnum(vnp,me->data.data.sysex_data->len);
		  i+=mf_copy_varnum(buf,vnp);

		  for(j=0;j<me->data.data.sysex_data->len;j++)
			 buf[i+j] = me->data.data.sysex_data->data[j];
		  i+=j;
		  if(verbose > 2)
			 fprintf(stderr,"write SYSEXstr len=%d(%d)\n",
					  me->data.data.meta_data->len,i);
		}
		else{
		  
		  for(j=0;j<md_msglen(me->data.status);j++)
			 buf[i+j] = me->data.data.message[j];
		  i+=j;

		  if(verbose > 2)
			 fprintf(stderr,"write MIDI %x len=%d(%d)\n",
					  me->data.status,j,i);
		}

		if(write(fd,buf,i) <= 0){
		  if(verbose > 0)
			 fprintf(stderr,"Error: writing MTevt to file %s\n",
						(outfile?outfile:"stdout"));
		  if(outfile)close(fd);
		  return NULL;
		};

		/* exept REALTIME use for running status */
		if(me->data.status < 0xF8 || me->data.status == 0xFF)
		  pme = me;
		me = me->next;
	 };

	 trk = trk->next;

	 if(verbose > 1)
		fprintf(stderr,"wrote Track with %ld Events\n",
				 evtcount);
  };



  if(outfile)
	 close(fd);

  if(verbose > 1)
	 fprintf(stderr,"midifile written on %s\n",(outfile?outfile:"stdout"));

  return mifi;
}

/* print the midifile */
int pheader = 1;
int ptracks = 1;
int pmeta = 0;
int psysex = 0;
int pmidi = 0;

void mf_print(MIDIFile *mifi)
{
  MThd *mifihd = mifi->MThd;
  MTrk *trk;
  MTevt *me;
  int i,j;
  unsigned long time;
  unsigned long evtcount;

  if(pheader > 0){

	 printf("MIDI-File %s \n", (mifi->name?mifi->name:"stdin"));
	 printf(" type:    %s\n",mifihd->type);
	 printf(" length:  %ld\n",mifihd->length);
	 printf(" format:  %d\n",mifihd->format);
	 printf(" ntracks: %d\n",mifihd->ntrks);

	 if(mifihd->divisions & 0x8000){
		printf("Time:   SMPTE, %d fr/s,%d ticks/fr\n",
				 -((mifihd->divisions & 0x7F00)>>8),
				 mifihd->divisions & 0x00FF);	
	 }
	 else{
		printf("Time: MIDI, Ticks/Quarter:  0x%x = %d \n",
				 mifihd->divisions,mifihd->divisions & 0x7FFF );
	 };
  }

  trk = mifihd->starttrack;
  i=0;

  while(trk != NULL){

	 if(ptracks > 0)
		printf("Track %d:length=%ld Bytes\n",i,trk->length);

	 me = trk->events;
	 j=0;
	 time = 0l;


	 evtcount = 0l;
	 while(me != NULL){

		time += me->dt;
		evtcount++;

		switch(me->data.status){

		case MF_META_EVENT :

		  if(pmeta > 0){
			 printf("%5d:time=%7ld:",j,time);
			 mf_print_meta(me->data.data.meta_data); 
		  }
		  break;

		case MF_SYSEX_EVENT :
		  if(psysex > 0){
			 printf("%5d:time=%7ld:",j,time);
			 mf_print_sysex(me->data.data.sysex_data);
		  }
			 break;

		case MF_ESC_EVENT :
		  if(psysex > 0){
			 printf("%5d:time=%7ld:",j,time);
			 mf_print_escape(me->data.data.sysex_data);
		  }
		  break;

		default: /* MIDI event */
		  if(pmidi > 0){
			 printf("%5d:time=%7ld:",j,time);
			 mf_print_midi(&(me->data));
		  }
		  break;
		}


		j++;
		me = me->next;
	 } /* while MTevts */

	 if(ptracks>0)printf("%ld MIDIFile events in Track %d\n",evtcount,i);
	 i++;
	 trk = trk->next;
  } /* while trk */
}


void mf_print_meta(METAstr *meta)

{
  int i;
  printf("MetaEvent %2x(l=%d):",
					meta->type,
					meta->len);

  
  switch(meta->type){
  case MF_META_SEQNR    :
	 printf("sequenz number %d\n",(((int) meta->data[0])<<8)+meta->data[1]);
	 break;

  case MF_META_TEXT     :
	 meta->data[meta->len]=0;
	 printf("text(%2d): %s\n",meta->len,meta->data);
	 break;

  case MF_META_CPRGHT   :
	 meta->data[meta->len]=0;
	 printf("copyright(%2d): %s\n",meta->len,meta->data);
	 break;

  case MF_META_TRKNAME  :
	 meta->data[meta->len]=0;
	 printf("track name(%2d): %s\n",meta->len,meta->data);
	 break;

  case MF_META_INSTNAME :
	 meta->data[meta->len]=0;
	 printf("instrument name(%2d): %s\n",meta->len,meta->data);
	 break;

  case MF_META_LYRIC    :
	 meta->data[meta->len]=0;
	 printf("lyrics(%2d): %s\n",meta->len,meta->data);
	 break;

  case MF_META_MARKER   :
	 meta->data[meta->len]=0;
	 printf("marker(%2d): %s\n",meta->len,meta->data);
	 break;

  case MF_META_CUEPOINT :
	 meta->data[meta->len]=0;
	 printf("cue point(%2d): %s\n",meta->len,meta->data);
	 break;
	
  case MF_META_MIDICHAN :
	 printf("midi channel prefix: %d\n",meta->data[0]);
	 break;

  case MF_META_ENDOFTRK :
	 printf("end of track\n");
	 break;

  case MF_META_SETTEMPO :
	 printf("tempo = %f sec/quarternote\n",
			  (float)(((int) (meta->data[0]<<16))+(int) (meta->data[1]<<8)
						 +meta->data[3])/1000000.0);
	 break;

  case MF_META_SMPTEOFF :
	 printf("SMPTE offset: %d hr %d min %d sec %f frames\n",
			  meta->data[0], meta->data[1] ,meta->data[2] ,
			  (float) meta->data[3] + (float) meta->data[4]/100.0 );
	 break;

  case MF_META_TIMESIGN :
	 printf("time signature: %d/%d, %d clocks/beat,%d 32nd per quarter\n",
			  meta->data[0],(int) 1<<meta->data[1],meta->data[2],meta->data[3]);
	 break;

  case MF_META_KEYSIGN  :
	 printf("key signature: %d %s, %s\n",
			  abs((signed char) meta->data[0]),
			  ((signed char) meta->data[0] >= 0)?"sharps":"flats",
			  (meta->data[1]==1)?"major":"minor");
	 break;

  case MF_META_SEQSPEC  :
	 printf("sequencer specific data (len=%d)>",meta->len);
	 for(i=0;i<meta->len;i++)printf(" %x",meta->data[i]);
	 printf("<\n");
	 break;

  default:
	 printf("unknown meta event (len=%d)>",meta->len);
	 for(i=0;i<meta->len;i++)printf(" %x",meta->data[i]);
	 printf("<\n");
	 break;

  }
  
}

void mf_print_sysex(MIDIstr *sysex)

{
  int i;
  printf("SysEx data (len=%d)>",sysex->len);
  for(i=0;i<sysex->len;i++)printf(" %x",sysex->data[i]);
  printf("<\n");
  return;
}

void mf_print_escape(MIDIstr *sysex)

{
  int i;
  printf("Escaped SysEx data (len=%d)>",sysex->len);
  for(i=0;i<sysex->len;i++)printf(" %x",sysex->data[i]);
  printf("<\n");
  return;
}

void mf_print_midi(MIDIevt *evt)
{
  switch(evt->status & 0xF0){

  case MD_NOTE_ON :
	 printf("noteon(%2d): %3d vel %3d\n",
			  (evt->status & 0x0F),evt->data.message[0],evt->data.message[1]);
	 break;
  case MD_NOTE_OFF :
	 printf("noteoff(%2d): %3d vel %3d\n",
			  (evt->status & 0x0F),evt->data.message[0],evt->data.message[1]);
	 break;
  case MD_POLY_TOUCH :
	 printf("polytouch(%2d): %3d  %3d\n",
			  (evt->status & 0x0F),evt->data.message[0],evt->data.message[1]);
	 break;
  case MD_CONTROL_CHANGE :
	 printf("controlchange(%2d): %3d  %3d\n",
			  (evt->status & 0x0F),evt->data.message[0],evt->data.message[1]);
	 break;
  case MD_PROGRAM_CHANGE :
	 printf("programchange(%2d): %3d\n",
			  (evt->status & 0x0F),evt->data.message[0]);
	 break;
  case  MD_CHANNEL_TOUCH :
	 printf("channeltouch(%2d): %3d \n",
			  (evt->status & 0x0F),evt->data.message[0]);
	 break;
  case  MD_PITCH_BEND :
	 printf("pitchbend(%2d): %3d  %3d\n",
			  (evt->status & 0x0F),evt->data.message[0],evt->data.message[1]);
	 break;
  case  MD_SYS_MESSAGES :

	 switch(evt->status){

		case MD_SYS_MTC_QUARTER :
		  printf("MTC quarter clock: %3d  \n",
					evt->data.message[0]);
		break;
		case MD_SYS_SONG_POSITION :
		  printf("song position: %3d  %3d\n",
					evt->data.message[0],evt->data.message[1]);
		break;
		case MD_SYS_SONG_SELECT :
		  printf("song select: %3d  %3d\n",
					evt->data.message[0],evt->data.message[1]);
		break;
		case MD_SYS_TUNE_REQUEST :
		  printf("tune request\n");
		break;
		case MD_SYS_TIMING_CLOCK :
		  printf("timing clock\n");
		break;
		case MD_SYS_START :
		  printf("realtime start\n");
		break;
		case MD_SYS_CONTINUE :
		  printf("realtime continue\n");
		break;
		case MD_SYS_STOP :
		  printf("realtime stop\n");
		break;
		case MD_SYS_ACTIVE_SENSE :
		  printf("active sensing\n");
		break;
		case MD_SYS_SYSTEM_RESET :
		  printf("system reset - ERROR never in MIDI-File !!!\n");
		break;

		default:
		  printf("Undefined Status %x in MIDI-Specs1.0\n",evt->status);
		break;	 
	 };
	 break;

  default:
	 printf("Undefined Status %x in MIDI-Specs1.0\n",evt->status);
	 break;	 
  };

  return;
}
/* ******************* HELPS ******************* */
int read_varlen(int fd,unsigned long *v)
{
  int i,j;
  unsigned char buf[10]; /* more than long */

  i=0;
  while( (j=read(fd,&buf[i],1)) > 0 && (buf[i]&0x80) == 0x80 && i<10)
	 i++;

  if(j<=0)
	 return 0;
	
  *v = mf_varnum_to_int(buf);

  return i+1;
}


unsigned long mf_varnum_to_int(byte *data)

{
  int i;
  unsigned long dt;

  dt= 0;
  i=0;

  while(data[i] & 0x80){
	 dt = dt * 0x80l + (unsigned long) (data[i] & 0x7F);
	 i++; 
  };

  dt = dt + (unsigned long) data[i];

  return(dt);
}


int mf_int_to_varnum(byte *b, unsigned long num)
{
  int i,j;
  
  b[0] = (byte) (num & 0x7Fl); /* last byte */
  i=1;
  while(num > 0x7Fl && i < 10){

 	 num = num>>7;

	 for(j=i;j>0;j--)
		  b[j] = b[j-1];

	 b[0] = (byte) (num & 0x7Fl) | 0x80;
	 i++;
  }

  return i ;
}

int mf_copy_varnum(byte *dest,const byte *src)
{
int i;

for(i=0;(src[i] & 0x80) != 0;i++)
  dest[i]=src[i];

dest[i]=src[i];

return i+1;
}

/* ------ SETTINGS -------------- */

int mf_verbose(int n)
{
if(n<0)
  return verbose;

if(n>MF_VERB_DEBUG)
  verbose = MF_VERB_DEBUG;

return verbose = n;
}



/* ------ MAIN -------------- */
#ifdef MAIN

void usage(char **argv) {
	fprintf(stderr,"usage: %s [-<options>] [<midi-file>] \n", argv[0]);
	fprintf(stderr,
			  "   if no <midi-file> argument, using name \'midifile.mid\' \n");
	fprintf(stderr,
			  "options:\n"
			  " -v <n> verbosity during reading"
			  " (0=no,1=errors,2=+trks,3=debug)\n");
	fprintf(stderr,
			  " -q ... quiet, no control output\n");

	fprintf(stderr,
			  " -f[h|H][t|T][m|M][s|S][e|E] ... print format\n"
			  "  where h/H ... no header/print header\n"
			  "        t/T ... no track header/print track header\n"
			  "        m/M ... no Metaevents/print Metaevents\n"
			  "        s/S ... no SYSEX/print SYSEX\n"
			  "        e/E ... no MIDI Event/print MIDI Event\n"
			  "        q/A ... no printing/printing ALL\n"
			  );
	return;
}


int main(int argc,char *argv[])

{
  MIDIFile *mymifi;
  int i;
  char *pchar;
  char *filename = NULL;
  char *outfile = NULL;
  int quiet = 0;
  int v = 1; /* default, errors only */

  for(i=1;i < argc;i++){
	 
	 /* option or filename */
	 if(argv[i][0] == '-'){ 

		switch(argv[i][1]){

		case 'q': /* verbosity */
		  quiet=1;
		  break;


		case 'f': /* what to print */
		  
		  pchar = &(argv[i][1]);

		  while(*(pchar++)){
			 switch(*pchar){
			 case 'h': pheader=0;break;
			 case 'H': pheader=1;break;
			 case 't': ptracks=0;break;
			 case 'T': ptracks=1;break;
			 case 'm': pmeta=0;break;
			 case 'M': pmeta=1;break;
			 case 's': psysex=0;break;
			 case 'S': psysex=1;break;
			 case 'e': pmidi=0;break;
			 case 'E': pmidi=1;break;
			 case 'q': pmidi=psysex=pmeta=ptracks=pheader=0;break;
			 case 'A': pmidi=psysex=pmeta=ptracks=pheader=1;break;
			 };
		  }
			 
		  break;

		case 'v': /* verbosity */
		  
		  if(sscanf(&(argv[i][2]),"%d",&v)  != 1){
		  
			 if((i+1) >= argc){
			 	fprintf(stderr,"-v <n>: no argument n, using n=1"); 
				v = 1;
			 }
			 else if(sscanf(&(argv[i+1][0]),"%d",&v)  != 1){
				fprintf(stderr,"-v <n>: no n, using n=1"); 
				v = 1;
			 }
			 else
				i++;
		  };

		  if(v < 0 || v > 3){
			 if(v > 3)v=3;
			 else v=0;
			 fprintf(stderr,"-v <n>: n should be between 0...3 using %d\n",v); 
		  };
		  break;

		case 'o':

		  if(argv[i][2] != 0)
		    outfile = &(argv[i][2]);
		  else
		    if((i+1) >= argc || argv[i+1][0] == 0){
				fprintf(stderr,"-o:cannot get outputname \n");
				usage(argv);
				exit(1);
			 }
			 else 
				outfile = argv[++i];

		  break;

		  default: 
			 fprintf(stderr,"Unkown option -%c \n",argv[i][0]);
		  case 'h': 
			 fprintf(stderr,"Midifile Parser V0.1 (c)iem 1998: "
						"reads Midifile outputs ASCII-Listing\n"); 
			 usage(argv); 
			 exit(1);
		}
	 }
	 else{
		/* filename */
		if(filename == NULL)
		  filename = argv[i];
		else{
		  fprintf(stderr,"Dont know what to do with %s\n",argv[i]);
		  usage(argv);
		  exit(1);
		}
	 }
  };

  if(!quiet)fprintf(stderr,"MIDI-FILE to ASCCII verbose mode %d:\n",v);

  mf_verbose(v);
  
  if(!quiet)fputs("reading ...",stderr);
  
  if((mymifi = mf_read(filename)) != NULL){

	 if(!quiet)fputs(" done\n",stderr);
	 
	 if(!quiet)fputs("printing:\n",stderr);

	 mf_print(mymifi);

	 if(!quiet)fputs("printing done\n",stderr);


	 if(outfile){
		if(!quiet)fprintf(stderr,"saving on %s ...",outfile);
		mf_save(mymifi,outfile);
		if(!quiet)fputs(" done\n",stderr);
	 }

 	 mf_free(mymifi); 
	 if(!quiet)fputs("all done ...\n    ... good bye !\n",stderr);

	 return(0);
  };

  if(!quiet)
	 fputs("something wrong happened trying to read ...\n    ... good bye !\n",
			 stderr);
  return(-1);
}
#endif
