/* Copyright (c) 1997 The Regents of the University of California.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

/* network */

#include "m_imp.h"

#include <sys/types.h>
#include <string.h>
#ifdef UNIX
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#define SOCKET_ERROR -1
#else
#include <winsock.h>
#endif

static t_class *netsend_class;

typedef struct _netsend
{
    t_object x_obj;
    int x_fd;
} t_netsend;

static void *netsend_new(void)
{
    t_netsend *x = (t_netsend *)pd_new(netsend_class);
    x->x_fd = -1;
    return (x);
}

static void netsend_connect(t_netsend *x, t_symbol *hostname,
    t_floatarg fportno)
{
    struct sockaddr_in server;
    struct hostent *hp;
    int sockfd;
    int portno = fportno;
    if (x->x_fd >= 0)
    {
    	error("netsend_connect: already connected");
    	return;
    }

    	/* create a socket */
    sockfd = socket(AF_INET, SOCK_STREAM, 0);
    if (sockfd < 0)
    {
    	sys_sockerror("socket");
    	return;
    }
    /* connect socket using hostname provided in command line */
    server.sin_family = AF_INET;
    hp = gethostbyname(hostname->s_name);
    if (hp == 0)
    {
	post("bad host?\n");
	return;
    }
    memcpy((char *)&server.sin_addr, (char *)hp->h_addr, hp->h_length);

    /* assign client port number */
    server.sin_port = htons((u_short)portno);

    post("connecting to port %d", portno);
	/* try to connect.  LATER make a separate thread to do this
	because it might block */
    if (connect(sockfd, (struct sockaddr *) &server, sizeof (server)) < 0)
    {
    	sys_sockerror("connecting stream socket");
    	sys_closesocket(sockfd);
    	return;
    }
    x->x_fd = sockfd;
}

static void netsend_disconnect(t_netsend *x)
{
    if (x->x_fd >= 0)
    {
    	sys_closesocket(x->x_fd);
    	x->x_fd = -1;
    }
}

static void netsend_send(t_netsend *x, t_symbol *s, int argc, t_atom *argv)
{
    if (x->x_fd > 0)
    {
	t_binbuf *b = binbuf_new();
	char *buf, *bp;
	int length, sent;
	t_atom at;
	binbuf_add(b, argc, argv);
	SETSEMI(&at);
	binbuf_add(b, 1, &at);
	binbuf_gettext(b, &buf, &length);
	for (bp = buf, sent = 0; sent < length;)
	{
    	    int res = send(x->x_fd, buf, length-sent, 0);
    	    if (res <= 0)
    	    {
    		sys_sockerror("netsend");
    		break;
    	    }
    	    else
    	    {
    		sent += res;
    		bp += res;
    	    }
	}
	t_freebytes(buf, length);
	binbuf_free(b);
    }
    else error("netsend: not connected");
}

static void netsend_free(t_netsend *x)
{
    netsend_disconnect(x);
}

static void netsend_setup(void)
{
    netsend_class = class_new(gensym("netsend"), netsend_new, netsend_free,
    	sizeof(t_netsend), 0, 0);
    class_addmethod(netsend_class, (t_method)netsend_connect,
    	gensym("connect"), A_SYMBOL, A_FLOAT, 0);
    class_addmethod(netsend_class, (t_method)netsend_disconnect,
    	gensym("disconnect"), 0);
    class_addmethod(netsend_class, (t_method)netsend_send, gensym("send"),
    	A_GIMME, 0);
}

/* ----------------------------- netreceive ------------------------- */

static t_class *netreceive_class;

typedef struct _netreceive
{
    t_object x_obj;
    int x_connectsocket;
} t_netreceive;

static void netreceive_connectpoll(t_netreceive *x)
{
    t_socketreceiver *y;
    int fd;

    fd = accept(x->x_connectsocket, 0, 0);
    y = socketreceiver_new();
    sys_addpollfn(fd, (t_fdpollfn)socketreceiver_read, y);
}

static void *netreceive_new(t_floatarg fportno)
{
    t_netreceive *x;
    struct sockaddr_in server;
    int sockfd, portno = fportno;
    
    	/* create a socket */
    sockfd = socket(AF_INET, SOCK_STREAM, 0);
    if (sockfd < 0)
    {
    	sys_sockerror("socket");
    	return (0);
    }
    server.sin_family = AF_INET;
    server.sin_addr.s_addr = INADDR_ANY;

#ifdef UNIX
    	/* this seems to work only in unix ...  ask Mark about it */
    if (setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, 0, 0) < 0)
    	post("setsockopt failed\n");
#endif

    	/* assign server port number */
    server.sin_port = htons((u_short)portno);
    post("listening to port number %d", portno);

    	/* name the socket */
    if (bind(sockfd, (struct sockaddr *)&server, sizeof(server)) < 0)
    {
    	sys_sockerror("bind");
    	sys_closesocket(sockfd);
    	return (0);
    }
    if (listen(sockfd, 5) < 0)
    {
    	sys_sockerror("listen");
    	sys_closesocket(sockfd);
    	return (0);
    }

    x = (t_netreceive *)pd_new(netreceive_class);

    sys_addpollfn(sockfd, (t_fdpollfn)netreceive_connectpoll, x);

    x->x_connectsocket = sockfd;
    return (x);
}

static void netreceive_print(t_netreceive *x)
{
}

static void netreceive_free(t_netreceive *x)
{
    	/* LATER make me clean up open connections */
    sys_rmpollfn(x->x_connectsocket);
}

static void netreceive_setup(void)
{
    netreceive_class = class_new(gensym("netreceive"),
    	(t_newmethod)netreceive_new, netreceive_free,
    	sizeof(t_netreceive), 0, A_DEFFLOAT, 0);
    class_addmethod(netreceive_class, netreceive_print, gensym("print"),
    	0);
}


void x_net_setup(void)
{
    netsend_setup();
    netreceive_setup();
}

