/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

#include "driver.h"

int gus_cards_count;
gus_card_t *gus_cards[ GUS_CARDS ];
gus_card_t *gus_cards_irq_index[ 16 ];

void gus_driver_init( void )
{
  short i;
  
  for ( i = 0; i < GUS_CARDS; i++ ) gus_cards[ i ] = NULL;
  for ( i = 0; i < 16; i++ ) gus_cards_irq_index[ i ] = NULL;
}

void gus_card_init( gus_card_t *card, struct GUS_STRU_CFG *cfg, int card_number )
{
  MEMSET( card, 0, sizeof( *card ) );	/* for sure */
  card -> number = cfg -> logical - 1;
  card -> dnumber = card_number;
  strncpy( card -> id, cfg -> id, sizeof( cfg -> id ) );
  SLEEP_PREPARE( card, dma1 );
  SLEEP_PREPARE( card, dma2 );
  SLEEP_PREPARE( card, wqueue );
  SLEEP_PREPARE( card, tqueue );
  SLEEP_PREPARE( card, rqueue );
  SLEEP_PREPARE( card, synth_tmp );
  MUTEX_PREPARE( &card -> gf1.mem_alloc, memory );
  MUTEX_PREPARE( &card -> gf1.mem_alloc, memory_find );
  MUTEX_PREPARE( &card -> gf1.mem_alloc, instruments );
  gus_mixer_init( card );
  gus_engine_init( card );
  gus_gf1_timer_init( card );
  gus_gf1_daemon_init( card );
  gus_pcm_init( card );
#ifdef GUSCFG_CODEC
  gus_init_codec_pcm_pre( card );
#endif
#if !defined( GUSCFG_GF1PCM ) && !defined( GUSCFG_CODEC )
#error "You must have defined GF1 PCM or CODEC PCM!!!"
#endif
#ifdef GUSCFG_MIDI
  gus_init_gf1_midi( card );
  gus_init_gf1_uart( card );
  gus_init_midi( card );
#endif
#ifdef GUSCFG_OSS
  gus_sequencer_init( card );
#endif
  card -> gf1.port = cfg -> port;
  card -> gf1.irq = cfg -> irq;
#ifdef GUSCFG_ESS
  card -> ess.irq = cfg -> ess_irq;
#endif
  card -> dma1 = cfg -> dma1;
  card -> dma1_size = cfg -> dma1_size;
  card -> dma2 = cfg -> dma2;
  card -> dma2_size = cfg -> dma2_size;
  card -> daughter_flag = cfg -> daughter_port != 0xffff;
#ifdef GUSCFG_CODEC
  card -> codec.port = cfg -> daughter_port;	/* if zero - codec not present or MAX */
  card -> codec.irq = cfg -> daughter_irq;
#endif
  card -> dmad = cfg -> daughter_dma;
  card -> dmad_size = cfg -> daughter_dma_size;
  card -> gf1.enh_mode = 1;
  card -> gf1.pcm_memory = 1;
  card -> use_codec = 1;
  card -> joystick_dac = cfg -> joystick_dac;
}

static void gus_set_dmas( gus_card_t *card )
{
  int idx;

  card -> dmas[ GUS_DMA_GPLAY ] = &card -> dmas_data[ 0 ];
  card -> dmas[ GUS_DMA_GPLAY ] -> dma = card -> dma1;
  card -> dmas[ GUS_DMA_GPLAY ] -> rsize = card -> dma1_size * 1024;
  if ( ( card -> dma1 > 0 && card -> dma2 > 0 && card -> dma1 == card -> dma2 ) || 
       card -> daughter_flag > 0 )
    {
      card -> equal_dma = 1;
      card -> dmas[ GUS_DMA_GRECORD ] = &card -> dmas_data[ 0 ];
    }
   else
    {
      card -> dmas[ GUS_DMA_GRECORD ] = &card -> dmas_data[ 1 ];
      card -> dmas[ GUS_DMA_GRECORD ] -> dma = card -> dma2;
      card -> dmas[ GUS_DMA_GRECORD ] -> rsize = card -> dma2_size * 1024;
    }
  if ( card -> daughter_flag )
    {
      card -> dmas[ GUS_DMA_CPLAY ] =
      card -> dmas[ GUS_DMA_CRECORD ] = &card -> dmas_data[ 1 ];
      card -> dmas[ GUS_DMA_CPLAY ] -> dma = card -> dmad;
      card -> dmas[ GUS_DMA_CPLAY ] -> rsize = card -> dmad_size * 1024;
    }
   else
    {
      card -> dmas[ GUS_DMA_CRECORD ] = card -> dmas[ GUS_DMA_GPLAY ];
      card -> dmas[ GUS_DMA_CPLAY ] = card -> dmas[ GUS_DMA_GRECORD ];
    }
  for ( idx = 0; idx < GUS_DMAS; idx++ )
    if ( card -> dmas[ idx ] )
      card -> dmas[ idx ] -> ursize = card -> dmas[ idx ] -> rsize;
}

static int gus_mdetect( gus_card_t *card )
{
  int l, idx, local;
  unsigned char d;

#if 0
  PRINTK( "gus_mdetect: 0x%x\n", card -> gf1.port );
#endif
  if ( card -> detection )
    {
      /* fill register variables for speedup */
      card -> gf1.reg_page = GUSP( card, GF1PAGE );
      card -> gf1.reg_regsel = GUSP( card, GF1REGSEL );
      card -> gf1.reg_data8 = GUSP( card, GF1DATAHIGH );
      card -> gf1.reg_data16 = GUSP( card, GF1DATALOW );
      card -> gf1.reg_irqstat = GUSP( card, IRQSTAT );
      card -> gf1.reg_dram = GUSP( card, DRAM );
      card -> gf1.reg_timerctrl = GUSP( card, TIMERCNTRL );
      card -> gf1.reg_timerdata = GUSP( card, TIMERDATA );
      /* ---- */
    
      card -> pnp_flag = 0;
      card -> gf1.memory = 0;
      card -> gf1.rom_memory = 0;
      gus_i_write8( card, GF1_GB_RESET, 0 );	/* reset GF1 */
#ifdef GUSCFG_DEBUG_DETECT
      if ( ((d = gus_i_look8( card, GF1_GB_RESET )) & 0x07) != 0 )
        {
          PRINTK( "gus: [0x%x] check 1 failed - 0x%x\n", card -> gf1.port, d );
          return 1;
        }
#else
      if ( (gus_i_look8( card, GF1_GB_RESET ) & 0x07) != 0 ) return 1;
#endif
      gus_delay1( 1 );
      gus_i_write8( card, GF1_GB_RESET, 1 );	/* release reset */
      gus_delay1( 1 );
#ifdef GUSCFG_DEBUG_DETECT
      if ( ((d = gus_i_look8( card, GF1_GB_RESET )) & 0x07) != 1 )
        {
          PRINTK( "gus: [0x%x] check 2 failed - 0x%x\n", card -> gf1.port, d );
          return 1;
        }
#else
      if ( (gus_i_look8( card, GF1_GB_RESET ) & 0x07) != 1 ) return 1;
#endif

      /*
       *  maybe we have InterWave board without RAM
       */
      {
        unsigned long flags;
        unsigned char rev1, rev2;
  
        CLI( &flags );
        rev1 = gus_look8( card, GF1_GB_VERSION_NUMBER );
        gus_write8( card, GF1_GB_VERSION_NUMBER, ~rev1 );
        rev2 = gus_look8( card, GF1_GB_VERSION_NUMBER );
        gus_write8( card, GF1_GB_VERSION_NUMBER, rev1 );
        STI( &flags );
#ifdef GUSCFG_DEBUG_DETECT
        PRINTK( "gus: [0x%x] InterWave check - rev1=0x%x, rev2=0x%x\n", card -> gf1.port, rev1, rev2 );
#endif
        if ( ( rev1 & 0xf0 ) == ( rev2 & 0xf0 ) &&
             ( rev1 & 0x0f ) != ( rev2 & 0x0f ) )
          {
#ifdef GUSCFG_DEBUG_DETECT
            PRINTK( "gus: [0x%x] InterWave check - passed\n", card -> gf1.port );
#endif
            card -> pnp_flag = 1;
            return 0;			/* ok.. don't check RAM with old method */
          }
      }
    }
  
  gus_poke( card, 0L, 0xaa );
  gus_poke( card, 1L, 0x55 );
  if ( gus_peek( card, 0L ) != 0xaa || gus_peek( card, 1L ) != 0x55 )
    {
      PRINTK( "gus: card at 0x%x without onboard DRAM?\n", card -> gf1.port );
      return 1;
    }
  for ( idx = 1, d = 0xaa; idx < GUS_MEMORY_BANKS; idx++, d++ )
    {
      local = idx << GUS_MEM_BANK_SHIFT;
      gus_poke( card, local, d );
      gus_poke( card, local + 1, d + 1 );
      if ( gus_peek( card, local ) != d ||
           gus_peek( card, local + 1 ) != d + 1 ||
           gus_peek( card, 0L ) != 0xaa ) break;
    }
#if 1
  card -> gf1.memory = idx << GUS_MEM_BANK_SHIFT;
#else
  card -> gf1.memory = 256 * 1024;
#endif
#if 0
  printk( "memory = 0x%x (%i)\n", card -> gf1.memory, card -> gf1.memory );
#endif
  for ( l = 0, local = card -> gf1.memory; l < GUS_MEMORY_BANKS; l++, local -= 256 * 1024 )
    {
      card -> gf1.banks_8[ l ].address =
      card -> gf1.banks_8[ l ].size = 0;
      card -> gf1.banks_16[ l ].address = l << GUS_MEM_BANK_SHIFT;
      card -> gf1.banks_16[ l ].size = local > 0 ? 256 * 1024 : 0;
    }
  card -> gf1.banks_8[ 0 ].size = card -> gf1.memory;
  return 0;		/* we have some memory */
}

#ifdef GUSCFG_INTERWAVE

static void gus_pnp_reset( gus_card_t *card )
{
  gus_write8( card, 0x4c, 0x00 );
  gus_delay1( 1 );
  gus_write8( card, 0x4c, 0x01 );
  gus_delay1( 1 );
}

static void gus_pnp_bank_sizes( gus_card_t *card, int *sizes )
{
  int idx;
  unsigned int local;
  unsigned char d;

  for ( idx = 0; idx < GUS_PNP_MEMORY_BANKS; idx++ )
    {
      sizes[ idx ] = 0;
      d = 0x55;
      for ( local = idx << 22;
            local < ( idx << 22 ) + 0x400000;
            local += 0x40000, d++ )
        {
          gus_poke( card, local, d );
          gus_poke( card, local + 1, d + 1 );
#if 0
          printk( "d = 0x%x, local = 0x%x, local + 1 = 0x%x, idx << 22 = 0x%x\n",
          		d,
                        gus_peek( card, local ),
                        gus_peek( card, local + 1 ),
                        gus_peek( card, idx << 22 ) );
#endif
          if ( gus_peek( card, local ) != d ||
               gus_peek( card, local + 1 ) != d + 1 ||
               gus_peek( card, idx << 22 ) != 0x55 ) break;
          sizes[ idx ]++;
        }
    }
#if 0
  printk( "sizes: %i %i %i %i\n", sizes[ 0 ], sizes[ 1 ], sizes[ 2 ], sizes[ 3 ] );
#endif
}

struct rom_hdr {
  /* 000 */ unsigned char iwave[ 8 ];
  /* 008 */ unsigned char rom_hdr_revision;
  /* 009 */ unsigned char series_number;
  /* 010 */ unsigned char series_name[ 16 ];
  /* 026 */ unsigned char date[ 10 ];
  /* 036 */ unsigned short vendor_revision_major;
  /* 038 */ unsigned short vendor_revision_minor;
  /* 040 */ unsigned int rom_size;
  /* 044 */ unsigned char copyright[ 128 ];
  /* 172 */ unsigned char vendor_name[ 64 ];
  /* 236 */ unsigned char rom_description[ 128 ];
  /* 364 */ unsigned char pad[ 147 ];
  /* 511 */ unsigned char csum;
};

static void gus_pnp_mdetect( gus_card_t *card )
{
  static unsigned int lmc[ 13 ] = {
    0x00000001, 0x00000101, 0x01010101, 0x00000401,
    0x04040401, 0x00040101, 0x04040101, 0x00000004,
    0x00000404, 0x04040404, 0x00000010, 0x00001010,
    0x10101010
  };

  int i, bank_pos, pages;
  unsigned int lmct;
  int psizes[ GUS_PNP_MEMORY_BANKS ];
  unsigned char csum;
  struct rom_hdr romh;

  gus_pnp_reset( card );
  gus_write8( card, 0x19, gus_read8( card, 0x19 ) | 0x01 );	/* enhanced mode */
  gus_write8( card, 0x53, 0x01 );				/* DRAM I/O cycles selected */
  gus_write16( card, 0x52, ( gus_look16( card, 0x52 ) & 0xff10 ) | 0x004c );
  /* ok.. simple test of memory size */
  pages = 0;
  gus_poke( card, 0, 0x55 );
  gus_poke( card, 1, 0xaa );
#if 1
  if ( gus_peek( card, 0 ) == 0x55 && gus_peek( card, 1 ) == 0xaa )
#else
  if ( 0 )		/* ok.. for testing of 0k RAM */
#endif
    {
      gus_pnp_bank_sizes( card, psizes );
      if ( psizes[ 0 ] == 1 && psizes[ 1 ] == 1 )
        card -> pnp_pro_flag = 1;
      lmct = ( psizes[ 3 ] << 24 ) | ( psizes[ 2 ] << 16 ) |
             ( psizes[ 1 ] << 8 ) | psizes[ 0 ];
#if 0
      printk( "lmct = 0x%08x\n", lmct );
#endif
      for ( i = 0; i < sizeof( lmc ) / sizeof( unsigned int ); i++ )
        if ( lmct == lmc[ i ] )
          {
#if 0
            printk( "found !!! %i\n", i );
#endif
            gus_write16( card, 0x52, ( gus_look16( card, 0x52 ) & 0xfff0 ) | i );
            gus_pnp_bank_sizes( card, psizes );
            break;
          }
      if ( i >= sizeof( lmc ) / sizeof( unsigned int ) && !card -> gf1.enh_mode )
        gus_write16( card, 0x52, ( gus_look16( card, 0x52 ) & 0xfff0 ) | 2 );
      for ( i = 0; i < GUS_PNP_MEMORY_BANKS; i++ )
        {
          card -> gf1.banks_8[ i ].address =
          card -> gf1.banks_16[ i ].address = i << 22;
          card -> gf1.banks_8[ i ].size =
          card -> gf1.banks_16[ i ].size = psizes[ i ] << 18;
          pages += psizes[ i ];
        }
    }
  pages <<= 18;
  card -> gf1.memory = pages;
  
  gus_write8( card, 0x53, 0x03 );		/* select ROM */
  gus_write16( card, 0x52, ( gus_look16( card, 0x52 ) & 0xff1f ) | ( 4 << 5 ) );
  card -> gf1.rom_banks = 0;
  card -> gf1.rom_memory = 0;
  for ( bank_pos = 0; bank_pos < 16L * 1024L * 1024L; bank_pos += 4L * 1024L * 1024L )
    {
      for ( i = 0; i < sizeof( struct rom_hdr ); i++ )
        *(((unsigned char *)&romh) + i) = gus_peek( card, i + bank_pos );
#ifdef GUSCFG_DEBUG_ROM
      printk( "gus: ROM at 0x%06x = %02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x\n", bank_pos,
        		romh.iwave[ 0 ], romh.iwave[ 1 ], romh.iwave[ 2 ], romh.iwave[ 3 ],
        		romh.iwave[ 4 ], romh.iwave[ 5 ], romh.iwave[ 6 ], romh.iwave[ 7 ] );
#endif
      if ( strncmp( romh.iwave, "INTRWAVE", 8 ) ) continue;  /* first check */
      csum = 0;
      for ( i = 0; i < sizeof( struct rom_hdr ) - 1; i++ )
        csum += *(((unsigned char *)&romh) + i); 
#ifdef GUSCFG_DEBUG_ROM
      printk( "gus: ROM checksum = 0x%x == 0x%x (computed)\n", romh.csum, (unsigned char)(256 - csum) );
#endif
      if ( 256 - csum != romh.csum ) continue;		  /* not valid rom */
      card -> gf1.rom_banks++;
      card -> gf1.rom_present |= 1 << ( bank_pos >> 22 );
      card -> gf1.rom_memory = gus_get_dword( (char *)&romh.rom_size, 0 );
    }
  if ( card -> gf1.rom_memory > 0 )
    {
#if 0
      for ( lmct = 0, i = card -> gf1.rom_memory >> 18; i > 1; i >>= 1 ) lmct++;
      gus_write16( card, 0x52, ( gus_look16( card, 0x52 ) & 0xff1f ) | ( lmct << 5 ) );
#endif
      if ( card -> gf1.rom_banks == 1 && card -> gf1.rom_present == 8 )
        card -> version = GUS_CARD_VERSION_DYNASONIC;
    }
#if 0
  printk( "0x52 = 0x%x\n", gus_look16( card, 0x52 ) );
#endif
  gus_write8( card, 0x53, 0x00 );		/* select RAM */
    
  if ( !card -> gf1.enh_mode ) gus_pnp_reset( card );
}

#endif /* GUSCFG_INTERWAVE */

static int gus_init_dma_irq( gus_card_t *card, int latches )
{
  unsigned long flags;
  unsigned char i, irq, dma;
  static irqs[ 16 ] = { 0, 0, 1, 3, 0, 2, 0, 4, 0, 1, 0, 5, 6, 0, 0, 7 };
  static dmas[ 16 ] = { 6, 1, 0, 2, 0, 3, 4, 5 };

#ifdef GUSCFG_PNP
  if ( latches && card -> pnp_flag )
    {
      /* ok.. some InterWave specific initialization */
      CLI( &flags );
      gus_write8( card, 0x45, 0x00 );
      gus_write8( card, 0x59, card -> codec.interwave_serial_port = 0x1f );
      gus_write8( card, 0x5a, 0x49 );
      gus_write8( card, 0x5b, 0x11 );
      gus_write8( card, 0x5c, 0x00 );
      gus_write8( card, 0x5d, 0x30 );
      gus_write8( card, 0x60, 0x00 );
      STI( &flags );
    }
#endif
  
  dma = dmas[ card -> dmas[ GUS_DMA_GPLAY ] -> dma & 7 ];
  i = card -> dmas[ GUS_DMA_GRECORD ] -> dma & 7;
  if ( card -> equal_dma || dmas[ i ] == 0 )
    {
      if ( dmas[ i ] == 0 )
      PRINTK( "gus: Warning! DMA2 isn't defined.\n" );
      dma |= 0x40;
    }
   else
    dma |= dmas[ i ] << 3;
    
  if ( ( dma & 7 ) == 0 )
    {
      PRINTK( "gus: Error! DMA isn't defined.\n" );
      return -EINVAL;
    }

  irq = irqs[ card -> gf1.irq & 0x0f ];
  if ( !irq )
    {
      PRINTK( "gus: Error! IRQ isn't defined.\n" );
      return -EINVAL;
    }
  irq |= 0x40;
#if 0
  card -> mixer.mix_ctrl_reg |= 0x10;
#endif

  CLI( &flags );
  OUTB( 5, GUSP( card, REGCNTRLS ) );
  OUTB( card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
  OUTB( 0x00, GUSP( card, IRQDMACNTRLREG ) );
  OUTB( 0, GUSP( card, REGCNTRLS ) );
  STI( &flags );

  gus_delay( card );

  CLI( &flags );
  OUTB( 0x00 | card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
  OUTB( dma, GUSP( card, IRQDMACNTRLREG ) );
  if ( latches )
    {
      OUTB( 0x40 | card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
      OUTB( irq, GUSP( card, IRQDMACNTRLREG ) );
    }
  STI( &flags );

  gus_delay( card );
  
  CLI( &flags );
  OUTB( 0x00 | card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
  OUTB( dma, GUSP( card, IRQDMACNTRLREG ) );
  if ( latches )
    {
      OUTB( 0x40 | card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
      OUTB( irq, GUSP( card, IRQDMACNTRLREG ) );
    }
  STI( &flags );

  gus_delay( card );
  
  if ( latches )
    card -> mixer.mix_ctrl_reg |= 0x08;		/* enable latches */
   else
    card -> mixer.mix_ctrl_reg &= ~0x08;	/* disable latches */
  CLI( &flags );
  OUTB( card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
  OUTB( 0, GUSP( card, GF1PAGE ) );
  STI( &flags );
  
  return 0; 
}

static int gus_verify_region( int show_error, int port, int size )
{
  if ( check_region( port, size ) ) {
    if ( show_error )  {
      PRINTK( "gus: unable to grab I/O ports for region 0x%x-0x%x\n",
              			port,
              			( port + size ) - 1 );
    }
    return 1;
  }  
  return 0;
}

#ifdef GUSCFG_ESS
static int gus_probe_extreme( gus_card_t *card, int show_error, int port )
{
  card -> ess.port = port;
  if ( ess_detect( card ) ) {
    card -> ess.port = 0;
    return 1;
  } else {
    card -> ess.port = 0;
    if ( gus_verify_region( show_error, card -> gf1.port, 16 ) ) return 1;
    if ( gus_verify_region( show_error, card -> gf1.port + 0x100, 16 ) ) return 1;
    card -> detection = 1;		/* mark detection phase */
    if ( gus_mdetect( card ) ) {
      PRINTK( "gus: Something is wrong.. ES 1688 chip from GUS Extreme was detected, but GF1 chip at 0x%x wasn't found..\n", card -> gf1.port );
      card -> detection = 0;
      return 1;
    }
    card -> ess_flag = 1;
    if ( gus_version( card ) ) {
      card -> ess_flag = 0;
      return 1;
    }
    if ( card -> version != GUS_CARD_VERSION_EXTREME ) {
      PRINTK( "gus: GF1 chip doesn't report Extreme ID!!!\n" );
      PRINTK( "gus: Aborting...\n" );
      card -> ess_flag = 0;
      return 1;
    }
    card -> detection = 0;
    card -> ess.port = port;
    card -> gf1.port = port + 0x20;
  }
  return 0;
}
#endif

int gus_probe( gus_card_t *card )
{
  struct gus_region {
    int add;
    int size;
  };

  static short ports[] = { 0x220, 0x240, 0x260, 0x210, 0x230, 0x250, -1 };
#ifndef __alpha__
  static short irqs[] = { 11, 5, 7, 15, 12, 9, 3, -1 };
#else		/* alpha probably have IRQ 11 allocated for NCR SCSI controller */
  static short irqs[] = { 15, 11, 5, 7, 12, 9, 3, -1 };
#endif
#ifdef GUSCFG_ESS
  static short ess_irqs[] = { 5, 7, 10, 9, -1 };
  static short ess_dmas[] = { 0, 1, 3, -1 };
#endif
  static short dmas[] = { 6, 7, 5, 3, 1, -1 };
#ifdef GUSCFG_INTERWAVE
  static short dmas_interwave[] = { 1, 0, 3, 5, 6, 7, -1 };
#endif
  int old_gus_port, i, port, show_error;
  short *curr_dmas;
  
  old_gus_port = card -> gf1.port;
  for ( i = 0; ports[ i ] >= 0; i++ )
    {
      show_error = old_gus_port != 0x100;
      port = show_error ? card -> gf1.port : ports[ i ];
      /* 1. check if port regions for GF1 chip at 0x2XX & 0x3XX are free */
      if ( gus_verify_region( show_error, port, 16 ) ) continue; 
      if ( gus_verify_region( show_error, port + 0x100, 16 ) ) {
#ifdef GUSCFG_ESS
        if ( gus_probe_extreme( card, show_error, port ) ) {
          if ( show_error ) return -ENODEV;
          continue;
        }
        break;		/* we found a card */
#else
        continue;
#endif
      }
      /* 2. try detect card */
      card -> gf1.port = port;
      card -> detection = 1;		/* mark detection phase */
      if ( gus_mdetect( card ) ) {
        card -> detection = 0;
#ifdef GUSCFG_ESS
        if ( !gus_probe_extreme( card, show_error, port ) )
          break;			/* we found a card */
#endif
        if ( show_error ) return -ENODEV;
        card -> gf1.port = 0x100;	/* detect again */
        break;
      }
      card -> detection = 0;		/* end of detection phase, we found card */
      /* 3. we know about chip type (GF1 or GFA1) */
      /* 3. detect now with what card we talking */
      /* 3. now check for additional regions (GUS MAX) */
      if ( gus_version( card ) ) {
        if ( show_error ) return -ENODEV;
        card -> gf1.port = 0x100;	/* detect again */
        break;
      }
      if ( card -> max_ctrl_flag ) {
        if ( gus_verify_region( show_error, port + ( 0x726 - 0x220 ), 1 ) ) {
          if ( show_error ) return -ENODEV;
          PRINTK( "gus: GUS MAX detected at 0x%x, but port 0x%x is used by another device!!!\n", port, port + ( 0x726 - 0x220 ) );
          card -> gf1.port = 0x100;	/* detect again */
          break;
        }
      }
      break;
    }
  if ( card -> gf1.port == 0x100 ) return -ENODEV;
  curr_dmas = dmas;
#ifdef GUSCFG_INTERWAVE
  if ( card -> pnp_flag ) curr_dmas = dmas_interwave;    
#endif
#ifdef GUSCFG_GUSDB16
  if ( card -> daughter_flag && gus_verify_region( 1, card -> codec.port, 4 ) )
    {
      PRINTK( "gus: unable to grab I/O ports (0x%x-0x%x) for DB16 card\n", 
      	card -> codec.port, card -> codec.port + 3 );
      return -ENODEV;
    }
#endif
  for ( i = 0; card -> gf1.irq == 0x100 && irqs[ i ] >= 0; i++ )
    if ( !gus_request_irq( irqs[ i ], gus_interrupt, SA_GUS_FLAGS, "gus", card ) )
      {
        gus_free_irq( card -> gf1.irq = irqs[ i ], card );
        break;
      } 
  if ( card -> gf1.irq == 0x100 ) 
    {
      PRINTK( "gus: unable to grab IRQ (all possible IRQ channels are used)\n" );
      return -EIO;
    }
#ifdef GUSCFG_ESS
  for ( i = 0; card -> ess.irq == 0x100 && card -> ess.irq != card -> gf1.irq && ess_irqs[ i ] >= 0; i++ )
    if ( !gus_request_irq( ess_irqs[ i ], gus_interrupt, SA_GUS_FLAGS, "gus ES 1688", card ) )
      {
        gus_free_irq( card -> ess.irq = ess_irqs[ i ], card );
        break;
      } 
  if ( card -> ess.irq == 0x100 ) 
    {
      PRINTK( "gus: unable to grab IRQ (all possible IRQ channels are used)\n" );
      return -EIO;
    }
  if ( card -> equal_dma ) {
    PRINTK( "gus: You cannot use equal DMA channels for GUS Extreme card!!!" );
    return -EIO;
  }
#endif
  for ( i = 0; card -> dma1 == 0x100 && curr_dmas[ i ] >= 0; i++ )
    {
      if ( !request_dma( curr_dmas[ i ], "1" ) )
        {
          free_dma( card -> dma1 = curr_dmas[ i ] );
          break;
        }
    }
  if ( card -> dma1 == 0x100 )
    {
      PRINTK( "gus: unable to grab DMA (all possible DMA channels are used)\n" );
      return -EIO;
    }
  if ( !card -> equal_dma )
    {
#ifdef GUSCFG_ESS
      if ( card -> ess_flag ) {
        curr_dmas = ess_dmas;
        if ( card -> dma2 != 0x100 ) {
          for ( i = 0; curr_dmas[ i ] >= 0; i++ )
            if ( curr_dmas[ i ] == card -> dma2 ) break;
          if ( curr_dmas[ i ] < 0 ) {
            PRINTK( "gus: 16-bit dma2 is invalid for ES 1688 chip...\n" );
            return -EIO;
          }
        }
      }
#endif
      for ( i = 0; card -> dma2 == 0x100 && curr_dmas[ i ] >= 0; i++ )
        {
          if ( curr_dmas[ i ] == card -> dma1 ) continue;
          if ( !request_dma( curr_dmas[ i ], "2" ) )
            {
              free_dma( card -> dma2 = curr_dmas[ i ] );
              break;
            }
        }
      if ( card -> dma2 == 0x100 )
        card -> dma2 = card -> dma1;
    }
  card -> equal_dma = card -> dma1 == card -> dma2;
  return 0;				/* ok.. we have gus card.. */
}

int gus_version( gus_card_t *card )
{
  unsigned long flags;
  int val, rev;

  card -> version = GUS_CARD_VERSION_CLASSIC;
  card -> ics_flag = card -> ics_flipped = 0;
  if ( !card -> daughter_flag )
    card -> codec_flag = card -> max_flag = card -> max_ctrl_flag = 0;
   else
    card -> codec_flag = 1;

  if ( !card -> pnp_flag )
    {
      if ( card -> dma1 == 0 || (card -> dma2 == 0 && !card -> ess_flag) )
        {
          PRINTK( "gus: DMA 0 is allowed only for InterWave and ES 1688 chip\n" );
          return 1;
        }
      card -> gf1.enh_mode = 0;
      CLI( &flags );
      OUTB( 0x20, GUSP( card, REGCNTRLS ) );
      MB();
      val = INB( GUSP( card, REGCNTRLS ) );
      rev = INB( GUSP( card, BOARDVERSION ) );
      STI( &flags );
#ifdef GUSCFG_DEBUG_DETECT
      PRINTK( "gus: [0x%x] init - val=0x%x, rev=0x%x\n", card -> gf1.port, val, rev );
#endif

      /*
       *  I don't know why, but sometimes isn't by OSS way GUS MAX detected properly.
       *  Revision value now have priority.
       */
  
      if ( ( val != 0xff && (val & 0x06) ) || ( rev >= 5 && rev != 0xff ) )
        {
          if ( rev == 255 || rev < 5 )
            card -> version = GUS_CARD_VERSION_CLASSIC1;
           else
          if ( rev < 10 )
            {
#if 1 /* simulate old GUS Classic card without mixer */
#if 1 /* simulate GUS ACE card */
              card -> version = GUS_CARD_VERSION_CLASSIC_ICS;
              card -> ics_flag++;
              card -> ics_flipped = val == 5;	/* if 5 - some channels are flipped */
#else
              card -> version = GUS_CARD_VERSION_ACE;
#endif
#endif
            }
           else
          if ( rev == 10 || rev == 11 )
            {
              card -> version = rev == 10 ? GUS_CARD_VERSION_MAX : GUS_CARD_VERSION_MAX1;
              card -> max_flag =
              card -> max_ctrl_flag =
              card -> codec_flag = 1;		/* heya - GUS MAX */
            }
           else
          if ( rev == 0x30 )
            card -> version = GUS_CARD_VERSION_ACE;
           else
          if ( rev == 0x50 )
            {
              card -> version = GUS_CARD_VERSION_EXTREME;
#ifndef GUSCFG_ESS
	      PRINTK( "gus: Please, compile driver with full support for Extreme card...\n" );
#else
              if ( !card -> ess_flag ) {
                PRINTK( "gus: ESS ES1688 chip wasn't detected on Extreme card!!!\n" );
                return 1;	/* abort */
              }
#endif
            }
           else
            {
              card -> version = GUS_CARD_VERSION_CLASSIC_ICS;
              card -> ics_flag++;
              PRINTK( "gus: unknown GF1 revision number - 0x%x (0x%x)\n", rev, val );
              PRINTK( "gus: please - report to <perex@jcu.cz>\n" );
            }
        }
    }
   else
    {
      card -> version = GUS_CARD_VERSION_PNP;
#ifdef GUSCFG_INTERWAVE
      gus_pnp_mdetect( card );
#endif
      card -> max_flag = card -> codec_flag = 1;
#ifdef GUSCFG_INTERWAVE
      if ( !card -> gf1.enh_mode )
        gus_mdetect( card );
#else
      card -> gf1.enh_mode = 0;
      gus_mdetect( card );
#endif
    }  
  return 0;
}

int gus_init_pre( gus_card_t *card )
{
  int val;

  if ( card -> dma1 < 4 && card -> dma1_size > 64 ) card -> dma1_size = 64;
  if ( card -> dma2 < 4 && card -> dma2_size > 64 ) card -> dma2_size = 64;

#ifndef GUSCFG_GUSSTD
  if ( card -> version == GUS_CARD_VERSION_CLASSIC ||
       card -> version == GUS_CARD_VERSION_CLASSIC1 ||
       card -> version == GUS_CARD_VERSION_CLASSIC_ICS ||
       card -> version == GUS_CARD_VERSION_ACE )
    {
      PRINTK( "gus: code for GUS Classic / ACE needed\n" );
      return 1;
    }
#endif
#ifndef GUSCFG_GUSDB16
  if ( card -> daughter_flag )
    {
      PRINTK( "gus: code for GUS 16-bit Daughter Board needed\n" );
      return 1;
    }
#endif
#ifndef GUSCFG_GUSMAX
  if ( card -> version == GUS_CARD_VERSION_MAX ||
       card -> version == GUS_CARD_VERSION_MAX1 )
    {
      PRINTK( "gus: code for GUS MAX needed\n" );
      return 1;
    }
#endif
#ifndef GUSCFG_GUSPNP
  if ( card -> version == GUS_CARD_VERSION_PNP )
    {
      PRINTK( "gus: code for GUS Plug & Play needed\n" );
      return 1;
    }
#endif

  if ( card -> version < 0xa0 )		/* cards without CODEC */
    {
      if ( card -> dma1_size > 64 ) card -> dma1_size = 64;
      if ( card -> dma2_size > 64 ) card -> dma2_size = 64;
    }
    
  gus_set_dmas( card );

  card -> gf1.active_voices = -1;
  card -> mixer.mix_ctrl_reg = 0;
    
#ifdef GUSCFG_CODEC
  if ( card -> max_flag )	/* GUS MAX */
    card -> codec.port = card -> gf1.port + ( 0x32c - 0x220 );
  if ( card -> codec_flag && card -> codec.port > 0x220 )
    {
      if ( card -> max_flag )
        {
          card -> codec.irq = card -> gf1.irq;	/* shared */
          card -> max_cntrl_val = ( card -> gf1.port >> 4 ) & 0x0f;
          if ( card -> dmas[ GUS_DMA_CRECORD ] -> dma > 3 ) card -> max_cntrl_val |= 0x10;
          if ( card -> dmas[ GUS_DMA_CPLAY ] -> dma > 3 ) card -> max_cntrl_val |= 0x20;
          card -> max_cntrl_val |= 0x40;
          OUTB( card -> max_cntrl_val, GUSP( card, MAXCNTRLPORT ) );
          MB();
        }

      card -> codec.version = val = codec_init( card );
      if ( !val )
        {
          PRINTK( "gus: %s not found at port 0x%x\n", 
          	card -> daughter_flag ? "daughter board" : "codec", card -> codec.port );
          if ( card -> daughter_flag ) return 1;	/* init error */
          card -> max_flag = card -> codec_flag = 0;	/* clear flags */
        }
       else
      if ( !card -> pnp_flag )
        if ( card -> max_flag )
          {
            card -> equal_irq = 1;
            card -> codec.irq = card -> gf1.irq;
          }
         else
          {
#ifdef GUSCFG_GUSDB16
            card -> ics_flag = 0;
#endif
          }
    }
#endif /* GUSCFG_CODEC */
  
  if ( !card -> max_flag ) card -> use_codec = 0;
  if ( card -> pnp_flag && card -> gf1.enh_mode )
    card -> use_codec = 1;

#ifdef GUSCFG_GF1PCM
  val = 0;
  if ( card -> use_codec || card -> ess_flag ) val = 1;
  gus_init_gf1_pcm( &card -> pcm[ val ] );
#endif
#ifdef GUSCFG_CODEC
  if ( card -> use_codec )
    gus_init_codec_pcm( &card -> pcm[ 0 ] );
#endif
#ifdef GUSCFG_ESS
  if ( card -> ess_flag )
    gus_init_ess_pcm( &card -> pcm[ 0 ] );
#endif

  if ( gus_init_dma_irq( card, 1 ) < 0 ) return -EIO;
#ifdef GUSCFG_ESS
  if ( card -> ess_flag )
    if ( ess_init( card, 1 ) < 0 ) return -EIO;
#endif
  return 0;
}

int gus_init( gus_card_t *card )
{
  mixer_init( card );
  if ( gus_gf1_start( card ) < 0 )
    {
      gus_done( card );
      return -EIO;
    }
  return 0;
}

void gus_done( gus_card_t *card )
{
  gus_gf1_stop( card );
#ifndef GUSCFG_LEAVE_MIXER
  gus_init_dma_irq( card, 0 );
#endif
#ifdef GUSCFG_ESS
  if ( card -> ess_flag )
    ess_init( card, 0 );
#endif
}

#ifdef GUSCFG_INTERWAVE 

void gus_init_set_enhanced_mode( gus_card_t *card, int enable )
{
  unsigned long flags;

  if ( !card -> pnp_flag ) return;
  enable = enable ? 1 : 0;
  CLI( &flags );
  if ( card -> gf1.enh_mode == enable )		/* already there */
    {
      STI( &flags );
      return;
    }
  if ( card -> gf1.mode & (GF1_MODE_ENGINE|GF1_MODE_PCM) )
    {
      STI( &flags );
      PRINTK( "gus: can't change enhance mode settings - device is busy\n" );
      return;
    }
  card -> gf1.mode |= GF1_MODE_ENGINE | GF1_MODE_PCM;
  STI( &flags );
  card -> gf1.enh_mode = enable;
  gus_gf1_stop( card );
  if ( card -> gf1.enh_mode )
    gus_pnp_mdetect( card );
   else
    gus_mdetect( card );
  gus_gf1_start( card );
  card -> gf1.mode &= ~(GF1_MODE_ENGINE | GF1_MODE_PCM);
}

#endif

void gus_init_set_pcm_memory( gus_card_t *card, int enable )
{
  unsigned long flags;

  enable = enable ? 1 : 0;
  CLI( &flags );
  if ( card -> gf1.pcm_memory == enable )	/* already there */
    {
      STI( &flags );
      return;
    }
  if ( card -> gf1.mode & (GF1_MODE_ENGINE|GF1_MODE_PCM) )
    {
      STI( &flags );
      PRINTK( "gus: can't change pcm memory settings - device is busy\n" );
      return;
    }
  card -> gf1.mode |= GF1_MODE_ENGINE | GF1_MODE_PCM;
  STI( &flags );
  gus_memory_done( card, &card -> gf1.mem_alloc );
  gus_memory_init( card, &card -> gf1.mem_alloc );
  card -> gf1.mode &= ~(GF1_MODE_ENGINE | GF1_MODE_PCM);
}

#ifdef GUSCFG_INTERWAVE

void gus_init_set_pcm_fifo( gus_card_t *card, int record, int value )
{
  unsigned long flags;
  unsigned short reg;
  int l;

  if ( value > 0 )
    {
      if ( value < 256 ) value = 256;
      if ( value > 256 * 1024 ) value = 256 * 1024;
      reg = 15;
      l = 256 * 1024;
      while ( l > 8 )
        {
          l >>= 1;
          if ( value > l ) break;
          reg--;
        }
      value = 8;
      l = reg;
      while ( l > 0 )
        {
          value <<= 1;
          l--;
        }
    }
   else
    {
      reg = 0;
      value = 0;                 /* disable this feature */
    }

  CLI( &flags );
  if ( (!record && card -> codec.playback_fifo_size == value) ||
       (record && card -> codec.record_fifo_size == value) )
    {
      STI( &flags );
      return;
    }
  if ( (card -> gf1.mode & (GF1_MODE_ENGINE|GF1_MODE_PCM)) ||
       card -> codec.mode != CODEC_MODE_NONE )
    {
      STI( &flags );
      PRINTK( "gus: can't change fifo settings - device(s) busy\n" );
      return;
    }
  card -> gf1.mode |= GF1_MODE_ENGINE | GF1_MODE_PCM;
  card -> codec.mode |= CODEC_MODE_OPEN;
  STI( &flags );
  if ( !record )
    {
      card -> codec.interwave_fifo_reg &= ~0x001f;
      if ( value > 0 )
        card -> codec.interwave_fifo_reg |= 0x0010 | reg;
      card -> codec.playback_fifo_size = value;
    }
   else
    {
      card -> codec.interwave_fifo_reg &= ~0x1f00;
      if ( value > 0 )
        card -> codec.interwave_fifo_reg |= ( 0x10 | reg ) << 8;
      card -> codec.record_fifo_size = value;
    }
  gus_gf1_stop( card );
  gus_gf1_start( card );
  card -> gf1.mode &= ~(GF1_MODE_ENGINE | GF1_MODE_PCM);
  card -> codec.mode &= ~CODEC_MODE_OPEN;
}

#endif

int gus_info( gus_card_t *card, gus_mem_t *alloc, struct GUS_STRU_INFO *a_info )
{
  int i;
  struct GUS_STRU_INFO info;

  if ( VERIFY_AREA( VERIFY_WRITE, a_info, sizeof( info ) ) ) return -EIO;

  strncpy( info.id, card -> id, sizeof( info.id ) );
  info.version = card -> version;
  info.flags = ( card -> daughter_flag ? GUS_STRU_INFO_F_DB16 : 0 ) |
               ( card -> gf1.pcm_memory > 0 ? GUS_STRU_INFO_F_PCM : 0 ) |
               ( card -> gf1.enh_mode ? GUS_STRU_INFO_F_ENHANCED : 0 ) |
               ( card -> gf1.mode & GF1_MODE_DAEMON ? GUS_STRU_INFO_F_DAEMON : 0 );
  
  info.port = card -> gf1.port;
  info.irq = card -> gf1.irq;
  info.dma1 = card -> dmas[ GUS_DMA_GPLAY ] -> dma;
  info.dma2 = card -> dmas[ GUS_DMA_GRECORD ] -> dma;

  info.mixing_freq = card -> gf1.playback_freq;

  info.memory_size = card -> gf1.memory;
  if ( alloc )
    {
      info.memory_free = gus_memory_free_size( card, alloc );
      info.memory_block_8 = gus_memory_free_block( card, alloc, 0 );
      info.memory_block_16 = gus_memory_free_block( card, alloc, 1 );
    }
   else
    {
      info.memory_free = 0;
      info.memory_block_8 = 0;
      info.memory_block_16 = 0;
    }
  if ( card -> gf1.enh_mode )
    {
      for ( i = 0; i < 4; i++ )
        {
          info.memory_banks[ i ].address = card -> gf1.banks_8[ i ].address;
          info.memory_banks[ i ].size    = card -> gf1.banks_8[ i ].size;
        }
    }
   else
    {
      info.memory_banks[ 0 ].address = 0;
      info.memory_banks[ 0 ].size = card -> gf1.memory;
      for ( i = 1; i < 4; i++ )
        {
          info.memory_banks[ i ].address = 0;
          info.memory_banks[ i ].size = 0;
        }
    }
  info.memory_rom_size = card -> gf1.rom_memory;
  info.memory_rom_banks = card -> gf1.rom_banks;
  info.memory_rom_present = card -> gf1.rom_present;

  MEMCPY_TOFS( a_info, &info, sizeof( info ) );
  return 0;
}

static void get_info_init_dma( gus_info_buffer_t *buffer, struct GUS_STRU_DMA *dma )
{
  if ( !dma -> used && !dma -> static_alloc )
    {
      gus_iprintf( buffer, "  unallocated (requested %i bytes)\n", (int)dma -> rsize );
    }
   else
    gus_iprintf( buffer, "  used          : %i%s\n"
    			 "  buffer ptr    : 0x%x\n"
  		         "  buffer size   : %i (real %li, requested %li/%li)\n"
  		         "  lock status   : %-6s %-6s %-6s\n"
  		         "  owner         : %-20s\n"
  		         "  old owner     : %-20s\n",
  		         dma -> used,
  		         dma -> mmaped ? " [mmaped]" : "",
		         (long)dma -> buf,
		         dma -> usize, dma -> size, dma -> rsize, dma -> ursize,
		         dma -> lock & WK_LOCK   ? "lock  " : "",
	                 dma -> lock & WK_WAKEUP ? "wakeup" : "",
	                 dma -> lock & WK_SLEEP  ? "sleep " : "",
		         dma -> owner,
		         dma -> old_owner );
}

static void get_info_init_card( gus_card_t *card, gus_info_buffer_t *buffer )
{
  unsigned int i;
  char manufacture[ 32 ];
  char model[ 32 ];
  char *str;

  strncpy( model, card -> id, sizeof( card -> id ) );
  model[ sizeof( card -> id ) ] = 0;
  gus_iprintf( buffer, "\n\nCARD #%i-%i (%s):\n", card -> dnumber + 1, card -> number + 1, model );
  for ( i = strlen( model ) + 13; i > 0; i-- )
    gus_iprintf( buffer, "=" );
  gus_iprintf( buffer, "\n" );
  
  strcpy( manufacture, "Gravis UltraSound " );
  strcpy( model, "???" );
  if ( card -> version < 0x50 )
    sprintf( model, "%i.%i", card -> version >> 4, card -> version & 0x0f );
   else
    switch ( card -> version ) {
      case GUS_CARD_VERSION_EXTREME:
        strcpy( model, "Extreme" );
        break;
      case GUS_CARD_VERSION_ACE:
        strcpy( model, "ACE" );
        break;
      case GUS_CARD_VERSION_MAX: 
      case GUS_CARD_VERSION_MAX1:
        strcpy( model, "MAX" );
        break;
      case GUS_CARD_VERSION_PNP:
        strcpy( model, "Plug & Play" );
        if ( card -> pnp_pro_flag )
          strcat( model, " Pro" );
        break;
      case GUS_CARD_VERSION_DYNASONIC:
        strcpy( manufacture, "" );
        strcpy( model, "DynaSonic 3-D" );
        break;
    }

  gus_iprintf( buffer, ">> %s%s (%dk) at 0x%x, irq %d, dma %d",
				manufacture,
  				model,
  				(int)card -> gf1.memory >> 10,
  				(int)card -> gf1.port,
  				(int)card -> gf1.irq,
  				(int)card -> dmas[ GUS_DMA_GPLAY ] -> dma );
  if ( !card -> equal_dma && !card -> ess_flag )
    gus_iprintf( buffer, "&%d", (int)card -> dmas[ GUS_DMA_GRECORD ] -> dma );
  gus_iprintf( buffer, "\n" );

#ifdef GUSCFG_CODEC
  if ( card -> codec_flag )
    {
      strcpy( model, "CS4231 rev A" );
      switch ( card -> codec.version ) {
        case 0x00: break;
        case 0x01: model[ 11 ] = 'B'; break;
        case 0x0a: model[ 11 ] = 'C'; break;
        default:   model[ 11 ] = '?';
      }
      gus_iprintf( buffer, ">> CODEC: %s%s at 0x%x, irq %d, dma %d", 
      			model,
      			card -> pnp_flag ? " (enhanced)" : "",
      			(int)card -> codec.port,
      			(int)card -> codec.irq,
      			(int)card -> dmas[ GUS_DMA_CPLAY ] -> dma );
      if ( !card -> equal_dma )
        gus_iprintf( buffer, "&%d", (int)card -> dmas[ GUS_DMA_CRECORD ] -> dma );
      gus_iprintf( buffer, "\n" );
    }
#endif
    
#ifdef GUSCFG_INTERWAVE
  if ( card -> pnp_flag )
    {
      int j;
    
      gus_iprintf( buffer, ">> InterWave: rev %c\n", 
      				(char)( ( gus_i_look8( card, 0x5b ) >> 4 ) + 'A' ) );
      for ( i = j = 0; i < GUS_PNP_MEMORY_BANKS; i++ )
        j += card -> gf1.banks_16[ i ].size >> 10;

      gus_iprintf( buffer, ">> InterWave: RAM %ik (", (int)j );
      for ( i = 0; i < GUS_PNP_MEMORY_BANKS; i++ )
        gus_iprintf( buffer, "%ik%s", (int)(card -> gf1.banks_16[ i ].size >> 10), i + 1 < GUS_PNP_MEMORY_BANKS ? "," : "" );
      gus_iprintf( buffer, ")\n" );

      gus_iprintf( buffer, ">> InterWave: ROM %ik (",
      				(int)( card -> gf1.rom_memory * card -> gf1.rom_banks >> 10 ) );
      for ( i = 0; i < GUS_PNP_MEMORY_BANKS; i++ )
        gus_iprintf( buffer, "%ik%s", (card -> gf1.rom_present & (1 << i)) ? card -> gf1.rom_memory >> 10 : 0, i + 1 < GUS_PNP_MEMORY_BANKS ? "," : ")" );
      gus_iprintf( buffer, "\n" );

      if ( !card -> gf1.enh_mode )
        gus_iprintf( buffer, ">> InterWave: Enhanced Mode is OFF\n" );
    }  
#endif

#ifdef GUSCFG_ESS
  if ( card -> ess_flag )
    gus_iprintf( buffer, ">> ES 1688: rev %d at 0x%x, irq %d, dma %d\n", 
      			card -> ess.version & 0x0f,
      			card -> ess.port,
      			card -> ess.irq,
      			card -> dmas[ GUS_DMA_ESS ] -> dma );
#endif
   
  gus_iprintf( buffer, "\nDMA%s:\n",
  	!card -> equal_dma ? "1" : "" );
  get_info_init_dma( buffer, card -> dmas[ GUS_DMA_GPLAY ] );
  if ( !card -> equal_dma )
    { 
      gus_iprintf( buffer, "DMA2:\n" );
      get_info_init_dma( buffer, card -> dmas[ GUS_DMA_GRECORD ] );
    }
  if ( card -> daughter_flag )
    {
      gus_iprintf( buffer, "DMA DB16:\n" );
      get_info_init_dma( buffer, card -> dmas[ GUS_DMA_CPLAY ] );
    }
  gus_iprintf( buffer, "Interrupts:\n" );
  gus_iprintf( buffer, "  TOTAL         : %i\n", card -> interrupt_stat );
#ifdef GUSCFG_INTERRUPTS_PROFILE
#ifdef GUSCFG_MIDI_DEVICES
  gus_iprintf( buffer, "  GF1 MIDI out  : %i\n", card -> gf1.interrupt_stat_midi_out );
  gus_iprintf( buffer, "  GF1 MIDI in   : %i\n", card -> gf1.interrupt_stat_midi_in );
#endif
  gus_iprintf( buffer, "  GF1 timer1    : %i\n", card -> gf1.interrupt_stat_timer1 );
  gus_iprintf( buffer, "  GF1 timer2    : %i\n", card -> gf1.interrupt_stat_timer2 );
  gus_iprintf( buffer, "  GF1 wave    S : %i\n", card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].interrupt_stat_wave );
  gus_iprintf( buffer, "  GF1 volume  S : %i\n", card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].interrupt_stat_volume );
  gus_iprintf( buffer, "  GF1 wave    P : %i\n", card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].interrupt_stat_wave );
  gus_iprintf( buffer, "  GF1 volume  P : %i\n", card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].interrupt_stat_volume );
  gus_iprintf( buffer, "  GF1 wave    E : %i\n", card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].interrupt_stat_wave );
  gus_iprintf( buffer, "  GF1 volume  E : %i\n", card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].interrupt_stat_volume );
  gus_iprintf( buffer, "  GF1 voice lost: %i\n", card -> gf1.interrupt_stat_voice_lost );
  gus_iprintf( buffer, "  GF1 DMA write : %i\n", card -> gf1.interrupt_stat_dma_write );
  gus_iprintf( buffer, "  GF1 DMA read  : %i\n", card -> gf1.interrupt_stat_dma_read );
#ifdef GUSCFG_CODEC
  if ( card -> codec_flag )
    {
      gus_iprintf( buffer, "  CODEC playback: %i\n", card -> codec.interrupt_stat_playback );
      gus_iprintf( buffer, "  CODEC record  : %i\n", card -> codec.interrupt_stat_record );
      gus_iprintf( buffer, "  CODEC timer   : %i\n", card -> codec.interrupt_stat_timer );
    }
#endif
#ifdef GUSCFG_ESS
  if ( card -> ess_flag )
    {
      gus_iprintf( buffer, "  ESS playback  : %i\n", card -> ess.interrupt_stat_playback );
      gus_iprintf( buffer, "  ESS record    : %i\n", card -> ess.interrupt_stat_record );
    }
#endif
#endif
  if ( card -> gf1.uart_enable )
    {
      gus_iprintf( buffer, "MIDI UART:\n" );
      gus_iprintf( buffer, "  Overrun errors: %i\n", card -> gf1.uart_overrun );
      gus_iprintf( buffer, "  Framing errors: %i\n", card -> gf1.uart_framing );
    }
  gus_iprintf( buffer, "%s:\n"
  		       "  flags         :%s%s%s\n",
  		       	   card -> gf1.enh_mode ? "Enhanced GF1" : "GF1",
                           card -> gf1.mode == 0 ? " none" :
                           card -> gf1.mode & GF1_MODE_DAEMON ? " daemon" : "",
                           card -> gf1.mode & GF1_MODE_MIDI ? " midi" : "",
                           card -> gf1.mode & GF1_MODE_SYNTH ? " synth" : "" );
  if ( card -> gf1.mode & (GF1_MODE_ENGINE | GF1_MODE_PCM) )
    {
      gus_iprintf( buffer, "  owner         :%s%s%s\n"
                           "  active voices : %i\n"
#ifdef GUSCFG_MIDI_DEVICES
                           "  midi voices   : %i\n"
#endif
                           "  mixing freq.  : %iHz\n",
                           card -> gf1.mode & GF1_MODE_ENGINE ? " engine" : "",
                           card -> gf1.mode & GF1_MODE_PCM_PLAY ? " play" : "",
                           card -> gf1.mode & GF1_MODE_PCM_RECORD ? " record" : "",
                           card -> gf1.active_voices,
#ifdef GUSCFG_MIDI_DEVICES
                           card -> gf1.midi_mix_voices,
#endif
                           card -> gf1.playback_freq
                            );
    }
  if ( card -> gf1.mode & GF1_MODE_ENGINE )
    {
      str = "unknown";
    
      switch ( card -> gf1.midi_emul ) {
        case GUS_MIDI_EMUL_GM: 	str = "General Midi"; break;
        case GUS_MIDI_EMUL_GS: 	str = "General Midi + GS"; break;
        case GUS_MIDI_EMUL_MT32: str = "MT-32"; break;
      }
      gus_iprintf( buffer, "  midi emulation: %s\n", str );
    }
#ifdef GUSCFG_GF1PCM
  if ( !card -> pnp_flag )
    gus_iprintf( buffer,   "  record overfw : %i\n",
      			   card -> gf1.pcm_record_overflow );
#endif
  gus_iprintf( buffer, "  UltraClick elimination defaults:\n" );
  if ( !card -> gf1.enh_mode )
    gus_iprintf( buffer, "    smooth pan    : %s\n"
  		         "    full range pan: %s\n",
  		         	card -> gf1.default_smooth_pan ? "enable" : "disable",
  		         	card -> gf1.default_full_range_pan ? "enable" : "disable" );
  gus_iprintf( buffer,   "    volume ramp   : %i\n", card -> gf1.default_volume_ramp );
  if ( card -> gf1.mode & GF1_MODE_ENGINE )
    {
      gus_iprintf( buffer, "  UltraClick elimination:\n" );
      if ( !card -> gf1.enh_mode )
        gus_iprintf( buffer, "    smooth pan    : %s\n"
  		             "    full range pan: %s\n",
  		         	card -> gf1.smooth_pan ? "enable" : "disable",
  		         	card -> gf1.full_range_pan ? "enable" : "disable" );
      gus_iprintf( buffer,   "    volume ramp   : %i\n", card -> gf1.volume_ramp );
      gus_iprintf( buffer, "  Instruments   : %i\n", card -> gf1.mem_alloc.instruments_count );
    }
}

int get_info_init( gus_info_buffer_t *buffer )
{
#ifdef MODULE_PARM
  static char *kernel_version = UTS_RELEASE;
#else
  extern char kernel_version[];
#endif
  int i;
  gus_card_t *card;
  char *str;

  *buffer -> buffer = 0;
  gus_iprintf( buffer, "Driver " VERSION_AND_COPYRIGHT "\n"
                       "Compiled in " __DATE__ " for kernel %s"
#ifdef __SMP__
		       " (SMP)"
#endif
#ifdef MODVERSIONS
                       " with versioned symbols"
#endif
                       ".\n", kernel_version );
  str =                "Compiled with support for GUS "
#ifdef GUSCFG_GUSSTD
                       "Classic, "
#endif
#ifdef GUSCFG_GUSDB16
		       "DB16, "
#endif
#ifdef GUSCFG_GUSEXTREME
		       "Extreme, "
#endif
#ifdef GUSCFG_GUSMAX
		       "MAX, "
#endif
#ifdef GUSCFG_GUSPNP
		       "PnP, "
#endif
		       ;
  if ( str[ strlen( str ) - 1 ] == ' ' )
    str[ strlen( str ) - 2 ] = 0;
  gus_iprintf( buffer, "%s soundcards.\n", str );

  gus_memory_info( buffer );

  for ( i = 0; i < gus_cards_count; i++ )
    {
      card = gus_cards[ i ];    
      get_info_init_card( card, buffer );
#ifdef GUSCFG_DEBUG_MEMORY
      gus_memory_manager_debug( card, buffer, &card -> gf1.mem_alloc );
#endif
#ifdef GUSCFG_DEBUG_INSTRUMENTS
      gus_memory_debug_instruments( card, buffer, &card -> gf1.mem_alloc );
#endif
      gus_timer_info( card, buffer );
      gus_pcm_info( card, buffer );
#ifdef GUSCFG_CODEC
      gus_codec_info( card, buffer );
#endif
    }
#ifdef GUSCFG_OSS
  gus_sequencer_info( buffer );
#endif
#ifdef GUSCFG_DEBUG_MEMORY
  gus_memory_debug( buffer );
#endif
  return (int)(buffer -> curr - buffer -> buffer);
}
