/*
 Selects specific columns from an rdbtable.

 Author: Carlo Strozzi <carlos@linux.it>
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define MAX_ESCAPE_LENGTH       256
#define MAX_ARG_LENGTH          1024
#define EMPTY                   ""

void show_help( char *my_name)
{
    printf("
        NoSQL operator: %s

Usage:  %s [options] 'column [column] ...'

Options:
    -help    Print this help info.
    -n       Strip header from output.
    -x       Debug option.
    -e \"abc\" Escape characters in set [a,b,c] by prepending a
             backslash to each of them on STDOUT.

Takes a list of column names of the input rdbtable and prints them to
STDOUT in the specified order.  Chars that are special to the UNIX shell
must be quoted. 

Column names must be referred to with the construct '$P[\"column_name\"]'.
If multiple columns are specified they must be separated by a comma.

For example, to select columns 'NAME' and 'JOB' from the input rdbtable
the statement is:

                     '$P[\"NAME\"],$P[\"JOB\"]' 

For readability, the comma may be surrounded by any number of spaces on
either side :

                    '$P[\"NAME\"] , $P[\"JOB\"]'

Note how the list of columns must be quoted, i.e. it must be one
single token. Long lists of column names may be folded over multiple
lines by ending each line with a backslash, i.e.:

            '$P[\"COL1\"] , $P[\"COL2\"] , $P[\"COL3\"] %s
             $P[\"COL4\"] , $P[\"COL5\"]'

Ordinary AWK field numbers may be used in expressions, i.e.:

                      '$(11) , $(3), $(21)'

This operator reads an rdbtable via STDIN and writes an rdbtable via
STDOUT. If an invalid column is specified, then the whole table row is
iprinted to STDOUT, at the point where the invalid column occurred.


'$Id: nsq-fcol.c,v 1.1 1998/05/29 20:43:01 carlos Exp $'

            ----------------------
NoSQL RDBMS, Copyright (C) 1998 Carlo Strozzi.
This program comes with ABSOLUTELY NO WARRANTY; for details
refer to the GNU General Public License.

You should have received a copy of the GNU General Public License
along with this program;  if not, write to the Free Software
Foundation, Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307
USA.
            ----------------------\n", my_name, my_name, "\\");
}

int main( int  argc, char *argv[] ) {

  /* For getopt() */
  extern char* optarg;
  extern int optind;

  /* for the rest of the program. */
  register int
    a_loop;
  char
    *my_name=argv[0],
    cmd_buf[MAX_COMMAND_LENGTH],
    e_set[MAX_ESCAPE_LENGTH] = EMPTY,
    c_list[MAX_ARG_LENGTH] = EMPTY;

  int no_hdr=0, debug=0;

  while ((a_loop = getopt(argc, argv, "xnhe:")) != EOF) {
    switch (a_loop) {
      case 'h':
        show_help(my_name);
        exit(0);
        break;
      case 'n': 
        no_hdr=1;
        break;
      case 'x': 
        debug=1;
        break;
      case 'e':
        snprintf(e_set,MAX_ESCAPE_LENGTH,"%s", optarg);
        break;
      default:
        show_help(my_name);
        exit(1);
    }
  }

  if( optind < argc )
    snprintf(c_list,MAX_ARG_LENGTH,"%s", argv[optind++]);
  else {
    show_help(my_name);
    exit(1);
  }

  snprintf(cmd_buf,MAX_COMMAND_LENGTH,"#
#
BEGIN {
  NULL = \"\"
  FS = \"\\t\"; OFS = FS;
  split( \"%s\", e_set, NULL)

  # Honour the 'debug' switch.
  if( %d ) {
    arg_vec = \"# ARGC = \" ARGC

    for( arg in ARGV )
      arg_vec = arg_vec \"\\n# ARGV[\" arg \"] = \" ARGV[arg]

    print arg_vec > \"/dev/stderr\"
  }
}
# Table comments.
r == 0 && $0 ~ /^ *#/ {
  if( ! %d ) print
  next
}
# Column names and positions.
r == 0 {
  while( ++p <= NF ) {
    # Make sure we pick the first occurrence of duplicated column
    # names (it may happen after a join).
    if( P[$p] == NULL ) {
      P[$p]=p
      N[p]=$p
    }
  }
  if( ! %d ) print %s
  r++
  next
}
# Column definitions.
r == 1 {
  if( ! %d ) print %s
  r++
  NR = 0
  next
}
# Table body.
{
  for( e in e_set ) gsub( e_set[e], \"\\\\\\\\\"e_set[e], $0 )
  print %s
}", e_set, debug, no_hdr, no_hdr, c_list, no_hdr, c_list, c_list);

  if( debug )
    fprintf (stderr, "Generated AWK program:
      ----------\n%s\n----------\n",cmd_buf);

  execlp(AWK,"awk",cmd_buf,NULL);
  exit(0);
}
