/* -*- Mode: C;-*-
 *
 * This file is part of XDelta - A binary delta generator.
 *
 * Copyright (C) 1997, 1998  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Josh MacDonald <jmacd@CS.Berkeley.EDU>
 *
 * xdelta.c,v 1.14 1998/10/03 02:32:09 jmacd Exp
 */

#include <string.h>
#include <stdlib.h>
#include <sys/param.h>  /* for ntohs, htons on some */
#include <netinet/in.h> /* on others */

#include "xdelta.h"
#include "xdeltapriv.h"

/*#define DEBUG_RSYNC_REQUEST*/
/*#define DEBUG_CONT*/
/*#define DEBUG_CONT2*/
/*#define DEBUG_CHECK_CONTROL*/

#define PACK_INSTRUCTIONS

/* about XDELTA_MAX_FILE_LEN: QUERY_SIZE bits for the index entry, 2
 * bits for the instruction.  This can be relaxed if someone
 * complains, for a little more memory. */

#ifdef DEBUG_CONT
static const char* md5_to_string (const guint8* key);
#endif

/* Control functions.
 */

static XdeltaControl* control_new    (XdeltaGenerator *gen);
static void           control_insert (XdeltaControl* cont, guint len);
static void           control_copy   (XdeltaControl* cont, XdeltaSource* src, guint from, guint to);
static gboolean       control_copy_normalize   (XdeltaGenerator* gen, XdeltaControl* cont, XdeltaSource* src, guint from, guint to);
static gboolean       control_add_info (XdeltaGenerator* gen, XdeltaControl* cont, XdeltaSource* src);

static const guint16 single_hash[256] =
{
  /* Random numbers generated using SLIB's pseudo-random number
   * generator. */
  0xbcd1, 0xbb65, 0x42c2, 0xdffe, 0x9666, 0x431b, 0x8504, 0xeb46,
  0x6379, 0xd460, 0xcf14, 0x53cf, 0xdb51, 0xdb08, 0x12c8, 0xf602,
  0xe766, 0x2394, 0x250d, 0xdcbb, 0xa678, 0x02af, 0xa5c6, 0x7ea6,
  0xb645, 0xcb4d, 0xc44b, 0xe5dc, 0x9fe6, 0x5b5c, 0x35f5, 0x701a,
  0x220f, 0x6c38, 0x1a56, 0x4ca3, 0xffc6, 0xb152, 0x8d61, 0x7a58,
  0x9025, 0x8b3d, 0xbf0f, 0x95a3, 0xe5f4, 0xc127, 0x3bed, 0x320b,
  0xb7f3, 0x6054, 0x333c, 0xd383, 0x8154, 0x5242, 0x4e0d, 0x0a94,
  0x7028, 0x8689, 0x3a22, 0x0980, 0x1847, 0xb0f1, 0x9b5c, 0x4176,
  0xb858, 0xd542, 0x1f6c, 0x2497, 0x6a5a, 0x9fa9, 0x8c5a, 0x7743,
  0xa8a9, 0x9a02, 0x4918, 0x438c, 0xc388, 0x9e2b, 0x4cad, 0x01b6,
  0xab19, 0xf777, 0x365f, 0x1eb2, 0x091e, 0x7bf8, 0x7a8e, 0x5227,
  0xeab1, 0x2074, 0x4523, 0xe781, 0x01a3, 0x163d, 0x3b2e, 0x287d,
  0x5e7f, 0xa063, 0xb134, 0x8fae, 0x5e8e, 0xb7b7, 0x4548, 0x1f5a,
  0xfa56, 0x7a24, 0x900f, 0x42dc, 0xcc69, 0x02a0, 0x0b22, 0xdb31,
  0x71fe, 0x0c7d, 0x1732, 0x1159, 0xcb09, 0xe1d2, 0x1351, 0x52e9,
  0xf536, 0x5a4f, 0xc316, 0x6bf9, 0x8994, 0xb774, 0x5f3e, 0xf6d6,
  0x3a61, 0xf82c, 0xcc22, 0x9d06, 0x299c, 0x09e5, 0x1eec, 0x514f,
  0x8d53, 0xa650, 0x5c6e, 0xc577, 0x7958, 0x71ac, 0x8916, 0x9b4f,
  0x2c09, 0x5211, 0xf6d8, 0xcaaa, 0xf7ef, 0x287f, 0x7a94, 0xab49,
  0xfa2c, 0x7222, 0xe457, 0xd71a, 0x00c3, 0x1a76, 0xe98c, 0xc037,
  0x8208, 0x5c2d, 0xdfda, 0xe5f5, 0x0b45, 0x15ce, 0x8a7e, 0xfcad,
  0xaa2d, 0x4b5c, 0xd42e, 0xb251, 0x907e, 0x9a47, 0xc9a6, 0xd93f,
  0x085e, 0x35ce, 0xa153, 0x7e7b, 0x9f0b, 0x25aa, 0x5d9f, 0xc04d,
  0x8a0e, 0x2875, 0x4a1c, 0x295f, 0x1393, 0xf760, 0x9178, 0x0f5b,
  0xfa7d, 0x83b4, 0x2082, 0x721d, 0x6462, 0x0368, 0x67e2, 0x8624,
  0x194d, 0x22f6, 0x78fb, 0x6791, 0xb238, 0xb332, 0x7276, 0xf272,
  0x47ec, 0x4504, 0xa961, 0x9fc8, 0x3fdc, 0xb413, 0x007a, 0x0806,
  0x7458, 0x95c6, 0xccaa, 0x18d6, 0xe2ae, 0x1b06, 0xf3f6, 0x5050,
  0xc8e8, 0xf4ac, 0xc04c, 0xf41c, 0x992f, 0xae44, 0x5f1b, 0x1113,
  0x1738, 0xd9a8, 0x19ea, 0x2d33, 0x9698, 0x2fe9, 0x323f, 0xcde2,
  0x6d71, 0xe37d, 0xb697, 0x2c4f, 0x4373, 0x9102, 0x075d, 0x8e25,
  0x1672, 0xec28, 0x6acb, 0x86cc, 0x186e, 0x9414, 0xd674, 0xd1a5
};

#define CHEW(x) (single_hash[(guint)x])

/* Compute the hash of length len for buf, a single byte array of
 * values, by first indexing the random array.
 */

static void
init_query_checksum (const guint8 *buf, XdeltaChecksum *cksum)
{
  gint i       = QUERY_SIZE_POW;
  guint16 low  = 0;
  guint16 high = 0;

  for (; i > 0; i -= 1)
    {
      low  += CHEW(*buf++);
      high += low;
    }

  cksum->low  = low;
  cksum->high = high;
}

static void
init_long_checksum (const guint8 *buf, guint len, XdeltaChecksum *cksum)
{
  guint16 low  = cksum->low;
  guint16 high = cksum->high;

  /* @@@ unroll me? */
  for (; len > 0; len -= 1)
    {
      low  += CHEW(*buf++);
      high += low;
    }

  cksum->low  = low;
  cksum->high = high;
}

/* Generate checksums
 */

static gboolean
generate_checksums (XdeltaGenerator *gen,
		    XdeltaStream    *stream,
		    XdeltaSource    *source)
{
  gint total_checksums  = handle_length (stream) / QUERY_SIZE_POW;
  gint checksum_index   = 0;
  XdeltaChecksum cksum;
  XdeltaChecksum *result;
  const guint8* segment = NULL, *segment_pointer;
  guint   segment_len, orig_segment_len;
  guint   segment_page = 0;

#ifdef DEBUG_CKSUM
  g_print ("Total base checksums: %d\n", total_checksums);
#endif

  source->ck_count = total_checksums;

  if (total_checksums == 0)
    return TRUE;

  result = g_new (XdeltaChecksum, total_checksums);
  source->cksums = result;

  for (; segment_page <= handle_pages (stream); segment_page += 1)
    {
      segment_len = handle_map_page (stream, segment_page, &segment);

      if (segment_len < 0)
	{
	  gen->xdp_errno = XDP_StreamMapFailed;
	  return FALSE;
	}

      orig_segment_len = segment_len;

      segment_len >>= QUERY_SIZE;

      for (segment_pointer = segment;
	   segment_len != 0;
	   segment_len -= 1, segment_pointer += QUERY_SIZE_POW)
	{
	  /* note: cheating at the boundaries */
	  init_query_checksum (segment_pointer, &cksum);

#ifdef DEBUG_CKSUM
	  g_print ("New cksum %04x %04x indices %d-%d\n",
		   cksum.low, cksum.high,
		   checksum_index * QUERY_SIZE_POW, (checksum_index * QUERY_SIZE_POW) + QUERY_SIZE_POW - 1);
#endif

	  result[checksum_index++] = cksum;
	}

      if (! handle_unmap_page (stream, segment_page, &segment))
	{
	  gen->xdp_errno = XDP_StreamUnmapFailed;
	  return FALSE;
	}
    }

  return TRUE;
}

/* $Format: "#define XDELTA_REQUIRED_VERSION \"$LibCurrent$.$LibRevision$.\"" $ */
#define XDELTA_REQUIRED_VERSION "1.0."

XdeltaGenerator*
__xdp_generator_new (const HandleFuncTable* table,
		     const XdeltaMD5Table* md5,
		     const char* version)
{
  XdeltaGenerator* xg;

  if (strncmp (version, XDELTA_REQUIRED_VERSION, strlen (XDELTA_REQUIRED_VERSION)) != 0)
    g_error ("XDelta library version mismatched, compiled for %s, running %s\n", version, XDELTA_VERSION);

  xg = g_malloc0 (sizeof (XdeltaGenerator) + md5->table_md5_ctx_size - 1);

  xg->sources = g_ptr_array_new ();

  memcpy (&xg->handle_table, table, sizeof (*table));
  memcpy (&xg->md5_table, md5, sizeof (*md5));

  return xg;
}

void
xdp_generator_reset (XdeltaGenerator *gen)
{
  if (gen->table) g_free (gen->table);

  gen->table = NULL;

  g_ptr_array_set_size (gen->sources, 0);
}

static void
init_pos (XdeltaGenerator* gen, XdeltaStream* str, XdeltaPos* pos)
{
  g_assert (str);

  memset (pos, 0, sizeof (*pos));

  pos->page_size = handle_pagesize (str);
}

static gboolean
check_stream_integrity (XdeltaGenerator* gen, XdeltaStream* str, const guint8* md5, guint len)
{
  const guint8* act_md5 = handle_checksum_md5 (str);

  if (! act_md5)
    return FALSE;

  /* change this to a delayed check later? */
  if (len != handle_length (str) || memcmp (md5, act_md5, 16) != 0)
    {
      g_free (act_md5);
      gen->xdp_errno = XDP_InvalidStream;
      return FALSE;
    }

  g_free (act_md5);

  return TRUE;
}

static gboolean
xdp_source_index_read (XdeltaGenerator *gen,
		       XdeltaSource    *xs,
		       XdeltaStream    *index_in)
{
  SerialSource *ss = serializeio_handle_source (index_in, &gen->handle_table);
  XdeltaIndex *index;

  if (! ss)
    {
      gen->xdp_errno = XDP_StreamSerializeFailed;
      return FALSE;
    }

  if (unserialize_xdeltaindex (ss, &index) != SerialSuccess)
    {
      gen->xdp_errno = XDP_InvalidChecksumCache;
      return FALSE;
    }

  if (! check_stream_integrity (gen, xs->source_in, index->file_md5, index->file_len))
    return FALSE;

  xs->ck_count = index->index_len;
  xs->cksums = index->index;

  /* @@@ how to free this? */

  return TRUE;
}

static gboolean
xdp_source_index_internal (XdeltaGenerator *gen,
			   XdeltaSource    *init,
			   XdeltaStream    *source_in,
			   XdeltaOutStream *index_out)
{
  if (! generate_checksums (gen, source_in, init))
    return FALSE;

  if (index_out)
    {
      const guint8* source_in_md5;
      SerialSink* sink = serializeio_handle_sink (index_out, &gen->handle_table);

      if (! sink)
	{
	  gen->xdp_errno = XDP_StreamSerializeFailed;
	  return FALSE;
	}

      if (! (source_in_md5 = handle_checksum_md5 (source_in)))
	return FALSE;

      if (serialize_xdeltaindex (sink,
				 handle_length (source_in),
				 source_in_md5,
				 init->ck_count,
				 init->cksums) != SerialSuccess)
	{
	  g_free (source_in_md5);
	  gen->xdp_errno = XDP_StreamWriteFailed;
	  return FALSE;
	}

      g_free (source_in_md5);

      if (! handle_close (index_out, 0))
	{
	  gen->xdp_errno = XDP_StreamCloseFailed;
	  return FALSE;
	}
    }

  return TRUE;
}

gboolean
xdp_source_index (XdeltaGenerator *gen,
		  XdeltaStream    *source_in,
		  XdeltaOutStream *index_out)
{
  XdeltaSource* xs = xdp_source_new (gen, NULL, source_in, NULL, index_out);

  if (xs)
    {
      xdp_source_free (xs);
      return TRUE;
    }

  return FALSE;
}

XdeltaSource*
xdp_source_new (XdeltaGenerator *gen,
		XdeltaControl   *delta_control_in,
		XdeltaStream    *source_in,
		XdeltaStream    *index_in,
		XdeltaOutStream *index_out)
{
  XdeltaSource* xs = g_new0 (XdeltaSource, 1);

  xs->delta_control_in = delta_control_in;
  xs->source_in        = source_in;

  g_return_val_if_fail (gen && source_in, NULL);
  g_return_val_if_fail (index_in ? ! index_out : TRUE, NULL);

  xs->index_in = index_in;
  xs->index_out = index_out;
  xs->source_pos.page_size = handle_pagesize (source_in);

  return xs;
}

static gboolean
xdp_source_check_index (XdeltaGenerator *gen,
			XdeltaSource    *xs)
{
  if (! xs->index_in)
    return xdp_source_index_internal (gen, xs, xs->source_in, xs->index_out);
  else
    return xdp_source_index_read (gen, xs, xs->index_in);
}

void
xdp_source_free (XdeltaSource* xs)
{
  if (xs)
    {
      /* if (xs->ckarray) @@@ this is troublesome now
	g_free (xs->ckarray);*/

      g_free (xs);
    }
}

const char*
xdp_strerror (gint e)
{
  switch ((XdpError) e)
    {
    case XDP_StreamMapFailed:
      return "Stream map failed";
    case XDP_StreamUnmapFailed:
      return "Stream unmap failed";
    case XDP_StreamWriteFailed:
      return "Stream write failed";
    case XDP_StreamCloseFailed:
      return "Stream close failed";
    case XDP_TooManySources:
      return "Too many source files";
    case XDP_SourceTooLong:
      return "Source file too long";
    case XDP_ControlEof:
      return "EOF in control data";
    case XDP_IllegalInstruction:
      return "Illegal delta instruction";
    case XDP_MismatchedFromMD5:
      return "Mismatched source MD5 checksum";
    case XDP_MismatchedToMD5:
      return "Mismatched constructed MD5 checksum";
    case XDP_BadDataMD5:
      return "Bad data MD5 checksum";
    case XDP_BadControlMagic:
      return "Bad control magic number";
    case XDP_UnrecognizedControlSeq:
      return "Unrecognized control format version";
    case XDP_BadDataLength:
      return "Wrong data length";
    case XDP_BadSrcControl:
      return "Source control damaged";
    case XDP_CannotNormalize:
      return "Missing normalization instructions";
    case XDP_CallbackFailed:
      return "Client callback failed";
    case XDP_CallbackShort:
      return "Client wrote incorrect length";
    case XDP_InvalidChecksumCache:
      return "Invalid checksum cache";
    case XDP_TooFewSources:
      return "Too few sources";
    case XDP_BadLength:
      return "Produced file with wrong length";
    case XDP_InvalidRsyncCache:
      return "Invalid rsync index cache";
    case XDP_StreamSerializeFailed:
      return "Serialization failed";
    case XDP_InvalidStream:
      return "Invalid stream construction";
    case XDP_StreamTooLong:
      return "Input stream too long";
    }

  return "No error";
}

gint
xdp_errno (XdeltaGenerator *gen)
{
  g_return_val_if_fail (gen, 0);

  return gen->xdp_errno;
}

void
xdp_source_add (XdeltaGenerator *gen,
		XdeltaSource    *src)
{
  if (gen->table)
    {
      g_free ((gpointer)gen->table);
      gen->table = NULL;
    }

  g_ptr_array_add (gen->sources, src);
}

static guint
c_hash (const XdeltaChecksum* c)
{
  const guint high = c->high;
  const guint low = c->low;
  const guint it = (high >> 2) + (low << 3) + (high << 16);

  return (it ^ high ^ low);
}

static gboolean
region_insert (XdeltaGenerator* gen, const XdeltaPos *xpos, guint len)
{
  /* This is a little bit cryptic: the xpos.mem + EXPR expression
   * computes the offset into the current page, which is guaranteed
   * to be correct since map_page has not occured yet. */
  const guint8* buf = xpos->mem + (gen->to_output_pos % xpos->page_size);

  if (len == 0)
    return TRUE;

#ifdef DEBUG_INST
  g_print ("insert %d at %d\n", len, gen->to_output_pos);
#endif

  if (! handle_write (gen->data_out, buf, len))
    return FALSE;

  control_insert (gen->control, len);

  gen->to_output_pos += len;

  return TRUE;
}

static gboolean
region_copy (XdeltaGenerator* gen, XdeltaSource* src, guint from, guint to)
{
#ifdef DEBUG_INST
  g_print ("copy %d - %d (%d) to %d\n", from, to, to-from, gen->to_output_pos);
#endif

  control_copy (gen->control, src, from, to);

  if (gen->normalize)
    {
      if (! control_copy_normalize (gen, gen->control, src, from, to))
	return FALSE;
    }

  gen->to_output_pos += (to-from);

  return TRUE;
}

static gboolean
unmap_page (XdeltaGenerator *gen, XdeltaStream* stream, XdeltaPos* pos)
{
  if (! pos->mem)
    return TRUE;

  if (handle_unmap_page (stream,
			 pos->mem_page,
			 &pos->mem))
    {
      pos->mem = NULL;
      return TRUE;
    }

  gen->xdp_errno = XDP_StreamUnmapFailed;

  return FALSE;
}

static gboolean
map_page (XdeltaGenerator *gen, XdeltaStream* stream, XdeltaPos* pos)
{
  gint on_page;

  if (pos->mem && pos->mem_page == pos->page)
    return TRUE;

  if (pos->mem)
    {
      if (! handle_unmap_page (stream,
			       pos->mem_page,
			       &pos->mem))
	{
	  gen->xdp_errno = XDP_StreamUnmapFailed;
	  return FALSE;
	}

      pos->mem = NULL;
    }

  pos->mem_page = pos->page;

  on_page = handle_map_page (stream,
			     pos->mem_page,
			     &pos->mem);

  if (on_page >= 0)
    {
      pos->mem_rem = on_page;
      return TRUE;
    }

  gen->xdp_errno = XDP_StreamMapFailed;
  return FALSE;
}

#define FLIP_FORWARD(p)  if ((p).off == (p).page_size) { (p).page += 1; (p).off = 0; }

static gboolean
try_match (XdeltaGenerator *gen,
	   XdeltaStream    *in,
	   XdeltaPos       *xpos_ptr,
	   XdeltaSource    *src,
	   guint            src_offset,
	   gboolean        *found)
{
  XdeltaPos xpos = *xpos_ptr;
  XdeltaPos ypos = src->source_pos;
  gint rem, remsave;
  gint match_forward  = 0;
  gint match_backward = 0;
  gint match_forward_max;
  gint match_backward_max;
  guint to_offset = XPOS (xpos);
  gboolean one_insert = FALSE;

  *found = FALSE;

  ypos.page = src_offset / ypos.page_size;
  ypos.off  = src_offset % ypos.page_size;

  match_forward_max  = MIN (handle_length (in)             - to_offset,
			    handle_length (src->source_in) - src_offset);
  match_backward_max = MIN (src_offset, to_offset - gen->to_output_pos);

  /* Don't allow backward paging */
  match_backward_max = MIN (match_backward_max, xpos.off);

  /* We're testing against the negative below. */
  match_backward_max = - match_backward_max;

  for (; match_backward > match_backward_max; )
    {
      g_assert (xpos.off != 0);

      if (ypos.off == 0)
	{
	  ypos.off = ypos.page_size;
	  ypos.page -= 1;
	}

      if (! map_page (gen, src->source_in, &ypos))
	goto bail;

      rem = MIN (xpos.off, ypos.off);
      rem = MIN (match_backward - match_backward_max, rem);

      for (; rem > 0; rem -= 1, match_backward -= 1)
	{
	  if (xpos.mem[xpos.off-1] != ypos.mem[ypos.off-1])
	    goto doneback;

	  xpos.off -= 1;
	  ypos.off -= 1;
	}
    }

doneback:

  xpos.page = to_offset / xpos.page_size;
  xpos.off  = to_offset % xpos.page_size;

  ypos.page = src_offset / ypos.page_size;
  ypos.off  = src_offset % ypos.page_size;

  for (; match_forward < match_forward_max; )
    {
      if (! map_page (gen, src->source_in, &ypos))
	goto bail;

      /* Fortunately, if this map happens it means that the match must
       * be long enough to succeed below, therefore it is safe to write
       * the insert out now. */
      if (! one_insert && xpos.page != xpos.mem_page)
	{
	  one_insert = TRUE;

	  if (! region_insert (gen, &xpos, (to_offset + match_backward) - gen->to_output_pos))
	    goto bail;
	}

      if (! map_page (gen, in, &xpos))
	goto bail;

      rem = MIN (xpos.mem_rem - xpos.off, ypos.mem_rem - ypos.off);
      rem = MIN (match_forward_max - match_forward, rem);

      /* Do a int-wise comparison if the regions are aligned. */
      if (rem > (4*sizeof(int)) && (xpos.off % sizeof (int)) == (ypos.off % sizeof(int)))
	{
	  gint is;
	  const int *xi, *yi;

	  for (; xpos.off % sizeof(int); rem -= 1, match_forward += 1)
	    {
	      if (xpos.mem[xpos.off] != ypos.mem[ypos.off])
		goto done;

	      xpos.off += 1;
	      ypos.off += 1;
	    }

	  remsave = rem;
	  rem /= sizeof(int);

	  xi = (const int*) (xpos.mem + xpos.off);
	  yi = (const int*) (ypos.mem + ypos.off);

	  is = rem;

	  for (; rem > 0 && *xi == *yi; )
	    {
	      rem -= 1;
	      xi += 1;
	      yi += 1;
	    }

	  is -= rem;

	  match_forward += is * sizeof(int);
	  xpos.off      += is * sizeof(int);
	  ypos.off      += is * sizeof(int);

 	  rem = (rem * sizeof(int)) + (remsave % sizeof(int));
	}

      for (; rem > 0; rem -= 1, match_forward += 1)
	{
	  if (xpos.mem[xpos.off] != ypos.mem[ypos.off])
	    goto done;

	  xpos.off += 1;
	  ypos.off += 1;
	}

      FLIP_FORWARD (xpos);
      FLIP_FORWARD (ypos);
    }

done:

  if (match_forward - match_backward >= QUERY_SIZE_POW)
    {
      *found = TRUE;

      if (! one_insert)
	{
	  if (! region_insert (gen, &xpos, (to_offset + match_backward) - gen->to_output_pos))
	    goto bail;
	}

      if (! region_copy (gen, src, src_offset + match_backward, src_offset + match_forward))
	goto bail;
    }
  else
    {
      g_assert (! one_insert);
    }

  *xpos_ptr = xpos;
  src->source_pos = ypos;
  return TRUE;

bail:
  *xpos_ptr = xpos;
  src->source_pos = ypos;
  return FALSE;
}

static gboolean
compute_copies (XdeltaGenerator* gen, XdeltaStream* stream)
{
  XdeltaChecksum cksum;
  const XdeltaChecksum *source_cksum;
  const guint8 *segment_pointer;
  guint source_offset, segment_index, index, prime = gen->table_size;
  guint source_index;
  const guint32* table = gen->table;
  guint16 old_c, new_c;
  guint save_page, save_off;
  XdeltaPos xpos;
  gboolean found;
  gboolean ret = TRUE;

  if (handle_length (stream) < QUERY_SIZE_POW)
    return TRUE;

  init_pos (gen, stream, &xpos);

  while (XPOS (xpos) <= (handle_length (stream) - QUERY_SIZE_POW))
    {
      if (!map_page (gen, stream, &xpos))
	return FALSE;

      g_assert (xpos.mem_rem > xpos.off);

      segment_index = (xpos.mem_rem - xpos.off);

      if (segment_index < QUERY_SIZE_POW)
	goto nextpage;

      segment_index -= QUERY_SIZE_POW;

      segment_pointer = xpos.mem + xpos.off;

      init_query_checksum (segment_pointer, &cksum);

      for (; ; segment_pointer += 1)
	{
	  index = c_hash (&cksum) % prime;

	  if (table[index])
	    {
	      source_index  = (table[index] &  QUERY_SIZE_MASK) - 1;
	      source_offset = (table[index] >> QUERY_SIZE);

	      source_cksum = ((XdeltaSource*)gen->sources->pdata[source_index])->cksums + source_offset;

	      if (cksum.high == source_cksum->high &&
		  cksum.low  == source_cksum->low)
		{
		  save_page = xpos.page;
		  save_off  = xpos.off;

		  if (! try_match (gen,
				   stream,
				   &xpos,
				   gen->sources->pdata[source_index],
				   source_offset << QUERY_SIZE,
				   &found))
		    {
		      ret = FALSE;
		      goto bail;
		    }

		  if (found)
		    {
		      g_assert (xpos.page*xpos.page_size+xpos.off == gen->to_output_pos);

		      goto reenter;
		    }
		  else
		    {
		      xpos.page = save_page;
		      xpos.off  = save_off;
		    }
		}
	    }

	  if (segment_index == 0)
	    goto nextpage;

	  segment_index -= 1;
	  xpos.off += 1;

	  old_c = CHEW(segment_pointer[0]);
	  new_c = CHEW(segment_pointer[QUERY_SIZE_POW]);

	  cksum.low -= old_c;
	  cksum.low += new_c;

	  cksum.high -= old_c << QUERY_SIZE;
	  cksum.high += cksum.low;
	}

    nextpage:

      if (xpos.mem_rem < xpos.page_size)
	break;

      xpos.page += 1;
      xpos.off = 0;

      if (xpos.page != xpos.mem_page)
	{
	  if (! region_insert (gen, &xpos, XPOS (xpos) - gen->to_output_pos))
	    return FALSE;
	}

    reenter:
    }

  xpos.off = gen->to_output_pos % handle_pagesize (stream);

  while (gen->to_output_pos < handle_length (stream))
    {
      if (! map_page (gen, stream, &xpos))
	return FALSE;

      if (! region_insert (gen, &xpos, xpos.mem_rem - xpos.off))
	ret = FALSE;

      xpos.off = 0;
      xpos.page += 1;
    }

bail:

  if (! unmap_page (gen, stream, &xpos))
      return FALSE;

  return ret;
}

static gboolean
just_output (XdeltaGenerator *gen,
	     XdeltaStream    *in)
{
  XdeltaPos pos;

  init_pos (gen, in, &pos);

  while (gen->to_output_pos < handle_length (in))
    {
      if (! map_page (gen, in, &pos))
	return FALSE;

      if (! region_insert (gen, &pos, pos.mem_rem - pos.off))
	return FALSE;

      pos.off = 0;
      pos.page += 1;
    }

  if (! unmap_page (gen, in, &pos))
    return FALSE;

  return TRUE;
}

static gboolean
xdp_generate_delta_int (XdeltaGenerator *gen,
			XdeltaStream    *in,
			XdeltaOutStream *control_out,
			XdeltaOutStream *data_out)
{
  gint i, j, total_from_ck_count = 0, prime = 0, index;
  gint total_from_len = 0;
  guint32* table;

  if (gen->sources->len == 0)
    {
      gen->xdp_errno = XDP_TooFewSources;
      return FALSE;
    }

  if (handle_length (in) >= XDELTA_MAX_FILE_LEN)
    {
      gen->xdp_errno = XDP_StreamTooLong;
      return FALSE;
    }

  for (i = 0; i < gen->sources->len; i += 1)
    {
      XdeltaSource* src = gen->sources->pdata[i];

      src->used = FALSE;

      total_from_len += handle_length (src->source_in);

      if (handle_length (src->source_in) >= XDELTA_MAX_FILE_LEN)
	{
	  gen->xdp_errno = XDP_SourceTooLong;
	  return FALSE;
	}
    }

  if (gen->sources->len >= QUERY_SIZE_POW)
    {
      gen->xdp_errno = XDP_TooManySources;
      return FALSE;
    }

  if (handle_length (in) < QUERY_SIZE_POW || total_from_len < QUERY_SIZE_POW)
    {
      if (! just_output (gen, in))
	return FALSE;
    }
  else
    {
#ifdef CLOBBER_ALGORITHM_C

#define CLOBBER_ALGORITHM_C

#define WEIGHT gfloat

      WEIGHT *running_weights;
      guint source_cksum_counts[QUERY_SIZE_POW];
#endif

      for (i = 0; i < gen->sources->len; i += 1)
	{
	  XdeltaSource* xs = (XdeltaSource*)gen->sources->pdata[i];

	  if (! xdp_source_check_index (gen, xs))
	    return FALSE;

	  total_from_ck_count += xs->ck_count;

#ifdef CLOBBER_ALGORITHM_C
	  source_cksum_counts[i] = xs->ck_count;
#endif

	  xs->source_index = i;
	}

      prime = g_spaced_primes_closest (total_from_ck_count);

      gen->table = table = g_new0 (guint32, prime);
      gen->table_size = prime;

#ifdef CLOBBER_ALGORITHM_C
      running_weights = g_new0 (WEIGHT, prime);
#endif

      for (i = 0; i < gen->sources->len; i += 1)
	{
	  XdeltaSource* xs = (XdeltaSource*)gen->sources->pdata[i];

	  for (j = xs->ck_count-1; j >= 0; j -= 1)
	    {
	      index = c_hash (xs->cksums + j) % prime;

#ifdef DEBUG_HASH
	      gen->hash_entries += 1;

	      if (table[index])
		{
		  gen->hash_conflicts += 1;

		  regions_similar (gen,
				   i,
				   j,
				   (table[index] & QUERY_SIZE_MASK) - 1,
				   table[index] >> QUERY_SIZE);
		}
#endif

	      /* Now, I have to decide whether to clobber the current
	       * entry, if there is one.
	       *
	       * Algorithm A: Clobber it always (its fast!).  The problem
	       *              is that this prefers matches at the front of
	       *              the file and leads to poor matches at the back
	       *              of the file (assuming I insert going backwards).
	       *
	       * Algorithm B: Keep a table of how many times there has
	       *              been a clobber at each index i, C[i].
	       *              With probability 1/(C[i]+1), replace the
	       *              previous entry.  This gives a uniform
	       *              probability of each entry surviving.
	       *              The problem (supposed) with this
	       *              algorithm is that the probabilities
	       *              should not be uniform (though uniform is
	       *              better than A) because there are more
	       *              chances to match a segment at the end of
	       *              the file than at the beginning.
	       *
	       * Algorithm C: Give a linear weight to each match
	       *              according to it's position in the file
	       *              -- number the segments from N down to 1
	       *              starting at the beginning.  Same as the
	       *              above but with a different weight.  The
	       *              weight for segment i, match at checksum
	       *              offset k, follows.  The total number of
	       *              checksums in the segment is C_i,
	       *              therefore the total checksum count is
	       *              C = sum_i (C_i).
	       *              Now the interior weight is the (C_i-k)
	       *              (the linear assumption) and the total
	       *              interior weight is sum_{j=1}^{N}{j}=(N)(N+1)/2
	       *              so the kth segment has interior weight
	       *
	       *                [2 (C_i - k)] / [(C_i) (C_i + 1)]
	       *
	       *              add in the exterior weight (after
	       *              cancelling a C_i):
	       *
	       *                w(i,k) = [2 (C_i - k)] / [(C_i + 1) (C)]
	       *
	       *              Now, as above, we will compute whether to
	       *              keep or replace the current value at the j-th
	       *              decision.  Let R_j be the running sum of
	       *              weights considered so far.  R_0 = 0.  At the
	       *              j-th decision,
	       *
	       *                P_ikj(use new value) = w(i,k)/R_{j+1}
	       *                R_{j+1} = R_j + w(i,k)
	       */

#ifdef CLOBBER_ALGORITHM_C
	      {
		/* i is the source index, j is the cksum index */

#ifndef CLOBBER_ALGORITHM_B  /* this is C -- linear assumption */
		WEIGHT this_weight =
		  ((WEIGHT) ((source_cksum_counts[i] - j) * 2)) /
		  ((WEIGHT) ((source_cksum_counts[i]+1) * total_from_ck_count));
#else           /* this is B -- no linear assumption */
		WEIGHT this_weight = source_cksum_counts[i] / total_from_ck_count;
#endif
		WEIGHT prob_of_replace;

		running_weights[index] += this_weight;

		prob_of_replace = this_weight / running_weights[index];

		if (drand48 () < prob_of_replace)
		  table[index] = (j << QUERY_SIZE) + 1 + i;
	      }
#else
	      table[index] = (j << QUERY_SIZE) + 1 + i;
#endif
	    }
	}

#ifdef CLOBBER_ALGORITHM_C
      g_free (running_weights);
#endif

#ifdef DEBUG_HASH
      for (i = 0; i < prime; i += 1)
	{
	  if (gen->table[i])
	    gen->hash_fill += 1;
	}

      g_print ("*** Hash stats:\n");
      g_print ("Hash conflicts: %d\n", gen->hash_conflicts);
      g_print ("Hash real conflicts: %d\n", gen->hash_real_conflicts);
      g_print ("Hash real real conflicts: %d\n", gen->hash_real_real_conflicts);
      g_print ("Hash fill:      %d\n", gen->hash_fill);
      g_print ("Hash size:      %d\n", gen->table_size);
      g_print ("Hash entries:   %d\n", gen->hash_entries);
      g_print ("Hash fill/entries: %f\n", (float)gen->hash_fill/(float)gen->hash_entries);
#endif

      if (! compute_copies (gen, in))
	return FALSE;
    }

  return TRUE;
}

XdeltaControl*
xdp_generate_delta (XdeltaGenerator *gen,
		    XdeltaStream    *in,
		    gboolean         normalize,
		    XdeltaOutStream *control_out,
		    XdeltaOutStream *data_out)
{
  gint i;
  const guint8* in_md5;
  const guint8* data_out_md5;

  gen->data_out = data_out;
  gen->control_out = control_out;
  gen->control = control_new (gen);
  gen->normalize = normalize;

  if (! xdp_generate_delta_int (gen, in, control_out, data_out))
    return NULL;

  gen->control->inst = &g_array_index (gen->control->inst_array, XdeltaInstruction, 0);
  gen->control->inst_len = gen->control->inst_array->len;

  for (i = 0; i < gen->sources->len; i += 1)
    {
      if (! control_add_info (gen, gen->control, gen->sources->pdata[i]))
	return NULL;
    }

  if (! handle_close (data_out, 0))
    {
      gen->xdp_errno = XDP_StreamCloseFailed;
      return NULL;
    }

  gen->control->data_len       = handle_length (data_out);
  gen->control->normalized     = gen->normalize;

  gen->control->source_info = (XdeltaSourceInfo**) gen->control->source_info_array->pdata;
  gen->control->source_info_len = gen->control->source_info_array->len;

  if (! (in_md5 = handle_checksum_md5 (in)))
    return FALSE;

  if (! (data_out_md5 = handle_checksum_md5 (data_out)))
    return FALSE;

  gen->control->to_info.length = handle_length (in);
  memcpy (gen->control->to_info.real_md5, in_md5,       16);
  memcpy (gen->control->to_info.md5,      data_out_md5, 16);

  if (control_out && ! xdp_control_write (gen, gen->control, control_out))
    return NULL;

  return gen->control;
}

/* Below here boring details mostly to do with reading and writing
 * control. */

XdeltaControl*
control_new (XdeltaGenerator* gen)
{
  XdeltaControl* it = g_new0 (XdeltaControl, 1);

  it->inst_array        = g_array_new (FALSE, FALSE, sizeof (XdeltaInstruction));
  it->source_info_array = g_ptr_array_new ();

  return it;
}

static void
control_reindex (XdeltaGenerator* gen, XdeltaControl* cont, XdeltaSource* src)
{
  gint i;
  gint new_index = cont->source_info_array->len;

  for (i = 0; i < cont->inst_len; i += 1)
    {
      XdeltaInstruction* inst = cont->inst + i;

      switch (inst->type)
	{
	case 'N':
	case 'C':
	  if (inst->index == src->source_index)
	    inst->index = new_index;
	}
    }
}

gboolean
control_add_info (XdeltaGenerator* gen, XdeltaControl* cont, XdeltaSource* src)
{
  XdeltaSourceInfo* si;

  if (! src->used)
    return TRUE;

  si = g_new0 (XdeltaSourceInfo, 1);

  si->length = handle_length (src->source_in);

  memcpy (si->md5, handle_checksum_md5 (src->source_in), 16);

  if (src->delta_control_in)
    memcpy (si->real_md5, src->delta_control_in->to_info.real_md5, 16);
  else
    {
      /* If the source is not a delta, record (real_md5 = md5) */
      memcpy (si->real_md5, si->md5, 16);
    }

  control_reindex (gen, cont, src);

  g_ptr_array_add (cont->source_info_array, si);

  return TRUE;
}

void
xdp_control_free (XdeltaControl* cont)
{
  if (cont->source_info_array)
    g_ptr_array_free (cont->source_info_array, TRUE);
  if (cont->inst_array)
    g_array_free (cont->inst_array, TRUE);
  g_free (cont);
}

void
control_insert (XdeltaControl* cont, guint len)
{
  XdeltaInstruction i;

  memset (&i, 0, sizeof(i));

  if (cont->inst_array->len > 0)
    {
      XdeltaInstruction* prev = &g_array_index (cont->inst_array, XdeltaInstruction, cont->inst_array->len-1);

      if (prev->type == 'I')
	{
	  prev->length += len;
	  return;
	}
    }

  i.type   = 'I'; /* I for insert */
  i.length = len;

  g_array_append_val (cont->inst_array, i);
}

void
control_copy (XdeltaControl* cont, XdeltaSource* src, guint from, guint to)
{
  XdeltaInstruction i;

  i.type = 'C'; /* C for copy */
  i.offset = from;
  i.length = to - from;
  i.index  = src->source_index;

  src->used = TRUE;

  g_array_append_val (cont->inst_array, i);
}

gboolean
control_copy_normalize (XdeltaGenerator* gen, XdeltaControl* cont, XdeltaSource* src, guint from, guint to)
{
  if (! src->delta_control_in)
    {
      XdeltaInstruction i;

      memset (&i, 0, sizeof(i));

      i.type = 'E'; /* E for error, can't apply this copy normalized. */

      g_array_append_val (cont->inst_array, i);

      return TRUE;
    }
  else
    {
      XdeltaControl *src_cont = src->delta_control_in;

      guint src_index = 0;
      guint copy_index = 0, new_copy_index;
      guint src_inst_index = 0;

      for (; from < to && src_inst_index < src_cont->inst_len; src_inst_index += 1)
	{
	  const XdeltaInstruction *src_inst = src_cont->inst + src_inst_index;

	  switch (src_inst->type)
	    {
	    case 'C':
	      src_index += src_inst->length;
	      break;
	    case 'I':

	      new_copy_index = copy_index + src_inst->length;

	      if (from < new_copy_index && from >= copy_index)
		{
		  XdeltaInstruction i;
		  guint copy_len = MIN (new_copy_index, to) - from;

		  i.type = 'N'; /* N for normalized copy */
		  i.offset = src_index + (from - copy_index);
		  i.length = copy_len;
		  i.index = src->source_index;

		  g_array_append_val (cont->inst_array, i);

		  from += copy_len;
		}

	      copy_index += src_inst->length;
	      src_index += src_inst->length;
	    }
	}

      if (to == from)
	return TRUE;

      gen->xdp_errno = XDP_BadSrcControl;

      return FALSE;
    }
}

#ifdef DEBUG_CONT
static const char*
md5_to_string (const guint8* key)
{
  static char buf[33];
  gint i;

  for (i = 0; i < 16; i += 1)
    sprintf (buf + 2*i, "%02x", key[i]);

  return buf;
}

static void
print_info (XdeltaSourceInfo* si)
{
  g_print (" ** info\n");
  g_print (" md5 %s\n", md5_to_string (si->md5));
  g_print (" real md5 %s\n", md5_to_string (si->real_md5));
  g_print (" length %d\n", si->length);
}

static void
print_inst (XdeltaInstruction* i)
{
  switch (i->type)
    {
    case 'C':
    case 'N':
      g_print ("    copy (%c) %d-%d (%d) from %d\n", i->type, i->offset, i->offset + i->length, i->length, i->index);
      break;
    case 'I':
      g_print ("    insert %d\n", i->length);
      break;
    case 'E':
      g_print ("    error (not normalizable)\n");
      break;
    }
}

static void
xdp_print_control (XdeltaControl *cont)
{
  gint i;

  g_print ("*** control\n");

  g_print (" data len: %d\n", cont->data_len);
  print_info (&cont->to_info);

  g_print (" source info len: %d\n", cont->source_info_len);

  for (i = 0; i < cont->source_info_len; i += 1)
    print_info (cont->source_info[i]);

  g_print ("   inst len: %d\n", cont->inst_len);

  for (i = 0; i < cont->inst_len; i += 1)
    print_inst (cont->inst + i);

}
#endif

#ifdef DEBUG_CHECK_CONTROL
void
check_control (XdeltaControl* cont)
{
  gint i;

  for (i = 0; i < cont->inst_len; i += 1)
    {
      switch (cont->inst[i].type)
	{
	case 'N':
	case 'C':
	  if (cont->inst[i].index >= cont->source_info_len)
	    g_error ("control has a too high instruction index\n");
	}
    }
}
#endif

static void
pack_instructions (XdeltaControl* cont)
{
#ifdef PACK_INSTRUCTIONS
  gint i;

  for (i = 0; i < cont->inst_len; i += 1)
    {
      g_assert (cont->inst[i].length < XDELTA_MAX_FILE_LEN);

      cont->inst[i].length <<= QUERY_SIZE;
      cont->inst[i].length |= cont->inst[i].index;
      cont->inst[i].length <<= 2;

      switch (cont->inst[i].type)
	{
	case 'N': cont->inst[i].length |= 0; break;
	case 'E': cont->inst[i].length |= 1; break;
	case 'C': cont->inst[i].length |= 2; break;
	case 'I': cont->inst[i].length |= 3; break;
	}
    }
#endif
}

static void
unpack_instructions (XdeltaControl* cont)
{
#ifdef PACK_INSTRUCTIONS
  gint i;

  for (i = 0; i < cont->inst_len; i += 1)
    {
      switch (cont->inst[i].length & 3)
	{
	case 0: cont->inst[i].type = 'N'; break;
	case 1: cont->inst[i].type = 'E'; break;
	case 2: cont->inst[i].type = 'C'; break;
	case 3: cont->inst[i].type = 'I'; break;
	}

      cont->inst[i].length >>= 2;
      cont->inst[i].index = cont->inst[i].length & QUERY_SIZE;
      cont->inst[i].length >>= QUERY_SIZE;
    }
#endif
}

gboolean
xdp_control_write (XdeltaGenerator *gen,
		   XdeltaControl   *cont,
		   XdeltaOutStream *cont_out)
{
  SerialSink* sink = serializeio_handle_sink (cont_out, &gen->handle_table);

#ifdef DEBUG_CONT
  xdp_print_control (cont);
#endif
#ifdef DEBUG_CHECK_CONTROL
  check_control (cont);
#endif

  if (! sink)
    {
      gen->xdp_errno = XDP_StreamSerializeFailed;
      return FALSE;
    }

  pack_instructions (cont);

  if (serialize_xdeltacontrol_obj (sink, cont) != SerialSuccess)
    {
      gen->xdp_errno = XDP_StreamWriteFailed;
      return FALSE;
    }

  unpack_instructions (cont);

  if (! handle_close (cont_out, 0))
    {
      gen->xdp_errno = XDP_StreamCloseFailed;
      return FALSE;
    }

  return TRUE;
}

XdeltaControl*
xdp_control_read (XdeltaGenerator *gen,
		  XdeltaStream    *cont_in)
{
  SerialSource* src = serializeio_handle_source (cont_in, &gen->handle_table);
  XdeltaControl* cont;

  if (! src)
    {
      gen->xdp_errno = XDP_StreamSerializeFailed;
      return NULL;
    }

  if (unserialize_xdeltacontrol (src, &cont) != SerialSuccess)
    {
      gen->xdp_errno = XDP_StreamMapFailed;
      return NULL;
    }

  unpack_instructions (cont);

#ifdef DEBUG_CHECK_CONTROL
  check_control (cont);
#endif

  return cont;
}

/* Applying patches.
 */

gboolean
xdp_apply_delta (XdeltaGenerator   *gen,
		 XdeltaControl     *cont,
		 XdeltaStream      *data,
		 XdpSourceCallback *call,
		 void              *user_data,
		 XdeltaOutStream   *res)
{
  if (! xdp_copy_delta_region (gen, cont, data, call, 0, cont->to_info.length, user_data, res))
    return FALSE;

  /* close the resulting stream */
  if (! handle_close (res, 0))
    {
      gen->xdp_errno = XDP_StreamCloseFailed;
      return FALSE;
    }

  if (! check_stream_integrity (gen, res, cont->to_info.real_md5, cont->to_info.length))
    return FALSE;

  return TRUE;
}

gboolean
xdp_copy_delta_region    (XdeltaGenerator   *gen,
			  XdeltaControl     *cont,
			  XdeltaStream      *data,
			  XdpSourceCallback *call,
			  guint              output_from,
			  guint              len,
			  void              *user_data,
			  XdeltaOutStream   *output_stream)
{
  gint i, l = cont->inst_len;
  guint data_index = 0;
  guint output_index = 0;
  guint output_to = output_from + len;
  gint write_from, write_to, writing, write_offset;
  gint save_written;

  /* Could use a state machine here, because the logic is a little
   * confusing. */

#define START_STATE 0        /* Accept a C or I instruction. */
#define IGNORE_STATE 1       /* Accept a N or E instruction, but ignore.  Move to IGNORE_DONE. */
#define IGNORE_DONE_STATE 2  /* Accept a N or E instruction, but ignore.  Return to START on C or I. */
#define REQUIRE_STATE 3      /* Accept a N or E instruction.  Move to REQUIRE_DONE. */
#define REQUIRE_DONE_STATE 4 /* Accept a N or E instruction.  Return to START on C or I. */

  /* But instead, use three variables:
   *
   * expecting_norm is true IFF state!=START
   * ignore_norm is true IFF (state=IGNORE_STATE || state=IGNORE_DONE_STATE)
   * need_norm is true IFF (state=IGNORE_STATE || state=REQUIRE_STATE)
   */

  /* And, if ! cont->normalized, state should never enter the norm
   * states. */

  /* So, there are two separate pieces of logic below: instruction/control logic
   * and the output logic.  The output logic is marked, and deals with outputting
   * only the region between OFF and OFF+LEN. */

  gboolean expecting_norm = FALSE;
  gboolean ignore_norm = FALSE;
  gboolean need_norm = FALSE;

  for (i = 0; i < l && output_index < output_to; i += 1)
    {
      const XdeltaInstruction *inst = cont->inst + i;

      /* BEGIN OUTPUT LOGIC */
      write_from = MAX (output_index, output_from);
      write_to = MIN (output_index+inst->length, output_to);
      writing = write_to - write_from;
      write_offset = write_from-output_index;
      save_written = handle_length (output_stream);
      /* END OUTPUT LOGIC */

      switch (inst->type)
	{
	case 'I':

	  if (need_norm)
	    goto badinst;

	  /* BEGIN OUTPUT LOGIC */
	  if (writing > 0 &&
	      ! handle_copy (data,
			     output_stream,
			     data_index+write_offset,
			     writing))
	    return FALSE;

	  data_index += inst->length;
	  output_index += inst->length;
	  /* END OUTPUT LOGIC */

	  expecting_norm = FALSE;

	  break;
	case 'C':

	  if (inst->index >= cont->source_info_len)
	    goto toofewsrc;

	  if (need_norm)
	    goto badinst;

	  need_norm = expecting_norm = cont->normalized;

	  /* BEGIN OUTPUT LOGIC */
	  if (writing > 0)
	    {
	      ignore_norm = (* call) (gen,
				      user_data,
				      cont->source_info[inst->index]->md5,
				      FALSE,
				      inst->offset + write_offset,
				      writing,
				      output_stream);

	      if (handle_length (output_stream) != save_written + ignore_norm * writing)
		goto shortcall;
	    }
	  else
	    ignore_norm = TRUE;

	  output_index += (ignore_norm * inst->length);
	  /* END OUTPUT LOGIC */

	  if (! ignore_norm && ! cont->normalized)
	    goto fail;

	  break;
	case 'N':

	  if (inst->index >= cont->source_info_len)
	    goto toofewsrc;

	  if (! expecting_norm)
	    goto badinst;

	  if (! ignore_norm )
	    {
	      /* BEGIN OUTPUT LOGIC */
	      if (writing > 0)
		{
		  if (! (* call) (gen,
				  user_data,
				  cont->source_info[inst->index]->real_md5,
				  FALSE,
				  inst->offset + write_offset,
				  writing,
				  output_stream))
		    goto fail;

		  if (handle_length (output_stream) != save_written + ignore_norm * writing)
		    goto shortcall;
		}

	      output_index += (ignore_norm * inst->length);
	      /* END OUTPUT LOGIC */
	    }

	  need_norm = FALSE;
	  break;
	case 'E':

	  if (! expecting_norm)
	    goto badinst;

	  if (! ignore_norm)
	    goto cannot;

	  need_norm = FALSE;
	  expecting_norm = FALSE;
	  break;

	default:

	  goto badinst;
	}
    }

  if (! check_stream_integrity (gen, data, cont->to_info.md5, cont->data_len))
    return FALSE;

  return TRUE;

badinst:
  gen->xdp_errno = XDP_IllegalInstruction;
  return FALSE;

toofewsrc:
  gen->xdp_errno = XDP_TooFewSources;
  return FALSE;

cannot:
  gen->xdp_errno = XDP_CannotNormalize;
  return FALSE;

fail:
  gen->xdp_errno = XDP_CallbackFailed;
  return FALSE;

shortcall:
  gen->xdp_errno = XDP_CallbackShort;
  return FALSE;
}

gint
xdp_control_depends_on_count (XdeltaControl *cont)
{
  return cont->source_info_len;
}

const guint8*
xdp_control_depends_on (XdeltaControl *cont, guint i)
{
  g_assert (i < cont->source_info_len);

  return cont->source_info[i]->md5;
}

const guint8*
xdp_control_depends_on_normalized (XdeltaControl *cont, guint i)
{
  g_assert (i < cont->source_info_len);

  return cont->source_info[i]->real_md5;
}

/* Rsync
 */

static XdeltaRsync*
xdp_rsync_index_int (XdeltaGenerator   *gen,
		     XdeltaStream      *str,
		     guint              seg_len)
{
  guint to_index = seg_len;
  XdeltaPos pos;
  XdeltaChecksum cksum;
  GArray    *index;

  index = g_array_new (FALSE, FALSE, sizeof (XdeltaRsyncElt));

  init_pos (gen, str, &pos);

  memset (&cksum, 0, sizeof (cksum));

  md5_init (md5_ctx);

  for (;;)
    {
      gint consume;

      if (! map_page (gen, str, &pos))
	return NULL;

      consume = MIN (to_index, pos.mem_rem - pos.off);

      if (consume == 0)
	break;

      to_index -= consume;

      md5_update (md5_ctx, pos.mem + pos.off, consume);
      init_long_checksum (pos.mem + pos.off, consume, &cksum);

      if (to_index == 0)
	{
	  XdeltaRsyncElt elt;

	  md5_final (elt.md5, md5_ctx);

	  elt.cksum = cksum;

	  g_array_append_val (index, elt);

	  md5_init (md5_ctx);
	  memset (&cksum, 0, sizeof (cksum));
	  to_index = seg_len;
	}

      pos.off += consume;

      FLIP_FORWARD (pos);
    }

  if (! unmap_page (gen, str, &pos))
    return NULL;

  {
    XdeltaRsync* rsync = g_new (XdeltaRsync, 1);

    rsync->seg_len = seg_len;
    rsync->file_len = handle_length (str);

    memcpy (rsync->file_md5, handle_checksum_md5 (str), 16);

    rsync->index = &g_array_index (index, XdeltaRsyncElt, 0);
    rsync->index_len = index->len;

    return rsync;
  }
}

static XdeltaRsync*
xdp_rsync_read_index (XdeltaGenerator* gen,
		      XdeltaStream*    cache_in)
{
  SerialSource* src = serializeio_handle_source (cache_in, &gen->handle_table);
  XdeltaRsync* rsync;

  if (! src)
    {
      gen->xdp_errno = XDP_StreamSerializeFailed;
      return NULL;
    }

  if (unserialize_rsyncindex (src, &rsync) != SerialSuccess)
    {
      gen->xdp_errno = XDP_InvalidRsyncCache;
      return NULL;
    }

  return rsync;
}

static gboolean
xdp_rsync_write_index (XdeltaGenerator* gen,
		       XdeltaRsync*     rsync,
		       XdeltaOutStream* cache_out)
{
  SerialSink* sink = serializeio_handle_sink (cache_out, &gen->handle_table);

  if (! sink)
    {
      gen->xdp_errno = XDP_StreamSerializeFailed;
      return NULL;
    }

  if (serialize_rsyncindex_obj (sink, rsync) != SerialSuccess)
    {
      gen->xdp_errno = XDP_StreamWriteFailed;
      return FALSE;
    }

  if (! handle_close (cache_out, 0))
    {
      gen->xdp_errno = XDP_StreamCloseFailed;
      return FALSE;
    }

  return TRUE;
}

XdeltaRsync*
xdp_rsync_index (XdeltaGenerator   *gen,
		 XdeltaStream      *str,
		 guint              seg_len,
		 XdeltaStream      *cache_in,
		 XdeltaOutStream   *cache_out)
{
  XdeltaRsync* rsync;

  if (cache_in)
    {
      if (! (rsync = xdp_rsync_read_index (gen, cache_in)))
	return NULL;

      if (seg_len != rsync->seg_len ||
	  (str && ! check_stream_integrity (gen, str, rsync->file_md5, rsync->file_len)))
	{
	  gen->xdp_errno = XDP_InvalidRsyncCache;
	  goto bail;
	}

      return rsync;
    }
  else
    {
      if (! (rsync = xdp_rsync_index_int (gen, str, seg_len)))
	return NULL;

      if (cache_out)
	{
	  if (! xdp_rsync_write_index (gen, rsync, cache_out))
	    goto bail;
	}

      return rsync;
    }

bail:

  xdp_rsync_index_free (rsync);

  return NULL;
}

void
xdp_rsync_index_free (XdeltaRsync *rsync)
{
  /* ??? */
}

static
gboolean xdp_rsync_hash (XdeltaRsync* rsync)
{
  guint i, index, prime;
  gboolean already_hashed = rsync->table != NULL;
  SerialRsyncIndexElt** table;

  if (! already_hashed)
    {
      prime = rsync->table_size = g_spaced_primes_closest (rsync->index_len);
      table = rsync->table = g_new0 (SerialRsyncIndexElt*, prime);
    }

  for (i = 0; i < rsync->index_len; i += 1)
    {
      SerialRsyncIndexElt* elt = rsync->index + i;

      elt->match_offset = -1;

      if (! already_hashed)
	{
	  index = c_hash (& elt->cksum) % prime;

	  elt->next = table[index];
	  table[index] = elt;
	}
    }

  return TRUE;
}

static void
incr_by (XdeltaPos* pos, gint incr)
{
  do
    {
      gint rem = MIN (incr, pos->mem_rem - pos->off);

      pos->off += incr;
      incr -= rem;
      FLIP_FORWARD (*pos);
    }
  while (incr > 0 && pos->mem_rem != pos->page_size);
}

GArray*
xdp_rsync_reqest (XdeltaGenerator   *gen,
		  XdeltaStream      *file,
		  XdeltaRsync       *rsync)
{
  XdeltaPos opos, npos;
  XdeltaChecksum cksum;
  guint max_buffer_index = handle_length (file);
  GArray *request = g_array_new (FALSE, FALSE, sizeof (guint));
  const guint8* n_pointer, *o_pointer;
  guint thistime;
  guint prime, index;
  SerialRsyncIndexElt **table;
  guint i;
  guint matched = 0;
  guint16 old_c, new_c;

  if (max_buffer_index < rsync->seg_len)
    return request;

  max_buffer_index -= rsync->seg_len;

  if (! xdp_rsync_hash (rsync))
    return NULL;

  g_assert (rsync->seg_len < handle_pagesize (file));

  init_pos (gen, file, &opos);
  init_pos (gen, file, &npos);
  memset (&cksum, 0, sizeof (cksum));

  prime = rsync->table_size;
  table = rsync->table;

  if (!map_page (gen, file, &opos))
    return NULL;

  init_long_checksum (opos.mem, rsync->seg_len, &cksum);

  npos.off += rsync->seg_len;

  for (; XPOS (opos) < max_buffer_index; )
    {
      if (!map_page (gen, file, &opos))
	return FALSE;

      if (!map_page (gen, file, &npos))
	return FALSE;

      if (matched == rsync->index_len)
	break;

      thistime = MIN (opos.mem_rem - opos.off,
		      npos.mem_rem - npos.off);

      o_pointer = opos.mem + opos.off;
      n_pointer = npos.mem + npos.off;

      for (; ; o_pointer += 1, n_pointer += 1)
	{
	  index = c_hash (&cksum) % prime;

	  if (table[index])
	    {
	      gboolean md5_computed = FALSE;
	      gboolean found = FALSE;
	      guint8 md5[16];
	      SerialRsyncIndexElt* elt;

	      for (elt = table[index]; elt; elt = elt->next)
		{
		  if (elt->match_offset >= 0)
		    continue;

		  if (elt->cksum.high != cksum.high ||
		      elt->cksum.low  != cksum.low)
		    continue;

		  if (! md5_computed)
		    {
		      md5_init (md5_ctx);

		      if (opos.page == npos.page)
			md5_update (md5_ctx, opos.mem + opos.off, rsync->seg_len);
		      else
			{
			  md5_update (md5_ctx, opos.mem + opos.off, opos.mem_rem - opos.off);
			  md5_update (md5_ctx, npos.mem, rsync->seg_len - (opos.mem_rem - opos.off));
			}

		      md5_final (md5, md5_ctx);

		      md5_computed = TRUE;
		    }

		  if (memcmp (md5, elt->md5, 16) == 0)
		    {
		      matched += 1;
		      found = TRUE;
		      elt->match_offset = XPOS (opos);
		    }
		}

	      if (found)
		{
		  incr_by (&opos, rsync->seg_len);
		  incr_by (&npos, rsync->seg_len);
		  goto reenter;
		}
	    }

	  if (thistime == 0)
	    goto nextpage;

	  thistime -= 1;
	  opos.off += 1;
	  npos.off += 1;

	  old_c = CHEW(*o_pointer);
	  new_c = CHEW(*n_pointer);

	  cksum.low -= old_c;
	  cksum.low += new_c;

	  cksum.high -= old_c * rsync->seg_len;
	  cksum.high += cksum.low;
	}

    nextpage:

      FLIP_FORWARD (opos);
      FLIP_FORWARD (npos);

    reenter:
    }

  for (i = 0; i < rsync->index_len; i += 1)
    {
      SerialRsyncIndexElt* elt = rsync->index + i;

      if (elt->match_offset < 0)
	{
#ifdef DEBUG_RSYNC_REQUEST
	  g_print ("request segment %d\n", i);
#endif
	  g_array_append_val (request, i);
	}
    }

  return request;
}

gboolean
xdp_apply_rsync_reply (XdeltaGenerator   *gen,
		       XdeltaRsync       *rsync,
		       XdeltaStream      *from,
		       XdeltaStream      *reply,
		       XdeltaStream      *out)
{
  gint i;
  guint reply_offset = 0;

  for (i = 0; i < rsync->index_len; i += 1)
    {
      SerialRsyncIndexElt* elt = rsync->index + i;

      if (elt->match_offset >= 0)
	{
	  if (! handle_copy (from, out, elt->match_offset, rsync->seg_len))
	    return FALSE;
	}
      else
	{
	  if (! handle_copy (reply, out, reply_offset, rsync->seg_len))
	    return FALSE;

	  reply_offset += rsync->seg_len;
	}
    }

  if (! handle_copy (reply, out, reply_offset, rsync->file_len % rsync->seg_len))
    return FALSE;

  if (! handle_close (out, 0))
    return FALSE;

  if (! check_stream_integrity (gen, out, rsync->file_md5, rsync->file_len))
    return FALSE;

  return TRUE;
}
