//<copyright>
//
// Copyright (c) 1995
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>
//
// Note: the code in this file is based on a sample program,
// see below for copyright notice of original source

//<file>
//
// Name:        remote.C
//
// Purpose:     VRweb remote calls
//
// Created:      6 Dec 1995   Michael Pichler
//
// Changed:      7 Dec 1995   Michael Pichler
//
// $Id: remote.C,v 1.7 1997/02/25 17:03:58 mpichler Exp $
//
//</file>



// implementation of remote calls to VRweb
//
// parts of this code are done in the same way as described for the
// Netscape API (http://home.netscape.com/newsref/std/x-remote.html)
//
// Copyright of the sample implementation follows
// (http://home.netscape.com/newsref/std/remote.c)
/*
 * Copyright  1995 Netscape Communications Corporation, all rights reserved.
 * Created: Jamie Zawinski <jwz@netscape.com>, 24-Dec-94.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 *
 */


#include "remote.h"

#include <hyperg/hyperg/message.h>
#include <hyperg/utils/verbose.h>

#include <InterViews/display.h>
#include <InterViews/enter-scope.h>
#include <InterViews/event.h>
#include <InterViews/window.h>

#include <IV-X11/xdisplay.h>
#include <IV-X11/xwindow.h>
#include <IV-X11/xevent.h>
extern "C" {
#include <IV-X11/Xdefs.h>
#  include <X11/Xlib.h>
#  include <X11/Xatom.h>
#  if defined(SUN4) || defined(SUN4_GNU)
/* if /usr/include/X11/Xmu/WinUtil.h does not provide a prototype */
     extern Window XmuClientWindow(Display*, Window);
#  else
#    include <X11/Xmu/WinUtil.h>	/* XmuClientWindow */
#  endif
#include <IV-X11/Xundefs.h>
}
// may have to include "vroot.h"

#include <string.h>
#include <iostream.h>


static Atom xaRemoteVersion = 0;
static Atom xaRemoteCommand = 0;


/*** helpers ***/

static void remoteInitAtoms (XDisplay* dpy)
{
  if (!xaRemoteVersion)
    xaRemoteVersion = XInternAtom (dpy, RemoteVersionProperty, False);
  if (!xaRemoteCommand)
    xaRemoteCommand = XInternAtom (dpy, RemoteCommandProperty, False);
}


// remoteFindWindow - find a remote window
// returns nonzero on success and fills out the window ID
// window is the client (application) window
// manwindow is the window managed by the window manager (child of root)

static int remoteFindWindow (DisplayRep* d, XWindow& window, XWindow& manwindow)
{
  remoteInitAtoms (d->display_);

  XWindow root = d->root_;

  window = 0;  // result
  manwindow = 0;

  XWindow root2, parent, *kids;
  unsigned int nkids;

  if (!XQueryTree (d->display_, d->root_, &root2, &parent, &kids, &nkids))
    return 0;

  if (root != root2 || parent || !kids || !nkids)
  { HgMessage::error ("XQueryTree failed");
    return 0;
  }

  for (unsigned i = 0;  i < nkids;  i++)
  {
    Atom type;
    int format;
    unsigned long nitems, bytesafter;
    unsigned char* version = 0;

    XWindow w = XmuClientWindow (d->display_, kids [i]);  // libXmu (misc. utils)

    int status = XGetWindowProperty (
      d->display_, w, xaRemoteVersion, 0, 1024, False /* do not delete */,
      XA_STRING, &type, &format, &nitems, &bytesafter, &version
    );

    if (!version)
      continue;
    DEBUGNL ("found remote version: " << version);
    XFree (version);
    if (status == Success && type != None)
    {
      window = w;
      manwindow = kids [i];
      return 1;  // found
    }

  } // for all children of root window

  DEBUGNL ("no remote VRweb window found");
  return 0;  // not found

} // remoteFindWindow


/*** remoteSetVersion ***/

void remoteSetVersion (DisplayRep* d, WindowRep* w, const char* version)
{
  remoteInitAtoms (d->display_);

  if (version)
    XChangeProperty (
      d->display_, w->xwindow_,
      xaRemoteVersion, XA_STRING, 8, PropModeReplace,
      (unsigned char*) version, strlen (version)
    );
  else
    XDeleteProperty (d->display_, w->xwindow_, xaRemoteVersion);
}


/*** remoteGetWindowID ***/

unsigned long remoteGetWindowID (DisplayRep* d)
{
  remoteInitAtoms (d->display_);

  XWindow win = 0, parent = 0;
  remoteFindWindow (d, win, parent);

  return (unsigned long) (parent);  // managed window
}


/*** remoteSendCommand ***/

int remoteSendCommand (DisplayRep* d, const char* command)
{
  // go through all windows on display and look whether a VRweb is among them
  // a window in "hold" state will not respond
  remoteInitAtoms (d->display_);

  XWindow win = 0, parent = 0;
  if (!remoteFindWindow (d, win, parent))
    return 0;

  DEBUGNL ("setting command " << command << "on window " << win);

  // send the command to the ready VRweb instance
  XChangeProperty (
    d->display_, win,
    xaRemoteCommand, XA_STRING, 8, PropModeReplace,
    (unsigned char*) command, strlen (command)
  );
  XFlush (d->display_);

  return 1;
}


/*** remoteRecieveCommand ***/

int remoteReceiveCommand (DisplayRep* d, WindowRep* w, const EventRep* e, RString& cmd)
{
  const XEvent& xe = e->xevent_;
  if (xe.type != PropertyNotify)  // only interested in PropertyNotify
    return 0;

  const XPropertyEvent& xev = xe.xproperty;
  if (xev.state == PropertyDelete)  // only interested in new/changed values
    return 0;

  remoteInitAtoms (d->display_);
  if (xev.atom != xaRemoteCommand)  // only interested in xaRemoteCommand
    return 0;

  Atom type;
  int format;
  unsigned long nitems, bytesafter;
  unsigned char* command;

  int status = XGetWindowProperty (
    d->display_, w->xwindow_, xaRemoteCommand, 0, 1024, True /* delete */,
    XA_STRING, &type, &format, &nitems, &bytesafter, &command
  );

  if (!command)
    return 0;

  if (status == Success && type != None)
  {
    cmd = RString ((const char*) command, nitems);
    XFree (command);
    return 1;  // successful
  }

  XFree (command);
  return 0;
}
