/*
 * IceWM
 *
 * Copyright (C) 1998 Marko Macek
 */

#include "icewm.h"

YApplication *app = 0;
YDesktop *desktop = 0;
XContext windowContext;

Atom _XA_WM_PROTOCOLS;
Atom _XA_WM_TAKE_FOCUS;
Atom _XA_WM_DELETE_WINDOW;
Atom _XA_WM_STATE;
Atom _XA_WM_CHANGE_STATE;
Atom _XATOM_MWM_HINTS;
//Atom _XA_MOTIF_WM_INFO;
Atom _XA_WM_COLORMAP_WINDOWS;

Atom _XA_WIN_PROTOCOLS;
Atom _XA_WIN_WORKSPACE;
Atom _XA_WIN_WORKSPACE_COUNT;
Atom _XA_WIN_WORKSPACE_NAMES;
Atom __XA_WIN_WORKSPACE_NAMES;
Atom _XA_WIN_WORKAREA;
Atom _XA_WIN_ICONS;
Atom _XA_WIN_STATE;
Atom _XA_WIN_LAYER;

Atom _XA_KWM_WIN_ICON;

Atom XA_XdndAware;
Atom XA_XdndEnter;
Atom XA_XdndLeave;
Atom XA_XdndPosition;
Atom XA_XdndStatus;
Atom XA_XdndDrop;
Atom XA_XdndFinished;

Atom XA_IcewmWinOptHint;

Colormap defaultColormap;

Cursor leftPointer;
Cursor rightPointer;
Cursor movePointer;

YColor *black = 0;
YColor *white = 0;

YColor *titleButtonBg = 0;
YColor *titleButtonFg = 0;

YColor *activeBorderBg = 0;
YColor *inactiveBorderBg = 0;

YColor *activeTitleBarBg = 0;
YColor *activeTitleBarFg = 0;

YColor *inactiveTitleBarBg = 0;
YColor *inactiveTitleBarFg = 0;

YColor *normalMinimizedWindowBg = 0;
YColor *normalMinimizedWindowFg = 0;

YColor *activeMinimizedWindowBg = 0;
YColor *activeMinimizedWindowFg = 0;

YColor *taskBarBg = 0;

GC blackGC;
GC whiteGC;
GC outlineGC;
GC clipPixmapGC;

YPixmap *startPixmap = 0;
YPixmap *windowsPixmap = 0;
YPixmap *mailPixmap = 0;
YPixmap *unreadMailPixmap = 0;
YPixmap *newMailPixmap = 0;


YPixmap *closePixmap[2] = { 0, 0 };
YPixmap *minimizePixmap[2] = { 0, 0 };
YPixmap *maximizePixmap[2] = { 0, 0 };
YPixmap *restorePixmap[2] = { 0, 0 };

YIcon *defaultAppIcon = 0;

#ifdef CONFIG_LOOK_PIXMAP
YPixmap *frameTL[2][2] = {{ 0, 0 }, { 0, 0 }};
YPixmap *frameT[2][2] = {{ 0, 0 }, { 0, 0 }};
YPixmap *frameTR[2][2] = {{ 0, 0 }, { 0, 0 }};
YPixmap *frameL[2][2] = {{ 0, 0 }, { 0, 0 }};
YPixmap *frameR[2][2] = {{ 0, 0 }, { 0, 0 }};
YPixmap *frameBL[2][2] = {{ 0, 0 }, { 0, 0 }};
YPixmap *frameB[2][2] = {{ 0, 0 }, { 0, 0 }};
YPixmap *frameBR[2][2] = {{ 0, 0 }, { 0, 0 }};

YPixmap *titleL[2] = { 0, 0 };
YPixmap *titleT[2] = { 0, 0 };
YPixmap *titleM[2] = { 0, 0 };
YPixmap *titleB[2] = { 0, 0 };
YPixmap *titleR[2] = { 0, 0 };

YPixmap *menuButton[2] = { 0, 0 };
#endif

YFont *titleFont = 0;
YFont *minimizedWindowFont = 0;
YFont *windowListFont = 0;

#ifdef SHAPE
int shapesSupported;
int shapeEventBase, shapeErrorBase;
#endif

#ifdef SM
int IceSMfd = -1;
IceConn IceSMconn = NULL;
SmcConn SMconn = NULL;
char *oldSessionId = NULL;
char *newSessionId = NULL;
char *sessionProg;

void iceWatchFD(IceConn conn,
                IcePointer /*client_data*/,
                Bool opening,
                IcePointer */*watch_data*/)
{
    if (opening) {
        if (IceSMfd != -1) { // shouldn't happen
            fprintf(stderr, "TOO MANY ICE CONNECTIONS -- not supported\n");
        } else {
            IceSMfd = IceConnectionNumber(conn);
        }
    } else {
        if (IceConnectionNumber(conn) == IceSMfd)
            IceSMfd = -1;
    }
}

void saveYourselfPhase2Proc(SmcConn conn, SmPointer /*client_data*/) {
    
    SmcSaveYourselfDone(conn, True);
}

void saveYourselfProc(SmcConn conn,
                      SmPointer /*client_data*/,
                      int /*save_type*/,
                      Bool /*shutdown*/,
                      int /*interact_style*/,
                      Bool /*fast*/)
{
    SmcRequestSaveYourselfPhase2(conn, &saveYourselfPhase2Proc, NULL);
}

void shutdownCancelledProc(SmcConn /*conn*/, SmPointer /*client_data*/) {
}

void saveCompleteProc(SmcConn /*conn*/, SmPointer /*client_data*/) {
}

void dieProc(SmcConn conn, SmPointer /*client_data*/) {
    SmcCloseConnection(conn, 0, NULL);
    if (conn == SMconn) {
        SMconn = NULL;
        IceSMconn = NULL;
    }
}

static void setSMProperties() {
    SmPropValue programVal = { 0, NULL };
    SmPropValue userIDVal = { 0, NULL };
    SmPropValue restartVal[3] = { { 0, NULL }, { 0, NULL }, { 0, NULL } };
    // broken headers in SMlib?
    SmProp programProp = { (char *)SmProgram, (char *)SmLISTofARRAY8, 1, &programVal };
    SmProp userIDProp = { (char *)SmUserID, (char *)SmARRAY8, 1, &userIDVal };
    SmProp restartProp = { (char *)SmRestartCommand, (char *)SmLISTofARRAY8, 3, (SmPropValue *)&restartVal };
    SmProp cloneProp = { (char *)SmCloneCommand, (char *)SmLISTofARRAY8, 2, (SmPropValue *)&restartVal };
    SmProp *props[] = {
        &programProp,
        &userIDProp,
        &restartProp,
        &cloneProp
    };

    char *user = getenv("USER");
    const char *clientId = "-clientId";

    programVal.length = strlen(sessionProg);
    programVal.value = sessionProg;
    userIDVal.length = strlen(user);
    userIDVal.value = (SmPointer)user;
    restartVal[0].length = strlen(sessionProg);
    restartVal[0].value = sessionProg;
    restartVal[1].length = strlen(clientId);
    restartVal[1].value = (char *)clientId;
    restartVal[2].length = strlen(newSessionId);
    restartVal[2].value = newSessionId;

    SmcSetProperties(SMconn,
                     sizeof(props)/sizeof(props[0]),
                     (SmProp **)&props);
}

static void initSM() {
    if (getenv("SESSION_MANAGER") == 0)
        return;
    if (IceAddConnectionWatch(&iceWatchFD, NULL) == 0) {
        fprintf(stderr, "IceAddConnectionWatch failed.");
        return ;
    }

    char error_str[256];
    SmcCallbacks smcall;

    memset(&smcall, 0, sizeof(smcall));
    smcall.save_yourself.callback = &saveYourselfProc;
    smcall.save_yourself.client_data = NULL;
    smcall.die.callback = &dieProc;
    smcall.die.client_data = NULL;
    smcall.save_complete.callback = &saveCompleteProc;
    smcall.save_complete.client_data = NULL;
    smcall.shutdown_cancelled.callback = &shutdownCancelledProc;
    smcall.shutdown_cancelled.client_data = NULL;
    
    if ((SMconn = SmcOpenConnection(NULL, /* network ids */
                                    NULL, /* context */
                                    1, 0, /* protocol major, minor */
                                    SmcSaveYourselfProcMask |
                                    SmcSaveCompleteProcMask |
                                    SmcShutdownCancelledProcMask |
                                    SmcDieProcMask,
                                    &smcall,
                                    oldSessionId, &newSessionId,
                                    sizeof(error_str), error_str)) == NULL)
    {
        fprintf(stderr, "session mgr init error: %s\n", error_str);
        return ;
    }
    IceSMconn = SmcGetIceConnection(SMconn);

    setSMProperties();
}
#endif

static void initAtoms() {
    _XA_WM_PROTOCOLS = XInternAtom(app->display(), "WM_PROTOCOLS", False);
    _XA_WM_TAKE_FOCUS = XInternAtom(app->display(), "WM_TAKE_FOCUS", False);
    _XA_WM_DELETE_WINDOW = XInternAtom(app->display(), "WM_DELETE_WINDOW", False);
    _XA_WM_STATE = XInternAtom(app->display(), "WM_STATE", False);
    _XA_WM_CHANGE_STATE = XInternAtom(app->display(), "WM_CHANGE_STATE", False);
    _XA_WM_COLORMAP_WINDOWS = XInternAtom(app->display(), "WM_COLORMAP_WINDOWS", False);
    _XATOM_MWM_HINTS = XInternAtom(app->display(), _XA_MOTIF_WM_HINTS, False);
    //_XA_MOTIF_WM_INFO = XInternAtom(app->display(), "_MOTIF_WM_INFO", False);
    _XA_KWM_WIN_ICON = XInternAtom(app->display(), "KWM_WIN_ICON", False);
    _XA_WIN_WORKSPACE = XInternAtom(app->display(), XA_WIN_WORKSPACE, False);
    _XA_WIN_WORKSPACE_COUNT = XInternAtom(app->display(), XA_WIN_WORKSPACE_COUNT, False);
    _XA_WIN_WORKSPACE_NAMES = XInternAtom(app->display(), XA_WIN_WORKSPACE_NAMES, False);
    __XA_WIN_WORKSPACE_NAMES = XInternAtom(app->display(), "_" XA_WIN_WORKSPACE_NAMES, False);
    _XA_WIN_WORKAREA = XInternAtom(app->display(), XA_WIN_WORKAREA, False);
    _XA_WIN_ICONS = XInternAtom(app->display(), XA_WIN_ICONS, False);
    _XA_WIN_LAYER = XInternAtom(app->display(), XA_WIN_LAYER, False);
    _XA_WIN_STATE = XInternAtom(app->display(), XA_WIN_STATE, False);
    _XA_WIN_PROTOCOLS = XInternAtom(app->display(), XA_WIN_PROTOCOLS, False);

    XA_XdndAware = XInternAtom(app->display(), "XdndAware", False);
    
    XA_XdndEnter = XInternAtom(app->display(), "XdndEnter", False);
    XA_XdndLeave = XInternAtom(app->display(), "XdndLeave", False);
    XA_XdndPosition = XInternAtom(app->display(), "XdndPosition", False);
    XA_XdndStatus = XInternAtom(app->display(), "XdndStatus", False);
    XA_XdndDrop = XInternAtom(app->display(), "XdndDrop", False);
    XA_XdndFinished = XInternAtom(app->display(), "XdndFinished", False);

    XA_IcewmWinOptHint = XInternAtom(app->display(), "_ICEWM_WINOPTHINT", False);
}

static void createGC(GC &gc, YColor fore, YColor back, YFont *font = 0) {
    XGCValues gcv;

    gcv.foreground = fore.pixel();
    gcv.background = back.pixel();

#ifndef I18N
    if (font)
        gcv.font = font->getFontStruct()->fid;
#else
    font = 0;
#endif
    
    gc = XCreateGC(app->display(),
                   desktop->handle(),
                   GCForeground | GCBackground | (font ? GCFont : 0),
                   &gcv);
}

static void initPointers() {
    leftPointer = XCreateFontCursor(app->display(), XC_left_ptr);
    rightPointer = XCreateFontCursor(app->display(), XC_right_ptr);
    movePointer = XCreateFontCursor(app->display(), XC_fleur);
}

static void initGCs() {
    menuFont = new YFont(menuFontName);
#ifdef CONFIG_WINLIST
    windowListFont = new YFont(windowListFontName);
#endif

    black = new YColor(clrBlack);
    white = new YColor(clrWhite);

    dialogBg = new YColor(clrDialog);

    menuBg = new YColor(clrNormalMenu);
    menuItemFg = new YColor(clrNormalMenuItemText);
    activeMenuItemBg = new YColor(clrActiveMenuItem);
    activeMenuItemFg = new YColor(clrActiveMenuItemText);
    disabledMenuItemFg = new YColor(clrDisabledMenuItemText);

#ifdef CONFIG_WINLIST
    scrollBarBg = new YColor(clrScrollBar);
    scrollBarArrow = new YColor(clrScrollBarArrow);
    scrollBarSlider= new YColor(clrScrollBarSlider);

    listBoxBg = new YColor(clrListBox);
    listBoxFg = new YColor(clrListBoxText);
    listBoxSelBg = new YColor(clrListBoxSelected);
    listBoxSelFg = new YColor(clrListBoxSelectedText);
#endif

    createGC(blackGC, YColor(clrBlack), YColor(clrBlack));
    createGC(whiteGC, YColor(clrWhite), YColor(clrWhite));

    {
        XGCValues gcv;

        gcv.foreground = YColor(clrActiveBorder).pixel();
        gcv.line_width = (wsBorderX + wsBorderY) / 2;
        gcv.subwindow_mode = IncludeInferiors;
        gcv.function = GXxor;

        outlineGC = XCreateGC(app->display(), desktop->handle(),
                              GCFunction | GCLineWidth |
                              GCForeground | GCSubwindowMode,
                              &gcv);

        clipPixmapGC = XCreateGC(app->display(), desktop->handle(),
                                 0, &gcv);
    }
}

void joinPath(char *path, pathelem *pe, const char *base, const char *name) {
    /// !!! this could be optimized
    strcpy(path, *pe->root);
    strcat(path, pe->rdir);
    if (pe->sub) {
        strcat(path, *pe->sub);
        strcat(path, "/");
    }
    if (base)
        strcat(path, base);
    strcat(path, name);
}

void verifyPaths(pathelem *search, const char *base) {
    unsigned int j = 0, i = 0;

    for (; search[i].root; i++) {
        char path[1024]; /// FIX
        
        joinPath(path, search + i, base, "");
        if (access(path, R_OK) == 0)
            search[j++] = search[i];
    }
    search[j] = search[i];
}

void loadPixmap(pathelem *pe, const char *base, const char *name, YPixmap **pixmap) {
    char path[1024]; /// FIX

    *pixmap = 0;

    for (; pe->root; pe++) {
        joinPath(path, pe, base, name);

        if (is_reg(path)) {
            *pixmap = new YPixmap(path);

            if (pixmap == 0)
                die(1, "out of memory for pixmap %s", path);

            return ;
        }
    }
    warn("could not find pixmap %s", name);
}


pathelem icon_paths[10];

static void initPixmaps() {
    static const char *home = getenv("HOME");
    static char themeSubdir[256];
    const char *base = 0;

    strcpy(themeSubdir, themeName);
    { char *p = strchr(themeSubdir, '/'); if (p) *p = 0; }

    static const char *themeDir = themeSubdir;
 
    pathelem tpaths[] = {
        { &home, "/.icewm/themes/", &themeDir },
        { &configDir, "/themes/", &themeDir },
        { &libDir, "/themes/", &themeDir },
        { 0, 0, 0 }
    };

    verifyPaths(tpaths, 0);
        
#ifdef CONFIG_LOOK_PIXMAP
    if (wmLook == lookPixmap || wmLook == lookMetal || wmLook == lookGtk) {
        loadPixmap(tpaths, 0, "closeI.xpm", &closePixmap[0]);
        loadPixmap(tpaths, 0, "maximizeI.xpm", &maximizePixmap[0]);
        loadPixmap(tpaths, 0, "minimizeI.xpm", &minimizePixmap[0]);
        loadPixmap(tpaths, 0, "restoreI.xpm", &restorePixmap[0]);
        loadPixmap(tpaths, 0, "closeA.xpm", &closePixmap[1]);
        loadPixmap(tpaths, 0, "maximizeA.xpm", &maximizePixmap[1]);
        loadPixmap(tpaths, 0, "minimizeA.xpm", &minimizePixmap[1]);
        loadPixmap(tpaths, 0, "restoreA.xpm", &restorePixmap[1]);
        loadPixmap(tpaths, 0, "frameITL.xpm", &frameTL[0][0]);
        loadPixmap(tpaths, 0, "frameIT.xpm",  &frameT[0][0]);
        loadPixmap(tpaths, 0, "frameITR.xpm", &frameTR[0][0]);
        loadPixmap(tpaths, 0, "frameIL.xpm",  &frameL[0][0]);
        loadPixmap(tpaths, 0, "frameIR.xpm",  &frameR[0][0]);
        loadPixmap(tpaths, 0, "frameIBL.xpm", &frameBL[0][0]);
        loadPixmap(tpaths, 0, "frameIB.xpm",  &frameB[0][0]);
        loadPixmap(tpaths, 0, "frameIBR.xpm", &frameBR[0][0]);
        loadPixmap(tpaths, 0, "frameATL.xpm", &frameTL[0][1]);
        loadPixmap(tpaths, 0, "frameAT.xpm",  &frameT[0][1]);
        loadPixmap(tpaths, 0, "frameATR.xpm", &frameTR[0][1]);
        loadPixmap(tpaths, 0, "frameAL.xpm",  &frameL[0][1]);
        loadPixmap(tpaths, 0, "frameAR.xpm",  &frameR[0][1]);
        loadPixmap(tpaths, 0, "frameABL.xpm", &frameBL[0][1]);
        loadPixmap(tpaths, 0, "frameAB.xpm",  &frameB[0][1]);
        loadPixmap(tpaths, 0, "frameABR.xpm", &frameBR[0][1]);
        
        loadPixmap(tpaths, 0, "dframeITL.xpm", &frameTL[1][0]);
        loadPixmap(tpaths, 0, "dframeIT.xpm",  &frameT[1][0]);
        loadPixmap(tpaths, 0, "dframeITR.xpm", &frameTR[1][0]);
        loadPixmap(tpaths, 0, "dframeIL.xpm",  &frameL[1][0]);
        loadPixmap(tpaths, 0, "dframeIR.xpm",  &frameR[1][0]);
        loadPixmap(tpaths, 0, "dframeIBL.xpm", &frameBL[1][0]);
        loadPixmap(tpaths, 0, "dframeIB.xpm",  &frameB[1][0]);
        loadPixmap(tpaths, 0, "dframeIBR.xpm", &frameBR[1][0]);
        loadPixmap(tpaths, 0, "dframeATL.xpm", &frameTL[1][1]);
        loadPixmap(tpaths, 0, "dframeAT.xpm",  &frameT[1][1]);
        loadPixmap(tpaths, 0, "dframeATR.xpm", &frameTR[1][1]);
        loadPixmap(tpaths, 0, "dframeAL.xpm",  &frameL[1][1]);
        loadPixmap(tpaths, 0, "dframeAR.xpm",  &frameR[1][1]);
        loadPixmap(tpaths, 0, "dframeABL.xpm", &frameBL[1][1]);
        loadPixmap(tpaths, 0, "dframeAB.xpm",  &frameB[1][1]);
        loadPixmap(tpaths, 0, "dframeABR.xpm", &frameBR[1][1]);
        
        loadPixmap(tpaths, 0, "titleIL.xpm", &titleL[0]);
        loadPixmap(tpaths, 0, "titleIT.xpm", &titleT[0]);
        loadPixmap(tpaths, 0, "titleIM.xpm", &titleM[0]);
        loadPixmap(tpaths, 0, "titleIB.xpm", &titleB[0]);
        loadPixmap(tpaths, 0, "titleIR.xpm", &titleR[0]);
        loadPixmap(tpaths, 0, "titleAL.xpm", &titleL[1]);
        loadPixmap(tpaths, 0, "titleAT.xpm", &titleT[1]);
        loadPixmap(tpaths, 0, "titleAM.xpm", &titleM[1]);
        loadPixmap(tpaths, 0, "titleAB.xpm", &titleB[1]);
        loadPixmap(tpaths, 0, "titleAR.xpm", &titleR[1]);
        
        loadPixmap(tpaths, 0, "menuButtonI.xpm", &menuButton[0]);
        loadPixmap(tpaths, 0, "menuButtonA.xpm", &menuButton[1]);
    } else
#endif
    {
        loadPixmap(tpaths, 0, "close.xpm", &closePixmap[0]);
        loadPixmap(tpaths, 0, "maximize.xpm", &maximizePixmap[0]);
        loadPixmap(tpaths, 0, "minimize.xpm", &minimizePixmap[0]);
        loadPixmap(tpaths, 0, "restore.xpm", &restorePixmap[0]);
    }

    defaultAppIcon = getIcon("app");
    
    pathelem paths[] = {
        { &home, "/.icewm/themes/", &themeDir },
        { &home, "/.icewm/", 0,},
        { &configDir, "/themes/", &themeDir },
        { &configDir, "/", 0 },
        { &libDir, "/themes/", &themeDir },
        { &libDir, "/", 0 },
        { 0, 0, 0 }
    };
    pathelem cpaths[sizeof(paths)/sizeof(paths[0])];
    verifyPaths(paths, 0);

    base = "taskbar/";
    memcpy(cpaths, paths, sizeof(cpaths));
    verifyPaths(cpaths, base);
    loadPixmap(cpaths, base, START_PIXMAP, &startPixmap);
    loadPixmap(cpaths, base, "windows.xpm", &windowsPixmap);

    base = "mailbox/";
    memcpy(cpaths, paths, sizeof(cpaths));
    verifyPaths(cpaths, base);
    loadPixmap(cpaths, base, "mail.xpm", &mailPixmap);
    loadPixmap(cpaths, base, "unreadmail.xpm", &unreadMailPixmap);
    loadPixmap(cpaths, base, "newmail.xpm", &newMailPixmap);
    
    base = "ledclock/";
    memcpy(cpaths, paths, sizeof(cpaths));
    verifyPaths(cpaths, base);
    loadPixmap(cpaths, base, "n0.xpm", &PixNum[0]);
    loadPixmap(cpaths, base, "n1.xpm", &PixNum[1]);
    loadPixmap(cpaths, base, "n2.xpm", &PixNum[2]);
    loadPixmap(cpaths, base, "n3.xpm", &PixNum[3]);
    loadPixmap(cpaths, base, "n4.xpm", &PixNum[4]);
    loadPixmap(cpaths, base, "n5.xpm", &PixNum[5]);
    loadPixmap(cpaths, base, "n6.xpm", &PixNum[6]);
    loadPixmap(cpaths, base, "n7.xpm", &PixNum[7]);
    loadPixmap(cpaths, base, "n8.xpm", &PixNum[8]);
    loadPixmap(cpaths, base, "n9.xpm", &PixNum[9]);
    loadPixmap(cpaths, base, "space.xpm", &PixSpace);
    loadPixmap(cpaths, base, "colon.xpm", &PixColon);
    loadPixmap(cpaths, base, "slash.xpm", &PixSlash);
    loadPixmap(cpaths, base, "dot.xpm", &PixDot);
    loadPixmap(cpaths, base, "a.xpm", &PixA);
    loadPixmap(cpaths, base, "p.xpm", &PixP);
    loadPixmap(cpaths, base, "m.xpm", &PixM);

    pathelem i_paths[10] = {
        { &home, "/.icewm/themes/", &themeDir },
        { &home, "/.icewm/", 0,},
        { &configDir, "/themes/", &themeDir },
        { &configDir, "/", 0 },
        { &libDir, "/themes/", &themeDir },
        { &libDir, "/", 0 },
        { 0, 0, 0 }
    };

    base = "icons/";
    verifyPaths(i_paths, base);
    memcpy(icon_paths, i_paths, sizeof(icon_paths));
}

YApplication::YApplication(int *argc, char ***argv, const char *displayName) {
    app = this;
    fLoopLevel = 0;
    fExitApp = 0;
    lastEventTime = CurrentTime;
    fGrabWindow = 0;
    fGrabTree = 0;
    fXGrabWindow = 0;
    fGrabMouse = 0;
    fPopup = 0;
    fFirstTimer = fLastTimer = 0;

    bool sync = false;

    for (int i = 1; i < *argc; i++) {
        if ((*argv)[i][0] == '-') {
            if (strcmp((*argv)[i], "-display") == 0) {
                displayName = (*argv)[++i];
#ifdef SM
            } else if (strcmp((*argv)[i], "-clientId") == 0) {
                oldSessionId = (*argv)[++i];
#endif
            } else if (strcmp((*argv)[i], "-sync") == 0) {
                sync = true;
            }
        }
    }

    if (displayName == 0)
        displayName = getenv("DISPLAY");
    else {
        static char disp[256] = "DISPLAY=";
        strcat(disp, displayName);
        putenv(disp);
    }

    if (!(fDisplay = XOpenDisplay(displayName)))
        die(1, "icewm: Can't open display: %s", displayName ? displayName : "<none>");

    if (sync)
        XSynchronize(display(), True);
    
    defaultColormap = DefaultColormap(display(), DefaultScreen(display()));

    DBG XSynchronize(display(), True);

    windowContext = XUniqueContext();

    initAtoms();
    initPointers();
    initModifiers();

#ifdef SHAPE
    shapesSupported = XShapeQueryExtension(display(),
                                           &shapeEventBase, &shapeErrorBase);
#endif

    desktop = new YDesktop(0, RootWindow(display(),
                                         DefaultScreen(display())));
    
    initGCs();
    initPixmaps();

#ifdef SM
    sessionProg = (*argv)[0]; //"icewm";
#endif

#ifdef SM
    initSM();
#endif

    struct sigaction sig;
    sig.sa_handler = SIG_IGN;
    sigemptyset(&sig.sa_mask);
    sig.sa_flags = 0;
    sigaction(SIGCHLD, &sig, &oldSignalCHLD);
    
    sigset_t mask;
    sigemptyset(&mask);
    sigprocmask(SIG_BLOCK, &mask, &oldSignalMask);
}

YApplication::~YApplication() {
    XCloseDisplay(display());
    app = 0;
}

void YApplication::registerTimer(YTimer *t) {
    t->fPrev = 0;
    t->fNext = fFirstTimer;
    if (fFirstTimer)
        fFirstTimer->fPrev = t;
    fFirstTimer = t;
}

void YApplication::unregisterTimer(YTimer *t) {
    if (t->fPrev)
        t->fPrev->fNext = t->fNext;
    else
        fFirstTimer = t->fNext;
    if (t->fNext)
        t->fNext->fPrev = t->fPrev;
    else
        fLastTimer = t->fPrev;
}

void YApplication::getTimeout(struct timeval *timeout) {
    YTimer *t;
    struct timeval curtime;
    
    gettimeofday(&curtime, 0);
    timeout->tv_sec += curtime.tv_sec;
    timeout->tv_usec += curtime.tv_usec;
    while (timeout->tv_usec >= 1000000) {
        timeout->tv_usec -= 1000000;
        timeout->tv_sec++;
    }
    
    t = fFirstTimer;
    while (t) {
        if (t->isRunning() && timercmp(timeout, &t->timeout, >))
            *timeout = t->timeout;
        t = t->fNext;
    }
    if (timercmp(&curtime, timeout, >)) {
        timeout->tv_sec = 0;
        timeout->tv_usec = 1;
    } else {
        timeout->tv_sec -= curtime.tv_sec;
        timeout->tv_usec -= curtime.tv_usec;
        while (timeout->tv_usec < 0) {
            timeout->tv_usec += 1000000;
            timeout->tv_sec--;
        }
    } 
    //printf("set: %d %d\n", timeout->tv_sec, timeout->tv_usec);
    assert(timeout->tv_sec >= 0);
    assert(timeout->tv_usec >= 0);
}

void YApplication::handleTimeouts() {
    YTimer *t;
    struct timeval curtime;
    gettimeofday(&curtime, 0);

    t = fFirstTimer;
    while (t) {
        if (t->isRunning() && timercmp(&curtime, &t->timeout, >)) {
            TimerListener *l = t->getListener();
            t->fRunning = false;
            if (l && l->handleTimer(t))
                t->startTimer();
        }
        t = t->fNext;
    }
}

int YApplication::mainLoop() {
    fLoopLevel++;
    fExitLoop = 0;

    struct timeval timeout;

    while (!fExitApp && !fExitLoop) {
        if (XPending(display()) > 0) {
            XEvent xev;

            XNextEvent(display(), &xev);
            
            saveEventTime(xev);
            
#ifdef DEBUG
            DBG logEvent(xev);
#endif
            
            if (xev.type == KeymapNotify) {
                XRefreshKeyboardMapping(&xev.xmapping);

                // ??? this should be ok
                // !!! but we should probably regrab everything ?
                initModifiers();
            } else {
                YWindow *window = 0;
                int rc = 0;
                int ge = (xev.type == ButtonPress ||
                          xev.type == ButtonRelease ||
                          xev.type == MotionNotify ||
                          xev.type == KeyPress ||
                          xev.type == KeyRelease ||
                          xev.type == EnterNotify ||
                          xev.type == LeaveNotify) ? 1 : 0;

                if (fPopup && ge) {
                    handleGrabEvent(fPopup, xev);
                } else if (fGrabWindow && ge) {
                    handleGrabEvent(fGrabWindow, xev);
                } else if ((rc = XFindContext(display(),
                                        xev.xany.window,
                                        windowContext,
                                        (XPointer *)&window)) == 0)
                {
                    window->handleEvent(xev);
                } else {
                    if (xev.type == MapRequest) {
                        // !!! java seems to do this ugliness
                        //YFrameWindow *f = getFrame(xev.xany.window);
                        fprintf(stderr,
                                "map request sent to destroyed frame! BUG? parent=%lX, window=%lX\n",
                                xev.xmaprequest.parent,
                                xev.xmaprequest.window);
                        desktop->handleEvent(xev);
                    } else if (xev.type != DestroyNotify) {
                        MSG(("unknown window 0x%lX event=%d", xev.xany.window, xev.type));
                    }
                }
            }
        } else {
            int rc;
            fd_set read_fds;

            handleIdle();

            FD_ZERO(&read_fds);
            FD_SET(ConnectionNumber(app->display()), &read_fds);
#ifdef SM
            if (IceSMfd != -1)
                FD_SET(IceSMfd, &read_fds);
#endif

            timeout.tv_sec = 0;
            timeout.tv_usec = 1000 * 1000;
            getTimeout(&timeout);
            //printf("timeout: %d %d\n", timeout.tv_sec, timeout.tv_usec);

#if defined(HPUX9)
            rc = select(sizeof(fd_set), (int *)&read_fds, 0, 0, &timeout);
#else
            rc = select(sizeof(fd_set), &read_fds, 0, 0, &timeout);
#endif

            {   // reap children on systems where SIGCHLD=SIG_IGN doesn't
                int s; 
                waitpid(-1, &s, WNOHANG); 
            }

            sigset_t mask;
            sigpending(&mask);
            if (sigismember(&mask, SIGINT))
                handleSignal(SIGINT);
            if (sigismember(&mask, SIGTERM))
                handleSignal(SIGTERM);
            if (sigismember(&mask, SIGHUP))
                handleSignal(SIGHUP);
            if (rc == 0) {
                handleTimeouts();
            } else {
                if (rc == -1) {
                    //fprintf(stderr, "select errno=%d\n", errno);
                    assert(rc != -1);
                }
            }

#ifdef SM
            if (IceSMfd != -1 && FD_ISSET(IceSMfd, &read_fds)) {
                Bool rep;
                if (IceProcessMessages(IceSMconn, NULL, &rep)
                    == IceProcessMessagesIOError)
                {
                    SmcCloseConnection(SMconn, 0, NULL);
                    IceSMconn = NULL;
                }
            }
#endif
        }
    }
    fLoopLevel--;
    return fExitCode;
}

void YApplication::dispatchEvent(YWindow *win, XEvent &xev) {
    if (xev.type == KeyPress || xev.type == KeyRelease) {
        YWindow *w = win;
        while (w && (w->handleKey(xev.xkey) == false)) {
            if (fGrabTree && w == fXGrabWindow)
                break;
            w = w->parent();
        }
    } else {
        Window child;
        
        if (xev.type == MotionNotify) {
            if (xev.xmotion.window != win->handle())
                if (XTranslateCoordinates(app->display(),
                                          xev.xany.window, win->handle(),
                                          xev.xmotion.x, xev.xmotion.y,
                                          &xev.xmotion.x, &xev.xmotion.y, &child) == True)
                    xev.xmotion.window = win->handle();
                else
                    return ;
        } else if (xev.type == ButtonPress || xev.type == ButtonRelease ||
                   xev.type == EnterNotify || xev.type == LeaveNotify)
        {
            if (xev.xbutton.window != win->handle())
                if (XTranslateCoordinates(app->display(),
                                          xev.xany.window, win->handle(),
                                          xev.xbutton.x, xev.xbutton.y,
                                          &xev.xbutton.x, &xev.xbutton.y, &child) == True)
                    xev.xbutton.window = win->handle();
                else
                    return ;
        } else if (xev.type == KeyPress || xev.type == KeyRelease) {
            if (xev.xkey.window != win->handle())
                if (XTranslateCoordinates(app->display(),
                                          xev.xany.window, win->handle(),
                                          xev.xkey.x, xev.xkey.y,
                                          &xev.xkey.x, &xev.xkey.y, &child) == True)
                    xev.xkey.window = win->handle();
                else
                    return ;
        }
        win->handleEvent(xev);
    }
}

void YApplication::handleGrabEvent(YWindow *winx, XEvent &xev) {
    YWindow *win = winx;

    assert(win != 0);
    if (fGrabTree) {
        if (xev.xbutton.subwindow != None) {
            if (XFindContext(display(),
                         xev.xbutton.subwindow,
                         windowContext,
                         (XPointer *)&win) != 0);
                if (xev.type == EnterNotify || xev.type == LeaveNotify)
                    win = 0;
                else
                    win = fGrabWindow;
        } else {
            if (XFindContext(display(),
                         xev.xbutton.window,
                         windowContext,
                         (XPointer *)&win) != 0) 
                if (xev.type == EnterNotify || xev.type == LeaveNotify)
                    win = 0;
                else
                    win = fGrabWindow;
        }
        if (win == 0)
            return ;
        {
            YWindow *p = win;
            while (p) {
                if (p == fXGrabWindow)
                    break;
                p = p->parent();
            }
            if (p == 0) {
                if (xev.type == EnterNotify || xev.type == LeaveNotify)
                    return ; 
                else
                    win = fGrabWindow;
            }
        }
        if (xev.type == EnterNotify || xev.type == LeaveNotify)
            if (win != fGrabWindow)
                return ;
        if (fGrabWindow != fXGrabWindow)
            win = fGrabWindow;
    }
    dispatchEvent(win, xev);
}

void YApplication::captureGrabEvents(YWindow *win) {
    if (fGrabWindow == fXGrabWindow && fGrabTree) {
        fGrabWindow = win;
        //puts("grabbed");
    }
}

void YApplication::releaseGrabEvents(YWindow *win) {
    if (win == fGrabWindow && fGrabTree) {
        fGrabWindow = fXGrabWindow;
        //puts("released");
    }
}

int YApplication::grabEvents(YWindow *win, Cursor ptr, unsigned int eventMask, int grabMouse, int grabKeyboard, int grabTree) {
    int rc;
    
    if (fGrabWindow != 0)
        return 0;

    XSync(display(), 0);
    fGrabTree = grabTree;
    if (grabMouse) {
        fGrabMouse = 1;
        rc = XGrabPointer(display(), win->handle(),
                          grabTree ? True : False,
                          eventMask,
                          GrabModeAsync, GrabModeAsync,
                          None, ptr, CurrentTime);
        
        if (rc != Success) {
            MSG(("grab status = %d\x7", rc));
            return 0;
        }
    } else {
        fGrabMouse = 0;

        XChangeActivePointerGrab(display(),
                                 eventMask,
                                 ptr, CurrentTime);
    }

    if (grabKeyboard) {
        rc = XGrabKeyboard(display(), win->handle(),
                           ///False,
                           grabTree ? True : False,
                           GrabModeAsync, GrabModeAsync, CurrentTime);
        if (rc != Success) {
            MSG(("grab status = %d\x7", rc));
            XUngrabPointer(display(), CurrentTime);
            return 0;
        }
        
        XSetInputFocus(display(),
                       win->handle(),
                       RevertToPointerRoot,
                       CurrentTime);
    }

    desktop->resetColormap(false);

    fXGrabWindow = win;
    fGrabWindow = win;
    return 1;
}

int YApplication::releaseEvents() {
    if (fGrabWindow == 0)
        return 0;
    fGrabWindow = 0;
    fXGrabWindow = 0;
    fGrabTree = 0;
    if (fGrabMouse) {
        XUngrabPointer(display(), CurrentTime);
        fGrabMouse = 0;
    }
    XUngrabKeyboard(display(), CurrentTime);
    desktop->resetInputFocus();
    desktop->resetColormap(true);
    return 1;
}

void YApplication::exitLoop(int exitCode) {
    fExitLoop = 1;
    fExitCode = exitCode;
}

void YApplication::exit(int exitCode) {
    fExitApp = 1;
    exitLoop(exitCode);
}

void YApplication::saveEventTime(XEvent &xev) {
    //lastEventTime = CurrentTime;
    //return ;
    switch (xev.type) {
    case ButtonPress:
    case ButtonRelease:
        lastEventTime = xev.xbutton.time;
        break;
        
    case MotionNotify:
        lastEventTime = xev.xmotion.time;
        break;
        
    case KeyPress:
    case KeyRelease:
        lastEventTime = xev.xkey.time;
        break;
        
    case EnterNotify:
    case LeaveNotify:
        lastEventTime = xev.xcrossing.time;
        break;
        
    case PropertyNotify:
        lastEventTime = xev.xproperty.time;
        break;
        
    case SelectionClear:
        lastEventTime = xev.xselectionclear.time;
        break;
        
    case SelectionRequest:
        lastEventTime = xev.xselectionrequest.time;
        break;
        
    case SelectionNotify:
        lastEventTime = xev.xselection.time;
        break;
    default:
        lastEventTime = CurrentTime;
        break;
    }
}

void YApplication::handleSignal(int /*sig*/) {
}

void YApplication::handleIdle() {
}

void YApplication::blockSignal(int sig) {
    sigset_t mask, old_mask;
    sigemptyset(&mask);
    sigaddset(&mask, sig);
    sigprocmask(SIG_BLOCK, &mask, &old_mask);
}

void YApplication::resetSignals() {
    sigset_t mask;
    struct sigaction old_sig;
    
    sigaction(SIGCHLD, &oldSignalCHLD, &old_sig);
    sigprocmask(SIG_SETMASK, &oldSignalMask, &mask);
}

void YApplication::initModifiers() {
    XModifierKeymap *xmk = XGetModifierMapping(app->display());
    if (xmk) {
        KeyCode numLockKeyCode = XKeysymToKeycode(app->display(), XK_Num_Lock);
        KeyCode scrollLockKeyCode = XKeysymToKeycode(app->display(), XK_Scroll_Lock);
        KeyCode altKeyCode = XKeysymToKeycode(app->display(), XK_Alt_L);
        KeyCode metaKeyCode = XKeysymToKeycode(app->display(), XK_Meta_L);

        int m, k;
        KeyCode *c = xmk->modifiermap;

        for (m = 0; m < 8; m++)
            for (k = 0; k < xmk->max_keypermod; k++) {
                if (*c == numLockKeyCode)
                    NumLockMask = (1 << m);
                if (*c == scrollLockKeyCode)
                    ScrollLockMask = (1 << m);
                if (*c == altKeyCode)
                    AltMask = (1 << m);
                if (*c == metaKeyCode)
                    MetaMask = (1 << m);
                c++;
            }
        XFreeModifiermap(xmk); // !!! x bug, should be ...Map
    }
    //fprintf(stderr, "alt:%d meta:%d num:%d scroll:%d\n",
    //        AltMask,
    //        MetaMask,
    //        NumLockMask,
    //        ScrollLockMask);

    // some hacks for "broken" modifier configurations

#if 1
    // this basically does what <0.9.13 versions did
    if (AltMask != 0 && MetaMask == Mod1Mask) {
        MetaMask = AltMask;
        AltMask = Mod1Mask;
    }
#endif

    if (AltMask == 0 && MetaMask != 0) {
        AltMask = MetaMask;
        MetaMask = 0;
    }

    if (AltMask == 0 && MetaMask != Mod1Mask)
        AltMask = Mod1Mask;
    
    KeyMask =
        ControlMask |
        ShiftMask |
        AltMask |
        MetaMask;
    ButtonMask =
        Button1Mask |
        Button2Mask |
        Button3Mask |
        Button4Mask |
        Button5Mask;

    ButtonKeyMask = KeyMask | ButtonMask;
}

void runProgram(const char *str, char **args) {
    if (fork() == 0) {
        app->resetSignals();
        if (args)
            execvp(str, args);
        else
            execlp(str, str, 0);
        exit(1);
    }
}

void runCommand(const char *prog) {
    if (fork() == 0) {
        app->resetSignals();
        exit(system(prog));
    }
}

void runRestart(char *str, char **args) {
    app->resetSignals();
    if (str) {
        if (args) {
            execvp(str, args);
        } else {
            execlp(str, str, 0);
        }
    } else {
        execlp("icewm"EXEEXT, "icewm"EXEEXT, 0);
    }
    XBell(app->display(), 100);
    fprintf(stderr, "icewm: Could not restart %s, not on $PATH?\n", str ? str : "icewm"EXEEXT );
}	
