/* $Id: binding.c,v 1.59 1998/08/13 01:09:38 gjb Exp $
 * binding.c
 * (C) 1997-1998 By Maciej Stachowiak and Greg J. Badros
 */

#define BINDING_IMPLEMENTATION

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <guile/gh.h>
#include <X11/keysym.h>
#include <ctype.h>

#include "scwm.h"
#include "screen.h"
#include "window.h"
#include "decor.h"
#include "errors.h"
#include "complex.h"
#include "util.h"
#include "miscprocs.h"
#include "add_window.h"
#include "binding.h"
#include "xmisc.h"
#include "syscompat.h"

#ifdef USE_DMALLOC
#include "dmalloc.h"
#endif


/* also used by window.c's set-window-focus! */
SCWM_GLOBAL_SYMBOL(sym_click,"click");

SCWM_SYMBOL(sym_motion,"motion");
SCWM_SYMBOL(sym_one_and_a_half_clicks,"one-and-a-half-clicks");
SCWM_SYMBOL(sym_double_click,"double-click");

struct symnum {
  SCM sym;
  int value;
};

struct symnum binding_contexts[] =
{
  {SCM_UNDEFINED, C_WINDOW},
  {SCM_UNDEFINED, C_TITLE},
  {SCM_UNDEFINED, C_ICON},
  {SCM_UNDEFINED, C_ROOT},
  {SCM_UNDEFINED, C_FRAME},
  {SCM_UNDEFINED, C_SIDEBAR},
  {SCM_UNDEFINED, C_L1},
  {SCM_UNDEFINED, C_R1},
  {SCM_UNDEFINED, C_L2},
  {SCM_UNDEFINED, C_R2},
  {SCM_UNDEFINED, C_L3},
  {SCM_UNDEFINED, C_R3},
  {SCM_UNDEFINED, C_L4},
  {SCM_UNDEFINED, C_R4},
  {SCM_UNDEFINED, C_L5},
  {SCM_UNDEFINED, C_R5},
  {SCM_UNDEFINED, C_ALL},
  {SCM_UNDEFINED, 0}
};

static int MetaMask = 0,
  AltMask = 0,
  HyperMask = 0,
  SuperMask = 0;

static unsigned char rgmapMouseButtons[XSERVER_MAX_BUTTONS];

static int cMouseButtons = 3;


/**CONCEPT: Key Specifier
   A key specifier is a string denoting a keystroke, perhaps including
modifiers.  The available modifiers include S-, C-, M-, A-, H-, and s-
for Shift, Control, Meta, Alt, Hyper, and Super, respectively.  They
can be combined arbitrarily, and in any order, but should precede the
key name.
*/


static const char *
PchModifiersToModmask(const char *pch, int *pmodifier, char *func_name)
{
  int modmask = 0;
  Bool fError = False;

  while (True) {
    if (pch[0] && pch[1] != '-') { /* be sure we do not look past end of string */
      break;
    }
    switch (pch[0]) {
    case 'S': /* Shift */
      modmask |= ShiftMask;
      break;
    case 'C': /* Control */
      modmask |= ControlMask;
      break;
    case 'M': /* Meta */
      if (!MetaMask)
	fError = True;
      modmask |= MetaMask;
      break;
    case 'A': /* Alt */
      if (!AltMask)
	fError = True;
      modmask |= AltMask;
      break;
    case 'H': /* Hyper */
      if (!HyperMask)
	fError = True;
      modmask |= HyperMask;
      break;
    case 's': /* super modifier [0x40] (emacs uses "s", so we do too) */
      if (!SuperMask)
	fError = True;
      modmask |= SuperMask;
      break;
    case 'P':
      /* FIXGJB this can get pulled out later-- I used 'P' at first to avoid
         confusion between 's-' and 'S-' (shift), but people didn't like it */
      scwm_msg(WARN,func_name,"Unrecognized modifier P- (super is now 's-')");
      return NULL;
    default:
      scwm_msg(WARN,func_name,"Unrecognized modifier %c-",pch[0]);
      return NULL;
    }
    if (fError)
      scwm_msg(WARN,func_name,"Unbound modifier %c-",
	       pch[0]);
    pch += 2;
  }

  if (fError) {
    *pmodifier = -1;
  } else {
    *pmodifier = modmask;
  }
  return pch;
}


Bool 
FKeyToKeysymModifiers(SCM key, KeySym *pkeysym, int *pmodifier, char *func_name)
{
  Bool fOk = True;
  int len;
  char *keyname;
  char *pch;

  if (!gh_string_p(key)) {
    scm_wrong_type_arg(func_name, 2, key);
  }

  keyname = gh_scm2newstr(key,&len);
  pch = (char *) PchModifiersToModmask(keyname,pmodifier, func_name);

  if (pch == 0 || *pmodifier < 0) {
    FREE(keyname);
    return False;
  }

  if (pch[1] == '\0' && isgraph(pch[0])) {  /* single character, so use tolower */
    pch[0] = tolower(pch[0]);
  }

  if ((*pkeysym = XStringToKeysym(pch)) == NoSymbol ||
	   (XKeysymToKeycode(dpy, *pkeysym)) == 0) { 
    scwm_msg(WARN,func_name,"No symbol `%s'",keyname);
    fOk = False; 
  }
  FREE(keyname);
  return fOk;
}


/* Permit "Mouse-1", "1", "M1", "Mouse1", "mouse1" all to
   be acceptable */
static
int
BnumFromSz(const char *sz)
{
  if (sz == 0)
    return -1;

  if (tolower(*sz) == 'a'  && (strcasecmp(sz,"any") == 0 ||
			       strcasecmp(sz,"all") == 0)) {
    return 0;
  } else {
    int ichFirstDigit = strcspn(sz,"0123456789");
    if (strncasecmp(sz,"mouse-",ichFirstDigit) != 0) {
      return -1; /* no match */
    } else {
      if (strlen(sz+ichFirstDigit) != 1) return -1;
      return strtol(sz + ichFirstDigit, NULL, 10);
    }
  }
}

Bool
FButtonToBnumModifiers(SCM button, int *pbnum, int *pmodifier, char *func_name)
{
  Bool fOk = True;
  int len;
  char *button_name = NULL;

  if (!gh_string_p(button)) {
    if (gh_number_p(button)) {
      *pbnum = gh_scm2int(button);
      if (*pbnum < 0 || *pbnum > cMouseButtons) {
	scwm_msg(WARN,func_name,"No button number `%d'",*pbnum);
	return False;
      }
    } else {
      scm_wrong_type_arg(func_name, 2, button);
    }
  } else { /* it is a string */
    button_name = gh_scm2newstr(button,&len);
  }

  if (NULL!=button_name) {
    *pbnum = BnumFromSz(PchModifiersToModmask(button_name, pmodifier, func_name));

    if (*pbnum < 0) {
      scwm_msg(WARN,func_name,"No button `%s'",button_name);
      fOk=False;
    }
    if (*pmodifier < 0) {
      scwm_msg(WARN,func_name,"Ignoring mouse bind/unbind request for %s",
	       button_name);
      fOk=False;
    }
    FREE(button_name);
  }

  return fOk;
}




/*
 *  Procedure:
 *	GrabKeys - grab needed keys for the window
 *
 *  Inputs:
 *	psw - the scwm window structure to use
 */
void 
GrabKeys(ScwmWindow *psw)
{
  Binding *tmp;

  for (tmp = Scr.AllBindings; tmp != NULL; tmp = tmp->NextBinding) {
    if ((tmp->Context & (C_WINDOW | C_TITLE | C_RALL | C_LALL | C_SIDEBAR)) &&
	(tmp->IsMouse == 0)) {
      XGrabKey(dpy, tmp->Button_Key, tmp->Modifier, psw->frame, True,
	       GrabModeAsync, GrabModeAsync);
      if (tmp->Modifier != AnyModifier) {
	XGrabKey(dpy, tmp->Button_Key, tmp->Modifier | LockMask,
		 psw->frame, True,
		 GrabModeAsync, GrabModeAsync);
      }
    }
  }
  return;
}


void
GrabButtonWithModifiers(int button, int modifier, 
			ScwmWindow *psw)
{
  if (button > 0) {
    XGrabButton(dpy, button, modifier, psw->w,
		True, ButtonPressMask | ButtonReleaseMask,
		GrabModeAsync, GrabModeAsync, None,
		Scr.ScwmCursors[CURSOR_DEFAULT]);
    if (modifier != AnyModifier) {
      XGrabButton(dpy, button, (modifier | LockMask), psw->w,
		  True, ButtonPressMask | ButtonReleaseMask,
		  GrabModeAsync, GrabModeAsync, None,
		  Scr.ScwmCursors[CURSOR_DEFAULT]);
    }
  } else {
    GrabButtonWithModifiers(1,modifier,psw);
    GrabButtonWithModifiers(2,modifier,psw);
    GrabButtonWithModifiers(3,modifier,psw);
  }
}
  

void
UngrabButtonWithModifiers(int button, int modifier, 
			  ScwmWindow *psw)
{
  if (button > 0) {
    XUngrabButton(dpy, button, modifier, psw->w);
    if (modifier != AnyModifier) {
      XUngrabButton(dpy, button, (modifier | LockMask), psw->w);
    }
  } else {
    UngrabButtonWithModifiers(1,modifier,psw);
    UngrabButtonWithModifiers(2,modifier,psw);
    UngrabButtonWithModifiers(3,modifier,psw);
  }
}

/*
 *  Procedure:
 *	GrabButtons - grab needed buttons for the window
 *
 *  Inputs:
 * 	psw - the scwm window structure to use
 */

/* FIXGJB: rewrite to use GrabButtonWithModifiers, above */
void 
GrabButtons(ScwmWindow * psw)
{
  Binding *MouseEntry;

  MouseEntry = Scr.AllBindings;
  while (MouseEntry != (Binding *) 0) {
    if ((MouseEntry->Action != NULL) && (MouseEntry->Context & C_WINDOW)
	&& (MouseEntry->IsMouse == 1)) {
      if (MouseEntry->Button_Key > 0) {
	XGrabButton(dpy, MouseEntry->Button_Key, MouseEntry->Modifier,
		    psw->w,
		    True, ButtonPressMask | ButtonReleaseMask,
		    GrabModeAsync, GrabModeAsync, None,
		    Scr.ScwmCursors[CURSOR_DEFAULT]);
	if (MouseEntry->Modifier != AnyModifier) {
	  XGrabButton(dpy, MouseEntry->Button_Key,
		      (MouseEntry->Modifier | LockMask),
		      psw->w,
		      True, ButtonPressMask | ButtonReleaseMask,
		      GrabModeAsync, GrabModeAsync, None,
		      Scr.ScwmCursors[CURSOR_DEFAULT]);
	}
      } else {
	XGrabButton(dpy, 1, MouseEntry->Modifier,
		    psw->w,
		    True, ButtonPressMask | ButtonReleaseMask,
		    GrabModeAsync, GrabModeAsync, None,
		    Scr.ScwmCursors[CURSOR_DEFAULT]);
	XGrabButton(dpy, 2, MouseEntry->Modifier,
		    psw->w,
		    True, ButtonPressMask | ButtonReleaseMask,
		    GrabModeAsync, GrabModeAsync, None,
		    Scr.ScwmCursors[CURSOR_DEFAULT]);
	XGrabButton(dpy, 3, MouseEntry->Modifier,
		    psw->w,
		    True, ButtonPressMask | ButtonReleaseMask,
		    GrabModeAsync, GrabModeAsync, None,
		    Scr.ScwmCursors[CURSOR_DEFAULT]);
	if (MouseEntry->Modifier != AnyModifier) {
	  XGrabButton(dpy, 1,
		      (MouseEntry->Modifier | LockMask),
		      psw->w,
		      True, ButtonPressMask | ButtonReleaseMask,
		      GrabModeAsync, GrabModeAsync, None,
		      Scr.ScwmCursors[CURSOR_DEFAULT]);
	  XGrabButton(dpy, 2,
		      (MouseEntry->Modifier | LockMask),
		      psw->w,
		      True, ButtonPressMask | ButtonReleaseMask,
		      GrabModeAsync, GrabModeAsync, None,
		      Scr.ScwmCursors[CURSOR_DEFAULT]);
	  XGrabButton(dpy, 3,
		      (MouseEntry->Modifier | LockMask),
		      psw->w,
		      True, ButtonPressMask | ButtonReleaseMask,
		      GrabModeAsync, GrabModeAsync, None,
		      Scr.ScwmCursors[CURSOR_DEFAULT]);
	}
      }
    }
    MouseEntry = MouseEntry->NextBinding;
  }
  return;
}

/* This grabs all the defined keys on all the windows */
static void
grab_all_keys_all_buttons_all_windows()
{
  ScwmWindow *psw;
  for (psw = Scr.ScwmRoot.next; psw != NULL; psw = psw->next) {
    GrabKeys(psw);
    GrabButtons(psw);
  }
}


/* This grabs all the defined keys on all the windows */
static void
grab_all_buttons_all_windows()
{
  ScwmWindow *psw;
  for (psw = Scr.ScwmRoot.next; psw != NULL; psw = psw->next) {
    GrabButtons(psw);
  }
}

/* Just grab a single key + modifier on all windows
   This needs to be done after a new key binding */
static void
grab_key_all_windows(int key, int modifier)
{
  ScwmWindow *psw;
  for (psw = Scr.ScwmRoot.next; psw != NULL; psw = psw->next) {
    XGrabKey(dpy, key, modifier, psw->frame, True, 
	     GrabModeAsync, GrabModeAsync);
    if (modifier != AnyModifier) {
      XGrabKey(dpy, key, modifier | LockMask, psw->frame, True,
	       GrabModeAsync, GrabModeAsync);
    }
  }
}

/* Just grab a mouse button + modifier on all windows
   This needs to be done after a new mouse binding */
static void
grab_button_all_windows(int button, int modifier)
{
  ScwmWindow *psw;
  for (psw = Scr.ScwmRoot.next; psw != NULL; psw = psw->next) {
    GrabButtonWithModifiers(button,modifier,psw);
  }
}


static void
ungrab_button_all_windows(int button, int modifier)
{
  ScwmWindow *psw;
  for (psw = Scr.ScwmRoot.next; psw != NULL; psw = psw->next) {
    UngrabButtonWithModifiers(button,modifier,psw);
  }
}

/*
   ** to remove a binding from the global list (probably needs more processing
   ** for mouse binding lines though, like when context is a title bar button).
 */
void 
remove_binding(int context, int mods, int button, KeySym keysym,
	       int mouse_binding)
{
  Binding *temp = Scr.AllBindings, *temp2, *prev = NULL;
  KeyCode keycode = 0;

  if (!mouse_binding) {
    keycode = XKeysymToKeycode(dpy, keysym);
  } else if (context & C_WINDOW) {
    ungrab_button_all_windows(button,mods);
  }

  while (temp) {
    temp2 = temp->NextBinding;
    if (temp->IsMouse == mouse_binding) {
      if ((temp->Button_Key == ((mouse_binding) ? (button) : (keycode))) &&
	  (temp->Context == context) &&
	  (temp->Modifier == mods)) {
	/* we found it, remove it from list */
	if (prev) {		/* middle of list */
	  prev->NextBinding = temp2;
	} else {		/* must have been first one, set new start */
	  Scr.AllBindings = temp2;
	}
	FREE(temp);
	temp = NULL;
      }
    }
    if (temp)
      prev = temp;
    temp = temp2;
  }
}

void 
add_binding(int context, int modmask, int bnum_or_keysym, int mouse_p, 
	    SCM proc, char *name)
{
  Binding *prev_binding = Scr.AllBindings;
  Scr.AllBindings = NEW(Binding);

  Scr.AllBindings->IsMouse = mouse_p;
  Scr.AllBindings->Button_Key = bnum_or_keysym;
  Scr.AllBindings->key_name = name;
  Scr.AllBindings->Context = context;
  Scr.AllBindings->Modifier = modmask;
  /* FIXMS: This field should go away from the binding struct. */
  Scr.AllBindings->Action = "Scheme";
  Scr.AllBindings->Thunk = proc;
  Scr.AllBindings->NextBinding = prev_binding;

  scm_protect_object(proc);

  if (mouse_p) {
    if ( (context & C_WINDOW) && Scr.fWindowsCaptured) {
      /* only grab the button press if we have already captured,
	 otherwise it's a waste of time since we will grab
	 them all later when we do the initial capture;
	 this is good, since initialization probably defines
	 lots of mouse  bindings */
      grab_button_all_windows(bnum_or_keysym ,modmask);
    } 
  } else {
    if (Scr.fWindowsCaptured) {
      /* only grab the key if we have already captured,
	 otherwise it's a waste of time since we will grab
	 them all later when we do the initial capture;
	 this is good, since initialization probably defines
	 lots of key bindings */
      grab_key_all_windows(bnum_or_keysym ,modmask);
    }
  }
}


/**CONCEPT: Event Contexts

 */

int 
lookup_context(SCM context)
{
  int i;

  if (!gh_symbol_p(context)) {
    return -2;
  }
  for (i = 0; binding_contexts[i].value != 0; i++) {
    if (gh_eq_p(binding_contexts[i].sym, context)) {
      return (binding_contexts[i].value);
    }
  }
  return -1;
}

int 
compute_contexts(SCM contexts, char *func_name)
{
  int tmp, retval;

  if (gh_list_p(contexts)) {
    for (tmp = 0, retval = 0; contexts != SCM_EOL; contexts = gh_cdr(contexts)) {
      if ((tmp = lookup_context(gh_car(contexts))) < 0) {
	return tmp;
      } else {
	retval |= tmp;
      }
    }
  } else {
   retval = lookup_context(contexts);
  }

  switch (retval) {
  case 0:
    /* FIXGJBERROR: do not error by number */
    scwm_error(func_name, 8);
    break;
  case -1:
    /* FIXGJBERROR: do not error by number */
    scwm_error(func_name, 9);
    break;
  case -2:
    /* FIXGJBERROR: do not error by number */
    scm_wrong_type_arg(func_name, 1, contexts);
    break;
  default:
    break;
  }
  
  return retval;
}


/* FIXGJB: abstract out stuff-- lots of duplication
   between this and unbind_mouse */
SCWM_PROC(unbind_key, "unbind-key", 2, 0, 0,
          (SCM contexts, SCM key))
     /** Remove any bindings attached to KEY in given CONTEXTS.
CONTEXTS is a list of event-contexts (e.g., '(button1 sidebar))
KEY is a string giving the key-specifier (e.g., M-Delete for Meta+Delete) */
#define FUNC_NAME s_unbind_key
{
  KeySym keysym;
  Bool fOkayKey;
  int modmask = 0;
  int context = 0;

  context = compute_contexts(contexts, FUNC_NAME);
  fOkayKey = FKeyToKeysymModifiers(key, &keysym, &modmask, FUNC_NAME);

  /*
   * Don't let a 0 keycode go through, since that means AnyKey to the
   * XGrabKey call in GrabKeys().
   */
  if (keysym == NoSymbol || !fOkayKey) {
    int len;
    char *keyname = gh_scm2newstr(key,&len);
    scwm_msg(WARN,FUNC_NAME,"Ignoring key unbind request for `%s'",keyname);
    FREE(keyname);
  } else {
    remove_binding(context,modmask,0,keysym,False);
  }
  return SCM_UNSPECIFIED;
}
#undef FUNC_NAME


SCWM_PROC(unbind_mouse, "unbind-mouse", 2, 0, 0,
          (SCM contexts, SCM button))
     /** Remove any bindings attached to mouse BUTTON in given CONTEXTS.
CONTEXTS is a list of event-contexts (e.g., '(button1 sidebar))
BUTTON is a string or integer giving the mouse button number */
#define FUNC_NAME s_unbind_mouse
{
  int bnum = 0;
  int modmask = 0;
  int context = 0;
  int fButtonOK = True;

  fButtonOK = FButtonToBnumModifiers(button, &bnum, &modmask, FUNC_NAME);
  context = compute_contexts(contexts, FUNC_NAME);

  if (!fButtonOK) {
    /* Need a better error */
    scm_wrong_type_arg(FUNC_NAME,2,button);
  }

  remove_binding(context,modmask,bnum,0,True /* Mouse binding */);

  return SCM_UNSPECIFIED;
}
#undef FUNC_NAME


SCWM_PROC(bind_key, "bind-key", 3, 0, 0,
          (SCM contexts, SCM key, SCM proc))
     /** Bind the given KEY within the CONTEXTS to invoke PROC.
CONTEXTS is a list of event-contexts (e.g., '(button1 sidebar))
KEY is a string giving the key-specifier (e.g., M-Delete for Meta+Delete)
PROC is a procedure (possibly a thunk) that should be invoked */
#define FUNC_NAME s_bind_key
{
  KeySym keysym;
  int len = 0;
  Bool fOkayKey = False;
  Bool fBoundKey = False;	/* for error checking */
  int i, min, max;
  int modmask = 0;
  int context = 0;

  if (!gh_procedure_p(proc)) {
    scm_wrong_type_arg(FUNC_NAME, 3, proc);
  }

  context = compute_contexts(contexts, FUNC_NAME);

  fOkayKey = FKeyToKeysymModifiers(key,&keysym,&modmask, FUNC_NAME);

  /*
   * Don't let a 0 keycode go through, since that means AnyKey to the
   * XGrabKey call in GrabKeys().
   */
  if (keysym ==  NoSymbol || !fOkayKey) {
    char *keyname = gh_scm2newstr(key,&len);
    gh_allow_ints();
    SCM_ALLOW_INTS;
    scwm_msg(WARN,FUNC_NAME,"Ignoring key binding `%s'",keyname);
    FREE(keyname);
    return SCM_BOOL_F;
  }
  /* 
   * One keycode might map to the same keysym -MS
   */
  
  XDisplayKeycodes(dpy, &min, &max);
  for (i = min; i <= max; i++) {
    if (XKeycodeToKeysym(dpy, i, 0) == keysym) {
      add_binding(context, modmask, i, 0, proc, gh_scm2newstr(key,&len));
      fBoundKey = True;
    }
  }

  if (!fBoundKey) {
    char *keyname = gh_scm2newstr(key,&len);
    scwm_msg(WARN,FUNC_NAME,"No matching keycode for symbol `%s'",keyname);
    FREE(keyname);
    return SCM_BOOL_F; /* Use False for error */
  }
  return SCM_UNSPECIFIED;
}
#undef FUNC_NAME


SCWM_PROC(bind_mouse, "bind-mouse", 3, 0, 0,
          (SCM contexts, SCM button, SCM proc))
     /** Bind the given mouse BUTTON within the CONTEXTS to invoke PROC.
CONTEXTS is a list of event-contexts (e.g., '(button1 sidebar))
BUTTON is a string or integer giving the mouse button number
PROC is a procedure (possibly a thunk) that should be invoked */
#define FUNC_NAME s_bind_mouse
{
  int bnum = 0;
  int j = 0;
  int k = 0;
  int modmask = 0;
  int context = 0;
  Bool fChangedNumButtons = False;

  int fButtonOK = True;

  if (!gh_procedure_p(proc)) {
    scm_wrong_type_arg(FUNC_NAME, 3, proc);
  }

  context = compute_contexts(contexts, FUNC_NAME);
  fButtonOK = FButtonToBnumModifiers(button, &bnum, &modmask, FUNC_NAME);


  if ((context != C_ALL) && (context & C_LALL)) {
    /* check for nr_left_buttons */
    k = 0;
    j = (context & C_LALL) / C_L1;
    while (j > 0) {
      k++;
      j = j >> 1;
    }
    if (Scr.nr_left_buttons < k) {
      Scr.nr_left_buttons = k;
      fChangedNumButtons = True;
    }
  }
  if ((context != C_ALL) && (context & C_RALL)) {
    /* check for nr_right_buttons */
    k = 0;
    j = (context & C_RALL) / C_R1;
    while (j > 0) {
      k++;
      j = j >> 1;
    }
    if (Scr.nr_right_buttons < k) {
      Scr.nr_right_buttons = k;
      fChangedNumButtons = True;
    }
  }

  if ((context & C_WINDOW) && ((modmask == 0) || modmask == AnyModifier)) {
    Scr.buttons2grab &= ~(1 << (bnum - 1));
  }
 
  add_binding(context, modmask, bnum, 1, proc, NULL);

  if (fChangedNumButtons && Scr.fWindowsCaptured) {
  /* FIXGJB - we should redraw the titlebars if necessary to reflect the new
     buttons */
#if 1 /* FIXGJB this doesn't work, just want to redraw buttons on all windows */
    ScwmDecor *fl = cur_decor ? cur_decor : &Scr.DefaultDecor;
    redraw_borders(fl);
#else
    recapture(); /* this stinks, but'll have to do for now --11/11/97 gjb */
#endif
  }
  return SCM_UNSPECIFIED;
}
#undef FUNC_NAME







/* to distinguish click, double-click, move */

SCM mouse_ev_type = SCM_BOOL_F;

Bool have_orig_position = False;
int orig_x, orig_y;

void 
find_mouse_event_type()
{
  XEvent d;

  gh_defer_ints();
  WXGetPointerWindowOffsets(Scr.Root, &orig_x, &orig_y);
  have_orig_position = True;

  mouse_ev_type = sym_motion;
  if (IsClick(orig_x, orig_y, ButtonReleaseMask, &d)) {
    mouse_ev_type = sym_click;
    /* If it was a click, wait to see if its a double click */
    if (IsClick(orig_x, orig_y, ButtonPressMask, &d)) {
      mouse_ev_type = sym_one_and_a_half_clicks;
      if (IsClick(orig_x, orig_y, ButtonReleaseMask, &d)) {
	mouse_ev_type = sym_double_click;
      }
    }
  }
  gh_allow_ints();
}

void 
clear_mouse_event_type()
{
  have_orig_position = False;
  mouse_ev_type = SCM_BOOL_F;
}

/* FIXGJB: a single, slow click with no movement should
   still count as a single click, see IsClick(), too */
SCWM_PROC(mouse_event_type, "mouse-event-type", 0, 0, 0,
          ())
     /** Return a symbol corresponding to the type of the most recent mouse event.
Return value is one of 'motion, 'click, 'one-and-a-half-clicks, 'double-click.
You can `case' on this symbol in a procedure bound to a mouse event
to determine, e.g., whether the user single clicked or double clicked. */
#define FUNC_NAME s_mouse_event_type
{
  return mouse_ev_type;
}
#undef FUNC_NAME


SCWM_PROC(mod_mask_meta,"mod-mask-meta", 0, 0, 0, ())
     /** Return the bit-mask for the Meta modifier key, or #f.
Returns #f if and only if there is no key bound to act as Meta, otherwise
returns a power of two corresponding to the bit-mask of the modifier */
#define FUNC_NAME s_mod_mask_meta
{ return MetaMask == 0? SCM_BOOL_F : gh_int2scm(MetaMask); }
#undef FUNC_NAME

SCWM_PROC(mod_mask_alt, "mod-mask-alt", 0, 0, 0, ())
     /** Return the bit-mask for the Alt modifier key, or #f.
Returns #f if and only if there is no key bound to act as Alt, otherwise
returns a power of two corresponding to the bit-mask of the modifier */
#define FUNC_NAME s_mod_mask_alt
{ return AltMask == 0? SCM_BOOL_F : gh_int2scm(AltMask); }
#undef FUNC_NAME

SCWM_PROC(mod_mask_hyper, "mod-mask-hyper", 0, 0, 0, ())
     /** Return the bit-mask for the Hyper modifier key, or #f.
Returns #f if and only if there is no key bound to act as Hyper, otherwise
returns a power of two corresponding to the bit-mask of the modifier */
#define FUNC_NAME s_mod_mask_hyper
{ return HyperMask == 0? SCM_BOOL_F : gh_int2scm (HyperMask); }
#undef FUNC_NAME


SCWM_PROC(mod_mask_super, "mod-mask-super", 0, 0, 0,())
     /** Return the bit-mask for the Super modifier key, or #f.
Returns #f if and only if there is no key bound to act as Super, otherwise
returns a power of two corresponding to the bit-mask of the modifier */
#define FUNC_NAME s_mod_mask_super
{ return SuperMask == 0? SCM_BOOL_F : gh_int2scm (SuperMask); }
#undef FUNC_NAME


SCWM_PROC(X_pointer_mapping, "X-pointer-mapping", 0, 0, 0,
          ())
     /** Return the mapping of physical->logical pointer buttons as a list.
The length of the returned list is the number of buttons available.  Each
element in the list is an integer.  E.g., '(1 2 3) is a normally mapped
3-button mouse, whereas '(3 2 1) is a 3-button mouse where the rightmost
physical button acts as logical button 1, and the leftmost acts as button 3. */
#define FUNC_NAME s_X_pointer_mapping
{
  SCM mapping = SCM_EOL;
  int imap = cMouseButtons - 1;
  while (imap >= 0) {
    mapping = gh_cons(gh_int2scm(rgmapMouseButtons[imap]), mapping);
    imap--;
  }
  return mapping;
}
#undef FUNC_NAME

void
init_pointer_mapping(void)
{
  cMouseButtons = XGetPointerMapping(dpy, rgmapMouseButtons, XSERVER_MAX_BUTTONS);
}

void
init_modifiers(void)
{
  int i, j, num;
  XModifierKeymap *mod;
  KeyCode *codes;
  KeySym *syms;

  MetaMask = AltMask = HyperMask = SuperMask = 0;

  mod = XGetModifierMapping(dpy);
  if (mod) {
    codes = mod->modifiermap;
    for (i = 0; i < 8; i++)
      for (j = 0; j < mod->max_keypermod; j++, codes++)
	if (*codes) {
	  syms = XGetKeyboardMapping(dpy, *codes, 1, &num);
	  if (syms) {
	    while (num--)
	      switch (syms[num]) {
	      case XK_Meta_L:
	      case XK_Meta_R:
		MetaMask = 1<<i;
		break;
	      case XK_Alt_L:
	      case XK_Alt_R:
		AltMask = 1<<i;
		break;
	      case XK_Super_L:
	      case XK_Super_R:
		SuperMask = 1<<i;
		break;
	      case XK_Hyper_L:
	      case XK_Hyper_R:
		HyperMask = 1<<i;
		break;
	      }
	    XFree(syms);
	  }
	}
    XFreeModifiermap(mod);
  }
}

void 
init_binding(void)
{
  int i;
  /* FIXGJB: buttons should have symbolic names, not numbered
     physically */
  static char *context_strings[] =
  {
    "window",
    "title",
    "icon",
    "root",
    "frame",
    "sidebar",
    "button-1",
    "button-2",
    "button-3",
    "button-4",
    "button-5",
    "button-6",
    "button-7",
    "button-8",
    "button-9",
    "button-10",
    "all",
    NULL
  };

  for (i = 0; context_strings[i] != NULL; i++) {
    binding_contexts[i].sym = gh_symbol2scm(context_strings[i]);
    scm_permanent_object(binding_contexts[i].sym);
  }

#ifndef SCM_MAGIC_SNARFER
#include "binding.x"
#endif
}


/* Local Variables: */
/* tab-width: 8 */
/* c-basic-offset: 2 */
/* End: */
