// remap from unicode to other encodings
// (C) Copyright 1998 Herbert Duerr

#include "encoding.h"
#include <string.h>

Encoding* Encoding::first	= 0;
Encoding* Encoding::last	= 0;

Encoding::Encoding( char* mapname)
: strName( mapname), lenName( strlen( mapname))
{
	if( !first) first = this;
	else last->next = this;
	last = this;
	next = 0;
}

int Encoding::parse( char* mapnames, Encoding** maps0, int maxcodes)
{
	Encoding** maps = maps0;

	for(; *mapnames; ++mapnames) {
		for( Encoding* m = first; m; m = m->next)
			if( !strncmp( mapnames, m->strName, m->lenName)) {
				*(maps++) = m;
				mapnames += m->lenName;
				break;
			}
		if( (*mapnames != ',') || (maps - maps0 >= maxcodes))
			break;
	}

	return (maps - maps0);
}

Encoding* Encoding::find( char* mapname)
{
	for( Encoding* m = first; m; m = m->next)
		if( !strcmp( mapname, m->strName))
			return m;
	return 0;
}

Encoding* Encoding::enumerate( Encoding* iterator)
{
	return iterator ? iterator->next : first;
}


//---
class iso8859_1
: public Encoding
{
public:
	iso8859_1() : Encoding("iso8859-1") {}
	int map2unicode( int code);
} exemplar_iso8859_1;

int iso8859_1::map2unicode( int code)
{
	if( code == 0x80) code = 0x20AC;	// euro currency symbol
	return code;
}


//---
class iso8859_2
: public Encoding
{
public:
	iso8859_2() : Encoding("iso8859-2") {}
	int map2unicode( int code);
} exemplar_iso8859_2;

int iso8859_2::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0a0, 0x104, 0x2d8, 0x141, 0x0a4, 0x13d, 0x15a, 0xa7,
		0x0a8, 0x160, 0x15e, 0x164, 0x179, 0x0ad, 0x17d, 0x17b,
		0x0b0, 0x105, 0x2db, 0x142, 0x0b4, 0x13e, 0x15b, 0x2c7,
		0x0b8, 0x161, 0x15f, 0x165, 0x17a, 0x2dd, 0x17e, 0x17c,
		0x154, 0x0c1, 0x0c2, 0x102, 0x0c4, 0x139, 0x106, 0x0c7,
		0x10c, 0x0c9, 0x118, 0x0cb, 0x11a, 0x0cd, 0x0ce, 0x10e,
		0x110, 0x143, 0x147, 0x0d3, 0x0d4, 0x150, 0x0d6, 0x0d7,
		0x158, 0x16e, 0x0da, 0x170, 0x0dc, 0x0dd, 0x162, 0x0df,
		0x155, 0x0e1, 0x0e2, 0x103, 0x0e4, 0x13a, 0x107, 0x0e7,
		0x10d, 0x0e9, 0x119, 0x0eb, 0x11b, 0x0ed, 0x0ee, 0x10f,
		0x111, 0x144, 0x148, 0x0f3, 0x0f4, 0x151, 0x0f6, 0x0f7,
		0x159, 0x16f, 0x0fa, 0x171, 0x0fc, 0x0fd, 0x163, 0x2d9
	};

	if( code == 0x80) return 0x20AC;	// euro currency symbol
	if( code <= 0xa0 || code >= 256) return code;
	return table[ code - 0xa0];
}

//---
class iso8859_3
: public Encoding
{
public:
	iso8859_3() : Encoding("iso8859-3") {}
	int map2unicode( int code);
} exemplar_iso8859_3;

int iso8859_3::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0a0, 0x126, 0x2d8, 0x0a3, 0x0a4, 0x0a5, 0x124, 0x0a7,
		0x0a8, 0x130, 0x15e, 0x11e, 0x134, 0x0ad, 0x0ae, 0x17b,
		0x0b0, 0x127, 0x0b2, 0x0b3, 0x0b4, 0x0b5, 0x125, 0x0b7,
		0x0b8, 0x131, 0x15f, 0x11f, 0x135, 0x0bd, 0x0be, 0x17c,
		0x0c0, 0x0c1, 0x0c2, 0x0c3, 0x0c4, 0x10a, 0x108, 0x0c7,
		0x0c8, 0x0c9, 0x0ca, 0x0cb, 0x0cc, 0x0cd, 0x0ce, 0x0cf,
		0x0d0, 0x0d1, 0x0d2, 0x0d3, 0x0d4, 0x120, 0x0d6, 0x0d7,
		0x11c, 0x0d9, 0x0da, 0x0db, 0x0dc, 0x16c, 0x15c, 0x0df,
		0x0e0, 0x0e1, 0x0e2, 0x0e3, 0x0e4, 0x10b, 0x109, 0x0e7,
		0x0e8, 0x0e9, 0x0ea, 0x0eb, 0x0ec, 0x0ed, 0x0ee, 0x0ef,
		0x0f0, 0x0f1, 0x0f2, 0x0f3, 0x0f4, 0x121, 0x0f6, 0x0f7,
		0x11d, 0x0f9, 0x0fa, 0x0fb, 0x0fc, 0x16d, 0x15d, 0x2d9
	};

	if( code == 0x80) return 0x20AC;	// euro currency symbol
	if( code <= 0xa0 || code >= 256) return code;
	return table[ code - 0xa0];
}

//---
class iso8859_4
: public Encoding
{
public:
	iso8859_4() : Encoding("iso8859-4") {}
	int map2unicode( int code);
} exemplar_iso8859_4;

int iso8859_4::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0a0, 0x104, 0x138, 0x156, 0x0a4, 0x128, 0x13b, 0x0a7,
		0x0a8, 0x160, 0x112, 0x122, 0x166, 0x0ad, 0x17d, 0x0af,
		0x0b0, 0x105, 0x2db, 0x157, 0x0b4, 0x129, 0x13c, 0x2c7,
		0x0b8, 0x161, 0x113, 0x123, 0x167, 0x14a, 0x17e, 0x14b,
		0x100, 0x0c1, 0x0c2, 0x0c3, 0x0c4, 0x0c5, 0x0c6, 0x12e,
		0x10c, 0x0c9, 0x118, 0x0cb, 0x116, 0x0cd, 0x0ce, 0x12a,
		0x110, 0x145, 0x14c, 0x136, 0x0d4, 0x0d5, 0x0d6, 0x0d7,
		0x0d8, 0x172, 0x0da, 0x0db, 0x0dc, 0x168, 0x16a, 0x0df,
		0x101, 0x0e1, 0x0e2, 0x0e3, 0x0e4, 0x0e5, 0x0e6, 0x12f,
		0x10d, 0x0e9, 0x119, 0x0eb, 0x117, 0x0ed, 0x0ee, 0x12b,
		0x111, 0x146, 0x14d, 0x137, 0x0f4, 0x0f5, 0x0f6, 0x0f7,
		0x0f8, 0x173, 0x0fa, 0x0fb, 0x0fc, 0x169, 0x16b, 0x2d9
	};

	if( code == 0x80) return 0x20AC;	// euro currency symbol
	if( code <= 0xa0 || code >= 256) return code;
	return table[ code - 0xa0];
}

//---
class iso8859_5
: public Encoding
{
public:
	iso8859_5() : Encoding("iso8859-5") {}
	int map2unicode( int code);
} exemplar_iso8859_5;

int iso8859_5::map2unicode( int code)
{
	if( code == 0x80) return 0x20AC;	// euro currency symbol
	if( code <= 0xa0 || code == 0xad || code >= 256) return code;
	if( code == 0xf0) return 0x2116;
	if( code == 0xfd) return 0xa7;
	return (code + 0x360);
}

//---
class iso8859_6
: public Encoding
{
public:
	iso8859_6() : Encoding("iso8859-6") {}
	int map2unicode( int code);
} exemplar_iso8859_6;

int iso8859_6::map2unicode( int code)
{
	if( code < 0x30 || code >= 256) return code;
	if( code <= 0x39) return (code + 0x630);
	if( code == 0x80) return 0x20AC;	// euro currency symbol
	if( code <= 0xa4) return code;
	return (code + 0x560);
}

//---
class iso8859_7
: public Encoding
{
public:
	iso8859_7() : Encoding("iso8859-7") {}
	int map2unicode( int code);
} exemplar_iso8859_7;

int iso8859_7::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0a0, 0x2bd, 0x2bc, 0x0a3, 0x0a4, 0x0a5, 0x0a6, 0x0a7,
		0x0a8, 0x0a9, 0x0aa, 0x0ab, 0x0ac, 0x0ad, 0x0ae, 0x2015,
		0x0b0, 0x0b1, 0x0b2, 0x0b3, 0x384, 0x385, 0x386, 0x0b7,
		0x388, 0x389, 0x38a, 0x0bb, 0x38c, 0x0bd
	};

	if( code == 0x80) return 0x20AC;	// euro currency symbol
	if( code <= 0xa0 || code >= 256) return code;
	if( code >= 0xbe) return (code + 0x2d0);
	return table[ code - 0xa0];
}

//---
class iso8859_8
: public Encoding
{
public:
	iso8859_8() : Encoding("iso8859-8") {}
	int map2unicode( int code);
} exemplar_iso8859_8;

int iso8859_8::map2unicode( int code)
{
	if( code <= 0xa9 || code >= 256) return code;
	if( code == 0xaa) return 0xd7;
	if( code == 0xaf) return 0x203e;
	if( code == 0xdf) return 0x2017;
	return (code + 0x4f0);	// thanks Stanislav Malyshev
}

//---
class iso8859_9
: public Encoding
{
public:
	iso8859_9() : Encoding("iso8859-9") {}
	int map2unicode( int code);
} exemplar_iso8859_9;

int iso8859_9::map2unicode( int code)
{
	if( code < 0xd0 || code >= 255) return code;
	switch( code) {
	case 0x80:	return 0x20AC;
	case 0xd0:	return 0x11e;
	case 0xdd:	return 0x130;
	case 0xde:	return 0x15e;
	case 0xf0:	return 0x11f;
	case 0xfd:	return 0x131;
	case 0xfe:	return 0x15f;
	}
	return code;
}


//--- thanks Vladimir Eltsov and Nikolay Grygoryev
class koi8_r
: public Encoding
{
public:
	koi8_r() : Encoding("koi8-r") {}
	int map2unicode( int code);
} exemplar_koi8_r;

int koi8_r::map2unicode( int code)
{
	static unsigned short table[] = {
		0x2500,0x2502,0x250c,0x2510,0x2514,0x2518,0x251c,0x2524,
		0x252c,0x2534,0x253c,0x2580,0x2584,0x2588,0x258c,0x2590,
		0x2591,0x2592,0x2593,0x2320,0x25a0,0x2219,0x221a,0x2248,
		0x2264,0x2265,0x00a0,0x2321,0x00b0,0x00b2,0x00b7,0x00f7,
		0x2550,0x2551,0x2552,0x0451,0x2553,0x2554,0x2555,0x2556,
		0x2557,0x2558,0x2559,0x255a,0x255b,0x255c,0x255d,0x255e,
		0x255f,0x2560,0x2561,0x0401,0x2562,0x2563,0x2564,0x2565,
		0x2566,0x2567,0x2568,0x2569,0x256a,0x256b,0x256c,0x00a9,
		0x044e,0x0430,0x0431,0x0446,0x0434,0x0435,0x0444,0x0433,
		0x0445,0x0438,0x0439,0x043a,0x043b,0x043c,0x043d,0x043e,
		0x043f,0x044f,0x0440,0x0441,0x0442,0x0443,0x0436,0x0432,
		0x044c,0x044b,0x0437,0x0448,0x044d,0x0449,0x0447,0x044a,
		0x042e,0x0410,0x0411,0x0426,0x0414,0x0415,0x0424,0x0413,
		0x0425,0x0418,0x0419,0x041a,0x041b,0x041c,0x041d,0x041e,
		0x041f,0x042f,0x0420,0x0421,0x0422,0x0423,0x0416,0x0412,
		0x042c,0x042b,0x0417,0x0428,0x042d,0x0429,0x0427,0x042a
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class windows_1251
: public Encoding
{
public:
	windows_1251() : Encoding("windows-1251") {}
	int map2unicode( int code);
} exemplar_windows_1251;

int windows_1251::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0402,0x0403,0x201A,0x0453,0x201E,0x2026,0x2020,0x2021,
		0x20AC,0x2030,0x0409,0x2039,0x040A,0x040C,0x040B,0x040F,
		0x0452,0x2018,0x2019,0x201C,0x201D,0x2022,0x2013,0x2014,
		0,0x2122,0x0459,0x203A,0x045A,0x045C,0x045B,0x045F,
		0x00A0,0x040E,0x045E,0x0408,0x00A4,0x0490,0x00A6,0x00A7,
		0x0401,0x00A9,0x0404,0x00AB,0x00AC,0x00AD,0x00AE,0x0407,
		0x00B0,0x00B1,0x0406,0x0456,0x0491,0x00B5,0x00B6,0x00B7,
		0x0451,0x2116,0x0454,0x00BB,0x0458,0x0405,0x0455,0x0457,
		0x0410,0x0411,0x0412,0x0413,0x0414,0x0415,0x0416,0x0417,
		0x0418,0x0419,0x041A,0x041B,0x041C,0x041D,0x041E,0x041F,
		0x0420,0x0421,0x0422,0x0423,0x0424,0x0425,0x0426,0x0427,
		0x0428,0x0429,0x042A,0x042B,0x042C,0x042D,0x042E,0x042F,
		0x0430,0x0431,0x0432,0x0433,0x0434,0x0435,0x0436,0x0437,
		0x0438,0x0439,0x043A,0x043B,0x043C,0x043D,0x043E,0x043F,
		0x0440,0x0441,0x0442,0x0443,0x0444,0x0445,0x0446,0x0447,
		0x0448,0x0449,0x044A,0x044B,0x044C,0x044D,0x044E,0x044F
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}

//--- thanks Petr Tomasek
class iso8859_10
: public Encoding
{
public:
	iso8859_10() : Encoding("iso8859-10") {}
	int map2unicode( int code);
} exemplar_iso8859_10;

int iso8859_10::map2unicode( int code)
{
	static unsigned short table[] = {
		0x20AC,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,
		0x0088,0x0089,0x008A,0x008B,0x008C,0x008D,0x008E,0x008F,
		0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,
		0x0098,0x0099,0x009A,0x009B,0x009C,0x009D,0x009E,0x009F,
		0x00A0,0x0104,0x0112,0x0122,0x012A,0x0128,0x0136,0x013B,
		0x0143,0x0156,0x0160,0x0166,0x017D,0x00AD,0x0138,0x014A,
		0x0111,0x0105,0x0113,0x0123,0x012B,0x0129,0x0137,0x013C,
		0x0144,0x0157,0x0161,0x0167,0x017E,0x00A7,0x00DF,0x014B,
		0x0100,0x00C1,0x00C2,0x00C3,0x00C4,0x00C5,0x00C6,0x012E,
		0x010C,0x00C9,0x0118,0x00CB,0x0116,0x00CD,0x00CE,0x00CF,
		0x0110,0x0145,0x014C,0x00D3,0x00D4,0x00D5,0x00D6,0x0168,
		0x00D8,0x0172,0x00DA,0x00DB,0x00DC,0x00DD,0x00DE,0x016A,
		0x0101,0x00E1,0x00E2,0x00E3,0x00E4,0x00E5,0x00E6,0x012F,
		0x010D,0x00E9,0x0119,0x00EB,0x0117,0x00ED,0x00EE,0x00EF,
		0x00F0,0x0146,0x014D,0x00F3,0x00F4,0x00F5,0x00F6,0x0169,
		0x00F8,0x0173,0x00FA,0x00FB,0x00FC,0x00FD,0x00FE,0x016B
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class windows_1250
: public Encoding
{
public:
	windows_1250() : Encoding("windows-1250") {}
	int map2unicode( int code);
} exemplar_windows_1250;

int windows_1250::map2unicode( int code)
{
	static unsigned short table[] = {
		0x20AC,0,0x201A,0,0x201E,0x22EF,0x2020,0x2021,
		0,0x2030,0x0160,0x2039,0x015A,0x0164,0x017D,0x0179,
		0,0x2018,0x2019,0x201C,0x201D,0x2219,0x2013,0x2014,
		0,0x2122,0x0161,0x203A,0x015B,0x0165,0x017E,0x017A,
		0x00A0,0x02C7,0x02D8,0x0141,0x00A4,0x0104,0x00A6,0x00A7,
		0x00A8,0x00A9,0x015E,0x00AB,0x00AC,0x00AD,0x00AE,0x017B,
		0x00B0,0x00B1,0x02DB,0x0142,0x00B4,0x00B5,0x00B6,0x00B7,
		0x00B8,0x0105,0x015F,0x00BB,0x0178,0x02DD,0x013D,0x017C,
		0x0154,0x00C1,0x00C2,0x0102,0x00C4,0x0139,0x0106,0x00C7,
		0x010C,0x00C9,0x0118,0x00CB,0x011A,0x00CD,0x00CE,0x010E,
		0x0110,0x0143,0x0147,0x00D3,0x00D4,0x0150,0x00D6,0x00D7,
		0x0158,0x016E,0x00DA,0x0170,0x00DC,0x00DD,0x0162,0x00DF,
		0x0155,0x00E1,0x00E2,0x0103,0x00E4,0x013A,0x0107,0x00E7,
		0x010D,0x00E9,0x0119,0x00EB,0x011B,0x00ED,0x00EE,0x010F,
		0x0111,0x0144,0x0148,0x00F3,0x00F4,0x0151,0x00F6,0x00F7,
		0x0159,0x016F,0x00FA,0x0171,0x00FC,0x00FD,0x0163,0x02D9
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class windows_1252
: public Encoding
{
public:
	windows_1252() : Encoding("windows-1252") {}
	int map2unicode( int code);
} exemplar_windows_1252;

int windows_1252::map2unicode( int code)
{
	static unsigned short table[] = {
		0x20AC,0,0x201A,0x0192,0x201E,0x22EF,0x2020,0x2021,
		0x0302,0x2030,0x0160,0x2039,0x0152,0,0,0,
		0,0x2018,0x2019,0x201C,0x201D,0x2219,0x2013,0x2014,
		0x0303,0x2122,0x0161,0x203A,0x0153,0,0,0x0178,
		0x00A0,0x00A1,0x00A2,0x00A3,0x00A4,0x00A5,0x00A6,0x00A7,
		0x00A8,0x00A9,0x00AA,0x00AB,0x00AC,0x00AD,0x00AE,0x00AF,
		0x00B0,0x00B1,0x00B2,0x00B3,0x00B4,0x00B5,0x00B6,0x00B7,
		0x00B8,0x00B9,0x00BA,0x00BB,0x00BC,0x00BD,0x00BE,0x00BF,
		0x00C0,0x00C1,0x00C2,0x00C3,0x00C4,0x00C5,0x00C6,0x00C7,
		0x00C8,0x00C9,0x00CA,0x00CB,0x00CC,0x00CD,0x00CE,0x00CF,
		0x00D0,0x00D1,0x00D2,0x00D3,0x00D4,0x00D5,0x00D6,0x00D7,
		0x00D8,0x00D9,0x00DA,0x00DB,0x00DC,0x00DD,0x00DE,0x00DF,
		0x00E0,0x00E1,0x00E2,0x00E3,0x00E4,0x00E5,0x00E6,0x00E7,
		0x00E8,0x00E9,0x00EA,0x00EB,0x00EC,0x00ED,0x00EE,0x00EF,
		0x00F0,0x00F1,0x00F2,0x00F3,0x00F4,0x00F5,0x00F6,0x00F7,
		0x00F8,0x00F9,0x00FA,0x00FB,0x00FC,0x00FD,0x00FE,0x00FF
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class windows_1253
: public Encoding
{
public:
	windows_1253() : Encoding("windows-1253") {}
	int map2unicode( int code);
} exemplar_windows_1253;

int windows_1253::map2unicode( int code)
{
	static unsigned short table[] = {
		0x20AC,0,0x201A,0x0192,0x201E,0x22EF,0x2020,0x2021,
		0,0x2030,0,0x2039,0,0,0,0,
		0,0x2018,0x2019,0x201C,0x201D,0x2219,0x2013,0x2014,
		0,0x2122,0,0x203A,0,0,0,0,
		0x00A0,0x0385,0x0386,0x00A3,0x00A4,0x00A5,0x00A6,0x00A7,
		0x00A8,0x00A9,0,0x00AB,0x00AC,0x00AD,0x00AE,0x2015,
		0x00B0,0x00B1,0x00B2,0x00B3,0x0384,0x00B5,0x00B6,0x0387,
		0x0388,0x0389,0x038A,0x00BB,0x038C,0x00BD,0x038E,0x038F,
		0x0390,0x0391,0x0392,0x0393,0x0394,0x0395,0x0396,0x0397,
		0x0398,0x0399,0x039A,0x039B,0x039C,0x039D,0x039E,0x039F,
		0x03A0,0x03A1,0,0x03A3,0x03A4,0x03A5,0x03A6,0x03A7,
		0x03A8,0x03A9,0x03AA,0x03AB,0x03AC,0x03AD,0x03AE,0x03AF,
		0x03B0,0x03B1,0x03B2,0x03B3,0x03B4,0x03B5,0x03B6,0x03B7,
		0x03B8,0x03B9,0x03BA,0x03BB,0x03BC,0x03BD,0x03BE,0x03BF,
		0x03C0,0x03C1,0x03C2,0x03C3,0x03C4,0x03C5,0x03C6,0x03C7,
		0x03C8,0x03C9,0x03CA,0x03CB,0x03CC,0x03CD,0x03CE,0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class windows_1254
: public Encoding
{
public:
	windows_1254() : Encoding("windows-1254") {}
	int map2unicode( int code);
} exemplar_windows_1254;

int windows_1254::map2unicode( int code)
{
	static unsigned short table[] = {
		0x20AC,0,0x201A,0x0192,0x201E,0x22EF,0x2020,0x2021,
		0x0302,0x2030,0x0160,0x2039,0x0152,0,0,0,
		0,0x2018,0x2019,0x201C,0x201D,0x2219,0x2013,0x2014,
		0x0303,0x2122,0x0161,0x203A,0x0153,0,0,0x0178,
		0x00A0,0x00A1,0x00A2,0x00A3,0x00A4,0x00A5,0x00A6,0x00A7,
		0x00A8,0x00A9,0x00AA,0x00AB,0x00AC,0x00AD,0x00AE,0x00AF,
		0x00B0,0x00B1,0x00B2,0x00B3,0x00B4,0x00B5,0x00B6,0x00B7,
		0x00B8,0x00B9,0x00BA,0x00BB,0x00BC,0x00BD,0x00BE,0x00BF,
		0x00C0,0x00C1,0x00C2,0x00C3,0x00C4,0x00C5,0x00C6,0x00C7,
		0x00C8,0x00C9,0x00CA,0x00CB,0x00CC,0x00CD,0x00CE,0x00CF,
		0x011E,0x00D1,0x00D2,0x00D3,0x00D4,0x00D5,0x00D6,0x00D7,
		0x00D8,0x00D9,0x00DA,0x00DB,0x00DC,0x0130,0x015E,0x00DF,
		0x00E0,0x00E1,0x00E2,0x00E3,0x00E4,0x00E5,0x00E6,0x00E7,
		0x00E8,0x00E9,0x00EA,0x00EB,0x00EC,0x00ED,0x00EE,0x00EF,
		0x011F,0x00F1,0x00F2,0x00F3,0x00F4,0x00F5,0x00F6,0x00F7,
		0x00F8,0x00F9,0x00FA,0x00FB,0x00FC,0x0131,0x015F,0x00FF
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class windows_1255
: public Encoding
{
public:
	windows_1255() : Encoding("windows-1255") {}
	int map2unicode( int code);
} exemplar_windows_1255;

int windows_1255::map2unicode( int code)
{
	static unsigned short table[] = {
		0x20AC,0,0x201A,0x0192,0x201E,0x22EF,0x2020,0x2021,
		0,0x2030,0,0x2039,0,0,0,0,
		0,0x2018,0x2019,0x201C,0x201D,0x2219,0x2013,0x2014,
		0,0x2122,0,0x203A,0,0,0,0,
		0x00A0,0,0x00A2,0x00A3,0x00A4,0x00A5,0x00A6,0x00A7,
		0x00A8,0x00A9,0x00D7,0x00AB,0x00AC,0x00AD,0x00AE,0x030D,
		0x00B0,0x00B1,0x00B2,0x00B3,0x00B4,0x00B5,0x00B6,0x00B7,
		0x00B8,0x00B9,0x00F7,0x00BB,0x00BC,0x00BD,0x00BE,0,
		0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0x2017,
		0x05D0,0x05D1,0x05D2,0x05D3,0x05D4,0x05D5,0x05D6,0x05D7,
		0x05D8,0x05D9,0x05DA,0x05DB,0x05DC,0x05DD,0x05DE,0x05DF,
		0x05E0,0x05E1,0x05E2,0x05E3,0x05E4,0x05E5,0x05E6,0x05E7,
		0x05E8,0x05E9,0x05EA,0,0,0,0,0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class windows_1256
: public Encoding
{
public:
	windows_1256() : Encoding("windows-1256") {}
	int map2unicode( int code);
} exemplar_windows_1256;

int windows_1256::map2unicode( int code)
{
	static unsigned short table[] = {
		0x060C,0,0x201A,0,0x201E,0x22EF,0x2020,0x2021,
		0,0x2030,0,0x2039,0,0,0,0,
		0,0x2018,0x2019,0x201C,0x201D,0x2219,0x2013,0x2014,
		0x061B,0x2122,0x061F,0x203A,0x0621,0x0622,0x0623,0x0178,
		0x00A0,0x0624,0x0625,0x00A3,0x00A4,0x0626,0x00A6,0x00A7,
		0x0627,0x00A9,0x0628,0x00AB,0x00AC,0x00AD,0x00AE,0,
		0x00B0,0x00B1,0x0629,0x062A,0x062B,0x00B5,0x00B6,0x00B7,
		0x062C,0,0x062D,0x00BB,0x062E,0x062F,0x0630,0x0631,
		0x00C0,0x0632,0x00C2,0,0x0633,0x0634,0x0635,0x00C7,
		0x00C8,0x00C9,0x00CA,0x00CB,0x0636,0x0637,0x00CE,0x00CF,
		0x0638,0x0639,0x063A,0x0640,0x00D4,0x0641,0x0642,0x00D7,
		0x0643,0x00D9,0x06AF,0x00DB,0x00DC,0x0644,0x0645,0x0646,
		0x00E0,0x0647,0x00E2,0,0x0648,0x0649,0x064A,0x00E7,
		0x00E8,0x00E9,0x00EA,0x00EB,0x064B,0x064C,0x00EE,0x00EF,
		0x064D,0x064E,0x064F,0x0650,0x00F4,0x0651,0x0652,0x00F7,
		0,0x00F9,0,0x00FB,0x00FC,0,0,0x00FF
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_437
: public Encoding
{
public:
	cp_437() : Encoding("cp-437") {}
	int map2unicode( int code);
} exemplar_cp_437;

int cp_437::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x00E0,0x00E5,0x00E7,
		0x00EA,0x00EB,0x00E8,0x00EF,0x00EE,0x00EC,0x00C4,0x00C5,
		0x00C9,0x00E6,0x00C6,0x00F4,0x00F6,0x00F2,0x00FB,0x00F9,
		0x00FF,0x00D6,0x00DC,0x00A2,0x00A3,0x00A5,0x20A7,0x0192,
		0x00E1,0x00ED,0x00F3,0x00FA,0x00F1,0x00D1,0x00AA,0x00BA,
		0x00BF,0x2310,0x00AC,0x00BD,0x00BC,0x00A1,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x03B1,0x00DF,0x0393,0x03C0,0x2211,0x03C3,0x00B5,0x03C4,
		0x03A6,0x0398,0x2126,0x03B4,0x221E,0x2205,0x2208,0x2229,
		0x2261,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,0x2248,
		0x00B0,0x2219,0x00B7,0x221A,0x207F,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_737
: public Encoding
{
public:
	cp_737() : Encoding("cp-737") {}
	int map2unicode( int code);
} exemplar_cp_737;

int cp_737::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0391,0x0392,0x0393,0x0394,0x0395,0x0396,0x0397,0x0398,
		0x0399,0x039A,0x039B,0x039C,0x039D,0x039E,0x039F,0x03A0,
		0x03A1,0x03A3,0x03A4,0x03A5,0x03A6,0x03A7,0x03A8,0x03A9,
		0x03B1,0x03B2,0x03B3,0x03B4,0x03B5,0x03B6,0x03B7,0x03B8,
		0x03B9,0x03BA,0x03BB,0x03BC,0x03BD,0x03BE,0x03BF,0x03C0,
		0x03C1,0x03C3,0x03C2,0x03C4,0x03C5,0x03C6,0x03C7,0x03C8,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x03C9,0x03AC,0x03AD,0x03AE,0x03CA,0x03AF,0x03CC,0x03CD,
		0x03CB,0x03CE,0x0386,0x0388,0x0389,0x038A,0x038C,0x038E,
		0x038F,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,0x2248,
		0x00B0,0x00A3,0x0387,0x221A,0x207F,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_850
: public Encoding
{
public:
	cp_850() : Encoding("cp-850") {}
	int map2unicode( int code);
} exemplar_cp_850;

int cp_850::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x00E0,0x00E5,0x00E7,
		0x00EA,0x00EB,0x00E8,0x00EF,0x00EE,0x00EC,0x00C4,0x00C5,
		0x00C9,0x00E6,0x00C6,0x00F4,0x00F6,0x00F2,0x00FB,0x00F9,
		0x00FF,0x00D6,0x00DC,0x00F8,0x00A3,0x00D8,0x00D7,0x0192,
		0x00E1,0x00ED,0x00F3,0x00FA,0x00F1,0x00D1,0x00AA,0x00BA,
		0x00BF,0x00AE,0x00AC,0x00BD,0x00BC,0x00A1,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x00C1,0x00C2,0x00C0,
		0x00A9,0x2563,0x2551,0x2557,0x255D,0x00A2,0x00A5,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x00E3,0x00C3,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x00A4,
		0x00F0,0x00D0,0x00CA,0x00CB,0x00C8,0x0131,0x00CD,0x00CE,
		0x00CF,0x2518,0x250C,0x2588,0x2584,0x00A6,0x00CC,0x2580,
		0x00D3,0x00DF,0x00D4,0x00D2,0x00F5,0x00D5,0x00B5,0x00FE,
		0x00DE,0x00DA,0x00DB,0x00D9,0x00FD,0x00DD,0x00AF,0x00B4,
		0x00AD,0x00B1,0x2017,0x00BE,0x00B6,0x00A7,0x00F7,0x00B8,
		0x00B0,0x00A8,0x00B7,0x00B9,0x00B3,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_851
: public Encoding
{
public:
	cp_851() : Encoding("cp-851") {}
	int map2unicode( int code);
} exemplar_cp_851;

int cp_851::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x00E0,0x0386,0x00E7,
		0x00EA,0x00EB,0x00E8,0x00EF,0x00EE,0x0388,0x00C4,0x0389,
		0x038A,0,0x038C,0x00F4,0x00F6,0x038E,0x00FB,0x00F9,
		0x038F,0x00D6,0x00DC,0x03AC,0x00A3,0x03AD,0x03AE,0x03AF,
		0x03CA,0x0390,0x03CC,0x03CD,0x0391,0x0392,0x0393,0x0394,
		0x0395,0x0396,0x0397,0x00BD,0x0398,0x0399,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x039A,0x039B,0x039C,
		0x039D,0x2563,0x2551,0x2557,0x255D,0x039E,0x039F,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x03A0,0x03A1,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x03A3,
		0x03A4,0x03A5,0x03A6,0x03A7,0x03A8,0x03A9,0x03B1,0x03B2,
		0x03B3,0x2518,0x250C,0x2588,0x2584,0x03B4,0x03B5,0x2580,
		0x03B6,0x03B7,0x03B8,0x03B9,0x03BA,0x03BB,0x03BC,0x03BD,
		0x03BE,0x03BF,0x03C0,0x03C1,0x03C3,0x03C2,0x03C4,0x0384,
		0x00AD,0x00B1,0x03C5,0x03C6,0x03C7,0x00A7,0x03C8,0x0385,
		0x00B0,0x00A8,0x03C9,0x03CB,0x03B0,0x03CE,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_852
: public Encoding
{
public:
	cp_852() : Encoding("cp-852") {}
	int map2unicode( int code);
} exemplar_cp_852;

int cp_852::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x016F,0x0107,0x00E7,
		0x0142,0x00EB,0x0150,0x0151,0x00EE,0x0179,0x00C4,0x0106,
		0x00C9,0x0139,0x013A,0x00F4,0x00F6,0x013D,0x013E,0x015A,
		0x015B,0x00D6,0x00DC,0x0164,0x0165,0x0141,0x00D7,0x010D,
		0x00E1,0x00ED,0x00F3,0x00FA,0x0104,0x0105,0x017D,0x017E,
		0x0118,0x0119,0,0x017A,0x010C,0x015F,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x00C1,0x00C2,0x011A,
		0x015E,0x2563,0x2551,0x2557,0x255D,0x017B,0x017C,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x0102,0x0103,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x00A4,
		0x0111,0x0110,0x010E,0x00CB,0x010F,0x0147,0x00CD,0x00CE,
		0x011B,0x2518,0x250C,0x2588,0x2584,0x0162,0x016E,0x2580,
		0x00D3,0x00DF,0x00D4,0x0143,0x0144,0x0148,0x0160,0x0161,
		0x0154,0x00DA,0x0155,0x0170,0x00FD,0x00DD,0x0163,0x00B4,
		0x00AD,0x02DD,0x02DB,0x02C7,0x02D8,0x00A7,0x00F7,0x00B8,
		0x00B0,0x00A8,0x02D9,0x0171,0x0158,0x0159,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_853
: public Encoding
{
public:
	cp_853() : Encoding("cp-853") {}
	int map2unicode( int code);
} exemplar_cp_853;

int cp_853::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x00E0,0x0109,0x00E7,
		0x00EA,0x00EB,0x00E8,0x00EF,0x00EE,0x00EC,0x00C4,0x0108,
		0x00C9,0x010B,0x010A,0x00F4,0x00F6,0x00F2,0x00FB,0x00F9,
		0x0130,0x00D6,0x00DC,0x011D,0x00A3,0x011C,0x00D7,0x0135,
		0x00E1,0x00ED,0x00F3,0x00FA,0x00F1,0x00D1,0x011E,0x011F,
		0x0124,0x0125,0,0x00BD,0x0134,0x015F,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x00C1,0x00C2,0x00C0,
		0x015E,0x2563,0x2551,0x2557,0x255D,0x017B,0x017C,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x015C,0x015D,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x00A4,
		0,0,0x00CA,0x00CB,0x00C8,0x0131,0x00CD,0x00CE,
		0x00CF,0x2518,0x250C,0x2588,0x2584,0,0x00CC,0x2580,
		0x00D3,0x00DF,0x00D4,0x00D2,0x0120,0x0121,0x00B5,0x0126,
		0x0127,0x00DA,0x00DB,0x00D9,0x016C,0x016D,0,0x00B4,
		0x00AD,0,0x2113,0x0149,0x02D8,0x00A7,0x00F7,0x00B8,
		0x00B0,0x00A8,0x02D9,0,0x00B3,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_855
: public Encoding
{
public:
	cp_855() : Encoding("cp-855") {}
	int map2unicode( int code);
} exemplar_cp_855;

int cp_855::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0452,0x0402,0x0453,0x0403,0x0451,0x0401,0x0454,0x0404,
		0x0455,0x0405,0x0456,0x0406,0x0457,0x0407,0x0458,0x0408,
		0x0459,0x0409,0x045A,0x040A,0x045B,0x040B,0x045C,0x040C,
		0x045E,0x040E,0x045F,0x040F,0x044E,0x042E,0x044A,0x042A,
		0x0430,0x0410,0x0431,0x0411,0x0446,0x0426,0x0434,0x0414,
		0x0435,0x0415,0x0444,0x0424,0x0433,0x0413,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x0445,0x0425,0x0438,
		0x0418,0x2563,0x2551,0x2557,0x255D,0x0439,0x0419,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x043A,0x041A,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x00A4,
		0x043B,0x041B,0x043C,0x041C,0x043D,0x041D,0x043E,0x041E,
		0x043F,0x2518,0x250C,0x2588,0x2584,0x041F,0x044F,0x2580,
		0x042F,0x0440,0x0420,0x0441,0x0421,0x0442,0x0422,0x0443,
		0x0423,0x0436,0x0416,0x0432,0x0412,0x044C,0x042C,0x2116,
		0x00AD,0x044B,0x042B,0x0437,0x0417,0x0448,0x0428,0x044D,
		0x042D,0x0449,0x0429,0x0447,0x0427,0,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_857
: public Encoding
{
public:
	cp_857() : Encoding("cp-857") {}
	int map2unicode( int code);
} exemplar_cp_857;

int cp_857::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x00E0,0x00E5,0x00E7,
		0x00EA,0x00EB,0x00E8,0x00EF,0x00EE,0x0131,0x00C4,0x00C5,
		0x00C9,0x00E6,0x00C6,0x00F4,0x00F6,0x00F2,0x00FB,0x00F9,
		0x0130,0x00D6,0x00DC,0x00F8,0x00A3,0x00D8,0x015E,0x015F,
		0x00E1,0x00ED,0x00F3,0x00FA,0x00F1,0x00D1,0x011E,0x011F,
		0x00BF,0x00AE,0x00AC,0x00BD,0x00BC,0x00A1,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x00C1,0x00C2,0x00C0,
		0x00A9,0x2563,0x2551,0x2557,0x255D,0x00A2,0x00A5,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x00E3,0x00C3,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x00A4,
		0x00BA,0x00AA,0x00CA,0x00CB,0x00C8,0,0x00CD,0x00CE,
		0x00CF,0x2518,0x250C,0x2588,0x2584,0x00A6,0x00CC,0x2580,
		0x00D3,0x00DF,0x00D4,0x00D2,0x00F5,0x00D5,0x00B5,0x00FE,
		0x00D7,0x00DA,0x00DB,0x00D9,0x00EC,0x00FF,0x00AF,0x00B4,
		0x00AD,0x00B1,0x2017,0x00BE,0x00B6,0x00A7,0x00F7,0x00B8,
		0x00B0,0x00A8,0x00B7,0x00B9,0x00B3,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_860
: public Encoding
{
public:
	cp_860() : Encoding("cp-860") {}
	int map2unicode( int code);
} exemplar_cp_860;

int cp_860::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E3,0x00E0,0x00C1,0x00E7,
		0x00EA,0x00CA,0x00E8,0x00CD,0x00D4,0x00EC,0x00C3,0x00C2,
		0x00C9,0x00C0,0x00C8,0x00F4,0x00F5,0x00F2,0x00DA,0x00F9,
		0x00CC,0x00D5,0x00DC,0x00A2,0x00A3,0x00D9,0x20A7,0x00D3,
		0x00E1,0x00ED,0x00F3,0x00FA,0x00F1,0x00D1,0x00AA,0x00BA,
		0x00BF,0x00D2,0x00AC,0x00BD,0x00BC,0x00A1,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x03B1,0x00DF,0x0393,0x03C0,0x2211,0x03C3,0x00B5,0x03C4,
		0x03A6,0x0398,0x2126,0x03B4,0x221E,0x2205,0x2208,0x2229,
		0x2261,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,0x2248,
		0x00B0,0x2219,0x00B7,0x221A,0x207F,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_861
: public Encoding
{
public:
	cp_861() : Encoding("cp-861") {}
	int map2unicode( int code);
} exemplar_cp_861;

int cp_861::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x00E0,0x00E5,0x00E7,
		0x00EA,0x00EB,0x00E8,0x00D0,0x00F0,0x00DE,0x00C4,0x00C5,
		0x00C9,0x00E6,0x00C6,0x00F4,0x00F6,0x00FE,0x00FB,0x00F9,
		0x00FF,0x00D6,0x00DC,0x00F8,0x00A3,0x00D8,0x20A7,0x0192,
		0x00E1,0x00ED,0x00F3,0x00FA,0x00C1,0x00CD,0x00D3,0x00DA,
		0x00BF,0x2310,0x00AC,0x00BD,0x00BC,0x00A1,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x03B1,0x00DF,0x0393,0x03C0,0x2211,0x03C3,0x00B5,0x03C4,
		0x03A6,0x0398,0x2126,0x03B4,0x221E,0x2205,0x2208,0x2229,
		0x2261,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,0x2248,
		0x00B0,0x2219,0x00B7,0x221A,0x207F,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_862
: public Encoding
{
public:
	cp_862() : Encoding("cp-862") {}
	int map2unicode( int code);
} exemplar_cp_862;

int cp_862::map2unicode( int code)
{
	static unsigned short table[] = {
		0x05D0,0x05D1,0x05D2,0x05D3,0x05D4,0x05D5,0x05D6,0x05D7,
		0x05D8,0x05D9,0x05DA,0x05DB,0x05DC,0x05DD,0x05DE,0x05DF,
		0x05E0,0x05E1,0x05E2,0x05E3,0x05E4,0x05E5,0x05E6,0x05E7,
		0x05E8,0x05E9,0x05EA,0x00A2,0x00A3,0x00A5,0x20A7,0x0192,
		0x00E1,0x00ED,0x00F3,0x00FA,0x00F1,0x00D1,0x00AA,0x00BA,
		0x00BF,0x2310,0x00AC,0x00BD,0x00BC,0x00A1,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x03B1,0x00DF,0x0393,0x03C0,0x2211,0x03C3,0x00B5,0x03C4,
		0x03A6,0x0398,0x2126,0x03B4,0x221E,0x2205,0x2208,0x2229,
		0x2261,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,0x2248,
		0x00B0,0x2219,0x00B7,0x221A,0x207F,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_863
: public Encoding
{
public:
	cp_863() : Encoding("cp-863") {}
	int map2unicode( int code);
} exemplar_cp_863;

int cp_863::map2unicode( int code)
{
	static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00C2,0x00E0,0x00B6,0x00E7,
		0x00EA,0x00EB,0x00E8,0x00EF,0x00EE,0x2017,0x00C0,0x00A7,
		0x00C9,0x00C8,0x00CA,0x00F4,0x00CB,0x00CF,0x00FB,0x00F9,
		0x00A4,0x00D4,0x00DC,0x00A2,0x00A3,0x00D9,0x00DB,0x0192,
		0x00A6,0x00B4,0x00F3,0x00FA,0x00A8,0x00B8,0x00B3,0x00AF,
		0x00CE,0x2310,0x00AC,0x00BD,0x00BC,0x00BE,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x03B1,0x00DF,0x0393,0x03C0,0x2211,0x03C3,0x00B5,0x03C4,
		0x03A6,0x0398,0x2126,0x03B4,0x221E,0x2205,0x2208,0x2229,
		0x2261,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,0x2248,
		0x00B0,0x2219,0x00B7,0x221A,0x207F,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_864
: public Encoding
{
public:
	cp_864() : Encoding("cp-864") {}
	int map2unicode( int code);
} exemplar_cp_864;

int cp_864::map2unicode( int code)
{
static unsigned short table[] = {
		0x00B0,0x00B7,0x2219,0x221A,0x2592,0x2500,0x2502,0x253C,
		0x2524,0x252C,0x251C,0x2534,0x2510,0x250C,0x2514,0x2518,
		0x00DF,0x221E,0x03A6,0x00B1,0x00BD,0x00BC,0x2248,0x00AB,
		0x00BB,0,0,0,0,0,0,0,
		0x00A0,0x00AD,0,0x00A3,0x00A4,0,0,0,
		0,0x0628,0x062A,0x062B,0x060C,0x062C,0x062D,0x062E,
		0x0660,0x0661,0x0662,0x0663,0x0664,0x0665,0x0666,0x0667,
		0x0668,0x0669,0x0641,0x061B,0x0633,0x0634,0x0635,0x061F,
		0x00A2,0x0621,0x0622,0x0623,0x0624,0,0,0x0627,
		0,0x0629,0,0,0,0,0,0x062F,
		0x0630,0x0631,0x0632,0,0,0,0,0x0637,
		0x0638,0,0,0x00A6,0x00AC,0x00F7,0x00D7,0x0639,
		0x0640,0,0,0,0,0,0,0,
		0x0648,0x0649,0,0x0636,0,0,0x063A,0x0645,
		0,0x0651,0x0646,0x0647,0,0,0,0,
		0x0642,0,0,0x0644,0x0643,0x064A,0x25A0,0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_865
: public Encoding
{
public:
	cp_865() : Encoding("cp-865") {}
	int map2unicode( int code);
} exemplar_cp_865;

int cp_865::map2unicode( int code)
{
static unsigned short table[] = {
		0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x00E0,0x00E5,0x00E7,
		0x00EA,0x00EB,0x00E8,0x00EF,0x00EE,0x00EC,0x00C4,0x00C5,
		0x00C9,0x00E6,0x00C6,0x00F4,0x00F6,0x00F2,0x00FB,0x00F9,
		0x00FF,0x00D6,0x00DC,0x00F8,0x00A3,0x00D8,0x20A7,0x0192,
		0x00E1,0x00ED,0x00F3,0x00FA,0x00F1,0x00D1,0x00AA,0x00BA,
		0x00BF,0x2310,0x00AC,0x00BD,0x00BC,0x00A1,0x00AB,0x00A4,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x03B1,0x00DF,0x0393,0x03C0,0x2211,0x03C3,0x00B5,0x03C4,
		0x03A6,0x0398,0x2126,0x03B4,0x221E,0x2205,0x2208,0x2229,
		0x2261,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,0x2248,
		0x00B0,0x2219,0x00B7,0x221A,0x207F,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_866
: public Encoding
{
public:
	cp_866() : Encoding("cp-866") {}
	int map2unicode( int code);
} exemplar_cp_866;

int cp_866::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0410,0x0411,0x0412,0x0413,0x0414,0x0415,0x0416,0x0417,
		0x0418,0x0419,0x041A,0x041B,0x041C,0x041D,0x041E,0x041F,
		0x0420,0x0421,0x0422,0x0423,0x0424,0x0425,0x0426,0x0427,
		0x0428,0x0429,0x042A,0x042B,0x042C,0x042D,0x042E,0x042F,
		0x0430,0x0431,0x0432,0x0433,0x0434,0x0435,0x0436,0x0437,
		0x0438,0x0439,0x043A,0x043B,0x043C,0x043D,0x043E,0x043F,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x0440,0x0441,0x0442,0x0443,0x0444,0x0445,0x0446,0x0447,
		0x0448,0x0449,0x044A,0x044B,0x044C,0x044D,0x044E,0x044F,
		0x0401,0x0451,0x0404,0x0454,0x0407,0x0457,0x040E,0x045E,
		0x00B0,0x2219,0x00B7,0x221A,0x2116,0x00A4,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_869
: public Encoding
{
public:
	cp_869() : Encoding("cp-869") {}
	int map2unicode( int code);
} exemplar_cp_869;

int cp_869::map2unicode( int code)
{
	static unsigned short table[] = {
		0,0,0,0,0,0,0x0386,0,
		0x0387,0x00AC,0x00A6,0x2018,0x2019,0x0388,0x2015,0x0389,
		0x038A,0x03AA,0x038C,0,0,0x038E,0x03AB,0x00A9,
		0x038F,0x00B2,0x00B3,0x03AC,0x00A3,0x03AD,0x03AE,0x03AF,
		0x03CA,0x0390,0x03CC,0x03CD,0x0391,0x0392,0x0393,0x0394,
		0x0395,0x0396,0x0397,0x00BD,0x0398,0x0399,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x039A,0x039B,0x039C,
		0x039D,0x2563,0x2551,0x2557,0x255D,0x039E,0x039F,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x03A0,0x03A1,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x03A3,
		0x03A4,0x03A5,0x03A6,0x03A7,0x03A8,0x03A9,0x03B1,0x03B2,
		0x03B3,0x2518,0x250C,0x2588,0x2584,0x03B4,0x03B5,0x2580,
		0x03B6,0x03B7,0x03B8,0x03B9,0x03BA,0x03BB,0x03BC,0x03BD,
		0x03BE,0x03BF,0x03C0,0x03C1,0x03C3,0x03C2,0x03C4,0x0384,
		0x00AD,0x00B1,0x03C5,0x03C6,0x03C7,0x00A7,0x03C8,0x0385,
		0x00B0,0x00A8,0x03C9,0x03CB,0x03B0,0x03CE,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class cp_895
: public Encoding
{
public:
   cp_895() : Encoding("cp-895") {}
   int map2unicode( int code);
} exemplar_cp_895;

int cp_895::map2unicode( int code)
{
	static unsigned short table[] = {
		0x010C,0x00FC,0x00E9,0x010F,0x00E4,0x010E,0x0164,0x010D,
		0x011B,0x011A,0x0139,0x00CD,0x013E,0x013A,0x00C4,0x00C1,
		0x00C9,0x017E,0x017D,0x00F4,0x00F6,0x00D3,0x016F,0x00DA,
		0x00FD,0x00D6,0x00DC,0x0160,0x013D,0x00DD,0x0158,0x0165,
		0x00E1,0x00ED,0x00F3,0x00FA,0x0148,0x0147,0x016E,0x00D4,
		0x0161,0x0159,0x0155,0x0154,0x00BC,0x00A1,0x00AB,0x00BB,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,
		0x2555,0x2563,0x2551,0x2557,0x255D,0x255C,0x255B,0x2510,
		0x2514,0x2534,0x252C,0x251C,0x2500,0x253C,0x255E,0x255F,
		0x255A,0x2554,0x2569,0x2566,0x2560,0x2550,0x256C,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256B,
		0x256A,0x2518,0x250C,0x2588,0x2584,0x258C,0x2590,0x2580,
		0x03B1,0x00DF,0x0393,0x03C0,0x2211,0x03C3,0x00B5,0x03C4,
		0x03A6,0x0398,0x2126,0x03B4,0x221E,0x2205,0x2208,0x2229,
		0x2261,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,0x2248,
		0x00B0,0x2219,0x00B7,0x221A,0x207F,0x00B2,0x25A0,0x00A0
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}


//---
class wingreek
: public Encoding
{
public:
   wingreek() : Encoding("wingreek-0") {}
   int map2unicode( int code);
} exemplar_wingreek;

int wingreek::map2unicode( int code)
{
	static unsigned short table[] = {
		0,0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,
		0x0007,0x0008,0x0009,0x000A,0x000B,0x000C,0x000D,0x000E,
		0x000F,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,
		0x0017,0x0018,0x0019,0x001A,0x001B,0x001C,0x001D,0x001E,
		0x001F,0x0020,0,0x03E0,0x03DC,0x03DA,0x03DE,0,
		0x1FBF,0x0028,0x0029,0x002A,0,0x002C,0x002D,0x002E,
		0x1FFD,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,
		0x0037,0x0038,0x0039,0x0387,0x037E,0x003C,0x003D,0x003E,
		0x003F,0x00A8,0x0391,0x0392,0x03A7,0x0394,0x0395,0x03A6,
		0x0393,0x0397,0x0399,0x1FF3,0x039A,0x039B,0x039C,0x039D,
		0x039F,0x03A0,0x0398,0x03A1,0x03A3,0x03A4,0x03A5,0x1FC3,
		0x03A9,0x039E,0x03A8,0x0396,0x005B,0x1FEF,0x005D,0x1FC0,
		0x005F,0x1FFE,0x03B1,0x03B2,0x03C7,0x03B4,0x03B5,0x03C6,
		0x03B3,0x03B7,0x03B9,0x03C2,0x03BA,0x03BB,0x03BC,0x03BD,
		0x03BF,0x03C0,0x03B8,0x03C1,0x03C3,0x03C4,0x03C5,0x1FB3,
		0x03C9,0x03BE,0x03C8,0x03B6,0x007B,0x007C,0x007D,0x007E,
		0x007F,0,0,0,0x1F31,0x1F30,0x1F77,0x1F35,
		0x1F34,0x1F76,0x1F35,0x1F32,0x1FD6,0x1F37,0x1F36,0x03CA,
		0x1FD3,0x1FD2,0x1FDF,0x1FCF,0x1FDE,0x1FCE,0x1FDD,0x1FCD,
		0x1FEE,0x1F11,0x1F10,0x1F73,0x1F15,0x1F14,0x1F72,0x1F13,
		0x1F12,0x1F84,0x1F01,0x1F00,0x1F71,0x1F05,0x1F04,0x1F70,
		0x1F03,0x1F02,0x1FB6,0x1F07,0x1F06,0x1F81,0x1F80,0x1FB4,
		0x1F85,0x1F84,0x1FB2,0x1F83,0x1F83,0x1FB7,0x1F87,0x1F86,
		0x1FE5,0x1FE4,0x1F21,0x1F20,0x1F75,0x1F25,0x1F24,0x1F74,
		0x1F23,0x1F22,0x1FC6,0x1F27,0x1F26,0x037A,0x1F91,0x1F90,
		0x1FC4,0x1F95,0x1F94,0x1FC2,0x1F93,0x1F92,0x1FC7,0x1F97,
		0x1F96,0x1F41,0x1F40,0x1F79,0x1F45,0x1F44,0x1F78,0x1F43,
		0x1F42,0x1F51,0x1F50,0x1F7B,0x1F55,0x1F54,0x1F7A,0x1F53,
		0x1F52,0x1FE6,0x1F57,0x1F56,0x03CB,0x1FE3,0x1FE2,0x1F61,
		0x1F60,0x1F7D,0x1F65,0x1F64,0x1F7C,0x1F63,0x1F62,0x1FF6,
		0x1F67,0x1F66,0x1FA1,0x1FA0,0x1FF4,0x1FA5,0x1FA4,0x1FF2,
		0x1FA3,0x1FA2,0x1FF7,0x1FA7,0x1FA6,0,0,0
	};

	if( code >= 256) return code;
	return table[ code];
}


//---
class symbol
: public Encoding
{
public:
   symbol() : Encoding("symbol-0") {}
   int map2unicode( int code);
} exemplar_symbol;

int symbol::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,
		0x0008,0x0009,0x000A,0x000B,0x000C,0x000D,0x000E,0x000F,
		0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,
		0x0018,0x0019,0x001A,0x001B,0x001C,0x001D,0x001E,0x001F,
		0x0020,0x0021,0x2200,0x0023,0x2203,0x0025,0x0026,0x220B,
		0x0028,0x0029,0x2217,0x002B,0x002C,0x2212,0x002E,0x002F,
		0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,
		0x0038,0x0039,0x003A,0x003B,0x003C,0x003D,0x003E,0x003F,
		0x2245,0x0391,0x0392,0x03A7,0x0394,0x0395,0x03A6,0x0393,
		0x0397,0x0399,0x03D1,0x039A,0x039B,0x039C,0x039D,0x039F,
		0x03A0,0x0398,0x03A1,0x03A3,0x03A4,0x03A5,0x03C2,0x03A9,
		0x039E,0x03A8,0x0396,0x005B,0x2234,0x005D,0x22A5,0x005F,
		0x203E,0x03B1,0x03B2,0x03C7,0x03B4,0x03B5,0x03C6,0x03B3,
		0x03B7,0x03B9,0x03D5,0x03BA,0x03BB,0x03BC,0x03BD,0x03BF,
		0x03C0,0x03B8,0x03C1,0x03C3,0x03C4,0x03C5,0x03D6,0x03C9,
		0x03BE,0x03C8,0x03B6,0x007B,0x007C,0x007D,0x223C,0x007F,
		0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,
		0x0088,0x0089,0x008A,0x008B,0x008C,0x008D,0x008E,0x008F,
		0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,
		0x0098,0x0099,0x009A,0x009B,0x009C,0x009D,0x009E,0x009F,
		0,0x03D2,0x2032,0x2264,0x2215,0x221E,0x0192,0x2663,
		0x2666,0x2665,0x2660,0x2194,0x2190,0x2191,0x2192,0x2193,
		0x00B0,0x00B1,0x2033,0x2265,0x00D7,0x221D,0x2202,0x2219,
		0x00F7,0x2260,0x2261,0x2248,0x22EF,0,0,0x21B5,
		0,0x2111,0x211C,0x2118,0x2297,0x2295,0x2205,0x2229,
		0x222A,0x2283,0x2287,0x2284,0x2282,0x2286,0x2208,0x2209,
		0x2220,0x2207,0x00AE,0x00A9,0x2122,0x220F,0x221A,0x22C5,
		0x00AC,0x2227,0x2228,0x21D4,0x21D0,0x21D1,0x21D2,0x21D3,
		0x25CA,0x2329,0,0,0,0x2211,0,0,
		0,0,0,0,0,0,0,0,
		0,0x232A,0x222B,0x2320,0,0x2321,0,0,
		0,0,0,0,0,0,0,0
	};

	if( code >= 256) return code;
	return table[ code];
}


//---
class wingding
: public Encoding
{
public:
   wingding() : Encoding("wingding-0") {}
   int map2unicode( int code);
} exemplar_wingding;

int wingding::map2unicode( int code)
{
	static unsigned short table[] = {
		0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,
		0x0008,0x0009,0x000A,0x000B,0x000C,0x000D,0x000E,0x000F,
		0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,
		0x0018,0x0019,0x001A,0x001B,0x001C,0x001D,0x001E,0x001F,
		0x0020,0x270F,0x2702,0x2701,0,0,0,0,
		0x260E,0x2706,0x2709,0,0,0,0,0,
		0,0,0,0,0,0,0,0x2328,
		0,0,0,0,0,0,0x2707,0x270D,
		0,0x270C,0,0,0,0x261C,0x261E,0x261D,
		0x261F,0,0x263A,0,0x2639,0,0x2620,0,
		0,0x2708,0x263C,0,0x2744,0x271D,0x271E,0,
		0x2720,0x2721,0x262A,0x262F,0,0x2638,0x2648,0x2649,
		0x264A,0x264B,0x264C,0x264D,0x264E,0x264F,0x2650,0x2651,
		0x2652,0x2653,0,0,0x25CF,0,0x25A0,0,
		0,0x2751,0x2752,0,0,0x25C6,0x2756,0,
		0x2327,0,0x2318,0x2740,0x273F,0x275D,0x275E,0,
		0,0x2780,0x2781,0x2782,0x2783,0x2784,0x2785,0x2786,
		0x2787,0x2788,0x2789,0,0x278A,0x278B,0x278C,0x278D,
		0x278E,0x278F,0x2790,0x2791,0x2792,0x2793,0,0,
		0,0,0,0,0,0,0,0,
		0,0x25CB,0,0,0,0,0,0,
		0x25A1,0,0x2726,0x2605,0x2736,0x2738,0x2739,0x2735,
		0,0x2316,0x2727,0x2311,0x2370,0x272A,0x2730,0,
		0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,
		0,0,0,0,0,0x232B,0x2326,0,
		0x27A2,0,0,0,0,0,0,0x2190,
		0x2192,0x2191,0x2193,0x2196,0x2197,0x2199,0x2198,0,
		0,0,0,0,0,0,0,0x21E6,
		0x21E8,0x21E7,0x21E9,0,0,0,0,0,
		0,0,0,0x2718,0x2714,0,0,0
	};

	if( code >= 256) return code;
	return table[ code];
}


//---
class atari_st
: public Encoding
{
public:
   atari_st() : Encoding("atari-st") {}
   int map2unicode( int code);
} exemplar_atarist;

int atari_st::map2unicode( int code)
{
   static unsigned short table[] = {
	0,0x00C7,0x00FC,0x00E9,0x00E2,0x00E4,0x00E0,0x00E5,
	0x00E7,0x00EA,0x00EB,0x00E8,0x00EF,0x00EE,0x00EC,0x00C4,
	0x00C5,0x00C9,0x00E6,0x00C6,0x00F4,0x00F6,0x00F2,0x00FB,
	0x00F9,0x00FF,0x00D6,0x00DC,0x00A2,0x00A3,0x00A5,0x00DF,
	0x0192,0x00E1,0x00ED,0x00F3,0x00FA,0x00F1,0x00D1,0x00AA,
	0x00BA,0x00BF,0x2310,0x00AC,0x00BD,0x00BC,0x00A1,0x00AB,
	0x00BB,0x00E3,0x00F5,0x00D8,0x00F8,0x0153,0x0152,0x00C0,
	0x00C3,0x00D5,0x00A8,0x00B4,0x2020,0x00B6,0x00A9,0x00AE,
	0x2122,0x0133,0x0132,0x05D0,0x05D1,0x05D2,0x05D3,0x05D4,
	0x05D5,0x05D6,0x05D7,0x05D8,0x05D9,0x05DB,0x05DC,0x05DE,
	0x05E0,0x05E1,0x05E2,0x05E4,0x05E6,0x05E7,0x05E8,0x05E9,
	0x05EA,0x05DF,0x05DA,0x05DD,0x05E3,0x05E5,0x00A7,0x2227,
	0x221E,0x03B1,0x03B2,0x0393,0x03C0,0x2211,0x03C3,0x00B5,
	0x03C4,0x03A6,0x0398,0x2126,0x03B4,0x222E,0x2205,0x2208,
	0x2229,0x2261,0x00B1,0x2265,0x2264,0x2320,0x2321,0x00F7,
	0x2248,0x00B0,0x2219,0x00B7,0x221A,0x207F,0x00B2,0x00B3,
	0x00AF   };

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}

//--- thanks Gilbert Baumann
class unicode_2
: public Encoding
{
public:
	unicode_2() : Encoding("unicode-2") {}
	int map2unicode( int code) { return code;}
} exemplar_unicode_2;

//--- thanks Mindaugas Riauba
class windows_1257
: public Encoding
{
public:
	windows_1257() : Encoding("windows-1257") {}
	int map2unicode( int code);
} exemplar_windows_1257;

int windows_1257::map2unicode( int code)
{
	static unsigned short table[] = {
		0,0,0x201a,0,0x201e,0x2026,0x2020,0x2021,
		0,0x2030,0,0x2039,0,0x00a8,0x02c7,0x00b8,
		0,0x2018,0x2019,0x201c,0x201d,0x2022,0x2013,0x2014,
		0,0x2122,0,0x203a,0,0x00af,0x02db,0,
		0x00a0,0xf8fc,0x00a2,0x00a3,0x00a4,0xf8fd,0x00a6,0x00a7,
		0x00d8,0x00a9,0x0156,0x00ab,0x00ac,0x00ad,0x00ae,0x00c6,
		0x00b0,0x00b1,0x00b2,0x00b3,0x00b4,0x00b5,0x00b6,0x00b7,
		0x00f8,0x00b9,0x0157,0x00bb,0x00bc,0x00bd,0x00be,0x00e6,
		0x0104,0x012e,0x0100,0x0106,0x00c4,0x00c5,0x0118,0x0112,
		0x010c,0x00c9,0x0179,0x0116,0x0122,0x0136,0x012a,0x013b,
		0x0160,0x0143,0x0145,0x00d3,0x014c,0x00d5,0x00d6,0x00d7,
		0x0172,0x0141,0x015a,0x016a,0x00dc,0x017b,0x017d,0x00df,
		0x0105,0x012f,0x0101,0x0107,0x00e4,0x00e5,0x0119,0x0113,
		0x010d,0x00e9,0x017a,0x0117,0x0123,0x0137,0x012b,0x013c,
		0x0161,0x0144,0x0146,0x00f3,0x014d,0x00f5,0x00f6,0x00f7,
		0x0173,0x0142,0x015b,0x016b,0x00fc,0x017c,0x017e,0x02d9
	};

	if( code < 128 || code >= 256) return code;
	return table[ code - 128];
}

void Encoding::getDefault( Encoding** maps, int maxcodes)
{
	maps[ 0] = &exemplar_iso8859_1;
	if( maxcodes <= 1) return;
	maps[ 1] = 0;
}

