// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: graphics.h,v 1.22 1998/05/13 01:07:30 jgg Exp $
/* ######################################################################
   
   Graphics - The graphics System
   GenGC - Generic Graphics Context
   TextGC - Textual GC (Character cells)
   GraphicsGC - Graphical GC
   
   This provides the basic abstraction of Output from a widget -- or 
   more generally an implementation defined region with clip and offset
   rectangles.
   
   GenGC handles the general graphics context. This extends to include 
   region management. Although regions technically do not fit inside
   a GC this is a convenient place to manipulate them from.

   The Widget class will interface with the GenGC::GC to manage regions 
   for the widgets when they are realized and so on. (See widget.h for a
   discussion of regions). Widget deals exclusively with GenGC, and
   makes sure it is in the proper state before it calls the widgets
   Render Function.

   The Static GC members inside each GC class represent the default
   GC. This is ment for use in non-render functions like extent.
   Widget may provide a different GC to the render function.

   CombinedGC is a way to pass all 3 of the GC's to a single function,
   operator ->() is overloaded to return the general gc. 
   
   Font selection is done to match the X system, we can specify a face,
   a weight, a slant and a point size. It will pick the best match,
   trying some backup faces if the main face is not present.
   
   16 colour constants are provided to match the normal Text Mode colours,
   these are coded to operate properly in graphics mode as well.
   
   ###################################################################### */
									/*}}}*/
// Header section: deity
#ifndef DEITY_GRAPHICS_H
#define DEITY_GRAPHICS_H

#include <string>
#include <iostream.h>

class Widget;

// Usefull structures
struct Point
{
   long x;
   long y;

   bool operator ==(const Point &rhs) {return rhs.x == x && rhs.y == y;};
   Point(long x,long y) : x(x), y(y) {};
   Point();
};
ostream &operator << (ostream &stream,const Point &P);

// Point and width/height rectangle
struct Rect
{
   long x;
   long y;
   long w;
   long h;

   void Clip(Rect Other);
   bool Inside(Point P);
   bool Inside(Rect R);
   inline bool operator ==(const Rect &rhs) {return x == rhs.x && y == rhs.y &&
	 w == rhs.w && h == rhs.h;};

   Rect(long x,long y,long w,long h) : x(x), y(y), w(w), h(h) {};
   Rect();
};
ostream &operator << (ostream &stream,const Rect &P);

// Rectangle with four absolute coordinates
struct AbsRect
{
   long x1;
   long y1;
   long x2;
   long y2;

   inline operator Rect() const {return Rect(x1,y1,x2-x1,y2-y1);};
   
   AbsRect(long x1,long y1,long x2,long y2) : x1(x1), y1(y1), x2(x2), y2(y2) {};
   AbsRect(const Rect &R) : x1(R.x), y1(R.y), x2(R.x + R.w), y2(R.y + R.h) {};
   AbsRect();
};
ostream &operator << (ostream &stream,const AbsRect &P);

// Generic colour value
class Color
{
   public:

   enum TextColor
   {
      Black = 1, Red, Green, Brown, Blue, Magenta, Cyan, LightGray, Gray, 
      BrightRed, BrightGree, Yellow, BrightBlue, BrightMagenta, BrightCyan,
      White, None = 0xFF
   };
   
   unsigned char RedLevel;
   unsigned char GreenLevel;
   unsigned char BlueLevel;
   TextColor Text;

   bool operator ==(const Color &rhs) const 
        {return RedLevel == rhs.RedLevel && GreenLevel == rhs.GreenLevel &&
	      BlueLevel == rhs.BlueLevel && Text == rhs.Text;};
   inline Color(unsigned char R,unsigned char G, unsigned char B, 
		TextColor T) : RedLevel(R), GreenLevel(G), BlueLevel(B), Text(T)
	        {};
   Color();
};

// Simplistic view of a font
class SimpleFont
{
   public:

   enum WeightT
   {
      Any = 0, Bold = 1, Medium, Regular, DemiBold, Black, Normal = Medium
   };
   
   // Any is also a valid slant
   enum SlantT
   {
      Upright = 1, Italic
   };
   
   string Face;
   unsigned long Weight;
   unsigned long Slant;
   unsigned long Points;     // 120 = 12pt

   inline bool operator ==(const SimpleFont &rhs) {return Face == rhs.Face && 
	 Slant == rhs.Slant && Points == rhs.Points && Weight == rhs.Weight;};
   
   SimpleFont(string Face,unsigned long Points,
	      WeightT Weight = Normal,SlantT Slant = Upright) : Face(Face), 
              Weight(Weight), Slant(Slant), Points(Points) {};
   SimpleFont();
};

// Colour defines
#define Wc_Black Color(0,0,0,Color::Black)
#define Wc_Red Color(128,0,0,Color::Red)
#define Wc_Green Color(0,128,0,Color::Green)
#define Wc_Brown Color(0,0,0,Color::Brown)
#define Wc_Blue Color(0,0,128,Color::Blue)
#define Wc_Magenta Color(0,0,0,Color::Magenta)
#define Wc_Cyan Color(0,0,0,Color::Cyan)
#define Wc_LightGray Color(200,200,200,Color::LightGray)
#define Wc_Gray Color(100,100,100,Color::Gray)
#define Wc_BrightRed Color(0xFF,0,0,Color::BrightRed)
#define Wc_BrightGreen Color(0,0xFF,0,Color::BrightGreen)
#define Wc_Yellow Color(0,0,0,Color::Yellow)
#define Wc_BrightBlue Color(0,0,0xFF,Color::BrightBlue)
#define Wc_BrightMagenta Color(0,0,0,Color::BrightMagenta)
#define Wc_BrightCyan Color(0,0,0,Color::BrightCyan)
#define Wc_White Color(0xFF,0xFF,0xFF,Color::White)
#define Wc_None Color()

// Generic Graphics Context
class GenGC
{
   protected:

   // GC clipping function
   virtual void ClipRect(Rect Win) = 0;
   void GenerateClip();
   Rect ClipStack[10];
   unsigned long ClipDepth;
   
   public:

   static GenGC *GC;
   
   // Text positioning flags, these define what Pos specifies
   enum TextFlags
   {
      XLeft = 0,
      XCenter = (1 << 0),
      XRight = (1 << 1),
      YTop = 0,
      YCenter = (1 << 2),
      YBottom = (1 << 3)
   };
   
   // Region things
   virtual void *CreateRegion(Widget *For,void *Parent);
   virtual void AttachRegion(void * /*Region*/) {};
   virtual void DeleteRegion(void * /*Region*/) {};
   virtual void ResizeRegion(Widget */*For*/,void * /*Region*/,Rect /*Size*/) {};
   virtual void GrabMouse(void * /*Region*/) {};
   virtual void SetTitle(void * /*Region*/,const char * /*Title*/,const char * /*Icon*/ = 0) {};
   
   // Clipping
   void EraseClippings();
   inline Rect GetClipping() const {return ClipDepth == 0?Rect():ClipStack[ClipDepth - 1];};
   void AddClipping(Rect Win);
   void PopClipping();

   // Aides
   void FlagPosition(Rect &Pos,unsigned long Flags);
      
   // Colors
   virtual void SetColor(Color C) = 0;
   virtual void Background(Color C) = 0;      // Only for DrawText
   
   // General
   virtual void Flush() = 0;
   virtual void Fill(Rect Area) = 0;
   virtual void BFill(Rect Area) = 0;

   // Text handling
   virtual Rect ExtentText(const char *Start,unsigned int Len) = 0;
   inline Rect ExtentText(string S) {return ExtentText(S.begin(),S.size());};
   inline Rect ExtentText(const char *S) {return ExtentText(S,strlen(S));};

   // Draw text with a background
   virtual void DrawText(Rect Pos,Point P,const char *Start,unsigned int Len,unsigned long Flags = 0) = 0;
   void DrawText(Rect Pos,const char *Start,unsigned int Len,unsigned long Flags = 0);
   inline void DrawString(Rect Pos,string S,unsigned long Flags = 0) {DrawText(Pos,S.begin(),S.size(),Flags);};
   inline void DrawString(Rect Pos,const char *S,unsigned long Flags = 0) {DrawText(Pos,S,strlen(S),Flags);};
   inline void DrawString(Rect Pos,Point P,string S,unsigned long Flags = 0) {DrawText(Pos,P,S.begin(),S.size(),Flags);};
   inline void DrawString(Rect Pos,Point P,const char *S,unsigned long Flags = 0) {DrawText(Pos,P,S,strlen(S),Flags);};

   virtual void DrawText(Point Pos,const char *Start,unsigned int Len,unsigned long Flags = 0) = 0;
   inline void DrawString(Point Pos,string S,unsigned long Flags = 0) {DrawText(Pos,S.begin(),S.size(),Flags);};
   inline void DrawString(Point Pos,const char *S,unsigned long Flags = 0) {DrawText(Pos,S,strlen(S),Flags);};

   // Wrap text
   virtual unsigned int WrapText(const char *Start,unsigned int Len,long Max) = 0;
   inline unsigned int WrapText(const char *Start,long Max) {return WrapText(Start,strlen(Start),Max);};
   long DrawWrappedText(Rect Loc,string S,bool Blank = false);

   virtual bool SetFont(SimpleFont Font) = 0;

   GenGC();
   virtual ~GenGC() {};
};

// Character cell based graphics context
class TextGC : public GenGC
{
   public:
   
   static TextGC *GC;

   // For DrawLineChar
   enum LineChars {HLineChar = 0,VLineChar,TeeChar,UlCornerChar,LlCornerChar,
                   UrCornerChar,LrCornerChar,TopTeeChar,LeftTeeChar,
                   RightTeeChar,BottomTeeChar};
      
   // Text operations
   virtual Rect ExtentText(const char *Start,unsigned int Len);
   virtual bool SetFont(SimpleFont /*Font*/) {};

   // Line/box drawing
   virtual void Box(Rect Size) = 0;
   virtual void HLine(Point Pos,long Length) = 0;
   virtual void Fill(Rect Area,char C) = 0;
   virtual void Fill(Rect Area) {Fill(Area,0);};
   virtual void BFill(Rect Area) {Fill(Area,' ');};
   virtual void DrawLineChar(Point Pos,LineChars Char) = 0;
			 
   virtual ~TextGC();
};

// Graphic based graphics context
class XPMImage;
class GraphicGC : public GenGC
{
   public:
   
   static GraphicGC *GC;
   
   // High level drawing
   virtual void Box3d(Rect Box,Color Ul,Color Lr,Color Fill,
		      unsigned long Width) = 0;
   virtual void Line(Point Start,Point Stop) = 0;
   
   // Font/Text handling
   virtual bool SetFont(SimpleFont Font) = 0;

   // Draw text with a clear (transparent) background
   virtual void DrawTextT(Point Pos,const char *Start,unsigned int Len,unsigned long Flags = 0) = 0;
   inline void DrawStringT(Point Pos,string S,unsigned long Flags = 0) {DrawTextT(Pos,S.begin(),S.size(),Flags);};
   inline void DrawStringT(Point Pos,const char *S,unsigned long Flags = 0) {DrawTextT(Pos,S,strlen(S),Flags);};
     
   // Bitmaps
   virtual void DrawBitmap(Point Pos,XPMImage &Image,unsigned long Flags = 0) = 0;
   void DrawBitmap(Rect Pos,XPMImage &Image,unsigned long Flags = 0);
   virtual void DrawBitmap(Rect Loc,Point Pos,XPMImage &Image,unsigned long Flags = 0) = 0;
   virtual void DrawBitmapT(Point Pos,XPMImage &Image,unsigned long Flags = 0) = 0;
   virtual bool ImagePreload(XPMImage &Image) = 0;
   
   virtual ~GraphicGC();
};

// Combination of all GCs
struct CombinedGC
{
   GenGC *GC;
   TextGC *tGC;
   GraphicGC *gGC;
   
   inline GenGC *operator ->() {return GC;};
   inline bool IsText() const {return tGC != 0;};
   inline bool IsGraphic() const {return gGC != 0;};

   inline CombinedGC() : GC(GenGC::GC), tGC(TextGC::GC), gGC(GraphicGC::GC) {};
};

#endif
