/*
  C K U A T H . C  --  Authentication for C-Kermit

  Copyright (C) 1998, Trustees of Columbia University in the City of New
  York.  The C-Kermit software may not be, in whole or in part, licensed or
  sold for profit as a software product itself, nor may it be included in or
  distributed with commercial products or otherwise distributed by commercial
  concerns to their clients or customers without written permission of the
  Office of Kermit Development and Distribution, Columbia University.  This
  copyright notice must not be removed, altered, or obscured.

  Author:  Jeffrey E Altman (jaltman@columbia.edu)
*/
/*
 * Based on a concatenation of all necessary source files distributed with the
 * Kerberos 5 NT Alpha 2 Telnet package from MIT with significant changes.
 * Additional copyrights included with affected code.
 */

#include "ckcdeb.h"
#include "ckucmd.h"                             /* For struct keytab */
#include "ckcnet.h"

#ifdef CRYPT_DLL
#ifndef LIBDES
#define LIBDES
#endif /* LIBDES */
#ifdef OS2
#ifdef NT
#include <windows.h>
#else /* NT */
#define INCL_DOSMODULEMGR
#include <os2.h>
#endif /* NT */
#endif /* OS2 */
#endif /* CRYPT_DLL */

#ifdef CK_AUTHENTICATION
#ifdef CK_KERBEROS
#ifdef NT
#define KRB4 1
#endif /* NT */
#define KRB5 1

#define KINIT
#define KLIST
#define KDESTROY
#endif /* CK_KERBEROS */

#ifdef CK_SRP
#define SRP 1
#endif /* CK_SRP */

#define CHECKADDRS

#ifdef CK_ENCRYPTION
#define ENCRYPTION
#endif /* CK_ENCRYPTION */

#ifdef KRB4
#ifdef NT
#define _WINDOWS
#endif /* NT */
#include <time.h>
#include <string.h>
#define  des_cblock Block
#define  des_key_schedule Schedule
#include "kerberosiv/krb.h"
#endif /* KRB4 */

#ifdef KRB5
#include <time.h>
#include <string.h>
#include "krb5.h"
#include "com_err.h"
#endif /* KRB5 */

#include <stdio.h>
#ifdef OS2
#include <io.h>
#endif /* OS2 */
#include <malloc.h>

#define ENCRYPT_NAMES
#define AUTH_NAMES
#ifdef SLC_NAMES
#undef SLC_NAMES
#endif /* SLC_NAMES */
#define SLC_NAMES
#include "ckuath.h"
#include "ckuat2.h"

#ifdef ENCRYPTION
#include <stdlib.h>
#endif /* ENCRYPTION */

#ifdef DEBUG
#undef DEBUG
#endif

extern char tn_msg[], hexbuf[];         /* from ckcnet.c */
extern char * tnopts[], * telcmds[];
extern char pwbuf[];
extern int deblog, debses;
extern int tn_auth;
#ifdef CK_ENCRYPTION
extern int tn_encrypt;
extern int cx_type;
#endif /* CK_ENCRYPTION */

#ifdef OutputDebugString
#undef OutputDebugString
#endif
#define OutputDebugString(x) debug(F000,"Kerberos",x,0);

/*
 * Implements Kerberos 4/5 authentication
 */

/*
 * Constants
 */
#define	KRB_AUTH		0	/* Authentication data follows */
#define	KRB_REJECT		1	/* Rejected (reason might follow) */
#define	KRB_ACCEPT		2	/* Accepted */
#ifdef KRB4
#define KRB4_CHALLENGE		3
#define KRB4_RESPONSE		4
#endif /* KRB4 */
#ifdef KRB5
#define	KRB5_RESPONSE		3	/* Response for mutual auth. */
#define KRB5_FORWARD            4       /* Forwarded credentials follow */
#define KRB5_FORWARD_ACCEPT     5       /* Forwarded credentials accepted */
#define KRB5_FORWARD_REJECT     6       /* Forwarded credentials rejected */
#endif /* KRB5 */
#ifdef SRP
#define	SRP_AUTH		0	/* Authentication data follows */
#define	SRP_REJECT		1	/* Rejected (reason might follow) */
#define	SRP_ACCEPT		2	/* Accepted */
#define SRP_CHALLENGE		3
#define SRP_RESPONSE		4
#define SRP_EXP                 8       /* */
#define SRP_PARAMS              9       /* */
#endif /* SRP */


#ifndef KSUCCESS                            /* Let K5 use K4 constants */
#define KSUCCESS    0
#define KFAILURE    255
#endif

/*
 * Globals
 */
int authentication_version = AUTHTYPE_NULL;
int krb_ver_usr      = AUTHTYPE_AUTO;
static int auth_how=0;
static int auth_crypt=0;
static int mutual_complete = 0;

#ifdef KRB4
static CREDENTIALS cred;
static KTEXT_ST k4_auth;
#ifdef ENCRYPTION
static Block    k4_session_key     = { 0 };
static Schedule k4_sched;
static Block    k4_challenge       = { 0 };
#endif /* ENCRYPTION */
#define KRB4_SERVICE_NAME    "rcmd"

_PROTOTYP(static int k4_auth_send,(VOID));
_PROTOTYP(static int k4_auth_reply,(unsigned char *, int));
#endif /* KRB4 */

#ifdef KRB5
static krb5_data          k5_auth;
static krb5_auth_context  auth_context;
static krb5_keyblock     *session_key = NULL;
#ifdef FORWARD
_PROTOTYP(void kerberos5_forward,(VOID));
#endif /* FORWARD */

#define KRB5_SERVICE_NAME    "host"

_PROTOTYP(static int k5_auth_send,(int,int));
_PROTOTYP(static int k5_auth_reply,(int, unsigned char *, int));
_PROTOTYP(static int SendK5AuthSB,(int, void *, int));
#endif /* KRB5 */

#ifdef SRP
_PROTOTYP(static int srp_reply,(int, unsigned char *, int));
#endif /* SRP */


#ifdef ENCRYPTION
int encrypt_flag = 1;
#endif
#ifdef FORWARD
int forward_flag = 0;       /* forward tickets? */
int forwardable_flag = 1;   /* get forwardable tickets to forward? */
int forwarded_tickets = 0;  /* were tickets forwarded? */
#endif

static unsigned char str_data[1024] = { IAC, SB, TELOPT_AUTHENTICATION, 0,
			  		AUTHTYPE_KERBEROS_V5, };

static char strTmp[1204];
static char szUserName[64];
static char szHostName[128];
static char szIP[16];

static struct kstream_crypt_ctl_block ctl;
static kstream g_kstream=NULL;

#ifdef KRB5
static krb5_context k5_context=NULL;
#endif /* KRB5 */

#ifdef SRP
#include <t_pwd.h>
#include <t_client.h>

static struct t_server * ts = NULL;
static struct t_client * tc = NULL;
static struct t_pw * tpw = NULL;
static struct t_conf * tconf = NULL;

static int waitresponse = 0;	/* Flag to indicate readiness for response */
static struct t_num * B;	/* Holder for B */

#define PWD_SZ 128

static char user_passwd[PWD_SZ];
#endif /* SRP */

#ifdef OS2
#include "ckoath.c"
#endif /* OS2 */

#ifdef CK_KERBEROS
extern char * krb4_d_srv, * krb5_d_srv;
#endif /* CK_KERBEROS */

int
ck_krb5_is_installed()
{
#ifdef KRB5
#ifdef OS2
    return(hKRB5_32 != NULL);
#else /* OS2 */
    return(1);
#endif /* OS2 */
#else /* KRB5 */
    return(0);
#endif /* KRB5 */
}

int
ck_krb4_is_installed()
{
#ifdef KRB4
#ifdef OS2
    return(hKRB4_32 != NULL);
#else /* OS2 */
    return(1);
#endif /* OS2 */
#else /* KRB4 */
    return(0);
#endif /* KRB4 */
}

int
ck_srp_is_installed()
{
#ifdef SRP
#ifdef SRPDLL
    return(hSRP != NULL);
#else /* SRPDLL */
    return(1);
#endif /* SRPDLL */
#else /* SRP */
    return(0);
#endif /* SRP */
}

int
ck_crypt_is_installed()
{
#ifdef ENCRYPTION
#ifdef CRYPT_DLL
    return(hCRYPT != NULL);
#else /* CRYPT_DLL */
    return(1);
#endif /* CRYPT_DLL */
#else /* ENCRYPTION */
    return(0);
#endif /* ENCRYPTION */
}

/* C K _ K R B _ I N I T
 * Initialize the Kerberos system for a pending connection
 *   hostname - a reverse DNS lookup of the hostname when possible
 *   ipaddr   - the ip address of the host
 *   username - the name the user wants to connect under not necessarily
 *              the same as principal
 *   socket   - the socket handle (ttyfd in Kermit speak)
 *
 * Returns: 1 on success and 0 on failure
 */

int
#ifdef CK_ANSIC
ck_krb_init( char * hostname, char * ipaddr, char * username, int socket )
#else /* CK_ANSIC */
ck_krb_init( hostname, ipaddr, username, socket )
    char * hostname; char * ipaddr; char * username; int socket;
#endif /* CK_ANSIC */
{
#ifdef OS2
    if ( !ck_krb_loaddll() ) {
        tn_auth = TN_NG_RF;
#ifdef CK_ENCRYPTION
        tn_encrypt = TN_NG_RF;
#endif /* CK_ENCRYPTION */
        return(0);
    }
#endif /* OS2 */

    strcpy( szUserName, username );
    strcpy( szHostName, hostname );
    strcpy( szIP, ipaddr );

    debug(F110,"Kerberos Username",username,0);
    debug(F110,"Kerberos Hostname",hostname,0);
    debug(F110,"Kerberos Ipaddr",ipaddr,0);

#ifdef KRB5
    /* create k5_context */
    krb5_init_context(&k5_context);
    krb5_init_ets(k5_context);
    memset(&k5_auth,0,sizeof(k5_auth));
    if (auth_context) {
        krb5_auth_con_free(k5_context, auth_context);
        auth_context = 0;
    }
#ifdef ENCRYPTION
    if (session_key) {
        krb5_free_keyblock(k5_context, session_key);
        session_key = 0;
    }
#endif /* ENCRYPTION */
#endif /* KRB5 */

#ifdef KRB4
    /* Initialize buffers used for authentication */
    memset(&k4_session_key, 0, sizeof(k4_session_key));
    memset(&k4_challenge, 0, sizeof(k4_challenge));
#endif /* KRB4 */

    kstream_destroy();

    auth_how = 0;
    auth_crypt = 0;
    mutual_complete = 0;

#ifdef SRP
    waitresponse = 0;
#endif /* SRP */

    /* create kstream from socket */
    /* a kstream is simply a structure containing the socket handle */
    /* and pointers to the appropriate functions for encryption,    */
    /* decryption, and the like.                                    */
    ctl.encrypt = auth_encrypt;
    ctl.decrypt = auth_decrypt;
    ctl.init = auth_init;
    ctl.destroy = auth_destroy;


    if (!kstream_create_from_fd(socket, &ctl, NULL))
        return(0);

    return(1);
}

/*  C K _ K R B _ T N _ S B _ A U T H
 *  An interface between the C-Kermit Telnet Command Parser and the Authent-
 *  ication option parser implemented in the Kerberos Telnet client.
 *
 *  sb   - the subnegotiation as calculated in ckcnet.c
 *  len  - the length of the buffer
 *
 *  Returns: 0 on success and -1 on failure
 */

int
#ifdef CK_ANSIC
ck_krb_tn_sb_auth(char * sb, int len)
#else /* CK_ANSIC */
ck_krb_tn_sb_auth(sb,len) char * sb; int len;
#endif /* CK_ANSIC */
{
    /* auth_parse() assumes that sb starts at pos 1 not 0 as in ckcnet.c */
    /* and it wants the length to exclude the IAC SE bytes                  */
    char buf[256];
    buf[0] = SB;
    memcpy( &buf[1], sb, len );
    buf[len+1] = '\0';
    if (auth_parse(buf,len+1-2)==KFAILURE) {
        authentication_version = AUTHTYPE_NULL;
#ifdef OS2
        ipadl25();
#endif /* OS2 */
        return(-1);
    }
#ifdef OS2
    ipadl25();
#endif /* OS2 */
    return(0);
}

/*  C K _ K R B _ T N _ S B _ E N C R Y P T
 *  An interface between the C-Kermit Telnet Command Parser and the Encryption
 *  option parser implemented in the Kerberos Telnet client.
 *
 *  sb   - the subnegotiation as calculated in ckcnet.c
 *  len  - the length of the buffer
 *
 *  Returns: Always returns 0 for success since encrypt_parse is void
 */


int
#ifdef CK_ANSIC
ck_krb_tn_sb_encrypt(char * sb, int len)
#else
ck_krb_tn_sb_encrypt(sb,len) char * sb; int len;
#endif /* CK_ANSIC */
{
    /* encrypt_parse() assumes that sb starts at pos 1 not 0 as in ckcnet.c */
    /* and it wants the length to exclude the IAC SE bytes                  */
#ifdef ENCRYPTION
    char buf[256];
    buf[0] = SB;
    memcpy( &buf[1], sb, len );
    buf[len+1] = '\0';
    encrypt_parse(buf,len+1-2);
#ifdef OS2
    ipadl25();
#endif /* OS2 */
#endif /* ENCRYPTION */
    return(0);
}


/*  C K _ K R B _ E N C R Y P T I N G
 *  Returns 1 if we are encrypting and 0 if we are not
 */

int
#ifdef CK_ANSIC
ck_krb_encrypting(VOID)
#else /* CK_ANSIC */
ck_krb_encrypting()
#endif /* CK_ANSIC */
{
    if ( g_kstream == NULL )
        return(0);
    if ( g_kstream->encrypt )
        return(g_kstream->encrypt_type);
    else 
        return(0);
}

/*  C K _ K R B _ D E C R Y P T I N G
 *  Returns 1 if we are decrypting and 0 if we are not
 */

int
#ifdef CK_ANSIC
ck_krb_decrypting(VOID)
#else
ck_krb_decrypting()
#endif /* CK_ANSIC */
{
    if ( g_kstream == NULL )
        return(0);
    if ( g_kstream->decrypt )
        return(g_kstream->decrypt_type);
    else
        return(0);
}

/*  C K _ K R B _ A U T H E N T I C A T E D
 *  Returns the authentication type: AUTHTYPE_NULL, AUTHTYPE_KERBEROS4,
 *  or AUTHTYPE_KERBEROS5
 */

int
#ifdef CK_ANSIC
ck_krb_authenticated(VOID)
#else
ck_krb_authenticated()
#endif
{
    extern int tn_auth;
    if ( !tn_auth )
        return(AUTHTYPE_NULL);
    return(authentication_version);
}

/*  C K _ K R B _ E N C R Y P T
 *  encrypts n characters in s if we are encrypting
 */

VOID
#ifdef CK_ANSIC
ck_krb_encrypt( char * s, int n )
#else
ck_krb_encrypt( s,n ) char * s; int n;
#endif
{
#ifdef ENCRYPTION
    struct kstream_data_block i;

    if (g_kstream->encrypt) {
#ifdef DEBUG
      hexdump("from plaintext", s, n);
#endif
        i.ptr = s;
        i.length = n;
        g_kstream->encrypt(&i, NULL);
#ifdef DEBUG
        hexdump("to cyphertext", s, n);
#endif
    }
    else debug(F101,"ck_krb_encrypt not encrypting","",n);
#endif /* ENCRYPTION */
}

/*  C K _ K R B _ D E C R Y P T
 *  decrypts n characters in s if we are decrypting
 */

VOID
#ifdef CK_ANSIC
ck_krb_decrypt( char * s, int n )
#else
ck_krb_decrypt( s,n ) char * s; int n;
#endif
{
#ifdef ENCRYPTION
    struct kstream_data_block i;

    if (g_kstream->decrypt) {

#ifdef DEBUG
        hexdump("from cyphertext", s, n);
#endif

        i.ptr = s;
        i.length = n;
        g_kstream->decrypt(&i, NULL);
#ifdef DEBUG
        hexdump("to plaintext", s, n);
#endif
    }
    else debug(F101,"ck_krb_decrypt not decrypting","",n);
#endif /* ENCRYPTION */
}

/*  S E N D K 5 A U T H S B
 *  Send a Kerberos 5 Authentication Subnegotiation to host and
 *  output appropriate Telnet Debug messages
 *
 *  type - Sub Negotiation type
 *  data - ptr to buffer containing data
 *  len  - len of buffer if not NUL terminated
 *
 *  returns number of characters sent or error value
 */

static int
#ifdef CK_ANSIC
SendK5AuthSB(int type, void *data, int len)
#else
SendK5AuthSB(type,data,len) int type; void *data; int len;
#endif
{
    int rc;
    unsigned char *p = str_data + 4;
    unsigned char *cd = (unsigned char *)data;


    if ( type < 0 || type > 6 )         /* Check for invalid values */
        return(0);

    if (len == -1)                        /* Use strlen() for len */
        len = strlen((char *)cd);


    /* Construct Message */
    *p++ = AUTHTYPE_KERBEROS_V5;
    *p = AUTH_WHO_CLIENT;
    *p |= auth_how;
#ifdef ENCRYPTION
    *p |= auth_crypt;
#endif
#ifdef USE_INI_CRED_FWD
    if ( forward_flag )
        *p |= INI_CRED_FWD_ON;
#endif /* USE_INI_CRED_FWD */
    p++;
    *p++ = type;
    while (len-- > 0) {
        if ((*p++ = *cd++) == IAC)
            *p++ = IAC;
        }
    *p++ = IAC;
    *p++ = SE;

    /* Handle Telnet Debugging Messages */
    if (deblog || debses) {
        int i;
        int deblen=p-str_data-2;
        char *s=NULL;
        int mode = AUTH_WHO_CLIENT | (auth_how | AUTH_HOW_MASK) |
            (auth_crypt?AUTH_ENCRYPT_ON:AUTH_ENCRYPT_OFF)
#ifdef USE_INI_CRED_FWD
                | ((authentication_version == AUTHTYPE_KERBEROS_V5 &&
                   forward_flag)?INI_CRED_FWD_ON:INI_CRED_FWD_OFF)
#endif /* USE_INI_CRED_FWD */
                    ;

        switch (type) {
        case 0:
            s = "AUTH";
            break;
        case 1:
            s = "REJECT";
            break;
        case 2:
            s = "ACCEPT";
            break;
        case 3:
            s = "RESPONSE";
            break;
        case 4:
            s = "FORWARD";
            break;
        case 5:
            s = "FORWARD_ACCEPT";
            break;
        case 6:
            s = "FORWARD_REJECT";
            break;
        }

	sprintf(tn_msg,"TELNET SENT SB %s IS %s %s %s ",
                 tnopts[TELOPT_AUTHENTICATION],
                 authtype_names[authentication_version],
                 authmode_names[mode],
                 s);
#ifdef HEXDISP
        {
            int was_hex = 1;
            for ( i=7;i<deblen;i++ ) {
                if ( str_data[i] < 32 || str_data[i] >= 127) {
                    sprintf(hexbuf,"%s%02X ",was_hex?"":"\" ",str_data[i]);
                    was_hex = 1;
                } else {
                    sprintf(hexbuf,"%s%c",was_hex?"\"":"",str_data[i]);
                    was_hex = 0;
                }
                strcat(tn_msg,hexbuf);
            }
            if ( !was_hex )
                strcat(tn_msg,"\" ");
        }
#else /* HEXDISP */
        memcpy(hexbuf,&str_data[7],deblen-7);
        hexbuf[deblen-7] = ' ';
        hexbuf[deblen-6] = '\0';
        strcat(tn_msg,hexbuf);
#endif /* HEXDISP */
        strcat(tn_msg,"IAC SE");
	debug(F100,tn_msg,"",0);
	if (debses) tn_debug(tn_msg);
    }

    /* Send data */
    rc = ttol((char *)str_data, p - str_data);
    return(rc);
}

#ifdef ENCRYPTION
/*
 * Function: Enable or disable the encryption process.
 *
 * Parameters:
 *	enable - TRUE to enable, FALSE to disable.
 */
static VOID
#ifdef CK_ANSIC
auth_encrypt_enable(BOOL enable)
#else
auth_encrypt_enable(enable) BOOL enable;
#endif
{
  encrypt_flag = enable;
}
#endif

/*
 * Function: Abort the authentication process
 *
 * Parameters:
 */
static VOID
#ifdef CK_ANSIC
auth_abort(char *errmsg, long r)
#else
auth_abort(errmsg,r) char *errmsg; long r;
#endif
{
    char buf[9];

    debug(F111,"auth_abort",errmsg,r);

    /* Construct Telnet Debugging messages */
    if (deblog || debses) {
	sprintf(tn_msg,"TELNET SENT SB %s IS %s %s IAC SE",
                 tnopts[TELOPT_AUTHENTICATION],
                 authtype_names[AUTHTYPE_NULL],
                 authtype_names[AUTHTYPE_NULL]);
	debug(F100,tn_msg,"",0);
	if (debses) tn_debug(tn_msg);
    }

    /* Construct the Abort message to send to the host   */
    /* Basicly we change the authentication type to NULL */
    sprintf(buf, "%c%c%c%c%c%c%c%c", IAC, SB, TELOPT_AUTHENTICATION,
             TELQUAL_IS, AUTHTYPE_NULL,
             AUTHTYPE_NULL, IAC, SE);
    ttol((char *)buf, 8);

    /* If there is an error message, and error number construct */
    /* an explanation to display to the user                    */
    if (errmsg != NULL) 
        strcpy(strTmp, errmsg);
    else
        strTmp[0] = '\0';


    if (r != KSUCCESS) {
        strcat(strTmp, "\n");
#ifdef KRB4
        if ( authentication_version == AUTHTYPE_KERBEROS_V4 )
            strcat(strTmp, krb_get_err_text_entry(r));
#endif  
#ifdef KRB5
        if ( authentication_version == AUTHTYPE_KERBEROS_V5 )
            strcat(strTmp, error_message(r));
#endif
    }
    printf("Authentication failed!\n%s\n",strTmp);
    authentication_version = AUTHTYPE_NULL;
}


/*
 * Function: Copy data to buffer, doubling IAC character if present.
 *
 */
static int
#ifdef CK_ANSIC
copy_for_net(unsigned char *to, unsigned char *from, int c)
#else
copy_for_net(to,from,c) unsigned char *to; unsigned char *from; int c;
#endif
{
  int n;

  n = c;

  while (c-- > 0) {
    if ((*to++ = *from++) == IAC) {
      n++;
      *to++ = IAC;
    }
  }

  return n;
}


/*
 * Function: Parse authentication send command
 *
 * Parameters:
 *  parsedat - the sub-command data.
 *
 *	end_sub - index of the character in the 'parsedat' array which
 *		is the last byte in a sub-negotiation
 *
 * Returns: Kerberos error code.
 */
static int
#ifdef CK_ANSIC
auth_send(unsigned char *parsedat, int end_sub)
#else
auth_send(parsedat,end_sub) unsigned char *parsedat; int end_sub;
#endif
{
    unsigned char buf[512];
    unsigned char *pname;
    int plen;
    int r;
    int i;
    int mode;

    auth_how = -1;              /* We have not found an auth method  */
    auth_crypt = 0;             /* We are not using encryption (yet) */


    /* Search the list of acceptable Authentication types sent from */
    /* the host and find one that we support                        */

    /* For Kerberos authentications, try to determine if we have a  */
    /* valid TGT, if not skip over the authentication type because  */
    /* we wouldn't be able to successfully login anyway.  Perhaps   */
    /* there is another supported authentication which we could use */

    if ( krb_ver_usr == AUTHTYPE_AUTO ) {
    for (i = 2; i+1 <= end_sub; i += 2) {
#ifdef SRP
        if ( parsedat[i] == AUTHTYPE_SRP
#ifdef SRPDLL
             && hSRP
#endif /* SRPDLL */
             ) {
            if ((parsedat[i+1] & AUTH_WHO_MASK) == AUTH_WHO_CLIENT) {
#ifdef CK_ENCRYPTION
                if ((parsedat[i+1] & AUTH_ENCRYPT_MASK) &&
                     tn_encrypt == TN_NG_RF)
                    continue;
                auth_crypt = parsedat[i+1] & AUTH_ENCRYPT_MASK;
#endif /* CK_ENCRYPTION */
                authentication_version = AUTHTYPE_SRP;
                auth_how = parsedat[i+1] & AUTH_HOW_MASK;
                break;
            }
        }
#endif /* SRP */
#ifdef KRB5
        if (parsedat[i] == AUTHTYPE_KERBEROS_V5 &&
#ifdef OS2
             hKRB5_32 &&
#endif /* OS2 */
             ck_krb5_is_tgt_valid()) {
            if ((parsedat[i+1] & AUTH_WHO_MASK) == AUTH_WHO_CLIENT) {
#ifdef CK_ENCRYPTION
                if ((parsedat[i+1] & AUTH_ENCRYPT_MASK) &&
                     tn_encrypt == TN_NG_RF)
                    continue;
                auth_crypt = parsedat[i+1] & AUTH_ENCRYPT_MASK;
#endif /* CK_ENCRYPTION */
                authentication_version = AUTHTYPE_KERBEROS_V5;
                auth_how = parsedat[i+1] & AUTH_HOW_MASK;
                break;
            }
        }
#endif /* KRB5 */
#ifdef KRB4
        if (parsedat[i] == AUTHTYPE_KERBEROS_V4 &&
#ifdef OS2
             hKRB4_32 &&
#endif /* OS2 */
             ck_krb4_is_tgt_valid()) {
            if ((parsedat[i+1] & AUTH_WHO_MASK) == AUTH_WHO_CLIENT) {
#ifdef CK_ENCRYPTION
                if ((parsedat[i+1] & AUTH_ENCRYPT_MASK) &&
                     tn_encrypt == TN_NG_RF)
                    continue;
                auth_crypt = parsedat[i+1] & AUTH_ENCRYPT_MASK;
#endif /* CK_ENCRYPTION */
                authentication_version = AUTHTYPE_KERBEROS_V4;
                auth_how = parsedat[i+1] & AUTH_HOW_MASK;
                break;
            }
        }
#endif /* KRB4 */
    }
    } else {
        for (i = 2; i+1 <= end_sub; i += 2) {
#ifdef SRP
            if ( krb_ver_usr == AUTHTYPE_SRP &&
                      parsedat[i] == AUTHTYPE_SRP
#ifdef SRPDLL
                 && hSRP
#endif /* SRPDLL */
                 ) {
                if ((parsedat[i+1] & AUTH_WHO_MASK) == AUTH_WHO_CLIENT) {
#ifdef CK_ENCRYPTION
                    if ((parsedat[i+1] & AUTH_ENCRYPT_MASK) &&
                         tn_encrypt == TN_NG_RF)
                        continue;
                    auth_crypt = parsedat[i+1] & AUTH_ENCRYPT_MASK;
#endif /* CK_ENCRYPTION */
                    authentication_version = AUTHTYPE_SRP;
                    auth_how = parsedat[i+1] & AUTH_HOW_MASK;
                    break;
                }
            }
#endif /* SRP */
#ifdef KRB5
            if ( krb_ver_usr == AUTHTYPE_KERBEROS_V5 &&
                 parsedat[i] == AUTHTYPE_KERBEROS_V5 &&
#ifdef OS2
                 hKRB5_32 &&
#endif /* OS2 */

                 ck_krb5_is_tgt_valid()) {
                if ((parsedat[i+1] & AUTH_WHO_MASK) == AUTH_WHO_CLIENT) {
#ifdef CK_ENCRYPTION
                    if ((parsedat[i+1] & AUTH_ENCRYPT_MASK) &&
                         tn_encrypt == TN_NG_RF)
                        continue;
                    auth_crypt = parsedat[i+1] & AUTH_ENCRYPT_MASK;
#endif /* CK_ENCRYPTION */
                    authentication_version = AUTHTYPE_KERBEROS_V5;
                    auth_how = parsedat[i+1] & AUTH_HOW_MASK;
                    break;
                }
            }
#endif /* KRB5 */
#ifdef KRB4
            if ( krb_ver_usr == AUTHTYPE_KERBEROS_V4 &&
                 parsedat[i] == AUTHTYPE_KERBEROS_V4 &&
#ifdef OS2
                 hKRB4_32 &&
#endif /* OS2 */
                 ck_krb4_is_tgt_valid()) {
                if ((parsedat[i+1] & AUTH_WHO_MASK) == AUTH_WHO_CLIENT) {
#ifdef CK_ENCRYPTION
                    if ((parsedat[i+1] & AUTH_ENCRYPT_MASK) &&
                         tn_encrypt == TN_NG_RF)
                        continue;
                    auth_crypt = parsedat[i+1] & AUTH_ENCRYPT_MASK;
#endif /* CK_ENCRYPTION */
                    authentication_version = AUTHTYPE_KERBEROS_V4;
                    auth_how = parsedat[i+1] & AUTH_HOW_MASK;
                    break;
                }
            }
#endif /* KRB4 */
        }
    }

    if (auth_how == -1) {               /* Did we find one? */
        auth_abort("No supported authentication method found", 0); 
                                        /* If not, abort the negotiation */
        return KFAILURE;
    }

#ifdef SRP
    /* Nothing to do here for SRP */
#endif /* SRP */


#ifdef KRB4
    if (authentication_version == AUTHTYPE_KERBEROS_V4) {
        r = k4_auth_send();
        if (r) {
            return KFAILURE;
        }
    }
#endif /* KRB4 */

#ifdef KRB5
    if (authentication_version == AUTHTYPE_KERBEROS_V5) {
        r = k5_auth_send(auth_how,auth_crypt);
        if (!r) {
            return KFAILURE;
        }
    }
#endif /* KRB5 */

    plen = strlen(szUserName);                 /* Set by k#_send if needed */
    pname = szUserName;

    /* Construct Telnet Debugging Message */
    if (deblog || debses) {
	sprintf(tn_msg,"TELNET SENT SB %s NAME %s IAC SE",
                 tnopts[TELOPT_AUTHENTICATION],
                 pname);
	debug(F100,tn_msg,"",0);
	if (debses) tn_debug(tn_msg);
    }

    /* Construct and send Authentication Name subnegotiation */
    sprintf(buf, "%c%c%c%c", IAC, SB, TELOPT_AUTHENTICATION, TELQUAL_NAME);
    memcpy(&buf[4], pname, plen);
    sprintf(&buf[plen + 4], "%c%c", IAC, SE);
    ttol((char *)buf, plen+6);

    /* Construct Authentication Mode subnegotiation message */
    mode = AUTH_WHO_CLIENT | (auth_how & AUTH_HOW_MASK) |
        (auth_crypt?AUTH_ENCRYPT_ON:AUTH_ENCRYPT_OFF)
#ifdef USE_INI_CRED_FWD
            | ((authentication_version == AUTHTYPE_KERBEROS_V5 &&
               forward_flag)?INI_CRED_FWD_ON:INI_CRED_FWD_OFF)
#endif /* USE_INI_CRED_FWD */
                ;
    sprintf(buf, "%c%c%c%c%c%c%c",
              IAC, SB, TELOPT_AUTHENTICATION, TELQUAL_IS,
              authentication_version,
              mode,
              KRB_AUTH);

#ifdef SRP
    if ( authentication_version == AUTHTYPE_SRP ) {
        sprintf(&buf[7], "%c%c", IAC, SE);
        if (deblog || debses) {
            int i;
            int mode = AUTH_WHO_CLIENT | (auth_how & AUTH_HOW_MASK) |
                (auth_crypt?AUTH_ENCRYPT_ON:AUTH_ENCRYPT_OFF)
#ifdef USE_INI_CRED_FWD
                    | ((authentication_version == AUTHTYPE_KERBEROS_V5 &&
                   forward_flag)?INI_CRED_FWD_ON:INI_CRED_FWD_OFF)
#endif /* USE_INI_CRED_FWD */
                        ;
            sprintf(tn_msg,"TELNET SENT SB %s IS %s %s AUTH ",
                     tnopts[TELOPT_AUTHENTICATION],
                     authtype_names[authentication_version],
                     authmode_names[mode]);
            strcat(tn_msg,"IAC SE");
            debug(F100,tn_msg,"",0);
            if (debses) tn_debug(tn_msg);
        }
        ttol((char *)buf, 9);
    }
#endif /* SRP */


#ifdef KRB4
    if ( authentication_version == AUTHTYPE_KERBEROS_V4 ) {
        k4_auth.length = copy_for_net(&buf[7], k4_auth.dat, k4_auth.length);
        sprintf(&buf[k4_auth.length+7], "%c%c", IAC, SE);

        if (deblog || debses) {
            int i;
            int mode = AUTH_WHO_CLIENT | (auth_how & AUTH_HOW_MASK) |
                (auth_crypt?AUTH_ENCRYPT_ON:AUTH_ENCRYPT_OFF)
#ifdef USE_INI_CRED_FWD
                    | ((authentication_version == AUTHTYPE_KERBEROS_V5 &&
                   forward_flag)?INI_CRED_FWD_ON:INI_CRED_FWD_OFF)
#endif /* USE_INI_CRED_FWD */
                        ;
            sprintf(tn_msg,"TELNET SENT SB %s IS %s %s AUTH ",
                     tnopts[TELOPT_AUTHENTICATION],
                     authtype_names[authentication_version],
                     authmode_names[mode]);
#ifdef HEXDISP
            {
                int was_hex = 1;
                for ( i=0;i<k4_auth.length;i++ ) {
                    if ( buf[i+7] < 32 || buf[i+7] >= 127) {
                        sprintf(hexbuf,"%s%02X ",was_hex?"":"\" ",buf[i+7]);
                        was_hex = 1;
                    } else {
                        sprintf(hexbuf,"%s%c",was_hex?"\"":"",buf[i+7]);
                        was_hex = 0;
                    }
                    strcat(tn_msg,hexbuf);
                }
                if ( !was_hex )
                    strcat(tn_msg,"\" ");
            }
#else /* HEXDISP */
            memcpy(hexbuf,&buf[7],k4_auth.length);
            hexbuf[k4_auth.length] = ' ';
            hexbuf[k4_auth.length+1] = '\0';
            strcat(tn_msg,hexbuf);
#endif /* HEXDISP */
            strcat(tn_msg,"IAC SE");
            debug(F100,tn_msg,"",0);
            if (debses) tn_debug(tn_msg);
        }
        ttol((char *)buf, k4_auth.length+9);

#ifndef REMOVE_FOR_EXPORT
#ifdef  ENCRYPTION
        /*
         * If we are doing mutual authentication, get set up to send
         * the challenge, and verify it when the response comes back.
         */
        if ((auth_how & AUTH_HOW_MASK) == AUTH_HOW_MUTUAL) {
            register int i;
            int rc = 0;

            memset(k4_sched,0,sizeof(Schedule));
            hexdump("auth_send",cred.session,8);

            rc = des_key_sched(cred.session, k4_sched);
            if ( rc == -1 ) {
                printf("?Invalid DES key specified\n");
                debug(F110,"auth_send","invalid DES Key specified",0);
            } else if ( rc == -2 ) {
                printf("?Weak DES key specified\n");
                debug(F110,"auth_send","weak DES Key specified",0);
            } else if ( rc != 0 ) {
                printf("?DES Key Schedule not set\n");
                debug(F110,"auth_send","DES Key Schedule not set",0);
            }

            hexdump("auth_send schedule",k4_sched,8*16);

            des_set_random_generator_seed(cred.session);

            do {
                des_new_random_key(k4_session_key);
                des_fixup_key_parity(k4_session_key);
            } while ( ck_des_is_weak_key(k4_session_key) );

            hexdump("des_new_random_key(k4_session_key)",
                     k4_session_key,8);

            /* Decrypt the session key so that we can send it to the */
            /* host as a challenge                                   */
            des_ecb_encrypt(k4_session_key, k4_session_key, k4_sched, 0);

            /* Prepare the result of the challenge */
            /* Decrypt the session_key, add 1, and then encrypt it */
            /* The result stored in k4_challenge should match the  */
            /* KRB4_RESPONSE value from the host.                  */
            des_ecb_encrypt(k4_session_key, k4_challenge, k4_sched, 0);
            /*
            * Increment the challenge by 1, and encrypt it for
            * later comparison.
            */
            for (i = 7; i >= 0; --i) {
                register int x;
                x = (unsigned int)k4_challenge[i] + 1;
                k4_challenge[i] = x;    /* ignore overflow */
                if (x < 256)            /* if no overflow, all done */
                    break;
            }
            hexdump("k4_challenge + 1",k4_challenge,8);
            des_ecb_encrypt(k4_challenge, k4_challenge, k4_sched, 1);
        }
#endif  /* ENCRYPTION */
#endif /* REMOVE_FOR_EXPORT */
    }
#endif /* KRB4 */
#ifdef KRB5
    if ( authentication_version == AUTHTYPE_KERBEROS_V5 ) {
        k5_auth.length = copy_for_net(&buf[7], k5_auth.data, k5_auth.length);
        sprintf(&buf[k5_auth.length+7], "%c%c", IAC, SE);

        if (deblog || debses) {
            int i;
            int mode = AUTH_WHO_CLIENT | (auth_how & AUTH_HOW_MASK) |
                (auth_crypt?AUTH_ENCRYPT_ON:AUTH_ENCRYPT_OFF)
#ifdef USE_INI_CRED_FWD
                    | ((authentication_version == AUTHTYPE_KERBEROS_V5 &&
                   forward_flag)?INI_CRED_FWD_ON:INI_CRED_FWD_OFF)
#endif /* USE_INI_CRED_FWD */
                        ;
            sprintf(tn_msg,"TELNET SENT SB %s IS %s %s AUTH ",
                     tnopts[TELOPT_AUTHENTICATION],
                     authtype_names[authentication_version],
                     authmode_names[mode]);
#ifdef HEXDISP
            {
                int was_hex = 1;
                for ( i=0;i<k5_auth.length;i++ ) {
                    if ( buf[i+7] < 32 || buf[i+7] >= 127) {
                        sprintf(hexbuf,"%s%02X ",was_hex?"":"\" ",buf[i+7]);
                        was_hex = 1;
                    } else {
                        sprintf(hexbuf,"%s%c",was_hex?"\"":"",buf[i+7]);
                        was_hex = 0;
                    }
                    strcat(tn_msg,hexbuf);
                }
                if ( !was_hex )
                    strcat(tn_msg,"\" ");
            }
#else /* HEXDISP */
            memcpy(hexbuf,&buf[7],k5_auth.length);
            hexbuf[k5_auth.length] = ' ';
            hexbuf[k5_auth.length+1] = '\0';
            strcat(tn_msg,hexbuf);
#endif /* HEXDISP */
            strcat(tn_msg,"IAC SE");
            debug(F100,tn_msg,"",0);
            if (debses) tn_debug(tn_msg);
        }
        ttol((char *)buf, k5_auth.length+9);
    }
#endif /* KRB5 */
    return KSUCCESS;
}

/*
 * Function: Parse authentication reply command
 *
 * Parameters:
 *  parsedat - the sub-command data.
 *
 *	end_sub - index of the character in the 'parsedat' array which
 *		is the last byte in a sub-negotiation
 *
 * Returns: Kerberos error code.
 */
static int
#ifdef CK_ANSIC
auth_reply(unsigned char *parsedat, int end_sub)
#else
auth_reply(parsedat,end_sub) unsigned char *parsedat; int end_sub;
#endif
{
    int n;

#ifdef KRB4
    if (authentication_version == AUTHTYPE_KERBEROS_V4)
        n = k4_auth_reply(parsedat, end_sub);
#endif
#ifdef KRB5
    if (authentication_version == AUTHTYPE_KERBEROS_V5)
        n = k5_auth_reply(auth_how|auth_crypt, parsedat, end_sub);
#endif
#ifdef SRP
    if (authentication_version == AUTHTYPE_SRP)
        n = srp_reply(auth_how|auth_crypt, parsedat, end_sub);
#endif /* SRP */
    return n;
}

/*
 * Function: Parse the athorization sub-options and reply.
 *
 * Parameters:
 *	parsedat - sub-option string to parse.
 *
 *	end_sub - last charcter position in parsedat.
 */
int
auth_parse(unsigned char *parsedat, int end_sub)
{
    if (parsedat[1] == TELQUAL_SEND)
        return(auth_send(parsedat, end_sub));

    if (parsedat[1] == TELQUAL_REPLY)
        return(auth_reply(parsedat, end_sub));
    return(KFAILURE);
}


/*
 * Function: Initialization routine called kstream encryption system.
 *
 * Parameters:
 *  data - user data.
 */
int
#ifdef CK_ANSIC
auth_init(kstream ks)
#else
auth_init(ks) kstream_ptr ks;
#endif
{
#ifdef FORWARD
    forwarded_tickets = 0;  /* were tickets forwarded? */
#endif /* FORWARD */
#ifdef ENCRYPTION
    encrypt_init(ks,cx_type);
#endif
    return 0;
}


/*
 * Function: Destroy routine called kstream encryption system.
 *
 * Parameters:
 *  data - user data.
 */
VOID
#ifdef CK_ANSIC
auth_destroy(void)
#else
auth_destroy()
#endif
{
}


/*
 * Function: Callback to encrypt a block of characters
 *
 * Parameters:
 *	out - return as pointer to converted buffer.
 *
 *  in - the buffer to convert
 *
 *  str - the stream being encrypted
 *
 * Returns: number of characters converted.
 */
int
#ifdef CK_ANSIC
auth_encrypt(struct kstream_data_block *out,
	     struct kstream_data_block *in)
#else
auth_encrypt(out,in)
    struct kstream_data_block *out; struct kstream_data_block *in;
#endif
{
    out->ptr = in->ptr;

    out->length = in->length;

    return(out->length);
}


/*
 * Function: Callback to decrypt a block of characters
 *
 * Parameters:
 *	out - return as pointer to converted buffer.
 *
 *  in - the buffer to convert
 *
 *  str - the stream being encrypted
 *
 * Returns: number of characters converted.
 */
int
#ifdef CK_ANSIC
auth_decrypt(struct kstream_data_block *out,
	     struct kstream_data_block *in)
#else
auth_decrypt(out,in)
    struct kstream_data_block *out; struct kstream_data_block *in;
#endif
{
    out->ptr = in->ptr;

    out->length = in->length;

    return(out->length);
}

#ifdef KRB4
/*
 *
 * K4_auth_send - gets authentication bits we need to send to KDC.
 *
 * Result is left in auth
 *
 * Returns: 0 on failure, 1 on success
 */
static int
#ifdef CK_ANSIC
k4_auth_send(void)
#else
k4_auth_send()
#endif
{
    int r;                                      /* Return value */
    char instance[INST_SZ];
    char *realm;
    char buf[256];

    memset(instance, 0, sizeof(instance));

    if (realm = krb_get_phost(szHostName))
        lstrcpy(instance, realm);

    realm = krb_realmofhost(szHostName);

    if (!realm) {
        strcpy(buf, "Can't find realm for host \"");
        strcat(buf, szHostName);
        strcat(buf, "\"");
        auth_abort(buf, 0);
        return KFAILURE;
    }

    r = krb_mk_req(&k4_auth, krb4_d_srv ? krb4_d_srv : KRB4_SERVICE_NAME, 
                    instance, realm, 0);

    if (r == 0)
        r = krb_get_cred(krb4_d_srv ? krb4_d_srv : KRB4_SERVICE_NAME, 
                          instance, realm, &cred);

    if (r) {
        strcpy(buf, "Can't get \"");
        strcat(buf, krb4_d_srv ? krb4_d_srv : KRB4_SERVICE_NAME);
        if (instance[0] != 0) {
            strcat(buf, ".");
            lstrcat(buf, instance);
        }
        strcat(buf, "@");
        lstrcat(buf, realm);
        strcat(buf, "\" ticket");
        auth_abort(buf, r);
        return r;
    }

#ifdef OS2
#ifdef COMMENT
    if (!szUserName[0])			/* Copy if not there */
        strcpy(szUserName, cred.pname);
#else
    if ( !szUserName[0] || !stricmp(szUserName,cred.pname) )
        strcpy(szUserName, cred.pname);
#endif
#endif /* OS2 */
    return(0);
}

/*
 * Function: K4 parse authentication reply command
 *
 * Parameters:
 *	ks - kstream to send abort message to.
 *
 *  parsedat - the sub-command data.
 *
 *	end_sub - index of the character in the 'parsedat' array which
 *		is the last byte in a sub-negotiation
 *
 * Returns: Kerberos error code.
 */
static int
#ifdef CK_ANSIC
k4_auth_reply(unsigned char *parsedat, int end_sub)
#else
k4_auth_reply(parsedat,end_sub) unsigned char *parsedat; int end_sub;
#endif
{
#ifdef ENCRYPTION
    Session_Key skey;
#endif
    time_t t;
    int x;
    char buf[512];
    int i;

    if (end_sub < 4)
        return KFAILURE;

    if (parsedat[2] != AUTHTYPE_KERBEROS_V4)
        return KFAILURE;

    if (parsedat[4] == KRB_REJECT) {
        buf[0] = 0;

        for (i = 5; i <= end_sub; i++) {
            if (parsedat[i] == IAC)
                break;
            buf[i-5] = parsedat[i];
            buf[i-4] = 0;
        }

        if (!buf[0])
            strcpy(buf, "Authentication rejected by remote machine!");
        printf("Kerberos authentication failed!\n%s\n",buf);
        return KFAILURE;
    }

    if (parsedat[4] == KRB_ACCEPT) {
        int net_len;
        if ((parsedat[3] & AUTH_HOW_MASK) == AUTH_HOW_ONE_WAY)
            return KSUCCESS;

        if ((parsedat[3] & AUTH_HOW_MASK) != AUTH_HOW_MUTUAL)
            return KFAILURE;

        sprintf(buf, "%c%c%c%c%c%c%c",
		IAC, SB, TELOPT_AUTHENTICATION, TELQUAL_IS,
		AUTHTYPE_KERBEROS_V4,
		AUTH_WHO_CLIENT|AUTH_HOW_MUTUAL, KRB4_CHALLENGE
		);

        net_len = copy_for_net(&buf[7],k4_session_key,sizeof(k4_session_key));
        sprintf(&buf[7+net_len], "%c%c", IAC, SE);
        if (deblog || debses) {
            int i;
            sprintf(tn_msg,
		   "TELNET SENT SB %s IS KERBEROS_V4 CLIENT|MUTUAL CHALLENGE ",
                    tnopts[TELOPT_AUTHENTICATION]);
#ifdef HEXDISP
            {
                int was_hex = 1;
                for ( i=0;i<net_len;i++ ) {
                    if ( buf[7+i] < 32 || buf[7+i] >= 127) {
                        sprintf(hexbuf,"%s%02X ",was_hex?"":"\" ",
                                 (unsigned char)buf[7+i]);
                        was_hex = 1;
                    } else {
                        sprintf(hexbuf,"%s%c",was_hex?"\"":"",
                                 (unsigned char)buf[7+i]);
                        was_hex = 0;
                    }
                    strcat(tn_msg,hexbuf);
                }
                if ( !was_hex )
                    strcat(tn_msg,"\" ");
            }
#else /* HEXDISP */
            memcpy(hexbuf,&buf[7],net_len);
            hexbuf[net_len] = ' ';
            hexbuf[net_len+1] = '\0';
            strcat(tn_msg,hexbuf);
#endif /* HEXDISP */
            strcat(tn_msg,"IAC SE");
            debug(F100,tn_msg,"",0);
            if (debses) tn_debug(tn_msg);
        }
        ttol((char *)buf, 9+net_len);

#ifndef REMOVE_FOR_EXPORT
#ifdef ENCRYPTION
     /* We have sent the decrypted session key to the host as a challenge.  */
     /* now we encrypt it to restore it to its original valid DES key value */
        des_ecb_encrypt(k4_session_key, k4_session_key, k4_sched, 1);

        /* And then use it to configure the encryption state machine. */
        skey.type = SK_DES;
        skey.length = 8;
        skey.data = k4_session_key;
        encrypt_session_key(&skey, AUTH_WHO_CLIENT);
#endif /* ENCRYPTION */
#endif /* REMOVE_FOR_EXPORT */
        return KSUCCESS;
    }

    if (parsedat[4] == KRB4_RESPONSE) {
        if (end_sub < 12)
            return KFAILURE;

        hexdump("KRB4_RESPONSE &parsedat[5]",&parsedat[5],8);
        hexdump("KRB4_RESPONSE k4_challenge",k4_challenge,8);

        /* The datablock returned from the host should match the value */
        /* we stored in k4_challenge.                                  */
        if (memcmp(&parsedat[5], k4_challenge, sizeof(k4_challenge)) != 0) {
            printf("Kerberos authentication failed!\n%s\n",
            "Remote machine is being impersonated!");
            return KFAILURE;
        }
        return KSUCCESS;
    }
    return KFAILURE;
}
#endif /* KRB4 */

#ifdef KRB5

/*
 *
 * K5_auth_send - gets authentication bits we need to send to KDC.
 *
 * Code lifted from telnet sample code in the appl directory.
 *
 * Result is left in k5_auth
 *
 * Returns: 0 on failure, 1 on success
 *
 */

static int
#ifdef CK_ANSIC
k5_auth_send(int how, int encrypt)
#else
k5_auth_send(how,encrypt) int how; int encrypt;
#endif
{
    krb5_error_code r=0;
    krb5_ccache ccache;
    krb5_creds creds;
    krb5_creds * new_creds=NULL;
    extern krb5_flags krb5_kdc_default_options;
    krb5_flags ap_opts;
    char type_check[2];
    krb5_data check_data;
    int len=0;
#ifdef ENCRYPTION
    krb5_keyblock *newkey = 0;
#endif

    if (r = krb5_cc_default(k5_context, &ccache)) {
        com_err(NULL, r, "while authorizing (0).");
        return(0);
    }

    memset((char *)&creds, 0, sizeof(creds));
    if (r = krb5_sname_to_principal(k5_context, szHostName, 
                                krb5_d_srv ? krb5_d_srv : KRB5_SERVICE_NAME,
                                KRB5_NT_SRV_HST, &creds.server)) {
        com_err(NULL, r, "while authorizing (1).");
        return(0);
    }

    if (r = krb5_cc_get_principal(k5_context, ccache, &creds.client)) {
        com_err(NULL, r, "while authorizing (2).");
        krb5_free_cred_contents(k5_context, &creds);
        return(0);
    }

    if (szUserName[0] == '\0') {                /* Get user name now */
        len  = krb5_princ_component(k5_context, creds.client, 0)->length;
        memcpy(szUserName,
                krb5_princ_component(k5_context, creds.client, 0)->data,
                len);
        szUserName[len] = '\0';
    } else {
        char * name = NULL;
        len  = krb5_princ_component(k5_context, creds.client, 0)->length;
        if ( len == strlen(szUserName) ) {
            name = krb5_princ_component(k5_context, creds.client, 0)->data;
#ifdef OS2
            if ( !strnicmp(szUserName,name,len) ) {
                memcpy(szUserName,name,len);
                szUserName[len] = '\0';
            }
#endif /* OS2 */
        }
    }
    if (r = krb5_get_credentials(k5_context, 0,
                                  ccache, &creds, &new_creds)) {
        com_err(NULL, r, "while authorizing (3).");
        krb5_free_cred_contents(k5_context, &creds);
        return(0);
    }

    ap_opts = 0;
    if ((how & AUTH_HOW_MASK) == AUTH_HOW_MUTUAL)
        ap_opts = AP_OPTS_MUTUAL_REQUIRED;

#ifdef ENCRYPTION
    if ( encrypt )
        ap_opts |= AP_OPTS_USE_SUBKEY;
#endif

    if (auth_context) {
        krb5_auth_con_free(k5_context, auth_context);
        auth_context = 0;
    }
    if ((r = krb5_auth_con_init(k5_context, &auth_context))) {
        com_err(NULL, r, "while initializing auth context");
        return(0);
    }

    krb5_auth_con_setflags(k5_context, auth_context,
                            KRB5_AUTH_CONTEXT_RET_TIME);

    type_check[0] = AUTHTYPE_KERBEROS_V5;
    type_check[1] = AUTH_WHO_CLIENT| (how & AUTH_HOW_MASK) | encrypt;
    check_data.magic = KV5M_DATA;
    check_data.length = 2;
    check_data.data = (char *)&type_check;

    r = krb5_mk_req_extended(k5_context, &auth_context, ap_opts,
                              &check_data, new_creds, &k5_auth);

#ifdef ENCRYPTION
    if ( encrypt ) {
        krb5_auth_con_getlocalsubkey(k5_context, auth_context, &newkey);
        if (session_key) {
            krb5_free_keyblock(k5_context, session_key);
            session_key = 0;
        }

        if (newkey) {
            /*
            * keep the key in our private storage, but don't use it
            * yet---see kerberos5_reply() below
            */
            if ((newkey->enctype != ENCTYPE_DES_CBC_CRC) &&
                 (newkey-> enctype != ENCTYPE_DES_CBC_MD5))
            {
                if ((new_creds->keyblock.enctype == ENCTYPE_DES_CBC_CRC) ||
                     (new_creds->keyblock.enctype == ENCTYPE_DES_CBC_MD5))
                    /* use the session key in credentials instead */
		  krb5_copy_keyblock(k5_context,
				     &new_creds->keyblock, &session_key);
                else
                    ; 	/* What goes here? XXX */
            } else {
                krb5_copy_keyblock(k5_context, newkey, &session_key);
            }
            krb5_free_keyblock(k5_context, newkey);
        }
    }
#endif  /* ENCRYPTION */

    krb5_free_cred_contents(k5_context, &creds);
    krb5_free_creds(k5_context, new_creds);

    if (r) {
        com_err(NULL, r, "while authorizing (4).");
        return(0);
    }
    return(1);
}

/*
 *
 * K5_auth_reply -- checks the reply for mutual authentication.
 *
 * Code lifted from telnet sample code in the appl directory.
 *
 */
static int
#ifdef CK_ANSIC
k5_auth_reply(int how, unsigned char *data, int cnt)
#else
k5_auth_reply(how,data,cnt) int how; unsigned char *data; int cnt;
#endif
{
#ifdef ENCRYPTION
    Session_Key skey;
#endif

    data += 4;                                  /* Point to status byte */

    switch (*data++) {
    case KRB_REJECT:
        cnt -=5;
        if (cnt > 0) {
            char *s;
            sprintf(strTmp,"Kerberos V5 refuses authentication because\n");
            s = strTmp + strlen(strTmp);
            memcpy(s, data, cnt);
            s[cnt] = 0;
        } else
            sprintf(strTmp, "Kerberos V5 refuses authentication");
        printf("Kerberos authentication failed!\n%s\n",strTmp);
        return KFAILURE;

    case KRB_ACCEPT:
        if (!mutual_complete) {
            if ((how & AUTH_HOW_MASK) == AUTH_HOW_MUTUAL && !mutual_complete) {
                sprintf(strTmp,
                          "Kerberos V5 accepted you, but didn't provide"
                          " mutual authentication");
                printf("Kerberos authentication failed!\n%s\n",strTmp);
                return KFAILURE;
            }
#ifdef ENCRYPTION
            if ((how & AUTH_ENCRYPT_MASK) == AUTH_ENCRYPT_ON) {
                if (session_key) {
                    skey.type = SK_DES;
                    skey.length = 8;
                    skey.data = session_key->contents;
                    encrypt_session_key(&skey, AUTH_WHO_CLIENT);
                }
            }
#endif
        }
        cnt -= 5;
        if ( cnt > 0 )
            memcpy(strTmp,data,cnt);
        strTmp[cnt] = 0;
        printf("Kerberos V accepts you as %s\n",strTmp);

#ifdef FORWARD
        if (forward_flag)
            kerberos5_forward();
#endif

        return KSUCCESS;
        break;

    case KRB5_RESPONSE:
        if ((how & AUTH_HOW_MASK) == AUTH_HOW_MUTUAL) {
            /* the rest of the reply should contain a krb_ap_rep */
            krb5_ap_rep_enc_part *reply;
            krb5_data inbuf;
            krb5_error_code r;

            inbuf.length = cnt;
            inbuf.data = (char *)data;

            if (r = krb5_rd_rep(k5_context, auth_context, &inbuf, &reply)) {
                com_err(NULL, r, "while authorizing. (5)");
                return KFAILURE;
            }
            krb5_free_ap_rep_enc_part(k5_context, reply);

#ifdef ENCRYPTION
            if ((how & AUTH_ENCRYPT_MASK) == AUTH_ENCRYPT_ON) {
                if (encrypt_flag && session_key) {
                    skey.type = SK_DES;
                    skey.length = 8;
                    skey.data = session_key->contents;
                    encrypt_session_key(&skey, AUTH_WHO_CLIENT);
                }
            }
#endif /* ENCRYPTION */
            mutual_complete = 1;
        }
        return KSUCCESS;

#ifdef FORWARD
    case KRB5_FORWARD_ACCEPT:
        forwarded_tickets = 1;
        return KSUCCESS;

    case KRB5_FORWARD_REJECT:
        forwarded_tickets = 0;
        if (cnt > 0) {
            char *s;

            sprintf(strTmp,
                      "Kerberos V5 refuses forwarded credentials because\n");
            s = strTmp + strlen(strTmp);
            memcpy(s, data, cnt);
            s[cnt] = 0;
        } else
            sprintf(strTmp, "Kerberos V5 refuses forwarded credentials");

        printf("%s\n",strTmp);
        return KSUCCESS;
#endif	/* FORWARD */

    default:
        return KFAILURE;                        /* Unknown reply type */
    }
}

#ifdef FORWARD
VOID
#ifdef CK_ANSIC
kerberos5_forward(void)
#else
kerberos5_forward()
#endif
{
    krb5_error_code r;
    krb5_ccache ccache;
    krb5_principal client = 0;
    krb5_principal server = 0;
    krb5_data forw_creds;

    forw_creds.data = 0;

    if ((r = krb5_cc_default(k5_context, &ccache))) {
        com_err(NULL, r, "Kerberos V5: could not get default ccache");
        return;
    }

    if ((r = krb5_cc_get_principal(k5_context, ccache, &client))) {
        com_err(NULL, r, "Kerberos V5: could not get default principal");
        goto cleanup;
    }

    if ((r = krb5_sname_to_principal(k5_context, szHostName, 
                               krb5_d_srv ? krb5_d_srv : KRB5_SERVICE_NAME,
				     KRB5_NT_SRV_HST, &server))) {
        com_err(NULL, r, "Kerberos V5: could not make server principal");
        goto cleanup;
    }

    if ((r = krb5_auth_con_genaddrs(k5_context, auth_context, g_kstream->fd,
			    KRB5_AUTH_CONTEXT_GENERATE_LOCAL_FULL_ADDR))) {
        com_err(NULL, r, "Kerberos V5: could not gen local full address");
        goto cleanup;
    }

    if (r = krb5_fwd_tgt_creds(k5_context, auth_context, 0, client, server,
			        ccache, forwardable_flag, &forw_creds)) {
        com_err(NULL, r, "Kerberos V5: error getting forwardable credentials");
        goto cleanup;
    }

    /* Send forwarded credentials */
    if (!SendK5AuthSB(KRB5_FORWARD, forw_creds.data, forw_creds.length)) {
        printf("Kerberos V5 forwarding error!\n%s\n",
                    "Not enough room for authentication data");
    }

cleanup:
    if (client)
        krb5_free_principal(k5_context, client);
    if (server)
        krb5_free_principal(k5_context, server);
#if 0 /* XXX */
    if (forw_creds.data)
        free(forw_creds.data);
#endif
    krb5_cc_close(k5_context, ccache);
}
#endif /* FORWARD */
#endif /* KRB5 */

#ifdef SRP
/*
 * Copyright (c) 1997 Stanford University
 *
 * The use of this software for revenue-generating purposes may require a
 * license from the owners of the underlying intellectual property.
 * Specifically, the SRP-3 protocol may not be used for revenue-generating
 * purposes without a license.
 *
 * Within that constraint, permission to use, copy, modify, and distribute
 * this software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notices and this permission
 * notice appear in all copies of the software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "t_pwd.h"
#include "t_server.h"
#include "t_client.h"

/*  S E N D S R P A U T H S B
 *  Send a SRP Authentication Subnegotiation to host and
 *  output appropriate Telnet Debug messages
 *
 *  type - Sub Negotiation type
 *  data - ptr to buffer containing data
 *  len  - len of buffer if not NUL terminated
 *
 *  returns number of characters sent or error value
 */

static int
#ifdef CK_ANSIC
SendSRPAuthSB(int type, void *data, int len)
#else
SendSRPAuthSB(type,data,len) int type; void *data; int len;
#endif
{
    int rc;
    unsigned char *p = str_data + 4;
    unsigned char *cd = (unsigned char *)data;

    /* Check for invalid values */
    if ( type != SRP_EXP && type != SRP_RESPONSE )
        return(0);

    if (len == -1)                        /* Use strlen() for len */
        len = strlen((char *)cd);


    /* Construct Message */
    *p++ = AUTHTYPE_SRP;
    *p = AUTH_WHO_CLIENT;
    *p |= auth_how;
#ifdef ENCRYPTION
    *p |= auth_crypt;
#endif
    p++;
    *p++ = type;
    while (len-- > 0) {
        if ((*p++ = *cd++) == IAC)
            *p++ = IAC;
        }
    *p++ = IAC;
    *p++ = SE;

    /* Handle Telnet Debugging Messages */
    if (deblog || debses) {
        int i;
        int deblen=p-str_data-2;
        char *s=NULL;
        int mode = AUTH_WHO_CLIENT | (auth_how | AUTH_HOW_MASK) |
            (auth_crypt?AUTH_ENCRYPT_ON:AUTH_ENCRYPT_OFF);

        switch (type) {
        case 0:
            s = "AUTH";
            break;
        case 1:
            s = "REJECT";
            break;
        case 2:
            s = "ACCEPT";
            break;
        case 3:
            s = "CHALLENGE";
            break;
        case 4:
            s = "RESPONSE";
            break;
        case 5:
            s = "FORWARD";
            break;
        case 6:
            s = "FORWARD_ACCEPT";
            break;
        case 7:
            s = "FORWARD_REJECT";
            break;
        case 8:
            s = "EXP";
            break;
        case 9:
            s = "PARAMS";
            break;
        }

	sprintf(tn_msg,"TELNET SENT SB %s IS %s %s %s ",
                 tnopts[TELOPT_AUTHENTICATION],
                 authtype_names[authentication_version],
                 authmode_names[mode],
                 s);
#ifdef HEXDISP
        {
            int was_hex = 1;
            for ( i=7;i<deblen;i++ ) {
                if ( str_data[i] < 32 || str_data[i] >= 127) {
                    sprintf(hexbuf,"%s%02X ",was_hex?"":"\" ",str_data[i]);
                    was_hex = 1;
                } else {
                    sprintf(hexbuf,"%s%c",was_hex?"\"":"",str_data[i]);
                    was_hex = 0;
                }
                strcat(tn_msg,hexbuf);
            }
            if ( !was_hex )
                strcat(tn_msg,"\" ");
        }
#else /* HEXDISP */
        memcpy(hexbuf,&str_data[7],deblen-7);
        hexbuf[deblen-7] = ' ';
        hexbuf[deblen-6] = '\0';
        strcat(tn_msg,hexbuf);
#endif /* HEXDISP */
        strcat(tn_msg,"IAC SE");
	debug(F100,tn_msg,"",0);
	if (debses) tn_debug(tn_msg);
    }

    /* Send data */
    rc = ttol((char *)str_data, p - str_data);
    return(rc);
}

static void
srp_encode_length(data, num)
     unsigned char * data;
     int num;
{
  *data = (num >> 8) & 0xff;
  *++data = num & 0xff;
}

static int
srp_decode_length(data)
     unsigned char * data;
{
  return (((int) *data & 0xff) << 8) | (*(data + 1) & 0xff);
}


static int
#ifdef CK_ANSIC
srp_reply(int how, unsigned char *data, int cnt)
#else
srp_reply(how,data,cnt) int how; unsigned char *data; int cnt;
#endif
{
    struct t_num n;
    struct t_num g;
    struct t_num s;

    struct t_num B;
    struct t_num * A;

    char hexbuf[MAXHEXPARAMLEN];
    int pflag;

#ifdef ENCRYPTION
    Session_Key skey;
#endif /* ENCRYPTION */

    char * str=NULL;

    data += 4;                          /* Point to status byte */
    cnt  -= 4;

    if(cnt-- < 1)
        return KFAILURE;

    switch(*data++) {
    case SRP_REJECT:
        if (cnt > 0) {
            sprintf(strTmp,
		    "[ SRP refuses authentication for '%s' (%.*s) ]\r\n",
                    szUserName, cnt, data);
            str = strTmp + strlen(strTmp);
            memcpy(str,data,cnt);
            str[cnt] = 0;
        } else
            sprintf(strTmp,"[ SRP refuses authentication for '%s' ]\r\n",
                     szUserName);
        printf("SRP authentication failed!\n%s\n",strTmp);
        return KFAILURE;

    case SRP_ACCEPT:
        if(tc == NULL || cnt < RESPONSE_LEN || !waitresponse) {
            printf("SRP Protocol error\r\n");
            return KFAILURE;
        }

        if(t_clientverify(tc, data) == 0) {
            printf("[ SRP authentication successful ]\n");

#ifdef  ENCRYPTION
            skey.type = SK_GENERIC;
            skey.length = SESSION_KEY_LEN;
            skey.data = tc->session_key;
            encrypt_session_key(&skey, AUTH_WHO_CLIENT);
#endif /* ENCRYPTION */

            return KSUCCESS;
        }
        else {
            printf("[ Warning: SRP server authentication failed ]\n");
            return KFAILURE;
        }
        break;

    case SRP_PARAMS:
        if(!szUserName) {
            printf("No username available\r\n");
            return KFAILURE;
        }

        n.len = srp_decode_length(data);
        data += 2;
        cnt -= 2;
        if(n.len > cnt) {
            printf("n too long\r\n");
            return KFAILURE;
        }
        n.data = data;
        data += n.len;
        cnt -= n.len;

        printf("[ Using %d-bit modulus for '%s' ]\n", 8 * n.len, szUserName);

        g.len = srp_decode_length(data);
        data += 2;
        cnt -= 2;
        if(g.len > cnt) {
            printf("g too long\r\n");
            return KFAILURE;
        }
        g.data = data;
        data += g.len;
        cnt -= g.len;

        s.len = srp_decode_length(data);
        data += 2;
        cnt -= 2;
        if(s.len > cnt) {
            printf("salt too long\r\n");
            return KFAILURE;
        }
        s.data = data;
        data += s.len;
        cnt -= s.len;

        tc = t_clientopen(szUserName, &n, &g, &s);

        A = t_clientgenexp(tc);

        SendSRPAuthSB(SRP_EXP, A->data, A->len);

        if ( pwbuf[0] ) {
            strncpy(user_passwd,pwbuf,sizeof(user_passwd)-1);
#ifdef OS2
            ck_encrypt((char *)user_passwd);
#endif /* OS2 */
        }
        else
            readpass("SRP Password: ",user_passwd,sizeof(user_passwd)-1);

        t_clientpasswd(tc, user_passwd);
        memset(user_passwd, 0, sizeof(user_passwd));
        return KSUCCESS;

    case SRP_CHALLENGE:
        if(tc == NULL) {
            printf("Protocol error\r\n");
            return KFAILURE;
        }

        B.data = data;
        B.len = cnt;
        t_clientgetkey(tc, &B);

        SendSRPAuthSB(SRP_RESPONSE, t_clientresponse(tc), RESPONSE_LEN);
        waitresponse = 1;
        return KSUCCESS;

    default:
        return KFAILURE;
        break;
    }
    return KSUCCESS;
}
#endif /* SRP */


#ifdef KRB5
#ifdef KINIT
/*
 * clients/kinit/kinit.c
 *
 * Copyright 1990 by the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * Export of this software from the United States of America may
 *   require a specific license from the United States Government.
 *   It is the responsibility of any person or organization contemplating
 *   export to obtain such a license before exporting.
 *
 * WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
 * distribute this software and its documentation for any purpose and
 * without fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of M.I.T. not be used in advertising or publicity pertaining
 * to distribution of the software without specific, written prior
 * permission.  M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 *
 *
 * Initialize a credentials cache.
 */

#include "k5-int.h"
#include "com_err.h"
#include "adm_proto.h"

#include <stdio.h>
#include <time.h>
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#define KRB5_DEFAULT_OPTIONS 0
#define KRB5_DEFAULT_LIFE 60*60*10 /* 10 hours */

static krb5_data tgtname = {
    0,
    KRB5_TGS_NAME_SIZE,
    KRB5_TGS_NAME
};

/* Internal prototypes */
static krb5_error_code krb5_validate_tgt
        KRB5_PROTOTYPE((krb5_context, krb5_ccache,
                        krb5_principal, krb5_data *));
static krb5_error_code krb5_renew_tgt
        KRB5_PROTOTYPE((krb5_context, krb5_ccache,
                        krb5_principal, krb5_data *));
static krb5_error_code krb5_tgt_gen
        KRB5_PROTOTYPE((krb5_context, krb5_ccache,
                        krb5_principal, krb5_data *, int opt));

/*
 * Try no preauthentication first; then try the encrypted timestamp
 */
static krb5_preauthtype * preauth = NULL;
static krb5_preauthtype preauth_list[2] = { 0, -1 };

#define NO_KEYTAB

struct tm *
#ifdef CK_ANSIC
cmdate2tm(char * date)
#else
cmdate2tm(date) char * date;
#endif
{
    /* date as "yyyymmdd hh:mm:ss" */
    static struct tm _tm;
    time_t now;

    if ( strlen(date) != 17 ||
         date[8] != ' ' ||
         date[11] != ':' ||
         date[14] != ':')
        return(NULL);

    time(&now);
    _tm = *localtime(&now);
    _tm.tm_year = (date[0]-'0')*1000 + (date[1]-'0')*100 +
                   (date[2]-'0')*10   + (date[3]-'0')-1900;
    _tm.tm_mon  = (date[4]-'0')*10   + (date[5]-'0')-1;
    _tm.tm_mday = (date[6]-'0')*10   + (date[7]-'0');
    _tm.tm_hour = (date[9]-'0')*10   + (date[10]-'0');
    _tm.tm_min  = (date[12]-'0')*10  + (date[13]-'0');
    _tm.tm_sec  = (date[15]-'0')*10  + (date[16]-'0');

    _tm.tm_wday = 0;
    _tm.tm_yday = 0;

    return(&_tm);
}

int
#ifdef CK_ANSIC
ck_krb5_initTGT( struct krb_op_data * op, struct krb5_init_data * init )
#else
ck_krb5_initTGT(op,init)
    krb_op_data * op; struct krb5_init_data * init;
#endif /* CK_ANSIC*/
{
    krb5_context kcontext;
    krb5_ccache ccache = NULL;
    krb5_deltat lifetime = KRB5_DEFAULT_LIFE;	/* -l option */
    krb5_timestamp starttime = 0;
    krb5_deltat rlife = 0;
    int options = KRB5_DEFAULT_OPTIONS;
    int option;
    int errflg = 0;
    krb5_error_code code;
    krb5_principal me;
    krb5_principal server;
    krb5_creds my_creds;
    krb5_timestamp now;
    krb5_address *null_addr = (krb5_address *)0;
    krb5_address **addrs = (krb5_address **)0;
#ifndef NO_KEYTAB
    int use_keytab = 0;			/* -k option */
    krb5_keytab keytab = NULL;
#endif /* NO_KEYTAB */
    struct passwd *pw = 0;
    int pwsize;
    char *client_name=NULL, realm[256]="", numstr[40]="";

#ifdef COMMENT
    printf("Kerberos V initialization\n");
#endif /* COMMENT */

    code = krb5_init_context(&kcontext);
    if (code) {
        com_err("krb5_kinit",code,"while init_context");
        return(-1);
    }

    if ((code = krb5_timeofday(kcontext, &now))) {
        com_err("krb5_kinit",code,"while getting time of day");
        goto exit_k5_init;
    }

    if ( init->renewable ) {
        options |= KDC_OPT_RENEWABLE;
        sprintf(numstr,"%dm",init->renewable);
        code = krb5_string_to_deltat(numstr, &rlife);
        if (code != 0 || rlife == 0) {
            printf("Bad renewable time value %s\n", numstr);
            errflg++;
        }
    }
    if ( init->renew ) {
        /* renew the ticket */
        options |= KDC_OPT_RENEW;
    }

    if ( init->validate ) {
        /* validate the ticket */
        options |= KDC_OPT_VALIDATE;
    }
    if ( init->proxiable ) {
        options |= KDC_OPT_PROXIABLE;
    }
    if ( init->forwardable ) {
        options |= KDC_OPT_FORWARDABLE;
    }
#ifndef NO_KEYTAB
    if (  ) {
        use_keytab = 1;
    }
    if (  ) {
        if (keytab == NULL && keytab_name != NULL) {
            code = krb5_kt_resolve(kcontext, keytab_name, &keytab);
            if (code != 0) {
                debug(F111,"krb5_init resolving keytab",
                         keytab_name,code);
                errflg++;
            }
        }
    }
#endif
    if ( init->lifetime ) {
        sprintf(numstr,"%dm",init->lifetime);
        code = krb5_string_to_deltat(numstr, &lifetime);
        if (code != 0 || lifetime == 0) {
            printf("Bad lifetime value %s\n", numstr);
            errflg++;
        }
    }
    if ( init->postdate ) {
        /* Convert cmdate() to a time_t value */
        struct tm * time_tm;
        time_tm = cmdate2tm(init->postdate);
        if ( time_tm )
            starttime = (krb5_timestamp) mktime(time_tm);

        if (code != 0 || starttime == 0 || starttime == -1) {
            krb5_deltat ktmp;
            code = krb5_string_to_deltat(init->postdate, &ktmp);
            if (code == 0 && ktmp != 0) {
		starttime = now + ktmp;
		options |= KDC_OPT_POSTDATED;
            } else {
		printf("Bad postdate start time value %s\n",
                        init->postdate);
		errflg++;
            }
        } else {
            options |= KDC_OPT_POSTDATED;
        }
    }
    if ( op->cache ) {
        if (ccache == NULL) {
            char cc_tmp[260];
            sprintf(cc_tmp,"FILE:%s",op->cache);
            code = krb5_cc_resolve (kcontext, cc_tmp, &ccache);
            if (code != 0) {
                com_err("krb5_kinit resolving ccache",code,
                          op->cache);
                errflg++;
            }
        }
    }

    if (ccache == NULL) {
        if ((code = krb5_cc_default(kcontext, &ccache))) {
            com_err("krb5_kinit",code,"while getting default ccache");
            goto exit_k5_init;
        }
    }

    if (init->principal == NULL) {       /* No principal name specified */
#ifndef NO_KEYTAB
        if (use_keytab) {
            /* Use the default host/service name */
            code = krb5_sname_to_principal(kcontext, NULL, NULL,
                                            KRB5_NT_SRV_HST, &me);
            if (code) {
                com_err("krb5_kinit",
			code,
			"when creating default server principal name");
                goto exit_k5_init;
            }
        } else
#endif
        {
            /* Get default principal from cache if one exists */
            code = krb5_cc_get_principal(kcontext, ccache, &me);
            if (code) {
#ifdef HAVE_PWD_H
                /* Else search passwd file for client */
                pw = getpwuid((int) getuid());
                if (pw) {
                    if ((code = krb5_parse_name(kcontext,pw->pw_name,
                                                 &me))) {
                        com_err("krb5_kinit",code,"when parsing name",
                                  pw->pw_name);
                        goto exit_k5_init;
                    }
                } else {
                    printf(
                        "Unable to identify user from password file\n");
                    goto exit_k5_init;
                }
#else /* HAVE_PWD_H */
                printf("Unable to identify user\n");
                goto exit_k5_init;
#endif /* HAVE_PWD_H */
            }
        }
    } /* Use specified name */	
    else if ((code = krb5_parse_name (kcontext, init->principal, &me))) {
	 com_err("krb5_kinit",code,"when parsing name",
                init->principal);
        goto exit_k5_init;
    }

    if ( init->realm == NULL ) {
        /* Save the realm */
        memcpy(realm,krb5_princ_realm(kcontext, me)->data,
                krb5_princ_realm(kcontext, me)->length);
        realm[krb5_princ_realm(kcontext, me)->length]='\0';
    } else {
        strcpy(realm,init->realm);
    }

    if ((code = krb5_unparse_name(kcontext, me, &client_name))) {
	com_err("krb5_kinit",code,"when unparsing name");
        goto exit_k5_init;
    }

    memset((char *)&my_creds, 0, sizeof(my_creds));

    my_creds.client = me;

    if (init->service == NULL) {
        if((code = krb5_build_principal_ext(kcontext, &server,
#ifdef COMMENT
					     krb5_princ_realm(kcontext, me)->length,
					     krb5_princ_realm(kcontext, me)->data,
#else /* COMMENT */
                                             strlen(realm),realm,
#endif /* COMMENT */
					     tgtname.length, tgtname.data,
#ifdef COMMENT
					     krb5_princ_realm(kcontext, me)->length,
					     krb5_princ_realm(kcontext, me)->data,
#else /* COMMENT */
                                             strlen(realm),realm,
#endif /* COMMENT */
					     0))) {
            com_err("krb5_kinit",code,"while building server name");
            goto exit_k5_init;
        }
    } else {
        if (code = krb5_parse_name(kcontext, init->service, &server)) {
            com_err("krb5_kinit",code,"while parsing service name",
                   init->service);
            goto exit_k5_init;
        }
    }
	
    my_creds.server = server;

    if (options & KDC_OPT_POSTDATED) {
        my_creds.times.starttime = starttime;
        my_creds.times.endtime = starttime + lifetime;
    } else {
        my_creds.times.starttime = 0;	/* start timer when request
					   gets to KDC */
        my_creds.times.endtime = now + lifetime;
    }
    if (options & KDC_OPT_RENEWABLE) {
	my_creds.times.renew_till = now + rlife;
    } else
	my_creds.times.renew_till = 0;

    if (options & KDC_OPT_VALIDATE) {
        /* don't use get_in_tkt, just use mk_req... */
        krb5_data outbuf;

        code = krb5_validate_tgt(kcontext, ccache, server, &outbuf);
	if (code) {
            com_err("krb5_kinit",code,"validating tgt");
            goto exit_k5_init;
	}
	/* should be done... */
        goto exit_k5_init;
    }

    if (options & KDC_OPT_RENEW) {
        /* don't use get_in_tkt, just use mk_req... */
        krb5_data outbuf;

        code = krb5_renew_tgt(kcontext, ccache, server, &outbuf);
	if (code) {
            com_err("krb5_kinit",code,"while renewing tgt");
            goto exit_k5_init;
	}
	/* should be done... */
        goto exit_k5_init;
    }

#ifndef NO_KEYTAB
    if (!use_keytab)
#endif
    {
#ifdef COMMENT
        (void) sprintf(prompt,"Password for %s: ", (char *) client_name);

        pwsize = sizeof(password);

        code = krb5_read_password(kcontext, prompt, 0, password, &pwsize);
        if (code || pwsize == 0) {
	      fprintf(stderr, "Error while reading password for '%s'\n",
		      client_name);
	      memset(password, 0, sizeof(password));
	      exit(1);
	 }
#else /* COMMENT */
        pwsize = strlen(init->password);
        if (pwsize == 0) {
            printf("A password must be specified for %s.\n",
		      client_name);
            memset(init->password, 0, pwsize);
            goto exit_k5_init;
        }
#endif /* COMMENT */

	 code = krb5_get_in_tkt_with_password(kcontext, options, addrs,
					      NULL, preauth, init->password,
                                               0, &my_creds, 0);
	 memset(init->password, 0, pwsize);
#ifndef NO_KEYTAB
    } else {
	 code = krb5_get_in_tkt_with_keytab(kcontext, options, addrs,
					    NULL, preauth, keytab, 0,
					    &my_creds, 0);
#endif
    }

    if (code) {
	if (code == KRB5KRB_AP_ERR_BAD_INTEGRITY)
	    printf("Password incorrect\n");
	else
	    com_err("krb5_kinit",code,"while getting initial credentials");
        goto exit_k5_init;
    }

    code = krb5_cc_initialize (kcontext, ccache, me);
    if (code != 0) {
	com_err("krb5_kinit",code,"when initializing cache",
		 op->cache);
        goto exit_k5_init;
    }

    code = krb5_cc_store_cred(kcontext, ccache, &my_creds);
    if (code) {
	com_err("krb5_kinit",code,"while storing credentials");
        goto exit_k5_init;
    }

exit_k5_init:

    /* my_creds is pointing at server */
    krb5_free_principal(kcontext, server);
    krb5_free_context(kcontext);

    printf("Result from realm %s: %s\n",realm,
            code?error_message(code):"OK");
    return(code?-1:0);
}

#define VALIDATE 0
#define RENEW 1

/* stripped down version of krb5_mk_req */
static krb5_error_code
#ifdef CK_ANSIC
krb5_validate_tgt( krb5_context context,
                   krb5_ccache ccache,
                   krb5_principal     server, /* tgtname */
                   krb5_data *outbuf )
#else
krb5_validate_tgt(context, ccache, server, outbuf)
     krb5_context context;
     krb5_ccache ccache;
     krb5_principal     server; /* tgtname */
     krb5_data *outbuf;
#endif
{
    return krb5_tgt_gen(context, ccache, server, outbuf, VALIDATE);
}

/* stripped down version of krb5_mk_req */
static krb5_error_code
#ifdef CK_ANSIC
krb5_renew_tgt(krb5_context context,
                krb5_ccache ccache,
                krb5_principal	  server, /* tgtname */
                krb5_data *outbuf)
#else
krb5_renew_tgt(context, ccache, server, outbuf)
     krb5_context context;
     krb5_ccache ccache;
     krb5_principal	  server; /* tgtname */
     krb5_data *outbuf;
#endif
{
    return krb5_tgt_gen(context, ccache, server, outbuf, RENEW);
}


/* stripped down version of krb5_mk_req */
static krb5_error_code
#ifdef CK_ANSIC
krb5_tgt_gen(krb5_context context,
              krb5_ccache ccache,
              krb5_principal	  server, /* tgtname */
              krb5_data *outbuf,
              int opt)
#else
krb5_tgt_gen(context, ccache, server, outbuf, opt)
     krb5_context context;
     krb5_ccache ccache;
     krb5_principal	  server; /* tgtname */
     krb5_data *outbuf;
     int opt;
#endif
{
    krb5_error_code 	  retval;
    krb5_creds 		* credsp;
    krb5_creds 		  creds;

    /* obtain ticket & session key */
    memset((char *)&creds, 0, sizeof(creds));
    if ((retval = krb5_copy_principal(context, server, &creds.server)))
	goto cleanup;

    if ((retval = krb5_cc_get_principal(context, ccache, &creds.client)))
	goto cleanup_creds;

    if(opt == VALIDATE) {
	    if ((retval = krb5_get_credentials_validate(context, 0,
							ccache, &creds, &credsp)))
		    goto cleanup_creds;
    } else {
	    if ((retval = krb5_get_credentials_renew(context, 0,
							ccache, &creds, &credsp)))
		    goto cleanup_creds;
    }

    /* we don't actually need to do the mk_req, just get the creds. */
cleanup_creds:
    krb5_free_cred_contents(context, &creds);

cleanup:

    return retval;
}
#endif /* KINIT */
#ifdef KDESTROY
#include "krb5.h"
#include "com_err.h"
#include <string.h>
#include <stdio.h>

int
#ifdef CK_ANSIC
ck_krb5_destroy(struct krb_op_data * op)
#else
ck_krb5_destroy(op) struct krb_op_data * op;
#endif
{
    krb5_context kcontext;
    krb5_error_code retval;
    int c;
    krb5_ccache cache = NULL;
    char *cache_name = NULL;
    int code;
    int errflg=0;
    int quiet = 0;	

    retval = krb5_init_context(&kcontext);
    if (retval) {
        debug(F101,"ck_krb5_destroy while initializing krb5","",retval);
        return(-1);
    }

    if (op->cache != NULL) {
        char cc_tmp[260];
        sprintf(cc_tmp,"FILE:%s",op->cache);
        cache_name = op->cache;
        code = krb5_cc_resolve (kcontext, cc_tmp, &cache);
        if (code != 0) {
            debug(F111,"ck_krb5_destroy while resolving", cache_name,code);
            errflg++;
        }
    }

    if (cache == NULL) {
	if (code = krb5_cc_default(kcontext, &cache)) {
	    debug(F101,"ck_krb5_destroy while getting default ccache",
                   "",code);
            krb5_free_context(kcontext);
	    return(-1);
	}
    }

    code = krb5_cc_destroy (kcontext, cache);
    if (code != 0) {
	debug(F101,"ck_krb5_destroy while destroying cache","",code);
        if ( code == KRB5_FCC_NOFILE )
            printf("No ticket cache to destroy.\n");
        else
            printf("Ticket cache NOT destroyed!\n");
        krb5_free_context(kcontext);
	return(-1);
    }
    krb5_free_context(kcontext);
    return (0);
}
#endif /* KDESTROY */
#ifdef KLIST
#include "k5-int.h"
#include "com_err.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

static int show_flags = 0, show_time = 0, status_only = 0, show_keys = 0;
static int show_etype = 0;
static char *defname;
static char *progname;
static krb5_int32 now;
static int timestamp_width;

static char * etype_string KRB5_PROTOTYPE((krb5_enctype ));
static void show_credential KRB5_PROTOTYPE((krb5_context,
                                             krb5_creds *));
	
static int do_ccache KRB5_PROTOTYPE((krb5_context,char *));
static int do_keytab KRB5_PROTOTYPE((krb5_context,char *));
static void printtime KRB5_PROTOTYPE((time_t));
static void fillit KRB5_PROTOTYPE((int, int));
	
#define DEFAULT 0
#define CCACHE 1
#define KEYTAB 2

#ifdef COMMENT
    /* Kept just so we know what the command line looks like for the */
    /* standalone program */
void usage()
{
     fprintf(stderr, "Usage: %s [[-c] [-f] [-e] [-s]] [-k [-t] [-K]] [name]\n",
	     progname);
     fprintf(stderr, "\t-c specifies credentials cache, -k specifies keytab");
     fprintf(stderr, ", -c is default\n");
     fprintf(stderr, "\toptions for credential caches:\n");
     fprintf(stderr, "\t\t-f shows credentials flags\n");
     fprintf(stderr, "\t\t-e shows the encryption type\n");
     fprintf(stderr, "\t\t-s sets exit status based on valid tgt existence\n");
     fprintf(stderr, "\toptions for keytabs:\n");
     fprintf(stderr, "\t\t-t shows keytab entry timestamps\n");
     fprintf(stderr, "\t\t-K shows keytab entry DES keys\n");
     exit(1);
}
#endif /* COMMENT */

int
#ifdef CK_ANSIC
ck_krb5_list_creds(struct krb_op_data * op, struct krb5_list_cred_data * lc)
#else
ck_krb5_list_creds(op,lc)
    struct krb_op_data * op; struct krb5_list_cred_data * lc;
#endif
{
    krb5_context kcontext;
    krb5_error_code retval;
    int code;
    char *name = op->cache;
    int mode;

    retval = krb5_init_context(&kcontext);
    if (retval) {
        debug(F101,"ck_krb5_list_creds while initializing krb5","",retval);
        return(-1);
    }

    name = op->cache;
    mode = DEFAULT;
    show_flags = 0;
    show_time = 0;
    status_only = 0;
    show_keys = 0;
    show_etype = 0;

#ifdef COMMENT
    while (*argv) {
	if ((*argv)[0] != '-') {
	    if (name) usage();
	    name = *argv;
	} else switch ((*argv)[1]) {
	case 'f':
	    show_flags = 1;
	    break;
	case 'e':
	    show_etype = 1;
	    break;
	case 't':
	    show_time = 1;
	    break;
	case 'K':
	    show_keys = 1;
	    break;
	case 's':
	    status_only = 1;
	    break;
	case 'c':
	    if (mode != DEFAULT) usage();
	    mode = CCACHE;
	    break;
	case 'k':
	    if (mode != DEFAULT) usage();
	    mode = KEYTAB;
	    break;
	default:
	    usage();
	    break;
	}
    }
#else /* COMMENT */
    show_flags = lc->flags;
    show_etype = lc->encryption;
    show_time = 1;
    show_keys = 1;
    mode = CCACHE;
#endif /* COMMENT */

    if ((code = krb5_timeofday(kcontext, &now))) {
        if (!status_only)
            debug(F101,"ck_krb5_list_creds while getting time of day.",
                   "",code);
        krb5_free_context(kcontext);
        return(-1);
    }
    else {
	char tmp[BUFSIZ];

	if (!krb5_timestamp_to_sfstring(now, tmp, 20, (char *) NULL) ||
	    !krb5_timestamp_to_sfstring(now, tmp, sizeof(tmp), (char *) NULL))
	    timestamp_width = (int) strlen(tmp);
	else
	    timestamp_width = 15;
    }

#ifdef COMMENT
    printf("Kerberos V\n");
    printf("----------\n");
#endif /* COMMENT */
    if (mode == DEFAULT || mode == CCACHE)
	 retval = do_ccache(kcontext,name);
    else
	 retval = do_keytab(kcontext,name);
    krb5_free_context(kcontext);
    return(retval);
}

static int
#ifdef CK_ANSIC
do_keytab(krb5_context kcontext, char * name)
#else
do_keytab(kcontext,name) krb5_context kcontext; char * name;
#endif
{
    krb5_keytab kt;
    krb5_keytab_entry entry;
    krb5_kt_cursor cursor;
    char buf[BUFSIZ]; /* hopefully large enough for any type */
    char *pname;
    int code;

    if (name == NULL) {
        if ((code = krb5_kt_default(kcontext, &kt))) {
            debug(F101,"ck_krb5_list_creds while getting default keytab",
                   "",code);
            return(-1);
        }
    } else {
        if ((code = krb5_kt_resolve(kcontext, name, &kt))) {
            debug(F111,"ck_krb5_list_creds while resolving keytab",
                     name,code);
            return(-1);
        }
    }

    if ((code = krb5_kt_get_name(kcontext, kt, buf, BUFSIZ))) {
        debug(F101,"ck_krb5_list_creds while getting keytab name",
               "",code);
        return(-1);
    }

     printf("Keytab name: %s\n", buf);

     if ((code = krb5_kt_start_seq_get(kcontext, kt, &cursor))) {
         debug(F101,"ck_krb5_list_creds while starting keytab scan",
                "",code);
	  return(-1);
     }

     if (show_time) {
	  printf("KVNO Timestamp");
          fillit(timestamp_width - sizeof("Timestamp") + 2, (int) ' ');
	  printf("Principal\n");
	  printf("---- ");
	  fillit(timestamp_width, (int) '-');
	  printf(" ");
	  fillit(78 - timestamp_width - sizeof("KVNO"), (int) '-');
	  printf("\n");
     } else {
	  printf("KVNO Principal\n");
	  printf(
"---- --------------------------------------------------------------------\
------\n");
     }

    while ((code = krb5_kt_next_entry(kcontext, kt, &entry, &cursor)) == 0) {
        if ((code = krb5_unparse_name(kcontext, entry.principal, &pname))) {
            debug(F101,"ck_krb5_list_creds while unparsing principal name",
                   "",code);
            return(-1);
        }
        printf("%4d ", entry.vno);
        if (show_time) {
            printtime(entry.timestamp);
            printf(" ");
        }
        printf("%s", pname);
        if (show_etype)
            printf(" (%s) " , etype_string(entry.key.enctype));
        if (show_keys) {
            printf(" (0x");
            {
                int i;
                for (i = 0; i < entry.key.length; i++)
                    printf("%02x", entry.key.contents[i]);
            }
            printf(")");
        }
        printf("\n");
        free(pname);
    }
    if (code && code != KRB5_KT_END) {
        debug(F101,"ck_krb5_list_creds while scanning keytab",
               "",code);
        return(-1);
    }
    if ((code = krb5_kt_end_seq_get(kcontext, kt, &cursor))) {
        debug(F101,"ck_krb5_list_creds while ending keytab scan",
               "",code);
        return(-1);
    }
    return(0);
}

static int
#ifdef CK_ANSIC
do_ccache(krb5_context kcontext, char * name)
#else
do_ccache(kcontext,name) krb5_context kcontext; char * name;
#endif
{
    krb5_ccache cache = NULL;
    krb5_cc_cursor cur;
    krb5_creds creds;
    krb5_principal princ;
    krb5_flags flags;
    krb5_error_code code;
    int	exit_status = 0;
	
    if (status_only)
	/* exit_status is set back to 0 if a valid tgt is found */
	exit_status = 1;

    if (name == NULL) {
	if ((code = krb5_cc_default(kcontext, &cache))) {
            debug(F101,"ck_krb5_list_creds while getting default ccache",
                   "",code);
	    return(-1);
        }
    } else {
        char cc_tmp[260];
        sprintf(cc_tmp,"FILE:%s",name);
	if ((code = krb5_cc_resolve(kcontext, cc_tmp, &cache))) {
            debug(F111,"ck_krb5_list_creds while resolving ccache",
                   name,code);
	    return(-1);
	}
    }

    flags = 0;				/* turns off OPENCLOSE mode */
    if ((code = krb5_cc_set_flags(kcontext, cache, flags))) {
	if (code == ENOENT) {
            debug(F111,"ck_krb5_list_creds (ticket cache)",
                   krb5_cc_get_name(kcontext, cache),code);
	} else {
            debug(F111,
		 "ck_krb5_list_creds while setting cache flags (ticket cache)",
                  krb5_cc_get_name(kcontext, cache),code);
	}
        printf("No ticket File.\n");
	return(-1);
    }
    if ((code = krb5_cc_get_principal(kcontext, cache, &princ))) {
        debug(F101,"ck_krb5_list_creds while retrieving principal name",
               "",code);
	return(-1);
    }
    if ((code = krb5_unparse_name(kcontext, princ, &defname))) {
        debug(F101,"ck_krb5_list_creds while unparsing principal name",
               "",code);
	return(-1);
    }
    if (!status_only) {
	printf("Ticket cache:      %s\nDefault principal: %s\n\n",
	       krb5_cc_get_name(kcontext, cache), defname);
	printf("Valid starting");
	fillit(timestamp_width - sizeof("Valid starting") + 3,
	       (int) ' ');
	printf("Expires");
	fillit(timestamp_width - sizeof("Expires") + 3,
	       (int) ' ');
	printf("Service principal\n");
    }
    if ((code = krb5_cc_start_seq_get(kcontext, cache, &cur))) {
        debug(F101,"ck_krb5_list_creds while starting to retrieve tickets",
               "",code);
	return(-1);
    }
    while (!(code = krb5_cc_next_cred(kcontext, cache, &cur, &creds))) {
	if (status_only) {
	    if (exit_status && creds.server->length == 2 &&
		strcmp(creds.server->realm.data, princ->realm.data) == 0 &&
		strcmp((char *)creds.server->data[0].data, "krbtgt") == 0 &&
		strcmp((char *)creds.server->data[1].data,
		       princ->realm.data) == 0 &&
		creds.times.endtime > now)
		exit_status = 0;
	} else {
	    show_credential(kcontext, &creds);
	}
	krb5_free_cred_contents(kcontext, &creds);
    }
    printf("\n");
    if (code == KRB5_CC_END) {
	if ((code = krb5_cc_end_seq_get(kcontext, cache, &cur))) {
            debug(F101,"ck_krb5_list_creds while finishing ticket retrieval",
                   "",code);
	    return(-1);
	}
	flags = KRB5_TC_OPENCLOSE;	/* turns on OPENCLOSE mode */
	if ((code = krb5_cc_set_flags(kcontext, cache, flags))) {
            debug(F101,"ck_krb5_list_creds while closing ccache",
                   "",code);
	    return(-1);
	}
	return(exit_status);
    } else {
        debug(F101,"ck_krb5_list_creds while retrieving a ticket","",code);
	return(-1);
    }	
    return(0);
}

static char *
#ifdef CK_ANSIC
etype_string(krb5_enctype enctype)
#else
etype_string(enctype) krb5_enctype enctype;
#endif
{
    static char buf[12];

    switch (enctype) {
    case ENCTYPE_DES_CBC_CRC:
	return "DES-CBC-CRC";
	break;
    case ENCTYPE_DES_CBC_MD4:
	return "DES-CBC-MD4";
	break;
    case ENCTYPE_DES_CBC_MD5:
	return "DES-CBC-MD5";
	break;
    case ENCTYPE_DES3_CBC_SHA:
	return "DES3-CBC-SHA";
	break;
    default:
	sprintf(buf, "etype %d", enctype);
	return buf;
	break;
    }
}

static char *
#ifdef CK_ANSIC
flags_string(register krb5_creds *cred)
#else
flags_string(cred) register krb5_creds *cred;
#endif
{
    static char buf[32];
    int i = 0;
	
    if (cred->ticket_flags & TKT_FLG_FORWARDABLE)
        buf[i++] = 'F';
    if (cred->ticket_flags & TKT_FLG_FORWARDED)
        buf[i++] = 'f';
    if (cred->ticket_flags & TKT_FLG_PROXIABLE)
        buf[i++] = 'P';
    if (cred->ticket_flags & TKT_FLG_PROXY)
        buf[i++] = 'p';
    if (cred->ticket_flags & TKT_FLG_MAY_POSTDATE)
        buf[i++] = 'D';
    if (cred->ticket_flags & TKT_FLG_POSTDATED)
        buf[i++] = 'd';
    if (cred->ticket_flags & TKT_FLG_INVALID)
        buf[i++] = 'i';
    if (cred->ticket_flags & TKT_FLG_RENEWABLE)
        buf[i++] = 'R';
    if (cred->ticket_flags & TKT_FLG_INITIAL)
        buf[i++] = 'I';
    if (cred->ticket_flags & TKT_FLG_HW_AUTH)
        buf[i++] = 'H';
    if (cred->ticket_flags & TKT_FLG_PRE_AUTH)
        buf[i++] = 'A';
    buf[i] = '\0';
    return(buf);
}

static char   *
#ifdef CK_ANSIC
short_date(long   *dp)
#else
short_date(dp) long   *dp;
#endif
{
    register char *cp;
    extern char *ctime();
    cp = ctime(dp) + 4;
    cp[15] = '\0';
    return (cp);
}


static VOID
#ifdef CK_ANSIC
printtime(time_t tv)
#else
printtime(tv) time_t tv;
#endif
{
    char timestring[BUFSIZ];
    char format[12];
    char fill;

    fill = ' ';
    sprintf(format,"%%-%ds",timestamp_width);
    if (!krb5_timestamp_to_sfstring((krb5_timestamp) tv,
                                     timestring,
                                     timestamp_width+1,
				     &fill)) {
        printf(format,timestring);
    }
    else {
        printf(format,short_date(&tv));
    }

}

static VOID
#ifdef CK_ANSIC
show_credential(krb5_context kcontext, register krb5_creds * cred)
#else
show_credential(kcontext, cred)
    krb5_context  	  kcontext;
    register krb5_creds * cred;
#endif
{
    krb5_error_code retval=0;
    krb5_ticket *tkt=NULL;
    char *name=NULL, *sname=NULL, *flags=NULL;
    int	extra_field = 0;

    retval = krb5_unparse_name(kcontext, cred->client, &name);
    if (retval) {
	debug(F101,"ck_krb5_list_creds while unparsing client name","",retval);
	return;
    }
    retval = krb5_unparse_name(kcontext, cred->server, &sname);
    if (retval) {
	debug(F101,"ck_krb5_list_creds while unparsing server name","",retval);
	free(name);
	return;
    }
    if (!cred->times.starttime)
	cred->times.starttime = cred->times.authtime;

    printtime(cred->times.starttime);
    printf("  ");

    if ( time(0) < cred->times.endtime )
        printtime(cred->times.endtime);
    else
        printf("** expired ** ");

    printf("  %s\n", sname);

    if (strcmp(name, defname)) {
        printf("\tfor client %s", name);
        extra_field++;
    }

    if (cred->times.renew_till) {
	if (!extra_field)
            printf("\t");
	else
            printf(", ");
	printf("renew until ");
        printtime(cred->times.renew_till);
	extra_field += 2;
    }

    if (extra_field > 3) {
	printf("\n");
	extra_field = 0;
    }

    if (show_flags) {
	flags = flags_string(cred);
	if (flags && *flags) {
	    if (!extra_field)
		printf("\t");
	    else
		printf(", ");
	    printf("Flags: %s", flags);
	    extra_field++;
        }
    }

    if (extra_field > 2) {
	printf("\n");
	extra_field = 0;
    }

    if (show_etype) {
	retval = decode_krb5_ticket(&cred->ticket, &tkt);
	if (!extra_field)
	    printf("\t");
	else
	    printf(", ");
	printf("Etype (skey, tkt): %s, %s ",
	       etype_string(cred->keyblock.enctype),
	       etype_string(tkt->enc_part.enctype));
	krb5_free_ticket(kcontext, tkt);
	extra_field++;
    }

    /* if any additional info was printed, extra_field is non-zero */
    if (extra_field)
	printf("\n");

    free(name);
    free(sname);
}

static VOID
#ifdef CK_ANSIC
fillit(int num, int c)
#else
fillit(num, c) int num; int c;
#endif
{
    int i;

    for (i=0; i<num; i++)
	printf("%c",c);
}
#endif /* KLIST */
#endif /* KRB5 */

#ifdef KRB4
#define KDEBUG 0

#ifdef KINIT
#define KRB_DEFAULT_LIFE 120 /* 10 hours in 5 minute intervals */

#ifdef SNK4
/* SNK4 is a hardware authentication system used to pre-authenticate    */
/* a ticket getting ticket.  We do not support this code at the present */
/* time in Kermit.                                                      */
void
get_input(s, size, stream)
char *s;
int size;
FILE *stream;
{
    char *p;

    if (fgets(s, size, stream) == NULL)
        exit(1);
    if ( (p = strchr(s, '\n')) != NULL)
        *p = '\0';
}
#endif /* SNK4 */

static char
#ifdef CK_ANSIC
hex_scan_nybble(char c)
#else
hex_scan_nybble(c) char c;
#endif
{
    if (c >= '0' && c <= '9')
        return c - '0';
    if (c >= 'A' && c <= 'F')
        return c - 'A' + 10;
    if (c >= 'a' && c <= 'f')
        return c - 'a' + 10;
    return -1;
}

/* returns: NULL for ok, pointer to error string for bad input */
static char*
#ifdef CK_ANSIC
hex_scan_four_bytes(char *out, char *in)
#else
hex_scan_four_bytes(out, in) char *out; char *in;
#endif
{
  int i;
  char c, c1;
  for (i=0; i<8; i++) {
    if(!in[i]) return "not enough input";
    c = hex_scan_nybble(in[i]);
    if(c<0) return "invalid digit";
    c1 = c; i++;
    if(!in[i]) return "not enough input";
    c = hex_scan_nybble(in[i]);
    if(c<0) return "invalid digit";
    *out++ = (c1 << 4) + c;
  }
  switch(in[i]) {
  case 0:
  case '\r':
  case '\n':
    return 0;
  default:
    return "extra characters at end of input";
  }
}

int
#ifdef CK_ANSIC
ck_krb4_initTGT(struct krb_op_data * op, struct krb4_init_data * init)
#else
ck_krb4_initTGT(op,init)
    struct krb_op_data * op, struct krb4_init_data * init
#endif
{
    char    aname[ANAME_SZ];
    char    inst[INST_SZ];
    char    realm[REALM_SZ];
    char    *password=NULL;
    char   *username = NULL;
    char   *usernameptr=NULL;
    int     iflag,      /* Instance */
            rflag,      /* Realm */
            vflag,      /* Verbose */
            lflag,      /* Lifetime */
            pflag,      /* Preauth */
            lifetime=KRB_DEFAULT_LIFE,   /* Life Time */
            k_errno;
    register char *cp;
    register i;

#if KDEBUG
/* This code is for testing time conversion in micro Kerberos ports */
    {
	extern int krb_debug;
	static long time_0 = 0L;
	static long time_1 = 1L;
	static long time_epoch;
	static long time_epoch2;

	krb_debug = 0xFFF;
	time_epoch = win_time_get_epoch();
	time_epoch2 = -time_epoch;
	printf ("STime at zero: %s\n", krb_stime(&time_0));
	printf ("STime at one: %s\n", krb_stime(&time_1));
	printf ("STime at Epoch: %s\n", krb_stime(&time_epoch));
	printf ("STime at -Epoch: %s\n", krb_stime(&time_epoch2));
	printf ("CTime at zero: %s\n", ctime(&time_0));
	printf ("CTime at one: %s\n", ctime(&time_1));
	printf ("CTime at Epoch: %s\n", ctime(&time_epoch));
	printf ("CTime at -Epoch: %s\n", ctime(&time_epoch2));
    }
#endif

    *inst = *realm = '\0';
    iflag = rflag = vflag = lflag = pflag = 0;

    if ( init->lifetime ) {
        lifetime = init->lifetime<5?1:init->lifetime/5;
        if ( lifetime > 255 ) lifetime = 255;
    }
    else
        lifetime = KRB_DEFAULT_LIFE;

    username = init->principal;
    password = init->password;
    vflag = 1;

    if (username &&
	(k_errno = kname_parse(aname, inst, realm, username))
	!= KSUCCESS) {
	printf("%s\n", krb_get_err_text_entry(k_errno));
	iflag = rflag = 1;
	username = NULL;
    }

    if ( init->realm )
        strcpy(realm,init->realm);

    if ( init->instance )
        strcpy(inst,init->instance);

#ifdef COMMENT
    if ( vflag )
        printf("Kerberos IV initialization\n");
#endif

    if (!username) {
        debug(F100,"ck_krb4_initTGT no username specified","",0);
        printf("?No principal specified\n");
        return(-1);
    }
    if (!*realm && krb_get_lrealm(realm, 1)) {
	printf("Error attempting to determine local Kerberos realm name\n");
	return(-1);
    }

    if (pflag) {
        k_errno = krb_get_pw_in_tkt_preauth( aname, inst, realm,
                                             "krbtgt", realm,
                                             lifetime,
                                             password[0]? password:
                                             NULL);
    } else {
        k_errno = krb_get_pw_in_tkt(aname,
                                     inst, realm,
                                     "krbtgt", realm,
                                     lifetime,
                                     password[0]? password:
                                     NULL);
    }

    if (vflag) {
	printf("Result from realm %s: ", realm);
	printf("%s\n", krb_get_err_text_entry(k_errno));
    } else if (k_errno) {
        printf("%s for principal %s%s%s@%s\n",
		krb_get_err_text_entry(k_errno), aname,
                inst[0]?".":"", inst, realm);
	return(-1);
    }

#if KDEBUG
    /* More testing code for micro ports */

    usernameptr = krb_get_default_user();
    printf ("Default user is `%s'.\n", usernameptr);

    k_errno = krb_set_default_user ("Bogon");
    printf ("Set default user result is %d\n", k_errno);

    printf ("Now default user is `%s'.\n", krb_get_default_user());

    k_errno = krb_set_default_user (usernameptr);

    /* Test credentials caching code
     * Tests are order dependent and depend on the results of previous tests.
     */
    {
        int numcreds;
        CREDENTIALS cr;
        char    aname[ANAME_SZ];
        char    inst[INST_SZ];
        char    realm[REALM_SZ];

        /* krb_get_num_cred
        * returns the number of credentials in the default cache.
        */
        numcreds = krb_get_num_cred();
        printf("After getting tgt...\n");
        printf("The number of credentials in the default cache:%d.\n",
	       numcreds);

        /* krb_get_tf_fullname
        * For the ticket file name input, return the principal's name,
        * instance, and realm.	 Currently the ticket file name is
        * unused because there is just one default ticket file/cache.
        */

        printf ("krb_get_tf_fullname returns %d.\n",
                 krb_get_tf_fullname (op->cache, aname, inst, realm));
        printf ("principal's name: %s, instance: %s, realm: %s. \n",
                 aname, inst, realm);		

	/* krb_get_nth_cred
	 * returns service name, service instance
	 * and realm of the nth credential.
	 */

	printf (
	"Get the service name, instance, and realm of the last credential.\n");
	printf ("krb_get_nth_cred returns %d.\n",
                 krb_get_nth_cred(aname, inst, realm, numcreds));
	printf ("service name: %s, instance: %s, realm: %s.\n",
                 aname, inst, realm);	
    		
        /* krb_get_cred
        * fills in a cred with info from the first cred in the default cache
        * with matching sname, inst, and realm
        */
        printf ("Make a new cred and fill it in with info from the cred\n");
        printf (
	"in the default cache with matching service name, inst and realm.\n");

        printf ("krb_get_cred returns: %d.\n",
		krb_get_cred(aname, inst, realm, &cr));

        printf ("service: %s, instance: %s, realm: %s\n",
		cr.service, cr.instance, cr.realm);

  	printf ("pname: %s, pinst: %s.\n", cr.pname, cr.pinst);

        /* krb_save_credentials
        * adds a new credential to the default cache, using information input
        */
        printf (
"Add a new cred to the cache, using mostly values from the cred made above\n");
        printf ("krb_save_credentials returns: %d.\n",
    		 krb_save_credentials ("bogus", cr.instance, cr.realm,
                                        cr.session, cr.lifetime, cr.kvno,
                                        &cr.ticket_st, cr.issue_date));
    							
        printf ("number of credentials in the cache: %d.\n",
		krb_get_num_cred());			
    							
        /* krb_delete_cred 	
        * deletes the first credential in the default cache
        * with matching service name, instance and realm.
        */		
        printf (
"Delete the first cred in the cache with matching service name, inst, realm\n"
		);
        printf ("krb_delete_cred returns: %d.\n",
                 krb_delete_cred ("bogus", inst, realm));

        printf("number of credentials in the cache: %d.\n",krb_get_num_cred());
    } /* end credentials caching tests */

    {
	char buf[1500];
	krb5_int32 buflen;
	des_cblock sessionKey;
	Key_schedule schedule;

	k_errno = krb_get_ticket_for_service ("rcmd.toad", buf, &buflen,
		0, sessionKey, schedule, "", 0);
	printf ("krb_get_ticket errno = %d (%s)\n", k_errno,
                 krb_get_err_text_entry (k_errno));
    }
#endif
    return(0);
}
#endif /* KINIT */
#ifdef KDESTROY
int
#ifdef CK_ANSIC
ck_krb4_destroy(struct krb_op_data * op)
#else
ck_krb4_destroy(op) struct krb_op_data * op;
#endif
{
    int k_errno=0;

    k_errno = dest_tkt();

    if (k_errno == 0)
        printf("Tickets destroyed.\n");
    else if (k_errno == RET_TKFIL)
        printf("No tickets to destroy.\n");
    else {
        printf("Tickets MAY NOT be destroyed.\n");
        return(-1);
    }
    return(0);
}
#endif /* KDESTROY */
#ifdef KLIST
_PROTOTYP(static int display_tktfile,(char *, int, int, int));

int
#ifdef CK_ANSIC
ck_krb4_list_creds(struct krb_op_data * op)
#else
ck_krb4_list_creds(op) struct krb_op_data * op;
#endif
{
    int     long_form = 1;
    int     tgt_test = 0;
    int     do_srvtab = 0;
    int	    show_kvnos = 0;
    char   *tkt_file = NULL;

    if ( op->cache )
        tkt_file = op->cache;

    if (do_srvtab)
	return(display_srvtab(tkt_file));
    else
	return(display_tktfile(tkt_file, tgt_test, long_form, show_kvnos));
}


static int
#ifdef CK_ANSIC
display_tktfile(char *file, int tgt_test, int long_form, int show_kvnos)
#else
display_tktfile(file,tgt_test,long_form,show_kvnos)
    char *file; int tgt_test; int long_form; int show_kvnos;
#endif
{
    char    pname[ANAME_SZ];
    char    pinst[INST_SZ];
    char    prealm[REALM_SZ];
    char    buf1[20], buf2[20];
    int     k_errno;
    CREDENTIALS c;
    int     header = 1;

    file = tkt_string();

    if (long_form) {
#ifdef COMMENT
        printf("Kerberos IV\n");
        printf("-----------\n");
#endif /* COMMENT */
	printf("Ticket cache:      %s\n", file);
    }

    /*
     * Since krb_get_tf_realm will return a ticket_file error,
     * we will call tf_init and tf_close first to filter out
     * things like no ticket file.  Otherwise, the error that
     * the user would see would be
     * klist: can't find realm of ticket file: No ticket file (tf_util)
     * instead of
     * klist: No ticket file (tf_util)
     */

    /* Open ticket file */
    if (k_errno = tf_init(file, R_TKT_FIL)) {
	if (!tgt_test)
            printf("%s\n", krb_get_err_text_entry (k_errno));
            return(-1);
    }


    /* Close ticket file */
    (void) tf_close();

    /*
     * We must find the realm of the ticket file here before calling
     * tf_init because since the realm of the ticket file is not
     * really stored in the principal section of the file, the
     * routine we use must itself call tf_init and tf_close.
     */
    if ((k_errno = krb_get_tf_realm(file, prealm)) != KSUCCESS) {
	if (!tgt_test)
	    printf("can't find realm of ticket file: %s\n",
		    krb_get_err_text_entry (k_errno));
	return(-1);
    }

    /* Open ticket file */
    if (k_errno = tf_init(file, R_TKT_FIL)) {
	if (!tgt_test)
            printf("%s\n", krb_get_err_text_entry (k_errno));
	return(-1);
    }
    /* Get principal name and instance */
    if ((k_errno = tf_get_pname(pname)) ||
         (k_errno = tf_get_pinst(pinst))) {
        if (!tgt_test)
            printf("%s\n", krb_get_err_text_entry (k_errno));
        return(-1);
    }

    /*
     * You may think that this is the obvious place to get the
     * realm of the ticket file, but it can't be done here as the
     * routine to do this must open the ticket file.  This is why
     * it was done before tf_init.
     */

    if (!tgt_test && long_form)
	printf("Default principal: %s%s%s%s%s\n\n", pname,
	       (pinst[0] ? "." : ""), pinst,
	       (prealm[0] ? "@" : ""), prealm);
    while ((k_errno = tf_get_cred(&c)) == KSUCCESS) {
	if (!tgt_test && long_form && header) {
	    printf("%-15s  %-15s  %s\n",
		   "Valid starting", "Expires", "Service principal");
	    header = 0;
	}
	if (tgt_test) {
	    c.issue_date += ((unsigned char) c.lifetime) * 5 * 60;
	    if (!strcmp(c.service, "krbtgt") &&
		!strcmp(c.instance, prealm)) {
		if (time(0) < c.issue_date)
		    return(0);		/* tgt hasn't expired */
		else
		    return(-1);		/* has expired */
	    }
	    continue;			/* not a tgt */
	}
	if (long_form) {
            timestamp_width = 17;
            printtime(c.issue_date);
	    c.issue_date += ((unsigned char) c.lifetime) * 5 * 60;
            if ( time(0) < c.issue_date )
                printtime(c.issue_date);
            else
                printf("*** expired ***  ");
	}
	if (show_kvnos)
	  printf("%s%s%s%s%s (%d)\n",
		 c.service, (c.instance[0] ? "." : ""), c.instance,
		 (c.realm[0] ? "@" : ""), c.realm, c.kvno);
	else
	  printf("%s%s%s%s%s\n",
		 c.service, (c.instance[0] ? "." : ""), c.instance,
		 (c.realm[0] ? "@" : ""), c.realm);
    }
    if (tgt_test)
	return(-1);			/* no tgt found */
    if (header && long_form && k_errno == EOF) {
	printf("No tickets in file.\n");
    }
    return(0);
}

#ifdef COMMENT
/* Just so we remember what the command line interface looked like */
usage()
{
    printf("Usage: [ -s | -t ] [ -file filename ] [ -srvtab ] [ -version ]\n");
    return(-1);
}
#endif /* COMMENT */

int
#ifdef CK_ANSIC
display_srvtab(char *file)
#else
display_srvtab(file) char *file;
#endif
{
    int stab;
    char serv[SNAME_SZ];
    char inst[INST_SZ];
    char rlm[REALM_SZ];
    unsigned char key[8];
    unsigned char vno;
    int count;

    printf("Server key file:   %s\n", file);
	
    if ((stab = open(file, O_RDONLY, 0400)) < 0) {
	perror(file);
	return(-1);
    }
    printf("%-15s %-15s %-10s %s\n","Service","Instance","Realm",
	   "Key Version");
    printf("------------------------------------------------------\n");

    /* argh. getst doesn't return error codes, it silently fails */
    while (((count = ok_getst(stab, serv, SNAME_SZ)) > 0)
	   && ((count = ok_getst(stab, inst, INST_SZ)) > 0)
	   && ((count = ok_getst(stab, rlm, REALM_SZ)) > 0)) {
	if (((count = read(stab,(char *) &vno,1)) != 1) ||
	     ((count = read(stab,(char *) key,8)) != 8)) {
	    if (count < 0)
		perror("reading from key file");
	    else
		printf("key file truncated\n");
	    return(-1);
	}
	printf("%-15s %-15s %-15s %d\n",serv,inst,rlm,vno);
    }
    if (count < 0)
	perror(file);
    (void) close(stab);
}

/* adapted from getst() in librkb */
/*
 * ok_getst() takes a file descriptor, a string and a count.  It reads
 * from the file until either it has read "count" characters, or until
 * it reads a null byte.  When finished, what has been read exists in
 * the given string "s".  If "count" characters were actually read, the
 * last is changed to a null, so the returned string is always null-
 * terminated.  ok_getst() returns the number of characters read, including
 * the null terminator.
 *
 * If there is a read error, it returns -1 (like the read(2) system call)
 */

static int
#ifdef CK_ANSIC
ok_getst(int fd, register char *s, int n)
#else
ok_getst(fd, s, n) int fd; register char *s; int n;
#endif
{
    register int count = n;
    int err;
    while ((err = read(fd, s, 1)) > 0 && --count)
        if (*s++ == '\0')
            return (n - count);
    if (err < 0)
	return(-1);
    *s = '\0';
    return (n - count);
}
#endif /* KLIST */
#else /* KRB4 */
#ifdef CK_KERBEROS
int
#ifdef CK_ANSIC
ck_krb4_initTGT(struct krb_op_data * op, struct krb4_init_data * init)
#else
ck_krb4_initTGT(op,init)
    struct krb_op_data * op, struct krb4_init_data * init
#endif
{
    return(-1);
}

#ifdef CK_ANSIC
ck_krb4_destroy(struct krb_op_data * op)
#else
ck_krb4_destroy(op) struct krb_op_data * op;
#endif
{
    return(-1);
}
int
#ifdef CK_ANSIC
ck_krb4_list_creds(struct krb_op_data * op)
#else
ck_krb4_list_creds(op) struct krb_op_data * op;
#endif
{
    return(-1);
}
#else /* CK_KERBEROS */
int ck_krb4_initTGT(void * a, void *b)
{
    return(-1);
}
int ck_krb4_destroy(void *a)
{
    return(-1);
}
int ck_krb4_list_creds(void *a)
{
    return(-1);
}
#endif /* CK_KERBEROS */
#endif /* KRB4 */

/* The following functions are used to implement the Kermit Script Language */
/* functions                                                                */

struct tkt_list_item {
    char * name;
    struct tkt_list_item * next;
};

static struct tkt_list_item * k4_tkt_list = NULL;

int
#ifdef CK_ANSIC
ck_krb4_get_tkts(VOID)
#else
ck_krb4_get_tkts()
#endif
{
#ifdef KRB4
    char   *file=NULL;
    char    pname[ANAME_SZ];
    char    pinst[INST_SZ];
    char    prealm[REALM_SZ];
    char    buf1[20], buf2[20];
    int     k_errno;
    CREDENTIALS c;
    int     tkt_count=0;
    struct  tkt_list_item ** list = &k4_tkt_list;

    while ( k4_tkt_list ) {
        struct tkt_list_item * next;
        next = k4_tkt_list->next;
        free(k4_tkt_list->name);
        free(k4_tkt_list);
        k4_tkt_list = next;
    }

    file = tkt_string();

    /*
     * Since krb_get_tf_realm will return a ticket_file error,
     * we will call tf_init and tf_close first to filter out
     * things like no ticket file.  Otherwise, the error that
     * the user would see would be
     * klist: can't find realm of ticket file: No ticket file (tf_util)
     * instead of
     * klist: No ticket file (tf_util)
     */

    /* Open ticket file */
    if (k_errno = tf_init(file, R_TKT_FIL)) {
        return(-1);
    }

    /* Close ticket file */
    (void) tf_close();

    /*
     * We must find the realm of the ticket file here before calling
     * tf_init because since the realm of the ticket file is not
     * really stored in the principal section of the file, the
     * routine we use must itself call tf_init and tf_close.
     */
    if ((k_errno = krb_get_tf_realm(file, prealm)) != KSUCCESS) {
	return(-1);
    }

    /* Open ticket file */
    if (k_errno = tf_init(file, R_TKT_FIL)) {
	return(-1);
    }
    /* Get principal name and instance */
    if ((k_errno = tf_get_pname(pname)) ||
         (k_errno = tf_get_pinst(pinst))) {
        return(-1);
    }

    /*
     * You may think that this is the obvious place to get the
     * realm of the ticket file, but it can't be done here as the
     * routine to do this must open the ticket file.  This is why
     * it was done before tf_init.
     */

    while ((k_errno = tf_get_cred(&c)) == KSUCCESS) {
        char tkt_buf[256];
        sprintf(tkt_buf,"%s%s%s%s%s",
		 c.service, (c.instance[0] ? "." : ""), c.instance,
		 (c.realm[0] ? "@" : ""), c.realm);
        *list = (struct tkt_list_item *) malloc(sizeof(struct tkt_list_item));
        (*list)->name = strdup(tkt_buf);
        (*list)->next = NULL;
        list = &((*list)->next);
        tkt_count++;
    }
    return(tkt_count);
#else /* KRB4 */
    return(0);
#endif /* KRB4 */
}

char *
#ifdef CK_ANSIC
ck_krb4_get_next_tkt(VOID)
#else
ck_krb4_get_next_tkt()
#endif
{
#ifdef KRB4
    static char * s=NULL;
    struct tkt_list_item * next=NULL;

    if ( s ) {
        free(s);
        s = NULL;
    }

    if ( k4_tkt_list == NULL )
        return(NULL);

    next = k4_tkt_list->next;
    s = k4_tkt_list->name;
    free(k4_tkt_list);
    k4_tkt_list = next;
    return(s);                          /* s must be freed by caller */
#else /* KRB4 */
    return(NULL);
#endif /* KRB4 */
}


#ifdef OS2
/* The Leash implementation of Kerberos 4 used by Kermit 95 */
/* has an extended Credentials structure that includes the  */
/* ip address of the ticket in readable form.               */
#ifdef KRB4
#ifndef ADDR_SZ
#define ADDR_SZ 40      /* From Leash krb.h */
#endif /* ADDR_SZ */

struct leash_credentials {
    char    service[ANAME_SZ];  /* Service name */
    char    instance[INST_SZ];  /* Instance */
    char    realm[REALM_SZ];    /* Auth domain */
    C_Block session;            /* Session key */
    int     lifetime;           /* Lifetime */
    int     kvno;               /* Key version number */
    KTEXT_ST ticket_st;         /* The ticket itself */
    long    issue_date;         /* The issue time */
    char    pname[ANAME_SZ];    /* Principal's name */
    char    pinst[INST_SZ];     /* Principal's instance */
    char    address[ADDR_SZ];      /* IP Address in ticket */
};

typedef struct leash_credentials LEASH_CREDENTIALS;
#endif /* KRB4 */
#endif /* OS2 */

int
#ifdef CK_ANSIC
ck_krb4_tkt_isvalid(char * tktname)
#else
ck_krb4_tkt_isvalid(tktname) char * tktname;
#endif
{
#ifdef KRB4
    char   *file=NULL;
    char    pname[ANAME_SZ];
    char    pinst[INST_SZ];
    char    prealm[REALM_SZ];
    char    buf1[20], buf2[20];
    int     k_errno;
#ifdef OS2
    LEASH_CREDENTIALS c;
#else /* OS2 */
    CREDENTIALS c;
#endif /* OS2 */

    file = tkt_string();

    /*
     * Since krb_get_tf_realm will return a ticket_file error,
     * we will call tf_init and tf_close first to filter out
     * things like no ticket file.  Otherwise, the error that
     * the user would see would be
     * klist: can't find realm of ticket file: No ticket file (tf_util)
     * instead of
     * klist: No ticket file (tf_util)
     */

    /* Open ticket file */
    if (k_errno = tf_init(file, R_TKT_FIL)) {
        return(-1);
    }

    /* Close ticket file */
    (void) tf_close();

    /*
     * We must find the realm of the ticket file here before calling
     * tf_init because since the realm of the ticket file is not
     * really stored in the principal section of the file, the
     * routine we use must itself call tf_init and tf_close.
     */
    if ((k_errno = krb_get_tf_realm(file, prealm)) != KSUCCESS) {
	return(-1);
    }

    /* Open ticket file */
    if (k_errno = tf_init(file, R_TKT_FIL)) {
	return(-1);
    }
    /* Get principal name and instance */
    if ((k_errno = tf_get_pname(pname)) ||
         (k_errno = tf_get_pinst(pinst))) {
        return(-1);
    }

    /*
     * You may think that this is the obvious place to get the
     * realm of the ticket file, but it can't be done here as the
     * routine to do this must open the ticket file.  This is why
     * it was done before tf_init.
     */

    while ((k_errno = tf_get_cred((CREDENTIALS *)&c)) == KSUCCESS) {
        char tkt_buf[256];
        sprintf(tkt_buf,"%s%s%s%s%s",
		 c.service, (c.instance[0] ? "." : ""), c.instance,
		 (c.realm[0] ? "@" : ""), c.realm);
        if ( !strcmp(tktname,tkt_buf) ) {
            /* we found the ticket we are looking for */
	    c.issue_date += ((unsigned char) c.lifetime) * 5 * 60;
            if ( time(0) < c.issue_date ) {
#ifdef OS2
#ifdef CHECKADDRS
                extern char myipaddr[20];       /* From ckcnet.c */
                if ( !myipaddr[0] ) {
                    if ( getlocalipaddr() < 0 )
                        return(1);              /* Unable to check IP Addr */
                }

                if ( strcmp(myipaddr,c.address) )
                    return(0);                  /* They're different */
                else
                    return(1);                  /* They're the same */
#else /* CHECKADDRS */
                return(1);      /* valid but no ip address check */
#endif /* CHECKADDRS */
#else /* OS2 */
                return(1);      /* Valid but no ip address check */
#endif /* OS2 */
            }
            else
                return(0);      /* expired */
        }
    }
    return(0);                  /* could not find the desired ticket */
#else /* KRB4 */
    return(-1);
#endif /* KRB4 */
}

int
#ifdef CK_ANSIC
ck_krb4_is_tgt_valid(VOID)
#else
ck_krb4_is_tgt_valid()
#endif
{
#ifdef KRB4
    char tgt[256];
    extern char * krb4_d_realm;
    char * s;
    int rc = 0;

    s = krb4_d_realm ? krb4_d_realm : ck_krb4_getrealm();
    sprintf(tgt,"krbtgt.%s@%s",s,s);
    rc = ck_krb4_tkt_isvalid(tgt);
    debug(F111,"ck_krb4_is_tgt_valid",tgt,rc);
    return(rc > 0);
#else /* KRB4 */
    return(0);
#endif /* KRB4 */
}

int
#ifdef CK_ANSIC
ck_krb4_tkt_time(char * tktname)
#else
ck_krb4_tkt_time(tktname) char * tktname;
#endif
{
#ifdef KRB4
    char   *file=NULL;
    char    pname[ANAME_SZ];
    char    pinst[INST_SZ];
    char    prealm[REALM_SZ];
    char    buf1[20], buf2[20];
    int     k_errno;
    CREDENTIALS c;

    file = tkt_string();

    /*
     * Since krb_get_tf_realm will return a ticket_file error,
     * we will call tf_init and tf_close first to filter out
     * things like no ticket file.  Otherwise, the error that
     * the user would see would be
     * klist: can't find realm of ticket file: No ticket file (tf_util)
     * instead of
     * klist: No ticket file (tf_util)
     */

    /* Open ticket file */
    if (k_errno = tf_init(file, R_TKT_FIL)) {
        return(-1);
    }

    /* Close ticket file */
    (void) tf_close();

    /*
     * We must find the realm of the ticket file here before calling
     * tf_init because since the realm of the ticket file is not
     * really stored in the principal section of the file, the
     * routine we use must itself call tf_init and tf_close.
     */
    if ((k_errno = krb_get_tf_realm(file, prealm)) != KSUCCESS) {
	return(-1);
    }

    /* Open ticket file */
    if (k_errno = tf_init(file, R_TKT_FIL)) {
	return(-1);
    }
    /* Get principal name and instance */
    if ((k_errno = tf_get_pname(pname)) ||
         (k_errno = tf_get_pinst(pinst))) {
        return(-1);
    }

    /*
     * You may think that this is the obvious place to get the
     * realm of the ticket file, but it can't be done here as the
     * routine to do this must open the ticket file.  This is why
     * it was done before tf_init.
     */

    while ((k_errno = tf_get_cred(&c)) == KSUCCESS) {
        char tkt_buf[256];
        sprintf(tkt_buf,"%s%s%s%s%s",
		 c.service, (c.instance[0] ? "." : ""), c.instance,
		 (c.realm[0] ? "@" : ""), c.realm);
        if ( !strcmp(tktname,tkt_buf) ) {
            /* we found the ticket we are looking for */
            int n = (c.issue_date + (((unsigned char) c.lifetime) * 5 * 60))
                - time(0);
            return(n <= 0 ? 0 : n);
        }
    }
    return(0);                  /* could not find the desired ticket */
#else /* KRB4 */
    return(-1);
#endif /* KRB4 */
}

char *
#ifdef CK_ANSIC
ck_krb4_getrealm(void)
#else
ck_krb4_getrealm()
#endif
{
#ifdef KRB4
    static char realm[256]="";
    realm[0]='\0';

    krb_get_lrealm(realm,1);

    return(realm);
#else /* KRB4 */
    return("");
#endif /* KRB4 */
}

static struct tkt_list_item * k5_tkt_list = NULL;

int
#ifdef CK_ANSIC
ck_krb5_get_tkts(char * cc_name)
#else
ck_krb5_get_tkts(cc_name) char * cc_name;
#endif
{
#ifdef KRB5
    krb5_context kcontext;
    krb5_error_code retval;
    krb5_ccache cache = NULL;
    krb5_cc_cursor cur;
    krb5_creds creds;
    krb5_principal princ;
    krb5_flags flags;
    krb5_error_code code;
    int	exit_status = 0;

    int     tkt_count=0;
    struct  tkt_list_item ** list = &k5_tkt_list;

    while ( k5_tkt_list ) {
        struct tkt_list_item * next;
        next = k5_tkt_list->next;
        free(k5_tkt_list->name);
        free(k5_tkt_list);
        k5_tkt_list = next;
    }

    retval = krb5_init_context(&kcontext);
    if (retval) {
        debug(F101,"ck_krb5_list_creds while initializing krb5","",retval);
        return(-1);
    }

    if (cc_name == NULL) {
	if ((code = krb5_cc_default(kcontext, &cache))) {
            tkt_count = -1;
            goto exit_k5_get_tkt;
        }
    } else {
        char cc_tmp[260];
        sprintf(cc_tmp,"FILE:%s",cc_name);
	if ((code = krb5_cc_resolve(kcontext, cc_tmp, &cache))) {
            tkt_count = -1;
            goto exit_k5_get_tkt;
	}
    }

    flags = 0;				/* turns off OPENCLOSE mode */
    if ((code = krb5_cc_set_flags(kcontext, cache, flags))) {
	if (code == ENOENT) {
            debug(F111,"ck_krb5_list_creds (ticket cache)",
                   krb5_cc_get_name(kcontext, cache),code);
	} else {
            debug(F111,
		 "ck_krb5_list_creds while setting cache flags (ticket cache)",
                  krb5_cc_get_name(kcontext, cache),code);
	}
        tkt_count = -1;
        goto exit_k5_get_tkt;
    }
    if ((code = krb5_cc_get_principal(kcontext, cache, &princ))) {
        debug(F101,"ck_krb5_list_creds while retrieving principal name",
               "",code);
        tkt_count = -1;
        goto exit_k5_get_tkt;
    }
    if ((code = krb5_unparse_name(kcontext, princ, &defname))) {
        debug(F101,"ck_krb5_list_creds while unparsing principal name",
               "",code);
        tkt_count = -1;
        goto exit_k5_get_tkt;
    }

    if ((code = krb5_cc_start_seq_get(kcontext, cache, &cur))) {
        debug(F101,"ck_krb5_list_creds while starting to retrieve tickets",
               "",code);
        tkt_count = -1;
        goto exit_k5_get_tkt;
    }

    while (!(code = krb5_cc_next_cred(kcontext, cache, &cur, &creds))) {
        char *sname=NULL;

        retval = krb5_unparse_name(kcontext, creds.server, &sname);
        if (retval) {
            debug(F101,
		  "ck_krb5_list_creds while unparsing server name","",retval);
            tkt_count = -1;
            goto exit_k5_get_tkt;
        }

        *list = (struct tkt_list_item *) malloc(sizeof(struct tkt_list_item));
        (*list)->name = sname;
        (*list)->next = NULL;
        list = &((*list)->next);

	krb5_free_cred_contents(kcontext, &creds);
        tkt_count++;
    }

    if (code == KRB5_CC_END) {
	if ((code = krb5_cc_end_seq_get(kcontext, cache, &cur))) {
            debug(F101,"ck_krb5_list_creds while finishing ticket retrieval",
                   "",code);
            tkt_count = -1;
            goto exit_k5_get_tkt;
	}
	flags = KRB5_TC_OPENCLOSE;	/* turns on OPENCLOSE mode */
	if ((code = krb5_cc_set_flags(kcontext, cache, flags))) {
            debug(F101,"ck_krb5_list_creds while closing ccache",
                   "",code);
            tkt_count = -1;
            goto exit_k5_get_tkt;
	}
    } else {
        debug(F101,"ck_krb5_list_creds while retrieving a ticket","",code);
        tkt_count = -1;
        goto exit_k5_get_tkt;
    }	

  exit_k5_get_tkt:
    krb5_free_context(kcontext);
    return(tkt_count);
#else /* KRB5 */
    return(0);
#endif /* KRB5 */
}

char *
#ifdef CK_ANSIC
ck_krb5_get_next_tkt(VOID)
#else
ck_krb5_get_next_tkt()
#endif
{
#ifdef KRB5
    static char * s=NULL;
    struct tkt_list_item * next=NULL;

    if ( s ) {
        free(s);
        s = NULL;
    }

    if ( k5_tkt_list == NULL )
        return(NULL);

    next = k5_tkt_list->next;
    s = k5_tkt_list->name;
    free(k5_tkt_list);
    k5_tkt_list = next;
    return(s);                          /* s must be freed by caller */
#else /* KRB5 */
    return(NULL);
#endif /* KRB5 */
}

int
#ifdef CK_ANSIC
ck_krb5_tkt_isvalid(char * cc_name, char * tktname)
#else
ck_krb5_tkt_isvalid(cc_name,tktname) char * cc_name; char * tktname;
#endif
{
#ifdef KRB5
    krb5_context kcontext;
    krb5_error_code retval;
    krb5_ccache cache = NULL;
    krb5_cc_cursor cur;
    krb5_creds creds;
    krb5_principal princ;
    krb5_flags flags;
    krb5_error_code code;
#ifdef CHECKADDRS
    krb5_address **	myAddrs;
    krb5_address **	p;
    BOOL	        Addrfound = FALSE;
#endif /*CHECKADDRS*/

    retval = krb5_init_context(&kcontext);
    if (retval) {
        debug(F101,"ck_krb5_list_creds while initializing krb5","",retval);
        return(-1);
    }

    if (cc_name == NULL) {
	if ((code = krb5_cc_default(kcontext, &cache))) {
            retval = -1;
            goto exit_k5_get_tkt;
        }
    } else {
        char cc_tmp[260];
        sprintf(cc_tmp,"FILE:%s",cc_name);
	if ((code = krb5_cc_resolve(kcontext, cc_tmp, &cache))) {
            retval = -1;
            goto exit_k5_get_tkt;
	}
    }

    flags = 0;				/* turns off OPENCLOSE mode */
    if ((code = krb5_cc_set_flags(kcontext, cache, flags))) {
	if (code == ENOENT) {
            debug(F111,"ck_krb5_list_creds (ticket cache)",
                   krb5_cc_get_name(kcontext, cache),code);
	} else {
            debug(F111,
		 "ck_krb5_list_creds while setting cache flags (ticket cache)",
                  krb5_cc_get_name(kcontext, cache),code);
	}
        retval = -1;
        goto exit_k5_get_tkt;
    }
    if ((code = krb5_cc_get_principal(kcontext, cache, &princ))) {
        debug(F101,"ck_krb5_list_creds while retrieving principal name",
               "",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }
    if ((code = krb5_unparse_name(kcontext, princ, &defname))) {
        debug(F101,"ck_krb5_list_creds while unparsing principal name",
               "",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }

    if ((code = krb5_cc_start_seq_get(kcontext, cache, &cur))) {
        debug(F101,"ck_krb5_list_creds while starting to retrieve tickets",
               "",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }

    if ((code = krb5_timeofday(kcontext, &now))) {
        if (!status_only)
            debug(F101,"ck_krb5_list_creds while getting time of day.",
                   "",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }

    while (!(code = krb5_cc_next_cred(kcontext, cache, &cur, &creds))) {
        char *sname=NULL;

        retval = krb5_unparse_name(kcontext, creds.server, &sname);
        if (retval) {
            debug(F101,
		  "ck_krb5_list_creds while unparsing server name","",retval);
            retval = -1;
            krb5_free_cred_contents(kcontext, &creds);
            goto exit_k5_get_tkt;
        }

        if ( !strcmp(sname,tktname) ) {
            /* we found the ticket we are looking for */
            retval = (creds.times.starttime &&
                       now > creds.times.starttime &&
                       now < creds.times.endtime &&
                       !(creds.ticket_flags & TKT_FLG_INVALID));

#ifdef CHECKADDRS
            if ( retval ) {
                /* if we think it is valid, then lets check the IP Addresses */
                /* to make sure it is valid for our current connection.      */
                /* Also make sure it's for the correct IP address */
                retval = krb5_os_localaddr(k5_context, &myAddrs);
                if( retval )
                {
                    com_err(NULL, retval, "retrieving my IP address");
                    krb5_free_cred_contents(kcontext, &creds);
                    code = KRB5_CC_END;
                    retval = -1;
                    break;
                }

#ifdef dbg
                for( p=myAddrs ; *p ; p++ )
                    com_err("Kclnt32", 0, "Address: %x", (int)(*p));
#endif

             /* See if any of our addresses match any in cached credentials */

                for( Addrfound=FALSE, p=myAddrs ;
                     (Addrfound==FALSE) && (*p) ; p++ )
                {

                    if( krb5_address_search(k5_context, *p, creds.addresses) )
                    {
#ifdef dbg
			com_err("Kclnt32", 0, "Found one of our addrs in creds!");
#endif
			Addrfound = TRUE;
                    }
                }
                krb5_free_addresses(k5_context, myAddrs);

                if( Addrfound )
                {
#ifdef dbg
                    com_err("Kclnt32", 0, "found my address OK");
#endif
                    krb5_free_cred_contents(kcontext, &creds);
                    code = KRB5_CC_END;
                    retval = 1;
                    break;
                }
                else
                {
#ifdef dbg
                    com_err("Kclnt32", 0, "did NOT find my address in ccache");
#endif
                    krb5_free_cred_contents(kcontext, &creds);
                    code = KRB5_CC_END;
                    retval = 0;
                    break;
                }
            }
#endif /* CHECKADDRS */

            krb5_free_cred_contents(kcontext, &creds);
            code = KRB5_CC_END;
            break;
        }
	krb5_free_cred_contents(kcontext, &creds);
    }

    if (code == KRB5_CC_END) {
	if ((code = krb5_cc_end_seq_get(kcontext, cache, &cur))) {
            debug(F101,"ck_krb5_list_creds while finishing ticket retrieval",
                   "",code);
            retval = -1;
            goto exit_k5_get_tkt;
	}
	flags = KRB5_TC_OPENCLOSE;	/* turns on OPENCLOSE mode */
	if ((code = krb5_cc_set_flags(kcontext, cache, flags))) {
            debug(F101,"ck_krb5_list_creds while closing ccache",
                   "",code);
            retval = -1;
            goto exit_k5_get_tkt;
	}
    } else {
        debug(F101,"ck_krb5_list_creds while retrieving a ticket","",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }	

  exit_k5_get_tkt:
    krb5_free_context(kcontext);
    return(retval);
#else /* KRB5 */
    return(-1);
#endif /* KRB5 */
}

int
#ifdef CK_ANSIC
ck_krb5_is_tgt_valid(VOID)
#else
ck_krb5_is_tgt_valid()
#endif
{
#ifdef KRB5
    char tgt[256];
    extern char * krb5_d_realm;
    extern char * krb5_d_cc;
    char * s;
    int rc = 0;

    s = krb5_d_realm ? krb5_d_realm : ck_krb5_getrealm(krb5_d_cc);
    sprintf(tgt,"krbtgt/%s@%s",s,s);
    rc = ck_krb5_tkt_isvalid(krb5_d_cc,tgt);
    debug(F111,"ck_krb5_is_tgt_valid",tgt,rc);
    return(rc>0);
#else /* KRB5 */
    return(0);
#endif /* KRB5 */
}

int
#ifdef CK_ANSIC
ck_krb5_tkt_time(char * cc_name, char * tktname)
#else
ck_krb5_tkt_time(cc_name, tktname) char * cc_name; char * tktname;
#endif
{
#ifdef KRB5
    krb5_context kcontext;
    krb5_error_code retval;
    krb5_ccache cache = NULL;
    krb5_cc_cursor cur;
    krb5_creds creds;
    krb5_principal princ;
    krb5_flags flags;
    krb5_error_code code;

    retval = krb5_init_context(&kcontext);
    if (retval) {
        debug(F101,"ck_krb5_list_creds while initializing krb5","",retval);
        return(-1);
    }

    if (cc_name == NULL) {
	if ((code = krb5_cc_default(kcontext, &cache))) {
            retval = -1;
            goto exit_k5_get_tkt;
        }
    } else {
        char cc_tmp[260];
        sprintf(cc_tmp,"FILE:%s",cc_name);
	if ((code = krb5_cc_resolve(kcontext, cc_tmp, &cache))) {
            retval = -1;
            goto exit_k5_get_tkt;
	}
    }

    flags = 0;				/* turns off OPENCLOSE mode */
    if ((code = krb5_cc_set_flags(kcontext, cache, flags))) {
	if (code == ENOENT) {
            debug(F111,"ck_krb5_list_creds (ticket cache)",
                   krb5_cc_get_name(kcontext, cache),code);
	} else {
            debug(F111,
		 "ck_krb5_list_creds while setting cache flags (ticket cache)",
                  krb5_cc_get_name(kcontext, cache),code);
	}
        retval = -1;
        goto exit_k5_get_tkt;
    }
    if ((code = krb5_cc_get_principal(kcontext, cache, &princ))) {
        debug(F101,"ck_krb5_list_creds while retrieving principal name",
               "",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }
    if ((code = krb5_unparse_name(kcontext, princ, &defname))) {
        debug(F101,"ck_krb5_list_creds while unparsing principal name",
               "",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }

    if ((code = krb5_cc_start_seq_get(kcontext, cache, &cur))) {
        debug(F101,"ck_krb5_list_creds while starting to retrieve tickets",
               "",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }

    if ((code = krb5_timeofday(kcontext, &now))) {
        if (!status_only)
            debug(F101,"ck_krb5_list_creds while getting time of day.",
                   "",code);
        krb5_free_context(kcontext);
        return(-1);
    }

    while (!(code = krb5_cc_next_cred(kcontext, cache, &cur, &creds))) {
        char *sname=NULL;

        retval = krb5_unparse_name(kcontext, creds.server, &sname);
        if (retval) {
            debug(F101,
		  "ck_krb5_list_creds while unparsing server name","",retval);
            retval = -1;
            krb5_free_cred_contents(kcontext, &creds);
            goto exit_k5_get_tkt;
        }

        if ( !strcmp(sname,tktname) ) {
            /* we found the ticket we are looking for */
            int valid = (creds.times.starttime &&
                       now > creds.times.starttime &&
                       now < creds.times.endtime &&
                       !(creds.ticket_flags & TKT_FLG_INVALID));
            if ( valid ) {
                retval = creds.times.endtime - now;
            }
            else
                retval = 0;
            krb5_free_cred_contents(kcontext, &creds);
            code = KRB5_CC_END;
            break;
        }
	krb5_free_cred_contents(kcontext, &creds);
    }

    if (code == KRB5_CC_END) {
	if ((code = krb5_cc_end_seq_get(kcontext, cache, &cur))) {
            debug(F101,"ck_krb5_list_creds while finishing ticket retrieval",
                   "",code);
            retval = -1;
            goto exit_k5_get_tkt;
	}
	flags = KRB5_TC_OPENCLOSE;	/* turns on OPENCLOSE mode */
	if ((code = krb5_cc_set_flags(kcontext, cache, flags))) {
            debug(F101,"ck_krb5_list_creds while closing ccache",
                   "",code);
            retval = -1;
            goto exit_k5_get_tkt;
	}
    } else {
        debug(F101,"ck_krb5_list_creds while retrieving a ticket","",code);
        retval = -1;
        goto exit_k5_get_tkt;
    }	

  exit_k5_get_tkt:
    krb5_free_context(kcontext);
    return(retval);
#else /* KRB5 */
    return(-1);
#endif /* KRB5 */
}

char *
#ifdef CK_ANSIC
ck_krb5_get_cc(void)
#else
ck_krb5_get_cc()
#endif
{
#ifdef KRB5
    static char cc_name[256]="";
    krb5_context kcontext;
    krb5_ccache ccache = NULL;
    krb5_error_code code;
    char * p=NULL;

    cc_name[0] = '\0';

    code = krb5_init_context(&kcontext);
    if (code) {
        com_err("ck_krb5_getrealm",code,"while init_context");
        return(cc_name);
    }
    if ((code = krb5_cc_default(kcontext, &ccache))) {
        com_err("ck_krb5_getrealm",code,"while getting default ccache");
        goto exit_k5_get_cc;
    }

    strcpy(cc_name,krb5_cc_get_name(kcontext,ccache));
    for ( p=cc_name; *p ; p++ )
        if ( *p == '\\' ) *p = '/';

  exit_k5_get_cc:
    krb5_free_context(kcontext);
    return(cc_name);
#else /* KRB5 */
    return("");
#endif /* KRB5 */
}

char *
#ifdef CK_ANSIC
ck_krb5_getrealm(char * cachefile)
#else
ck_krb5_getrealm(cachefile) char * cachefile;
#endif
{
#ifdef KRB5
    static char realm[256]="";
    krb5_context kcontext;
    krb5_ccache ccache = NULL;
    krb5_error_code code;
    krb5_principal me;

    realm[0] = '\0';

    code = krb5_init_context(&kcontext);
    if (code) {
        debug(F111,"ck_krb5_getrealm","while init_context",code);
        return(realm);
    }

    if ( cachefile ) {
        char cc_tmp[260];
        sprintf(cc_tmp,"FILE:%s",cachefile);
        if (ccache == NULL) {
            code = krb5_cc_resolve(kcontext, cc_tmp, &ccache);
            if (code != 0) {
                com_err("ck_krb5_getrealm resolving ccache",code,
                          cachefile);
            }
        }
    }

    if (ccache == NULL) {
        if ((code = krb5_cc_default(kcontext, &ccache))) {
            com_err("ck_krb5_getrealm",code,"while getting default ccache");
            goto exit_k5_getrealm;
        }
    }

    if ((code = krb5_parse_name (kcontext, "foo", &me))) {
	 com_err("ck_krb5_getrealm",code,"when parsing name",
                "foo");
        goto exit_k5_getrealm;
    }
    memcpy(realm,krb5_princ_realm(kcontext, me)->data,
            krb5_princ_realm(kcontext, me)->length);
    realm[krb5_princ_realm(kcontext, me)->length]='\0';

  exit_k5_getrealm:
    krb5_free_context(kcontext);
    return(realm);
#else /* KRB5 */
    return("");
#endif /* KRB5 */
}

#ifndef CRYPT_DLL
int
ck_get_crypt_table(struct keytab ** pTable, int * pN)
{
#ifdef ENCRYPTION
    return(get_crypt_table(pTable, pN));
#else /* ENCRYPTION */
    int i=0;
#ifndef OS2 
    char * tmpstring = NULL;
#endif /* OS2 */

    if ( *pTable )
    {
        for ( i=0 ; i < *pN ; i++ )
            free( (*pTable)[i].kwd ) ;
        free ( *pTable )  ;
    }
    *pTable = NULL;
    *pN = 0;

    *pTable = malloc( sizeof(struct keytab) * 2 ) ;
    if ( !(*pTable) )
        return(0);

#ifdef OS2
        (*pTable)[0].kwd =strdup("automatic");
#else /* OS2 */
	makestr(&tmpstring,"automatic");
        (*pTable)[0].kwd = tmpstring;
#endif /* OS2 */
        (*pTable)[0].kwval = ENCTYPE_ANY;
        (*pTable)[0].flgs = 0;
#ifdef OS2
        (*pTable)[1].kwd =strdup("none");
#else /* OS2 */
	makestr(&tmpstring,"none");
        (*pTable)[1].kwd = tmpstring;
#endif /* OS2 */
        (*pTable)[1].kwval = 999;
        (*pTable)[1].flgs = 0;
        (*pN) = 2;

    return(2);
#endif /* ENCRYPTION */
}

VOID
ck_encrypt_send_support()
{
#ifdef ENCRYPTION
    encrypt_send_support();
#endif /* ENCRYPTION */
}
#endif /* CRYPT_DLL */

/*-
 * Copyright (c) 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* based on @(#)genget.c	8.1 (Berkeley) 6/4/93 */

#include <ctype.h>

#define	LOWER(x) (isupper(x) ? tolower(x) : (x))
/*
 * The prefix function returns 0 if *s1 is not a prefix
 * of *s2.  If *s1 exactly matches *s2, the negative of
 * the length is returned.  If *s1 is a prefix of *s2,
 * the length of *s1 is returned.
 */
int
#ifdef CK_ANSIC
isprefix(register char *s1, register char *s2)
#else
isprefix(s1,s2) register char *s1; register char *s2;
#endif
{
    char *os1;
    register char c1, c2;

    if (*s1 == '\0')
        return(-1);
    os1 = s1;
    c1 = *s1;
    c2 = *s2;
    while (LOWER(c1) == LOWER(c2)) {
        if (c1 == '\0')
            break;
        c1 = *++s1;
        c2 = *++s2;
    }
    return(*s1 ? 0 : (*s2 ? (s1 - os1) : (os1 - s1)));
}

static char *ambiguous;		/* special return value for command routines */

char **
#ifdef CK_ANSIC
genget(	char	*name,		/* name to match */
	char	**table,	/* name entry in table */
	int	stlen)
#else
genget(name,table,stlen) char *name; char **table; int stlen;
#endif
{
    register char **c, **found;
    register int n;

    if (name == 0)
        return 0;

    found = 0;
    for (c = table; *c != 0; c = (char **)((char *)c + stlen)) {
        if ((n = isprefix(name, *c)) == 0)
            continue;
        if (n < 0)		/* exact match */
            return(c);
        if (found)
            return(&ambiguous);
        found = c;
    }
    return(found);
}

/*
 * Function call version of Ambiguous()
 */
int
Ambiguous(s) char *s;
{
    return((char **)s == &ambiguous);
}

/*
 *
 * Kstream
 * 	
 * Emulates the kstream package in Kerberos 4
 *
 */

int
kstream_destroy()
{
    if (g_kstream != NULL) {
        auth_destroy();                       /* Destroy authorizing */
        free(g_kstream);
        g_kstream=NULL;
    }
    return 0;
}

VOID
#ifdef CK_ANSIC
kstream_set_buffer_mode(int mode)
#else
kstream_set_buffer_mode(mode) int mode;
#endif
{
}


int
#ifdef CK_ANSIC
kstream_create_from_fd(int fd,
		       const struct kstream_crypt_ctl_block *ctl,
		       kstream_ptr data)
#else
kstream_create_from_fd(fd,ctl,data)
    int fd; const struct kstream_crypt_ctl_block *ctl; kstream_ptr data;
#endif
{
    int n;

    g_kstream = malloc(sizeof(struct kstream_int));
    if (g_kstream == NULL)
        return 0;

    g_kstream->fd = fd;

    n = auth_init(g_kstream);                   /* Initialize authorizing */
    if (n) {
        free(g_kstream);
        g_kstream = NULL;
        return 0;
    }

    g_kstream->encrypt = NULL;
    g_kstream->decrypt = NULL;
    g_kstream->encrypt_type = ENCTYPE_ANY;
    g_kstream->decrypt_type = ENCTYPE_ANY;
    return 1;
}
#endif /* CK_AUTHENTICATION */
