#include "xmame.h"
#include "driver.h"
#include "audit.h"

/* Mame frontend interface & commandline */
/* parsing rountines by Maurizio Zanello */

static const struct GameDriver *gamedrv;

#ifdef MESS /* MESS doesn't have a neogeo_bios but to work around a lott
               off #defines later on, we create one here ;) */
static struct GameDriver neogeo_bios;
#else
extern struct GameDriver neogeo_bios;
#endif

/* compare string[8] using standard(?) wildchars ('?' & '*')          */
/* for this to work correctly, the shells internal wildcard expansion */
/* mechanism has to be disabled, use quotes */
int strwildcmp(const char *sp1, const char *sp2)
{
   char s1[9], s2[9];
   int i, l1, l2;
   char *p;

   strncpy(s1, sp1, 8); s1[8] = 0; if (s1[0] == 0) strcpy(s1, "*");

   strncpy(s2, sp2, 8); s2[8] = 0; if (s2[0] == 0) strcpy(s2, "*");

   p = strchr(s1, '*');
   if (p)
   {
      for (i = p - s1; i < 8; i++) s1[i] = '?';
      s1[8] = 0;
   }

   p = strchr(s2, '*');
   if (p)
   {
      for (i = p - s2; i < 8; i++) s2[i] = '?';
      s2[8] = 0;
   }

   l1 = strlen(s1);
   if (l1 < 8)
   {
      for (i = l1 + 1; i < 8; i++) s1[i] = ' ';
      s1[8] = 0;
   }

   l2 = strlen(s2);
   if (l2 < 8)
   {
      for (i = l2 + 1; i < 8; i++) s2[i] = ' ';
      s2[8] = 0;
   }

   for (i = 0; i < 8; i++)
   {
      if (s1[i] == '?' && s2[i] != '?') s1[i] = s2[i];
      if (s2[i] == '?' && s1[i] != '?') s2[i] = s1[i];
   }

   return stricmp(s1, s2);
}


int frontend_list(int list, char *gamename, int showclones)
{
   int i, j;
   const char *header[] = {
       NAME" currently supports:\n",
       "Name:     Description:\n",
       "Year: Manufacturer:                        Description:\n",
       "", /* listclones */
       "name      driver      cpu 1    cpu 2    cpu 3    sound 1   sound 2   sound 3\n"
       "--------  ----------  -----    -----    -----    -------   -------   -------\n",
       "", /* listroms */
       "", /* listsamples */
       "Name:     Samples dir:\n",
       "CRC:     Filename:    Description:\n",
       "", /* crcdup */
       "Name:     Colors:\n"
       "", /* lmr */
       "", /* wrongorientation */
       "", /* verifyroms */
       ""  /* verifysamples */
   };
       
   int matching     = 0;
   int correct      = 0;
   int incorrect    = 0;
   int skipped      = 0;
   int not_found    = 0;
      
   fprintf(stdout_file, header[list-1]);
      
   for (i=0;drivers[i];i++)
   {
         if ((showclones || drivers[i]->clone_of == 0 || drivers[i]->clone_of == &neogeo_bios) &&
            !strwildcmp(gamename, drivers[i]->name))
         {
            matching++;
            
            switch(list)
            {
               case LIST_LIST: /* simple games list */
                  fprintf(stdout_file, "%10s",drivers[i]->name);
                  if (!(matching % 7)) fprintf(stdout_file, "\n");
                  break;
               case LIST_FULL: /* games list with descriptions */
                  fprintf(stdout_file, "%-10s\"%s\"\n",drivers[i]->name,drivers[i]->description);
                  break;
               case LIST_GAMES:
                  fprintf(stdout_file, "%-5s %-36s %s\n",drivers[i]->year,drivers[i]->manufacturer,drivers[i]->description);
                  break;
               case LIST_DETAILS: /* A detailed MAMELIST.TXT type roms lister */
                  /* First, the rom name */
                  fprintf(stdout_file, "%-8s  ",drivers[i]->name);

                  /* source file (skip the leading "src/drivers/" */
                  fprintf(stdout_file, "%-10s  ",&drivers[i]->source_file[12]);

                  /* Then, cpus */
                  for(j=0;j<MAX_CPU-1;j++) /* Increase to table to 4, when a game with 4 cpus will appear */
                  {
                     const struct MachineCPU *x_cpu = drivers[i]->drv->cpu;
                     switch(x_cpu[j].cpu_type & (~CPU_FLAGS_MASK | CPU_AUDIO_CPU))
                     {
                        case 0:         fprintf(stdout_file, "         "); break;
                        case CPU_Z80:   fprintf(stdout_file, "Z80      "); break;
                        case CPU_8085A: fprintf(stdout_file, "I8085    "); break;
                        case CPU_M6502: fprintf(stdout_file, "M6502    "); break;
                        case CPU_I86:   fprintf(stdout_file, "I86      "); break;
                        case CPU_I8039: fprintf(stdout_file, "I8039    "); break;
                        case CPU_M6803: fprintf(stdout_file, "M6808    "); break;
                        case CPU_M6805: fprintf(stdout_file, "M6805    "); break;
                        case CPU_M6809: fprintf(stdout_file, "M6809    "); break;
                        case CPU_M68000:fprintf(stdout_file, "M68000   "); break;
                        case CPU_T11   :fprintf(stdout_file, "T-11     "); break;
                        case CPU_S2650 :fprintf(stdout_file, "S2650    "); break;
#ifdef MESS
                        case CPU_PDP1  :fprintf(stdout_file, "PDP-1    "); break;
#else
                        case CPU_TMS34010 :fprintf(stdout_file, "TMS34010 "); break;
#endif                           
                        case CPU_Z80   |CPU_AUDIO_CPU: fprintf(stdout_file, "[Z80]    "); break; /* Brackets mean that the cpu is only needed for sound. In cpu flags, 0x8000 means it */
                        case CPU_8085A |CPU_AUDIO_CPU: fprintf(stdout_file, "[I8085]  "); break;
                        case CPU_M6502 |CPU_AUDIO_CPU: fprintf(stdout_file, "[M6502]  "); break;
                        case CPU_I86   |CPU_AUDIO_CPU: fprintf(stdout_file, "[I86]    "); break;
                        case CPU_I8039 |CPU_AUDIO_CPU: fprintf(stdout_file, "[I8039]  "); break;
                        case CPU_M6803 |CPU_AUDIO_CPU: fprintf(stdout_file, "[M6808]  "); break;
                        case CPU_M6805 |CPU_AUDIO_CPU: fprintf(stdout_file, "[M6805]  "); break;
                        case CPU_M6809 |CPU_AUDIO_CPU: fprintf(stdout_file, "[M6809]  "); break;
                        case CPU_M68000|CPU_AUDIO_CPU: fprintf(stdout_file, "[M68000] "); break;
                        case CPU_T11   |CPU_AUDIO_CPU: fprintf(stdout_file, "[T-11]   "); break;
                        case CPU_S2650 |CPU_AUDIO_CPU: fprintf(stdout_file, "[S2650]  "); break;
#ifdef MESS
                        case CPU_PDP1  |CPU_AUDIO_CPU: fprintf(stdout_file, "[PDP-1]  "); break;
#else
                        case CPU_TMS34010 |CPU_AUDIO_CPU: fprintf(stdout_file, "[TMS34010] "); break;
#endif                           
                     }
                  }
               
                  for(j=0;j<MAX_CPU-1;j++) /* Increase to table to 4, when a game with 4 cpus will appear */
                  {
                     /* Dummy int to hold the number of specific sound chip.
                        In every multiple-chip interface, number of chips
                        is defined as the first variable, and it is integer. */
                     const struct MachineSound *x_sound = drivers[i]->drv->sound;
                     int *x_num = x_sound[j].sound_interface;
                     
                     switch(x_sound[j].sound_type)
                     {
                        /* These don't have a number of chips, only one possible */
                        case 0: fprintf(stdout_file, "          "); break; 
                        case SOUND_CUSTOM:  fprintf(stdout_file, "Custom    "); break;
                        case SOUND_SAMPLES: fprintf(stdout_file, "Samples   "); break;
                        case SOUND_NAMCO:   fprintf(stdout_file, "Namco     "); break;
                        case SOUND_TMS5220: fprintf(stdout_file, "TMS5520   "); break;
                        case SOUND_VLM5030: fprintf(stdout_file, "VLM5030   "); break;
                        default:
                           /* Let's print out the number of the chips */
                           fprintf(stdout_file, "%dx",*x_num);
                           /* Then the chip's name */
                           switch(x_sound[j].sound_type)
                           {
                              case SOUND_DAC:     fprintf(stdout_file, "DAC     "); break;
                              case SOUND_AY8910:  fprintf(stdout_file, "AY-8910 "); break;
                              case SOUND_YM2203:  fprintf(stdout_file, "YM-2203 "); break;
                              case SOUND_YM2151:  fprintf(stdout_file, "YM-2151 "); break;
                              case SOUND_YM2151_ALT: fprintf(stdout_file, "YM-2151a"); break;
                              case SOUND_YM2413:  fprintf(stdout_file, "YM-2413 "); break;
                              case SOUND_YM3812:  fprintf(stdout_file, "YM-3812 "); break;
                              case SOUND_SN76496: fprintf(stdout_file, "SN76496 "); break;
                              case SOUND_POKEY:   fprintf(stdout_file, "Pokey   "); break;
                              case SOUND_NES:     fprintf(stdout_file, "NES     "); break;
                              case SOUND_ADPCM:   fprintf(stdout_file, "ADPCM   "); break;
                              case SOUND_OKIM6295:fprintf(stdout_file, "OKI6295 "); break;
                              case SOUND_MSM5205: fprintf(stdout_file, "MSM5205 "); break;
                              case SOUND_ASTROCADE: fprintf(stdout_file, "ASTRCADE"); break;
                              default:            fprintf(stdout_file, "Unknown "); break;
                           }
                           break;
                     }
                  }
                  fprintf(stdout_file, "\n");
                  break;
               case LIST_ROMS: /* game roms list */
                  printromlist(drivers[i]->rom,drivers[i]->name);
                  fprintf(stdout_file, "\n");
                  break;
               case LIST_SAMPLES: /* game samples list */
                  gamedrv = drivers[i];
                  if (gamedrv->samplenames != 0 && gamedrv->samplenames[0] != 0)
                  {
                     j=0;
                     fprintf(stdout_file, "Samples need by %s:\n", gamedrv->name);
                     if(gamedrv->samplenames[0][0]=='*')
                     {
                        fprintf(stdout_file, "Uses samples from: %s\n", gamedrv->samplenames[0]+1);
                        j++;
                     } 
                     while(gamedrv->samplenames[j])
                     {
                           fprintf(stdout_file, "%s\n",gamedrv->samplenames[j]);
                           j++;
                     }
                  } else skipped++;
                  break;
               case LIST_SAMDIR: /* games list with samples directories */
                  if (drivers[i]->samplenames != 0 && drivers[i]->samplenames[0] != 0)
                  {
                        fprintf(stdout_file, "%-10s",drivers[i]->name);
                        if (drivers[i]->samplenames[0][0] == '*')
                              fprintf(stdout_file, "%s\n",drivers[i]->samplenames[0]+1);
                        else
                              fprintf(stdout_file, "%s\n",drivers[i]->name);
                  } else skipped++;
                  break;
               case LIST_CRC: /* list all crc-32 */
                  {
                     const struct RomModule *romp = drivers[i]->rom;

                     while (romp->name || romp->offset || romp->length)
                     {
                        if (romp->name && romp->name != (char *)-1)
                           fprintf(stdout_file, "%08x %-12s %s\n",romp->crc,romp->name,drivers[i]->description);
                        romp++;
                     }
                  }
                  break;
               case LIST_DUPCRC:
                  {
                     const struct RomModule *romp = drivers[i]->rom;

                     while (romp->name || romp->offset || romp->length)
                     {
                        if (romp->name && romp->name != (char *)-1)
                        {
                           j = i+1;
                           while (drivers[j])
                           {
                              const struct RomModule *romp1 = drivers[j]->rom;

                              while (romp1->name || romp1->offset || romp1->length)
                              {
                                 if (romp1->name && romp1->name != (char *)-1 &&
                                    strcmp(romp->name,romp1->name) &&
                                    romp1->crc == romp->crc)
                                 {
                                    printf("%08x %-12s %-8s <-> %-12s %-8s\n",romp->crc,
                                       romp->name,drivers[i]->name, romp1->name,drivers[j]->name);
                                 }
                                 romp1++;
                              }
                              j++;
                           }
                        }
                        romp++;
                     }                  
                  }
                  break;
               case LIST_COLORS:
                  fprintf(stdout_file, "%s  %d\n", drivers[i]->name,
                     drivers[i]->drv->total_colors);
                  break;
               case LIST_LMR: /* missing romsets list */
                  /* if the game is a clone, also try loading the ROM from the main version */
                  if (!osd_faccess (drivers[i]->name, OSD_FILETYPE_ROM) &&
                     (drivers[i]->clone_of == 0 || !osd_faccess(drivers[i]->clone_of->name,OSD_FILETYPE_ROM)))
                  {
                     fprintf(stdout_file, "%s\n", drivers[i]->name);
                     not_found++;
                  }
                  break;
               case LIST_WRONGORIENTATION: /* list drivers which incorrectly use the orientation and visible area fields */
                  if ((drivers[i]->drv->video_attributes & VIDEO_TYPE_VECTOR) == 0 &&
                              drivers[i]->drv->visible_area.max_x - drivers[i]->drv->visible_area.min_x + 1 <=
                              drivers[i]->drv->visible_area.max_y - drivers[i]->drv->visible_area.min_y + 1)
                  {
                      fprintf(stdout_file, "%s %dx%d\n",drivers[i]->name,
                         drivers[i]->drv->visible_area.max_x - drivers[i]->drv->visible_area.min_x + 1,
                         drivers[i]->drv->visible_area.max_y - drivers[i]->drv->visible_area.min_y + 1);
                      incorrect++;
                  } else correct++;
                  break;
               case VERIFY_ROMS: /* verify options */
               case VERIFY_SAMPLES:
                  if(list==VERIFY_ROMS)
                  {
                      /* some drivers don't use roms,
                        this happens in MESS */
                     if (drivers[i]->rom == NULL)
                     {
                        skipped++;
                        continue;
                     }
                     j = VerifyRomSet (i, (verify_printf_proc)printf);
                     fprintf(stdout_file, "romset %s ", drivers[i]->name);
                  }
                  else
                  {
                     /* ignore games that need no samples */
                        if (drivers[i]->samplenames == 0 ||
                        drivers[i]->samplenames[0] == 0)
                     {
                        skipped++;
                        continue;
                     }
                     j = VerifySampleSet (i,(verify_printf_proc)printf);
                     fprintf(stdout_file, "sampleset %s ", drivers[i]->name);
                  }
                  
                  switch (j)
                  {
                     case 0:
                        fprintf(stdout_file, "correct\n");
                        correct++;
                        break;
                     case 1:
                        fprintf(stdout_file, "not found\n");
                        not_found++;
                        break;
                     case 2:
                        fprintf(stdout_file, "incorrect\n");
                        incorrect++;
                        break;
                  }
                  break;
            }
         }
   }
   if (matching == 0)
   {
      fprintf(stderr_file, "Error: \"%s\" is not supported!\n", gamename);
      return 1;
   }
      
   fprintf(stdout_file, "\n\n");
   fprintf(stdout_file, "Total Supported: %d", i);
   if (matching  != i)
   {
      fprintf(stdout_file, ", Matching \"%s\": %d\n", gamename, matching);
   }
   else
   {
      fprintf(stdout_file, "\n");
   }
   if (skipped) fprintf(stdout_file, "Displayed: %d, Skipped: %d, because they don't use any roms/samples\n", matching-skipped, skipped);
   if (correct+incorrect) fprintf(stdout_file, "Found: %d, of which %d correct and %d incorrect\n", correct+incorrect, correct, incorrect);
   if (not_found) fprintf(stdout_file, "Not found: %d\n", not_found);
      
   if (incorrect > 0)
      return 2;
   else
      return 0;
}

int frontend_list_clones(char *gamename)
{
   /* listclones is a special case since the strwildcmp */
   /* also has to be done on clone_of. */
   int i;
   
   fprintf(stdout_file, "Name:    Clone of:\n");
   for (i=0;drivers[i];i++)
   {
      if (drivers[i]->clone_of &&
            (!strwildcmp(gamename,drivers[i]->name) || !strwildcmp(gamename,drivers[i]->clone_of->name)))
         fprintf(stdout_file, "%-8s %-8s\n",drivers[i]->name,drivers[i]->clone_of->name);
   }
   return 0;
}
