/* Written by Mark Vojkovich (mvojkovi@ucsd.edu) */
/* modified by Harald Deischinger */

#include "../cthugha.h"
#include "../cthugha-X11.h"
#include "../display.h"
#include "../disp-sys.h"
#include "dga.h"

#include <X11/Xlib.h>
#include <X11/X.h>
#include <X11/extensions/xf86dga.h>
#include <X11/extensions/xf86vmode.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

char*		DGAMemPtr;
char*		DGAMemBase;		
char*		DGASecondPage;

Bool		DGADoubleBuffer = 0;
int		DGABankSwitching;
int		DGACurrentBank = 0;	

int		DGARAMSize;
int		DGABankSize;
int		DGAPageSize;		

int 		DGAFrameBufferWidth;

void (*DGADoubleBuffer_function)();
void (*DGAClearScreen_function)();


int InitDGAEnvironment() {
    int MajorVersion, MinorVersion;
    int EventBase, ErrorBase;
    int dotclock;
    XF86VidModeModeLine modeline;

    /* test, if we get root privileges */
    seteuid(0);
    if(geteuid()) {
	printfe("Program must run with root permissions.\n");
	return 1;
    }
    seteuid(getuid());

    if (!XF86DGAQueryVersion(xcth_display,&MajorVersion,&MinorVersion)) { 
        printfe("Unable to query video extension version.\n");
        return 1;
    }

    if (!XF86DGAQueryExtension(xcth_display,&EventBase,&ErrorBase)) {
        printfe("Unable to query video extension information.\n");
        return 1;
    }

    if ( (MajorVersion < MINMAJOR) ||
	 (MajorVersion == MINMAJOR && MinorVersion < MINMINOR)) {
        printfe("Xserver is running an old XFree86-DGA version"
                " (%d.%d)\n", MajorVersion, MinorVersion);
        printfe("Minimum required version is %d.%d\n",
                MINMAJOR, MINMINOR);
        return 1;
    }

    XF86VidModeGetModeLine(xcth_display, xcth_screen, &dotclock,
			   &modeline);
    
    window_size.x = modeline.hdisplay;
    window_size.y = modeline.vdisplay;

    return 0;
}

int alloc_image_dga() {

    /* here comes some privilleged work */
    seteuid(0);

    /* get information */
    XF86DGAGetVideo(xcth_display, xcth_screen, &DGAMemBase,
                &DGAFrameBufferWidth, &DGABankSize, &DGARAMSize);

    bytes_per_line = DGAFrameBufferWidth * bypp;
    
    /* shanghai the keyboard */
    XGrabKeyboard(xcth_display,xcth_window,True,GrabModeAsync, 
		 GrabModeAsync,CurrentTime);  
    /* shanghai the mouse */
    XGrabPointer(xcth_display,xcth_window,True,
		 ButtonPressMask|PointerMotionMask, 
		 GrabModeAsync,GrabModeAsync,None,None,CurrentTime); 

    /* here goes nothing! */
    XF86DGADirectVideo(xcth_display,xcth_screen,
		       XF86DGADirectGraphics|XF86DGADirectMouse|
		       XF86DGADirectKeyb);
    seteuid(getuid());

    printfv(3,
	    "DGA information:\n"
	    "    MemBase         : 0x%x\n"
	    "    FrameBufferWidth: %d\n"
	    "    BankSize        : %d\n"
	    "    VideoRam        : %d kbytes\n",
	    (int)DGAMemBase,
	    DGAFrameBufferWidth,
	    DGABankSize,
	    DGARAMSize);

    /* defaults */
    DGASecondPage = DGAMemBase;
    DGAMemPtr = DGAMemBase;
    DGAPageSize = DGAFrameBufferWidth * window_size.y * bypp;
    DGARAMSize *= 1024;
    
    if(DGABankSize < DGARAMSize) {
	DGABankSwitching = DGAPageSize/DGABankSize;
        if((DGABankSwitching * DGABankSize) < DGAPageSize)
	    DGABankSwitching++;
	XF86DGASetVidPage(xcth_display,xcth_screen,0);
	printfv(2,"    Bankswitching... First frame fits on %i pages",
		DGABankSwitching);
	DGAClearScreen_function = DGAClearScreen_Banked;

	display_direct = 0;			/* must use software doublebuffering */
    } else {
	DGABankSwitching = 0;
	DGAClearScreen_function = DGAClearScreen_Linear;
    }
    
    XF86DGASetViewPort(xcth_display,xcth_screen,0,0);
    
    if(DGABankSwitching) {
	/* software double buffering only */
	printfv(0, "Only software double-buffering available with bank-switching.\n");
	nobuff = 1;
    }
    
    if(! DGAInitDoubleBuffer((nobuff ? SoftwareDB : HardwareDB),SubOK)) {
	exit_graph_mode();
	printfe("Unable to set up double-buffering!\n");
	return 1;
    } 	

    /* clear the screen completely */
    CLEARSCREEN();
    DOUBLEBUFFER(2); 
    CLEARSCREEN(); 
	
    return 0;
}

int free_image_dga() {

    XF86DGADirectVideo(xcth_display,xcth_screen,0);

    XUngrabPointer(xcth_display,CurrentTime);
    XUngrabKeyboard(xcth_display,CurrentTime);

    if(DGADoubleBuffer && 
       (DGADoubleBuffer_function != DGADoubleBuffer_Hardware))  {
	free(DGASecondPage);
    }

    return 0;
}


/******************************************************************/

void DGAClearScreen_Linear() {
    bzero(DGAMemPtr,DGAPageSize);
}

void DGAClearScreen_Banked() {
    for(DGACurrentBank = 0;
	DGACurrentBank < (DGABankSwitching - 1);
	DGACurrentBank++) 
    {
	XF86DGASetVidPage(xcth_display,xcth_screen,DGACurrentBank);
	bzero(DGAMemBase,DGABankSize);
    }

    /* is this sort of precaution necessary? */
    XF86DGASetVidPage(xcth_display,xcth_screen,DGACurrentBank);
    bzero(DGAMemBase,DGAPageSize - (DGABankSize * DGACurrentBank));
}

/******************************************************************/

void DGADoubleBuffer_Hardware(int clear) {
    if((char*)DGAMemPtr == DGAMemBase) {
	DGAMemPtr = DGASecondPage;
   	XF86DGASetViewPort(xcth_display,xcth_screen,0,0);
    } else {
	DGAMemPtr = DGAMemBase;
   	XF86DGASetViewPort(xcth_display,xcth_screen,0,window_size.y);
    }
}



void DGADoubleBuffer_Software_Linear(int clear) {
    int i;
    char * src;
    char * dst;

    switch(clear) {
    case 2:	/* copy the complete screen */
	memcpy(DGAMemBase,DGAMemPtr,DGAPageSize);
	break;

    case 1:	/* copy the display size */
	dst = DGAMemBase + 
	    (window_size.y - disp_size.y)/2 * bytes_per_line +	
	    (window_size.x - disp_size.x)/2 * bypp;			
	src = DGAMemPtr + 
	    (window_size.y - disp_size.y)/2 * bytes_per_line +	
	    (window_size.x - disp_size.x)/2 * bypp;			
	
	for(i=0; i < disp_size.y; i++) {
	    memcpy(dst, src, disp_size.x * bypp);
	    dst += bytes_per_line;
	    src += bytes_per_line;
	}
	break;

    default:	/* copy the drawing area only */
	dst = DGAMemBase + 
	    (window_size.y - draw_size.y)/2 * bytes_per_line +	
	    (window_size.x - draw_size.x)/2 * bypp;			
	src = DGAMemPtr + 
	    (window_size.y - draw_size.y)/2 * bytes_per_line +	
	    (window_size.x - draw_size.x)/2 * bypp;			
	
	for(i=0; i < draw_size.y; i++) {
	    memcpy(dst, src, draw_size.x * bypp);
	    dst += bytes_per_line;
	    src += bytes_per_line;
	}
    }

}

void DGADoubleBuffer_Software_Banked(int clear) {
     for(DGACurrentBank = 0;
	 DGACurrentBank < (DGABankSwitching - 1);
	 DGACurrentBank++) 
    {
	XF86DGASetVidPage(xcth_display,xcth_screen,DGACurrentBank);
	memcpy(DGAMemBase,DGAMemPtr + (DGABankSize * DGACurrentBank),
					DGABankSize);
    }
    /* is this sort of precaution necessary?  */
    XF86DGASetVidPage(xcth_display,xcth_screen,DGACurrentBank);
    memcpy(DGAMemBase,DGAMemPtr + (DGABankSize * DGACurrentBank),
		DGAPageSize - (DGABankSize * DGACurrentBank)); 
}



int DGAInitDoubleBuffer(int type,Bool sub) {
    DGADoubleBuffer = 0;

    if(type == HardwareDB) {
	if((DGAPageSize*2) < DGARAMSize) {
	    printfv(2,"    Yahoo! Both pages fit in the videoram.\n"
		    "\tHardware DoubleBuffering.\n");
	    DGASecondPage = DGAMemBase + DGAPageSize;
	    DGADoubleBuffer = 1;
	    DGADoubleBuffer_function = DGADoubleBuffer_Hardware;
        } else {
	    printfv(2,"    Not enough videoram for Hardware Double "
		    "Buffering.\n");
	    if(sub == NoSub)
	    	return DGADoubleBuffer;
 	}
    }
    
    if(!DGADoubleBuffer) {
	if((DGASecondPage = (char*)malloc(DGAPageSize))) {
	    printfv(2,"    Software DoubleBuffering.\n");
	    DGADoubleBuffer = 1;
	    if(DGABankSwitching) {
	       DGADoubleBuffer_function = DGADoubleBuffer_Software_Banked;
		/* we need to overwrite the clearing function since
		   we don't need to bank switch when writing host memory! */
	       DGAClearScreen_function = DGAClearScreen_Linear;
            }
	    else 
	       DGADoubleBuffer_function = DGADoubleBuffer_Software_Linear;
	} else {
	    printfv(2,"    Unable to allocate bitmap. Double Buffering "
		    "not available.\n");
	    return DGADoubleBuffer;
	}	
    }

    DGAMemPtr = DGASecondPage;
    
    return DGADoubleBuffer;
}



