/* --------------------------------------------------------------------------
 * Test driver for the Haskell server.
 *
 * Copyright (c) Mark P Jones 1991-1996.   All rights reserved.
 * See NOTICE for details and conditions of use etc...
 * Hugs version 1.3, August 1996
 *
 * $RCSfile: test.c,v $
 * $Revision: 1.1 $
 * $Date: 1997/01/07 15:45:56 $
 * ------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#include "config.h"
#include "options.h"

/* where is the data for this experiment? */
#define DATADIR "Expt/e0000" 

typedef unsigned Bool;
#define TRUE  1
#define FALSE 0


extern void main      (int, char*[]);
static void check     (Bool);
static void loadHugs  (int, char*[]);

#include "server.h"
static HugsServerAPI *hugs = 0;
#define hugsAp(f,x) f; x; hugs->apply();  /* prefix apply macro */

static void check(haltOnError)
Bool haltOnError;
{
    char* err = hugs->clearError();
    if (err) {
        fprintf(stderr,"Error\n%s\n",err);
        fflush(stderr);
	if (haltOnError)
	    exit(1);
    }
}

/* --------------------------------------------------------------------------
 * Dynamic loading
 * ------------------------------------------------------------------------*/

#if HAVE_DLFCN_H /* eg LINUX, SOLARIS, ULTRIX */
#include <stdio.h>
#include <dlfcn.h>

/* SunOS4 doesn't define define RTLD_NOW - so we use RTLD_LAZY instead     */
#ifdef RTLD_NOW
#define MODE RTLD_NOW
#else
#ifdef RTLD_LAZY
#define MODE RTLD_LAZY
#else
#undef MODE
#endif
#endif

#define GetDLL(nm)     dlopen(nm, MODE)
#define GetProc(h,nm)  dlsym(h,nm)
typedef void           *DLLInstance;
#define DLLError()     dlerror()

#elif HAVE_DL_H /* eg HPUX */

#include <dl.h>
#define GetDLL(nm)     shl_open(nm,BIND_IMMEDIATE,0L)
#define GetProc(h,nm)  shl_lookup(h,nm)
typedef void           *DLLInstance;
#define DLLError()     ??? /* ToDo: fill this in - use "" if nothing else works */

#elif HAVE_WINDOWS_H && !defined(__GNUC__)

#include <windows.h>
#define GetDLL(nm)     LoadLibrary(nm)
#define GetProc(h,nm)  GetProcAddress(h,nm)
typedef HINSTANCE      DLLInstance;

static char* DLLError() {
#if 1
    /* ADR note: errors loading DLLs cause a message box to popup in the
     * users window and clear the "LastError" variable.  This causes the
     * more complicated code below to return "" - so we prefer this 
     * simpler piece of code.
     */
    return "";
#else
    static char errorMessageBuffer[100];
    FormatMessage(
		  0,
		  (LPTSTR) NULL,
		  GetLastError(),
		  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		  (LPTSTR) errorMessageBuffer,
		  100,
		  NULL
		  );
    return errorMessageBuffer;
#endif
}

#else /* Dynamic loading not available */

/* leaving this section blank will cause link errors - this is what we want */

#endif /* Dynamic loading not available */


static void loadHugs(argc,argv)
int    argc;
char* argv[]; {
    HugsServerInitFun initHugsServer;
    DLLInstance handle = GetDLL("./hugs.so");
    if (NULL == handle) {
	fprintf(stderr,"Can't Find Hugs dll\n");
	fflush(stderr);
	exit(1);
    }
    initHugsServer = (HugsServerInitFun) GetProc(handle,"initHugsServer");
    if (NULL == initHugsServer) {
	fprintf(stderr,"Can't find \"initHugsServer\"\n");
	fflush(stderr);
	exit(1);
    }
    hugs = initHugsServer(argc,argv);
    check(TRUE);
}

/* --------------------------------------------------------------------------
 * main
 * ------------------------------------------------------------------------*/

void main(argc,argv)
int    argc;
char* argv[]; {
    int    baseScript; /* number of original script */
    int    i;
    char* s;
    HVal   h;

    loadHugs(argc,argv);

    baseScript = hugs->getNumScripts();
    hugs->loadFile( DATADIR "/Foo.hs" );
    check(TRUE);

    hugs->lookupName("Foo","test1");
    i = hugs->evalInt();
    check(TRUE); printf("Result1: %d\n", i); fflush(stdout);

    hugs->reset(baseScript);
    hugs->loadFile( DATADIR "/Foo.hs");
    check(TRUE);

    hugs->lookupName("Foo","test2");
    i = hugs->evalInt();
    check(TRUE); printf("Result3: %d\n", i); fflush(stdout);

    hugs->mkInt(3);
    i = hugs->evalInt();
    check(TRUE); printf("Result4: %d\n", i); fflush(stdout);

    /* Deliberately load a bogus file to check the error message */
    baseScript = hugs->getNumScripts();
    hugs->loadFile("This File Disnae Exist");
    check(FALSE);

    /* Check that the error was cleared */
    hugs->reset(baseScript);
    check(TRUE);

    hugsAp(hugs->lookupName("Foo","fac"),
	   hugs->mkInt(3));
    i = hugs->evalInt();
    check(TRUE); printf("Result5: %d\n", i); fflush(stdout);

    hugs->mkString("hello");
    s = hugs->evalString();
    check(TRUE); printf("Result6: %s\n", s); free(s); fflush(stdout);

    hugsAp(hugs->lookupName("Prelude","putStrLn"),
	   hugs->mkString("hello"));
    hugs->doIO();
    check(TRUE); fflush(stdout);

    hugsAp(hugs->lookupName("Foo","putStrLn"),
	   hugs->mkString("hello again"));
    hugs->doIO();
    check(TRUE); fflush(stdout);

    hugsAp(hugsAp(hugs->lookupName("Foo","++"),
		  hugs->mkString("hello")),
	   hugs->mkString(" again"));
    s = hugs->evalString();
    check(TRUE); printf("String: \"%s\"\n", s); fflush(stdout);

    h = hugs->compileExpr("Foo","print [1..10]");
    hugs->pushHVal(h);
    hugs->doIO();
    check(TRUE); fflush(stdout);

#if 0
    hugs->changeDir("../../Version7");
    hugs->loadProject("rbmh.prj");
    hugsAp(hugs->lookupName("DynamicTest","disp"),
	   hugs->lookupName("DynamicTest","p17"));
    hugs->doIO();
    fflush(stdout);
#endif

    exit(0);
}


