/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995-1998.  The Regents of the University of California.  All     */
/*   rights reserved.                                                      */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <ctype.h>
#include <Xm/Xm.h>
#include "xdir.h"
#include "str.h"


/*
 * parse_short_entry - Parse "entry" to extract the entry "name" and "type".
 *                     "entry" consists of a name with, possibly, a symbol
 *                     appended ('@', '*', '=', or '/').  The caller is
 *                     responsible for releasing the memory pointed to
 *                     by "name" with a call to XtFree().  "system"
 *                     identifies the system (e.g.,  SYS_UNIX).
 */
parse_short_entry(system, entry, name, type)
int system;
char *entry;
char **name;
char *type;
{
	switch (system) {
	case SYS_VMS:
		vms_parse_short_entry(entry, name, type);
		break;
	default:
		unix_parse_short_entry(entry, name, type);
	}
}


/*
 * unix_parse_short_entry - Parse the Unix "entry" to extract the entry
 *                          "name" and "type".  "entry" consists of a
 *                          name with, possibly, a symbol appended
 *                          ('@', '*', '=', or '/').  The caller is
 *                          responsible for releasing the memory
 *                          pointed to by "name" with a call to XtFree().
 */
unix_parse_short_entry(entry, name, type)
char *entry;
char **name;
char *type;
{
	int len;

	*name = XtNewString(entry);
	len = strlen(*name);

	switch ((*name)[len-1]) {
	case '/':
		*type = DIRECTORY_TYPE;
		(*name)[len-1] = '\0';
	break;
	case '*':
		*type = EXECUTABLE_TYPE;
	(*name)[len-1] = '\0';
		break;
	case '@':
		*type = LINK_TYPE;
		(*name)[len-1] = '\0';
		break;
	case '=':
		*type = SOCKET_TYPE;
		(*name)[len-1] = '\0';
		break;
	default:
		*type = FILE_TYPE;
	}
}


/*
 * vms_parse_short_entry - Parse the VMS "entry" to extract the entry
 *                         "name" and "type".  The caller is
 *                         responsible for releasing the memory
 *                         pointed to by "name" with a call to XtFree().
 */
vms_parse_short_entry(entry, name, type)
char *entry;
char **name;
char *type;
{
	/* For name, just return a copy of the entry */
	*name = XtNewString(entry);

	/* Is the entry a directory or a file? */
	if (!strcmp(&(entry[strlen(entry)-4]), ".dir") || strstr(entry, ".dir;"))
		*type = DIRECTORY_TYPE;
	else
		*type = FILE_TYPE;
}


/*
 * parse_long_entry - Parse "entry" to extract the "name", "info" and
 *                    "type".  The caller is responsible for releasing
 *                    the memory pointed to by "name" and "info" with
 *                    a call to XtFree().  "system" identifies the
 *                    system (e.g.,  SYS_UNIX) and "server" identifies the
 *                    server.
 */
parse_long_entry(system, server, entry, name, info, type)
int system;
int server;
char *entry;
char **name;
char **info;
char *type;
{
	switch (system) {
	case SYS_VMS:
		vms_parse_long_entry(server, entry, name, info, type);
		break;
	case SYS_MAC:
		mac_parse_long_entry(entry, name, info, type);
		break;
	case SYS_NT:
		nt_parse_long_entry(server, entry, name, info, type);
		break;
	default:
		unix_parse_long_entry(entry, name, info, type);
	}
}


/*
 * mac_parse_long_entry - Parse the Macintosh "entry" to extract the "name",
 *                        "info" and "type".  The caller is responsible
 *                        for releasing the memory pointed to by "name"
 *                        and "info" with a call to XtFree().  This function
 *                        assumes that SERVER_MAC_FTPD is the server.
 */
mac_parse_long_entry(entry, name, info, type)
char *entry;
char **name;
char **info;
char *type;
{
	char *perms;
	char *temp;

	/* Info is simply a copy of entry */
	*info = XtNewString(entry);

	/* Assume that entry name starts at fixed location */
	*name = XtNewString(entry+54);

	/* Assume that the first blank-delimited token is the permissions */
	temp = XtNewString(entry);
	if (((perms = strtok(temp, " ")) == NULL) || (strlen(perms) != 10)) {
		*type = UNKNOWN_TYPE;
		XtFree(temp);
		return;
	}

	/* Determine file type */
	switch (perms[0]) {
	case '-':
		if ((perms[3] == 'x') || (perms[6] == 'x') || (perms[9] == 'x'))
			*type = EXECUTABLE_TYPE;
		else
			*type = FILE_TYPE;
		break;
	case 'd':
		*type = DIRECTORY_TYPE;
		break;
	case 'l':
		*type = LINK_TYPE;
		break;
	default:
		*type = UNKNOWN_TYPE;
	}
	XtFree(temp);
}


/*
 * nt_parse_long_entry - Parse the Windows NT "entry" to extract the "name",
 *                       "info" and "type".  The caller is responsible
 *                       for releasing the memory pointed to by "name"
 *                       and "info" with a call to XtFree().  "server"
 *                       identifies the server.
 */
nt_parse_long_entry(server, entry, name, info, type)
int server;
char *entry;
char **name;
char **info;
char *type;
{
	char *p;
	int i;
	char *perms;
	char *temp;

	/* Treat nonstandard NT server like Unix */
	if (server != SERVER_NT || strlen(entry) < 60 || entry[58] != ' ') {
		unix_parse_long_entry(entry, name, info, type);
		return;
	}

	/* Info is simply a copy of entry with any trailing blanks removed */
	*info = XtNewString(entry);

	/* Assume that entry name starts at fixed location */
	*name = XtNewString(entry+59);

	/* Assume that the first blank-delimited token is the permissions */
	temp = XtNewString(entry);
	if (((perms = strtok(temp, " ")) == NULL) || (strlen(perms) != 10)) {
		*type = UNKNOWN_TYPE;
		XtFree(temp);
		return;
	}

	/* Determine file type */
	switch (perms[0]) {
	case '-':
		if ((perms[3] == 'x') || (perms[6] == 'x') || (perms[9] == 'x'))
			*type = EXECUTABLE_TYPE;
		else
			*type = FILE_TYPE;
		break;
	case 'd':
		*type = DIRECTORY_TYPE;
		break;
	default:
		*type = UNKNOWN_TYPE;
	}
	XtFree(temp);
}


/*
 * unix_parse_long_entry - Parse the Unix "entry" to extract the "name",
 *                         "info" and "type".  The caller is responsible
 *                         for releasing the memory pointed to by "name"
 *                         and "info" with a call to XtFree().
 */
unix_parse_long_entry(entry, name, info, type)
char *entry;
char **name;
char **info;
char *type;
{
	char *p;
	int i;
	char *perms;
	char *temp;

	/* Info is simply a copy of entry with any trailing blanks removed */
	*info = XtNewString(entry);
	for (i=strlen(*info)-1; i>=0; i--)
		if ((*info)[i] == ' ')
			(*info)[i] = '\0';
		else
			break;

	/* Assume that the first blank-delimited token is the permissions */
	temp = XtNewString(entry);
	if (((perms = strtok(temp, " ")) == NULL) || (strlen(perms) != 10))
		*type = UNKNOWN_TYPE;
	else
		switch (perms[0]) {
		case 'b':
		case 'c':
		case 'p':
		case '-':
		case 'm':
			if ((perms[3] == 'x') || (perms[6] == 'x') || (perms[9] == 'x'))
				*type = EXECUTABLE_TYPE;
			else
				*type = FILE_TYPE;
			break;
		case 'd':
			*type = DIRECTORY_TYPE;
			break;
		case 'l':
			*type = LINK_TYPE;
			break;
		case 's':
			*type = SOCKET_TYPE;
			break;
		default:
			*type = UNKNOWN_TYPE;
		}
	XtFree(temp);

	/* Get entry name.  Look for links */
	temp = XtNewString(entry);
	if ((p = strrchr(temp, ' ')) == NULL)
		fatal_error("Problem in unix_parse_long_entry()");
	if (*type == LINK_TYPE) {
		*p = '\0';
		if ((p = strrchr(temp, ' ')) == NULL)
			fatal_error("Problem in unix_parse_long_entry()");
		*p = '\0';
		if ((p = strrchr(temp, ' ')) == NULL)
			fatal_error("Problem in unix_parse_long_entry()");
	}
	*name = XtNewString(p+1);
	XtFree(temp);
}


/*
 * vms_parse_long_entry - Parse the VMS "entry" to extract the "name",
 *                        "info" and "type".  The caller is responsible
 *                        for releasing the memory pointed to by "name"
 *                        and "info" with a call to XtFree().  "server"
 *                        identifies the server.
 */
vms_parse_long_entry(server, entry, name, info, type)
int server;
char *entry;
char **name;
char **info;
char *type;
{
	char *p;
	char *temp;
	int i;
	int len;

	/* Info is basically a copy of entry with some changes to white space */
	*info = XtNewString(entry);
	for (i=strlen(*info)-1; i>=0; i--)
		if ((*info)[i] == ' ')
			(*info)[i] = '\0';
		else
			break;
	for (i=strlen(*info)-1; i>=0; i--)
		if ((*info)[i] == '\t')
			(*info)[i] = ' ';

	/* Assume that first blank-delimited token is the entry name */
	temp = XtNewString(entry);
	if (server == SERVER_VMS_MULTINET) {
		p = strtok(temp, " \t;");
		len = strlen(p);
		if (p[len-1] == '.')
			p[len-1] = '\0';
	} else
		p = strtok(temp, " \t");
	if (p == NULL)
		fatal_error("Problem in vms_parse_long_entry()");
	for (i=strlen(p)-1; i>=0; i--)
		p[i] = tolower(p[i]);
	*name = XtNewString(p);

	/* Is the entry a directory or a file? */
	if (strstr(p, ".dir"))
		*type = DIRECTORY_TYPE;
	else
		*type = FILE_TYPE;

	XtFree(temp);
}

